/*
 * This file is part of the ESO Telluric Correction Library
 * Copyright (C) 2001-2018 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */


/*----------------------------------------------------------------------------*/
/**
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include <cpl.h>


#include "mf_wrap_calc.h"


/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Enumeration types
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Defines
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Global variables
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Macros
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Structured types
 */
/*----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------*/
/**
 *                 Functions prototypes
 */
/*----------------------------------------------------------------------------*/


cpl_error_code mf_wrap_calc_molecules(cpl_frameset *frameset, cpl_table **molecules, const char *arm)
{
    /* Load TAG = MOLECULES */

    cpl_error_code err           = CPL_ERROR_NONE;
    cpl_table     *molecules_int = NULL;
    if (!err) {
        const char *input_tag = mf_wrap_tag_suffix(MOLECFIT_MODEL_MOLECULES, arm, CPL_FALSE);
        cpl_msg_info(cpl_func, "Loading %s cpl_table", input_tag);
        molecules_int = mf_wrap_load_unique_table(frameset, input_tag);

        if (!molecules_int) {
            err = CPL_ERROR_ILLEGAL_INPUT;
        }
        else {
            err = cpl_error_get_code();
        }
    }
    if (molecules_int) {
        *molecules = cpl_table_duplicate(molecules_int);
        cpl_table_delete(molecules_int);
    }
    return err;
}

cpl_error_code mf_wrap_calc_kernel_library(
    mf_wrap_fits               **kernel_data,
    cpl_table                  **mapping_kernel,
    cpl_frameset                *frameset,
    mf_wrap_calctrans_parameter *parameters,
    const char                  *arm
)
{
    cpl_error_code err                = CPL_ERROR_NONE;
    cpl_table     *mapping_kernel_int = NULL;

    if (!err) {
        const char *input_tag = mf_wrap_tag_suffix(MOLECFIT_CALCTRANS_KERNEL_LIBRARY, arm, CPL_FALSE);

        *kernel_data = mf_wrap_load_kernel_tag(
            frameset, input_tag, parameters->use_input_kernel, parameters->mf_config->parameters, CPL_TRUE
        );
        err = cpl_error_get_code();

        if (*kernel_data && !err) {
            if (parameters->mapping_kernel_table) {
                /* Mapping in the recipe parameters */
                mapping_kernel_int = cpl_table_duplicate(parameters->mapping_kernel_table);
            }
            else {
                /* Mapping in the static_calib input FITS files */
                cpl_errorstate pre_state = cpl_errorstate_get();
                input_tag                = mf_wrap_tag_suffix(MOLECFIT_CALCTRANS_MAPPING_KERNEL, arm, CPL_FALSE);
                const cpl_frame *input_mapping_kernel = cpl_frameset_find(frameset, input_tag);
                if (input_mapping_kernel) {
                    cpl_msg_info(cpl_func, "Loading %s cpl_table", input_tag);
                    mapping_kernel_int = mf_wrap_load_unique_table(frameset, input_tag);
                }
                else {
                    cpl_errorstate_set(pre_state);
                    input_tag            = mf_wrap_tag_suffix(MOLECFIT_MAPPING_KERNEL, arm, CPL_FALSE);
                    input_mapping_kernel = cpl_frameset_find(frameset, input_tag);
                    if (input_mapping_kernel) {
                        cpl_msg_info(cpl_func, "Loading %s cpl_table", input_tag);
                        mapping_kernel_int = mf_wrap_load_unique_table(frameset, input_tag);
                    }
                    else {
                        err = cpl_error_set_message(
                            cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                            "Mapping kernel data not found in frameset and neither in the recipe parameter!"
                        );
                    }
                }
            }

            if (!mapping_kernel_int) {
                err = CPL_ERROR_ILLEGAL_INPUT;
            }
            else {
                err = cpl_error_get_code();
            }
        }
        if (mapping_kernel_int) {
            *mapping_kernel = cpl_table_duplicate(mapping_kernel_int);
            cpl_table_delete(mapping_kernel_int);
        }

        if (!err) {
            cpl_msg_info(cpl_func, "Using the default molecfit kernels -> With the values inside BEST_FIT_PARMS!");
        }
    }

    return err;
}

cpl_error_code mf_wrap_calc_mapping_atm(
    cpl_frameset                *frameset,
    cpl_table                  **mapping_atmospheric,
    mf_wrap_calctrans_parameter *parameters,
    const char                  *arm
)
{
    cpl_error_code err                     = CPL_ERROR_NONE;
    cpl_table     *mapping_atmospheric_int = NULL;

    if (!err) {
        if (parameters->mapping_atmospheric_table) {
            mapping_atmospheric_int = cpl_table_duplicate(parameters->mapping_atmospheric_table);
        }
        else {
            const char *input_tag = mf_wrap_tag_suffix(MOLECFIT_MAPPING_ATMOSPHERIC, arm, CPL_FALSE);
            cpl_msg_info(cpl_func, "Loading %s cpl_table", input_tag);
            mapping_atmospheric_int = mf_wrap_load_unique_table(frameset, input_tag);
        }


        if (!mapping_atmospheric_int) {
            err = CPL_ERROR_ILLEGAL_INPUT;
        }
        else {
            err = cpl_error_get_code();
        }
    }
    if (mapping_atmospheric_int) {
        *mapping_atmospheric = cpl_table_duplicate(mapping_atmospheric_int);
        cpl_table_delete(mapping_atmospheric_int);
    }
    return err;
}

cpl_error_code mf_wrap_calc_mapping_conv(
    cpl_frameset                *frameset,
    cpl_table                  **mapping_convolve,
    mf_wrap_calctrans_parameter *parameters,
    const char                  *arm
)
{
    cpl_error_code err                  = CPL_ERROR_NONE;
    cpl_table     *mapping_convolve_int = NULL;

    if (!err) {
        if (parameters->mapping_convolve_table) {
            mapping_convolve_int = cpl_table_duplicate(parameters->mapping_convolve_table);
        }
        else {
            const char *input_tag = mf_wrap_tag_suffix(MOLECFIT_MAPPING_CONVOLVE, arm, CPL_FALSE);
            cpl_msg_info(cpl_func, "Loading %s cpl_table", input_tag);
            mapping_convolve_int = mf_wrap_load_unique_table(frameset, input_tag);
        }

        if (!mapping_convolve_int) {
            err = CPL_ERROR_ILLEGAL_INPUT;
        }
        else {
            err = cpl_error_get_code();
        }
    }
    if (mapping_convolve_int) {
        *mapping_convolve = cpl_table_duplicate(mapping_convolve_int);
        cpl_table_delete(mapping_convolve_int);
    }
    return err;
}
