/*
 * This file is part of the ESO Telluric Correction Library
 * Copyright (C) 2001-2018 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/*----------------------------------------------------------------------------*/
/**
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

//#include "mf_wrap.h"
#include "mf_wrap_config_calc.h"
#include <regex.h>


/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Enumeration types
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Defines
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Global variables
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Macros
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Structured types
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Functions prototypes
 */
/*----------------------------------------------------------------------------*/

// LOAD TAG FUNCTIONS

cpl_error_code mf_wrap_load_tag_science(mf_wrap_fits **data, cpl_frame *input_frame)
{
    /* Load TAG = SCIENCE_CALCTRANS/SCIENCE */

    cpl_error_code err = CPL_ERROR_NONE;

    if (!err) {
        const char *science_file = cpl_frame_get_filename(input_frame);

        *data = mf_wrap_fits_load(science_file, CPL_FALSE);

        if (!data) {
            err = CPL_ERROR_ILLEGAL_INPUT;
        }
        else {
            err = cpl_error_get_code();
        }
    }
    return err;
}


/*cpl_error_code mf_wrap_config_calc_setup_input_filenames(cpl_parameterlist* ilist, const char* suffix, const char* input_name){

        cpl_error_code err= CPL_ERROR_NONE;

        const char* suf;
        if(!suffix) suf = "";
        else suf = "_%s" % (suffix);
        const char* type;
        if(strcmp(input_name,"SCI")){
            type ="_SCI";
        } else if(strcmp(input_name,"STD") || strcmp(input_name,"TELL")){
            type ="_STD";
        } else {
            type = "";
        }

        cpl_parameterlist_append(ilist,cpl_parameter_new_value(cpl_sprintf("ATM_PARAMETERS%s%s",type,suf),CPL_TYPE_STRING,NULL,NULL,MOLECFIT_ATM_PARAMETERS));
        cpl_parameterlist_append(ilist,cpl_parameter_new_value(cpl_sprintf("BEST_FIT_PARAMETERS%s%s",type,suf),CPL_TYPE_STRING,NULL,NULL,MOLECFIT_BEST_FIT_PARAMETERS));
        cpl_parameterlist_append(ilist,cpl_parameter_new_value(cpl_sprintf("BEST_FIT_MODEL%s%s",type,suf),CPL_TYPE_STRING,NULL,NULL,MOLECFIT_BEST_FIT_MODEL));
        cpl_parameterlist_append(ilist,cpl_parameter_new_value(cpl_sprintf("MODEL_MOLECULES%s",suf),CPL_TYPE_STRING,NULL,NULL,MOLECFIT_MODEL_MOLECULES));
        cpl_parameterlist_append(ilist,cpl_parameter_new_value(cpl_sprintf("KERNEL_LIBRARY%s",suf),CPL_TYPE_STRING,NULL,NULL,MOLECFIT_KERNEL_LIBRARY));

        return CPL_ERROR_NONE;
    }

cpl_error_code mf_wrap_config_calc_setup_output_filenames(cpl_parameterlist* ilist, const char* suffix){

        cpl_error_code err = CPL_ERROR_NONE;

        const char* suf;
        if(!suffix) suf = "";
        else suf = "_%s" % (suffix);

        cpl_parameterlist_append(ilist,cpl_parameter_new_value(cpl_sprintf("TELLURIC_CORR%s",suf),CPL_TYPE_STRING,NULL,NULL,MOLECFIT_TELLURIC_CORR));
        cpl_parameterlist_append(ilist,cpl_parameter_new_value(cpl_sprintf("CALCTRANS_KERNEL_LIBRARY%s",suf),CPL_TYPE_STRING,NULL,NULL,MOLECFIT_CALCTRANS_KERNEL_LIBRARY));
        cpl_parameterlist_append(ilist,cpl_parameter_new_value(cpl_sprintf("LBLRTM_RESULTS%s",suf),CPL_TYPE_STRING,NULL,NULL,MOLECFIT_LBLRTM_RESULTS));

        return CPL_ERROR_NONE;
}
*/


/**@}*/


/*----------------------------------------------------------------------------*/
/**
 * @brief  Generate the internal configuration file for the recipe and check values
 *
 * @param  list   parameterlist with the parameters
 *
 * @return configuration file or NULL if exist an error
 *
 */
/*----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------*/
/**
 * @brief    Deallocate the given parameter configuration object and its contents
 *
 * @param    parameters       The parameter configuration variable in the recipe.
 */
/*----------------------------------------------------------------------------*/

void mf_wrap_calctrans_parameter_delete(mf_wrap_calctrans_parameter *parameters)
{
    if (parameters) {
        if (parameters->mapping_kernel_table) {
            cpl_table_delete(parameters->mapping_kernel_table);
        }

        if (parameters->mapping_atmospheric_table) {
            cpl_table_delete(parameters->mapping_atmospheric_table);
        }

        if (parameters->mapping_convolve_table) {
            cpl_table_delete(parameters->mapping_convolve_table);
        }

        if (parameters->atm_parameters) {
            mf_wrap_fits_delete(parameters->atm_parameters);
        }
        if (parameters->best_fit_parameters) {
            mf_wrap_fits_delete(parameters->best_fit_parameters);
        }

        if (parameters->telluriccorr_head) {
            for (cpl_size ext = 0; ext < parameters->n_ext; ext++) {
                if (parameters->telluriccorr_head[ext]) {
                    cpl_propertylist_delete(parameters->telluriccorr_head[ext]);
                }
            }
            cpl_free(parameters->telluriccorr_head);
        }

        if (parameters->telluriccorr_data) {
            for (cpl_size ext = 0; ext < parameters->n_ext; ext++) {
                if (parameters->telluriccorr_data[ext]) {
                    cpl_table_delete(parameters->telluriccorr_data[ext]);
                }
            }
            cpl_free(parameters->telluriccorr_data);
        }

        if (parameters->best_fit_parameters_table) {
            for (cpl_size ext = 0; ext < parameters->n_ext; ext++) {
                if (parameters->best_fit_parameters_table[ext]) {
                    cpl_table_delete(parameters->best_fit_parameters_table[ext]);
                }
            }
            cpl_free(parameters->best_fit_parameters_table);
        }

        if (parameters->results_lblrtm) {
            for (cpl_size ext = 0; ext < parameters->n_ext; ext++) {
                if (parameters->results_lblrtm[ext]) {
                    mf_calctrans_lblrtm_results_delete(parameters->results_lblrtm[ext]);
                }
            }
            cpl_free(parameters->results_lblrtm);
        }

        if (parameters->results_convolution) {
            for (cpl_size ext = 0; ext < parameters->n_ext; ext++) {
                if (parameters->results_convolution[ext]) {
                    mf_calctrans_convolution_results_delete(parameters->results_convolution[ext]);
                }
            }
            cpl_free(parameters->results_convolution);
        }

        if (parameters->mf_config) {
            mf_configuration_delete(parameters->mf_config);
        }

        if (parameters->pl) {
            cpl_propertylist_delete(parameters->pl);
        }

        cpl_free(parameters);
    }
}


/*----------------------------------------------------------------------------*/
/**
 * @brief  Generate the internal configuration file required for CALCTRANS .... mf_model and check values
 *
 * @param  parlist   parameterlist with the parameters
 *
 * @return mf_wrap_model_parameter parameters structure with instrument-independent defaults
 *
 */
/*----------------------------------------------------------------------------*/
mf_wrap_calctrans_parameter *mf_wrap_config_calc_init(
    cpl_frameset            *frameset,
    const cpl_parameterlist *list,
    cpl_propertylist        *scientific_head_pri,
    cpl_size                 n_ext,
    const char              *combined_suffix
)
// cpl_frameset *frameset --> was this needed ?
{
    /* copy of molecfit_calctrans_parameters from molecfit_calctrans.c */
    cpl_msg_info(cpl_func, "mf_wrap_config_calc_init");

    cpl_frameset_dump(frameset, stdout);
    /* Check input */
    cpl_error_ensure(frameset && list, CPL_ERROR_NULL_INPUT, return NULL, "list input is NULL!");

    /* Get preState */
    cpl_errorstate       preState = cpl_errorstate_get();
    const cpl_parameter *p;

    /* Create the configuration parameter */
    // mf_wrap_calctrans_parameter is defined in mf_wrap_config_calc.h

    mf_wrap_calctrans_parameter *parameters = cpl_malloc(sizeof(mf_wrap_calctrans_parameter));
    parameters->mapping_kernel_table        = NULL;
    parameters->mapping_atmospheric_table   = NULL;
    parameters->mapping_convolve_table      = NULL;
    parameters->atm_parameters              = NULL;
    parameters->best_fit_parameters         = NULL;
    parameters->telluriccorr_head           = NULL;
    parameters->telluriccorr_data           = NULL;
    parameters->best_fit_parameters_table   = NULL;
    parameters->results_lblrtm              = NULL;
    parameters->results_convolution         = NULL;
    parameters->mf_config                   = NULL;
    parameters->pl                          = cpl_propertylist_new();

    parameters->scale_pwv        = MOLECFIT_PARAMETER_CALCTRANS_SCALE_PWV_INIT;
    parameters->mjd_pwv          = MOLECFIT_PARAMETER_CALCTRANS_PWV_HDR_MJD_INIT;
    parameters->exp_pwv          = MOLECFIT_PARAMETER_CALCTRANS_PWV_HDR_EXP_INIT;
    parameters->air1_pwv         = MOLECFIT_PARAMETER_CALCTRANS_PWV_HDR_AIR1_INIT;
    parameters->air2_pwv         = MOLECFIT_PARAMETER_CALCTRANS_PWV_HDR_AIR2_INIT;
    parameters->pwv_ratio        = 1.0;
    parameters->pwv_sci          = -99.9;
    parameters->h2o_col_mm       = -99.9;
    parameters->sg_window_length = MOLECFIT_PARAMETER_CALCTRANS_SG_WINDOW_LENGTH_INIT;         //15
    parameters->sg_as_max_length = MOLECFIT_PARAMETER_CALCTRANS_SG_WINDOW_LENGTH_AS_MAX_INIT;  //CPL_FALSE

    cpl_msg_info(cpl_func, "molecfit_calctrans_parameter parameters mallocd %s", cpl_error_get_message());

    /* Use only primary extension in the input FITS file ? */
    p                                  = cpl_parameterlist_find_const(list, MOLECFIT_PARAMETER_USE_ONLY_INPUT_PRI);
    parameters->use_only_input_pri_ext = cpl_parameter_get_bool(p);
    cpl_propertylist_update_bool(
        parameters->pl, MF_PARAMETERS_CONTEX_DEFAULT " " MOLECFIT_PARAMETER_USE_ONLY_INPUT_PRI,
        parameters->use_only_input_pri_ext
    );

    cpl_msg_info(cpl_func, "MOLECFIT_PARAMETER_USE_ONLY_INPUT_PRI %s", cpl_error_get_message());

    /* If parameters->use_only_input_pri_ext == CPL_TRUE, you can provide a error flux extension */
    p                                = cpl_parameterlist_find_const(list, MOLECFIT_PARAMETER_DFLUX_EXTENSION_DATA);
    parameters->dflux_extension_data = cpl_parameter_get_int(p);
    cpl_propertylist_update_int(
        parameters->pl, MF_PARAMETERS_CONTEX_DEFAULT " " MOLECFIT_PARAMETER_DFLUX_EXTENSION_DATA,
        parameters->dflux_extension_data
    );

    cpl_msg_info(cpl_func, "MOLECFIT_PARAMETER_DFLUX_EXTENSION_DATA %s", cpl_error_get_message());

    /* If parameters->use_only_input_pri_ext == CPL_TRUE, you can provide a mask extension */
    p                               = cpl_parameterlist_find_const(list, MOLECFIT_PARAMETER_MASK_EXTENSION_DATA);
    parameters->mask_extension_data = cpl_parameter_get_int(p);
    cpl_propertylist_update_int(
        parameters->pl, MF_PARAMETERS_CONTEX_DEFAULT " " MOLECFIT_PARAMETER_MASK_EXTENSION_DATA,
        parameters->mask_extension_data
    );

    cpl_msg_info(cpl_func, "MOLECFIT_PARAMETER_MASK_EXTENSION_DATA %s", cpl_error_get_message());

    /* User input kernel ? */
    p                            = cpl_parameterlist_find_const(list, MOLECFIT_PARAMETER_USE_INPUT_KERNEL);
    parameters->use_input_kernel = cpl_parameter_get_bool(p);
    cpl_propertylist_update_bool(
        parameters->pl, MF_PARAMETERS_CONTEX_DEFAULT " " MOLECFIT_PARAMETER_USE_INPUT_KERNEL,
        parameters->use_input_kernel
    );

    cpl_msg_info(cpl_func, "MOLECFIT_PARAMETER_USE_INPUT_KERNEL %s", cpl_error_get_message());
    /* Mapping kernel */
    p                          = cpl_parameterlist_find_const(list, MOLECFIT_PARAMETER_CALCTRANS_MAPPING_KERNEL);
    parameters->mapping_kernel = cpl_parameter_get_string(p);
    cpl_propertylist_update_string(
        parameters->pl, MF_PARAMETERS_CONTEX_DEFAULT " " MOLECFIT_PARAMETER_CALCTRANS_MAPPING_KERNEL,
        parameters->mapping_kernel
    );
    cpl_msg_info(cpl_func, "MOLECFIT_PARAMETER_USE_INPUT_KERNEL 2 %s", cpl_error_get_message());
    /* Create mapping kernel cpl_table */
    if (strcmp(parameters->mapping_kernel, MF_PARAMETERS_NULL)) {
        cpl_msg_info(cpl_func, "MOLECFIT_PARAMETER_USE_INPUT_KERNEL 3 %s", cpl_error_get_message());
        parameters->mapping_kernel_table =
            mf_wrap_config_table_mapping(parameters->mapping_kernel, MOLECFIT_MAPPING_KERNEL_EXT);
    }
    cpl_msg_info(cpl_func, "MOLECFIT_PARAMETER_CALCTRANS_MAPPING_KERNEL %s", cpl_error_get_message());


    /* Scale PWV  */
    p = cpl_parameterlist_find_const(list, MOLECFIT_PARAMETER_CALCTRANS_SCALE_PWV);
    if (p) {
        parameters->scale_pwv = cpl_parameter_get_string(p);
        cpl_propertylist_update_string(
            parameters->pl, MF_PARAMETERS_CONTEX_DEFAULT " " MOLECFIT_PARAMETER_CALCTRANS_SCALE_PWV,
            parameters->scale_pwv
        );
    }

    /* Other header keywords used for airmass calculation. We do not ask for RA, DEC and LST as these are more standard. */

    p = cpl_parameterlist_find_const(list, MOLECFIT_PARAMETER_CALCTRANS_PWV_HDR_MJD);
    if (p) {
        parameters->mjd_pwv = cpl_parameter_get_string(p);
        cpl_propertylist_update_string(
            parameters->pl, MF_PARAMETERS_CONTEX_DEFAULT " " MOLECFIT_PARAMETER_CALCTRANS_PWV_HDR_MJD,
            parameters->mjd_pwv
        );
    }

    p = cpl_parameterlist_find_const(list, MOLECFIT_PARAMETER_CALCTRANS_PWV_HDR_EXP);
    if (p) {
        parameters->exp_pwv = cpl_parameter_get_string(p);
        cpl_propertylist_update_string(
            parameters->pl, MF_PARAMETERS_CONTEX_DEFAULT " " MOLECFIT_PARAMETER_CALCTRANS_PWV_HDR_EXP,
            parameters->exp_pwv
        );
    }

    p = cpl_parameterlist_find_const(list, MOLECFIT_PARAMETER_CALCTRANS_PWV_HDR_AIR1);
    if (p) {
        parameters->air1_pwv = cpl_parameter_get_string(p);
        cpl_propertylist_update_string(
            parameters->pl, MF_PARAMETERS_CONTEX_DEFAULT " " MOLECFIT_PARAMETER_CALCTRANS_PWV_HDR_AIR1,
            parameters->air1_pwv
        );
    }
    p = cpl_parameterlist_find_const(list, MOLECFIT_PARAMETER_CALCTRANS_PWV_HDR_AIR2);
    if (p) {
        parameters->air2_pwv = cpl_parameter_get_string(p);
        cpl_propertylist_update_string(
            parameters->pl, MF_PARAMETERS_CONTEX_DEFAULT " " MOLECFIT_PARAMETER_CALCTRANS_PWV_HDR_AIR2,
            parameters->air2_pwv
        );
    }

    /* Savitzky-Golay filter parameters */
    p = cpl_parameterlist_find_const(list, MOLECFIT_PARAMETER_CALCTRANS_SG_WINDOW_LENGTH);
    if (p) {
        parameters->sg_window_length = cpl_parameter_get_int(p);
        cpl_propertylist_update_int(
            parameters->pl, MF_PARAMETERS_CONTEX_DEFAULT " " MOLECFIT_PARAMETER_CALCTRANS_SG_WINDOW_LENGTH,
            parameters->sg_window_length
        );
    }

    p = cpl_parameterlist_find_const(list, MOLECFIT_PARAMETER_CALCTRANS_SG_WINDOW_LENGTH_AS_MAX);
    if (p) {
        parameters->sg_as_max_length = cpl_parameter_get_bool(p);
        cpl_propertylist_update_bool(
            parameters->pl, MF_PARAMETERS_CONTEX_DEFAULT " " MOLECFIT_PARAMETER_CALCTRANS_SG_WINDOW_LENGTH_AS_MAX,
            parameters->sg_as_max_length
        );
    }


    /* Mapping ATMOSPHERIC : SCIENCE_CALCTRANS/SCIENCE - ATM_PARAMETERS */
    p                               = cpl_parameterlist_find_const(list, MOLECFIT_PARAMETER_MAPPING_ATMOSPHERIC);
    parameters->mapping_atmospheric = cpl_parameter_get_string(p);
    cpl_msg_info(cpl_func, "parameters->mapping_atmospheric == %s", parameters->mapping_atmospheric);
    cpl_propertylist_update_string(
        parameters->pl, MF_PARAMETERS_CONTEX_DEFAULT " " MOLECFIT_PARAMETER_MAPPING_ATMOSPHERIC,
        parameters->mapping_atmospheric
    );
    /* Create mapping atmospheric cpl_table */
    if (strcmp(parameters->mapping_atmospheric, MF_PARAMETERS_NULL)) {
        cpl_msg_info(cpl_func, "parameters->mapping_atmospheric != NULL");
        parameters->mapping_atmospheric_table =
            mf_wrap_config_table_mapping(parameters->mapping_atmospheric, MOLECFIT_MAPPING_ATMOSPHERIC_EXT);
    }
    cpl_msg_info(cpl_func, "MOLECFIT_PARAMETER_MAPPING_ATMOSPHERIC %s", cpl_error_get_message());


    /* Mapping CONVOLVE : LBLRTM_RESULTS - TELLURIC_CORR */
    p                            = cpl_parameterlist_find_const(list, MOLECFIT_PARAMETER_MAPPING_CONVOLVE);
    parameters->mapping_convolve = cpl_parameter_get_string(p);
    cpl_propertylist_update_string(
        parameters->pl, MF_PARAMETERS_CONTEX_DEFAULT " " MOLECFIT_PARAMETER_MAPPING_CONVOLVE,
        parameters->mapping_convolve
    );
    /* Create mapping convolve cpl_table */
    if (strcmp(parameters->mapping_convolve, MF_PARAMETERS_NULL)) {
        parameters->mapping_convolve_table =
            mf_wrap_config_table_mapping(parameters->mapping_convolve, MOLECFIT_MAPPING_CONVOLVE_EXT);
    }
    cpl_msg_info(cpl_func, "MOLECFIT_PARAMETER_MAPPING_CONVOLVE %s", cpl_error_get_message());


    /* Load TAG = MOLECFIT_ATM_PARAMETERS */
    const char *input_tag = mf_wrap_tag_suffix(MOLECFIT_ATM_PARAMETERS, combined_suffix, CPL_FALSE);
    cpl_msg_info(cpl_func, "Load %s from molecfit_model(...)", input_tag);
    const cpl_frame *frm_atm_parameters = cpl_frameset_find_const(frameset, input_tag);
    if (frm_atm_parameters) {
        const char *filename = cpl_frame_get_filename(frm_atm_parameters);
        cpl_msg_info(cpl_func, "Loading %s from %s from molecfit_model(...)", filename, input_tag);
        parameters->atm_parameters = mf_wrap_fits_load(filename, CPL_FALSE);
    }
    cpl_msg_info(cpl_func, "MOLECFIT_ATM_PARAMETERS %s", cpl_error_get_message());

    /* Load TAG = MOLECFIT_BEST_FIT_PARAMETERS */
    input_tag = mf_wrap_tag_suffix(MOLECFIT_BEST_FIT_PARAMETERS, combined_suffix, CPL_FALSE);
    cpl_msg_info(cpl_func, "Load %s from molecfit_model(...)", input_tag);
    const cpl_frame *frm_best_fit_parameters = cpl_frameset_find_const(frameset, input_tag);
    if (frm_best_fit_parameters) {
        const char *filename = cpl_frame_get_filename(frm_best_fit_parameters);
        cpl_msg_info(cpl_func, "Loading %s from %s from molecfit_model(...)", filename, input_tag);
        parameters->best_fit_parameters = mf_wrap_fits_load(filename, CPL_FALSE);
    }
    cpl_msg_info(cpl_func, "MOLECFIT_BEST_FIT_PARAMETERS %s", cpl_error_get_message());

    /* Check if MOLECFIT_ATM_PARAMETERS and MOLECFIT_BEST_FIT_PARAMETERS were loaded */
    if (!(parameters->atm_parameters) || !(parameters->best_fit_parameters)) {
        //||   parameters->atm_parameters->n_ext  !=   parameters->best_fit_parameters->n_ext) {
        cpl_msg_info(cpl_func, "MOLECFIT_BEST_FIT_PARAMETERS 2 %s", cpl_error_get_message());
        if (!(parameters->atm_parameters)) {
            cpl_msg_info(cpl_func, "MOLECFIT_BEST_FIT_PARAMETERS 2.1");
        }
        if (!(parameters->best_fit_parameters)) {
            cpl_msg_info(cpl_func, "MOLECFIT_BEST_FIT_PARAMETERS 2.2");
        }
        if (parameters->atm_parameters->n_ext != parameters->best_fit_parameters->n_ext) {
            cpl_msg_info(cpl_func, "MOLECFIT_BEST_FIT_PARAMETERS 2.3");
        }


        mf_wrap_calctrans_parameter_delete(parameters);
        cpl_error_set_message(
            cpl_func, CPL_ERROR_ILLEGAL_INPUT, "molecfit_model output files not correct or not provided!"
        );
        return NULL;
    }
    //cpl_size n_ext = cpl_fits_count_extensions(science_fname);

    /* Reserve memory for the output results */
    parameters->n_ext                     = n_ext;
    parameters->telluriccorr_head         = cpl_calloc(parameters->n_ext, sizeof(cpl_table *));
    parameters->telluriccorr_data         = cpl_calloc(parameters->n_ext, sizeof(cpl_table *));
    parameters->best_fit_parameters_table = cpl_calloc(parameters->n_ext, sizeof(cpl_table *));
    parameters->results_lblrtm            = cpl_calloc(parameters->n_ext, sizeof(mf_calctrans_lblrtm_results *));
    parameters->results_convolution       = cpl_calloc(parameters->n_ext, sizeof(mf_calctrans_convolution_results *));
    for (cpl_size ext = 0; ext < n_ext; ext++) {
        parameters->telluriccorr_head[ext]         = NULL;
        parameters->telluriccorr_data[ext]         = NULL;
        parameters->best_fit_parameters_table[ext] = NULL;
        parameters->results_lblrtm[ext]            = NULL;
        parameters->results_convolution[ext]       = NULL;
    }
    cpl_msg_info(cpl_func, "setup memory for output results %s", cpl_error_get_message());

    //IS THIS CORRECT?
    //cpl_propertylist* scientific_head_pri = cpl_propertylist_load(science_fname,0);//parameters->best_fit_parameters->v_ext[0].header;
    //scientific_head_pri

    cpl_msg_info(cpl_func, "scientific_head_pri %s", cpl_error_get_message());
    /* Get recipe parameters in the recipe molecfit_model --> Best_fit_parameters */
    parameters->mf_config = mf_wrap_config_load_header_parameters(
        parameters->best_fit_parameters->v_ext[0].header, scientific_head_pri, parameters->pl
    );

    cpl_msg_info(cpl_func, "molecfit_config_load_header_parameters %s", cpl_error_get_message());

    /* Get recipe parameters in the recipe molecfit_model --> Best_fit_parameters */
    parameters->mf_config = mf_wrap_config_load_header_parameters(
        parameters->best_fit_parameters->v_ext[0].header, scientific_head_pri, parameters->pl
    );

    regex_t regex;
    int     reti;
    reti = regcomp(&regex, "[a-zA-Z0-9]", 0);
    if (reti) {
        cpl_error_set_message(
            cpl_func, CPL_ERROR_ILLEGAL_INPUT, "molecfit_calctrans: could not compile regex required to check SCALE_PWV"
        );
        return NULL;
    }
    reti = regexec(&regex, parameters->scale_pwv, 0, NULL, 0);

    /* This needs to be updated to work for all n_ext */
    int ridx;
    int nrows;
    if (n_ext > 0 && parameters->atm_parameters->v_ext[1].table != NULL) {
        nrows = cpl_table_get_nrow(parameters->best_fit_parameters->v_ext[1].table);
        for (ridx = 0; ridx < nrows; ridx++) {
            if (!strcmp(
                    cpl_table_get_string(parameters->best_fit_parameters->v_ext[1].table, "parameter", ridx),
                    "h2o_col_mm"
                )) {
                parameters->h2o_col_mm =
                    cpl_table_get_double(parameters->best_fit_parameters->v_ext[1].table, "value", ridx, NULL);
                break;
            }
        }
    }
    /*Do we need to throw an error here if we cannot read in h2o_col_mm ?*/

    /*Calculate pwv_sci */
    if (!strcmp(parameters->scale_pwv, "auto")) {
        if (cpl_propertylist_has(scientific_head_pri, "ESO TEL AMBI IWV START") &&
            cpl_propertylist_has(scientific_head_pri, "ESO TEL AMBI IWV END")) {
            double iwv_start = cpl_propertylist_get_double(scientific_head_pri, "ESO TEL AMBI IWV START");
            double iwv_end   = cpl_propertylist_get_double(scientific_head_pri, "ESO TEL AMBI IWV END");
            if (iwv_start == 0.0 && iwv_end == 0.0) {
                cpl_error_set_message(
                    cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                    "molecfit_calctrans: The ESO TEL AMBI IWV START/END science header keywords are both 0.0 which is "
                    "not permitted by SCALE_PWV=auto"
                );
                return NULL;
            }
            parameters->pwv_sci = 0.5 * (iwv_start + iwv_end);
            if (parameters->h2o_col_mm != -99.9 && parameters->h2o_col_mm != 0.0) {
                parameters->pwv_ratio = parameters->pwv_sci / parameters->h2o_col_mm;
            }
        }
        else {
            /* IWV data not in headers, raise error */
            cpl_error_set_message(
                cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                "molecfit_calctrans: Missing ESO TEL AMBI IWV START/END science header keywords required by "
                "SCALE_PWV=auto"
            );
            return NULL;
        }
    }
    //if the supplied value is a header keyword (i.e. it has characters in it) and not equal to none
    else if (!reti && strcmp(parameters->scale_pwv, "none")) {  // && parameters->scale_pwv has chars and != NONE) {
        if (cpl_propertylist_has(scientific_head_pri, parameters->scale_pwv)) {
            parameters->pwv_sci = cpl_propertylist_get_double(scientific_head_pri, parameters->scale_pwv);
            if (parameters->h2o_col_mm != -99.9 && parameters->h2o_col_mm != 0.0) {
                parameters->pwv_ratio = parameters->pwv_sci / parameters->h2o_col_mm;
            }
        }
        else {
            cpl_error_set_message(
                cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                "molecfit_calctrans: Missing science header keyword %s required by SCALE_PWV", parameters->scale_pwv
            );
            return NULL;
        }
    }
    else if (reti && strcmp(parameters->scale_pwv, "none")) {
        /* numerical value supplied, convert to float and use it */
        parameters->pwv_sci = atof(parameters->scale_pwv);
        if (parameters->h2o_col_mm != -99.9 && parameters->h2o_col_mm != 0.0) {
            parameters->pwv_ratio = parameters->pwv_sci / parameters->h2o_col_mm;
        }
    }
    else if (strcmp(parameters->scale_pwv, "none")) {
        cpl_error_set_message(
            cpl_func, CPL_ERROR_ILLEGAL_INPUT, "molecfit_calctrans: Error converting supplied value of SCALE_PWV: %s",
            parameters->scale_pwv
        );
        return NULL;
    }
    regfree(&regex);
    cpl_msg_info(
        cpl_func,
        "Value of parameters->pwv_sci: %e; parameters->h2o_col_mm (pwv_tell): %e; parameters->pwv_ratio: %e; "
        "parameters->scale_pwv: %s\n",
        parameters->pwv_sci, parameters->h2o_col_mm, parameters->pwv_ratio, parameters->scale_pwv
    );
    /* Rescale values in parameters->atm_parameters and write out to CALCTRANS_ATM_PARAMETERS.fits */
    if (n_ext > 0 && parameters->atm_parameters->v_ext[1].table != NULL &&
        cpl_table_has_column(parameters->atm_parameters->v_ext[1].table, "H2O")) {
        if (parameters->pwv_ratio == 1.0) {
            cpl_msg_warning(cpl_func, "molecfit_calctrans: pwv_ratio is 1.0. No scaling based on PWV performed.\n");
        }
        nrows = cpl_table_get_nrow(parameters->atm_parameters->v_ext[1].table);
        for (ridx = 0; ridx < nrows; ridx++) {
            double rval = cpl_table_get_double(parameters->atm_parameters->v_ext[1].table, "H2O", ridx, NULL) *
                          parameters->pwv_ratio;
            cpl_table_set_double(parameters->atm_parameters->v_ext[1].table, "H2O", ridx, rval);
        }
    }
    else {
        cpl_msg_warning(
            cpl_func, "molecfit_calctrans: ATM_PARAMETERS missing H2O column. No scaling based on PWV performed.\n"
        );
    }
    /*write out the new table to disk */
    /* This does not work on complex datasets - probably have to use:
   molecfit_save / molecfit_save_mf_results (as in molecfit_model)
   instead of molecfit_fits_write */

    if (strcmp(parameters->scale_pwv, "none")) {
        cpl_frameset *frameset_output = cpl_frameset_new();
        cpl_size      n_frames        = cpl_frameset_get_size(frameset);
        for (cpl_size frame = 0; frame < n_frames; frame++) {
            /* Check all inputs in frameset to get only the ATM_PARAMETERS frame*/
            cpl_frame  *frame_data = cpl_frameset_get_position(frameset, frame);
            const char *tag        = cpl_frame_get_tag(frame_data);
            if (!strcmp(tag, "ATM_PARAMETERS")) {
                cpl_frameset_insert(frameset_output, cpl_frame_duplicate(frame_data));
                mf_wrap_fits_write(
                    frameset, frameset_output, list, RECIPE_NAME, "ATM_PARAMETERS", parameters->atm_parameters,
                    "CALCTRANS_ATM_PARAMETERS.fits"
                );
                break;
            }
        }
    }

    /* Combine extensions in the FITS file ? */
    p                           = cpl_parameterlist_find_const(list, MOLECFIT_PARAMETER_CHIP_EXTENSIONS);
    parameters->chip_extensions = cpl_parameter_get_bool(p);
    cpl_propertylist_update_bool(
        parameters->pl, MF_PARAMETERS_CONTEX_DEFAULT " " MOLECFIT_PARAMETER_CHIP_EXTENSIONS, parameters->chip_extensions
    );


    /* Check status */
    if (!cpl_errorstate_is_equal(preState) || !(parameters->mf_config)) {
        /* Configuration failed */
        mf_wrap_calctrans_parameter_delete(parameters);
        return NULL;
    }

    return parameters;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief  delete the memory allocations that are allocated
 *         from calling molecfit_config_load_header_parameters.
 *         Note: this is primarily used for unit tests
 *
 * @param  mf_configuration     mf_configuration object created from a call
 *                              to molecfit_config_load_header_parameters
 *
 * @return void
 */
/*----------------------------------------------------------------------------*/
void mf_wrap_config_delete_configuration(mf_configuration *cfg)
{
    mf_configuration_delete(cfg);
}

/*----------------------------------------------------------------------------*/
/**
 * @brief  Parse output header and add to the mf_parameters_config and the header propertylist
 *
 * @param  parameters_header      Header with input parameters.
 * @param  data_header            Header with the data parameters.
 * @param  pl                     Output header.
 *
 * @return mf_configuration       Molecfit configuration
 */
/*----------------------------------------------------------------------------*/

mf_configuration *mf_wrap_config_load_header_parameters(
    const cpl_propertylist *parameters_header,
    const cpl_propertylist *data_header,
    cpl_propertylist       *pl
)
{
    /* Check input */
    cpl_error_ensure(parameters_header && data_header && pl, CPL_ERROR_NULL_INPUT, return NULL, "Input is NULL!");

    /* Get preState */
    cpl_errorstate pre_state = cpl_errorstate_get();
    const char    *p;


    /*** Get the molecfit parameters from the BEST_FIT_PARAMETERS primary header (cpl_propertylist *) ***/
    cpl_msg_info(cpl_func, "Loading input parameters saved in a output FITS header ...");

    /*** Molecfit default configuration ***/
    cpl_msg_info(cpl_func, "Load Molecfit default configuration ...");
    mf_configuration *mf_config = mf_configuration_create();
    if (!mf_config) {
        cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_OUTPUT, "Error loading Molecfit default configuration");
        return NULL;
    }

    /*** Set Molecfit configuration with recipe parameters ***/
    mf_parameters_directories  *directories  = &(mf_config->parameters->directories);
    mf_parameters_inputs       *inputs       = &(mf_config->parameters->inputs);
    mf_parameters_fitting      *fitting      = &(mf_config->parameters->fitting);
    mf_parameters_ambient      *ambient      = &(mf_config->parameters->ambient);
    mf_parameters_instrumental *instrumental = &(mf_config->parameters->instrumental);
    mf_parameters_atmospheric  *atmospheric  = &(mf_config->parameters->atmospheric);


    /*** Directories ***/
    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_TELLURICCORR_PATH;
    if (cpl_propertylist_has(parameters_header, p)) {
        if (directories->telluric_path) {
            cpl_free(directories->telluric_path);
        }
        directories->telluric_path = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
        cpl_propertylist_update_string(pl, p, directories->telluric_path);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_TELLURICCORR_DATA_PATH;
    if (cpl_propertylist_has(parameters_header, p)) {
        if (directories->telluriccorr_data_path) {
            cpl_free(directories->telluriccorr_data_path);
        }
        directories->telluriccorr_data_path = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
        cpl_propertylist_update_string(pl, p, directories->telluriccorr_data_path);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_TMP_PATH;
    if (cpl_propertylist_has(parameters_header, p)) {
        if (directories->tmp_path) {
            cpl_free(directories->tmp_path);
        }
        directories->tmp_path = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
        cpl_propertylist_update_string(pl, p, directories->tmp_path);
    }

    /*p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_OUTPUT_PATH;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (directories->output_path) cpl_free(directories->output_path);
      directories->output_path = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, directories->output_path);
  }*/

    /*p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_OUTPUT_NAME;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (directories->output_name) cpl_free(directories->output_name);
      directories->output_name = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, directories->output_name);
  }*/


    /*** Input parameters ***/
    /*p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_OMP_NUM_THREADS;
  if (cpl_propertylist_has(parameters_header, p)) {
      inputs->omp_num_threads = cpl_propertylist_get_int(parameters_header, p);
      cpl_propertylist_update_int(pl, p, inputs->omp_num_threads);
  }*/

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_SILENT_EXTERNAL_BINS;
    if (cpl_propertylist_has(parameters_header, p)) {
        inputs->silent_external_bins = cpl_propertylist_get_bool(parameters_header, p);
        cpl_propertylist_update_bool(pl, p, inputs->silent_external_bins);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_TRANSMISSION;
    if (cpl_propertylist_has(parameters_header, p)) {
        inputs->transmission = cpl_propertylist_get_bool(parameters_header, p);
        cpl_propertylist_update_bool(pl, p, inputs->transmission);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_COLUMN_LAMBDA;
    if (cpl_propertylist_has(parameters_header, p)) {
        if (inputs->column_lambda) {
            cpl_free(inputs->column_lambda);
        }
        inputs->column_lambda = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
        cpl_propertylist_update_string(pl, p, inputs->column_lambda);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_COLUMN_FLUX;
    if (cpl_propertylist_has(parameters_header, p)) {
        if (inputs->column_flux) {
            cpl_free(inputs->column_flux);
        }
        inputs->column_flux = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
        cpl_propertylist_update_string(pl, p, inputs->column_flux);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_COLUMN_DFLUX;
    if (cpl_propertylist_has(parameters_header, p)) {
        if (inputs->column_dflux) {
            cpl_free(inputs->column_dflux);
        }
        inputs->column_dflux = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
        cpl_propertylist_update_string(pl, p, inputs->column_dflux);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_COLUMN_MASK;
    if (cpl_propertylist_has(parameters_header, p)) {
        if (inputs->column_mask) {
            cpl_free(inputs->column_mask);
        }
        inputs->column_mask = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
        cpl_propertylist_update_string(pl, p, inputs->column_mask);
    }


    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_DEFAULT_ERROR;
    if (cpl_propertylist_has(parameters_header, p)) {
        inputs->default_error = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, inputs->default_error);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_WLG_TO_MICRON;
    if (cpl_propertylist_has(parameters_header, p)) {
        inputs->wlg_to_micron = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, inputs->wlg_to_micron);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_WAVELENGTH_FRAME;
    if (cpl_propertylist_has(parameters_header, p)) {
        if (inputs->wavelengths_frame) {
            cpl_free(inputs->wavelengths_frame);
        }
        inputs->wavelengths_frame = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
        cpl_propertylist_update_string(pl, p, inputs->wavelengths_frame);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_OBSERVATORY_ERF_RV_KEYWORD;
    if (cpl_propertylist_has(parameters_header, p)) {
        if (inputs->observing_erv_rv.key) {
            cpl_free(inputs->observing_erv_rv.key);
        }
        inputs->observing_erv_rv.key = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
        cpl_propertylist_update_string(pl, p, inputs->observing_erv_rv.key);
    }
    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_OBSERVATORY_ERF_RV_VALUE;
    if (cpl_propertylist_has(parameters_header, p)) {
        inputs->observing_erv_rv.value = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, inputs->observing_erv_rv.value);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_CLEAN_MODEL_FLUX;
    if (cpl_propertylist_has(parameters_header, p)) {
        inputs->clean_flux = cpl_propertylist_get_bool(parameters_header, p);
        cpl_propertylist_update_bool(pl, p, inputs->clean_flux);
    }


    /*** Convergence criterion ***/
    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_FTOL;
    if (cpl_propertylist_has(parameters_header, p)) {
        fitting->ftol = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, fitting->ftol);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_XTOL;
    if (cpl_propertylist_has(parameters_header, p)) {
        fitting->xtol = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, fitting->xtol);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_FLUX_UNIT;
    if (cpl_propertylist_has(parameters_header, p)) {
        fitting->flux_unit = cpl_propertylist_get_int(parameters_header, p);
        cpl_propertylist_update_int(pl, p, fitting->flux_unit);
    }


    /*** Telescope background ***/
    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_FIT_TELESCOPE_BACK;
    if (cpl_propertylist_has(parameters_header, p)) {
        fitting->fit_telescope_background.fit = cpl_propertylist_get_bool(parameters_header, p);
        cpl_propertylist_update_bool(pl, p, fitting->fit_telescope_background.fit);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_TELESCOPE_BACK_CONST;
    if (cpl_propertylist_has(parameters_header, p)) {
        fitting->fit_telescope_background.const_val = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_int(pl, p, fitting->fit_telescope_background.const_val);
    }


    /*** Continuum ***/
    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_FIT_CONTINUUM;
    if (cpl_propertylist_has(parameters_header, p)) {
        fitting->fit_continuum.fit = cpl_propertylist_get_bool(parameters_header, p);
        cpl_propertylist_update_bool(pl, p, fitting->fit_continuum.fit);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_CONTINUUM_N;
    if (cpl_propertylist_has(parameters_header, p)) {
        fitting->fit_continuum.n = cpl_propertylist_get_int(parameters_header, p);
        cpl_propertylist_update_int(pl, p, fitting->fit_continuum.n);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_CONTINUUM_CONST;
    if (cpl_propertylist_has(parameters_header, p)) {
        fitting->fit_continuum.const_val = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, fitting->fit_continuum.const_val);
    }

    /*p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_OBSERVATORY_BARY_RV;
  if (cpl_propertylist_has(parameters_header, p)) {
      fitting->obs_bary_rv = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, fitting->obs_bary_rv);
  }
  */


    /*** Wavelength solution fit/adjustment ***/
    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_FIT_WLC;
    if (cpl_propertylist_has(parameters_header, p)) {
        fitting->fit_wavelenght.fit = cpl_propertylist_get_bool(parameters_header, p);
        cpl_propertylist_update_bool(pl, p, fitting->fit_wavelenght.fit);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_WLC_N;
    if (cpl_propertylist_has(parameters_header, p)) {
        fitting->fit_wavelenght.n = cpl_propertylist_get_int(parameters_header, p);
        cpl_propertylist_update_int(pl, p, fitting->fit_wavelenght.n);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_WLC_CONST;
    if (cpl_propertylist_has(parameters_header, p)) {
        fitting->fit_wavelenght.const_val = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, fitting->fit_wavelenght.const_val);
    }

    /* p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_WLC_REF;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (fitting->wlc_ref) cpl_free(fitting->wlc_ref);
      fitting->wlc_ref = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, fitting->wlc_ref);
  } */


    /*** Default kernel: Boxcar kernel ***/
    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_FIT_RES_BOX;
    if (cpl_propertylist_has(parameters_header, p)) {
        fitting->fit_res_box.fit = cpl_propertylist_get_bool(parameters_header, p);
        cpl_propertylist_update_bool(pl, p, fitting->fit_res_box.fit);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_RES_BOX;
    if (cpl_propertylist_has(parameters_header, p)) {
        fitting->fit_res_box.const_val = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, fitting->fit_res_box.const_val);
    }


    /*** Default kernel: Gaussian kernel ***/
    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_FIT_GAUSS;
    if (cpl_propertylist_has(parameters_header, p)) {
        fitting->fit_gauss.fit = cpl_propertylist_get_bool(parameters_header, p);
        cpl_propertylist_update_bool(pl, p, fitting->fit_gauss.fit);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_RES_GAUSS;
    if (cpl_propertylist_has(parameters_header, p)) {
        fitting->fit_gauss.const_val = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, fitting->fit_gauss.const_val);
    }


    /*** Default kernel: Lorentz kernel ***/
    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_FIT_LORENTZ;
    if (cpl_propertylist_has(parameters_header, p)) {
        fitting->fit_lorentz.fit = cpl_propertylist_get_bool(parameters_header, p);
        cpl_propertylist_update_bool(pl, p, fitting->fit_lorentz.fit);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_RES_LORENTZ;
    if (cpl_propertylist_has(parameters_header, p)) {
        fitting->fit_lorentz.const_val = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, fitting->fit_lorentz.const_val);
    }


    /*** Default kernels: Generic parameters ***/
    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_KERN_MODE;
    if (cpl_propertylist_has(parameters_header, p)) {
        fitting->kern_mode = cpl_propertylist_get_bool(parameters_header, p);
        cpl_propertylist_update_bool(pl, p, fitting->kern_mode);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_KERN_FAC;
    if (cpl_propertylist_has(parameters_header, p)) {
        fitting->kern_fac = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, fitting->kern_fac);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_VAR_KERN;
    if (cpl_propertylist_has(parameters_header, p)) {
        fitting->var_kern = cpl_propertylist_get_bool(parameters_header, p);
        cpl_propertylist_update_bool(pl, p, fitting->var_kern);
    }


    /*** Ambient ***/
    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_OBSERVING_DATE_KEYWORD;
    if (cpl_propertylist_has(parameters_header, p)) {
        if (ambient->observing_date.key) {
            cpl_free(ambient->observing_date.key);
        }
        ambient->observing_date.key = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
        cpl_propertylist_update_string(pl, p, ambient->observing_date.key);
    }
    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_OBSERVING_DATE_VALUE;
    if (cpl_propertylist_has(parameters_header, p)) {
        ambient->observing_date.value = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, ambient->observing_date.value);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_UTC_KEYWORD;
    if (cpl_propertylist_has(parameters_header, p)) {
        if (ambient->utc.key) {
            cpl_free(ambient->utc.key);
        }
        ambient->utc.key = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
        cpl_propertylist_update_string(pl, p, ambient->utc.key);
    }
    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_UTC_VALUE;
    if (cpl_propertylist_has(parameters_header, p)) {
        ambient->utc.value = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, ambient->utc.value);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_TELESCOPE_ANGLE_KEYWORD;
    if (cpl_propertylist_has(parameters_header, p)) {
        if (ambient->telescope_angle.key) {
            cpl_free(ambient->telescope_angle.key);
        }
        ambient->telescope_angle.key = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
        cpl_propertylist_update_string(pl, p, ambient->telescope_angle.key);
    }
    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_TELESCOPE_ANGLE_VALUE;
    if (cpl_propertylist_has(parameters_header, p)) {
        ambient->telescope_angle.value = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, ambient->telescope_angle.value);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_RELATIVE_HUMIDITY_KEYWORD;
    if (cpl_propertylist_has(parameters_header, p)) {
        if (ambient->relative_humidity.key) {
            cpl_free(ambient->relative_humidity.key);
        }
        ambient->relative_humidity.key = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
        cpl_propertylist_update_string(pl, p, ambient->relative_humidity.key);
    }
    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_RELATIVE_HUMIDITY_VALUE;
    if (cpl_propertylist_has(parameters_header, p)) {
        ambient->relative_humidity.value = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, ambient->relative_humidity.value);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_PRESSURE_KEYWORD;
    if (cpl_propertylist_has(parameters_header, p)) {
        if (ambient->pressure.key) {
            cpl_free(ambient->pressure.key);
        }
        ambient->pressure.key = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
        cpl_propertylist_update_string(pl, p, ambient->pressure.key);
    }
    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_PRESSURE_VALUE;
    if (cpl_propertylist_has(parameters_header, p)) {
        ambient->pressure.value = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, ambient->pressure.value);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_TEMPERATURE_KEYWORD;
    if (cpl_propertylist_has(parameters_header, p)) {
        if (ambient->temperature.key) {
            cpl_free(ambient->temperature.key);
        }
        ambient->temperature.key = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
        cpl_propertylist_update_string(pl, p, ambient->temperature.key);
    }
    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_TEMPERATURE_VALUE;
    if (cpl_propertylist_has(parameters_header, p)) {
        ambient->temperature.value = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, ambient->temperature.value);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_MIRROR_TEMPERATURE_KEYWORD;
    if (cpl_propertylist_has(parameters_header, p)) {
        if (ambient->mirror_temperature.key) {
            cpl_free(ambient->mirror_temperature.key);
        }
        ambient->mirror_temperature.key = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
        cpl_propertylist_update_string(pl, p, ambient->mirror_temperature.key);
    }
    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_MIRROR_TEMPERATURE_VALUE;
    if (cpl_propertylist_has(parameters_header, p)) {
        ambient->mirror_temperature.value = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, ambient->mirror_temperature.value);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_ELEVATION_KEYWORD;
    if (cpl_propertylist_has(parameters_header, p)) {
        if (ambient->elevation.key) {
            cpl_free(ambient->elevation.key);
        }
        ambient->elevation.key = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
        cpl_propertylist_update_string(pl, p, ambient->elevation.key);
    }
    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_ELEVATION_VALUE;
    if (cpl_propertylist_has(parameters_header, p)) {
        ambient->elevation.value = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, ambient->elevation.value);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_LONGITUDE_KEYWORD;
    if (cpl_propertylist_has(parameters_header, p)) {
        if (ambient->longitude.key) {
            cpl_free(ambient->longitude.key);
        }
        ambient->longitude.key = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
        cpl_propertylist_update_string(pl, p, ambient->longitude.key);
    }
    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_LONGITUDE_VALUE;
    if (cpl_propertylist_has(parameters_header, p)) {
        ambient->longitude.value = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, ambient->longitude.value);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_LATITUDE_KEYWORD;
    if (cpl_propertylist_has(parameters_header, p)) {
        if (ambient->latitude.key) {
            cpl_free(ambient->latitude.key);
        }
        ambient->latitude.key = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
        cpl_propertylist_update_string(pl, p, ambient->latitude.key);
    }
    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_LATITUDE_VALUE;
    if (cpl_propertylist_has(parameters_header, p)) {
        ambient->latitude.value = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, ambient->latitude.value);
    }


    /*** Instrumental ***/
    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_SLIT_WIDTH_KEYWORD;
    if (cpl_propertylist_has(parameters_header, p)) {
        if (instrumental->slit_width.key) {
            cpl_free(instrumental->slit_width.key);
        }
        instrumental->slit_width.key = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
        cpl_propertylist_update_string(pl, p, instrumental->slit_width.key);
    }
    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_SLIT_WIDTH_VALUE;
    if (cpl_propertylist_has(parameters_header, p)) {
        instrumental->slit_width.value = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, instrumental->slit_width.value);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_PIXEL_SCALE_KEYWORD;
    if (cpl_propertylist_has(parameters_header, p)) {
        if (instrumental->pixel_scale.key) {
            cpl_free(instrumental->pixel_scale.key);
        }
        instrumental->pixel_scale.key = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
        cpl_propertylist_update_string(pl, p, instrumental->pixel_scale.key);
    }
    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_PIXEL_SCALE_VALUE;
    if (cpl_propertylist_has(parameters_header, p)) {
        instrumental->pixel_scale.value = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, instrumental->pixel_scale.value);
    }


    /*** Atmospheric ***/
    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_REFERENCE_ATMOSPHERIC;
    if (cpl_propertylist_has(parameters_header, p)) {
        if (atmospheric->ref_atm) {
            cpl_free(atmospheric->ref_atm);
        }
        atmospheric->ref_atm = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
        cpl_propertylist_update_string(pl, p, atmospheric->ref_atm);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_GDAS_PROFILE;
    if (cpl_propertylist_has(parameters_header, p)) {
        if (atmospheric->gdas_prof) {
            cpl_free(atmospheric->gdas_prof);
        }
        atmospheric->gdas_prof = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
        cpl_propertylist_update_string(pl, p, atmospheric->gdas_prof);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_LAYERS;
    if (cpl_propertylist_has(parameters_header, p)) {
        atmospheric->layers = cpl_propertylist_get_bool(parameters_header, p);
        cpl_propertylist_update_bool(pl, p, atmospheric->layers);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_EMIX;
    if (cpl_propertylist_has(parameters_header, p)) {
        atmospheric->emix = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, atmospheric->emix);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_PWV;
    if (cpl_propertylist_has(parameters_header, p)) {
        atmospheric->pwv = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, atmospheric->pwv);
    }


    /*** LNFL ***/
    mf_io_lnfl_config *lnfl = mf_config->lnfl;

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_LNFL_LINE_DB;
    if (cpl_propertylist_has(parameters_header, p)) {
        if (lnfl->line_db) {
            cpl_free(lnfl->line_db);
        }
        lnfl->line_db = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
        cpl_propertylist_update_string(pl, p, lnfl->line_db);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_LNFL_LINE_DB_FMT;
    if (cpl_propertylist_has(parameters_header, p)) {
        lnfl->line_db_fmt = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, lnfl->line_db_fmt);
    }


    /*** LBLRTM ***/
    mf_io_lblrtm_config *lblrtm = mf_config->lblrtm;

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_LBLRTM_ICNTNM;
    if (cpl_propertylist_has(parameters_header, p)) {
        lblrtm->icntnm = cpl_propertylist_get_int(parameters_header, p);
        cpl_propertylist_update_int(pl, p, lblrtm->icntnm);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_LBLRTM_IAERSL;
    if (cpl_propertylist_has(parameters_header, p)) {
        lblrtm->iaersl = cpl_propertylist_get_int(parameters_header, p);
        cpl_propertylist_update_int(pl, p, lblrtm->iaersl);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_LBLRTM_MPTS;
    if (cpl_propertylist_has(parameters_header, p)) {
        lblrtm->mpts = cpl_propertylist_get_int(parameters_header, p);
        cpl_propertylist_update_int(pl, p, lblrtm->mpts);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_LBLRTM_NPTS;
    if (cpl_propertylist_has(parameters_header, p)) {
        lblrtm->npts = cpl_propertylist_get_int(parameters_header, p);
        cpl_propertylist_update_int(pl, p, lblrtm->npts);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_LBLRTM_V1;
    if (cpl_propertylist_has(parameters_header, p)) {
        lblrtm->v[0] = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, lblrtm->v[0]);
    }
    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_LBLRTM_V2;
    if (cpl_propertylist_has(parameters_header, p)) {
        lblrtm->v[1] = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, lblrtm->v[1]);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_LBLRTM_SAMPLE;
    if (cpl_propertylist_has(parameters_header, p)) {
        lblrtm->sample = cpl_propertylist_get_int(parameters_header, p);
        cpl_propertylist_update_int(pl, p, lblrtm->sample);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_LBLRTM_ALFAL0;
    if (cpl_propertylist_has(parameters_header, p)) {
        lblrtm->alfal0 = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, lblrtm->alfal0);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_LBLRTM_AVMASS;
    if (cpl_propertylist_has(parameters_header, p)) {
        lblrtm->avmass = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, lblrtm->avmass);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_LBLRTM_DPTMIN;
    if (cpl_propertylist_has(parameters_header, p)) {
        lblrtm->dptmin = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, lblrtm->dptmin);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_LBLRTM_DPTFAC;
    if (cpl_propertylist_has(parameters_header, p)) {
        lblrtm->dptfac = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, lblrtm->dptfac);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_LBLRTM_TBOUND;
    if (cpl_propertylist_has(parameters_header, p)) {
        lblrtm->tbound = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, lblrtm->tbound);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_LBLRTM_SREMIS1;
    if (cpl_propertylist_has(parameters_header, p)) {
        lblrtm->sremis[0] = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, lblrtm->sremis[0]);
    }
    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_LBLRTM_SREMIS2;
    if (cpl_propertylist_has(parameters_header, p)) {
        lblrtm->sremis[1] = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, lblrtm->sremis[1]);
    }
    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_LBLRTM_SREMIS3;
    if (cpl_propertylist_has(parameters_header, p)) {
        lblrtm->sremis[2] = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, lblrtm->sremis[2]);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_LBLRTM_SRREFL1;
    if (cpl_propertylist_has(parameters_header, p)) {
        lblrtm->srrefl[0] = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, lblrtm->srrefl[0]);
    }
    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_LBLRTM_SRREFL2;
    if (cpl_propertylist_has(parameters_header, p)) {
        lblrtm->srrefl[1] = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, lblrtm->srrefl[1]);
    }
    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_LBLRTM_SRREFL3;
    if (cpl_propertylist_has(parameters_header, p)) {
        lblrtm->srrefl[2] = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, lblrtm->srrefl[2]);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_LBLRTM_MODEL;
    if (cpl_propertylist_has(parameters_header, p)) {
        lblrtm->model = cpl_propertylist_get_int(parameters_header, p);
        cpl_propertylist_update_int(pl, p, lblrtm->model);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_LBLRTM_ITYPE;
    if (cpl_propertylist_has(parameters_header, p)) {
        lblrtm->itype = cpl_propertylist_get_int(parameters_header, p);
        cpl_propertylist_update_int(pl, p, lblrtm->itype);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_LBLRTM_NOZERO;
    if (cpl_propertylist_has(parameters_header, p)) {
        lblrtm->nozero = cpl_propertylist_get_int(parameters_header, p);
        cpl_propertylist_update_int(pl, p, lblrtm->nozero);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_LBLRTM_NOPRNT;
    if (cpl_propertylist_has(parameters_header, p)) {
        lblrtm->noprnt = cpl_propertylist_get_int(parameters_header, p);
        cpl_propertylist_update_int(pl, p, lblrtm->noprnt);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_LBLRTM_IPUNCH;
    if (cpl_propertylist_has(parameters_header, p)) {
        lblrtm->ipunch = cpl_propertylist_get_int(parameters_header, p);
        cpl_propertylist_update_int(pl, p, lblrtm->ipunch);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_LBLRTM_RE;
    if (cpl_propertylist_has(parameters_header, p)) {
        lblrtm->re = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, lblrtm->re);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_LBLRTM_HSPACE;
    if (cpl_propertylist_has(parameters_header, p)) {
        lblrtm->hspace = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, lblrtm->hspace);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT
        " " MF_PARAMETERS_LATITUDE_VALUE;  // MF_LBLRTM_REF_LAT;   // MF_PARAMETERS_LATITUDE_VALUE
    if (cpl_propertylist_has(parameters_header, p)) {
        lblrtm->ref_lat = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, lblrtm->ref_lat);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_PARAMETERS_ELEVATION_VALUE;  // MF_LBLRTM_H1;
    if (cpl_propertylist_has(parameters_header, p)) {
        lblrtm->h[0] = ambient->elevation.value / 1000.00;  // cpl_propertylist_get_double(parameters_header, p);
        /*cpl_propertylist_update_double(pl, p, lblrtm->h[0]);*/
    }
    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_LBLRTM_H2;
    if (cpl_propertylist_has(parameters_header, p)) {
        lblrtm->h[1] = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, lblrtm->h[1]);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_LBLRTM_RANGE;
    if (cpl_propertylist_has(parameters_header, p)) {
        lblrtm->range = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, lblrtm->range);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_LBLRTM_BETA;
    if (cpl_propertylist_has(parameters_header, p)) {
        lblrtm->beta = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, lblrtm->beta);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_LBLRTM_LEN;
    if (cpl_propertylist_has(parameters_header, p)) {
        lblrtm->len = cpl_propertylist_get_int(parameters_header, p);
        cpl_propertylist_update_int(pl, p, lblrtm->len);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_LBLRTM_HOBS;
    if (cpl_propertylist_has(parameters_header, p)) {
        lblrtm->hobs = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, lblrtm->hobs);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_LBLRTM_AVTRAT;
    if (cpl_propertylist_has(parameters_header, p)) {
        lblrtm->avtrat = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, lblrtm->avtrat);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_LBLRTM_TDIFF1;
    if (cpl_propertylist_has(parameters_header, p)) {
        lblrtm->tdiff[0] = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, lblrtm->tdiff[0]);
    }
    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_LBLRTM_TDIFF2;
    if (cpl_propertylist_has(parameters_header, p)) {
        lblrtm->tdiff[1] = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, lblrtm->tdiff[1]);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_LBLRTM_ALTD1;
    if (cpl_propertylist_has(parameters_header, p)) {
        lblrtm->altd[0] = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, lblrtm->altd[0]);
    }
    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_LBLRTM_ALTD2;
    if (cpl_propertylist_has(parameters_header, p)) {
        lblrtm->altd[1] = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, lblrtm->altd[1]);
    }

    p = MF_PARAMETERS_CONTEX_DEFAULT " " MF_LBLRTM_DELV;
    if (cpl_propertylist_has(parameters_header, p)) {
        lblrtm->delv = cpl_propertylist_get_double(parameters_header, p);
        cpl_propertylist_update_double(pl, p, lblrtm->delv);
    }


    /*** Update the Molecfit configuration with the header ***/
    cpl_error_code e = mf_parameters_config_update_with_header_keywords(mf_config->parameters, data_header);


    cpl_msg_info(
        cpl_func, "Columns : wave[%s], flux[%s], dflux[%s], mask[%s]", inputs->column_lambda, inputs->column_flux,
        inputs->column_dflux, inputs->column_mask
    );


    /* Check possible errors */
    if (!cpl_errorstate_is_equal(pre_state) || e != CPL_ERROR_NONE) {
        mf_configuration_delete(mf_config);
        cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_OUTPUT, "Error loading Molecfit default configuration");
        return NULL;
    }

    return mf_config;
}

cpl_vector *convol(cpl_vector *v, cpl_vector *kernel)
{
    /* Code taken from cpl_vector_filter_lowpass_create */
    int         nv       = cpl_vector_get_size(v);
    int         nk       = cpl_vector_get_size(kernel);
    cpl_vector *filtered = cpl_vector_new(nv);
    int         hw       = floor(nk / 2);
    double      replace;
    cpl_size    i, j;

    /* compute edge effects for the first hw elements */
    for (i = 0; i < hw; i++) {
        replace = 0.0;
        for (j = -hw; j <= hw; j++) {
            if (i + j < 0) {
                replace += cpl_vector_get(kernel, hw + j) * cpl_vector_get(v, 0);
            }
            else {
                replace += cpl_vector_get(kernel, hw + j) * cpl_vector_get(v, i + j);
            }
        }
        cpl_vector_set(filtered, i, replace);
    }

    /* compute edge effects for the last hw elements */
    for (i = nv - hw; i < nv; i++) {
        replace = 0.0;
        for (j = -hw; j <= hw; j++) {
            if (i + j > nv - 1) {
                replace += cpl_vector_get(kernel, hw + j) * cpl_vector_get(v, nv - 1);
            }
            else {
                replace += cpl_vector_get(kernel, hw + j) * cpl_vector_get(v, i + j);
            }
        }
        cpl_vector_set(filtered, i, replace);
    }

    /* compute all other elements */
    for (i = hw; i < nv - hw; i++) {
        replace = 0.0;
        for (j = -hw; j <= hw; j++) {
            replace += cpl_vector_get(kernel, hw + j) * cpl_vector_get(v, i + j);
        }
        cpl_vector_set(filtered, i, replace);
    }
    //cpl_vector_delete(kernel);
    //cpl_tools_add_flops(4 * hw * nv);
    return filtered;
}

cpl_vector *calc_rms(
    cpl_table *t,
    int        wlength,
    int        nrows_inc,
    double    *w1_inc,
    double    *w2_inc,
    int        nrows_exc,
    double    *w1_exc,
    double    *w2_exc,
    double    *fres,
    double    *cres,
    double    *sres
)
{
    //1. Duplicate tdata
    //2. Add columns for smoothed data using Savitzky-Golay filter with window_length=wlength
    //3. Filter table based on *inc/*exc data
    //4. Calculate statistics and populate the results
    cpl_table *tdata      = cpl_table_duplicate(t);
    int        tdata_rows = cpl_table_get_nrow(tdata);
    //calculate smoothed versions of the columns flux and cflux
    cpl_msg_info(cpl_func, "calc_rms: Calculating Savitzky-Golay coefficients for window_length %d\n", wlength);
    cpl_vector *coeffs = savgol(floor(wlength / 2), floor(wlength / 2), 0, 3);

    /*before we do anything, we need to remove the bad quality (null rows), since these can introduce nan values */
    int *qual_col_orig = cpl_table_get_data_int(tdata, MF_COL_OUT_MASK);
    cpl_table_unselect_all(tdata);
    for (int i = 0; i < tdata_rows; i++) {
        if (qual_col_orig[i] == 1) {
            cpl_table_select_row(tdata, i);
        }
    }

    tdata      = cpl_table_extract_selected(tdata);
    tdata_rows = cpl_table_get_nrow(tdata);
    //lambda == MF_COL_IN_LAMBDA, mlambda == MF_COL_MOD_LAMBDA
    double     *wave_col   = cpl_table_get_data_double(tdata, MF_COL_MOD_LAMBDA);
    double     *flux_col   = cpl_table_get_data_double(tdata, MF_COL_IN_FLUX);
    double     *sflux_col  = cpl_table_get_data_double(tdata, MF_COL_MOD_WEIGHT);
    double     *cflux_col  = cpl_table_get_data_double(tdata, MF_COL_OUT_FLUX);
    cpl_vector *flux       = cpl_vector_wrap(tdata_rows, flux_col);
    cpl_vector *sflux      = cpl_vector_wrap(tdata_rows, sflux_col);
    cpl_vector *cflux      = cpl_vector_wrap(tdata_rows, cflux_col);
    cpl_vector *flux_sm    = convol(flux, coeffs);
    cpl_vector *cflux_sm   = convol(cflux, coeffs);
    cpl_vector *flux_norm  = cpl_vector_new(tdata_rows);
    cpl_vector *cflux_norm = cpl_vector_new(tdata_rows);
    cpl_vector *sflux_norm = cpl_vector_new(tdata_rows);
    cpl_msg_info(cpl_func, "calc_rms: Calculating normalised flux columns");
    for (int i = 0; i < tdata_rows; i++) {
        cpl_vector_set(flux_norm, i, 1.0 - (cpl_vector_get(flux, i) / cpl_vector_get(flux_sm, i)));
        cpl_vector_set(cflux_norm, i, 1.0 - (cpl_vector_get(cflux, i) / cpl_vector_get(cflux_sm, i)));
        cpl_vector_set(sflux_norm, i, (cpl_vector_get(sflux, i) / cpl_vector_get(flux_sm, i)));
    }

    //calculate the normalised columns

    //populate the new columns
    //(add the new columns)
    cpl_table_new_column(tdata, "flux_sm", CPL_TYPE_DOUBLE);
    cpl_table_new_column(tdata, "cflux_sm", CPL_TYPE_DOUBLE);
    cpl_table_new_column(tdata, "flux_norm", CPL_TYPE_DOUBLE);
    cpl_table_new_column(tdata, "cflux_norm", CPL_TYPE_DOUBLE);
    cpl_table_new_column(tdata, "sflux_norm", CPL_TYPE_DOUBLE);
    cpl_table_copy_data_double(tdata, "flux_sm", cpl_vector_get_data(flux_sm));
    cpl_table_copy_data_double(tdata, "cflux_sm", cpl_vector_get_data(cflux_sm));
    cpl_table_copy_data_double(tdata, "flux_norm", cpl_vector_get_data(flux_norm));
    cpl_table_copy_data_double(tdata, "cflux_norm", cpl_vector_get_data(cflux_norm));
    cpl_table_copy_data_double(tdata, "sflux_norm", cpl_vector_get_data(sflux_norm));
    //(fill the new columns)

    //filter the table...
    cpl_msg_info(cpl_func, "calc_rms: Filtering results table");
    int *qual_col = cpl_table_get_data_int(tdata, MF_COL_OUT_MASK);
    cpl_table_unselect_all(tdata);
    for (int i = 0; i < tdata_rows; i++) {
        //cpl_msg_info(cpl_func,"qual_col[%d] = %d\n",i,qual_col[i]);
        if (qual_col[i] == 1) {
            cpl_boolean sel_row = CPL_FALSE;
            //first select the row if it lies in an inclusion range
            if (nrows_inc != -1) {
                for (int j = 0; j < nrows_inc; j++) {
                    if (wave_col[i] >= w1_inc[j] && wave_col[i] <= w2_inc[j]) {
                        sel_row = CPL_TRUE;
                    }
                }
            }
            //then exclude any rows if they lie in an exclusion range
            if (nrows_exc != -1) {
                for (int j = 0; j < nrows_exc; j++) {
                    if (wave_col[i] >= w1_exc[j] && wave_col[i] <= w2_exc[j]) {
                        sel_row = CPL_FALSE;
                    }
                }
            }
            if (nrows_inc == -1 && nrows_exc == -1) {
                sel_row = CPL_TRUE;
            }
            //if the row was selected, actually select it
            if (sel_row == CPL_TRUE) {
                cpl_table_select_row(tdata, i);
            }
        }
    }
    tdata      = cpl_table_extract_selected(tdata);
    tdata_rows = cpl_table_get_nrow(tdata);

    /* Calculate statistics from the cleaned up table */
    cpl_msg_info(cpl_func, "calc_rms: Calculating statistics on filtered results table");

    //get the table data
    double     *fnorm_col  = cpl_table_get_data_double(tdata, "flux_norm");
    double     *cfnorm_col = cpl_table_get_data_double(tdata, "cflux_norm");
    double     *sfnorm_col = cpl_table_get_data_double(tdata, "sflux_norm");
    cpl_vector *fnorm      = cpl_vector_wrap(tdata_rows, fnorm_col);
    cpl_vector *cfnorm     = cpl_vector_wrap(tdata_rows, cfnorm_col);
    cpl_vector *sfnorm     = cpl_vector_wrap(tdata_rows, sfnorm_col);

    //now the calculations
    double flux_norm_mean  = cpl_vector_get_mean(fnorm);
    double cflux_norm_mean = cpl_vector_get_mean(cfnorm);
    double sflux_norm_mean = cpl_vector_get_mean(sfnorm);

    cpl_vector *flux_norm_res  = cpl_vector_new(tdata_rows);
    cpl_vector *cflux_norm_res = cpl_vector_new(tdata_rows);
    cpl_vector *sflux_norm_res = cpl_vector_new(tdata_rows);
    for (int i = 0; i < tdata_rows; i++) {
        cpl_vector_set(flux_norm_res, i, fabs(cpl_vector_get(flux_norm, i) - flux_norm_mean));
        cpl_vector_set(cflux_norm_res, i, fabs(cpl_vector_get(cflux_norm, i) - cflux_norm_mean));
        cpl_vector_set(sflux_norm_res, i, fabs(cpl_vector_get(sflux_norm, i) - sflux_norm_mean));
    }
    *fres = cpl_vector_get_mean(flux_norm_res);
    *cres = cpl_vector_get_mean(cflux_norm_res);
    *sres = cpl_vector_get_mean(sflux_norm_res);

    return cpl_vector_duplicate(cflux_sm);
}
cpl_table *mean_absolute_difference(cpl_table *telluric_data, mf_wrap_calctrans_parameter *parameters)
{
    //TODO: very important, smooth the full spectrum and then filter wavelength ranges afterwards...ugh
    //this is necessary to avoid introducing bad artefacts into the smoothed functions...

    //TODO: ASSERT maxwl >= 3

    //First find the wavelength region information from molecfit_model via best_fit_parameters
    //If we have WAVE_INCLUDE or WAVE_EXCLUDE in BEST_FIT_PARAMETERS, use them,
    //otherwise,  use entire wavelength range
    //extension indices
    //number of rows in the tables
    int nrows_inc = -1;
    int nrows_exc = -1;
    //wavelength ranges to include
    double *w1_inc = NULL;
    double *w2_inc = NULL;
    //wavelength ranges to exclude
    double *w1_exc = NULL;
    double *w2_exc = NULL;

    cpl_msg_info(cpl_func, "mean_absolute_difference: calculating wave ranges");
    for (int i = 0; i < parameters->best_fit_parameters->n_ext; i++) {
        if (cpl_propertylist_has(parameters->best_fit_parameters->v_ext[i].header, "EXTNAME")) {
            if (!strcmp(
                    cpl_propertylist_get_string(parameters->best_fit_parameters->v_ext[i].header, "EXTNAME"),
                    "WAVE_INCLUDE"
                )) {
                if (cpl_table_has_column(parameters->best_fit_parameters->v_ext[i].table, MF_COL_WAVE_RANGE_LOWER) &&
                    cpl_table_has_column(parameters->best_fit_parameters->v_ext[i].table, MF_COL_WAVE_RANGE_UPPER)) {
                    nrows_inc = cpl_table_get_nrow(parameters->best_fit_parameters->v_ext[i].table);
                    w1_inc    = cpl_table_get_data_double(
                        parameters->best_fit_parameters->v_ext[i].table, MF_COL_WAVE_RANGE_LOWER
                    );
                    w2_inc = cpl_table_get_data_double(
                        parameters->best_fit_parameters->v_ext[i].table, MF_COL_WAVE_RANGE_UPPER
                    );
                }
            }
            if (!strcmp(
                    cpl_propertylist_get_string(parameters->best_fit_parameters->v_ext[i].header, "EXTNAME"),
                    "WAVE_EXCLUDE"
                )) {
                if (cpl_table_has_column(parameters->best_fit_parameters->v_ext[i].table, MF_COL_WAVE_RANGE_LOWER) &&
                    cpl_table_has_column(parameters->best_fit_parameters->v_ext[i].table, MF_COL_WAVE_RANGE_UPPER)) {
                    nrows_exc = cpl_table_get_nrow(parameters->best_fit_parameters->v_ext[i].table);
                    w1_exc    = cpl_table_get_data_double(
                        parameters->best_fit_parameters->v_ext[i].table, MF_COL_WAVE_RANGE_LOWER
                    );
                    w2_exc = cpl_table_get_data_double(
                        parameters->best_fit_parameters->v_ext[i].table, MF_COL_WAVE_RANGE_UPPER
                    );
                }
            }
        }
    }


    int wlength;
    if (parameters->sg_as_max_length == CPL_FALSE) {
        double      fres, cres, sres;
        cpl_vector *cflux_sm;
        wlength = parameters->sg_window_length;
        cpl_msg_info(cpl_func, "mean_absolute_difference: single window_length specified: %d", wlength);
        cflux_sm =
            calc_rms(telluric_data, wlength, nrows_inc, w1_inc, w2_inc, nrows_exc, w1_exc, w2_exc, &fres, &cres, &sres);

        /*First fixup the cflux_sm vector so that it includes any bad data from the original table */
        cpl_size    nrows_orig      = cpl_table_get_nrow(telluric_data);
        cpl_vector *cflux_sm_proper = cpl_vector_new(nrows_orig);
        int        *qual_col        = cpl_table_get_data_int(telluric_data, MF_COL_OUT_MASK);
        int         counter         = 0;
        for (int i = 0; i < nrows_orig; i++) {
            if (qual_col[i] == 0) {
                cpl_vector_set(cflux_sm_proper, i, 0.0);
            }
            else {
                cpl_vector_set(cflux_sm_proper, i, cpl_vector_get(cflux_sm, counter));
                counter = counter + 1;
            }
        }
        /*Add column to telluric data with smoothed spectrum (cflux_sm) */
        cpl_table_new_column(telluric_data, "cflux_savgol", CPL_TYPE_DOUBLE);
        cpl_table_copy_data_double(telluric_data, "cflux_savgol", cpl_vector_get_data(cflux_sm_proper));

        //fres = cres = sres = 1.0;
        cpl_msg_info(cpl_func, "mean_absolute_difference: calc_rms called: %e %e %e", fres, cres, sres);
        cpl_table *res = cpl_table_new(1);
        cpl_table_new_column(res, "window_length", CPL_TYPE_INT);
        cpl_table_new_column(res, "flux_res", CPL_TYPE_DOUBLE);
        cpl_table_new_column(res, "cflux_res", CPL_TYPE_DOUBLE);
        cpl_table_new_column(res, "sflux_res", CPL_TYPE_DOUBLE);
        cpl_table_new_column(res, "res_ratio", CPL_TYPE_DOUBLE);
        cpl_table_set(res, "window_length", 0, wlength);
        cpl_table_set(res, "flux_res", 0, fres);
        cpl_table_set(res, "cflux_res", 0, cres);
        cpl_table_set(res, "sflux_res", 0, sres);
        cpl_table_set(res, "res_ratio", 0, fres / cres);
        return cpl_table_duplicate(res);
    }
    else {
        int maxwl = parameters->sg_window_length;
        int minwl = 5;
        int count = 0;
        for (wlength = maxwl; wlength >= minwl; wlength -= 2) {
            count += 1;
        }
        cpl_table *res = cpl_table_new(count);
        cpl_table_new_column(res, "window_length", CPL_TYPE_INT);
        cpl_table_new_column(res, "flux_res", CPL_TYPE_DOUBLE);
        cpl_table_new_column(res, "cflux_res", CPL_TYPE_DOUBLE);
        cpl_table_new_column(res, "sflux_res", CPL_TYPE_DOUBLE);
        cpl_table_new_column(res, "res_ratio", CPL_TYPE_DOUBLE);
        int         idx             = 0;
        cpl_vector *cflux_sm        = NULL;
        cpl_vector *cflux_sm_keep   = NULL;
        double      running_res_max = 0.;
        for (wlength = maxwl; wlength >= minwl; wlength -= 2) {
            double fres, cres, sres;
            cflux_sm = calc_rms(
                telluric_data, wlength, nrows_inc, w1_inc, w2_inc, nrows_exc, w1_exc, w2_exc, &fres, &cres, &sres
            );

            cpl_table_set(res, "window_length", idx, wlength);
            cpl_table_set(res, "flux_res", idx, fres);
            cpl_table_set(res, "cflux_res", idx, cres);
            cpl_table_set(res, "sflux_res", idx, sres);
            double res_ratio = fres / cres;
            cpl_table_set(res, "res_ratio", idx, res_ratio);
            //keep track of the max res ratio.
            if (wlength == maxwl) {
                running_res_max = res_ratio;
                cflux_sm_keep   = cpl_vector_duplicate(cflux_sm);
            }
            if (res_ratio > running_res_max) {
                running_res_max = res_ratio;
                if (cflux_sm_keep) {
                    cpl_vector_delete(cflux_sm_keep);
                }
                cflux_sm_keep = cpl_vector_duplicate(cflux_sm);
            }
            idx += 1;
        }
        /*First fixup the cflux_sm vector so that it includes any bad data from the original table */
        cpl_size    nrows_orig      = cpl_table_get_nrow(telluric_data);
        cpl_vector *cflux_sm_proper = cpl_vector_new(nrows_orig);
        int        *qual_col        = cpl_table_get_data_int(telluric_data, MF_COL_OUT_MASK);
        int         counter         = 0;
        for (int i = 0; i < nrows_orig; i++) {
            if (qual_col[i] == 0) {
                cpl_vector_set(cflux_sm_proper, i, 0.0);
            }
            else {
                cpl_vector_set(cflux_sm_proper, i, cpl_vector_get(cflux_sm_keep, counter));
                counter = counter + 1;
            }
        }
        /*Add column to telluric data with smoothed spectrum (cflux_sm) */
        cpl_table_new_column(telluric_data, "cflux_savgol", CPL_TYPE_DOUBLE);
        cpl_table_copy_data_double(telluric_data, "cflux_savgol", cpl_vector_get_data(cflux_sm_proper));

        return cpl_table_duplicate(res);
    }
}

/* Computes the Savitzky-Golay filter coefficients following the IDL function savgol.pro.
   Results match those from savgol_coeffs function from scipy.
*/
cpl_vector *savgol(int nl, int nr, int order, int degree)
{
    cpl_matrix *a = cpl_matrix_new(degree + 1, degree + 1);
    cpl_matrix *b = cpl_matrix_new(degree + 1, 1);
    //cpl_vector *b = cpl_vector_new(degree + 1);
    cpl_vector *cr    = cpl_vector_new(nr);
    cpl_vector *cl    = cpl_vector_new(nl);
    cpl_vector *power = cpl_vector_new(degree + 1);
    int         np    = nl + nr + 1;
    cpl_vector *c     = cpl_vector_new(np);

    if (nl < 0 || nr < 0) {
        cpl_error_set_message(
            cpl_func, CPL_ERROR_ILLEGAL_INPUT, "savgol: Both nl (%d) and nr (%d) must be positive", nl, nr
        );
        return NULL;
    }
    if (order > degree) {
        cpl_error_set_message(
            cpl_func, CPL_ERROR_ILLEGAL_INPUT, "savgol: The order (%d) must be less than or equal to the degree (%d)",
            order, degree
        );
        return NULL;
    }
    if (degree >= np) {
        cpl_error_set_message(
            cpl_func, CPL_ERROR_ILLEGAL_INPUT,
            "savgol: The degree (%d) must be less than the filter width (nl+nr+1=%d).", degree, np
        );
        return NULL;
    }

    int i, j, k;
    for (i = 0; i < nr; i++) {
        cpl_vector_set(cr, i, 1.0 + i * 1.0);
    }
    for (i = 0; i < nl; i++) {
        cpl_vector_set(cl, i, -(1.0 + i * 1.0));
    }
    for (i = 0; i < degree + 1; i++) {
        cpl_vector_set(power, i, i * 1.0);
        if (i == order) {
            cpl_matrix_set(b, i, 0, 1.0);
            //cpl_vector_set(b, i, 1.0);
        }
        else {
            cpl_matrix_set(b, i, 0, 0.0);
            //cpl_vector_set(b, i, 0.0);
        }
    }

    for (i = 0; i <= degree * 2; i++) {
        double sum = (i == 0) ? 1.0 : 0.0;
        if (nr > 0) {
            for (j = 0; j < nr; j++) {
                sum = sum + pow(cpl_vector_get(cr, j), i);
            }
        }
        if (nl > 0) {
            for (j = 0; j < nl; j++) {
                sum = sum + pow(cpl_vector_get(cl, j), i);
            }
        }
        double mm = i < (2 * degree - i) ? i : (2 * degree - i);
        for (j = -mm; j <= mm; j += 2) {
            cpl_matrix_set(a, (i + j) / 2, (i - j) / 2, sum);
        }
    }
    //gsl_matrix_fprintf(stdout,a,"%.2f");
    /*for (i = 0; i < degree+1; i++) {
        for (j = 0; j < degree+1; j++) {
            printf("a[%d][%d] = %.2f\n",i,j,cpl_matrix_get(a,i,j));
        }
    }*/
    //unsigned long N = cpl_matrix_get_nrow(a);//->size1;//, N = A->size2;
    //unsigned long N = a->size1;//, N = A->size2;

    //int signum;
    //int s = 0;

    //gsl_permutation * perm = gsl_permutation_alloc(N);
    //gsl_matrix * lu  = cpl_matrix_new(N, N);
    //gsl_vector * res = cpl_vector_new(N);

    //printf("A size: %d %d\n",A->size1,A->size2);
    //printf("b size: %d\n",b->size);
    //gsl_matrix_memcpy(lu, a);

    //s += gsl_linalg_LU_decomp(lu, perm, &signum);
    //s += gsl_linalg_LU_solve(lu, perm, b, res);
    cpl_matrix *res = cpl_matrix_solve(a, b);
    if (res == NULL) {
        cpl_error_set_message(
            cpl_func, CPL_ERROR_NULL_INPUT, "Could not solve matrices for Savitzky-Golay coefficients: %s",
            cpl_error_get_message()
        );
        return NULL;
    }
    //cpl_msg_info(cpl_func,"Error: %s",cpl_error_get_message());
    //cpl_matrix_dump(res,stdout);
    for (k = -nl; k <= nr; k++) {
        double sum = 0.0;
        for (i = 0; i < degree + 1; i++) {
            sum = sum + (cpl_matrix_get(res, i, 0) * pow(k, cpl_vector_get(power, i)));
            //sum = sum+(cpl_vector_get(res,i)*pow(k,cpl_vector_get(power,i)));
        }
        cpl_vector_set(c, k + nl, sum);
    }
    //these are the Savitzky-Golay coefficients
    //for(i=0;i<np;i++){
    //    printf("c[%d] = %.8f\n",i,cpl_vector_get(c,i));
    //}
    return c;
}
