/*
 * This file is part of the ESO Telluric Correction Library
 * Copyright (C) 2001-2018 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef MF_WRAP_CONFIG_CALC_H
#define MF_WRAP_CONFIG_CALC_H

/*----------------------------------------------------------------------------*/
/**
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include <cpl.h>

//#include <telluriccorr.h>
/*#include <mf_wrap_fits.h>
#include <mf_wrap_data.h>
#include <mf_wrap_dfs.h>*/

#include <mf_wrap_utils.h>
#include "mf_wrap_config.h"
#include <mf_calctrans.h>
//#include <telluriccorr.h>
//#include <mf_calctrans.h>

//#include "mf_constants.h" //?
//#include "mf_configuration.h"

CPL_BEGIN_DECLS

/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Enumeration types
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Defines
 */
/*----------------------------------------------------------------------------*/

#define MOLECFIT_CALCTRANS "molecfit_calctrans"

#undef RECIPE_NAME
#define RECIPE_NAME MOLECFIT_CALCTRANS
#define CONTEXT     "molecfit." RECIPE_NAME

/*----------------------------------------------------------------------------*/
/**
 *                 Global variables
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Macros
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Structured types
 */
/*----------------------------------------------------------------------------*/

typedef struct
{
    cpl_boolean
        use_only_input_pri_ext; /* If the user want to use only the  SCIENCE_CALCTRANS/SCIENCE input FITS primary extension                                    */
    int dflux_extension_data; /* If use_only_input_pri_ext == CPL_TRUE, you can provide a extension as DFLUX (error flux)                                    */
    int mask_extension_data; /* If use_only_input_pri_ext == CPL_TRUE, you can provide a extension as MASK                                                  */

    cpl_boolean
        use_input_kernel; /* If the user provided, use or not the user kernel                                                                            */

    const char *
        mapping_kernel; /* Mapping extensions : SCIENCE-CALCTRANS_KERNEL_LIBRARY; i.e. {Science_ext1-Kernel_lib_ext, ..., Science_extN-Kernel_lib_ext} */
    cpl_table *
        mapping_kernel_table; /* Mapping kernel extensions cpl_table. Contains 1 columns [KERNEL_LIBRARY_EXT]                                                */

    const char *
        mapping_atmospheric; /* Mapping extensions : SCIENCE-ATM_PARAMETERS, i.e. {Science_ext1-Lblrtm_results_ext, ..., Science_extN-Lblrtm_results_ext}   */
    cpl_table *
        mapping_atmospheric_table; /* Mapping lblrtm_results extensions cpl_table. Contains 1 columns [LBLRTM_RESULTS_EXT]                                        */

    const char *
        mapping_convolve; /* Mapping extensions : SCIENCE-LBLRTM_RESULTS, i.e. {Science_ext1-Lblrtm_results_ext, ..., Science_extN-Lblrtm_results_ext}   */
    cpl_table *
        mapping_convolve_table; /* Mapping lblrtm_results extensions cpl_table. Contains 1 columns [LBLRTM_RESULTS_EXT]                                        */

    /* Calibration */
    mf_wrap_fits *
        atm_parameters; /* cpl_table's in the ATM_PARAMETERS      (output of molecfit_model)                                                           */
    mf_wrap_fits *
        best_fit_parameters; /* cpl_table's in the BEST_FIT_PARAMETERS (output of molecfit_model)                                                           */
    const char *scale_pwv;
    const char *mjd_pwv;
    const char *exp_pwv;
    const char *air1_pwv;
    const char *air2_pwv;
    double      pwv_sci;
    double      pwv_ratio;
    double h2o_col_mm; /* Variable to store value read in from BEST_FIT_PARAMETERS table (output of molecfit_model) */

    int         sg_window_length; /* Savitzky-Golay filter smoothing window length in pixels */
    cpl_boolean sg_as_max_length; /* Treat sg_window_length as a maximum length */

    /* Science */
    cpl_size
        n_ext; /* Number of extensions                                                                                                        */
    cpl_propertylist **
        telluriccorr_head; /* Science spectrum header array                                                                                               */
    cpl_table **
        telluriccorr_data; /* Science spectrum data   array                                                                                               */
    cpl_table **
        best_fit_parameters_table; /* Best fit parameters array                                                                                                   */
    mf_calctrans_lblrtm_results **
        results_lblrtm; /* Results execute mf_calctrans_lblrtm for every input extension                                                               */
    mf_calctrans_convolution_results **
        results_convolution; /* Results of telluric corrections after execute mf_calctrans_convolution                                                      */

    mf_configuration *
        mf_config; /* Telluric Correction configuration parameter                                                                                 */

    cpl_boolean
        chip_extensions; /* If the user want to combine the input FITS extensions                                                                       */

    cpl_propertylist *
        pl; /* All input parameters to save in the output files                                                                            */

} mf_wrap_calctrans_parameter;

/*----------------------------------------------------------------------------*/
/**
 *                 Functions prototypes
 */
/*----------------------------------------------------------------------------*/

cpl_error_code mf_wrap_load_tag_science(mf_wrap_fits **data, cpl_frame *input_frame);

//cpl_error_code mf_wrap_config_calc_setup_input_filenames(cpl_parameterlist* ilist, const char* suffix, const char* input_name);

//cpl_error_code mf_wrap_config_calc_setup_output_filenames(cpl_parameterlist* ilist, const char* suffix);

/* Parse output header and add to the mf_parameters_config and the header propertylist */
mf_configuration *mf_wrap_config_load_header_parameters(
    const cpl_propertylist *header,
    const cpl_propertylist *data_header,
    cpl_propertylist       *pl
);

/* Cleanup any memory allocations that were created by calling
 molecfit_config_load_header_parameters */
void mf_wrap_config_delete_configuration(mf_configuration *cfg);

mf_wrap_calctrans_parameter *mf_wrap_config_calc_init(
    cpl_frameset            *frameset,
    const cpl_parameterlist *list,
    cpl_propertylist        *scientific_head_pri,
    cpl_size                 n_ext,
    const char              *combined_suffix
);

cpl_vector *savgol(int nl, int nr, int order, int degree);
cpl_vector *convol(cpl_vector *v, cpl_vector *kernel);
cpl_table  *mean_absolute_difference(cpl_table *telluric_data, mf_wrap_calctrans_parameter *parameters);
cpl_vector *calc_rms(
    cpl_table *t,
    int        wlength,
    int        nrows_inc,
    double    *w1_inc,
    double    *w2_inc,
    int        nrows_exc,
    double    *w1_exc,
    double    *w2_exc,
    double    *fres,
    double    *cres,
    double    *sres
);

void mf_wrap_calctrans_parameter_delete(mf_wrap_calctrans_parameter *parameters);


// mf_wrap_sciframe(cpl_frameset *frameset, const cpl_parameterlist  *parlist, mf_wrap_model_parameter *parameters); // DO WE NEED THIS ??


CPL_END_DECLS


#endif /* MF_WRAP_CONFIG_CALC_H */
