/*
 * This file is part of the Molecfit Pipeline
 * Copyright (C) 2001-2019 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include "assert.h"

#include "mf_wrap_dfs.h"


#define CPL_DFS_PRO_DID "PRO-1.16"

static cpl_error_code cpl_dfs_product_save(
    cpl_frameset *,
    cpl_propertylist *,
    const cpl_parameterlist *,
    const cpl_frameset *,
    const cpl_frame *,
    const cpl_imagelist *,
    const cpl_image *,
    cpl_type,
    const cpl_table *,
    const cpl_vector *,
    const cpl_propertylist *,
    const char *,
    const cpl_propertylist *,
    const char *,
    const char *,
    const char *
);

/*----------------------------------------------------------------------------*/
/**
 * @defgroup molecfit_dfs  DFS related functions
 *
 * TBD
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
 * @brief    Set the group as RAW or CALIB in a frameset
 *
 * @param    set     the input frameset
 *
 * @return   The function returns @c CPL_ERROR_NONE on success or a CPL error code otherwise.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code mf_wrap_dfs_set_groups(cpl_frameset *set)
{
    /* Check entries */
    cpl_ensure_code(set != NULL, CPL_ERROR_NULL_INPUT);

    /* Initialize */
    cpl_size nframes = cpl_frameset_get_size(set);

    /* Loop on frames */
    for (cpl_size i = 0; i < nframes; i++) {
        cpl_frame  *cur_frame = cpl_frameset_get_position(set, i);
        const char *tag       = cpl_frame_get_tag(cur_frame);

        if (!strcmp(tag, MOLECFIT_STD_MODEL) || !strcmp(tag, MOLECFIT_SCIENCE_CALCTRANS) ||
            !strcmp(tag, MOLECFIT_SCIENCE)) {
            /* RAW frames */
            cpl_frame_set_group(cur_frame, CPL_FRAME_GROUP_RAW);
        }
        else if (!strcmp(tag, MOLECFIT_MOLECULES) || !strcmp(tag, MOLECFIT_MODEL_MOLECULES) ||

                 !strcmp(tag, MOLECFIT_WAVE_INCLUDE) || !strcmp(tag, MOLECFIT_WAVE_EXCLUDE) ||
                 !strcmp(tag, MOLECFIT_PIXEL_EXCLUDE) ||

                 !strcmp(tag, MOLECFIT_GDAS) || !strcmp(tag, MOLECFIT_GDAS_BEFORE) ||
                 !strcmp(tag, MOLECFIT_GDAS_AFTER) ||

                 !strcmp(tag, MOLECFIT_ATM_PROFILE_STANDARD) || !strcmp(tag, MOLECFIT_ATM_PROFILE_COMBINED) ||

                 !strcmp(tag, MOLECFIT_KERNEL_LIBRARY) || !strcmp(tag, MOLECFIT_MAPPING_KERNEL) ||

                 !strcmp(tag, MOLECFIT_MODEL_KERNEL_LIBRARY) || !strcmp(tag, MOLECFIT_MODEL_MAPPING_KERNEL) ||

                 !strcmp(tag, MOLECFIT_MODEL_INIT_FIT_PARAMETERS) ||

                 !strcmp(tag, MOLECFIT_CALCTRANS_KERNEL_LIBRARY) || !strcmp(tag, MOLECFIT_CALCTRANS_MAPPING_KERNEL) ||

                 !strcmp(tag, MOLECFIT_MAPPING_ATMOSPHERIC) || !strcmp(tag, MOLECFIT_MAPPING_CONVOLVE) ||
                 !strcmp(tag, MOLECFIT_MAPPING_CORRECT) || !strcmp(tag, MOLECFIT_FIT_MODEL) ||

                 !strcmp(tag, MOLECFIT_ATM_PARAMETERS) || !strcmp(tag, MOLECFIT_BEST_FIT_PARAMETERS) ||
                 !strcmp(tag, MOLECFIT_BEST_FIT_MODEL) ||

                 !strcmp(tag, MOLECFIT_LBLRTM_RESULTS) ||

                 !strcmp(tag, MOLECFIT_TELLURIC_DATA) || !strcmp(tag, MOLECFIT_TELLURIC_CORR)) {
            /* CALIB frames */
            cpl_frame_set_group(cur_frame, CPL_FRAME_GROUP_CALIB);
        }
        else if (!strcmp(tag, MOLECFIT_MODEL_CHIPS_COMBINED) || !strcmp(tag, MOLECFIT_CALCTRANS_CHIPS_COMBINED) ||
                 !strcmp(tag, MOLECFIT_CORRECT_CHIPS_COMBINED) ||

                 !strcmp(tag, MOLECFIT_SCIENCE_TELLURIC_CORR) || !strcmp(tag, MOLECFIT_SPECTRUM_TELLURIC_CORR)) {
            /* PRODUCTS frames */
            cpl_frame_set_group(cur_frame, CPL_FRAME_GROUP_PRODUCT);
        }
        else {
            /* unknown-frame */
            cpl_frame_set_group(cur_frame, CPL_FRAME_GROUP_NONE);
            cpl_msg_warning(cpl_func, "Frame:%lld with tag:%s, unknown!", i, tag);
        }
    }

    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 * @brief  Save an image as a DFS-compliant pipeline product
 *
 * @param  allframes  The list of input frames for the recipe
 * @param  header     NULL, or filled with properties written to product header
 * @param  parlist    The list of input parameters
 * @param  usedframes The list of raw/calibration frames used for this product
 * @param  inherit    NULL or product frames inherit their header from this frame
 * @param  vector     The vector to be saved
 * @param  type       The type used to represent the data in the file
 * @param  recipe     The recipe name
 * @param  applist    Propertylist to append to primary header, w. PRO.CATG
 * @param  remregexp  Optional regexp of properties not to put in main header
 * @param  pipe_id    PACKAGE "/" PACKAGE_VERSION
 * @param  filename   Filename of created product
 *
 * @note The image may be NULL in which case only the header information is saved but passing a NULL image is deprecated, use cpl_dfs_save_propertylist()..
 *
 * @note remregexp may be NULL.
 *
 * @note applist must contain a string-property with key CPL_DFS_PRO_CATG.
 *
 * @note On success and iff header is non-NULL, it will be emptied and then filled with the properties written to the primary header of the product.
 *
 * @note The FITS header of the created product is created from the provided applist and the cards copied by cpl_dfs_setup_product_header(), with exception of the cards whose keys match the provided remregexp.
 *
 * @return CPL_ERROR_NONE or the relevant CPL error code on error (See cpl_dfs_setup_product_header(), cpl_vector_save()).
 *
 */
/*----------------------------------------------------------------------------*/
cpl_error_code mf_wrap_dfs_save_vector(
    cpl_frameset            *allframes,
    cpl_propertylist        *header,
    const cpl_parameterlist *parlist,
    const cpl_frameset      *usedframes,
    const cpl_frame         *inherit,
    const cpl_vector        *vector,
    cpl_type                 type,
    const char              *recipe,
    const cpl_propertylist  *applist,
    const char              *remregexp,
    const char              *pipe_id,
    const char              *filename
)
{
    return cpl_dfs_product_save(
        allframes, header, parlist, usedframes, inherit, NULL, NULL, type, NULL, vector, NULL, recipe, applist,
        remregexp, pipe_id, filename
    );
}


/** @cond PRIVATE */

/*----------------------------------------------------------------------------*/
/**
  @internal
  @brief  Save either an image or table as a pipeline product
  @param  allframes  The list of input frames for the recipe
  @param  header     NULL, or filled with properties written to product header
  @param  parlist    The list of input parameters
  @param  usedframes The list of raw/calibration frames used for this product
  @param  inherit    NULL, or frame from which header information is inherited
  @param  imagelist  The imagelist to be saved or NULL
  @param  image      The image to be saved or NULL
  @param  type       The type used to represent the data in the file
  @param  table      The table to be saved or NULL
  @param  tablelist  Optional propertylist to use in table extension or NULL
  @param  recipe     The recipe name
  @param  applist    Propertylist to append to primary header, w. PRO.CATG
  @param  remregexp  Optional regexp of properties not to put in main header
  @param  pipe_id    PACKAGE "/" PACKAGE_VERSION
  @param  filename   Filename of created product
  @return CPL_ERROR_NONE or the relevant CPL error code on error
  @note At most one of imagelist, image and table may be non-NULL, if all are
        NULL the product frame type will be that of an image
  @see cpl_dfs_save_image()

 */
/*----------------------------------------------------------------------------*/
static cpl_error_code cpl_dfs_product_save(
    cpl_frameset            *allframes,
    cpl_propertylist        *header,
    const cpl_parameterlist *parlist,
    const cpl_frameset      *usedframes,
    const cpl_frame         *inherit,
    const cpl_imagelist     *imagelist,
    const cpl_image         *image,
    cpl_type                 type,
    const cpl_table         *table,
    const cpl_vector        *vector,
    const cpl_propertylist  *tablelist,
    const char              *recipe,
    const cpl_propertylist  *applist,
    const char              *remregexp,
    const char              *pipe_id,
    const char              *filename
)
{
    const char       *procat;
    cpl_propertylist *plist;
    cpl_frame        *product_frame;
    /* Inside this function the product-types are numbered:
       0: imagelist
       1: table
       2: image
       3: propertylist only
    */
    const unsigned pronum    = imagelist != NULL ? 0 : table != NULL ? 1 : image != NULL ? 2 : (vector != NULL ? 3 : 4);
    const char    *proname[] = { "imagelist", "table", "image", "propertylist", "unknown" };
    /* FIXME: Define a frame type for an imagelist and when data-less */
    const int      protype[] = { CPL_FRAME_TYPE_ANY, CPL_FRAME_TYPE_TABLE, CPL_FRAME_TYPE_IMAGE, CPL_FRAME_TYPE_IMAGE,
                                 CPL_FRAME_TYPE_ANY };
    cpl_error_code error     = CPL_ERROR_NONE;


    /* No more than one of imagelist, table and image may be non-NULL */
    /* tablelist may only be non-NULL when table is non-NULL */
    if (imagelist != NULL) {
        assert(pronum == 0);
        assert(image == NULL);
        assert(table == NULL);
        assert(tablelist == NULL);
        assert(vector == NULL);
    }
    else if (table != NULL) {
        assert(pronum == 1);
        assert(imagelist == NULL);
        assert(image == NULL);
        assert(vector == NULL);
    }
    else if (image != NULL) {
        assert(pronum == 2);
        assert(imagelist == NULL);
        assert(table == NULL);
        assert(tablelist == NULL);
        assert(vector == NULL);
    }
    else if (vector != NULL) {
        assert(pronum == 3);
        assert(imagelist == NULL);
        assert(table == NULL);
        assert(tablelist == NULL);
        assert(image == NULL);
    }
    else {
        assert(pronum == 4);
        assert(imagelist == NULL);
        assert(table == NULL);
        assert(tablelist == NULL);
        assert(image == NULL);
        assert(vector == NULL);
    }

    cpl_ensure_code(allframes != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(parlist != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(usedframes != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(recipe != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(applist != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(pipe_id != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(filename != NULL, CPL_ERROR_NULL_INPUT);

    procat = cpl_propertylist_get_string(applist, CPL_DFS_PRO_CATG);

    cpl_ensure_code(procat != NULL, cpl_error_get_code());

    cpl_msg_info(cpl_func, "Writing FITS %s product(%s): %s", proname[pronum], procat, filename);


    product_frame = cpl_frame_new();

    /* Create product frame.
       NB: With multiple errors, error will differ from the actual CPL error */
    error |= cpl_frame_set_filename(product_frame, filename);
    error |= cpl_frame_set_tag(product_frame, procat);
    error |= cpl_frame_set_type(product_frame, protype[pronum]);
    error |= cpl_frame_set_group(product_frame, CPL_FRAME_GROUP_PRODUCT);
    error |= cpl_frame_set_level(product_frame, CPL_FRAME_LEVEL_FINAL);

    cpl_msg_info(cpl_func, "After Create product frame... %s", cpl_error_get_message());

    if (error) {
        cpl_frame_delete(product_frame);
        return error;
    }

    if (header != NULL) {
        cpl_propertylist_empty(header);
        plist = header;
    }
    else {
        plist = cpl_propertylist_new();
    }

    /* Add any QC parameters here */
    error = cpl_propertylist_append(plist, applist);

    /* Add DataFlow keywords */
    if (!error) {
        error = cpl_dfs_setup_product_header(
            plist, product_frame, usedframes, parlist, recipe, pipe_id, CPL_DFS_PRO_DID, inherit
        );
    }

    if (remregexp != NULL && !error) {
        cpl_errorstate prestate = cpl_errorstate_get();
        (void)cpl_propertylist_erase_regexp(plist, remregexp, 0);
        if (!cpl_errorstate_is_equal(prestate)) {
            error = cpl_error_get_code();
        }
    }

    if (!error) {
        switch (pronum) {
            case 0:
                error = cpl_imagelist_save(imagelist, filename, type, plist, CPL_IO_CREATE);
                break;
            case 1:
                error = cpl_table_save(table, plist, tablelist, filename, CPL_IO_CREATE);
                break;
            case 2:
                error = cpl_image_save(image, filename, type, plist, CPL_IO_CREATE);
                break;
            case 3:
                error = cpl_vector_save(vector, filename, type, plist, CPL_IO_CREATE);
                break;
            default:
                /* case 4: */
                error = cpl_propertylist_save(plist, filename, CPL_IO_CREATE);
        }
    }

    if (!error) {
        /* Insert the frame of the saved file in the input frameset */
        error = cpl_frameset_insert(allframes, product_frame);
    }
    else {
        cpl_frame_delete(product_frame);
    }

    if (plist != header) {
        cpl_propertylist_delete(plist);
    }

    return error;
}


/** @endcond */

/**@}*/
