/*
 * This file is part of the Molecfit Pipeline
 * Copyright (C) 2001-2019 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef MF_WRAP_FITS_H
#define MF_WRAP_FITS_H

/*----------------------------------------------------------------------------*/
/**
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include <cpl.h>

#include "mf_wrap_dfs.h"


CPL_BEGIN_DECLS

/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Enumeration types
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Defines
 */
/*----------------------------------------------------------------------------*/

/* Header FITS files: Valid values for XTENSION keyword */
#define MOLECFIT_FITS_KEYWORD_XTENSION "XTENSION"
#define MOLECFIT_FITS_KEYWORD_BINTABLE "BINTABLE"
#define MOLECFIT_FITS_KEYWORD_IMAGE    "IMAGE"

/* Header FITS files: General definition keywords */
#define MOLECFIT_FITS_KEYWORD_SIMPLE  "SIMPLE"
#define MOLECFIT_FITS_KEYWORD_BITPIX  "BITPIX"
#define MOLECFIT_FITS_KEYWORD_EXTEND  "EXTEND"
#define MOLECFIT_FITS_KEYWORD_DATE    "DATE"
#define MOLECFIT_FITS_KEYWORD_EXPTIME "EXPTIME"
#define MOLECFIT_FITS_KEYWORD_EXTNAME "EXTNAME"
#define MOLECFIT_FITS_KEYWORD_EXTVER  "EXTVER"
#define MOLECFIT_FITS_KEYWORD_TFIELDS "TFIELDS"
#define MOLECFIT_FITS_KEYWORD_NAXIS   "NAXIS"
#define MOLECFIT_FITS_KEYWORD_NAXIS1  "NAXIS1"
#define MOLECFIT_FITS_KEYWORD_NAXIS2  "NAXIS2"
#define MOLECFIT_FITS_KEYWORD_NAXIS3  "NAXIS3"
#define MOLECFIT_FITS_KEYWORD_CRPIX1  "CRPIX1"
#define MOLECFIT_FITS_KEYWORD_CRPIX2  "CRPIX2"
#define MOLECFIT_FITS_KEYWORD_CRPIX3  "CRPIX3"
#define MOLECFIT_FITS_KEYWORD_CRVAL1  "CRVAL1"
#define MOLECFIT_FITS_KEYWORD_CRVAL2  "CRVAL2"
#define MOLECFIT_FITS_KEYWORD_CRVAL3  "CRVAL3"
/* Now obsolete keywords. CDELT1 = CD1_1, CDELT2 = CD2_2, CDELT3 = CD3_3 */
/* Uncommented MOLECFIT_FITS_KEYWORD_CDELT1 as we use it when CD1_1 not present */
#define MOLECFIT_FITS_KEYWORD_CDELT1 "CDELT1"
/*
#define MOLECFIT_FITS_KEYWORD_CDELT2            "CDELT2"
#define MOLECFIT_FITS_KEYWORD_CDELT3            "CDELT3"
*/
#define MOLECFIT_FITS_KEYWORD_CTYPE1 "CTYPE1"
#define MOLECFIT_FITS_KEYWORD_CTYPE2 "CTYPE2"
#define MOLECFIT_FITS_KEYWORD_CTYPE3 "CTYPE3"
#define MOLECFIT_FITS_KEYWORD_CUNIT1 "CUNIT1"
#define MOLECFIT_FITS_KEYWORD_CUNIT2 "CUNIT2"
#define MOLECFIT_FITS_KEYWORD_CUNIT3 "CUNIT3"
#define MOLECFIT_FITS_KEYWORD_CD1_1  "CD1_1"
#define MOLECFIT_FITS_KEYWORD_CD1_2  "CD1_2"
#define MOLECFIT_FITS_KEYWORD_CD1_3  "CD1_3"
#define MOLECFIT_FITS_KEYWORD_CD2_1  "CD2_1"
#define MOLECFIT_FITS_KEYWORD_CD2_2  "CD2_2"
#define MOLECFIT_FITS_KEYWORD_CD2_3  "CD2_3"
#define MOLECFIT_FITS_KEYWORD_CD3_1  "CD3_1"
#define MOLECFIT_FITS_KEYWORD_CD3_2  "CD3_2"
#define MOLECFIT_FITS_KEYWORD_CD3_3  "CD3_3"

#define MOLECFIT_FITS_KEYWORD_CTYPE1_SPECTRUM "WAVE"


/*----------------------------------------------------------------------------*/
/**
 *                 Global variables
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Macros
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Structured types
 */
/*----------------------------------------------------------------------------*/

/* Enumeration structure for identifying the kind of FITS files */
typedef enum
{
    mf_wrap_fits_file_none = -1, /* Not exist file                                     */
    mf_wrap_fits_file_invalid,   /* Invalid format                                     */
    mf_wrap_fits_file_no_data,   /* No data in the FITS file (only headers in all ext) */
    mf_wrap_fits_file_table,     /* Table                                              */
    mf_wrap_fits_file_image_1D,  /* Image 1D                                           */
    mf_wrap_fits_file_image_2D,  /* Image 2D                                           */
    mf_wrap_fits_file_image_3D   /* Image 3D                                           */
} mf_wrap_fits_file_format;

/* Enumeration structure for identifying integer and double type */
typedef enum
{
    mf_wrap_fits_table = 1, /*                                                    */
    mf_wrap_fits_vector,    /*                                                    */
    mf_wrap_fits_matrix,    /*                                                    */
    mf_wrap_fits_image,     /*                                                    */
    mf_wrap_fits_cube       /*                                                    */
} mf_wrap_fits_type;

/* Structure for data of FITS file */
typedef struct
{
    cpl_propertylist *header; /* CPL property list                                  */
    cpl_table        *table;  /* CPL table                                          */
    cpl_vector       *vector; /* CPL image 1D (vector)                              */
    cpl_matrix       *matrix; /* CPL matrix                                         */
    cpl_image        *image;  /* CPL image 2D                                       */
    cpl_imagelist    *cube;   /* CPL image 3D (cube)                                */

    cpl_propertylist *spectrum_head; /* CPL property list                                  */
    cpl_table        *spectrum_data; /* Data converted in a cpl_table spectrum             */

} mf_wrap_fits_ext;

/* Structure for data of FITS file */
typedef struct
{
    char                    *filename; /* Original FITS file name                            */
    mf_wrap_fits_file_format format;   /* Original FITS file format                          */
    mf_wrap_fits_type        type;     /* Data type in the content                           */
    int                      n_ext;    /* Number of FITS extensions                          */
    mf_wrap_fits_ext        *v_ext;    /* Array of extensions                                */
} mf_wrap_fits;


/*----------------------------------------------------------------------------*/
/**
 *                 Functions prototypes
 */
/*----------------------------------------------------------------------------*/

/* Get the file format of a fits file*/
mf_wrap_fits_file_format mf_wrap_fits_get_file_format(const char *filename, const cpl_boolean verbose);

/* Create mf_wrap_fits without data */
mf_wrap_fits *mf_wrap_fits_create(
    const char              *filename,
    mf_wrap_fits_file_format format,
    mf_wrap_fits_type        type,
    const cpl_size           n_ext
);

/* Delete mf_wrap_fits structure */
void mf_wrap_fits_delete(mf_wrap_fits *data);

/* Load mf_wrap_fits structure */
mf_wrap_fits *mf_wrap_fits_load(const char *filename, cpl_boolean is_matrix);

/* Load mf_wrap_fits structure with verbose option */
mf_wrap_fits *mf_wrap_fits_load_verbose(const char *filename, cpl_boolean is_matrix, cpl_boolean verbose);

/* Write mf_wrap_fits structure */
cpl_error_code mf_wrap_fits_write(
    cpl_frameset            *all_frameset,
    cpl_frameset            *used_frameset,
    const cpl_parameterlist *parlist,
    const char              *recipe,
    const char              *tag,
    const mf_wrap_fits      *data,
    const char              *out_filename
);


CPL_END_DECLS


#endif /* MOLECFIT_FITS_H */
