/*
 * This file is part of the ESO Telluric Correction Library
 * Copyright (C) 2001-2018 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/*----------------------------------------------------------------------------*/
/**
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include "mf_constants.h"

#include "mf_parameters.h"
#include "mf_spectrum.h"
#include "mf_atm_combined_gdas.h"

#include <cpl_test.h>

/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Enumeration types
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Defines
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Global variables
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Macros
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Structured types
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Functions prototypes
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Program
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @defgroup test_mf_atm_create   .
 *
 * @brief
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/* ---------------------------------------------------------------------------*/
/**
 * @brief .
 *
 * @param .                  .
 * @param                    .
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK.
 *                           If not, these are the errors:
 *                           - .
 *                           - Error in subroutine (see subroutines).
 *
 * @description .
 *
 * @note .
 *
 */
/* ---------------------------------------------------------------------------*/
int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_DEBUG);

    /* Initialize */
    cpl_errorstate errstate = cpl_errorstate_get();

    cpl_msg_info(cpl_func, "\n============================");
    cpl_msg_info(cpl_func, "Testing mf_atm_combined_gdas");
    cpl_msg_info(cpl_func, "============================\n");

    /* Get time */
    double cs = cpl_test_get_walltime();

    /* Create configuration */
    mf_parameters_config *config_parameters = mf_parameters_config_create();
    cpl_test_error(CPL_ERROR_NONE);

    if (config_parameters->inputs.column_lambda) {
        cpl_free(config_parameters->inputs.column_lambda);
    }
    if (config_parameters->inputs.column_flux) {
        cpl_free(config_parameters->inputs.column_flux);
    }
    if (config_parameters->inputs.column_dflux) {
        cpl_free(config_parameters->inputs.column_dflux);
    }
    if (config_parameters->inputs.column_mask) {
        cpl_free(config_parameters->inputs.column_mask);
    }

    //  if (config_parameters->directories.output_name) cpl_free(config_parameters->directories.output_name);

    config_parameters->inputs.column_lambda = cpl_sprintf("%s", "Wavelength");
    config_parameters->inputs.column_flux   = cpl_sprintf("%s", "Extracted_OPT");
    config_parameters->inputs.column_dflux  = cpl_sprintf("%s", "Error_OPT");
    config_parameters->inputs.column_mask   = cpl_sprintf("%s", "NULL");

    config_parameters->inputs.wlg_to_micron = 1e-3;
    //  config_parameters->directories.output_name              = cpl_sprintf("%s", "telluriccorr_test_crires");
    config_parameters->fitting.ftol                         = 1e-2;
    config_parameters->fitting.xtol                         = 1e-2;
    config_parameters->fitting.fit_telescope_background.fit = CPL_FALSE;
    config_parameters->fitting.fit_continuum.n              = 3;
    config_parameters->fitting.fit_wavelenght.n             = 3;
    config_parameters->fitting.fit_res_box.fit              = CPL_FALSE;
    config_parameters->fitting.fit_res_box.const_val        = 0.;
    config_parameters->fitting.fit_lorentz.fit              = CPL_FALSE;
    config_parameters->fitting.fit_lorentz.const_val        = 0.5;
    config_parameters->fitting.kern_fac                     = 300.;

    /* Initialize parameters */
    mf_parameters *params = mf_parameters_initialize(config_parameters, NULL);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull(params);

    /* Create ATM profile */
    /*
  cpl_table *gdas_user        = NULL;
  cpl_table *gdas_profile1    = NULL;
  cpl_table *gdas_profile2    = NULL;
  cpl_table *gdas_interpolate = NULL;
  cpl_table *atm_profile      = mf_atm(params, gdas_user, &gdas_profile1, &gdas_profile2, &gdas_interpolate);
  if (gdas_profile1   ) cpl_table_delete(gdas_profile1   );
  if (gdas_profile2   ) cpl_table_delete(gdas_profile2   );
  if (gdas_interpolate) cpl_table_delete(gdas_interpolate);
  cpl_test_error(CPL_ERROR_NONE);
  cpl_test_nonnull(atm_profile);
  */

    /* Show results */
    //cpl_test_rel(cpl_table_get(atm_profile, "TEM",  0, NULL), 289.848,  1e-5);
    //cpl_test_rel(cpl_table_get(atm_profile, "HGT",  0, NULL),   2.,     1e-5);
    //cpl_test_rel(cpl_table_get(atm_profile, "TEM", 31, NULL), 256.38,   1e-5);
    //cpl_test_rel(cpl_table_get(atm_profile, "HGT", 31, NULL),  39.4286, 1e-5);

    /* Cleanup */
    //cpl_table_delete(atm_profile);
    mf_parameters_delete(params);
    mf_parameters_config_delete(config_parameters);
    cpl_test_error(CPL_ERROR_NONE);

    /* Show time */
    double ce = cpl_test_get_walltime();
    cpl_msg_info(cpl_func, "test_mf_atm_create() -> Run time: %g min\n", (ce - cs) / 60.);

    /* Show errors and return */
    cpl_errorstate_dump(errstate, CPL_TRUE, cpl_errorstate_dump_one);
    return cpl_test_end(0);
}


/** @cond PRIVATE */

/** @endcond */

/**@}*/
