/*
 * This file is part of the ESO Telluric Correction Library
 * Copyright (C) 2001-2018 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/*----------------------------------------------------------------------------*/
/**
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include "mf_constants.h"

#include "mf_parameters.h"
#include "mf_model.h"
#include "mf_kernel_synthetic.h"

#include <cpl_test.h>

/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Enumeration types
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Defines
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Global variables
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Macros
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Structured types
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Functions prototypes
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Program
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @defgroup test_mf_model   .
 *
 * @brief
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/* ---------------------------------------------------------------------------*/
/**
 * @brief .
 *
 * @param .                  .
 * @param                    .
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK.
 *                           If not, these are the errors:
 *                           - .
 *                           - Error in subroutine (see subroutines).
 *
 * @description .
 *
 * @note .
 *
 */
/* ---------------------------------------------------------------------------*/

/* TO TEST:
    mf_kernel_synthetic    
*/


int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_DEBUG);

    cpl_msg_info(cpl_func, "Testing mf_kerenel_synthetic.c");

    cpl_error_code err;

    /* Create dummy configuration needed for parameter object */
    mf_parameters_config *config_parameters = mf_parameters_config_create();

    if (config_parameters->inputs.column_lambda) {
        cpl_free(config_parameters->inputs.column_lambda);
    }
    if (config_parameters->inputs.column_flux) {
        cpl_free(config_parameters->inputs.column_flux);
    }
    if (config_parameters->inputs.column_dflux) {
        cpl_free(config_parameters->inputs.column_dflux);
    }
    if (config_parameters->inputs.column_mask) {
        cpl_free(config_parameters->inputs.column_mask);
    }

    //    if (config_parameters->directories.output_name) cpl_free(config_parameters->directories.output_name);

    config_parameters->inputs.column_lambda = cpl_sprintf("%s", "Wavelength");
    config_parameters->inputs.column_flux   = cpl_sprintf("%s", "Extracted_OPT");
    config_parameters->inputs.column_dflux  = cpl_sprintf("%s", "Error_OPT");
    config_parameters->inputs.column_mask   = cpl_sprintf("%s", "NULL");

    config_parameters->inputs.wlg_to_micron = 1e-3;
    //    config_parameters->directories.output_name              = cpl_sprintf("%s", "telluriccorr_test_crires");
    config_parameters->fitting.ftol                         = 1e-2;
    config_parameters->fitting.xtol                         = 1e-2;
    config_parameters->fitting.fit_telescope_background.fit = CPL_FALSE;
    config_parameters->fitting.fit_continuum.n              = 3;
    config_parameters->fitting.fit_wavelenght.n             = 3;
    config_parameters->fitting.fit_res_box.fit              = CPL_FALSE;
    config_parameters->fitting.fit_res_box.const_val        = 0.;
    config_parameters->fitting.fit_lorentz.fit              = CPL_FALSE;
    config_parameters->fitting.fit_lorentz.const_val        = 0.5;
    config_parameters->fitting.kern_fac                     = 300.;

    /* Create dummy parameter object */
    mf_parameters *params = mf_parameters_initialize(config_parameters, NULL);

    /* Create a dummy sec table */
    cpl_size   size = 100;
    cpl_table *spec = cpl_table_new(size);
    cpl_table_new_column(spec, MF_COL_IN_LAMBDA, CPL_TYPE_DOUBLE);
    cpl_table_new_column(spec, MF_COL_IN_FLUX, CPL_TYPE_DOUBLE);

    /* Create a dummy fitpar array */
    cpl_size   npar   = 3;
    cpl_array *fitpar = cpl_array_new(npar, CPL_TYPE_DOUBLE);


    /* ---------------------------------- */
    /* Test function mf_kerenel_synthetic */
    /* ---------------------------------- */

    cpl_msg_info(cpl_func, "Testing function mf_kerenel_synthetic");
    err = mf_kernel_synthetic(spec, params, fitpar);

    if (err) {
        cpl_msg_error(cpl_func, "mf_kerenel_synthetic test failed!");
    }
    else {
        cpl_msg_info(cpl_func, "mf_kerenel_synthetic test  ---> PASS");
    }


    /* Cleanup*/
    mf_parameters_delete(params);
    mf_parameters_config_delete(config_parameters);
    cpl_table_delete(spec);
    cpl_array_delete(fitpar);

    /* Show errors and return */
    if (err) {
        cpl_test_end(1);
    }
    return cpl_test_end(0);
}


/** @cond PRIVATE */

/** @endcond */

/**@}*/
