/*
 * This file is part of the ESO Telluric Correction Library
 * Copyright (C) 2001-2018 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/*----------------------------------------------------------------------------*/
/**
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include "mf_constants.h"

#include "mf_parameters.h"
#include "mf_model.h"

#include <cpl_test.h>

/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Enumeration types
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Defines
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Global variables
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Macros
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Structured types
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Functions prototypes
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Program
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @defgroup test_mf_model   .
 *
 * @brief
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/* ---------------------------------------------------------------------------*/
/**
 * @brief .
 *
 * @param .                  .
 * @param                    .
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK.
 *                           If not, these are the errors:
 *                           - .
 *                           - Error in subroutine (see subroutines).
 *
 * @description .
 *
 * @note .
 *
 */
/* ---------------------------------------------------------------------------*/
int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_DEBUG);

    int nfail = 0;

    /* Create configuration */
    cpl_test_error(CPL_ERROR_NONE);

    cpl_msg_info(cpl_func, "Testing mf_molecules");

    cpl_error_code err;


    /* -----------------------------------*/
    /* Test mf_molecules_create_array()   */
    /* -----------------------------------*/
    cpl_array *molecs;
    molecs = mf_molecules_create_array();
    cpl_array_delete(molecs);


    /* ----------------------------------------------------- */
    /* Test mf_molecules_str_check with good and bad samples */
    /* ----------------------------------------------------- */
    const char *molec_str = "H2O";
    const char *badml_str = "X2O";

    /* Check that we pass a good molecule string*/
    err = mf_molecules_str_check(molec_str);
    if (err != CPL_ERROR_NONE) {
        nfail++;
        cpl_msg_error(cpl_func, "Error in checking good molecular string %s", molec_str);
    }

    /* Check that we fail a bad molecule string*/
    err = mf_molecules_str_check(badml_str);
    if (err == CPL_ERROR_NONE) {
        nfail++;
        cpl_msg_error(cpl_func, "Error in checking bad molecular string %s", badml_str);
    }

    /* ----------------------------------------------------------- */
    /* Test molecule routines  with a table of H20,CO2,O2 and CH4  */
    /* ----------------------------------------------------------- */
    /* Create Molecules table */
    cpl_table *table;
    table = cpl_table_new(4); /* Allocate 4 rows */
    cpl_table_new_column(table, MF_COL_LIST_MOLECULES, CPL_TYPE_STRING);
    cpl_table_new_column(table, MF_COL_FIT_MOLECULES, CPL_TYPE_INT);
    cpl_table_new_column(table, MF_COL_REL_COL, CPL_TYPE_DOUBLE);
    cpl_table_set_string(table, MF_COL_LIST_MOLECULES, 0, "H20");
    cpl_table_set_string(table, MF_COL_LIST_MOLECULES, 1, "CO2");
    cpl_table_set_string(table, MF_COL_LIST_MOLECULES, 2, "O2");
    cpl_table_set_string(table, MF_COL_LIST_MOLECULES, 3, "CH4");

    /* Create a molecules object */
    mf_molecules *my_molecules;
    my_molecules             = cpl_malloc(sizeof(mf_molecules));
    my_molecules->n_molec    = MF_MOLECULES_NUMBER_INIT;
    my_molecules->lbl_molecs = cpl_sprintf("%s", MF_MOLECULES_FIT_ARRAY_FLAGS_INIT);

    /* Test mf_molecules_lbl_fill */
    int   *nptr   = &(my_molecules->n_molec);
    char **strptr = &(my_molecules->lbl_molecs);
    err           = mf_molecules_lbl_fill(table, nptr, strptr);
    if (err != CPL_ERROR_NONE) {
        nfail++;
        cpl_msg_error(cpl_func, "Error in testing mf_molecules_lbl_fill");
    }

    /* Test mf_molecules_check */
    err = mf_molecules_check(table);
    if (err != CPL_ERROR_NONE) {
        nfail++;
        cpl_msg_error(cpl_func, "Error in testing mf_molecules_check");
    }


    cpl_table_delete(table);
    cpl_free(my_molecules->lbl_molecs);
    cpl_free(my_molecules);


    /* Show errors and return */
    return cpl_test_end(nfail);
}


/** @cond PRIVATE */

/** @endcond */

/**@}*/
