/*
 * This file is part of the ESO Telluric Correction Library
 * Copyright (C) 2001-2018 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/*----------------------------------------------------------------------------*/
/**
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include "mf_constants.h"

#include "mf_spectrum.h"
#include "mf_spectrum.c"
#include <cpl_test.h>

/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Enumeration types
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Defines
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Global variables
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Macros
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Structured types
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Functions prototypes
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Program
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @defgroup test_mf_spectrum   .
 *
 * @brief
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/* ---------------------------------------------------------------------------*/
/**
 * @brief .
 *
 * @param .                  .
 * @param                    .
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK.
 *                           If not, these are the errors:
 *                           - .
 *                           - Error in subroutine (see subroutines).
 *
 * @description .
 *
 * @note .
 *
 */
/* ---------------------------------------------------------------------------*/
int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_DEBUG);

    /* Initialize */
    cpl_errorstate errstate = cpl_errorstate_get();

    /* Get time */
    double cs = cpl_test_get_walltime();

    /* Create configuration */
    mf_parameters_config *config_parameters = mf_parameters_config_create();
    cpl_test_error(CPL_ERROR_NONE);

    if (config_parameters->inputs.column_lambda) {
        cpl_free(config_parameters->inputs.column_lambda);
    }
    if (config_parameters->inputs.column_flux) {
        cpl_free(config_parameters->inputs.column_flux);
    }
    if (config_parameters->inputs.column_dflux) {
        cpl_free(config_parameters->inputs.column_dflux);
    }
    if (config_parameters->inputs.column_mask) {
        cpl_free(config_parameters->inputs.column_mask);
    }

    //  if (config_parameters->directories.output_name) cpl_free(config_parameters->directories.output_name);

    config_parameters->inputs.column_lambda = cpl_sprintf("%s", "Wavelength");
    config_parameters->inputs.column_flux   = cpl_sprintf("%s", "Extracted_OPT");
    config_parameters->inputs.column_dflux  = cpl_sprintf("%s", "Error_OPT");
    config_parameters->inputs.column_mask   = cpl_sprintf("%s", "NULL");

    config_parameters->inputs.wlg_to_micron = 1e-3;
    //  config_parameters->directories.output_name              = cpl_sprintf("%s", "telluriccorr_test_crires");
    config_parameters->fitting.ftol                         = 1e-2;
    config_parameters->fitting.xtol                         = 1e-2;
    config_parameters->fitting.fit_telescope_background.fit = CPL_FALSE;
    config_parameters->fitting.fit_continuum.n              = 3;
    config_parameters->fitting.fit_wavelenght.n             = 3;
    config_parameters->fitting.fit_res_box.fit              = CPL_FALSE;
    config_parameters->fitting.fit_res_box.const_val        = 0.;
    config_parameters->fitting.fit_lorentz.fit              = CPL_FALSE;
    config_parameters->fitting.fit_lorentz.const_val        = 0.5;
    config_parameters->fitting.kern_fac                     = 300.;
    //config_parameters->fitting.fit_ranges                   = NULL;


    /* Read spectrum */
    char *filename = cpl_sprintf("%s/%s", TELLURICCORR_SOURCEDIR, "input/crires_spec_jitter_extracted_0000.fits");
    cpl_propertylist *header_spec = cpl_propertylist_load(filename, 0);
    cpl_table        *spec        = cpl_table_load(filename, 1, 0);
    cpl_free(filename);
    cpl_test_nonnull(header_spec);
    cpl_test_nonnull(spec);

    /* Update the telluriccorr configuration with the header */
    mf_parameters_config_update_with_header_keywords(config_parameters, header_spec);

    /* Create input telluriccorr spec format */
    cpl_table *spec_telluriccorr = mf_spectrum_create(config_parameters, spec);

    /* Initialize parameters */
    mf_parameters *params = mf_parameters_initialize(config_parameters, NULL);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull(params);

    /* Generate a single row wavelength include ranges */
    double     lam1 = 3.281, lam2 = 3.358;
    cpl_table *wave_ranges_table = cpl_table_new(1);
    cpl_table_new_column(wave_ranges_table, "LOWER_LIMIT", CPL_TYPE_DOUBLE);
    cpl_table_new_column(wave_ranges_table, "UPPER_LIMIT", CPL_TYPE_DOUBLE);
    cpl_table_new_column(wave_ranges_table, "MAPPED_TO_CHIP", CPL_TYPE_INT);
    cpl_table_set(wave_ranges_table, "LOWER_LIMIT", 0, lam1);
    cpl_table_set(wave_ranges_table, "UPPER_LIMIT", 0, lam2);
    cpl_table_set(wave_ranges_table, "MAPPED_TO_CHIP", 0, 1);
    cpl_size size;

    size = cpl_table_get_nrow(params->chiptab);
    if (size == 0) {
        printf("TABLE INITIALLY OF SIZE ZERO");
        cpl_table_set_size(params->chiptab, 1);
    }
    cpl_table_set_double(params->chiptab, "wl_min", 0, lam1);
    cpl_table_set_double(params->chiptab, "wl_max", 0, lam1);
    cpl_table_set_int(params->chiptab, "fit_chip", 0, 0);


    cpl_table_dump(params->chiptab, 0, size, NULL);

    /* Generate a cpl_table telluriccorr spectrum from header and spec data */
    cpl_table *spec_telluriccorr_format =
        mf_spectrum_ranges_apply(params, spec_telluriccorr, wave_ranges_table, NULL, NULL);
    cpl_test_nonnull(spec_telluriccorr_format);
    cpl_test_error(CPL_ERROR_NONE);

    /* Test air to vacuum conversion */
    const char    *frame_type  = MF_PARAMETERS_WAVELENGTH_FRAME_AIR;
    double         lambda0     = cpl_table_get_data_double(spec_telluriccorr_format, MF_COL_IN_LAMBDA)[0];
    cpl_error_code err         = mf_spectrum_air_to_vacuum(frame_type, 0., spec_telluriccorr_format);
    double         new_lambda0 = cpl_table_get_data_double(spec_telluriccorr_format, MF_COL_IN_LAMBDA)[0];
    cpl_test_abs(new_lambda0 - lambda0, 0.000893136, 1.e-9);
    cpl_test_eq(err, CPL_ERROR_NONE);
    cpl_test_error(CPL_ERROR_NONE);

    frame_type  = MF_PARAMETERS_WAVELENGTH_FRAME_VACUUM;  // No action, wavelength already in vacuum
    lambda0     = cpl_table_get_data_double(spec_telluriccorr_format, MF_COL_IN_LAMBDA)[0];
    err         = mf_spectrum_air_to_vacuum(frame_type, 0., spec_telluriccorr_format);
    new_lambda0 = cpl_table_get_data_double(spec_telluriccorr_format, MF_COL_IN_LAMBDA)[0];
    cpl_test_abs(lambda0, new_lambda0, 1.e-9);
    cpl_test_eq(err, CPL_ERROR_NONE);
    cpl_test_error(CPL_ERROR_NONE);

    frame_type  = MF_PARAMETERS_WAVELENGTH_FRAME_AIR_RV;
    lambda0     = cpl_table_get_data_double(spec_telluriccorr_format, MF_COL_IN_LAMBDA)[0];
    err         = mf_spectrum_air_to_vacuum(frame_type, 299792.458, spec_telluriccorr_format);
    new_lambda0 = cpl_table_get_data_double(spec_telluriccorr_format, MF_COL_IN_LAMBDA)[0];
    cpl_test_abs(new_lambda0 - lambda0, -1.6371656646, 1.e-9);
    cpl_test_eq(err, CPL_ERROR_NONE);
    cpl_test_error(CPL_ERROR_NONE);

    /* Test to vacuum to air conversion */
    frame_type  = MF_PARAMETERS_WAVELENGTH_FRAME_VACUUM;
    lambda0     = cpl_table_get_data_double(spec_telluriccorr_format, MF_COL_IN_LAMBDA)[0];
    err         = mf_spectrum_vacuum_to_air(frame_type, 0., spec_telluriccorr_format);
    new_lambda0 = cpl_table_get_data_double(spec_telluriccorr_format, MF_COL_IN_LAMBDA)[0];
    cpl_test_abs(new_lambda0 - lambda0, -0.000447393, 1.e-9);
    cpl_test_eq(err, CPL_ERROR_NONE);
    cpl_test_error(CPL_ERROR_NONE);

    frame_type  = MF_PARAMETERS_WAVELENGTH_FRAME_AIR;  // No action, wavelength already in air
    lambda0     = cpl_table_get_data_double(spec_telluriccorr_format, MF_COL_IN_LAMBDA)[0];
    err         = mf_spectrum_vacuum_to_air(frame_type, 0., spec_telluriccorr_format);
    new_lambda0 = cpl_table_get_data_double(spec_telluriccorr_format, MF_COL_IN_LAMBDA)[0];
    cpl_test_abs(lambda0, new_lambda0, 1.e-9);
    cpl_test_eq(err, CPL_ERROR_NONE);
    cpl_test_error(CPL_ERROR_NONE);

    frame_type  = MF_PARAMETERS_WAVELENGTH_FRAME_VACUUM_RV;
    lambda0     = cpl_table_get_data_double(spec_telluriccorr_format, MF_COL_IN_LAMBDA)[0];
    err         = mf_spectrum_vacuum_to_air(frame_type, 299792.458, spec_telluriccorr_format);
    new_lambda0 = cpl_table_get_data_double(spec_telluriccorr_format, MF_COL_IN_LAMBDA)[0];
    cpl_test_abs(new_lambda0 - lambda0, 1.6367185151, 1.e-9);
    cpl_test_eq(err, CPL_ERROR_NONE);
    cpl_test_error(CPL_ERROR_NONE);

    /* Test to vacuum -> air -> vacuum conversion */
    frame_type = MF_PARAMETERS_WAVELENGTH_FRAME_VACUUM;
    lambda0    = cpl_table_get_data_double(spec_telluriccorr_format, MF_COL_IN_LAMBDA)[0];
    err        = mf_spectrum_vacuum_to_air(frame_type, 0., spec_telluriccorr_format);
    cpl_test_eq(err, CPL_ERROR_NONE);
    frame_type = MF_PARAMETERS_WAVELENGTH_FRAME_AIR;
    err        = mf_spectrum_air_to_vacuum(frame_type, 0., spec_telluriccorr_format);
    cpl_test_eq(err, CPL_ERROR_NONE);
    new_lambda0 = cpl_table_get_data_double(spec_telluriccorr_format, MF_COL_IN_LAMBDA)[0];
    cpl_test_abs(lambda0, new_lambda0, 1.e-13);
    cpl_test_error(CPL_ERROR_NONE);

    /* Test to air -> vacuum -> air conversion */
    frame_type = MF_PARAMETERS_WAVELENGTH_FRAME_AIR;
    lambda0    = cpl_table_get_data_double(spec_telluriccorr_format, MF_COL_IN_LAMBDA)[0];
    err        = mf_spectrum_air_to_vacuum(frame_type, 0., spec_telluriccorr_format);
    cpl_test_eq(err, CPL_ERROR_NONE);
    frame_type = MF_PARAMETERS_WAVELENGTH_FRAME_VACUUM;
    err        = mf_spectrum_vacuum_to_air(frame_type, 0., spec_telluriccorr_format);
    cpl_test_eq(err, CPL_ERROR_NONE);
    new_lambda0 = cpl_table_get_data_double(spec_telluriccorr_format, MF_COL_IN_LAMBDA)[0];
    cpl_test_abs(lambda0, new_lambda0, 1.e-13);
    cpl_test_error(CPL_ERROR_NONE);

    /* Cleanup */
    cpl_propertylist_delete(header_spec);
    cpl_table_delete(spec);
    cpl_table_delete(spec_telluriccorr);
    cpl_table_delete(spec_telluriccorr_format);
    cpl_table_delete(wave_ranges_table);
    mf_parameters_delete(params);
    mf_parameters_config_delete(config_parameters);

    /* Show time */
    double ce = cpl_test_get_walltime();
    cpl_msg_info(cpl_func, "mf_spectrum -> Run time: %g min\n", (ce - cs) / 60.);

    /* Show errors and return */
    cpl_errorstate_dump(errstate, CPL_TRUE, cpl_errorstate_dump_one);
    return cpl_test_end(0);
}


/** @cond PRIVATE */

/** @endcond */

/**@}*/
