import configparser
import itertools
import logging
import os
import shutil
from collections import Counter
from dataclasses import dataclass
from pathlib import Path
from typing import List, Optional, Dict, Set
from uuid import UUID

import panel as pn
from edps import ReportInput
from edps.EDPS import EDPS
from edps.client.FlatOrganization import DatasetsDTO
from edps.client.GraphType import GraphType
from edps.client.JobInfo import JobInfo
from edps.client.ParameterSetDTO import ParameterSetDTO
from edps.client.ProcessingJob import ProcessingJob
from edps.client.ProcessingJobStatus import ProcessingJobStatus, JobStatus
from edps.client.ProcessingRequest import ProcessingRequest
from edps.client.ReportsConfiguration import ReportsConfiguration
from edps.client.RequestParameters import RequestParameters
from edps.client.RunReportsRequestDTO import RunReportsRequestDTO
from edps.client.search import SearchFilter
from edps.generator.association_preference import AssociationPreference
from edps.generator.constants import ASSOCIATION_THRESHOLD
from edps.scripts.client import REPORT_TYPES
from edps.scripts.server import setup_edps

from .reduction_repository import Reduction, ReductionRepository, ReductionConfig, ClassifiedFile

edps_banner = """
'########:'########::'########:::'######::
 ##.....:: ##.... ##: ##.... ##:'##... ##:
 ##::::::: ##:::: ##: ##:::: ##: ##:::..::
 ######::: ##:::: ##: ########::. ######::
 ##...:::: ##:::: ##: ##.....::::..... ##:
 ##::::::: ##:::: ##: ##::::::::'##::: ##:
 ########: ########:: ##::::::::. ######::
........::........:::..::::::::::......:::
"""


@dataclass
class RecipeParameter:
    task: str
    param_name: str
    value: str
    default: str


class EDPSWrapper:
    EDPS_HOME = os.environ.get("HOME") + "/.edps"
    APPLICATION_CONFIG = f"{EDPS_HOME}/application.properties"
    LOGGING_CONFIG = f"{EDPS_HOME}/logging.yaml"

    def __init__(self):
        self.edps: Optional[EDPS] = None
        self.repository = ReductionRepository(self.get_reduction_db_path())
        self.logger = logging.getLogger('EDPSW')

    def get_reduction_db_path(self) -> str:
        config = configparser.ConfigParser()
        config.read(self.APPLICATION_CONFIG)
        base_dir = config.get('executor', 'base_dir')
        return os.path.join(base_dir, 'reductions.json')

    def start(self):
        if self.edps is None:
            self.edps = setup_edps(self.APPLICATION_CONFIG, self.LOGGING_CONFIG)

    def stop(self):
        if self.edps:
            self.edps.shutdown()
            self.edps = None

    def is_running(self) -> bool:
        return self.edps is not None

    @property
    def base_dir(self) -> str:
        return self.edps.configuration.base_dir if self.is_running() else None

    @property
    def package_base_dir(self) -> str:
        return self.edps.configuration.package_base_dir if self.is_running() else None

    def get_installed_pipelines(self) -> List[str]:
        esorex_path = self.edps.configuration.esorex_path
        esorex_path = Path(shutil.which(esorex_path) or esorex_path)
        pipeline_path = esorex_path.parent.parent / 'lib/esopipes-plugins'
        if not pipeline_path.exists():
            pipeline_path = esorex_path.parent.parent / 'lib64/esopipes-plugins'
            if not pipeline_path.exists():
                self.logger.error(f"Pipeline path '{pipeline_path}' does not exist.")
                return []
        return [p.name for p in pipeline_path.iterdir()]

    def workflows(self) -> List[str]:
        return self.edps.list_workflows()

    def get_workflow_path(self, workflow: str) -> Optional[str]:
        return self.edps.workflows.get(workflow)

    def get_meta_targets(self, workflow: str) -> List[str]:
        wkf = self.edps.get_workflow(workflow)
        return list(set(itertools.chain.from_iterable([task.meta_targets for task in wkf.tasks])))

    def get_targets(self, workflow: str, meta_target: str) -> List[str]:
        wkf = self.edps.get_workflow(workflow)
        all_targets = [task.name for task in wkf.tasks]
        return [task.name for task in wkf.tasks if meta_target in task.meta_targets] if meta_target else all_targets

    def classify(self, workflow: str, inputs: List[str]) -> List[ClassifiedFile]:
        request = ProcessingRequest(inputs=inputs, workflow=workflow)
        classified_files = self.edps.classify_files(request)
        return [ClassifiedFile(f.name, f.category) for f in classified_files]

    def create_datasets(self, workflow: str, inputs: List[str], targets: List[str],
                        association_preference: str, association_level: float) -> DatasetsDTO:

        parameters = RequestParameters(workflow_parameters={ASSOCIATION_THRESHOLD: association_level})
        request = ProcessingRequest(workflow=workflow, inputs=inputs, targets=targets,
                                    association_preference=AssociationPreference.from_str(association_preference),
                                    parameters=parameters)
        return self.edps.organise_data_flatten(request)

    def organise(self, workflow: str, inputs: List[str]) -> List[JobInfo]:
        request = ProcessingRequest(workflow=workflow, inputs=inputs)
        return self.edps.organise_data(request)

    def process_reduction(self, reduction: Reduction, package_dir: str = None):
        report_config = ReportsConfiguration(task_names=['ALL'],
                                             report_types=REPORT_TYPES[reduction.config.report_type])
        parameters = RequestParameters(workflow_parameter_set=reduction.config.parameter_set,
                                       workflow_parameters=reduction.config.workflow_parameters,
                                       recipe_parameter_set=reduction.config.parameter_set,
                                       recipe_parameters=reduction.config.recipe_parameters)
        request = ProcessingRequest(workflow=reduction.workflow, inputs=[f.name for f in reduction.input_files],
                                    targets=[reduction.target], meta_targets=[],
                                    parameters=parameters, package_base_dir=package_dir,
                                    reports_configuration=report_config)
        response = self.edps.process_data(request)
        job_ids = [job.job_id for job in response.jobs]
        if job_ids:
            self.set_reduction_jobs(reduction.id, job_ids)

    # def reduce_dataset(self, workflow: str, dataset: LabelledDatasetDTO,
    #                    config: ReductionConfiguration) -> ProcessingResponse:
    #     reports_config = ReportsConfiguration(task_names=['ALL'],
    #                                           report_types=REPORT_TYPES.get(config.report_type, 'reduced'))
    #     parameters = RequestParameters(workflow_parameter_set=config.parameter_set,
    #                                    workflow_parameters=config.workflow_parameters,
    #                                    recipe_parameter_set=config.parameter_set,
    #                                    recipe_parameters=config.recipe_parameters)
    #     target = dataset.dataset.task_name
    #     files = get_dataset_files(dataset.dataset)
    #     request = ProcessingRequest(workflow=workflow, inputs=files, targets=[target], meta_targets=[],
    #                                 parameters=parameters, reports_configuration=reports_config)
    #     # important: reduction time must be antecedent to the jobs' submission time!
    #     reduction_timestamp = datetime.now().isoformat(timespec='milliseconds')
    #     response = self.edps.process_data(request)
    #     reduction = Reduction(timestamp=reduction_timestamp, dataset=dataset.dataset_name, workflow=workflow,
    #                           target=target, input_files=files, parameter_set=config.parameter_set,
    #                           workflow_parameters=config.workflow_parameters,
    #                           recipe_parameters=config.recipe_parameters,
    #                           job_ids=[job.job_id for job in response.jobs], comment=config.comment)
    #     self.repository.add(reduction)
    #
    #     return response

    def monitor_jobs(self, job_ids) -> List[ProcessingJob]:
        jobs = [self.edps.get_job_details(UUID(job_id)) for job_id in job_ids]
        active_jobs = [job for job in jobs if job.status in ('CREATED', 'RUNNING')]
        return jobs if active_jobs else []

    def get_jobs(self) -> List[ProcessingJob]:
        return self.edps.get_jobs_filter(SearchFilter())

    def get_scheduled_jobs(self) -> List[str]:
        return [str(job_id) for job_id in self.edps.get_scheduled_jobs()]

    def get_job_details(self, job_id) -> ProcessingJob:
        return self.edps.get_job_details(UUID(job_id))

    def get_job_status(self, job_id) -> JobStatus:
        return self.edps.get_job_status(UUID(job_id))

    def get_report_panel(self, job_id, panel_name) -> bytes:
        return self.edps.get_job_report(UUID(job_id), panel_name)[1]

    def get_log(self, job_id, log_name) -> str:
        return self.edps.get_job_log(UUID(job_id), log_name).decode()

    def get_simple_graph(self, workflow: str) -> str:
        return self.edps.get_graph(workflow, GraphType.SIMPLE)

    def get_detailed_graph(self, workflow: str) -> str:
        return self.edps.get_graph(workflow, GraphType.DETAILED)

    def get_assoc_map(self, workflow: str) -> str:
        return self.edps.get_assoc_map(workflow)

    def delete_all_jobs(self):
        jobs = self.edps.get_jobs_filter(SearchFilter())
        job_ids = [job.configuration.job_id for job in jobs]
        for job_id in job_ids:
            self.edps.delete_job_cascade(UUID(job_id))

    def get_parameter_sets(self, workflow: str) -> List[ParameterSetDTO]:
        return self.edps.get_parameter_sets(workflow)

    def get_parameter_metadata(self, workflow: str) -> Dict[str, str]:
        for param_set in self.get_parameter_sets(workflow):
            if param_set.default:
                return param_set.parameter_metadata
        return {}

    def get_workflow_parameters(self, workflow: str, param_set_name: str) -> Dict[str, str]:
        for param_set in self.get_parameter_sets(workflow):
            if param_set.name == param_set_name:
                return param_set.workflow_parameters
        return {}

    def get_recipe_parameters(self, workflow: str, param_set_name: str) -> Dict[str, Dict[str, str]]:
        for param_set in self.get_parameter_sets(workflow):
            if param_set.name == param_set_name:
                return param_set.recipe_parameters
        return {}

    def get_default_parameters(self, workflow: str, task: str) -> Dict[str, object]:
        workflow_task = f"{workflow}.{task}"
        if workflow_task not in pn.state.cache:
            try:
                pn.state.cache[workflow_task] = self.edps.get_default_params(workflow, task)
            except Exception as e:
                self.logger.error(e, exc_info=e)
        return pn.state.cache.get(workflow_task, {})

    def get_combined_parameters(self, workflow: str, param_set: str, task: str) -> List[RecipeParameter]:
        result = []
        recipe_parameters = self.get_recipe_parameters(workflow, param_set).get(task, {})
        for param_name, default_value in self.get_default_parameters(workflow, task).items():
            result.append(RecipeParameter(task=task, param_name=param_name, value=recipe_parameters.get(param_name),
                                          default=str(default_value)))
        return result

    def stop_reductions(self):
        self.edps.halt_executions()

    def get_tasks(self, workflow: str) -> List[str]:
        return [task.name for task in self.edps.get_workflow(workflow).tasks]

    def run_reports(self, job_id: str, report_inputs: str):
        report_types_for_inputs = {
            'raw': [ReportInput.RECIPE_INPUTS],
            'reduced': [ReportInput.RECIPE_INPUTS_OUTPUTS],
            'both': [ReportInput.RECIPE_INPUTS, ReportInput.RECIPE_INPUTS_OUTPUTS]
        }
        report_types = report_types_for_inputs.get(report_inputs, [ReportInput.RECIPE_INPUTS_OUTPUTS])
        request = RunReportsRequestDTO(job_ids=[job_id], report_types=report_types)
        self.edps.run_reports(request)

    def get_all_reductions_as_list(self) -> List[Reduction]:
        return self.repository.get_all_as_list()

    def get_all_reductions_as_set(self) -> Set[Reduction]:
        return self.repository.get_all_as_set()

    def get_reduction_by_id(self, reduction_id: str) -> Reduction:
        return self.repository.get_by_id(reduction_id)

    def filter_reductions(self, dataset: str, target: str) -> List[Reduction]:
        return self.repository.get_by_dataset_and_target(dataset, target)

    def update_reduction_comment(self, reduction_id: str, comment: str, mode: str = 'replace'):
        self.repository.update_comment(reduction_id, comment, mode)

    def update_report_type(self, reduction_id: str, report_type: str):
        self.repository.update_report_type(reduction_id, report_type)

    def update_reduction_config(self, reduction_id: str, config: ReductionConfig, update_comment: bool = False):
        self.repository.update_config(reduction_id, config, update_comment)

    def clone_reduction(self, reduction: Reduction, new_config: ReductionConfig) -> Optional[Reduction]:
        return self.repository.clone(reduction, new_config)

    def delete_reduction(self, reduction_id: str) -> Reduction:
        deleted = self.repository.delete(reduction_id)
        self.edps.delete_independent_jobs_subset([UUID(job_id) for job_id in deleted.job_ids])
        return deleted

    def add_reduction(self, reduction: Reduction):
        self.repository.add(reduction)

    def archive_reduction(self, reduction_id: str):
        self.repository.set_archived(reduction_id, True)

    def unarchive_reduction(self, reduction_id: str):
        self.repository.set_archived(reduction_id, False)

    def set_reduction_jobs(self, reduction_id: str, job_ids: List[str]):
        self.repository.set_jobs(reduction_id, job_ids)

    def get_job_status_str(self, job_id: str) -> str:
        job_status = self.get_job_status(job_id)
        if job_status.interrupted:
            return "ABORTED"
        if job_status.status == ProcessingJobStatus.CREATED:
            return "PENDING"
        return job_status.status.value

    def get_detailed_reduction_status(self, reduction: Reduction) -> Dict[str, int]:
        if not reduction.job_ids:
            return {"NEW": 0}
        if reduction.completed:
            return {"COMPLETED": len(reduction.job_ids)}
        status_counter = Counter(self.get_job_status_str(job_id) for job_id in reduction.job_ids)
        if set(status_counter) == {"COMPLETED"} and not reduction.completed:
            self.repository.set_completed(reduction.id)
        return status_counter

    def get_reduction_status(self, reduction: Reduction) -> str:
        status = self.get_detailed_reduction_status(reduction)
        if len(status) == 1:
            return list(status)[0]
        elif "RUNNING" in status:
            return "RUNNING"
        elif "ABORTED" in status:
            return "ABORTED"
        elif "FAILED" in status:
            return "FAILED"
        elif "MISSING" in status:
            return "MISSING"
        elif "PENDING" in status:
            return "PENDING"
        else:
            return "UNKNOWN"
