import io
import os
import subprocess
from dataclasses import dataclass
from datetime import datetime
from typing import List, Dict, Tuple

import pandas as pd
from astropy.io import fits
from astropy.table import Table

MAX_TIMESTAMP_US = datetime.max.isoformat(timespec='microseconds')
MAX_TIMESTAMP_MS = datetime.max.isoformat(timespec='milliseconds')


@dataclass
class HDUInfo:
    index: int
    name: str
    type: str
    shape: Tuple[int, ...]
    prodcatg: str


def get_fits_info_old(filename: str) -> Tuple[str, int]:
    output = io.StringIO()
    with fits.open(filename) as hdul:
        hdul.info(output)
        num_hdus = len(hdul)
    file_info = output.getvalue()
    output.close()
    return file_info, num_hdus


def get_fits_info(filename: str) -> List[HDUInfo]:
    with fits.open(filename) as hdul:
        prodcatg = hdul[0].header.get('PRODCATG', '')
        return [
            HDUInfo(
                index=i,
                name=hdu.name,
                type=hdu.__class__.__name__,
                shape=hdu.data.shape if hdu.data is not None else (),
                prodcatg=prodcatg
            )
            for i, hdu in enumerate(hdul)
        ]


def get_fits_header(filename: str, hdunum: int = 0) -> fits.Header:
    with fits.open(filename) as hdul:
        return hdul[hdunum].header


def get_keywords(filename: str, keywords: List[str]) -> Dict[str, object]:
    keyword_values = {}
    with fits.open(filename) as hdul:
        for kw in keywords:
            keyword_values[kw] = hdul[0].header.get(kw)
    return keyword_values


def get_table(filename: str, hdunum: int) -> pd.DataFrame:
    with fits.open(filename) as hdul:
        table = Table.read(hdul[hdunum])
        column_shapes = [table[col].shape for col in table.colnames]
        if all(len(shape) <= 1 for shape in column_shapes):
            return table.to_pandas()
        else:
            # SDP format
            return pd.DataFrame({col: table[col][0].astype('float32') for col in table.colnames})


def open_file_in_fv(filename: str):
    fv = os.environ.get('EDPSGUI_FV') or 'fv'
    subprocess.Popen(args=[fv, filename],
                     start_new_session=True,
                     stdout=subprocess.DEVNULL,
                     stderr=subprocess.DEVNULL)


def open_file_in_ds9(filename: str):
    ds9 = os.environ.get('EDPSGUI_DS9') or 'ds9'
    subprocess.Popen(args=[ds9, filename],
                     start_new_session=True,
                     stdout=subprocess.DEVNULL,
                     stderr=subprocess.DEVNULL)
