import copy
from typing import Dict, List, Tuple

import pandas as pd
import panel as pn
import panel_material_ui as pmui
import param
from panel.viewable import Viewer, Viewable

from edpsgui.domain.reduction_repository import Reduction, ReductionConfig
from .edps_ctl import get_edps
from .tooltips import COMMENT_TOOLTIP, CONFIGURATION_TOOLTIP

WorkflowParameters = Dict[str, str]
RecipeParameters = Dict[str, Dict[str, str]]

COLUMNS = ['Dataset', 'Target', 'Object', 'Configuration Date', 'Status']


def flatten_recipe_parameters(recipe_parameters: RecipeParameters) -> List[Tuple[str, str, str]]:
    """
    Flatten the recipe parameters dictionary into a list of tuples.
    Each tuple contains (task, parameter, value).
    """
    flattened = []
    for task, params in recipe_parameters.items():
        for param_name, value in params.items():
            flattened.append((task, param_name, value))
    return flattened


class ReductionConfigurationEditor(Viewer):
    parameter_set = param.Selector(default=[])
    task = param.Selector(default=[])
    selected_workflow_param = param.String(allow_None=True)
    workflow_param_df = param.DataFrame()
    recipe_param_df = param.DataFrame()
    custom_workflow_parameters = param.Dict(default={})
    custom_recipe_parameters = param.Dict(default={})
    comment = param.String()
    reduction_updated = param.Event()

    def __init__(self, reduction: Reduction, reductions_df: pd.DataFrame, **params):
        super().__init__(**params)
        self.edps = get_edps()
        self.reduction = reduction
        self.status = self.edps.get_reduction_status(self.reduction)
        sorters = [{'field': 'Configuration Date', 'dir': 'desc'}]
        editors = {
            'Dataset': None,
            'Target': None,
            'Object': None,
            'Configuration Date': None,
            'Status': None,
            'Comment': None,
            'Parameters': None,
        }
        formatters = {
            'Comment': 'textarea',
            'Parameters': 'textarea',
        }
        widths = {
            'Comment': 150,
            'Parameters': 150,
        }
        other_reductions_df = reductions_df[reductions_df['ID'] != reduction.id]
        self.current_configuration = pn.widgets.Tabulator(self.current_reduction_df, show_index=False,
                                                          hidden_columns=['ID', 'Comment', 'Parameters'],
                                                          editors=editors, formatters=formatters, widths=widths,
                                                          selectable='checkbox', selection=[0])
        self.other_configurations = pn.widgets.Tabulator(other_reductions_df, show_index=False,
                                                         hidden_columns=['ID', 'Comment', 'Parameters'],
                                                         header_filters=True, editors=editors, formatters=formatters,
                                                         sorters=sorters, groupby=['Dataset', 'Target', 'Object'],
                                                         widths=widths, pagination='local', page_size=10,
                                                         selectable='checkbox')
        self.comment_update_mode = pn.widgets.RadioBoxGroup(options=['append', 'replace'], value='append')
        config = self.reduction.config
        self.comment = config.comment
        self.workflow = self.reduction.workflow
        self.updating_recipe_parameters = pn.indicators.LoadingSpinner(value=False, size=20, visible=False)
        self.custom_workflow_parameters = copy.deepcopy(config.workflow_parameters)
        self.custom_recipe_parameters = copy.deepcopy(config.recipe_parameters)
        self.parameter_metadata: Dict[str, str] = dict()

        self.report_selector = pn.widgets.RadioBoxGroup(options={'Raw data': 'raw',
                                                                 'Reduced data': 'reduced',
                                                                 'All reports': 'all',
                                                                 'No reports': 'none'},
                                                        value=config.report_type,
                                                        inline=True)

        parameter_sets = [ps.name for ps in self.edps.get_parameter_sets(self.workflow)]
        self.parameter_set = config.parameter_set if config.parameter_set in parameter_sets else None
        self.parameter_set_selector = pn.widgets.Select.from_param(self.param.parameter_set,
                                                                   name='Parameter set',
                                                                   align='start', width=200,
                                                                   options=parameter_sets)

        sorted_tasks = sorted(self.reduction.tasks)
        self.task = sorted_tasks[0] if sorted_tasks else None
        self.task_selector = pn.widgets.Select.from_param(self.param.task,
                                                          name='Task',
                                                          align='start', width=200,
                                                          options=sorted_tasks)

        self.workflow_parameter_table = self.create_workflow_parameters_table()
        self.recipe_parameter_table = self.create_recipe_parameters_table()
        self.status_alert = pn.pane.Alert(alert_type='info')
        self.config_alert = pn.pane.Alert(visible=False)
        self.comment_alert = pn.pane.Alert(visible=False)

    @pn.depends('comment')
    def save_comment_disabled(self) -> bool:
        save_comment_enabled = self.comment != self.reduction.config.comment
        if save_comment_enabled:
            self.comment_alert.visible = False
        return not save_comment_enabled

    @pn.depends('reduction_updated', 'parameter_set', 'custom_workflow_parameters', 'custom_recipe_parameters')
    def save_config_disabled(self) -> bool:
        save_config_enabled = self.status == "NEW" and self.get_config() != self.reduction.config
        if save_config_enabled:
            self.config_alert.visible = False
        return not save_config_enabled

    @pn.depends('reduction_updated', 'parameter_set', 'custom_workflow_parameters', 'custom_recipe_parameters')
    def save_as_new_config_disabled(self) -> bool:
        save_config_enabled = self.get_config() != self.reduction.config
        if save_config_enabled:
            self.config_alert.visible = False
        return not save_config_enabled

    @pn.depends('other_configurations.selection')
    def no_configurations_selected(self) -> bool:
        return not self.other_configurations.selection

    @pn.depends('other_configurations.selection')
    def no_new_configurations_selected(self) -> bool:
        df = self.other_configurations.value.iloc[self.other_configurations.selection]
        return df.empty or df[df['Status'] == 'NEW'].empty

    def get_selected_reduction_ids(self) -> List[str]:
        return self.other_configurations.value.iloc[self.other_configurations.selection]['ID'].to_list()

    @staticmethod
    def create_parameters_table(df, callback) -> pn.widgets.Tabulator:
        editors = {
            'Parameter': None,
            'Default': None,
        }
        titles = {
            'Default': 'Default value',
            'Custom': 'Custom value',
        }
        formatters = {
            'Default': 'textarea',
            'Custom': 'textarea',
        }
        widths = {
            'Default': 300,
            'Custom': 300,
        }
        table = pn.widgets.Tabulator.from_param(df, show_index=False, theme='default', editors=editors, titles=titles,
                                                formatters=formatters, widths=widths)
        table.on_edit(callback)
        return table

    def create_workflow_parameters_table(self) -> pn.widgets.Tabulator:
        table = self.create_parameters_table(self.param.workflow_param_df, self.edit_workflow_parameter)
        table.on_click(
            lambda event: setattr(self, 'selected_workflow_param', event.value)
        )
        return table

    def create_recipe_parameters_table(self) -> pn.widgets.Tabulator:
        return self.create_parameters_table(self.param.recipe_param_df, self.edit_recipe_parameter)

    def get_workflow_parameters(self) -> Dict[str, str]:
        return {param: value for param, value in self.custom_workflow_parameters.items() if value}

    def get_recipe_parameters(self) -> Dict[str, Dict[str, str]]:
        result = {}
        for task in self.custom_recipe_parameters:
            result[task] = {param: value for param, value in self.custom_recipe_parameters[task].items() if value}
            if len(result[task]) == 0:
                result.pop(task)
        return result

    def get_config(self) -> ReductionConfig:
        return ReductionConfig(comment=self.comment,
                               report_type=self.report_selector.value,
                               parameter_set=self.parameter_set,
                               workflow_parameters=self.get_workflow_parameters(),
                               recipe_parameters=self.get_recipe_parameters())

    @pn.depends('parameter_set', watch=True)
    def update_workflow_parameter_table(self):
        if self.parameter_set:
            workflow_parameters = self.edps.get_workflow_parameters(self.workflow, self.parameter_set)
            columns = ['Parameter', 'Default', 'Custom']
            data = [(parameter, value, self.custom_workflow_parameters.get(parameter)) for
                    parameter, value in workflow_parameters.items()]
            self.workflow_param_df = pd.DataFrame(data, columns=columns)
            self.parameter_metadata = self.edps.get_parameter_metadata(self.workflow)

    def set_loading_spinner(self, is_active: bool):
        self.updating_recipe_parameters.name = f'Loading recipe parameters for task {self.task}' if is_active else ''
        self.updating_recipe_parameters.value = is_active
        self.updating_recipe_parameters.visible = is_active

    def should_update_recipe_parameters(self) -> bool:
        return self.parameter_set and self.task and self.task in self.edps.get_tasks(self.workflow)

    @pn.depends('parameter_set', 'task', watch=True)
    def update_recipe_parameter_table(self):
        if self.should_update_recipe_parameters():
            self.set_loading_spinner(True)
            columns = ['Parameter', 'Default', 'Custom']
            self.recipe_param_df = pd.DataFrame([], columns=columns)
            recipe_parameters = self.edps.get_combined_parameters(self.workflow, self.parameter_set, self.task)
            data = [(rp.param_name,
                     rp.value or rp.default,
                     self.custom_recipe_parameters.get(self.task, {}).get(rp.param_name)) for
                    rp in recipe_parameters]
            self.recipe_param_df = pd.DataFrame(data, columns=columns)
            self.set_loading_spinner(False)

    def edit_workflow_parameter(self, event):
        parameter = self.workflow_parameter_table.value['Parameter'][event.row]
        # Create a new dictionary to trigger the update event
        new_custom_workflow_parameters = copy.deepcopy(self.custom_workflow_parameters)
        new_custom_workflow_parameters[parameter] = event.value
        self.custom_workflow_parameters = new_custom_workflow_parameters

    def edit_recipe_parameter(self, event):
        parameter = self.recipe_parameter_table.value['Parameter'][event.row]
        # Create a new dictionary to trigger the update event
        new_custom_recipe_parameters = copy.deepcopy(self.custom_recipe_parameters)
        if self.task not in new_custom_recipe_parameters:
            new_custom_recipe_parameters[self.task] = dict()
        new_custom_recipe_parameters[self.task][parameter] = event.value
        self.custom_recipe_parameters = new_custom_recipe_parameters

    @pn.depends('selected_workflow_param')
    def workflow_parameter_description(self):
        parameter = self.selected_workflow_param
        if not parameter:
            return pn.pane.Markdown("_Click on a parameter to view its description_")
        else:
            description = self.parameter_metadata.get(parameter, 'No description available.')
            return pn.pane.Markdown(f"**{parameter}**: {description}")

    def save_comment(self, event):
        old_comment = self.reduction.config.comment
        new_comment = self.comment
        if new_comment != old_comment:
            self.edps.update_reduction_comment(self.reduction.id, new_comment)
            self.param.trigger('comment')
            self.comment_alert.object = f"Comment updated from '{old_comment}' to '{new_comment}'."
            self.comment_alert.alert_type = 'success'
        else:
            self.comment_alert.object = "No changes made to the comment."
            self.comment_alert.alert_type = 'warning'
        self.comment_alert.visible = True

    def copy_comment(self, event):
        selected_ids = self.get_selected_reduction_ids()
        self.other_configurations.selection = []
        for reduction_id in selected_ids:
            self.edps.update_reduction_comment(reduction_id, self.comment, mode=self.comment_update_mode.value)
        num_selected = len(selected_ids)
        self.comment_alert.object = f"Updated comment for {num_selected} selected configurations: '{self.comment}'."
        self.comment_alert.alert_type = 'success'
        self.comment_alert.visible = True

    def save_configuration(self, event):
        old_config = self.reduction.config
        new_config = self.get_config()
        if new_config != old_config:
            self.edps.update_reduction_config(self.reduction.id, new_config)
            self.param.trigger('reduction_updated')
            self.config_alert.object = "Configuration updated."
            self.config_alert.alert_type = 'success'
        else:
            self.config_alert.object = "No changes made to the configuration."
            self.config_alert.alert_type = 'warning'
        self.config_alert.visible = True

    def save_as_new_configuration(self, event):
        old_config = self.reduction.config
        new_config = self.get_config()
        if new_config != old_config:
            new_reduction = self.edps.clone_reduction(self.reduction, new_config)
            if new_reduction:
                self.reduction = new_reduction
                self.status = self.edps.get_reduction_status(self.reduction)
                self.param.trigger('reduction_updated')
                message = f"New configuration created with creation date {new_reduction.config.timestamp}."
                message_type = 'success'
            else:
                message = "Failed to create a new configuration with the same parameters as an existing one."
                message_type = 'danger'
        else:
            message = "A new configuration cannot be created with the same parameters as the current one."
            message_type = 'warning'
        self.config_alert.object = message
        self.config_alert.alert_type = message_type
        self.config_alert.visible = True

    def copy_configuration(self, event):
        selected_ids = self.get_selected_reduction_ids()
        self.other_configurations.selection = []
        num_updated = 0
        num_created = 0
        new_config = self.get_config()
        for reduction_id in selected_ids:
            reduction = self.edps.get_reduction_by_id(reduction_id)
            reduction_status = self.edps.get_reduction_status(reduction)
            if reduction_status == "NEW":
                self.edps.update_reduction_config(reduction_id, new_config)
                num_updated += 1
            else:
                new_reduction = self.edps.clone_reduction(reduction, new_config)
                if new_reduction:
                    num_created += 1
        self.config_alert.object = f"Updated configuration for {num_updated} selected configurations. " \
                                   f"Created {num_created} new configurations for non-NEW selected configurations."
        self.config_alert.alert_type = 'success'
        self.config_alert.visible = True

    @pn.depends('reduction_updated')
    def current_reduction_df(self) -> pd.DataFrame:
        return pd.DataFrame(data=[(
            self.reduction.dataset,
            self.reduction.target,
            self.reduction.obs_target,
            self.reduction.config.timestamp,
            self.status
        )], columns=COLUMNS)

    @pn.depends('custom_workflow_parameters')
    def custom_workflow_parameters_table(self):
        workflow_parameters = [f'- {param_name}: {value}' for param_name, value in
                               self.get_workflow_parameters().items()]
        if workflow_parameters:
            newline = '\n'
            return f"""
#### Custom Workflow Parameters
{newline.join(workflow_parameters)}
"""

    @pn.depends('custom_recipe_parameters')
    def custom_recipe_parameters_table(self):
        recipe_parameters = [f'- {task}.{param_name}: {value}' for task, param_name, value in
                             flatten_recipe_parameters(self.get_recipe_parameters())]
        if recipe_parameters:
            newline = '\n'
            return f"""
#### Custom Recipe Parameters
{newline.join(recipe_parameters)}
"""

    def __panel__(self) -> Viewable:
        comment_input = pmui.TextAreaInput.from_param(self.param.comment, sizing_mode='stretch_width')
        save_comment_btn = pn.widgets.Button(name='Save', button_type='primary', disabled=self.save_comment_disabled)
        save_comment_btn.on_click(self.save_comment)
        copy_comment_btn = pn.widgets.Button(name='Copy to selected configurations', button_type='primary',
                                             disabled=self.no_configurations_selected)
        copy_comment_btn.on_click(self.copy_comment)

        save_btn = pn.widgets.Button(name='Save', button_type='primary', disabled=self.save_config_disabled)
        save_btn.on_click(self.save_configuration)
        save_as_new_btn = pn.widgets.Button(name='Save as new configuration', button_type='primary',
                                            disabled=self.save_as_new_config_disabled)
        save_as_new_btn.on_click(self.save_as_new_configuration)
        copy_config_btn = pn.widgets.Button(name='Copy to selected configurations', button_type='primary',
                                            disabled=self.no_configurations_selected)
        copy_config_btn.on_click(self.copy_configuration)
        comment_tooltip = pn.widgets.TooltipIcon(value=COMMENT_TOOLTIP)
        configuration_tooltip = pn.widgets.TooltipIcon(value=CONFIGURATION_TOOLTIP)
        current_configuration = pn.Column(
            self.current_configuration,
        )
        comment = pn.Column(
            comment_input,
            pn.Row(save_comment_btn, copy_comment_btn, self.comment_update_mode, comment_tooltip, self.comment_alert)
        )
        parameters = pn.Column(
            self.parameter_set_selector,
            pmui.Typography('Workflow parameters', variant='h5'),
            self.workflow_parameter_table,
            self.workflow_parameter_description,
            pmui.Typography('Recipe parameters', variant='h5'),
            pn.Row(self.task_selector, self.updating_recipe_parameters),
            self.recipe_parameter_table,
            pn.Row(save_btn, save_as_new_btn, copy_config_btn, configuration_tooltip, self.config_alert),
        )
        current_configuration_header = pn.Row(
            pmui.Typography('Current Configuration', variant='h5'),
            pn.widgets.TooltipIcon(value="Displays the currently selected dataset configuration. "
                                         "Click to expand or collapse this section.")
        )
        other_configurations_header = pn.Row(
            pmui.Typography('Other Configurations', variant='h5'),
            pn.widgets.TooltipIcon(
                value="Displays other dataset configurations (excluding the currently selected one). "
                      "You can select one or more configurations from the table, "
                      "then use the buttons below to apply the comment or parameters to them.")
        )
        comment_header = pn.Row(
            pmui.Typography('Comment', variant='h5'),
            pn.widgets.TooltipIcon(value="Edit the comment for the current configuration. "
                                         "You can also apply this comment to one or more configurations "
                                         "selected in the <i>Other Configurations</i> section.")
        )
        parameters_header = pn.Row(
            pmui.Typography('Parameters', variant='h5'),
            pn.widgets.TooltipIcon(value="Edit the parameters for the current configuration. "
                                         "You can also apply these parameters to one or more configurations "
                                         "selected in the <i>Other Configurations</i> section.")
        )
        accordion = pmui.Accordion(
            (current_configuration_header, current_configuration),
            (other_configurations_header, self.other_configurations),
            (comment_header, comment),
            (parameters_header, parameters),
            active=[0], title_variant='h5'
        )
        return pn.Column(
            pmui.Typography('Dataset Configuration Editor', variant='h2'),
            # pn.Row(self.custom_workflow_parameters_table, self.custom_recipe_parameters_table),
            accordion,
            width=1000, height=700, scroll=True
        )
