import logging
import textwrap
import threading
from collections import Counter

import pandas as pd
import panel as pn
import panel_material_ui as pmui
import param
import time
from panel.viewable import Viewer, Viewable
from time import sleep

from .edps_ctl import get_edps

COLUMNS = ['Dataset', 'Configuration Date', 'Pending', 'Running', 'Completed', 'Aborted', 'Failed']


class ReductionMonitor(Viewer):
    edps_status = param.Boolean(default=None, allow_refs=True)
    workflow = param.String(default=None, allow_refs=True)
    reduction_df = param.DataFrame(default=pd.DataFrame(columns=COLUMNS))

    def __init__(self, **params):
        super().__init__(**params)
        self.logger = logging.getLogger('ReductionMonitor')
        self.edps = get_edps()
        self.stop_reduction_btn = pn.widgets.Button(name='Stop All Reductions', button_type='danger',
                                                    width=200, height=50, align='center', icon='skull',
                                                    disabled=self.reduction_not_running)
        self.stop_reduction_btn.on_click(self.stop_reduction)

        self.pending = pn.indicators.Number(name='Pending', value=0, font_size='32pt', default_color='orange')
        self.running = pn.indicators.Number(name='Running', value=0, font_size='32pt', default_color='blue')
        self.completed = pn.indicators.Number(name='Completed', value=0, font_size='32pt', default_color='green')
        self.aborted = pn.indicators.Number(name='Aborted', value=0, font_size='32pt', default_color='grey')
        self.failed = pn.indicators.Number(name='Failed', value=0, font_size='32pt', default_color='red')
        self.monitor_table = pn.widgets.Tabulator.from_param(self.param.reduction_df, show_index=False, disabled=True,
                                                             pagination='local', page_size=20)
        self.keep_running = True
        threading.Thread(target=self.periodic_update, daemon=True).start()
        pn.state.on_session_destroyed(self.stop_periodic_update)

    @pn.depends('reduction_df')
    def reduction_not_running(self):
        return self.reduction_df.empty

    def stop_reduction(self, event):
        with self.stop_reduction_btn.param.update(loading=True, disabled=True):
            self.edps.stop_reductions()

    def stop_periodic_update(self, session_context):
        self.keep_running = False

    def periodic_update(self):
        num_updates = 0
        total_time = 0.
        while self.keep_running:
            if self.edps.is_running():
                scheduled_jobs = set(self.edps.get_scheduled_jobs())
                if scheduled_jobs:
                    start = time.perf_counter()
                    self.update_monitors(scheduled_jobs)
                    total_time += time.perf_counter() - start
                    num_updates += 1
                else:
                    if not self.reduction_df.empty:
                        self.reset_monitors()
            else:
                if not self.reduction_df.empty:
                    self.reset_monitors()
            sleep(5)
        self.logger.info(
            'Stopping periodic update thread. Performed %d updates in %.2f s (average %.3f s per update)',
            num_updates, total_time, total_time / num_updates if num_updates > 0 else 0
        )

    def reset_monitors(self):
        self.reduction_df = pd.DataFrame(columns=COLUMNS)
        self.pending.value = 0
        self.running.value = 0
        self.completed.value = 0
        self.aborted.value = 0
        self.failed.value = 0

    def update_monitors(self, scheduled_jobs):
        scheduled_reductions = [reduction for reduction in self.edps.get_all_reductions_as_list()
                                if set(reduction.job_ids).intersection(scheduled_jobs)]
        reduction_status_counters = [self.edps.get_detailed_reduction_status(r) for r in scheduled_reductions]
        self.reduction_df = pd.DataFrame(
            data=[(reduction.dataset,
                   reduction.config.timestamp,
                   status_counter.get('PENDING', 0),
                   status_counter.get('RUNNING', 0),
                   status_counter.get('COMPLETED', 0),
                   status_counter.get('ABORTED', 0),
                   status_counter.get('FAILED', 0))
                  for reduction, status_counter in zip(scheduled_reductions, reduction_status_counters)],
            columns=COLUMNS
        )
        job_status_counter = Counter(self.edps.get_job_status_str(job_id) for job_id in scheduled_jobs)
        self.pending.value = job_status_counter.get('PENDING', 0)
        self.running.value = job_status_counter.get('RUNNING', 0)
        self.completed.value = job_status_counter.get('COMPLETED', 0)
        self.aborted.value = job_status_counter.get('ABORTED', 0)
        self.failed.value = job_status_counter.get('FAILED', 0)

    def __panel__(self) -> Viewable:
        counters = pn.Row(
            self.pending, pn.Spacer(width=50),
            self.running, pn.Spacer(width=50),
            self.completed, pn.Spacer(width=50),
            self.aborted, pn.Spacer(width=50),
            self.failed,
        )
        tooltip = textwrap.dedent("""
            The Real Time Job Execution Monitor displays the status of ongoing data reduction tasks in real-time.
            It provides an overview of the number of jobs that are pending, running, completed, aborted, or failed.
            Click to expand or collapse this section.
        """)
        header = pn.Row(
            pmui.Typography('Real Time Job Execution Monitor', variant='h5'),
            pn.widgets.TooltipIcon(value=tooltip)
        )
        return pn.Column(
            self.stop_reduction_btn,
            pn.Card(
                counters, self.monitor_table,
                header=header,
                collapsed=True,
                sizing_mode='stretch_width',
            ),
        )
