import asyncio
import logging
import os
import threading
import time
from pathlib import Path
from time import sleep
from typing import List

import pandas as pd
import panel as pn
import param
from panel.viewable import Viewer, Viewable
from panel.widgets import Tqdm

from edpsgui.domain.reduction_repository import Reduction
from edpsgui.gui.edps_ctl import get_edps
from edpsgui.gui.file_table import FileTable
from edpsgui.gui.job_table import JobTable
from edpsgui.gui.job_viewer import JobViewer
from edpsgui.gui.reduction_config_editor import (ReductionConfigurationEditor, flatten_recipe_parameters,
                                                 WorkflowParameters, RecipeParameters)
from edpsgui.gui.reduction_summary import reduction_summary_viewer

status_color = {
    'PENDING': 'orange',
    'RUNNING': 'blue',
    'COMPLETED': 'green',
    'ABORTED': 'grey',
    'FAILED': 'red',
    'NEW': 'purple'
}

COLUMNS = ['ID', 'Dataset', 'Target', 'Object', 'Configuration Date', 'Status', 'Comment', 'Parameters']


class ReductionTable(Viewer):
    edps_status = param.Boolean(default=None, allow_refs=True)
    workflow = param.String(default=None, allow_refs=True)
    output_dir = param.String()
    ALL_BUTTONS = {
        "view": "<i class='fa-regular fa-file-lines' title='View reduction details'></i>",
        "view_target": "<i class='fa-solid fa-magnifying-glass' title='View target job'></i>",
        "run": "<i class='fa-solid fa-play' title='Run this reduction'></i>",
        "archive": "<i class='fa-solid fa-file-export' title='Archive this reduction'></i>",
        "unarchive": "<i class='fa-solid fa-file-import' title='Unarchive this reduction'></i>",
        "configure": "<i class='fa-solid fa-gear' title='Configure this reduction'></i>",
    }

    def __init__(self, **params):
        super().__init__(**params)
        self.logger = logging.getLogger('ReductionTable')
        self.edps = get_edps()
        self.job_icon = False
        sorters = [{'field': 'Configuration Date', 'dir': 'desc'}]
        editors = {
            'ID': None,
            'Dataset': None,
            'Target': None,
            'Object': None,
            'Configuration Date': None,
            'Status': None,
            'Parameters': None,
        }
        formatters = {
            'Comment': 'textarea',
            'Parameters': 'textarea',
        }
        widths = {
            'Comment': 150,
            'Parameters': 150,
        }
        self.reduction_table = pn.widgets.Tabulator(pd.DataFrame(columns=COLUMNS), show_index=False,
                                                    hidden_columns=['ID'], header_filters=True,
                                                    buttons=self.buttons(), editors=editors, sorters=sorters,
                                                    row_content=self.reduction_jobs,
                                                    groupby=['Dataset', 'Target', 'Object'],
                                                    pagination='local', page_size=20, selectable='checkbox',
                                                    formatters=formatters, widths=widths, layout='fit_data',
                                                    stylesheets=[":host .tabulator {font-size: 13px;}"],
                                                    configuration={
                                                        'columnDefaults': {
                                                            'tooltip': True
                                                        },
                                                    })
        self.reduction_table.on_click(self.on_table_clicked)
        self.reduction_table.on_edit(self.edit_reduction_comment)
        self.update_selector = pn.widgets.RadioBoxGroup(options={'All workflows': 'all',
                                                                 'Selected workflow': 'selected'},
                                                        value='selected', align='center')
        self.update_selector.param.watch(self.update_table, 'value')
        self.collapse_btn = pn.widgets.Button(name='Collapse', button_type='primary', disabled=self.all_collapsed)
        self.collapse_btn.on_click(self.collapse_rows)
        self.delete_btn = pn.widgets.Button(name='Delete', button_type='danger', disabled=self.empty_selection)
        self.delete_btn.on_click(self.delete_reductions)
        self.close_btn = pn.widgets.Button(name='Close', button_type='primary')
        self.close_btn.on_click(self.close_modal)
        self.new = pn.indicators.Number(name='New', font_size='12pt', title_size='14pt', default_color='purple')
        self.pending = pn.indicators.Number(name='Pending', font_size='12pt', title_size='14pt', default_color='orange')
        self.running = pn.indicators.Number(name='Running', font_size='12pt', title_size='14pt', default_color='blue')
        self.completed = pn.indicators.Number(name='Completed', font_size='12pt', title_size='14pt',
                                              default_color='green')
        self.aborted = pn.indicators.Number(name='Aborted', font_size='12pt', title_size='14pt', default_color='grey')
        self.failed = pn.indicators.Number(name='Failed', font_size='12pt', title_size='14pt', default_color='red')
        self.tqdm = Tqdm()
        self.layout = None
        self.modal = None
        self.keep_running = True
        threading.Thread(target=self.periodic_update, daemon=True).start()
        pn.state.on_session_destroyed(self.stop_periodic_update)
        self.job_tables = {}

    @pn.depends('edps_status', watch=True)
    def reset_table_when_edps_not_running(self):
        if not self.edps.is_running():
            self.reduction_table.value = pd.DataFrame(columns=COLUMNS)

    def buttons(self):
        # This method should be implemented in subclasses
        raise NotImplementedError

    def is_selected(self, reduction: Reduction) -> bool:
        # This method should be implemented in subclasses
        raise NotImplementedError

    def create_layout(self) -> pn.Column:
        # This method should be implemented in subclasses
        raise NotImplementedError

    def open_modal(self, modal_content) -> pn.Modal:
        modal = pn.Modal(modal_content)
        self.layout.pop(-1)
        self.layout.append(modal)
        modal.open = True
        return modal

    def close_modal(self, event):
        if self.modal:
            self.modal.open = False
            self.modal = None

    @pn.depends('reduction_table.selection')
    def empty_selection(self):
        return not self.reduction_table.selection

    @pn.depends('reduction_table.expanded')
    def all_collapsed(self):
        return not self.reduction_table.expanded

    @pn.depends('workflow', watch=True)
    def update_table_when_workflow_changes(self):
        if self.workflow:
            self.update_table()

    @pn.depends('reduction_table.value', 'reduction_table.selection')
    def reduction_counters(self):
        num_reductions = len(self.reduction_table.value)
        num_selected = len(self.reduction_table.selection)
        return pn.Row(
            pn.indicators.Number(name='Reductions', value=num_reductions, font_size='12pt', title_size='14pt',
                                 default_color='black'),
            pn.indicators.Number(name='Selected', value=num_selected, font_size='12pt', title_size='14pt',
                                 default_color='blue'),
        )

    def get_selected_reductions(self) -> List[str]:
        if self.reduction_table.selection:
            return self.reduction_table.value.iloc[self.reduction_table.selection]['ID'].to_list()
        else:
            return []

    def select_new_reductions(self):
        df = self.reduction_table.current_view
        if df.empty:
            return
        new_reductions = df[df['Status'] == 'NEW']
        if not new_reductions.empty:
            self.reduction_table.selection = new_reductions.index.to_list()

    def stop_periodic_update(self, session_context):
        self.keep_running = False

    def periodic_update(self):
        num_updates = 0
        total_time = 0.
        while self.keep_running:
            if self.edps.is_running():
                start = time.perf_counter()
                self.update_status()
                self.update_expanded_reduction_jobs()
                self.update_counters()
                total_time += time.perf_counter() - start
                num_updates += 1
            sleep(5)
        self.logger.info(
            'Stopping periodic update thread. Performed %d updates in %.2f s (average %.3f s per update)',
            num_updates, total_time, total_time / num_updates if num_updates > 0 else 0
        )

    def update_status(self):
        df = self.reduction_table.current_view
        if df.empty:
            return
        rows_to_check = df[df['Status'] != 'COMPLETED']
        if rows_to_check.empty:
            return
        current_statuses = rows_to_check['Status']
        reductions_to_check = [self.edps.get_reduction_by_id(rid) for rid in rows_to_check['ID']]
        new_statuses = [self.edps.get_reduction_status(r) for r in reductions_to_check]
        if current_statuses.to_list() != new_statuses:
            self.reduction_table.patch(
                {'Status': [(index, status) for index, status in zip(rows_to_check.index, new_statuses)]}
            )

    def update_expanded_reduction_jobs(self):
        for row in self.reduction_table.expanded:
            reduction_id = self.reduction_table.value.iloc[row]['ID']
            if reduction_id in self.job_tables:
                self.job_tables[reduction_id].update_status()

    def update_counters(self):
        df = self.reduction_table.current_view
        if df.empty:
            self.new.value = 0
            self.pending.value = 0
            self.running.value = 0
            self.completed.value = 0
            self.aborted.value = 0
            self.failed.value = 0
            return
        status_counts = df['Status'].value_counts()
        self.new.value = status_counts.get('NEW', 0)
        self.pending.value = status_counts.get('PENDING', 0)
        self.running.value = status_counts.get('RUNNING', 0)
        self.completed.value = status_counts.get('COMPLETED', 0)
        self.aborted.value = status_counts.get('ABORTED', 0)
        self.failed.value = status_counts.get('FAILED', 0)

    def stop_loading_spinner_on_modal_close(self, event):
        if event and event.name == 'open' and not event.new:
            self.reduction_table.loading = False

    def update_table_on_modal_close(self, event):
        if event and event.name == 'open' and not event.new:
            self.update_table()
            self.stop_loading_spinner_on_modal_close(event)

    def update_table(self, event=None):
        self.collapse_rows(None)
        reductions = [r for r in self.edps.get_all_reductions_as_list() if self.is_selected(r)]
        if self.update_selector.value == 'selected':
            reductions = [r for r in reductions if r.workflow == self.workflow]
        self.reduction_table.value = pd.DataFrame(
            [(r.id,
              r.dataset,
              r.target,
              r.obs_target,
              r.config.timestamp,
              self.edps.get_reduction_status(r),
              r.config.comment,
              self.format_recipe_parameters(r.config.recipe_parameters)) for r in reductions],
            columns=COLUMNS
        )
        self.select_new_reductions()

    def update_status_color(self):
        self.reduction_table.style.map(lambda status: f"color: {status_color.get(status, 'black')}", subset=['Status'])

    def collapse_rows(self, event):
        self.reduction_table.expanded = []
        self.job_tables = {}

    @staticmethod
    def format_workflow_parameters(params: WorkflowParameters) -> str:
        if not params:
            return ""
        return "\n".join([f"{param}={value}" for param, value in params.items()])

    @staticmethod
    def format_recipe_parameters(params: RecipeParameters) -> str:
        if not params:
            return ""
        return "\n".join([f"{task}.{param}={value}" for task, param, value in flatten_recipe_parameters(params)])

    def reduction_jobs(self, row):
        reduction_id = row["ID"]
        reduction = self.edps.get_reduction_by_id(reduction_id)
        if reduction.job_ids:
            job_table = JobTable(reduction.job_ids, reduction_timestamp=reduction.timestamp if self.job_icon else None,
                                 main_layout=self.layout, obs_target=reduction.obs_target)
            self.job_tables[reduction_id] = job_table
            return pn.Row(pn.Spacer(width=50), job_table)
        else:
            return pn.pane.Markdown("This dataset has not been processed yet.")

    def edit_reduction_comment(self, event):
        reduction_id = self.reduction_table.value.iloc[event.row]['ID']
        self.edps.update_reduction_comment(reduction_id, event.value)

    def archive_dataset(self, reduction: Reduction):
        dataset_reductions = self.edps.filter_reductions(reduction.dataset, reduction.target)
        reduction_statuses = {self.edps.get_reduction_status(r) for r in dataset_reductions}
        self.logger.info('Archiving dataset %s with configurations: %s', reduction.dataset,
                         [r.config.timestamp for r in dataset_reductions])
        if "COMPLETED" not in reduction_statuses:
            pn.state.notifications.warning("Cannot archive dataset with no COMPLETED reductions.")
        elif "RUNNING" in reduction_statuses:
            pn.state.notifications.warning("Cannot archive dataset with RUNNING reductions.")
        elif "PENDING" in reduction_statuses:
            pn.state.notifications.warning("Cannot archive dataset with PENDING reductions.")
        else:
            num_archived = 0
            num_deleted = 0
            for r in dataset_reductions:
                if r.completed:
                    self.logger.debug('Archiving dataset %s, configuration %s', r.dataset, r.config.timestamp)
                    self.edps.archive_reduction(r.id)
                    num_archived += 1
                else:
                    self.logger.info('Deleting dataset %s, configuration %s', r.dataset, r.config.timestamp)
                    self.edps.delete_reduction(r.id)
                    num_deleted += 1
            # message = (f"Archived dataset {reduction.dataset} with {num_archived} configurations. "
            #            f"Deleted {num_deleted} unused configurations.")
            # pn.state.notifications.success(message, duration=7000)

    def unarchive_dataset(self, reduction: Reduction):
        dataset_reductions = self.edps.filter_reductions(reduction.dataset, reduction.target)
        self.logger.debug('Unarchiving dataset %s with configurations: %s', reduction.dataset,
                          [r.config.timestamp for r in dataset_reductions])
        for r in dataset_reductions:
            self.edps.unarchive_reduction(r.id)
        # message = f"Unarchived dataset {reduction.dataset} with {len(dataset_reductions)} configurations."
        # pn.state.notifications.success(message, duration=5000)

    def on_table_clicked(self, event):
        reduction_id = self.reduction_table.value.iloc[event.row]['ID']
        reduction = self.edps.get_reduction_by_id(reduction_id)
        if event.column == 'view':
            self.open_dataset_viewer(reduction)
        elif event.column == 'view_target':
            self.open_job_viewer(reduction)
        elif event.column == 'configure':
            self.open_configuration_editor(reduction)

    def open_dataset_viewer(self, reduction: Reduction):
        status = self.edps.get_reduction_status(reduction)
        file_table = FileTable()
        file_table.update(reduction.input_files)
        modal_content = pn.Column(
            f"# Dataset {reduction.dataset}",
            reduction_summary_viewer(reduction, status),
            "### Input Files",
            file_table,
            width=1000, height=700, scroll=True
        )
        modal_content.scroll_to(0)
        self.open_modal(modal_content)

    def open_job_viewer(self, reduction: Reduction):
        jobs = [self.edps.get_job_details(job_id) for job_id in reduction.job_ids]
        if not jobs:
            pn.state.notifications.warning("This dataset has not been processed yet.")
            return
        target_jobs = [job for job in jobs if job.configuration.task_name == reduction.target]
        if not target_jobs:
            pn.state.notifications.error("Unexpected error: target job not found.")
            return
        target_job = target_jobs[0]
        if target_job.status.value != "COMPLETED":
            pn.state.notifications.warning(
                f"The target job '{target_job.configuration.task_name}' has not been processed yet.")
            return
        self.reduction_table.loading = True
        job_viewer = JobViewer(target_job.configuration.job_id, reduction.obs_target)
        modal_content = pn.Column(job_viewer, width=1000, height=700, scroll=True)
        modal = self.open_modal(modal_content)
        modal.param.watch(self.stop_loading_spinner_on_modal_close, "open")

    def open_configuration_editor(self, reduction: Reduction):
        self.reduction_table.loading = True
        modal_content = ReductionConfigurationEditor(reduction, self.reduction_table.current_view)
        modal = self.open_modal(modal_content)
        modal.param.watch(self.update_table_on_modal_close, "open")

    def delete_reductions(self, event):
        to_delete = self.get_selected_reductions()
        with self.delete_btn.param.update(loading=True, disabled=True):
            for reduction_id in self.tqdm(to_delete):
                self.edps.delete_reduction(reduction_id)
            self.reduction_table.selection = []
            self.update_table()
            self.close_modal(None)

    def open_delete_dialog(self, event):
        selected_df = self.reduction_table.value.iloc[self.reduction_table.selection]
        modal_content = pn.Column(
            f"### Delete {len(selected_df)} reductions?",
            pn.Column(
                pn.widgets.Tabulator(selected_df, show_index=False, groupby=['Dataset'], theme='default',
                                     hidden_columns=['ID', 'Target', 'Object', 'Comment']),
                height=200, scroll=True
            ),
            pn.layout.Divider(),
            pn.Row(self.delete_btn, self.close_btn),
            self.tqdm
        )
        self.modal = self.open_modal(modal_content)

    def __panel__(self) -> Viewable:
        self.layout.append(pn.Spacer())  # Placeholder for the modal
        return self.layout


class ReductionArchive(ReductionTable):
    def __init__(self, **params):
        super().__init__(**params)
        self.export_selector = pn.widgets.RadioBoxGroup(
            options={'Export all selected reductions': 'all',
                     'Export the latest reduction for each selected dataset': 'latest'},
            value='all'
        )
        self.export_btn = pn.widgets.Button(name='Export', button_type='success', disabled=self.export_disabled)
        self.export_btn.on_click(self.export_reductions)
        self.unarchive_btn = pn.widgets.Button(name='Unarchive', button_type='primary', align='center', height=30,
                                               disabled=self.empty_selection, icon='archive-off')
        self.unarchive_btn.on_click(self.unarchive_reductions)
        self.output_dir_input = pn.widgets.TextInput.from_param(self.param.output_dir)
        self.output_dir_input.name = 'Output directory'
        self.output_dir_msg = pn.widgets.StaticText()
        self.layout = self.create_layout()

    def is_selected(self, reduction: Reduction) -> bool:
        return reduction.archived

    def buttons(self):
        return {
            name: tooltip for name, tooltip in self.ALL_BUTTONS.items() if name in {'view', 'view_target'}
        }

    def unarchive_reductions(self, event):
        selected_df = self.reduction_table.value.iloc[self.reduction_table.selection]
        # Get the latest configuration date for each dataset
        reduction_indexes = selected_df.groupby('Dataset')['Configuration Date'].idxmax()
        reduction_ids = selected_df.loc[reduction_indexes]['ID'].to_list()
        for reduction_id in reduction_ids:
            reduction = self.edps.get_reduction_by_id(reduction_id)
            self.unarchive_dataset(reduction)
        self.reduction_table.selection = []
        self.update_table()

    def export_reductions(self, event):
        if self.export_selector.value == 'all':
            to_export = self.get_selected_reductions()
        else:
            selected_df = self.reduction_table.value.iloc[self.reduction_table.selection]
            idx = selected_df.groupby('Dataset')['Configuration Date'].idxmax()
            to_export = selected_df.loc[idx]['ID'].to_list()
        with self.export_btn.param.update(loading=True, disabled=True):
            for reduction_id in self.tqdm(to_export):
                reduction = self.edps.get_reduction_by_id(reduction_id)
                self.edps.process_reduction(reduction, self.output_dir)
        self.close_modal(None)

    def open_export_dialog(self, event):
        selected_df = self.reduction_table.value.iloc[self.reduction_table.selection]
        modal_content = pn.Column(
            "### Selected reductions",
            pn.Column(
                pn.widgets.Tabulator(selected_df, show_index=False, groupby=['Dataset'], theme='default',
                                     hidden_columns=['ID', 'Target', 'Object', 'Comment']),
                height=200, scroll=True
            ),
            pn.layout.Divider(),
            self.export_selector,
            self.output_dir_input,
            self.output_dir_msg,
            pn.Row(self.export_btn, self.close_btn),
            self.tqdm
        )
        self.modal = self.open_modal(modal_content)

    @pn.depends('output_dir')
    def export_disabled(self):
        if self.output_dir:
            if self.is_valid_output_dir():
                self.output_dir_msg.value = "✅ Output directory is valid"
                return False
            else:
                self.output_dir_msg.value = "❌ Output directory does not exist or is not writable"
                return True
        else:
            return True

    def is_valid_output_dir(self) -> bool:
        path = Path(self.output_dir)
        if not path.exists():
            # If path doesn't exist, check if parent directory is writable
            return path.parent.exists() and os.access(path.parent, os.W_OK)
        # If path exists, check if it's a directory and writable
        return path.is_dir() and os.access(path, os.W_OK)

    def create_layout(self) -> pn.Column:
        export_btn = pn.widgets.Button(name='Export', button_type='success', align='center', height=30,
                                       disabled=self.empty_selection, icon='file-export')
        export_btn.on_click(self.open_export_dialog)
        delete_btn = pn.widgets.Button(name='Delete', button_type='danger', align='center', height=30,
                                       disabled=self.empty_selection, icon='trash')
        delete_btn.on_click(self.open_delete_dialog)
        return pn.Column(
            pn.Row(
                self.update_selector,
                export_btn,
                self.unarchive_btn,
                delete_btn,
                pn.Card(self.reduction_counters, hide_header=True),
            ),
            self.reduction_table,
        )


class ReductionQueue(ReductionTable):

    def __init__(self, **params):
        super().__init__(**params)
        self.job_icon = True
        self.reduce_btn = pn.widgets.Button(name='Reduce', button_type='success', align='center', height=30,
                                            disabled=self.empty_selection, icon='run')
        self.reduce_btn.on_click(self.start_reduction)
        self.archive_btn = pn.widgets.Button(name='Archive', button_type='primary', align='center', height=30,
                                             disabled=self.empty_selection, icon='archive')
        self.archive_btn.on_click(self.archive_reductions)
        self.layout = self.create_layout()

    def is_selected(self, reduction: Reduction) -> bool:
        return not reduction.archived

    def buttons(self):
        return {
            name: tooltip for name, tooltip in self.ALL_BUTTONS.items() if name in {'view', 'view_target', 'configure'}
        }

    async def start_reduction(self, event, inline=False):
        with self.reduce_btn.param.update(loading=True, disabled=True):
            if inline:  # If the event is a reduction object, process it directly
                reductions = [event]
            else:
                reductions = [self.edps.get_reduction_by_id(rid) for rid in self.get_selected_reductions()]
            runnable_reductions = [r for r in reductions if
                                   self.edps.get_reduction_status(r) not in ("RUNNING", "COMPLETED")]
            num_skipped_reductions = len(reductions) - len(runnable_reductions)
            if not runnable_reductions:
                pn.state.notifications.error("Cannot process RUNNING or COMPLETED reductions.")
                return
            msg = f"Processing {len(runnable_reductions)} reductions."
            if num_skipped_reductions > 0:
                msg += f" Skipping {num_skipped_reductions} RUNNING or COMPLETED reductions."
            pn.state.notifications.success(msg)
            df = self.reduction_table.current_view
            for reduction in runnable_reductions:
                await asyncio.to_thread(self.edps.process_reduction, reduction)
                index = df.index[df['ID'] == reduction.id].tolist()
                self.reduction_table.patch({
                    'Status': [(ix, 'PENDING') for ix in index]
                })
                if index:
                    self.reduction_table.selection = [ix for ix in self.reduction_table.selection if ix != index[0]]

    def archive_reductions(self, event):
        selected_df = self.reduction_table.value.iloc[self.reduction_table.selection]
        completed_df = selected_df[selected_df['Status'] == 'COMPLETED']
        if completed_df.empty:
            pn.state.notifications.warning("No completed reductions selected for archiving.")
            return
        # Get the latest configuration date for each dataset
        reduction_indexes = completed_df.groupby('Dataset')['Configuration Date'].idxmax()
        reduction_ids = completed_df.loc[reduction_indexes]['ID'].to_list()
        for reduction_id in reduction_ids:
            reduction = self.edps.get_reduction_by_id(reduction_id)
            self.archive_dataset(reduction)
        self.reduction_table.selection = []
        self.update_table()

    def create_layout(self) -> pn.Column:
        delete_btn = pn.widgets.Button(name='Delete', button_type='danger', align='center', height=30,
                                       disabled=self.empty_selection, icon='trash')
        delete_btn.on_click(self.open_delete_dialog)
        monitors = pn.Row(
            self.new,
            self.pending,
            self.running,
            self.completed,
            self.aborted,
            self.failed,
        )
        return pn.Column(
            pn.Row(
                self.update_selector,
                self.reduce_btn,
                self.archive_btn,
                delete_btn,
                pn.Card(self.reduction_counters, hide_header=True),
                pn.Card(monitors, hide_header=True)
            ),
            self.reduction_table,
        )
