import itertools

import pandas as pd
import panel as pn
import param
from panel.viewable import Viewer, Viewable

from edpsgui.gui.edps_ctl import get_edps


class Organiser(Viewer):
    edps_status = param.Boolean(default=None, allow_refs=True)
    workflow = param.String(default=None, allow_refs=True)
    inputs = param.List(default=None, allow_refs=True)
    jobs = param.Dict(default={})

    def __init__(self, **params):
        super().__init__(**params)
        self.edps = get_edps()
        self.organise_btn = pn.widgets.Button(name='Organise', button_type='primary', disabled=self.edps_not_running)
        self.organise_btn.on_click(self.organise)

    @pn.depends('edps_status', 'inputs')
    def edps_not_running(self):
        return not self.edps.is_running() or not self.inputs

    @pn.depends('jobs')
    def data(self):
        jobs = list(self.jobs.values())
        df = pd.DataFrame(
            [(job.job_id,
              job.instrument,
              job.task_name,
              job.command,
              job.complete) for job in jobs],
            columns=['ID', 'Instrument', 'Task', 'Recipe', 'Complete']
        )
        setup_keywords = set(itertools.chain.from_iterable([job.setup.keys() for job in jobs]))
        for keyword in setup_keywords:
            df[keyword] = [job.setup.get(keyword, None) for job in jobs]
        return df

    def organise(self, event):
        with self.organise_btn.param.update(loading=True, disabled=True):
            jobs = self.edps.organise(workflow=self.workflow, inputs=self.inputs)
            self.jobs = {job.job_id: job for job in jobs}

    @pn.cache
    def job_content(self, row):
        job = self.jobs[row['ID']]
        files = '\n'.join([f"{f.name} {f.category}" for f in job.input_files + job.associated_files])
        return pn.pane.Markdown(f"{files if files else '**no input files**'}")

    def __panel__(self) -> Viewable:
        formatters = {
            'Complete': {'type': 'tickCross'}
        }
        jobs_table = pn.widgets.Tabulator(self.data, show_index=False, hidden_columns=['ID'],
                                          row_content=self.job_content, embed_content=True,
                                          header_filters=True, pagination='local', page_size=20, formatters=formatters)
        return pn.Column(
            self.organise_btn,
            jobs_table,
        )
