from dataclasses import dataclass, field
from typing import List, Set, Dict

import itertools
from edps.client.FlatOrganization import DatasetDTO, LabelledDatasetDTO

from .reduction_repository import ClassifiedFile

NamedDataset = LabelledDatasetDTO


@dataclass
class DatasetJob:
    job_id: str
    task_name: str
    complete: bool
    input_files: List[ClassifiedFile]
    associated_files: List[ClassifiedFile]
    assoc_level: float

    def __hash__(self):
        return hash(self.job_id)

    def __eq__(self, other: 'DatasetJob'):
        return self.job_id == other.job_id


@dataclass
class Node:
    id: str
    text: str
    icon: str
    children: List['Node']
    state: Dict[str, bool] = field(default_factory=lambda: {'opened': False})


def astree(dataset: DatasetDTO, opened: bool = False) -> Node:
    children = [astree(job.dataset) for job in dataset.main_input_jobs + dataset.associated_input_jobs]
    for file in dataset.main_input_files + dataset.associated_input_files:
        children.append(Node(id=file.name, text=f"{file.name} [{file.category}]", icon="jstree-file", children=[]))
    for missing_input in dataset.missing_associated_inputs:
        html = missing_input_html(missing_input.task_name, missing_input.optional)
        children.append(Node(id=missing_input.task_name, text=html, icon="jstree-folder", children=[]))
    html = complete_dataset_html(dataset.task_name, dataset.complete)
    return Node(id=dataset.job_id, text=html, icon="jstree-folder", children=children, state={'opened': opened})


@dataclass
class MenuItem:
    label: str
    secondary: str
    icon: str
    color: str


@dataclass
class ContainerMenuItem(MenuItem):
    items: List['MenuItem']
    open: bool


def asmenulist(dataset: DatasetDTO, open: bool = False) -> MenuItem:
    items = [asmenulist(job.dataset) for job in dataset.main_input_jobs + dataset.associated_input_jobs]
    for file in dataset.main_input_files + dataset.associated_input_files:
        items.append(
            MenuItem(label=file.category, secondary=file.name, icon="image", color="success")
        )
    for missing_input in dataset.missing_associated_inputs:
        missing = "MISSING, OPTIONAL" if missing_input.optional else "MISSING, MANDATORY"
        color = "warning" if missing_input.optional else "error"
        items.append(
            MenuItem(label=missing_input.task_name, secondary=missing, icon="image", color=color)
        )
    complete = "COMPLETE" if dataset.complete else "INCOMPLETE"
    color = "success" if dataset.complete else "error"
    return ContainerMenuItem(label=dataset.task_name, secondary=complete, icon="collections", color=color, items=items,
                             open=open)


def get_dataset_jobs(dataset: DatasetDTO) -> Set[DatasetJob]:
    jobs = [get_dataset_jobs(job.dataset) for job in dataset.main_input_jobs + dataset.associated_input_jobs]
    job = DatasetJob(job_id=dataset.job_id, task_name=dataset.task_name, complete=dataset.complete,
                     input_files=[ClassifiedFile(f.name, f.category) for f in dataset.main_input_files],
                     associated_files=[ClassifiedFile(f.name, f.category) for f in dataset.associated_input_files],
                     assoc_level=dataset.assoc_level)
    return {job}.union(itertools.chain.from_iterable(jobs))


def get_dataset_files(dataset: DatasetDTO) -> List[ClassifiedFile]:
    files = []
    for job in get_dataset_jobs(dataset):
        files.extend([ClassifiedFile(f.name, f.category) for f in job.input_files + job.associated_files])
    return list(set(files))


def get_dataset_tasks(dataset: DatasetDTO) -> List[str]:
    return list({job.task_name for job in get_dataset_jobs(dataset)})


def get_dataset_assoc_level(dataset: DatasetDTO) -> float:
    return max(job.assoc_level for job in get_dataset_jobs(dataset))


def missing_input_html(name: str, optional: bool) -> str:
    color = '#ff9966' if optional else '#cc3300'
    text = '[missing, optional]' if optional else '[missing, mandatory]'
    return f'<span style="color:{color};">{name} {text}</span>'


def complete_dataset_html(name: str, complete: bool) -> str:
    color = '#339900' if complete else '#cc3300'
    text = '[complete]' if complete else '[incomplete]'
    return f'<span style="color:{color};">{name} {text}</span>'


def check_or_cross(status: bool):
    return "✅" if status else "❌"
