import panel as pn
import param
from panel.viewable import Viewer, Viewable

from .edps_ctl import get_edps
from .file_table import FileTable


class Classifier(Viewer):
    edps_status = param.Boolean(default=None, allow_refs=True)
    workflow = param.String(default=None, allow_refs=True)
    inputs = param.List(default=None, allow_refs=True)

    def __init__(self, **params):
        super().__init__(**params)
        self.edps = get_edps()
        self.classify_btn = pn.widgets.Button(name='Inspect Inputs', button_type='primary',
                                              disabled=self.edps_not_running)
        self.classify_btn.on_click(self.classify)
        self.layout = self.create_layout()

    @pn.depends('edps_status', 'inputs')
    def edps_not_running(self):
        return not self.edps.is_running() or not self.inputs

    def classify(self, event):
        with self.classify_btn.param.update(loading=True, disabled=True):
            classified_files = self.edps.classify(workflow=self.workflow, inputs=self.inputs)
            classified_files = [f for f in classified_files if f.category is not None]
            if not classified_files:
                msg = f"No file classified. Please select input data that can be processed with workflow '{self.workflow}'"
                pn.state.notifications.warning(msg, duration=0)
            else:
                file_table = FileTable()
                file_table.update(classified_files)
                modal_content = pn.Column(file_table, width=1000, height=800, scroll=True)
                modal = pn.Modal(modal_content)
                self.layout.pop(-1)
                self.layout.append(modal)
                modal.open = True

    def create_layout(self) -> pn.Column:
        return pn.Column(
            self.classify_btn,
            pn.Spacer()  # This is a placeholder!
        )

    def __panel__(self) -> Viewable:
        return self.layout
