import dataclasses
import logging
import os
from typing import List, Dict

import itertools
import pandas as pd
import panel as pn
import panel_material_ui as pmui
import param
import time
from panel.viewable import Viewer, Viewable

from edpsgui.domain.dataset import (
    get_dataset_jobs,
    get_dataset_files,
    get_dataset_tasks,
    get_dataset_assoc_level,
    asmenulist,
    NamedDataset,
)
from edpsgui.domain.reduction_repository import Reduction, ReductionConfig, get_timestamp
from edpsgui.domain.utils import get_keywords
from .edps_ctl import get_edps
from .file_table import FileTable
from .tooltips import CALIBRATION_PREFERENCE

COLUMNS = ['ID', 'Dataset', 'Target', 'Object', 'Files', 'Jobs', 'CalibLevel', 'Complete', 'Submitted', 'Archived']
OBS_TARG = 'HIERARCH ESO OBS TARG NAME'
INF = float('inf')


class DatasetCreator(Viewer):
    edps_status = param.Boolean(default=None, allow_refs=True)
    workflow = param.String(default=None, allow_refs=True)
    inputs = param.List(allow_refs=True)
    targets = param.List(allow_refs=True)
    parameter_set = param.Selector(allow_refs=True)
    workflow_parameters = param.Dict(allow_refs=True)
    association_preference = param.Selector(default='RAW_PER_QUALITY_LEVEL',
                                            objects=['RAW', 'MASTER', 'RAW_PER_QUALITY_LEVEL',
                                                     'MASTER_PER_QUALITY_LEVEL'])
    association_level = param.Selector(default=INF,
                                       objects={'CALIBRATION PLAN (BEST)': 0., 'LEVEL 1': 1., 'LEVEL 2': 2.,
                                                'ANY': INF})
    dataset_df = param.DataFrame(default=pd.DataFrame(columns=COLUMNS))
    datasets = param.Dict(default={})

    def __init__(self, **params):
        super().__init__(**params)
        self.edps = get_edps()
        self.create_datasets_btn = pn.widgets.Button(name='Create Datasets', button_type='success', align='center',
                                                     height=30, disabled=self.creation_disabled)
        self.create_datasets_btn.on_click(self.create_datasets)
        formatters = {
            'Complete': {'type': 'tickCross'},
            'Submitted': {'type': 'tickCross'},
            'Archived': {'type': 'tickCross'},
        }
        buttons = {
            "view": "<i class='fa-regular fa-file-lines' title='View dataset details'></i>",
        }
        self.dataset_table = pn.widgets.Tabulator.from_param(self.param.dataset_df, show_index=False, buttons=buttons,
                                                             hidden_columns=['ID'],
                                                             pagination='local', page_size=20, formatters=formatters,
                                                             selectable_rows=self.selectable_rows,
                                                             header_align='center', text_align='center',
                                                             selectable='checkbox', disabled=True,
                                                             stylesheets=[":host .tabulator {font-size: 13px;}"])
        self.dataset_table.on_click(self.dataset_actions)
        self.layout = self.create_layout()
        self.logger = logging.getLogger('DatasetCreator')

    @pn.depends('edps_status', 'inputs', 'dataset_df')
    def creation_disabled(self):
        return not (self.edps.is_running() and self.inputs and self.dataset_df.empty)

    @pn.depends('dataset_df')
    def deletion_disabled(self):
        return self.dataset_df.empty

    @pn.depends('workflow', 'targets', 'parameter_set', 'workflow_parameters', 'inputs', 'association_preference',
                watch=True)
    def reset_datasets_on_input_change(self):
        self.delete_datasets(None)

    @pn.depends('dataset_table.selection')
    def empty_selection(self):
        return not self.dataset_table.selection

    @pn.depends('dataset_df', 'dataset_table.selection')
    def dataset_counters(self):
        df = self.dataset_df
        num_datasets = len(df)
        num_selected = len(self.dataset_table.selection)
        num_complete = len(df[df['Complete'] == True])
        num_submitted = len(df[df['Submitted'] == True])
        return pn.Row(
            pn.indicators.Number(name='Datasets', value=num_datasets, font_size='12pt', title_size='14pt',
                                 default_color='black'),
            pn.indicators.Number(name='Selected', value=num_selected, font_size='12pt', title_size='14pt',
                                 default_color='blue'),
            pn.indicators.Number(name='Complete', value=num_complete, font_size='12pt', title_size='14pt',
                                 default_color='green'),
            pn.indicators.Number(name='Submitted', value=num_submitted, font_size='12pt', title_size='14pt',
                                 default_color='orange'),
        )

    @staticmethod
    def selectable_rows(df: pd.DataFrame) -> List[int]:
        if df is None or df.empty:
            return []
        return df.index[df['Submitted'] == False].tolist()

    def show_dataset(self, dataset):
        files = [job.input_files + job.associated_files for job in get_dataset_jobs(dataset.dataset)]
        files = set(itertools.chain.from_iterable(files))
        file_table = FileTable()
        file_table.update(list(files))
        dataset_menu = pmui.MenuList(items=[dataclasses.asdict(asmenulist(dataset.dataset, open=True))],
                                     dense=True)
        modal_content = pn.Column(f"### Dataset {dataset.dataset_name}",
                                  pn.Tabs(
                                      ('Task hierarchy', dataset_menu),
                                      ('Files', file_table),
                                      stylesheets=[":host {font-size: 16px;}"]
                                  ),
                                  width=1000, height=800, scroll=True)
        modal = pn.Modal(modal_content)
        self.layout.pop(-1)
        self.layout.append(modal)
        modal.open = True

    def dataset_actions(self, event):
        job_id = self.dataset_table.value.iloc[event.row]['ID']
        dataset = self.datasets[job_id]
        if event.column == "view":
            self.show_dataset(dataset)

    def get_dataset_keywords(self, dataset: NamedDataset) -> Dict[str, object]:
        keywords = [OBS_TARG]
        main_file = [f.name for f in get_dataset_files(dataset.dataset) if
                     os.path.basename(f.name).startswith(dataset.dataset_name)]
        if len(main_file) != 1:
            self.logger.error('Could not find main file for dataset %s', dataset.dataset_name)
            return {kw: None for kw in keywords}
        else:
            return get_keywords(main_file[0], keywords)

    def is_dataset_submitted(self, dataset: NamedDataset) -> bool:
        reductions = self.edps.filter_reductions(dataset.dataset_name, dataset.dataset.task_name)
        dataset_files = set(get_dataset_files(dataset.dataset))
        submitted = [r for r in reductions if set(r.input_files) == dataset_files]
        return len(submitted) > 0

    def is_dataset_archived(self, dataset: NamedDataset) -> bool:
        reductions = self.edps.filter_reductions(dataset.dataset_name, dataset.dataset.task_name)
        dataset_files = set(get_dataset_files(dataset.dataset))
        archived = [r for r in reductions if r.archived and set(r.input_files) == dataset_files]
        return len(archived) > 0

    def select_unsubmitted_datasets(self):
        df = self.dataset_table.current_view
        if df.empty:
            return
        unsubmitted_datasets = df[(df['Submitted'] == False) & (df['Complete'] == True)]
        if not unsubmitted_datasets.empty:
            self.dataset_table.selection = unsubmitted_datasets.index.to_list()

    def update_dataset_df(self):
        dataset_keywords = {ds.dataset.job_id: self.get_dataset_keywords(ds) for ds in self.datasets.values()}
        self.dataset_df = pd.DataFrame([(ds.dataset.job_id,
                                         ds.dataset_name,
                                         ds.dataset.task_name,
                                         dataset_keywords[ds.dataset.job_id][OBS_TARG],
                                         len(get_dataset_files(ds.dataset)),
                                         len(get_dataset_jobs(ds.dataset)),
                                         get_dataset_assoc_level(ds.dataset),
                                         ds.dataset.complete,
                                         self.is_dataset_submitted(ds),
                                         self.is_dataset_archived(ds)) for ds in self.datasets.values()],
                                       columns=COLUMNS)
        self.select_unsubmitted_datasets()

    def create_datasets(self, event):
        self.delete_datasets(None)
        with self.create_datasets_btn.param.update(loading=True, disabled=True):
            start = time.time()
            self.logger.info(
                "Creating datasets: workflow=%s, inputs=%s, targets=%s, association_preference=%s, association_level=%.1f",
                self.workflow, self.inputs, self.targets, self.association_preference, self.association_level)
            datasets = self.edps.create_datasets(workflow=self.workflow, inputs=self.inputs, targets=self.targets,
                                                 association_preference=self.association_preference,
                                                 association_level=self.association_level,
                                                 parameter_set=self.parameter_set,
                                                 workflow_parameters=self.workflow_parameters)
            self.logger.info("Created %d datasets in %.3fs", len(datasets.datasets), time.time() - start)
            self.datasets = {ds.dataset.job_id: ds for ds in datasets.datasets}
            self.update_dataset_df()
            if not self.datasets:
                msg = f"No dataset created. Please select input data that can be processed with workflow '{self.workflow}'"
                pn.state.notifications.warning(msg, duration=10_000)

    def submit_datasets(self, event):
        indexes = self.dataset_table.selection
        dataset_ids = self.dataset_table.value.iloc[indexes]['ID'].to_list()
        config = ReductionConfig(
            parameter_set=self.parameter_set,
            workflow_parameters=self.workflow_parameters
        )
        for dataset_id in dataset_ids:
            dataset = self.datasets[dataset_id]
            dataset_name = dataset.dataset_name
            obs_target = self.get_dataset_keywords(dataset).get(OBS_TARG)
            reduction = Reduction(
                timestamp=get_timestamp(),
                dataset=dataset_name,
                workflow=self.workflow,
                tasks=get_dataset_tasks(dataset.dataset),
                target=dataset.dataset.task_name,
                obs_target=str(obs_target),
                input_files=get_dataset_files(dataset.dataset),
                config=config
            )
            self.logger.info('Enqueuing dataset %s for reduction', dataset_name)
            self.edps.add_reduction(reduction)
        self.dataset_table.selection = []
        self.update_dataset_df()

    def delete_datasets(self, event):
        self.datasets = {}
        self.dataset_df = pd.DataFrame(columns=COLUMNS)

    def create_layout(self):
        # association_level_select = pn.widgets.Select.from_param(self.param.association_level,
        #                                                         name='Acceptable Calibration Quality', align='start',
        #                                                         width=200)
        association_preference_select = pn.widgets.Select.from_param(self.param.association_preference,
                                                                     name='Calibration Preference', align='start',
                                                                     width=200)
        association_preference_tooltip = pn.widgets.TooltipIcon(value=CALIBRATION_PREFERENCE)

        submit_datasets_btn = pn.widgets.Button(name='Submit to Reduction Queue', button_type='success', align='center',
                                                height=30, disabled=self.empty_selection)
        submit_datasets_btn.on_click(self.submit_datasets)

        delete_datasets_btn = pn.widgets.Button(name='Delete Datasets', button_type='danger',
                                                disabled=self.deletion_disabled)
        delete_datasets_btn.on_click(self.delete_datasets)

        return pn.Column(
            pn.Row(association_preference_select, association_preference_tooltip),
            pn.Row(self.create_datasets_btn, submit_datasets_btn,
                   pn.Card(self.dataset_counters, hide_header=True)),
            self.dataset_table,
            pn.Spacer()  # This is a placeholder!
        )

    def __panel__(self) -> Viewable:
        return self.layout
