import os
from typing import List

import pandas as pd
import panel as pn
import param
from panel.viewable import Viewer, Viewable

from edpsgui.domain.reduction_repository import ClassifiedFile
from edpsgui.domain.utils import open_file_in_fv, open_file_in_ds9, get_fits_info
from .data_viewer import ImageViewer, TableViewer, DataViewer, SpectrumViewer
from .header_viewer import HeaderViewer


class FileTable(Viewer):
    filename = param.String()
    category = param.String()
    hdu = param.Selector(objects=[0], label='Select HDU')
    files_df = param.DataFrame(default=pd.DataFrame())

    def __init__(self, header_filters=True, group_by_catg=True, **params):
        super().__init__(**params)
        self.fits_info = []
        formatters = {
            'File': 'textarea',
        }
        widths = {
            'File': 600,
            'Category': 200,
        }
        buttons = {
            "fv": "<button>fv</button>",
            "ds9": "<button>ds9</button>"
        }
        groupby = ['Category'] if group_by_catg else []
        self.file_table = pn.widgets.Tabulator.from_param(self.param.files_df, show_index=False,
                                                          header_filters=header_filters, groupby=groupby,
                                                          formatters=formatters, widths=widths, buttons=buttons,
                                                          theme='default', disabled=True, pagination='local',
                                                          page_size=15, configuration={'groupStartOpen': False},
                                                          stylesheets=[":host .tabulator {font-size: 11px;}"])
        self.file_table.on_click(self.file_actions)

    def update(self, fits_files: List[ClassifiedFile]):
        self.files_df = pd.DataFrame({
            'File': [file.name for file in fits_files],
            'Category': [file.category for file in fits_files],
        })
        self.filename = ''
        self.category = ''

    def file_actions(self, event):
        self.filename = self.file_table.value.iloc[event.row]['File']
        self.category = self.file_table.value.iloc[event.row]['Category']
        if event.column == 'fv':
            open_file_in_fv(self.filename)
        elif event.column == 'ds9':
            open_file_in_ds9(self.filename)

    def data_viewer(self, hdu: int):
        hdu_info = self.fits_info[hdu]
        if hdu == 1 and hdu_info.prodcatg == 'SCIENCE.SPECTRUM':
            return SpectrumViewer(filename=self.filename, hdu_info=hdu_info)
        elif hdu_info.type in ('ImageHDU', 'PrimaryHDU', 'CompImageHDU') and len(hdu_info.shape) == 2:
            return ImageViewer(filename=self.filename, hdu_info=hdu_info)
        elif hdu_info.type in ('TableHDU', 'BinTableHDU'):
            return TableViewer(filename=self.filename, hdu_info=hdu_info)
        else:
            return DataViewer(filename=self.filename, category=self.category, hdu_info=hdu_info)

    def hdu_viewer(self, hdu: int):
        header_viewer = HeaderViewer(filename=self.filename, hdu=hdu)
        data_viewer = self.data_viewer(hdu)
        return pn.Tabs(
            ('Data', data_viewer),
            ('Header', header_viewer),
        )

    @pn.depends('filename')
    def file_viewer(self):
        if self.filename and os.path.exists(self.filename):
            self.fits_info = get_fits_info(self.filename)
            options = {f'{info.index:02} {info.name} {info.type} {info.shape}': info.index for info in self.fits_info}
            select = pn.widgets.Select(name='Select HDU', options=options, value=0)
            return pn.Column(
                select,
                pn.bind(self.hdu_viewer, select.param.value)
            )

    @pn.depends('filename')
    def file_title(self):
        if not self.filename:
            return pn.pane.Markdown('#### No file selected')
        elif not os.path.exists(self.filename):
            msg = f'File {self.filename} does not exist'
            return pn.pane.Markdown(f'<span style="color:red">{msg}</span>')
        else:
            return pn.pane.Markdown(f'#### File: {self.filename}')

    def __panel__(self) -> Viewable:
        layout = pn.Column(
            self.file_table,
            self.file_title,
            self.file_viewer
        )
        return layout
