from typing import List

import networkx as nx
import pandas as pd
import panel as pn
from panel.viewable import Viewer, Viewable

from edpsgui.domain.utils import MAX_TIMESTAMP_US, MAX_TIMESTAMP_MS
from .edps_ctl import get_edps
from .job_viewer import JobViewer


class JobTable(Viewer):

    def __init__(self, job_ids: List[str], reduction_timestamp: str = None, main_layout=None, obs_target=None,
                 **params):
        super().__init__(**params)
        self.edps = get_edps()
        sorted_jobs = self.sort_jobs(self.edps.get_job_details(job_id) for job_id in job_ids)
        self.reduction_timestamp = reduction_timestamp or MAX_TIMESTAMP_MS
        sorters = [
            {'field': 'Order', 'dir': 'asc'},
        ]
        buttons = {
            'view': "<i class='fa-solid fa-magnifying-glass'></i>"
        }
        hidden_columns = ['ID', 'Order']
        if not reduction_timestamp:
            hidden_columns.append('New')
        df = pd.DataFrame([(job.configuration.job_id,
                            order,
                            self.get_task_name(job),
                            job.configuration.command,
                            self.get_submission_date(job),
                            self.get_completion_date(job),
                            self.get_status(job)) for order, job in enumerate(sorted_jobs)],
                          columns=['ID', 'Order', 'Task', 'Recipe', 'Created', 'Completed', 'Status'])
        self.job_table = pn.widgets.Tabulator(df, show_index=False, hidden_columns=hidden_columns,
                                              buttons=buttons, sorters=sorters, disabled=True)
        self.job_table.on_click(self.job_actions)
        self.layout = self.create_layout()
        # we keep a reference to the main layout to add the modal to it
        self.main_layout = main_layout or self.layout
        self.obs_target = obs_target

    def is_new_job(self, job) -> bool:
        return job.submission_date > self.reduction_timestamp

    def get_task_name(self, job) -> str:
        if self.reduction_timestamp == MAX_TIMESTAMP_MS:
            icon = ''
        else:
            icon = '🆕 ' if self.is_new_job(job) else '♻️ '
        return icon + job.configuration.task_name

    @staticmethod
    def get_status(job) -> str:
        return "ABORTED" if job.interrupted else "PENDING" if job.status.value == "CREATED" else job.status.value

    @staticmethod
    def get_submission_date(job) -> str:
        return job.submission_date.split('.')[0]

    @staticmethod
    def get_completion_date(job) -> str:
        return job.completion_date.split('.')[0] if job.completion_date != MAX_TIMESTAMP_US else ''

    @staticmethod
    def sort_jobs(jobs):
        graph = nx.DiGraph()
        for job in jobs:
            graph.add_node(job.configuration.job_id, job=job)
            for parent_id in job.configuration.input_job_ids + job.configuration.associated_job_ids:
                graph.add_edge(parent_id, job.configuration.job_id)
        return [graph.nodes[node]['job'] for node in nx.topological_sort(graph)]

    def stop_loading_spinner_on_modal_close(self, event):
        if event and event.name == 'open' and not event.new:
            self.job_table.loading = False

    def open_job_viewer(self, job_id: str):
        self.job_table.loading = True
        modal_content = pn.Column(JobViewer(job_id, self.obs_target), width=1000, height=700, scroll=True)
        modal = pn.Modal(modal_content)
        self.main_layout.pop(-1)
        self.main_layout.append(modal)
        modal.open = True
        modal.param.watch(self.stop_loading_spinner_on_modal_close, "open")

    def job_actions(self, event):
        job_id = self.job_table.value.iloc[event.row]['ID']
        if event.column == 'view':
            self.open_job_viewer(job_id)

    def create_layout(self) -> pn.Column:
        return pn.Column(
            self.job_table,
            pn.Spacer()  # This is a placeholder!
        )

    def update_status(self):
        df = self.job_table.current_view
        if df.empty:
            return
        rows_to_check = df[df['Status'] != 'COMPLETED']
        if rows_to_check.empty:
            return
        current_statuses = rows_to_check['Status']
        jobs_to_check = [self.edps.get_job_details(job_id) for job_id in rows_to_check['ID']]
        new_statuses = [self.get_status(job) for job in jobs_to_check]
        completion_dates = [self.get_completion_date(job) for job in jobs_to_check]
        if current_statuses.to_list() != new_statuses:
            self.job_table.patch(
                {
                    'Status': [(index, status) for index, status in zip(rows_to_check.index, new_statuses)],
                    'Completed': [(index, date) for index, date in zip(rows_to_check.index, completion_dates)],
                }
            )

    def __panel__(self) -> Viewable:
        return self.layout
