import copy
from typing import Dict

import pandas as pd
import panel as pn
import param
from panel.viewable import Viewer, Viewable

from edpsgui.gui.edps_ctl import get_edps
from .tooltips import PARAMETER_SET

SCIENCE_PARAMETERS = 'science_parameters'
COLUMNS = ['Parameter', 'Default', 'Custom']


class ParameterSelector(Viewer):
    workflow = param.String(default=None, allow_refs=True)
    parameter_set = param.Selector(default=[])
    clicked_parameter = param.String(allow_None=True)
    workflow_param_df = param.DataFrame(columns=COLUMNS)
    custom_workflow_parameters = param.Dict(default={})

    def __init__(self, **params):
        super().__init__(**params)
        self.edps = get_edps()
        parameter_sets = self.get_parameter_sets()
        self.parameter_set = parameter_sets[0] if parameter_sets else None
        self.parameter_set_select = pn.widgets.Select.from_param(self.param.parameter_set,
                                                                 name='Parameter Set',
                                                                 align='start', width=200,
                                                                 options=self.get_parameter_sets,
                                                                 value=self.parameter_set)
        self.parameter_metadata: Dict[str, str] = dict()
        self.workflow_parameter_table = self.create_parameter_table()

    def create_parameter_table(self) -> pn.widgets.Tabulator:
        editors = {
            'Parameter': None,
            'Default': None,
        }
        titles = {
            'Default': 'Default value',
            'Custom': 'Custom value',
        }
        formatters = {
            'Default': 'textarea',
            'Custom': 'textarea',
        }
        widths = {
            'Default': 300,
            'Custom': 300,
        }
        table = pn.widgets.Tabulator.from_param(self.param.workflow_param_df, show_index=False, theme='default',
                                                editors=editors, titles=titles, formatters=formatters, widths=widths)
        table.on_edit(self.edit_workflow_parameter)
        table.on_click(
            lambda event: setattr(self, 'clicked_parameter', event.value)
        )
        return table

    def edit_workflow_parameter(self, event):
        parameter = self.workflow_parameter_table.value['Parameter'][event.row]
        # Create a new dictionary to trigger the update event
        new_custom_workflow_parameters = copy.deepcopy(self.custom_workflow_parameters)
        if not event.value:
            new_custom_workflow_parameters.pop(parameter, None)
        else:
            new_custom_workflow_parameters[parameter] = event.value
        self.custom_workflow_parameters = new_custom_workflow_parameters

    @pn.depends('workflow')
    def get_parameter_sets(self):
        if self.workflow:
            parameter_sets = [ps.name for ps in self.edps.get_parameter_sets(self.workflow)]
            if SCIENCE_PARAMETERS in parameter_sets:
                parameter_sets = [SCIENCE_PARAMETERS] + [ps for ps in parameter_sets if ps != SCIENCE_PARAMETERS]
            self.parameter_set = parameter_sets[0] if parameter_sets else None
            return parameter_sets
        else:
            return []

    @pn.depends('workflow', 'parameter_set', watch=True)
    def update_workflow_parameter_table(self):
        self.clicked_parameter = None
        if getattr(self, 'workflow_parameter_table', None):
            self.workflow_parameter_table.selection = []
        if self.workflow and self.parameter_set:
            workflow_parameters = self.edps.get_workflow_parameters(self.workflow, self.parameter_set)
            data = [(parameter, value, self.custom_workflow_parameters.get(parameter)) for
                    parameter, value in workflow_parameters.items()]
            self.workflow_param_df = pd.DataFrame(data, columns=COLUMNS)
            self.parameter_metadata = self.edps.get_parameter_metadata(self.workflow)
        else:
            self.workflow_param_df = pd.DataFrame([], columns=COLUMNS)
            self.parameter_metadata = dict()

    @pn.depends('clicked_parameter')
    def workflow_parameter_description(self):
        if not self.clicked_parameter:
            return pn.pane.Markdown("_Click on a parameter to view its description_")
        else:
            description = self.parameter_metadata.get(self.clicked_parameter, 'No description available.')
            return pn.pane.Markdown(f"**{self.clicked_parameter}**: {description}")

    def __panel__(self) -> Viewable:
        parameter_set_tooltip = pn.widgets.TooltipIcon(value=PARAMETER_SET)
        return pn.Column(
            pn.Row(
                self.parameter_set_select, parameter_set_tooltip,
            ),
            self.workflow_parameter_table,
            self.workflow_parameter_description
        )
