from typing import List, Optional

import panel as pn
import param
from panel.viewable import Viewer, Viewable

from .edps_ctl import get_edps


class TargetSelector(Viewer):
    workflow = param.String(default=None, allow_refs=True)
    meta_target = param.Selector()
    targets = param.List()

    def __init__(self, **params):
        super().__init__(**params)
        self.edps = get_edps()
        self.meta_target_select = pn.widgets.Select.from_param(self.param.meta_target, name='Target Category',
                                                               align='start', width=200, options=self.get_meta_targets,
                                                               value=self.initial_meta_target)
        self.meta_target_select.param.watch(self.update_metatarget_value, 'options')
        self.targets_multi_choice = pn.widgets.MultiChoice(name='Targets', align='start', sizing_mode='stretch_width',
                                                           options=self.get_targets, value=self.get_targets)
        self.targets_multi_choice.param.watch(self.update_targets, 'value')

    @staticmethod
    def select_meta_target(values: List[str]) -> Optional[str]:
        return 'science' if 'science' in values else values[0] if values else None

    @property
    def initial_meta_target(self) -> Optional[str]:
        return self.select_meta_target(self.get_meta_targets())

    def update_metatarget_value(self, event=None):
        if event and event.new:
            self.meta_target_select.value = self.select_meta_target(event.new)

    def update_targets(self, event=None):
        if not event:
            return
        elif event.new:
            self.targets = event.new
        else:
            # If the new value is empty, we reset to the old value
            self.targets_multi_choice.value = event.old

    @pn.depends('workflow')
    def get_meta_targets(self):
        if self.workflow:
            meta_targets = sorted(self.edps.get_meta_targets(self.workflow))
            self.meta_target = self.select_meta_target(meta_targets)
            return meta_targets
        else:
            return []

    @pn.depends('workflow', 'meta_target')
    def get_targets(self):
        if self.workflow and self.meta_target:
            self.targets = self.edps.get_targets(self.workflow, self.meta_target)
            return self.targets
        else:
            return []

    def __panel__(self) -> Viewable:
        return pn.Row(
            self.meta_target_select,
            self.targets_multi_choice
        )
