# This file is part of PyEsorex the Python ESO Recipe Execution Tool
# Copyright (C) 2020-2025 European Southern Observatory
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""CLI for PyEsoRex."""

import argparse
import math
import sys
import traceback
from pathlib import Path
from typing import Optional

from cpl import DESCRIPTION as LIBRARIES_DESCRIPTION
from cpl import core
from cpl.ui import FrameSet

from pyesorex.pyesorex import Pyesorex


def pyesorex_cli():
    """
    Runs the PyEsoRex command line interface.

    Creates a pyesorex.pyesorex.Pyesorex instance, parses the command line arguments,
    passes the arguments through to Pyesorex, and then calls the Pyesorex.run method
    to execute the selected recipe. Enter `pyesorex --help` at the command line for
    full details of the supported command line arguments.

    See Also
    --------
    pyesorex.pyesorex.Pyesorex : class that implements the recipe execution environment.
    """
    try:
        # Need a Pyesorex instance to set up the command line argument parser,
        # but don't want it to start doing anything until after we've parsed
        # the args.
        pyesorex = Pyesorex()
        print(f"\n{pyesorex.version_string}\n")

        parser = argparse.ArgumentParser(
            usage="pyesorex [pyesorex options] recipe [recipe options] sof",
            argument_default=argparse.SUPPRESS,
            add_help=False,
            epilog=f"Libraries used: {LIBRARIES_DESCRIPTION}",
        )
        parser.add_argument(
            "--version",
            action="version",
            version=parser.epilog,
            help="Display version information and exit.",
        )

        parser.add_argument(
            "--show-products",
            action="store",
            default=["off"],
            choices=["off", "summary", "details", "dump"],
            dest="list_products",
            nargs=1,
            help="Display the set of product frames after the recipe execution is complete.",
        )

        pyesorex_group = parser.add_argument_group("pyesorex options")
        for param in pyesorex.parameters:
            # Pyesorex parameters have associated argparse Actions that take care of
            # setting the parameter values, etc.
            pyesorex_group.add_argument(
                f"--{param.cli_alias}",
                action=param.action,
                pyesorex=pyesorex,
                dest=param.name,
                nargs=param.nargs,
                const=param.const,
            )

        # Assume any arguments not recognised as pyesorex options are the recipe name, its options
        # and the set-of-frames and store them for later. The action of parsing the arguments will
        # also set all the corresponding parameter values in the Pyesorex object.
        pyesorex_args, recipe_args = parser.parse_known_args()

        # The first pass of parsing the command-line arguments processed only the options of
        # pyesorex itself. Therefore the first unrecognized argument stored in recipe_args
        # should be the recipe name, if any is given. If the recipe is not found in the recipe
        # search path the program exits at this point with an error. If the first element starts
        # with an option prefix character it is considered an unrecognized option, otherwise it
        # would already been consumed by the first parsing pass.
        recipe_name = None
        if len(recipe_args) > 0:
            if recipe_args[0].startswith(tuple(chr for chr in parser.prefix_chars)):
                err = f"Expected a recipe name but got an unrecognized argument: {recipe_args[0]}"
                parser.error(err)
            if pyesorex.is_c_recipe(recipe_args[0]) or pyesorex.is_python_recipe(recipe_args[0]):
                recipe_name = recipe_args[0]
            else:
                recipe_dirs = pyesorex.parameters["recipe_dirs"].value.split(sep=":")
                err = f"No recipe named {recipe_args[0]!r} found in {recipe_dirs}."
                parser.error(err)
        else:
            nargs = 0
            for argname in pyesorex.parameters.as_dict():
                if hasattr(pyesorex_args, argname):
                    nargs += 1
            # if no arguments are given at all display the usage information and exit
            # at this point.
            if nargs == 0:
                parser.exit(message=parser.format_usage())

        # Handle the creation of the config file if it is requested. After the configuration
        # file is created the program will exit as this point.
        if hasattr(pyesorex_args, "create_config"):
            _write_config_file(pyesorex, pyesorex_args, recipe_name)
            parser.exit()

        # If we haven't already exited we're in normal (recipe execution) mode, so load the recipe.
        # This will automatically load the recipe config file too, if one was set with command line
        # options or if one exists at the default location.
        pyesorex.load_recipe(recipe_name)

        # If there are recipe args from the command line parse and apply them. This will also
        # set the SOF location in the Pyesorex instance if any was given on the command line
        pyesorex._parse_recipe_arguments(recipe_args[1:])

        # Run recipe
        products = pyesorex.run()

        # Show created products if requested
        _product_sof_show(products, pyesorex_args.list_products[0])

    except (Exception, core.Error) as err:
        # Generally don't want to flood the terminal with Python Tracebacks in case
        # of an error. Instead we catch exceptions, display a short, human readable message
        # that an exception occured and, if the log level is det to debug, put the Traceback
        # in the log.
        #
        # In most cases Pyesorex *should* display and log a human readable error message
        # prior to getting to this point too.
        #
        # Finally we call sys.exit() to set a non-zero exit code just in case someone is
        # using the Pyesorex CLI inside other code (don't!).
        exception_name = err.__class__.__qualname__
        if err.__class__.__module__ != "builtins":
            exception_name = ".".join([err.__class__.__module__, exception_name])
            exception_details = f"file {Path(err.file).name}, line {err.line} in {err.function}"
        else:
            tb = traceback.extract_tb(err.__traceback__)[-1]
            exception_details = f"file {Path(tb.filename).name}, line {tb.lineno} in {tb.name}"
        core.Msg.error("pyesorex", f"Pyesorex raised {exception_name} ({exception_details}), exiting")
        if pyesorex.parameters["log_level"].value.lower() == "debug":
            # If we want the traceback in the log file we need to do it manually
            # because the CPL message system truncates at 1023 characters.
            core.Msg.debug("pyesorex", f"{exception_name} details:")
            core.Msg.stop_file()
            log_dir = Path(pyesorex.parameters["log_dir"].value)
            log_file = Path(pyesorex.parameters["log_file"].value)
            with Path(log_dir / log_file).open("a") as log_file:
                traceback.print_exc(file=log_file)
        sys.exit(1)


def _write_config_file(pyesorex: Pyesorex, pyesorex_args: argparse.Namespace, recipe_name: Optional[str]) -> None:
    """
    Write the configuration of the Pyesorex instance or the recipe to a file

    This either writes the current configuration of the Pyesorex instance or the current
    configuration of a given recipe to a configuration file. The argument pyesorex_args
    must have an attribute create_config when this function is called. This attribute
    represents the arguments given to the create-config command line option, if any
    argument was given. If an argument was given, the attribute is a string which can
    either be a recipe name or the name of the target configuration file.

    If recipe_name is not given, and the create_config attribute is a string it is
    checked whether the attribute value is a recipe or not. In the former case the
    recipe configuration is saved, in the latter the Pyesorex configuration.

    If recipe_name is given, the value of the attribute create_config is assumed to be
    a file name, to which the recipe configuration is written. The recipe_name argument
    is not verified, it is assumed that it refers to an actual recipe.

    If the create_config attribute is not a string, it is True indicating the option
    create-config is present on the command line. In this case and if recipe_name is
    not given the Pyesorex configuration is saved with its default name. If recipe_name
    is given the recipe configuration is saved using defaults. The assumption that
    recipe_name is a valid recipe name also holds in this case. No verification is done.

    This is used internally by pyesorex.pyesorex.__main__.pyesorex_cli(). It should not be
    necessary to call this directly.

    Parameters
    ----------
    pyesorex : pyesorex.pyesorex.Pyesorex
        Pyesorex instance to be used to find recipes.
    pyesorex_args : argparse.Namespace
        Parsed command line options applicable to the Pyesorex instance.
    recipe_name : str, optional
        The name of a valid recipe.
    """

    assert hasattr(pyesorex_args, "create_config")

    if isinstance(pyesorex_args.create_config, str):
        config_arg = pyesorex_args.create_config
        if recipe_name is None:
            if pyesorex.is_c_recipe(config_arg) or pyesorex.is_python_recipe(config_arg):
                pyesorex.load_recipe(config_arg)
                pyesorex.write_recipe_config()
            else:
                pyesorex.write_config(config_arg)
        else:
            pyesorex.load_recipe(recipe_name)
            pyesorex.write_recipe_config(config_arg)
    elif recipe_name is None:
        pyesorex.write_config()
    else:
        pyesorex.load_recipe(recipe_name)
        pyesorex.write_recipe_config()


def _product_sof_summary(sof: FrameSet) -> None:
    nframes = len(sof)
    column_widths = [
        max(3, math.ceil(math.log10(nframes))),
        max([len(f.file) for f in sof]),
    ]
    column_format = f"{{:>{column_widths[0]}}} {{:<{column_widths[1]}}} ({{:<}})"
    for idx, frame in enumerate(sof):
        print(column_format.format(idx, frame.file, frame.tag))


def _product_sof_show(sof: FrameSet, level: str) -> None:
    if level not in ["summary", "details", "dump"]:
        return
    print(f"FrameSet with {len(sof)} frames:")
    if level == "summary":
        _product_sof_summary(sof)
    elif level == "details":
        print(sof)
    elif level == "dump":
        sof.dump()
    return


if __name__ == "__main__":
    pyesorex_cli()
