# This file is part of PyEsorex the Python ESO Recipe Execution Tool
# Copyright (C) 2020-2025 European Southern Observatory
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Custom argparse Action classes for use by the PyEsoRex command line interface."""

import argparse

from pyesorex.pyesorex import Pyesorex


class ParameterAction(argparse.Action):
    """argparse Action used for pyesorex options that correspond to parameter settings.

    These require a custom Action in order to directly set the corresponding pyesorex parameter
    value while the command line arguments are being parsed. This is necessary so that if special
    options such as --create-config or --recipes stop the processing of command line options
    before it is complete the parameter settings will still take effect.

    This is used internally by the pyesorex.pyesore.Pyesorex class and
    pyesorex.pyesorex.__main__.pyesorex_cli(). It should not be necessary to instantiate this
    directly, it is instead passed to an argparse.ArgumentParser instance.

    Parameters
    ----------
    option_strings : list of str
        A list of command-line option strings which should be associated with this action.
    pyesorex : pyesorex.pyesorex.Pyesorex
        Pyesorex instance that will have its parameter value set.
    dest : str
        The name of the attribute to be added to the object returned by parse_args() and,
        more importantly, the name of corresponding parameter in the Pyesorex object.
    nargs : str or int, default="?"
        The number of command line arguments that should be consumed, default "?" (1 or 0)
    const:  str or int or bool or float or complex, default=True
        The value used if nargs is "?" (the default) and the option string is present on the
        command line but is not followed by a value. The type of this argument must match
        the data type of the parameter or be convertable to it by the data type's constructor,
        e.g. a string literal for a value of the correct type.
    metavar: str, optional
        A name for the argument in usage messages.

    Raises
    ------
    KeyError:
        If the Pyesorex object does not have a parameter with a name matching dest.
    """

    def __init__(
        self,
        option_strings,
        pyesorex,
        dest,
        nargs="?",
        const=True,
        metavar=None,
        **kwargs,
    ):
        if not isinstance(pyesorex, Pyesorex):
            msg = f"{self.__class__} requires a Pyesorex instance but got {pyesorex!r}."
            raise ValueError(msg)
        self.pyesorex = pyesorex
        self.param = pyesorex.parameters[dest]
        if metavar is None:
            metavar = self.param.name.upper()
        super().__init__(
            option_strings=option_strings,
            dest=dest,
            nargs=nargs,
            const=const,
            default=argparse.SUPPRESS,
            metavar=metavar,
            help=self.param.cli_help,
        )

    def __call__(self, parser, namespace, values, option_string=None):
        """Sets the corresponding parameter value in the Pyesorex object.

        It should not be necessary to call this directly, it is intended to be called by
        the ArgumentParser via the parse_args() or parse_known_args() methods.

        Parameters
        ----------
        parser : argparse.ArgumentParser
            The calling ArgumentParser object.
        namespace : argparse.Namespace
            The Namespace object that will be returned by parse_args().
        values : tuple
            The associated command line arguments, with any type conversions applied.
        option_string : str, optional
            The option string used to invoke this Action. Will be absent if invoke with a
            positional argument.

        Raises
        ------
        KeyError:
            If the Pyesorex object does not have a parameter with a name matching the dest
            attribute of the Action.

        Notes
        -----
        Directly set the corresponding pyesorex parameter value while the command line arguments
        are being parsed so that if special options such as --create-config or --recipes stop the
        processing of command line options before it is complete the parameter settings will still
        take effect.
        """
        setattr(namespace, self.dest, values)
        # Inconveniently the type of values depends on the `nargs` parameter. If `nargs` is an
        # int, or "*", or "+" then values will be a list of values (even if `nargs=1`), but if
        # `nargs` is "?" then values with be a single item, not a list.
        if self.nargs == "?":
            self.pyesorex.parameters[self.dest].set_from_string(values)
        else:
            self.pyesorex.parameters[self.dest].set_from_string(values[0])


class ConfigAction(ParameterAction):
    """argparse Action used when loading a Pyesorex config file using the command line option.

    This is implemented as an Action so that the config file parameter is set and the config file
    is read before parsing the remaining arguments. This is necessary so that if special
    options such as --create-config or --recipes stop the processing of command line options
    before it is complete the correct config will still be used.

    This is used internally by the pyesorex.pyesore.Pyesorex class and
    pyesorex.pyesorex.__main__.pyesorex_cli(). It should not be necessary to instantiate this
    directly, it is instead passed to an argparse.ArgumentParser instance.

    Parameters
    ----------
    option_strings : list of str
        A list of command-line option strings which should be associated with this action.
    pyesorex : pyesorex.pyesorex.Pyesorex
        Pyesorex instance that will use the config.
    """

    def __init__(self, option_strings, pyesorex, dest="config", **kwargs):
        super().__init__(
            option_strings=option_strings,
            pyesorex=pyesorex,
            dest=dest,
            nargs=1,
            metavar="CONFIG",
        )

    def __call__(self, parser, namespace, values, option_string=None):
        """Sets the config parameter of the Pyesorex object & reads the config file.

        It should not be necessary to call this directly, it is intended to be called by
        the ArgumentParser via the parse_args() or parse_known_args() methods.

        Parameters
        ----------
        parser : argparse.ArgumentParser
            The calling ArgumentParser object.
        namespace : argparse.Namespace
            The Namespace object that will be returned by parse_args().
        values : tuple
            The associated command line arguments, with any type conversions applied.
        option_string : str, optional
            The option string used to invoke this Action. Will be absent if invoke with a
            positional argument.

        Notes
        -----
        Directly set the pyesorex config file parameter and read the config file before parsing
        the remaining arguments. This is necessary so that if special options such as
        --create-config or --recipes stop the processing of command line options before it is
        complete the correct config will still be used.
        """
        # Method of parent class just sets the parameter value
        super().__call__(parser, namespace, values, option_string=None)
        # Having set the parameter value read the config now.
        self.pyesorex.read_config(self.pyesorex.parameters["config"].value)


class CreateConfigAction(ParameterAction):
    """argparse Action used when creating a PyEsoRex/recipe config file at the command line.

    This is implemented as an Action but only adds the relevant attribute to the namespace.
    The actual creation of the config file is done in pyesorex.pyesorex.__main__.pyesorex_cli()
    to handle the creation of the pyesorex config file, and the creation of recipe config files
    with the same command line option syntax that is used by esorex.

    This is used internally by the pyesorex.pyesore.Pyesorex class and
    pyesorex.pyesorex.__main__.pyesorex_cli(). It should not be necessary to instantiate this
    directly, it is instead passed to an argparse.ArgumentParser instance.

    Parameters
    ----------
    option_strings : list of str
        A list of command-line option strings which should be associated with this action.
    pyesorex : pyesorex.pyesorex.Pyesorex
        Pyesorex instance to be used to create config files.
    """

    def __init__(self, option_strings, pyesorex, dest="create_config", **_kwargs):
        super().__init__(
            option_strings=option_strings,
            pyesorex=pyesorex,
            dest=dest,
            nargs="?",
            metavar="[CONFIG] [RECIPE]",
        )

    def __call__(self, _parser, namespace, value, _option_string):
        setattr(namespace, self.dest, value)


class ParamsAction(ParameterAction):
    """argparse Action used when listing current parameter values.

    This is implemented as an Action in order to interrupt parsing of command line arguments
    after listing the parameter values. This is necessary because otherwise there would be
    error due to missing positonal arguments (sof_location, recipe).

    This is used internally by the pyesorex.pyesore.Pyesorex class and
    pyesorex.pyesorex.__main__.pyesorex_cli(). It should not be necessary to instatiate this
    directly, it is instead passed to an argparse.ArgumentParser instance.

    Parameters
    ----------
    option_strings : list of str
        A list of command-line option strings which should be associated with this action.
    pyesorex : pyesorex.pyesorex.Pyesorex
        Pyesorex instance to be used to find recipes.
    """

    def __init__(self, option_strings, pyesorex, dest="params", **kwargs):
        super().__init__(
            option_strings=option_strings,
            pyesorex=pyesorex,
            dest=dest,
            nargs="?",
            metavar="RECIPE",
        )

    def __call__(self, parser, namespace, values, option_string=None):
        """Lists the current parameter values then causes the parser to exit.

        It should not be necessary to call this directly, it is intended to be called by
        the ArgumentParser via the parse_args() or parse_known_args() methods.

        Parameters
        ----------
        parser : argparse.ArgumentParser
            The calling ArgumentParser object.
        namespace : argparse.Namespace
            The Namespace object that will be returned by parse_args().
        values : tuple
            The associated command line arguments, with any type conversions applied.
        option_string : str, optional
            The option string used to invoke this Action. Will be absent if invoke with a
            positional argument.
        """
        self.pyesorex.parameters[self.dest].value = True
        # Values will either be True (--params option with no recipe name) or it
        # will be a string (--params option followed by a recipe name).
        if values is not True:
            # Got passed a recipe name, load it.
            try:
                self.pyesorex.load_recipe(recipe_name=values)
            except Exception as err:
                # Couldn't load the recipe, but list Pyesorex parameters anyway.
                print(self.pyesorex.get_params_text(include_recipe_params=False))
                raise err
        print(self.pyesorex.get_params_text(include_recipe_params=(values is not True)))
        print(parser.epilog + "\n")
        parser.exit()


class RecipesAction(ParameterAction):
    """argparse Action used when listing available recipes on the command line.

    This is implemented as an Action in order to interrupt parsing of command line arguments
    after listing the recipes. This is necessary because otherwise there would be error
    due to missing positonal arguments (sof_location, recipe).

    This is used internally by the pyesorex.pyesore.Pyesorex class and
    pyesorex.pyesorex.__main__.pyesorex_cli(). It should not be necessary to instatiate this
    directly, it is instead passed to an argparse.ArgumentParser instance.

    Parameters
    ----------
    option_strings : list of str
        A list of command-line option strings which should be associated with this action.
    pyesorex : pyesorex.pyesorex.Pyesorex
        Pyesorex instance to be used to find recipes.
    """

    def __init__(self, option_strings, pyesorex, dest="recipes", **kwargs):
        super().__init__(option_strings=option_strings, pyesorex=pyesorex, dest=dest, nargs=0)

    def __call__(self, parser, namespace, values, option_string=None):
        """Lists the available recipes then causes the parser to exit.

        It should not be necessary to call this directly, it is intended to be called by
        the ArgumentParser via the parse_args() or parse_known_args() methods.

        Parameters
        ----------
        parser : argparse.ArgumentParser
            The calling ArgumentParser object.
        namespace : argparse.Namespace
            The Namespace object that will be returned by parse_args().
        values : tuple
            The associated command line arguments, with any type conversions applied.
        option_string : str, optional
            The option string used to invoke this Action. Will be absent if invoke with a
            positional argument.
        """
        print(self.pyesorex.get_recipes_text())
        parser.exit()


class HelpAction(ParameterAction):
    """argparse Action used for displaying the help page, for both pyesorex itself and for recipes

    This is implemented as an Action in order to interrupt parsing of command line arguments
    after listing the recipes. This is necessary because otherwise there would be error
    due to missing positonal arguments (sof_location, recipe).

    This is used internally by the pyesorex.pyesore.Pyesorex class and
    pyesorex.pyesorex.__main__.pyesorex_cli(). It should not be necessary to instatiate this
    directly, it is instead passed to an argparse.ArgumentParser instance.

    Parameters
    ----------
    option_strings : list of str
        A list of command-line option strings which should be associated with this action.
    pyesorex : pyesorex.pyesorex.Pyesorex
        Pyesorex instance to be used to find recipes.
    """

    def __init__(self, option_strings, pyesorex, dest="help", **kwargs):
        if not isinstance(pyesorex, Pyesorex):
            msg = f"{self.__class__} requires a Pyesorex instance but got {pyesorex!r}."
            raise ValueError(msg)
        self.pyesorex = pyesorex
        super().__init__(
            option_strings=option_strings,
            pyesorex=pyesorex,
            dest=dest,
            nargs="?",
            default="",
        )

    def __call__(self, parser, namespace, value, option_string=None):
        """Prints the help page, either the recipe help if passed or the main PyEsoRex page with all
        possible command line args. Quits when page is shown.

        It should not be necessary to call this directly, it is intended to be called by
        the ArgumentParser via the parse_args() or parse_known_args() methods.

        Parameters
        ----------
        parser : argparse.ArgumentParser
            The calling ArgumentParser object.
        namespace : argparse.Namespace
            The Namespace object that will be returned by parse_args().
        value : str or bool
            Name of the recipe passed, or bool True if no recipe passed.
        option_string : str, optional
            The option string used to invoke this Action. Will be absent if invoke with a
            positional argument.
        """
        if value is True:  # value resolves to boolean True if not passed any cmd values
            parser.print_help()  # Print default argparse help (pyesorex commands)
        else:
            # value is a recipe name instead of a bool
            self.pyesorex.load_recipe(value)  # Get pyesorex to load the recipe first
            print(self.pyesorex.get_recipe_help())  # Print recipe help page using the loaded recipe
        parser.exit()


class ManPageAction(ParameterAction):
    """argparse Action used for displaying the manual page for a given recipe

    This is implemented as an Action in order to interrupt parsing of command line arguments
    after listing the recipes. This is necessary because otherwise there would be error
    due to missing positonal arguments (sof_location, recipe).

    This is used internally by the pyesorex.pyesore.Pyesorex class and
    pyesorex.pyesorex.__main__.pyesorex_cli(). It should not be necessary to instatiate this
    directly, it is instead passed to an argparse.ArgumentParser instance.

    Parameters
    ----------
    option_strings : list of str
        A list of command-line option strings which should be associated with this action.
    pyesorex : pyesorex.pyesorex.Pyesorex
        Pyesorex instance to be used to find recipes.
    """

    def __init__(self, option_strings, pyesorex, dest="help", **kwargs):
        if not isinstance(pyesorex, Pyesorex):
            msg = f"{self.__class__} requires a Pyesorex instance but got {pyesorex!r}."
            raise ValueError(msg)
        self.pyesorex = pyesorex
        super().__init__(
            option_strings=option_strings,
            pyesorex=pyesorex,
            dest=dest,
            nargs=1,
            default="",
        )

    def __call__(
        self,
        parser,
        namespace,
        values,  # Note: is a 1 element list so will have to fetch from index 0
        option_string=None,
    ):
        """Prints the manual page by using the pyesorex object to load the recipe and passing
        it to the docutils

        It should not be necessary to call this directly, it is intended to be called by
        the ArgumentParser via the parse_args() or parse_known_args() methods.

        Parameters
        ----------
        parser : argparse.ArgumentParser
            The calling ArgumentParser object.
        namespace : argparse.Namespace
            The Namespace object that will be returned by parse_args().
        value : list or bool
            1 element list with name of the recipe passed, or bool True if no recipe passed.
        option_string : str, optional
            The option string used to invoke this Action. Will be absent if invoke with a
            positional argument.
        """
        self.pyesorex.load_recipe(values[0])  # Get pyesorex to load the recipe first
        print(self.pyesorex.get_recipe_manual())  # Print the manual
        parser.exit()
