# This file is part of PyEsorex the Python ESO Recipe Execution Tool
# Copyright (C) 2020-2025 European Southern Observatory
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Alternatives to the Parameter and ParameterList classes from cpl.ui for use by PyEsoRex."""

from collections import UserList
from typing import Optional

from cpl import core, ui

from pyesorex.action import ParameterAction

type_string = {
    core.Type.CHAR: "char",
    core.Type.UCHAR: "uchar",
    core.Type.BOOL: "bool",
    core.Type.SHORT: "short",
    core.Type.USHORT: "ushort",
    core.Type.INT: "int",
    core.Type.UINT: "uint",
    core.Type.LONG: "long",
    core.Type.ULONG: "ulong",
    core.Type.LONG_LONG: "long_long",
    core.Type.SIZE: "size",
    core.Type.FLOAT: "float",
    core.Type.DOUBLE: "double",
    core.Type.FLOAT_COMPLEX: "float_complex",
    core.Type.DOUBLE_COMPLEX: "double_complex",
    core.Type.STRING: "string",
    core.Type.ARRAY: "array",
}


def get_string_converter(parameter):
    """Returns a function to convert a string to the data type of the given parameter.

    Parameters
    ----------
    parameter : cpl.ui.Parameter or pyesorex.pyesorex.Parameter
        Parameter object used to determine the required data type.

    Returns
    -------
    callable
        Function that will convert a str argument to the data type needed to set the value
        of parameter. May be true_of_false, float, int, complex or str.
    """
    # Use data_type property to find a suitable Python type to set the parameter value with.
    if parameter.data_type == core.Type.BOOL:
        conversion_function = true_or_false
    elif parameter.data_type in {core.Type.FLOAT, core.Type.DOUBLE}:
        conversion_function = float
    elif parameter.data_type in {
        core.Type.CHAR,
        core.Type.UCHAR,
        core.Type.SHORT,
        core.Type.USHORT,
        core.Type.INT,
        core.Type.UINT,
        core.Type.LONG,
        core.Type.ULONG,
        core.Type.LONG_LONG,
        core.Type.SIZE,
    }:
        conversion_function = int
    elif parameter.data_type in {core.Type.FLOAT_COMPLEX, core.Type.DOUBLE_COMPLEX}:
        conversion_function = complex
    else:
        conversion_function = str

    return conversion_function


def true_or_false(string):
    """Converts a string "True" or "False" to the corresponding bool value.

    Parameters
    ----------
    string : str
        String to be converted to a bool. Must be either "True" or "False",
        but the case does not matter.

    Returns
    -------
    bool
        The boolean value that was spelled out in string.
    """
    if isinstance(string, str):
        string_low = string.lower()
        if string_low == "true":
            return True
        if string_low == "false":
            return False
        msg = f"Value must be 'True' or 'False', got {string!r}"
        raise ValueError(msg)
    # Not a string
    return bool(string)


class Parameter:
    """Base class.

    This base class is an alternative to that provided by PyCPL, providing some additional
    properties and methods that are helpful for PyEsoRex as well as an alternative string
    representation. This class is not intended to be instantiated itself, one of the derived
    classes in this module should be used instead.

    See Also
    --------
    cpl.ui.Parameter : Parameter base class from PyCPL
    """

    @property
    def cfg_help(self):
        """Constructs a help string for use in PyEsoRex/recipe configuration files.

        If the parameter has env_alias attribute that differs from the parameter name, i.e the
        parameter can be set with an environment variable, a note will be added to the help string
        to that effect, otherwise the help string is simply the parameter's description attribute.

        Returns
        -------
        str
            Configuration help string.
        """
        help_string = f"{self.description}"
        if self.env_alias != self.name:
            help_string += f" This option may also be set using the environment variable {self.env_alias}."
        return help_string

    @property
    def cli_help(self):
        """Constructs a help string for use in the PyEsoRex command line help.

        The command line help string is the came as the config help string, except that the
        current value of the parameter is appended to the end in square brakets.

        Returns
        -------
        str
            CLI help string.

        See Also
        --------
        cfg_help : Constructs a help string for use in PyEsoRex/recipe configuration files.
        """
        return self.cfg_help + f" [{self.value}]"

    @property
    def valtype(self):
        """String representation of the parameter data type."""
        return type_string[self.data_type]

    def set_from_string(self, string_value):
        """Set the value of the parameter from a string, performing any necessary conversions.

        Uses an appropriate function to convert the string to the data type of the parameter
        (e.g. float(), int(), true_of_false()) then sets the value with it.

        Parameters
        ----------
        string_value : str
            String containing the new value for the parameter.

        See Also
        --------
        get_string_converter : Returns a function to convert a string to the data type of the
            given parameter.
        """
        conversion_function = get_string_converter(self)
        self.value = conversion_function(string_value)

    @classmethod
    def from_cplui(cls, parameter):
        """
        Creates a PyEsoRex parameter object from a cpl.ui.Parameter* object

        Parameters
        ----------
        parameter : cpl.ui.Parameter
            An instance of one of the cpl.ui.Parameter* classes.
        """
        _parameter = None
        if isinstance(parameter, ui.ParameterRange):
            return ParameterRange(
                name=parameter.name,
                description=parameter.description,
                context=parameter.context,
                default=parameter.default,
                min=parameter.min,
                max=parameter.max,
                tag=parameter.tag,
                value=parameter.value,
                cli_alias=parameter.cli_alias,
                env_alias=parameter.env_alias,
                cfg_alias=parameter.cfg_alias,
                presence=parameter.presence,
            )
        if isinstance(parameter, ui.ParameterEnum):
            return ParameterEnum(
                name=parameter.name,
                description=parameter.description,
                context=parameter.context,
                default=parameter.default,
                alternatives=parameter.alternatives,
                tag=parameter.tag,
                value=parameter.value,
                cli_alias=parameter.cli_alias,
                env_alias=parameter.env_alias,
                cfg_alias=parameter.cfg_alias,
                presence=parameter.presence,
            )
        if isinstance(parameter, ui.ParameterValue):
            return ParameterValue(
                name=parameter.name,
                description=parameter.description,
                context=parameter.context,
                default=parameter.default,
                tag=parameter.tag,
                value=parameter.value,
                cli_alias=parameter.cli_alias,
                env_alias=parameter.env_alias,
                cfg_alias=parameter.cfg_alias,
                presence=parameter.presence,
            )
        msg = f"parameter must be an instance of a cpl.ui.Parameter subclass, got {parameter!r}"
        raise ValueError(msg)

    @classmethod
    def to_cplui(cls, parameter):
        """
        Creates a PyCPL parameter object from a pyesorex.parameter.Parameter object

        Parameters
        ----------
        parameter : pyesorex.parameter.Parameter
            An instance of one of the pyesorex.parameter.Parameter classes.
        """
        _parameter = None
        if isinstance(parameter, Parameter):
            if isinstance(parameter, ParameterValue):
                _parameter = ui.ParameterValue(
                    name=parameter.name,
                    description=parameter.description,
                    context=parameter.context,
                    default=parameter.default,
                )
            if isinstance(parameter, ParameterRange):
                _parameter = ui.ParameterRange(
                    name=parameter.name,
                    description=parameter.description,
                    context=parameter.context,
                    default=parameter.default,
                    min=parameter.min,
                    max=parameter.max,
                )
            if isinstance(parameter, ParameterEnum):
                _parameter = ui.ParameterEnum(
                    name=parameter.name,
                    description=parameter.description,
                    context=parameter.context,
                    default=parameter.default,
                    alternatives=parameter.alternatives,
                    max=parameter.max,
                )
            if _parameter is not None:
                _parameter.tag = parameter.tag
                _parameter.cli_alias = parameter.cli_alias
                _parameter.cfg_alias = parameter.cfg_alias
                _parameter.env_alias = parameter.env_alias
                _parameter.value = parameter.value
                # Copy the presence flag after the parameter value is set!
                # Setting the value will update the flag. To keep the flag
                # from the source parameter this is copied after the value
                # was set.
                _parameter.presence = parameter.presence
                return _parameter
        msg = f"parameter must be an instance of a pyesorex.parameter.Parameter subclass, got {parameter!r}"
        raise ValueError(msg)

    def __repr__(self):
        return (
            f"<{self.__class__.__module__}.{self.__class__.__qualname__}: "
            + f"name={self.name!r}, value={self.value!r}>"
        )


class ParameterValue(Parameter, ui.ParameterValue):
    """ParameterValue class.

    This class is an alternative to that provided by PyCPL, adding the option to set the tag,
    value, cli_alias, env_alias and cfg_alias attributes when creating an instance.

    Parameters
    ----------
    name : str
        Name of the parameter.
    description : str
        Description of the parameter.
    context : str
        Context of the parameter.
    default : str or int or bool or float or complex
        Default value for the parameter. The type of this argument is used to set the data type
        of the parameter.
    tag : str, optional
        Tag of the parameter.
    value : str or int or bool or float or complex, optional
        Initial value for the parameter. The type of this argument must match the data type of
        the parameter. If not given the initial value of the parameter will be equal to default.
    cli_alias : str, optional
        Alternative name used to refer to the parameter on the command line. If not given will
        be set to be the same value as name.
    env_alias : str, optional
        Alternative name used when setting the parameter from environment variables. If not given
        will be set to be the same value as name.
    cfg_alias : str, optional
        Alternative name use to refer to the parameter in configuration files. If not given will
        be set to the same value as name.
    action : argparse.Action, optional
        Action class to be used when the parameter is being set using command line arguments,
        default pyesorex.pyesorex.ParameterAction.
    nargs : str or int, optional
        "Number of args" for the corresponding command line option. Can be an integer >= 0, "?",
        "*" or "+", see the Python Standard Library documentation for
        `argparse.ArgumentParser.add_argument()` for details. The default is "?", i.e. 1 or 0.
    const : str or int or bool or float or complex, optional
        Value the parameter will be set to if its command line option is used without a following
        argument, if nargs="?", see the Python Standard Library documentation for
        `argparse.ArgumentParser.add_argument()` for details. The type of this argument must match
        the data type of the parameter or be convertable to it by the data type's constructor,
        e.g. a string literal for a value of the correct type. The default is True.
    callback : callable, default=None
        A function of two variables (the Parameter object and the new value) that will be called
        whenever the parameter value is changed. The callback is called before setting the
        parameter value so it can be used to validate the new value (raising an Exception to
        prevent the change of value).

    See Also
    --------
    cpl.ui.ParameterValue : ParameterValue class from PyCPL
    """

    def __init__(
        self,
        name,
        description,
        context,
        default,
        tag=None,
        value=None,
        cli_alias=None,
        env_alias=None,
        cfg_alias=None,
        action=ParameterAction,
        nargs="?",
        const=True,
        callback=None,
        presence: Optional[bool] = None,
    ):
        super().__init__(name, description, context, default)
        if tag is not None:
            self.tag = tag
        if cli_alias is not None:
            self.cli_alias = cli_alias
        if env_alias is not None:
            self.env_alias = env_alias
        if cfg_alias is not None:
            self.cfg_alias = cfg_alias
        self.action = action
        self.nargs = nargs
        self.const = const
        self.callback = callback
        if value is not None:
            self.value = value
        # Copy the presence flag after the parameter value is set!
        # Setting the value will update the flag. To keep the flag
        # from the source parameter this is copied after the value
        # was set.
        if presence is not None:
            self.presence = presence

    @ui.ParameterValue.value.setter
    def value(self, new_value):
        """Override the value setter to enable callback functionality."""
        if self.callback is not None:
            self.callback(self, new_value)
        # Call base class setter to actually set the value
        ui.ParameterValue.value.fset(self, new_value)

    @property
    def partype(self):
        """String representation of the parameter type, in this case 'value'."""
        return "value"

    def as_dict(self, defaults=False):
        """Returns a dict representation suitable for serialisation by json.dump().

        Parameters
        ----------
        defaults : bool, default False
            If True the value linked to the "value" key in the returned dictionary will
            be set to the parameter's default value, not its current value.

        Returns
        -------
        dict
            Dictionary containing the key: value pairs required for JSON format config files.
        """
        return {
            "name": self.cfg_alias,
            "value": self.default if defaults else self.value,
            "valtype": self.valtype,
            "partype": self.partype,
            "display_name": self.cli_alias,
            "description": self.description,
        }


class ParameterRange(Parameter, ui.ParameterRange):
    """ParameterRange class.

    This class is an alternative to that provided by PyCPL, adding the option to set the tag,
    value, cli_alias, env_alias and cfg_alias attributes when creating an instance.

    Parameters
    ----------
    name : str
        Name of the parameter.
    description : str
        Description of the parameter.
    context : str
        Context of the parameter.
    default : str or int or bool or float or complex
        Default value for the parameter. The type of this argument is used to set the data type
        of the parameter.
    min : int or float
        Minimum allowable value for the parameter.
    max : int or float
        Maximum allowable value for the parameter.
    tag : str, optional
        Tag of the parameter.
    value : str or int or bool or float or complex, optional
        Initial value for the parameter. The type of this argument must match the data type of
        the parameter. If not given the initial value of the parameter will be equal to default.
    cli_alias : str, optional
        Alternative name used to refer to the parameter on the command line. If not given will
        be set to be the same value as name.
    env_alias : str, optional
        Alternative name used when setting the parameter from environment variables. If not given
        will be set to be the same value as name.
    cfg_alias : str, optional
        Alternative name use to refer to the parameter in configuration files. If not given will
        be set to the same value as name.
    action : argparse.Action, optional
        Action class to be used when the parameter is being set using command line arguments,
        default pyesorex.pyesorex.ParameterAction
    nargs : str or int, optional
        "Number of args" for the corresponding command line option. Can be an integer >= 0, "?",
        "*" or "+", see the Python Standard Library documentation for
        `argparse.ArgumentParser.add_argument()` for details. The default is "?", i.e. 1 or 0.
    const : str or int or bool or float or complex, optional
        Value the parameter will be set to if its command line option is used without a following
        argument, if nargs="?", see the Python Standard Library documentation for
        `argparse.ArgumentParser.add_argument()` for details. The type of this argument must match
        the data type of the parameter or be convertable to it by the data type's constructor,
        e.g. a string literal for a value of the correct type. The default is True.
    callback : callable, default=None
        A function of two variables (the Parameter object and the new value) that will be called
        whenever the parameter value is changed. The callback is called before setting the
        parameter value so it can be used to validate the new value (raising an Exception to
        prevent the change of value).

    See Also
    --------
    cpl.ui.ParameterRange : ParameterRange class from PyCPL
    """

    def __init__(
        self,
        name,
        description,
        context,
        default,
        min,
        max,
        tag=None,
        value=None,
        cli_alias=None,
        env_alias=None,
        cfg_alias=None,
        action=ParameterAction,
        nargs="?",
        const=True,
        callback=None,
        presence: Optional[bool] = None,
    ):
        super().__init__(name, description, context, default, min, max)
        if tag is not None:
            self.tag = tag
        if cli_alias is not None:
            self.cli_alias = cli_alias
        if env_alias is not None:
            self.env_alias = env_alias
        if cfg_alias is not None:
            self.cfg_alias = cfg_alias
        self.action = action
        self.nargs = nargs
        self.const = const
        self.callback = callback
        if value is not None:
            self.value = value
        # Copy the presence flag after the parameter value is set!
        # Setting the value will update the flag. To keep the flag
        # from the source parameter this is copied after the value
        # was set.
        if presence is not None:
            self.presence = presence

    @property
    def cfg_help(self):
        """Constructs a help string for use in PyEsoRex/recipe configuration files.

        The string starts with the parameter's description attribute, then if the parameter has
        env_alias attribute that differs from the parameter name, i.e the parameter can be set
        with an environment variable, a note will be added to the help string to that effect.
        Finally the minimum and maximum values for the parameter are added to the end of the
        string.

        Returns
        -------
        str
            Configuration help string.
        """
        help_string = super().cfg_help
        help_string += f" ({self.min}, {self.max})"
        return help_string

    @ui.ParameterRange.value.setter
    def value(self, new_value):
        """Override the value setter to enable callback functionality."""
        if self.callback is not None:
            self.callback(self, new_value)
        # Call base class setter to actually set the value
        ui.ParameterRange.value.fset(self, new_value)

    @property
    def partype(self):
        """String representation of the parameter type, in this case 'value'."""
        return "range"

    def as_dict(self, defaults=False):
        """Returns a dict representation suitable for serialisation by json.dump().

        Parameters
        ----------
        defaults : bool, default False
            If True the value linked to the "value" key in the returned dictionary will
            be set to the parameter's default value, not its current value.

        Returns
        -------
        dict
            Dictionary containing the key: value pairs required for JSON format config files.
        """
        return {
            "name": self.cfg_alias,
            "value": self.default if defaults else self.value,
            "valtype": self.valtype,
            "partype": self.partype,
            "valmin": self.min,  # TODO: confirm that EsoRex would call this valmin
            "valmax": self.max,  # TODO: confirm that EsoRex would call this valmax
            "display_name": self.cli_alias,
            "description": self.description,
        }

    def __repr__(self):
        return (
            f"<{self.__class__.__module__}.{self.__class__.__qualname__}: "
            + f"name={self.name!r}, value={self.value!r}, min={self.min!r}, max={self.max!r}>"
        )


class ParameterEnum(Parameter, ui.ParameterEnum):
    """ParameterEnum class.

    This class is an alternative to that provided by PyCPL, adding the option to set the tag,
    value, cli_alias, env_alias and cfg_alias attributes when creating an instance.

    Parameters
    ----------
    name : str
        Name of the parameter.
    description : str
        Description of the parameter.
    context : str
        Context of the parameter.
    default : str or int or bool or float or complex
        Default value for the parameter. The type of this argument is used to set the data type
        of the parameter.
    alternatives : sequence of str or int or float
        Sequence containing the allowed values for the parameter.
    tag : str, optional
        Tag of the parameter.
    value : str or int or float or complex, optional
        Initial value for the parameter. The type of this argument must match the data type of
        the parameter. If not given the initial value of the parameter will be equal to default.
    cli_alias : str, optional
        Alternative name used to refer to the parameter on the command line. If not given will
        be set to be the same value as name.
    env_alias : str, optional
        Alternative name used when setting the parameter from environment variables. If not given
        will be set to be the same value as name.
    cfg_alias : str, optional
        Alternative name use to refer to the parameter in configuration files. If not given will
        be set to the same value as name.
    action : argparse.Action, optional
        Action class to be used when the parameter is being set using command line arguments,
        default pyesorex.pyesorex.ParameterAction.
    nargs : str or int, optional
        "Number of args" for the corresponding command line option. Can be an integer >= 0, "?",
        "*" or "+", see the Python Standard Library documentation for
        `argparse.ArgumentParser.add_argument()` for details. The default is "?", i.e. 1 or 0.
    const : str or int or bool or float or complex, optional
        Value the parameter will be set to if its command line option is used without a following
        argument, if nargs="?", see the Python Standard Library documentation for
        `argparse.ArgumentParser.add_argument()` for details. The type of this argument must match
        the data type of the parameter or be convertable to it by the data type's constructor,
        e.g. a string literal for a value of the correct type. The default is True.
    callback : callable, default=None
        A function of two variables (the Parameter object and the new value) that will be called
        whenever the parameter value is changed. The callback is called before setting the
        parameter value so it can be used to validate the new value (raising an Exception to
        prevent the change of value).

    See Also
    --------
    cpl.ui.ParameterEnum : ParameterEnum class from PyCPL
    """

    def __init__(
        self,
        name,
        description,
        context,
        default,
        alternatives,
        tag=None,
        value=None,
        cli_alias=None,
        env_alias=None,
        cfg_alias=None,
        action=ParameterAction,
        nargs="?",
        const=True,
        callback=None,
        presence: Optional[bool] = None,
    ):
        super().__init__(name, description, context, default, alternatives)
        if tag is not None:
            self.tag = tag
        if cli_alias is not None:
            self.cli_alias = cli_alias
        if env_alias is not None:
            self.env_alias = env_alias
        if cfg_alias is not None:
            self.cfg_alias = cfg_alias
        self.action = action
        self.nargs = nargs
        self.const = const
        self.callback = callback
        if value is not None:
            self.value = value
        # Copy the presence flag after the parameter value is set!
        # Setting the value will update the flag. To keep the flag
        # from the source parameter this is copied after the value
        # was set.
        if presence is not None:
            self.presence = presence

    @property
    def cfg_help(self):
        """Constructs a help string for use in PyEsoRex/recipe configuration files.

        The string starts with the parameter's description attribute, then if the parameter has
        env_alias attribute that differs from the parameter name, i.e the parameter can be set
        with an environment variable, a note will be added to the help string to that effect.
        Finally the alternative values for the parameter are added to the end of the
        string.

        Returns
        -------
        str
            Configuration help string.
        """
        help_string = super().cfg_help
        help_string += f" <{'|'.join([str(alt) for alt in self.alternatives])}>"
        return help_string

    def __repr__(self):
        return (
            f"<{self.__class__.__module__}.{self.__class__.__qualname__}: "
            + f"name={self.name!r}, value={self.value!r}, alternatives={self.alternatives!r}>"
        )

    @ui.ParameterEnum.value.setter
    def value(self, new_value):
        """Override the value setter to enable callback functionality."""
        if self.callback is not None:
            self.callback(self, new_value)
        # Call base class setter to actually set the value
        ui.ParameterEnum.value.fset(self, new_value)

    @property
    def partype(self):
        """String representation of the parameter type, in this case 'value'."""
        return "enum"

    def as_dict(self, defaults=False):
        """Returns a dict representation suitable for serialisation by json.dump().

        Parameters
        ----------
        defaults : bool, default False
            If True the value linked to the "value" key in the returned dictionary will
            be set to the parameter's default value, not its current value.

        Returns
        -------
        dict
            Dictionary containing the key: value pairs required for JSON format config files.
        """
        return {
            "name": self.cfg_alias,
            "value": self.default if defaults else self.value,
            "valtype": self.valtype,
            "partype": self.partype,
            "valenum": self.alternatives,
            "display_name": self.cli_alias,
            "description": self.description,
        }


class ParameterList(UserList):
    """ParameterList class.

    This class is an alternative to that provided by PyCPL, adding new methods and a more
    useful string format useful for PyEsoRex. This class allows parameters to be accessed by
    integer index, by the parameter name, or by the parameter's cfg_alias.

    Parameters
    ----------
    parameters : sequence of cpl.ui.Parameter or pyesorex.pyesorex.Parameter, optional
        Initial set of parameters for the parameter list.

    See Also
    --------
    cpl.ui.ParameterList : ParameterListClass from PyCPL

    Notes
    -----
    Needed to replace PyCPL ParameterList because that can't hold the derived Parameter classes
    from this module. The PyCPL ParameterList isn't really like a Python list, basically it
    implement the Sequence interface but adds a custom append instead of implementing __setitem__,
    __delitem__ and insert as would be required for MutableSequence, then throws in an inefficient
    alternative __getitem__ for lookup by parameter name instead of index. The closest match is
    probably subclassing UserList and overriding __getitem__ and __setitem__ to add the lookup by
    name option.
    """

    def update(self, other):
        """Update the values of the parameters in the list from a dictionary, like the update
        method of Python dictionaries.

        Parameters
        ----------
        other : dict
            Dictionary containing parameters names & new values.
        """
        for key, value in other.items():
            self[key] = value

    def as_dict(self, modified_only=True):
        """
        Return the names and values of the parameters in the list as a dictionary.

        Parameters
        ----------
        modified_only: bool, optional
            If True only parameters which were modified, as indicated by the parameter presence attribute,
            are returned. If False all parameters are returned ignoring the presence attribute of the parameter.

        Returns
        -------
        dict
            Dictionary containing the names and values of the parameters in the list. By default
            only modified parameters are returned.
        """
        if modified_only is True:
            parameters = {}
            for param in self:
                if param.presence is True:
                    parameters[param.name] = param.value
            return parameters
        return {param.name: param.value for param in self}

    def __getitem__(self, key):
        try:
            return self.data[key]
        except TypeError:
            # Key is not an integer or slice, might be a name. Do an inefficient name search.
            for parameter in self.data:
                # Check the config alias as well as name, useful when reading config files.
                if key in [parameter.name, parameter.cfg_alias]:
                    return parameter
            msg = f"No parameter named {key!r} in list."
            raise KeyError(msg) from None

    def __setitem__(self, key, value):
        try:
            self.data[key] = value
        except TypeError:
            # Key is not an integer or slice, might be a name. Do an inefficient name search.
            for parameter in self.data:
                # Check the config alias as well as name, useful when reading config files.
                if key in [parameter.name, parameter.cfg_alias]:
                    parameter.value = value
                    return
            msg = f"No parameter named {key!r} in list."
            raise KeyError(msg) from None

    def __str__(self):
        s = "ParameterList:\n"
        for parameter in self.data:
            s += f"    {parameter}\n"
        return s
