# This file is part of PyEsorex the Python ESO Recipe Execution Tool
# Copyright (C) 2020-2025 European Southern Observatory
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Main module for the Pyesorex PyCPL recipe execution environment."""

import argparse
import copy
import importlib.util
import inspect
import json
import os
import shutil
import textwrap
import time
from collections import namedtuple
from contextlib import suppress
from datetime import datetime
from pathlib import Path

from cpl import core, dfs, ui

from pyesorex.__version__ import __version__

RecipeMetadata = namedtuple("RecipeMetadata", ["name", "synopsis", "description", "version"])


class Pyesorex:
    """
    PyCPL recipe execution environment.

    The Pyesorex class is a tool for executing CPL recipes, both classic CPL recipes written
    in C and PyCPL recipes written in Python.

    Parameters
    ----------
    config : str, optional
        Path of a PyEsoRex configuration file. If given Pyesorex will load it on instantiation,
        otherwise it will try to load a configuration from the default location,
        ~/.pyesorex/pyesorex.rc.
    config_format : str, optional
        Format of the config file, either "rc" or "json" (not yet implemented). If not given
        Pyesorex will infer the format of the config file from the file extension.

    Other Parameters
    ----------------
    display_import_errors : bool, optional
        When True any import errors encountered while searching for Python recipes will be
        displayed as warnings. When False import errors will be suppressed.
    link_dir : str, optional
        The directory in which a symbolic link to each of the product files should be
        written. The enable/disable switch to control whether the link is actually made is the
        `suppress_link` parameter.
    log_dir : str, optional
        Directory where to place the logfile.
    log_file : str, optional
        Filename of logfile.
    log_level : str, optional
        Controls the severity level of messages that will be printed to the logfile. Must be
        one of "debug", "info", "warning", "error", "off".
    msg_level : str, optional
        Controls the severity level of messages that will be printed to the terminal. Must be
        one of "debug", "info", "warning", "error", "off".
    no_datamd5 : bool, optional
        Disables the computation of the MD5 data hash for FITS product files.
    no_checksum : bool, optional
        Disables the computation of the standard FITS product checksums
    output_dir : str, optional
        The directory where the product files should be finally moved to (all products are
        first created in the current dir).
    output_mkdir : bool, optional
        When True output and link directories will be created if they do not already exist.
        When False an output or link directory that does not exist will result in an error,
        as it does with EsoRex.
    output_prefix : str, optional
        Prefix applied to any output file. For example, specifying 'pre' would translate
        'filename.fits' to 'pre_0000.fits'. See also the `suppress_prefix` parameter.
    products_sof : str, optional
        Output file which contains the FITS files created by the recipe. If the filename
        ends with extension .json then a machine-readable JSON format will be used.
    products_sof_abspath : bool, optional
        When True file paths in products SOF files will be converted to absolute paths,
        otherwise they will not be modified.
    recipe_config : str, optional
        Configuration file for any selected recipe.
    recipe_dirs : str, optional
        Directory containing recipe libraries. Note that esorex will recursively search not
        only the specified directory, but all sub-directories below it as well. Multiple
        directory heads may be specified, by separating the starting paths with colons (:).
    suppress_link : bool, optional
        When True, no symbolic link is created to the output product. However, if False,
        then a symbolic link is created in the directory specified with the parameter
        `link-dir` for each product that is created by the recipe.
    suppress_prefix : bool, optional
        When True, the original name of the output product, as produced by the recipe, is
        maintained. If False, then the name of the output file is changed to the
        "prefix_number" format. The prefix can be altered using the `output_prefix` option.
    time : bool, optional
        Measure and show the recipe's execution time.
    """

    def __init__(self, config=None, config_format=None, **kwargs):
        # Deferred imports necessary to avoid circular import problems.
        from pyesorex.action import (
            ConfigAction,
            CreateConfigAction,
            HelpAction,
            ManPageAction,
            ParamsAction,
            RecipesAction,
        )
        from pyesorex.parameter import ParameterEnum, ParameterList, ParameterValue

        # Set up parameters with default values.
        self.parameters = ParameterList(
            (
                ParameterValue(
                    name="help",  # cli_alias automatically set to help
                    context="pyesorex",
                    description="Display this help and exit. If a recipe name is also given, then help will be given "
                    "for it as well. PyEsoRex will exit after this option. It should always be the last option on the "
                    "command line.",
                    default=False,
                    # Empty cfg_alias string prevent this appearing in the config file.
                    cfg_alias="",
                    nargs="?",
                    action=HelpAction,
                ),
                ParameterValue(
                    name="config",
                    context="pyesorex",
                    description="Configuration file to be used for PyEsoRex.",
                    default="~/.pyesorex/pyesorex.rc",
                    # Empty cfg_alias string prevent this appearing in the config file.
                    cfg_alias="",
                    cli_alias="config",
                    env_alias="PYESOREX_CONFIG",
                    action=ConfigAction,
                ),
                # Need to make this parameter string type because it needs to be 'False', 'True' or a filename
                ParameterValue(
                    name="create_config",
                    context="pyesorex",
                    description="Creates a configuration file for PyEsoRex. If set a config file 'pyesorex.rc' is "
                    "created in the '.pyesorex' directory in $HOME of the user. If a filename is specified, a config "
                    "file will be created accordingly. If a recipe is specified in the command line then the "
                    "configuration file will be created for the recipe instead (called 'recipename.rc'). Note that an "
                    "existing file will be overwritten but a backup will be copied to 'filename.rc.bak' in the same "
                    "directory. If the filename ends with extension .json then a machine-readable JSON format will be "
                    "used. PyEsoRex will exit after this option. It should always be the last option on the "
                    "command line.",
                    default="False",
                    # Empty cfg_alias string prevents this appearing in the config file.
                    cfg_alias="",
                    cli_alias="create-config",
                    action=CreateConfigAction,
                ),
                ParameterValue(
                    name="display_import_errors",
                    context="pyesorex",
                    description="When True any import errors encountered while searching for Python recipes will "
                    "be displayed as warnings. When False import errors will be suppressed.",
                    default=False,
                    cfg_alias="pyesorex.display-import-errors",
                    cli_alias="display-import-errors",
                    env_alias="PYESOREX_DISPLAY_IMPORT_ERRORS",
                ),
                ParameterValue(
                    name="link_dir",
                    context="pyesorex",
                    description="The directory in which a symbolic link to each of the product files should be "
                    "written. The enable/disable switch to control whether the link is actually made is "
                    "the '--suppress-link' option.",
                    default="/tmp",
                    cfg_alias="pyesorex.log-dir",
                    cli_alias="link-dir",
                    env_alias="PYESOREX_LINK_DIR",
                    nargs=1,
                ),
                ParameterValue(
                    name="log_dir",
                    context="pyesorex",
                    description="Directory where to place the logfile.",
                    default=".",
                    cfg_alias="pyesorex.log-dir",
                    cli_alias="log-dir",
                    env_alias="PYESOREX_LOG_DIR",
                    nargs=1,
                    callback=lambda _param, new_value: self._start_logging(
                        self.parameters["log_level"].value,
                        self.parameters["log_file"].value,
                        new_value,
                    ),
                ),
                ParameterValue(
                    name="log_file",
                    context="pyesorex",
                    description="Filename of logfile.",
                    default="pyesorex.log",
                    cfg_alias="pyesorex.log-file",
                    cli_alias="log-file",
                    env_alias="PYESOREX_LOG_FILE",
                    nargs=1,
                    callback=lambda _param, new_value: self._start_logging(
                        self.parameters["log_level"].value,
                        new_value,
                        self.parameters["log_dir"].value,
                    ),
                ),
                ParameterEnum(
                    name="log_level",
                    context="pyesorex",
                    description="Controls the severity level of messages that will be printed to the logfile.",
                    default="info",
                    alternatives=("debug", "info", "warning", "error", "off"),
                    cfg_alias="pyesorex.log-level",
                    cli_alias="log-level",
                    env_alias="PYESOREX_LOG_LEVEL",
                    nargs=1,
                    callback=lambda _param, new_value: self._start_logging(
                        new_value,
                        self.parameters["log_file"].value,
                        self.parameters["log_dir"].value,
                    ),
                ),
                ParameterValue(
                    name="man_page",
                    context="pyesorex",
                    description="Display a manual page for the specified. Note that this option only applies to "
                    "recipes, and that it does nothing for pyesorex by itself. See also the '--help' option"
                    "PyEsoRex will exit after this option. It should always be the last option on the command line.",
                    default=False,
                    cli_alias="man-page",
                    # Empty cfg_alias string prevent this appearing in the config file.
                    cfg_alias="",
                    nargs=1,
                    action=ManPageAction,
                ),
                ParameterEnum(
                    name="msg_level",
                    context="pyesorex",
                    description="Controls the severity level of messages that will be printed to the terminal.",
                    default="info",
                    alternatives=("debug", "info", "warning", "error", "off"),
                    cfg_alias="pyesorex.msg-level",
                    cli_alias="msg-level",
                    env_alias="PYESOREX_MSG_LEVEL",
                    nargs=1,
                    callback=lambda _param, new_value: self._set_msg_level(new_value),
                ),
                ParameterValue(
                    name="no_datamd5",
                    context="pyesorex",
                    description="Disables the computation of the MD5 data hash for FITS product files.",
                    default=False,
                    cfg_alias="pyesorex.no-datamd5",
                    cli_alias="no-datamd5",
                    env_alias="PYESOREX_NO_DATAMD5",
                ),
                ParameterValue(
                    name="no_checksum",
                    context="pyesorex",
                    description="Disables the computation of the standard FITS product checksums.",
                    default=False,
                    cfg_alias="pyesorex.no-checksum",
                    cli_alias="no-checksum",
                    env_alias="PYESOREX_NO_CHECKSUM",
                ),
                ParameterValue(
                    name="output_dir",
                    context="pyesorex",
                    description="The directory where the product files should be finally moved to (all products "
                    "are first created in the current dir).",
                    default=".",
                    cfg_alias="pyesorex.output-dir",
                    cli_alias="output-dir",
                    env_alias="PYESOREX_OUTPUT_DIR",
                    nargs=1,
                ),
                ParameterValue(
                    name="output_mkdir",
                    context="pyesorex",
                    description="When True output and link directories will be created if they do not already "
                    "exist. When False an output or link directory that does not exist will result in an error, "
                    "as it does with EsoRex.",
                    default=False,
                    cfg_alias="pyesorex.output-mkdir",
                    cli_alias="output-mkdir",
                    env_alias="PYESOREX_OUTPUT_MKDIR",
                ),
                ParameterValue(
                    name="output_prefix",
                    context="pyesorex",
                    description="Prefix applied to any output file. For example, specifying 'pre' would translate "
                    "'filename.fits' to 'pre_0000.fits'. See also the '--suppress-prefix' option.",
                    default="out",
                    cfg_alias="pyesorex.output-prefix",
                    cli_alias="output-prefix",
                    env_alias="PYESOREX_OUTPUT_PREFIX",
                    nargs=1,
                    callback=lambda _param, new_value: self._set_output_prefix(new_value),
                ),
                ParameterValue(
                    name="params",
                    context="pyesorex",
                    description="List the input parameters and their current settings (whether from the command "
                    "line or a configuration file) for the pyesorex application. Parameters are labelled using "
                    "the parameter's CLI alias. If a recipe is also specified, then the list of its parameters "
                    "will also be generated in the same way. PyEsoRex will exit after this option. It should "
                    "always be the last option on the command line.",
                    default=False,
                    cfg_alias="",
                    cli_alias="params",
                    action=ParamsAction,
                ),
                ParameterValue(
                    name="products_sof",
                    context="pyesorex",
                    description="Output file which contains the FITS files created by the recipe. If the filename "
                    "ends with extension .json then a machine-readable JSON format will be used.",
                    default="",
                    cfg_alias="pyesorex.products-sof",
                    cli_alias="products-sof",
                    env_alias="PYESOREX_PRODUCTS_SOF",
                    nargs=1,
                ),
                ParameterValue(
                    name="products_sof_abspaths",
                    context="pyesorex",
                    description="When True file paths in products SOF files will be converted to absolute paths, "
                    "otherwise they will not be modified.",
                    default=False,
                    cfg_alias="pyesrex.products-sof-abspaths",
                    cli_alias="products-sof-abspaths",
                    env_alias="PYESOREX_PRODUCTS_SOF_ABSPATHS",
                ),
                ParameterValue(
                    name="recipes",
                    context="pyesorex",
                    description="Display a list of all available recipes (that are available in the directory tree "
                    "specified with '--recipe-dir'). PyEsoRex will exit after this option. It should always be "
                    "the last option on the command line.",
                    default=False,
                    cfg_alias="",
                    cli_alias="recipes",
                    action=RecipesAction,
                ),
                ParameterValue(
                    name="recipe_config",
                    context="pyesorex",
                    description="Configuration file for any selected recipe.",
                    default="",
                    cfg_alias="pyesorex.recipe-config",
                    cli_alias="recipe-config",
                    env_alias="PYESOREX_RECIPE_CONFIG",
                    nargs=1,
                ),
                ParameterValue(
                    name="recipe_dirs",
                    context="pyesorex",
                    description="Directory containing recipe libraries. Note that esorex will recursively search "
                    "not only the specified directory, but all sub-directories below it as well. Multiple "
                    "directory heads may be specified, by separating the starting paths with colons (:).",
                    default=ui.CRecipe.recipe_dir[0] if ui.CRecipe.recipe_dir else "",
                    cfg_alias="pyesorex.recipe-dir",
                    cli_alias="recipe-dir",
                    env_alias="PYESOREX_PLUGIN_DIR",
                    nargs=1,
                ),
                ParameterValue(
                    name="suppress_link",
                    context="pyesorex",
                    description="When True, no symbolic link is created to the output product. However, if False, "
                    "then a symbolic link is created in the directory specified with the option '--link-dir' for "
                    "each product that is created by the recipe.",
                    default=True,
                    cfg_alias="pyesorex.suppress-prefix",
                    cli_alias="suppress-link",
                    env_alias="PYESOREX_SUPPRESS_LINK",
                ),
                ParameterValue(
                    name="suppress_prefix",
                    context="pyesorex",
                    description="When True, the original name of the output product, as produced by the recipe, is "
                    'maintained. If False, then the name of the output file is changed to the "prefix_number" '
                    "format. The prefix can be altered using the '--output-prefix' option.",
                    default=True,
                    cfg_alias="pyesorex.suppress-prefix",
                    cli_alias="suppress-prefix",
                    env_alias="PYESOREX_SUPPRESS_PREFIX",
                ),
                ParameterValue(
                    name="time",
                    context="pyesorex",
                    description="Measure and show the recipe's execution time.",
                    default=True,
                    cfg_alias="pyesorex.time",
                    env_alias="PYESOREX_TIME",
                ),
            )
        )

        self.version_string = f"     ***** ESO Recipe Execution Tool, Python version {__version__} *****"
        self._recipe = None
        self.recipe_parameters = None
        self._sof_location = None
        self._sof = ui.FrameSet()

        if not config:
            # Check for a config in the default location, and load it if possible.
            self.read_config()
        # Check for any configuration overrides from environment variables.
        self.get_envars()
        # If given a config file name read it now.
        if config:
            self.read_config(config, config_format)

        if kwargs:
            for param in self.parameters:
                if param.name in kwargs:
                    param.set_from_string(kwargs[param.name])

        # Just in case it hasn't been triggered automatically yet setup messages
        self._set_msg_level(self.parameters["msg_level"].value)

        # Likewise for logging.
        self._start_logging(
            self.parameters["log_level"].value,
            self.parameters["log_file"].value,
            self.parameters["log_dir"].value,
        )

        core.Msg.info("", f"This is PyEsoRex, version {__version__}.")

    # Makes some of the parameters into properties for easy getting/setting.
    @property
    def link_dir(self):
        """The currently set link directory for recipe products.

        This property is a `pathlib.Path` object corresponding to current value of the
        `Pyesorex.properties['link_dir']` parameter. It can be set by assigning either a
        `Path` or a string to this property, using
        `Pyesorex.properties['link_dir'].set_from_string()`, the `PYESOREX_LINK_DIR`
        environment variable, the `pyesorex.link-dir` config file entry or the `--link-dir`
        command line option.

        Returns
        -------
        pathlib.Path
            Path object for the output directory.
        """
        return Path(self.parameters["link_dir"].value)

    @link_dir.setter
    def link_dir(self, new_link_dir):
        """Set the link directory for recipe products.

        Parameters
        ----------
        new_link_dir : pathlib.Path or str
            New link directory.

        Raises
        ------
        FileNotFoundError
            if new_link_dir does not exist
        """
        # The link directory will be validated just before running a recipe.
        self.parameters["link_dir"].set_from_string(new_link_dir)

    @property
    def output_dir(self):
        """The currently set output directory for recipe products.

        This property is a `pathlib.Path` object corresponding to current value of the
        `Pyesorex.properties['output_dir']` parameter. It can be set by assigning either a
        `Path` or a string to this property, using
        `Pyesorex.properties['output_dir'].set_from_string()`, the `PYESOREX_OUTPUT_DIR`
        environment variable, the `pyesorex.output-dir` config file entry or the `--output-dir`
        command line option.

        Returns
        -------
        pathlib.Path
            Path object for the output directory.
        """
        return Path(self.parameters["output_dir"].value)

    @output_dir.setter
    def output_dir(self, new_output_dir):
        """Set the output directory for recipe products.

        Parameters
        ----------
        new_output_dir : pathlib.Path or str
            New output directory.

        Raises
        ------
        FileNotFoundError
            if new_output_dir does not exist
        """
        # The output directory will be validated just before running a recipe.
        self.parameters["output_dir"].set_from_string(new_output_dir)

    @property
    def recipe(self):
        """The currently loaded recipe.

        Returns
        -------
        cpl.ui.Recipe or None
            The currently loaded recipe, or None is no recipe is loaded.
        """
        return self._recipe

    @recipe.setter
    def recipe(self, recipe_name):
        """Set the current recipe by name.

        Parameters
        ----------
        recipe_name : str
            Name of a recipe to load.

        See Also
        --------
        load_recipe : Loads a recipe.
        """
        self.load_recipe(recipe_name)

    @property
    def sof_location(self):
        """The path to the SOF file.

        Returns
        -------
        str or None
            Path to the SOF file containing the details of the input FITs files.
            A None value indicates the sof was not yet set or that it was
            alternatively assigned via an iterable containing cpl.ui.Frame objects.
        """
        return self._sof_location

    @sof_location.setter
    def sof_location(self, sof_location):
        """Set the path to the SOF file.

        Parameters
        ----------
        sof_location : str
            Path to the SOF file containing the details of the input FITs files.

        See Also
        ----------
        sof: Alternative way to set the SOF via iterable containing cpl.ui.Frame objects or Path str.
        """
        self._sof_location = os.path.abspath(os.path.expanduser(os.path.expandvars(sof_location)))
        self._sof = ui.FrameSet(self._sof_location)
        core.Msg.debug("sof_location", f"sof_location set to {self.sof_location!r}")

    @property
    def sof(self):
        """The currently loaded SOF.

        Returns
        -------
        cpl.ui.FrameSet
            A FrameSet containing the set of input files for the recipe.
        """
        return self._sof

    @sof.setter
    def sof(self, sof):
        """Set the SOF directly.

        Parameters
        ----------
        sof: iterable or str
            An iterable containing cpl.ui.Frame objects
            (e.g. a cpl.ui.FrameSet object or a tuple of Frame objects)
            representing the set of input files for the recipe.
            Alternatively, a string representing the sof_location path.

        See Also
        ----------
        sof_location: Alternative way to set the SOF via Path str.
        """
        if isinstance(sof, str):
            # use existing sof_location handler
            self.sof_location = sof
        else:
            try:
                self._sof = ui.FrameSet(sof)
            except TypeError:
                core.Msg.error(
                    "sof",
                    f"sof must be an iterable containing cpl.ui.Frame objects (e.g. cpl.ui.FrameSet object) {sof!r}",
                )
                raise
            except Exception:
                core.Msg.error("sof", f"error converting sof to cpl.ui.FrameSet object {sof!r}")
                raise
            if len(self._sof) == 0:
                core.Msg.warning("sof", "Assigned SOF is empty!")
            # indicate the sof_location is no longer relevant, as the sof was not assigned via a file
            self._sof_location = None
            core.Msg.debug("sof", f"sof assigned successfully {self._sof!r}")

    def read_config(
        self,
        config_path=None,
        format=None,
        ignore_unsupported=False,
        suppress_read_error=False,
        recipe=False,
    ):
        """Read a PyEsoRex configuration file.

        Reads a configuration file and sets the values of the parameters in self.parameters to
        the values found in the file.

        It is recommended to load any required configuration file when creating the Pyesorex object
        by passing a `config` argument to the constructor. This ensures that the desired logging /
        console message settings are applied from the start. This method can be used to load a
        configuration file after instantiation, however.

        Parameters
        ----------
        config_path : str, optional
            Path to the config file. If not given this method will attempt to load a configuration
            file from the default location, i.e. ~/.pyesorex/pyesorex.rc.
        format : str, optional
            Format of the config file, either "rc" or "json" (not yet implemented). If not given
            Pyesorex will infer the format of the config file from the file extension if it is
            present. Otherwise "rc" is assumed as format.
        ignore_unsupported : bool, default=False
            If True unrecognised keys in the config file will result in a DEBUG message only, if
            False (default) a KeyError will be raised. Setting this to True may be useful for
            loading an EsoRex config file that contains settings that PyEsoRex does not yet
            support.

        Other Parameters
        ----------------
        recipe : bool, default=False
            Set to True to load a recipe configuration file instead of a PyEsoRex configuration
            file. This is not inteded for use by user code, the recommended way to load a recipe
            configuration is to use the `read_recipe_config` method instead.
        suppress_read_error : bool, default=False
            Used by `read_recipe_config` to suppress errors when attempting to load recipe configs
            from the default location, where there may or may not be a confif file. User code
            should not need to set this. If True a failure to open the config file will result in
            a DEBUG message only, if False (default) an OSError will be raised.

        See Also
        --------
        read_recipe_config : Read a recipe configuration file.
        """
        if not config_path:
            config_path = self.parameters["config"].default
        config_path = self._normalise_path(config_path)

        if not suppress_read_error:
            # Automatically suppress read errors if trying to read a Pyesorex config from
            # the default location, where a config file may or may not exist.
            suppress_read_error = config_path == self._normalise_path(self.parameters["config"].default)

        if not format:
            format = self._guess_format(config_path)

        try:
            with Path(config_path).open() as config_file:
                core.Msg.debug("read_config", f"Reading config file {config_path!r}...")

                if format == "rc":
                    for line in config_file:
                        _line = line.strip()
                        if _line and not _line.startswith("#"):
                            name, value = _line.split("=", maxsplit=1)
                            context, _ = name.rsplit(".", maxsplit=1)
                            core.Msg.debug(
                                "read_config",
                                f"line={_line!r}, name={name!r}, context={context!r}, value={value!r}",
                            )
                            try:
                                # Custom ParameterList class allows lookup by cfg_alias
                                param = self.recipe_parameters[name] if recipe else self.parameters[name]
                                param.set_from_string(value)
                            except KeyError:
                                if ignore_unsupported:
                                    core.Msg.debug(
                                        "read_config",
                                        f"Unsupported parameter name {name!r} in {config_path!r} ignored.",
                                    )
                                else:
                                    # Failed to parse config file so set parameter value to "" to indicate this.
                                    if recipe:
                                        self.parameters["recipe_config"].set_from_string("")
                                    else:
                                        self.parameters["config"].set_from_string("")
                                    err = f"Parameter name {name!r} found in config file is invalid or not supported."
                                    core.Msg.error("read_config", err)
                                    raise KeyError(err) from None

                elif format == "json":
                    try:
                        param_dicts = json.load(config_file)
                    except json.JSONDecodeError as json_error:
                        # Failed to parse config file so set parameter value to "" to indicate this.
                        if recipe:
                            self.parameters["recipe_config"].set_from_string("")
                        else:
                            self.parameters["config"].set_from_string("")
                        core.Msg.error(
                            "read_config",
                            f"Error decoding {config_path!r}: {json_error}",
                        )
                        raise json_error
                    for param_dict in param_dicts:
                        try:
                            # Custom ParameterList class allos lookup by cfg_alias
                            if recipe:
                                param = self.recipe_parameters[param_dict["name"]]
                            else:
                                param = self.parameters[param_dict["name"]]
                            param.value = param_dict["value"]
                        except KeyError:
                            if ignore_unsupported:
                                core.Msg.debug(
                                    "read_config",
                                    f"Unsupported parameter name {name!r} in {config_path!r} ignored.",
                                )
                            else:
                                # Failed to parse config file so set parameter value to "" to indicate this.
                                if recipe:
                                    self.parameters["recipe_config"].set_from_string("")
                                else:
                                    self.parameters["config"].set_from_string("")
                                err = f"Parameter name {name!r} found in config file is invalid or not supported."
                                core.Msg.error("read_config", err)
                                raise KeyError(err) from None

                else:
                    err = f"Config file format must be 'rc' or 'json', got {format!r}."
                    core.Msg.error("read_config", err)
                    raise ValueError(err)

        except OSError as os_error:
            # Failed to load the config so set the parameter value to "" to indicate this.
            if recipe:
                self.parameters["recipe_config"].set_from_string("")
            else:
                self.parameters["config"].set_from_string("")

            if suppress_read_error:
                core.Msg.debug("read_config", f"Failed to open {config_path!r}.")
                return
            core.Msg.error("read_config", f"Failed to open {config_path!r}: {os_error}")
            raise os_error

        core.Msg.info("read_config", f"Read config file {config_path!r}.")
        # Set the parameter value to the path of the loaded config.
        if recipe:
            self.parameters["recipe_config"].set_from_string(config_path)
        else:
            self.parameters["config"].set_from_string(config_path)

    def read_recipe_config(self, config_path=None, format=None, ignore_unsupported=False):
        """Read a recipe configuration file.

        Reads a configuration file and sets the values of the parameters in self.recipe_parameters
        to the values found in the file.

        A recipe must be loaded before calling this method so that config file parser knows
        the names of the parameters to parse.

        Parameters
        ----------
        config_path : str, optional
            Path to the config file. If not given this method will attempt to load a configuration
            file from the locaation specified in the "recipe_config" parameter value, or if that
            is not set it will try the default location, i.e. ~/.pyesorex/<recipe_name>.rc
        format : str, optional
            Format of the config file, either "rc" or "json" (not yet implemented). If not given
            Pyesorex will infer the format of the config file from the file extension if it is
            present. Otherwise "rc" is assumed as format.
        ignore_unsupported : bool, default=False
            If True unrecognised keys in the config file will result in a DEBUG message only, if
            False (default) a KeyError will be raised. Setting this to True may be useful for
            loading an EsoRex config file that contains settings that PyEsoRex does not yet
            support.

        Raises
        ------
        RuntimeError
            Raised if this method is called when no recipe has been loaded, of if the recipe
            has no `parameters` attribute.

        See Also
        --------
        read_config : Read a PyEsoRex configuration file.
        """
        if not self.recipe:
            err = "No recipe loaded, cannot read recipe config."
            core.Msg.error("read_recipe_config", err)
            raise RuntimeError(err)

        if self.recipe_parameters is None:
            err = "Recipe has no parameters attribute, cannot read recipe config."
            core.Msg.error("read_recipe_config", err)
            raise RuntimeError(err)

        if config_path is None:
            # Not passed a config path, try to get one from parameters.
            config_path = self.parameters["recipe_config"].value
            if config_path:
                core.Msg.debug(
                    "read_recipe_config",
                    f"Using recipe config from parameters, {config_path!r}.",
                )

        if not config_path:
            # No config path in the parameters, so try the default based on recipe name.
            if not format:
                format = "rc"
            config_path = f"~/.pyesorex/{self.recipe.name}.{format}"
            core.Msg.debug(
                "read_recipe_config",
                f"Using default location for recipe config, {config_path!r}",
            )
            # There may or may not be a recipe config file at the default location
            # so failure to read it should not be an error.
            suppress_read_error = True
        else:
            # Config path explictly set either when calling this method or in parameters
            # so failure to read should be an error.
            suppress_read_error = False

        self.read_config(
            config_path,
            format=format,
            recipe=True,
            ignore_unsupported=ignore_unsupported,
            suppress_read_error=suppress_read_error,
        )

    def write_config(self, config_path=None, format=None, defaults=False, recipe=False):
        """Write a PyEsoRex configuration file.

        Creates a new PyEsoRex configuration file containing the current values of the parameters
        in self.parameters.

        Parameters
        ----------
        config_path : str, optional
            Path for the config file. If not given this method will attempt to create a
            configuration file at the default location, i.e. ~/.pyesorex/pyesorex.rc
        format : str, optional
            Format of the config file, either "rc" or "json" (not yet implemented). If not given
            Pyesorex will infer the format of the config file from the file extension, if present.
            By default the "rc" format will be used.
        defaults : bool, default=False
            Set to True to write the default values for the PyEsoRex parameters to the config
            file instead of the current values.

        Other Parameters
        ----------------
        recipe : bool, default=False
            Set to True to create a recipe configuration file instead of a PyEsoRex configuration
            file. This is not inteded for use by user code, the recommended way to create a recipe
            configuration is to use the `write_recipe_config` method instead.

        See Also
        --------
        write_recipe_config : Create a recipe configuration file.
        """
        if not config_path:
            config_path = self.parameters["config"].default
        config_path = self._normalise_path(config_path)
        if not format:
            format = self._guess_format(config_path)

        if Path(config_path).exists():
            # Config file already exists, back it up before replacing it.
            core.Msg.info("write_config", f"Saving existing config as {config_path}.bak")
            Path(config_path).replace(config_path + ".bak")
        else:
            # Config file doesn't already exist, make sure the directory for it does.
            Path(config_path).parent.mkdir(parents=True, exist_ok=True)

        parameters = self.parameters if not recipe else self.recipe_parameters

        if format.lower() == "rc":
            with Path(config_path).open("w") as config_file:
                core.Msg.debug("write_config", f"Writing config file {config_path!r}...")
                config_file.write(f"# File: {config_path}\n#\n")
                config_file.write("# Note: This configuration file has been automatically\n")
                config_file.write(f"#       generated by the pyesorex (v{__version__}) program.\n#\n")
                config_file.write(f"# Date: {datetime.now().isoformat(sep=' ', timespec='seconds')}\n#\n\n")
                for param in parameters:
                    if param.cfg_alias:
                        value = param.default if defaults else param.value
                        core.Msg.debug(
                            "write_config",
                            f"name={param.name!r}, cli_alias={param.cli_alias!r}, cfg_alias={param.cfg_alias!r}, "
                            f"value={value!r}",
                        )
                        config_file.write(f"# --{param.cli_alias}\n")
                        try:
                            help_text = param.cfg_help
                        except AttributeError:
                            help_text = param.help
                        config_file.write(
                            textwrap.fill(
                                help_text,
                                width=80,
                                initial_indent="# ",
                                subsequent_indent="# ",
                            )
                        )
                        config_file.write(f"\n{param.cfg_alias}={value}\n\n")
                config_file.write("#\n# End of file\n")

        elif format.lower() == "json":
            with Path(config_path).open("w") as config_file:
                core.Msg.debug("write_config", f"Writing config file {config_path!r}...")
                param_dicts = [param.as_dict(defaults) for param in parameters if param.cfg_alias]
                if recipe:
                    for param_dict in param_dicts:
                        param_dict["recipe"] = self.recipe.name
                json.dump(param_dicts, config_file, indent=2)

        else:
            pass

        # Update parameter values with path to file containing config that was just written.
        if recipe:
            self.parameters["recipe_config"].set_from_string(config_path)
        else:
            self.parameters["config"].set_from_string(config_path)

        core.Msg.info("write_config", f"Wrote config file {config_path!r}.")

    def write_recipe_config(self, config_path=None, format=None, defaults=False):
        """Write a PyEsoRex configuration file.

        Creates a new recipe configuration file containing the current values of the parameters
        in self.recipe_parameters.

        A recipe must be loaded before calling this method in order for there to be recipe
        parameters to write to the file.

        Parameters
        ----------
        config_path : str, optional
            Path for the config file. If not given this method will attempt to create a
            configuration file at the default location, i.e. ~/.pyesorex/<recipe_name>.rc
        format : str, optional
            Format of the config file, either "rc" or "json" (not yet implemented). If not given
            Pyesorex will infer the format of the config file from the file extension, if present.
            By default the "rc" format will be used.
        defaults : bool, default=False
            Set to True to write the default values for the recipe parameters to the config
            file instead of the current values.

        See Also
        --------
        write_config : Create a PyEsoRex configuration file.
        """
        if not self.recipe:
            err = "No recipe loaded, cannot write recipe config."
            core.Msg.error("write_recipe_config", err)
            raise RuntimeError(err)

        if self.recipe_parameters is None:
            err = "Recipe has no parameters attribute, cannot write recipe config."
            core.Msg.error("write_recipe_config", err)
            raise RuntimeError(err)

        if not config_path:
            if not format:
                format = "rc"
            config_path = f"~/.pyesorex/{self.recipe.name}.{format}"
            core.Msg.debug(
                "write_recipe_config",
                f"Using default location for recipe config, {config_path!r}",
            )

        self.write_config(config_path, format, recipe=True, defaults=defaults)

    def write_sof(self, frames, filepath, format=None, abspaths=False):
        """Creates a SOF file for the given FrameSet.

        Parameters
        ----------
        frames : iterable of cpl.ui.Frame
            An iterable (e.g. a cpl.ui.FrameSet) that contains the set of Frame objects that
            are to be written to the SOF file.
        filepath : pathlib.Path or str
            Path for the SOF file, either as a pathlib.Path object or a string.
        format : str, optional
            If "json" the SOF file will be written in JSON format, if "sof" the original SOF
            file format will be used. If not given the format will be inferred from filepath,
            if it ends with `.json` then JSON format will be used, otherwise SOF file format.
        abspaths : bool, optional
            If True the file paths from the Frames will be converted to absolute paths when
            writing them to the SOF file, otherwise they will be left unchanged. Default False.
        """
        # Validate filepath
        filepath = Path(filepath)
        if filepath.exists():
            core.Msg.warning(
                "pyesorex",
                f"Product SOF file already exists and will be overwritten ({filepath.resolve()})",
            )
        else:
            try:
                filepath.parent.resolve(strict=True)
            except FileNotFoundError as err:
                if self.parameters["output_mkdir"].value:
                    core.Msg.info(
                        "pyesorex",
                        f"Creating products SOF directory {filepath.parent.resolve()}.",
                    )
                    filepath.parent.mkdir(parents=True)
                else:
                    core.Msg.error(
                        "pyesorex",
                        f"Products SOF directory {filepath.parent} does not exist.",
                    )
                    raise err

        # Validate format
        if format is None:
            format = "json" if filepath.suffix == ".json" else "sof"
        elif format.lower() not in {"json", "sof"}:
            msg = f'Format must be "sof" or "json", got {format!r}.'
            core.Msg.error("pyesorex", msg)
            raise core.IllegalInputError(msg)

        if abspaths:
            frames = copy.deepcopy(frames)
            for frame in frames:
                frame.file = str(Path(frame.file).resolve())

        # Finally write SOF file
        with filepath.open("w") as sof_file:
            core.Msg.debug("write_sof", f"Writing SOF file {str(filepath)!r}...")
            if format.lower() == "json":
                frame_dicts = [{"name": frame.file, "category": frame.tag} for frame in frames]
                json.dump(frame_dicts, sof_file, indent=2)
            else:
                for frame in frames:
                    sof_file.write(f"{frame.file}     {frame.tag}\n")
        core.Msg.info("pyesorex", f"Wrote SOF file {str(filepath)!r}.")

    def get_envars(self):
        """Get parameter values from environment variables.

        Checks for environment variables with names matching the `env_alias` attributes of
        the Pyesorex parameters, and if any are found sets the value of the parameter to
        the value of the environment variable. This should be called after loading a config file
        in order to (re)apply overrides to the config file parameter values.
        """
        # If the PYESOREX_CONFIG environment variable has been used to specify a config file
        # then load that first so that any other environment variable settings will override
        # the ones from that config file.
        envars = copy.deepcopy(os.environ)
        with suppress(KeyError):
            config_file = envars.pop("PYESOREX_CONFIG")
            core.Msg.debug("get_envars", f"Loading config {config_file!r} from $PYESOREX_CONFIG...")
            self.read_config(config_file)

        for param in self.parameters:
            if param.env_alias in envars:
                param.set_from_string(envars[param.env_alias])
                core.Msg.debug(
                    "get_envars",
                    f"Set {param.name!r} to {param.value!r} from ${param.env_alias}.",
                )

    def load_recipe(self, recipe_name, recipe_config=None):
        """Loads a recipe by name.

        Parameters
        ----------
        recipe_name : str
            Named of the recipe to load.
        recipe_config : str, optional
            Path to recipe config file. If given Pyesorex will load the recipe config file after
            loading the recipe, otherwise Pyesorex will try to load a recipe config from the
            default location, ~/.pyesorex/<recipe_name>.rc.

        See Also
        --------
        get_recipes : Gets a dictionary with metadata of all available recipes.
        get_recipes_text : Gets a summary of available recipes in text form.
        """
        from pyesorex.parameter import Parameter, ParameterList

        self._recipe = None
        self.recipe_parameters = None

        # recipe_dirs can be a colon separated list of paths (environment variable style),
        # convert to a Python list.
        recipe_dirs = self.parameters["recipe_dirs"].value.split(sep=":")
        core.Msg.debug("load_recipe", f"Recipe directories: {recipe_dirs}")

        # Get all C recipes and check for matching recipe name
        c_recipes = self.get_c_recipes()
        if recipe_name in c_recipes:
            self._recipe = ui.CRecipe(recipe_name)

        # Get all python recipes and check for matching recipe name.
        python_recipes = self.get_python_recipes(self.parameters["display_import_errors"].value)
        if recipe_name in python_recipes:
            if self._recipe is not None:
                # Found named recipe in both C recipes and Python recipes
                core.Msg.warning(
                    "load_recipe",
                    f"Both C and Python recipes named {recipe_name!r} found in {recipe_dirs}, using Python recipe.",
                )
            self._recipe = python_recipes[recipe_name]()

        if self._recipe is None:
            # Didn't find named recipe in C recipes or Python recipes
            # Should raise a CPL error here, will use ValueError for now.
            self.recipe_parameters = None
            err = f"No recipe named {recipe_name!r} found in {recipe_dirs}."
            core.Msg.error("load_recipe", err)
            raise ValueError(err)

        core.Msg.info("load_recipe", f"Loaded recipe {recipe_name!r}.")

        # Loaded a recipe, store a copy of the recipe parameters for modification.
        try:
            # Convert from cpl.ui.ParameterList to pyesorex.parameter.ParameterList of
            # pyesorex.parameter.Parameter* for extra features.
            params = [Parameter.from_cplui(param) for param in self._recipe.parameters]
            self.recipe_parameters = ParameterList(params)
        except AttributeError:
            self.recipe_parameters = ParameterList()
            core.Msg.warning(
                "load_recipe",
                f"Recipe {recipe_name!r} has no parameters attribute, cannot access recipe settings or use recipe "
                "config files. Using empty parameter list.",
            )

        if self.recipe_parameters:
            # Can only load configs if recipe has .parameters.
            if not recipe_config:
                # No recipe config file specified, try to read from paramters or default location
                self.read_recipe_config()
            else:
                # Recipe config path was given, if it can't be read it should raise an OSError.
                self.read_recipe_config(config_path=recipe_config)

    def run(self):
        """Run the currently loaded recipe.

        Runs the currently loaded recipe using the current recipe parameter values from
        `recipe_parameters` and the input FITs files from `sof`.

        Returns
        -------
        cpl.ui.FrameSet
            FrameSet object containing the details of the FITs file output by the recipe.
        """
        time_recipe = self.parameters["time"].value
        if time_recipe:
            start_time = time.monotonic()

        # Before actually running the recipe check that we have a set of setting we can use.
        if not self._settings_valid():
            core.Msg.error("pyesorex", "Invalid settings, recipe execution aborted.")
            return None

        core.Msg.info("run", f"Running recipe {self.recipe.name!r}...")
        if len(self.sof) == 0:
            core.Msg.warning("pyesorex", "Running recipe with empty SOF!")
        core.Msg.set_config(domain=self.recipe.name)
        if self.recipe_parameters:
            result = self.recipe.run(self.sof, self.recipe_parameters.as_dict())
        else:
            # Handles case where recipe has no parameters.
            result = self.recipe.run(self.sof, {})
        core.Msg.set_config(domain="pyesorex")
        core.Msg.info("run", f"Recipe {self.recipe.name!r} complete.")

        # Need to sign the output Frames returned by the recipe.
        core.Msg.info("pyesorex", "Calculating product checksums")
        dfs.sign_products(
            result,
            compute_md5=not self.parameters["no_datamd5"].value,
            compute_checksum=not self.parameters["no_checksum"].value,
        )

        # Move the output FITS files, if required.
        result = self._move_products(result)

        # Create links to output FITS files, if required.
        if not self.parameters["suppress_link"].value:
            self._create_links(result)

        # Create SOF file for output FITS files, if required.
        if self.parameters["products_sof"].value:
            self.write_sof(
                result,
                self.parameters["products_sof"].value,
                abspaths=self.parameters["products_sof_abspaths"].value,
            )

        if time_recipe:
            elapsed_time = time.monotonic() - start_time
            core.Msg.info(
                "pyesorex",
                f"Recipe operation(s) took {elapsed_time:.2f} seconds to complete.",
            )
            input_frames = self.sof
            input_bytes = 0
            for frame in input_frames:
                try:
                    input_bytes += Path(frame.file).stat().st_size
                except OSError:
                    core.Msg.warning("pyesorex", f"Input file {frame.file} cannot be accessed.")
            core.Msg.info(
                "pyesorex",
                f"Total size of {len(input_frames)} raw input frames = {input_bytes / 1e6:.2f} MB",
            )
            core.Msg.info(
                "pyesorex",
                f"=> processing rate of {input_bytes / (1e6 * elapsed_time):.2f} MB/s",
            )

        return result

    def get_params_text(self, include_recipe_params=False):
        """Returns the current parameter values as a formatted multiline string.

        By default only the PyEsoRex parameters are listed, however the parameters
        of the currently loaded recipe can also be included.

        Parameters
        ----------
        include_recipe_params : bool, optional
            If True the current parameter values for the currently loaded recipe
            will be included in the output. Default False.

        Returns
        -------
        str
            Multiline string listing the current parameter values.
        """
        params_text = "Pyesorex parameters:\n\n"
        for parameter in self.parameters:
            params_text += f"{parameter.cli_alias:21s} : {parameter.value}\n"
        if include_recipe_params:
            if self.recipe_parameters:
                params_text += f"\n{self.recipe.name} recipe parameters:\n\n"
                for parameter in self.recipe_parameters:
                    params_text += f"{parameter.cli_alias:21s} : {parameter.value}\n"
            else:
                if not self.recipe:
                    msg = "Recipe parameters requested but no recipe loaded."
                else:
                    msg = "Recipe parameters requested but recipe has no parameters."
                params_text += f"{msg}\n"
                core.Msg.warning("get_params_text", msg)
        return params_text

    def get_recipes(self):
        """Get a dictionary with metadata of all available recipes.

        Calling this method triggers a recursive search for recipes in the directories
        set by the recipe_dirs parameter.

        Returns
        -------
        dict of str, RecipeMetadata
            A dictionary containing recipe names and RecipeMetadata objects, a NamedTuple with
            name, synopsis, description and version attributes.
        """
        recipe_meta = self.get_c_recipes()
        python_meta = {}
        for name, Recipe in self.get_python_recipes().items():
            recipe = Recipe()
            python_meta[name] = RecipeMetadata(recipe.name, recipe.synopsis, recipe.description, recipe.version)
        recipe_meta.update(python_meta)
        return recipe_meta

    def get_recipes_text(self):
        """Get a summary of available recipes in text form.

        Calling this method triggers a recursive search for recipes in the directories
        set by the recipe_dirs parameter.

        Returns
        -------
        str
            Multiline string containing a formatted, human readable summary of all available
            recipes. The summary lists the name and synopsis of each recipe.
        """
        recipe_meta = self.get_recipes()
        text = "List of available recipes:\n\n"
        for name, meta in recipe_meta.items():
            text += f"  {name:22s}: {meta.synopsis}\n"
        return text

    def get_c_recipes(self):
        """Get a dictionary with metadata of all available classic CPL (written in C) recipes.

        Calling this method triggers a recursive search for recipes in the directories
        set by the recipe_dirs parameter.

        Returns
        -------
        dict of str, RecipeMetadata
            A dictionary containing recipe names and RecipeMetadata objects, a NamedTuple with
            name, synopsis, description and version attributes.
        """
        recipe_dirs = [os.path.expanduser(dir) for dir in self.parameters["recipe_dirs"].value.split(sep=":")]
        ui.CRecipe.recipe_dir = recipe_dirs
        c_recipe_names = ui.CRecipe.recipes
        c_recipes = {}
        for name in c_recipe_names:
            recipe = ui.CRecipe(name)
            c_recipes[name] = RecipeMetadata(recipe.name, recipe.synopsis, recipe.description, recipe.version)
        core.Msg.debug("get_c_recipes", f"C recipes: {c_recipes}")
        return c_recipes

    def get_python_recipes(self, display_import_errors=False):
        """Get a dictionary containing all available PyCPL (written in Python) recipes.

        Calling this method triggers a recursive search for recipes in the directories
        set by the recipe_dirs parameter.

        Returns
        -------
        dict of str, cpl.ui.PyRecipe
            A dictionary containing recipe names and the corresponding cpl.ui.PyRecipe class.
        """
        recipe_dirs = [os.path.expanduser(dir) for dir in self.parameters["recipe_dirs"].value.split(sep=":")]
        recipes = {}
        for recipe_dir in recipe_dirs:
            for dirpath, _dirnames, filenames in os.walk(recipe_dir, followlinks=True):
                core.Msg.debug(
                    "get_python_recipes",
                    f"Looking for Python recipes in {dirpath!r}...",
                )
                for filename in filenames:
                    if filename.endswith(".py"):
                        spec = importlib.util.spec_from_file_location(
                            os.path.splitext(filename)[0],
                            os.path.join(dirpath, filename),
                        )
                        mod = importlib.util.module_from_spec(spec)
                        try:
                            spec.loader.exec_module(mod)
                            for name, obj in inspect.getmembers(mod, inspect.isclass):
                                # Python recipes should be a subclass of PyRecipe
                                if issubclass(obj, ui.PyRecipe) and obj != ui.PyRecipe:
                                    recipe = obj()
                                    recipes[recipe.name] = obj
                                    core.Msg.debug(
                                        "get_python_recipes",
                                        f"Found Python recipe: {name!r}",
                                    )
                        except Exception as err:
                            # Don't fail on errors while looking for recipes, optionally display.
                            if display_import_errors:
                                core.Msg.warning(
                                    "get_python_recipes",
                                    f"Exception while searching for Python recipes: {err!r}",
                                )
                            else:
                                core.Msg.debug(
                                    "get_python_recipes",
                                    f"Exception while searching for Python recipes: {err!r}",
                                )

        core.Msg.debug("get_python_recipes", f"Python recipes: {recipes}")
        return recipes

    def is_c_recipe(self, recipe_name):
        """Is there an available classic CPL (written in C) recipe with the given recipe name?

        Calling this method triggers a recursive search for recipes in the directories
        set by the recipe_dirs parameter.

        Parameters
        ----------
        recipe_name : str
            Name of the recipe to check.

        Returns
        -------
        bool:
            True if there is a C recipe with name recipe_name.
        """
        c_recipes = self.get_c_recipes()
        return recipe_name in c_recipes

    def is_python_recipe(self, recipe_name):
        """Is there an available PyCPL (written in Python) recipe with the given recipe name?

        Calling this method triggers a recursive search for recipes in the directories
        set by the recipe_dirs parameter.

        Parameters
        ----------
        recipe_name : str
            Name of the recipe to check.

        Returns
        -------
        bool:
            True if there is a Python recipe with name recipe_name.
        """
        python_recipes = self.get_python_recipes(self.parameters["display_import_errors"].value)
        return recipe_name in python_recipes

    def get_recipe_manual(self):
        """Generates the recipe manual for the currently loaded recipe, based off esorex's --man-page.

        Details recipe information including its name, description and how its used, options available for
        the recipe, copyright information and any contact details.

        Returns
        -------
        str:
            The recipe manual page for printing

        Raises
        ------
        RuntimeError
            If no recipe is yet loaded by pyesorex, or pyesorex failed to load the recipe parameters
        """
        if not self.recipe:
            err = "No recipe loaded, cannot generate recipe manual."
            core.Msg.error("get_recipe_manual", err)
            raise RuntimeError(err)

        wrap_width = shutil.get_terminal_size().columns
        recipe_instance = self.recipe
        formatted_description = recipe_instance.description.replace(
            "\n", "\n  "
        )  # Slightly indent each new line with 2 spaces
        formatted_copyright = recipe_instance.copyright.replace(
            "\n", "\n  "
        )  # Slightly indent each new line with 2 spaces

        formatted_paramlist_note = "\n  ".join(
            textwrap.wrap(
                "Note that it is also possible to create a configuration file containing these "
                "options, along with suitable default values. Please refer to the details "
                "provided by the 'pyesorex --help' command.",
                width=wrap_width - 2,
            )
        )

        return f"""NAME
    {recipe_instance.name} -- {recipe_instance.synopsis}

SYNOPSIS
  pyesorex [pyesorex-options] {recipe_instance.name} [{recipe_instance.name}-options] sof

DESCRIPTION
  {formatted_description}

OPTIONS
  The following options are provided by this recipe.

{self._parameterlist_string()}

  {formatted_paramlist_note}

AUTHOR
  {recipe_instance.author}

BUGS
  Please report any problems with this recipe to {recipe_instance.email}

RELEASE
  {recipe_instance.name} -- version {recipe_instance.version}

LICENSE
  {formatted_copyright}"""

    def get_recipe_help(self):
        """Generates the recipe help for the currently loaded recipe, based off esorex --help (recipe name)

        Details recipe information including its name, usage, brief synopsis and available options

        Returns
        -------
        str:
            The recipe help page for printing

        Raises
        ------
        RuntimeError
            If no recipe is yet loaded by pyesorex, or pyesorex failed to load the recipe parameters
        """
        if not self.recipe:
            err = "No recipe loaded, cannot generate recipe manual."
            core.Msg.error("get_recipe_manual", err)
            raise RuntimeError(err)
        recipe_instance = self.recipe
        wrap_width = shutil.get_terminal_size().columns
        formatted_footer = "\n".join(
            textwrap.wrap(
                "For help on the options of pyesorex itself, please use the command 'pyesorex --help' (that"
                " is, without specifying any recipe name). For more information about the recipe, one can "
                f"also use the command 'pyesorex --man-page {recipe_instance.name}'.",
                width=wrap_width - 2,
            )
        )  # Format footer into 80 character lines

        return f"""Recipe: {recipe_instance.name} -- {recipe_instance.synopsis}

Usage: pyesorex [pyesorex-options] {recipe_instance.name} [{recipe_instance.name}-options] sof

Options:

{self._parameterlist_string()}

{formatted_footer}"""

    def _parameterlist_string(self):
        if self.recipe_parameters is None:
            err = "Recipe has no parameters attribute, cannot parse recipe settings"
            core.Msg.error("_parameterlist_string", err)
            raise RuntimeError(err)

        if len(self.recipe_parameters) < 1:
            return "No options are provided by this recipe"
        return "\n".join([self._parameter_string(p) for p in self.recipe_parameters])

    def _parameter_string(self, parameter):
        wrap_width = shutil.get_terminal_size().columns
        prefix = f"  --{parameter.cli_alias}"
        description = parameter.help
        if description[-1] != ".":  # Append full-stop if none
            description += "."

        # Env alias component. Determine if its the a.b.c string, i.e. original parameter name
        # (e.g. giraffe.biasremoval.remove). This is if at least 2 . is in the string
        if parameter.env_alias and parameter.env_alias.count(".") <= 1:
            description += f" This option may also be set using the environment variable {parameter.env_alias}."

        # Enum component (if the parameter is an enum)
        if isinstance(parameter, ui.ParameterEnum):
            enum_component = f"<{' | '.join(str(alternative) for alternative in parameter.alternatives)}> "
        else:
            enum_component = ""  # Empty string if its not an enum

        # Add value component
        value_component = f"[{parameter.value}]"
        description = f"{description} {enum_component}{value_component}"

        # Pad each line out to 26 characters. First line of 26 characters reserved for name (prefix)
        description_formatted = ("\n" + " " * 26).join(textwrap.wrap(description, width=wrap_width - 26 - 2))
        # Pad spaces to place : at positions 24.
        return f"{prefix}{' ' * max(24 - len(prefix), 0)}: {description_formatted}"

    def _create_links(self, frames):
        core.Msg.debug("pyesorex", "Creating links to product files...")
        link_dir = Path(self.parameters["link_dir"].value)
        for frame in frames:
            fits_path = Path(frame.file)
            try:
                # Makes path absolute to avoid creating broken links like esorex does.
                fits_path = fits_path.resolve(strict=True)
            except FileNotFoundError as err:
                core.Msg.error("pyesorex", f"Product file {fits_path} missing, cannot create link")
                raise err
            link_path = link_dir / fits_path.name
            try:
                link_path.symlink_to(fits_path)
            except FileExistsError:
                core.Msg.warning(
                    "pyesorex",
                    f"A file already exists at the link path and will be overwritten ({link_path})",
                )
                link_path.unlink()
                link_path.symlink_to(fits_path)
        core.Msg.info("pyesorex", f"Created links to product files in {link_dir}")

    def _guess_format(self, config_path):
        if config_path.endswith((".json", ".json.bak")):
            return "json"
        if config_path.endswith((".rc", ".rc.bak")):
            return "rc"
        msg = (
            "Format not specified & could not automatically determine format from filename "
            f"{Path(config_path).name!r}, assuming 'rc'!"
        )
        core.Msg.warning("_guess_format", msg)
        return "rc"

    def _move_products(self, frames):
        # Build output paths, if required.
        original_paths = [Path(frame.file) for frame in frames]
        output_paths = []
        if not self.output_dir.samefile("."):
            output_paths = [self.output_dir / original.name for original in original_paths]

        if not self.parameters["suppress_prefix"].value:
            if not output_paths:
                # Populate the output paths list if not already done.
                output_paths = list(original_paths)
            # Change filenames to "prefix_NNNN.fits"
            output_paths = [
                output_path.with_name(f"{self.parameters['output_prefix'].value}_{i:04d}.fits")
                for i, output_path in enumerate(output_paths)
            ]

        if output_paths:
            # Now move the FITS files, or...
            for original_path, output_path, frame in zip(original_paths, output_paths, frames):
                # If existing files or directories are about to be clobbered issue a warning.
                if output_path.exists():
                    core.Msg.warning(
                        "pyesorex",
                        "A file already exists at the product output path and will be overwritten "
                        f"({output_path.resolve()})",
                    )
                shutil.move(original_path, output_path)
                core.Msg.info("pyesorex", f"Created product {output_path.resolve()}")
                frame.file = str(output_path)
        else:
            # ...note that the FITS files haven't been moved.
            for original_path in original_paths:
                core.Msg.info("pyesorex", f"Created product {original_path.name} (in place)")
        core.Msg.info("pyesorex", f"{len(frames)} products created")
        return frames

    def _normalise_path(self, orig_path):
        return os.path.abspath(os.path.expanduser(os.path.expandvars(orig_path)))

    def _parse_recipe_arguments(self, recipe_args):
        from pyesorex.parameter import get_string_converter

        if not self.recipe:
            err = "No recipe loaded, cannot parse recipe settings."
            core.Msg.error("_parse_recipe_arguments", err)
            raise RuntimeError(err)

        if self.recipe_parameters is None:
            err = "Recipe has no parameters attribute, cannot parse recipe settings"
            core.Msg.error("_parse_recipe_arguments", err)
            raise RuntimeError(err)

        # Need to construct a parser to turn argument list into a dictionary of name:value pairs.
        parser = argparse.ArgumentParser(argument_default=argparse.SUPPRESS)

        for parameter in self.recipe_parameters:
            # Use dataType property to find a suitable Python type to set the parameter value with.
            param_type = get_string_converter(parameter)

            # Add an argument to the parser for the recipe parameter
            parser.add_argument(f"--{parameter.cli_alias}", dest=parameter.name, type=param_type)

        # Handle a paossibly given SOF as an argument of the recipe, even if it formally is
        # and argument of pyesorex. This allows the SOF to be optional.
        parser.add_argument("sof_location", metavar="sof", type=str, nargs="?")

        # Parse the list of command line arguments.
        _recipe_args = parser.parse_args(recipe_args)

        # If a SOF is present on the command line its location is added as an attribute
        # to the Pyesorex instance, to be used when the recipe is actually called. It is
        # then removed from the list of parsed arguments because in the following step
        # it is expected that the parsed arguments are only recipe parameters.
        if hasattr(_recipe_args, "sof_location"):
            self.sof_location = _recipe_args.sof_location
            delattr(_recipe_args, "sof_location")

        recipe_settings = vars(_recipe_args)
        core.Msg.debug("_parse_recipe_arguments", f"Parsed recipe arguments: {recipe_settings}")

        # Update the recipe parameter values with the parsed arguments
        self.recipe_parameters.update(recipe_settings)

    def _set_msg_level(self, msg_level):
        core.Msg.set_config(
            level=getattr(core.Msg.SeverityLevel, msg_level.upper()),
            domain="pyesorex",
            show_domain=True,
        )

    def _check_dir(self, new_dir, dir_type):
        try:
            new_dir = new_dir.resolve(strict=True)
        except FileNotFoundError as err:
            if self.parameters["output_mkdir"].value:
                core.Msg.info("pyesorex", f"Creating {dir_type} directory {new_dir}")
                new_dir.mkdir(parents=True)
            else:
                core.Msg.error(
                    "pyesorex",
                    f"{dir_type.capitalize()} directory {new_dir} does not exist.",
                )
                raise err

        if not new_dir.is_dir():
            msg = f"{dir_type.capitalize()} directory {new_dir} is not a directory."
            core.Msg.error("pyesorex", msg)
            raise NotADirectoryError(msg)

    def _set_output_prefix(self, new_output_prefix):
        # Simple callback just to check the prefix is valid, i.e. not an empty string.
        new_output_prefix = str(new_output_prefix)
        if new_output_prefix == "":
            msg = (
                "Output prefix cannot be an empty string. "
                + "Use suppress prefix option instead to disable output file renaming."
            )
            core.Msg.error("pyesorex", msg)
            raise ValueError(msg)

    def _settings_valid(self):
        core.Msg.debug("pyesorex", "Validating settings.")
        # Check output and link directories
        self._check_dir(self.output_dir, "output")
        self._check_dir(self.link_dir, "link")

        # EsoRex doesn't allow output file renaming with the default output prefix, it needs to
        # have been explicitly set. PyEsoRex should probably do the same.
        if not self.parameters["suppress_prefix"].value and not self.parameters["output_prefix"].presence:
            core.Msg.warning(
                "pyesorex",
                "Output file prefixes have been enabled by setting the suppress prefix option to False.",
            )
            core.Msg.error(
                "pyesorex",
                "However, no valid prefix has been given by setting the output prefix option.",
            )
            return False

        return True

    def _start_logging(self, log_level, log_file, log_dir):
        log_path = os.path.join(log_dir, log_file)
        core.Msg.start_file(verbosity=getattr(core.Msg.SeverityLevel, log_level.upper()), name=log_path)
