import copy
from pathlib import Path
import subprocess
import sys

import pytest

from pyesorex.__main__ import pyesorex_cli

PATH_HERE = Path(__file__).parent
PATH_RECIPES = PATH_HERE / "recipes"
PATH_PYESOREX = PATH_HERE.parent / "src" / "p"


@pytest.mark.parametrize("recipe_name", [
    "helloworld_no_parameters",
    "helloworld_zero_parameters",
    "helloworld_none_parameters",
    "helloworld_one_parameter",
])
def test_manpage_pyesorex_cli(capsys, recipe_name):
    """Test the --man-page by calling pyesorex_cli directly.

    However, this seems to never trigger the OSError that is detected
    through test_manpage_subprocess, so that test is still necessary.
    """
    sys_argv_org = copy.copy(sys.argv)
    sys.argv = [
        "pyesorex",
        "--recipe-dir",
        str(PATH_RECIPES),
        "--man-page",
        recipe_name,
    ]
    with pytest.raises(SystemExit) as excinfo:
        pyesorex_cli()
        assert excinfo.value.code == 0
        _, err = capsys.readouterr()
        assert b"OSError" not in err
    sys.argv = sys_argv_org


@pytest.mark.xfail(reason="pyesorex might not be installed")
@pytest.mark.parametrize("recipe_name", [
    "helloworld_no_parameters",
    "helloworld_zero_parameters",
    "helloworld_none_parameters",
    "helloworld_one_parameter",
])
def test_manpage_subprocess(recipe_name):
    """Test to see whether pyesorex --man-page works in a subprocess.

    Using subprocess might trigger this error:

    OSError: [Errno 25] Inappropriate ioctl for device

    This test checks whether such a problem occurs. However, it requires
    pyesorex to be already installed, e.g. through "pip install -e .".
    """
    proc = subprocess.run(
        [
            "pyesorex",
            "--recipe-dir",
            str(PATH_RECIPES),
            "--man-page",
            "helloworld_zero_parameters"
        ],
        capture_output=True,
    )
    assert proc.returncode == 0
    assert b"OSError" not in proc.stderr
