import os
import astropy.io.fits as fits
from adari_core.data_libs.master_rawdisp import MasterRawdispReport
from .sphere_utils import SphereSetupInfo, SphereReportMixin

class SphereRawdispReport(SphereReportMixin, MasterRawdispReport):
    def __init__(self):
        super().__init__("sphere_rawdisp")
        self.extensions = []
        self.tasks = {
            "IFS_CAL_BACKGROUND_RAW": "ifs_cal_background",
            "IFS_DARK_RAW": "dark",
            "IFS_DETECTOR_FLAT_FIELD_RAW": "ifs_flat",
            "IFS_DISTORTION_MAP_RAW": "ifs_distortion_map",
            "IFS_FLAT_FIELD_RAW": "ifs_flat",
            "IFS_SCIENCE_DR_RAW": "ifs_standard",
            "IFS_SPECPOS_RAW": "ifs_specpos",
            "IFS_WAVECALIB_RAW": "ifs_wave_cal",
            "IRD_DARK_RAW": "dark",
            "IRD_DISTORTION_MAP_RAW": "ird_distortion_map",
            "IRD_FLAT_FIELD_RAW": "ird_flat",
            "IRD_INS_BG_RAW": "ird_ins_bg",
            "IRD_SCIENCE_IMAGING_RAW": "ird_science_imaging",
            "IRD_WAVECALIB_RAW": "ird_wave_cal",
            "ZPL_BIAS_IMAGING_RAW": "zpl_bias",
            "ZPL_BIAS_RAW": "zpl_bias",
            "ZPL_DARK_IMAGING_RAW": "zpl_dark",
            "ZPL_DARK_RAW": "zpl_dark",
            "ZPL_INT_FLAT_FIELD_IMAGING_RAW": "zpl_int_flat",
            "ZPL_INT_FLAT_FIELD_RAW": "zpl_int_flat",
            "ZPL_MODEM_EFF_RAW": "zpl_modem_efficiency",
            "ZPL_SCIENCE_IMAGING_RAW": "zpl_standard",
            "ZPL_SCIENCE_P1_RAW": "zpl_standard",
            "ZPL_SCIENCE_P2_RAW": "zpl_standard",
        }
        self.setup_info = SphereSetupInfo
        self.data_readers["filename"] = SphereRawdispReport.data_reader

    def parse_sof(self):
        # we building multiple report sets, so we append multiple reports to file_lists
        # get a list of tags
        ext = "PRIMARY"
        tags = list(self.tasks.keys())
        added = {}
        file_lists = []
        for filename, catg in self.inputs:
            if catg in tags:
                if filename is not None and catg not in added:
                    file_lists.append({"filename": filename})
                    added[catg] = self.tasks[catg]
                    self.sof_tag.append(catg)
                    if "ZPL" in catg:
                        self.extensions.append(1)
                        file_lists.append({"filename": filename})
                        self.extensions.append(2)
                        self.sof_tag.append(catg)
                    else:
                        self.extensions.append(ext)
        return file_lists

    def data_reader(filename):
        hdu = fits.open(filename, mode="readonly")
        for item in hdu.info(output=False):
            if len(item[5]) == 3:
                ext = item[1]
                hdu[ext].data = hdu[ext].data[0]
        return hdu

    def generate_panels(self, **kwargs):
        panels = {}
        new_panels = super().generate_panels(ext=self.extensions, **kwargs)
        for i, (panel, panel_descr) in enumerate(new_panels.items()):
            # depending on task name
            try:
                task_name = panel_descr["task_name"]
            except KeyError:
                raise RuntimeError(
                    "A report has been created by "
                    "MasterRawdispReport that did "
                    "not come back with a task name "
                    "attached!"
                )
            panel_descr["report_name"] = "sphere_rawdisp_{}_{}_{}_{}".format(
                task_name,
                self.extensions[i],
                self.sof_tag[i].lower(),
                os.path.basename(panel_descr["filename"]),
            )
            panel_descr["report_description"] = (
                f"SPHERE rawdisp panel - "
                f"{panel_descr['task_name']}, "
                f"{panel_descr['tag']}, "
                f"{os.path.basename(panel_descr['filename'])}, "
                f"{panel_descr['ext']}"
            )
        panels = {**panels, **new_panels}

        return panels


rep = SphereRawdispReport()
