/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <cpl.h>
#include <math.h>
#include "sph_common_keywords.h"
#include "sph_keyword_manager.h"
#include "sph_ifs_master_detector_flat.h"
#include "sph_master_frame.h"
#include "sph_error.h"
#include "sph_ifs_keywords.h"
#include "sph_ifs_tags.h"
#include "sph_cube.h"
#include "sph_utils.h"
#include "sph_filemanager.h"
#include "sph_create_flat.h"
#include "sph_frame_validator.h"

/*-----------------------------------------------------------------------------
 Error Codes
 -----------------------------------------------------------------------------*/

extern sph_error_code SPH_IFS_MASTER_DETECTOR_FLAT_GENERAL;
extern sph_error_code SPH_IFS_MASTER_DETECTOR_FLAT_FRAMES_MISSING;

/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_ifs_master_detector_flat_run Run Algorithm of the Master DFF Recipe
 *
 * This module provides the algorithm inplementation for the master detector flat
 * recipe for IFS.
 *
 * @par Synopsis:
 * @code
 *   #include "sph_ifs_master_detector_flat.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/
/**@{*/

#define SPH_IFS_LAMP1_LAMBDA    1.020
#define SPH_IFS_LAMP2_LAMBDA    1.230
#define SPH_IFS_LAMP3_LAMBDA    1.300
#define SPH_IFS_LAMP4_LAMBDA    1.540
#define SPH_IFS_LAMP5_LAMBDA    0.0
#define SPH_IFS_LAMP6_LAMBDA    0.0

static sph_error_code sph_ifs_master_detector_flat_process_set(
        sph_ifs_master_detector_flat* self, cpl_frameset* rawframes,
        sph_master_frame* dark, cpl_mask* badpix,
        sph_master_frame** preamp_flat, double lambda);

static sph_master_frame* sph_ifs_master_detector_flat_create_lss_flat(
        sph_ifs_master_detector_flat* self, sph_master_frame* temp_master);

static cpl_frameset*
sph_ifs_master_detector_flat_extract_lamp_frames(cpl_frameset* rawframes,
        const char* tag);

static sph_master_frame*
sph_ifs_master_detector_flat_preamp_flat(sph_master_frame* flat,
        cpl_propertylist* pl);

#ifdef SPH_IFS_EXTRA
static
int sph_ifs_master_detector_flat_check_lamp(cpl_propertylist* plist,
        const char* key, double lambda, double *outlambda);

static double
sph_ifs_master_detector_flat_extrac_wavelength(cpl_frameset* frames);
#endif

static cpl_error_code
sph_ifs_master_detector_flat_add_qc(cpl_propertylist* self,
                                    const cpl_frameset* inframes,
                                    cpl_size llx,
                                    cpl_size lly,
                                    cpl_size urx,
                                    cpl_size ury);

/*----------------------------------------------------------------------------*/
/**
 @brief    Interpret the command line options and execute the data processing
 @param    frameset   the frames list
 @param    parlist    the parameters list
 @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_ifs_master_detector_flat_run(
        sph_ifs_master_detector_flat* self) {
    sph_error_code rerr = CPL_ERROR_NONE;
    cpl_image* hot_image = NULL;
    cpl_mask* badpix = NULL;
    sph_master_frame* dark = NULL;
    sph_master_frame* preamp_flat = NULL;
    double lambda = 0.0;
    int set = 0;
    sph_frame_validator_result fcheck = SPH_FRAME_VALIDATOR_ERROR;
    cpl_frameset* lamp_set_frames = NULL;
    const char* lamp_keys[6] = { SPH_IFS_KEYWORD_LAMP1_SWITCH,
            SPH_IFS_KEYWORD_LAMP2_SWITCH, SPH_IFS_KEYWORD_LAMP3_SWITCH,
            SPH_IFS_KEYWORD_LAMP4_SWITCH, SPH_IFS_KEYWORD_LAMP5_SWITCH,
            SPH_IFS_KEYWORD_LAMP6_SWITCH };

    double lamp_lambdas[6] = { SPH_IFS_LAMP1_LAMBDA, SPH_IFS_LAMP2_LAMBDA,
            SPH_IFS_LAMP3_LAMBDA, SPH_IFS_LAMP4_LAMBDA, SPH_IFS_LAMP5_LAMBDA,
            SPH_IFS_LAMP6_LAMBDA };
    /*------------------------------------------------------------------
     -  Placing master dark in cube structure for later use
     --------------------------------------------------------------------*/

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE

    fcheck = sph_frame_validator_warn_keyword_tolerance(self->inframes, 0,
            SPH_IFS_KEYWORD_BENCH_TEMP, 0.5, NULL);
    if (fcheck == SPH_FRAME_VALIDATOR_NOT_VALIDATED) {
        SPH_ERROR_RAISE_WARNING(CPL_ERROR_INCOMPATIBLE_INPUT,
                "The frames provided do not all have the same "
                "bench temperature. Check your raw frames/detector to "
                "ensure all is OK."
                "Will proceed with recipe execution anyway...");
    }
    if (fcheck == SPH_FRAME_VALIDATOR_ERROR) {
        SPH_ERROR_RAISE_WARNING(
                CPL_ERROR_INCOMPATIBLE_INPUT,
                "The frames provided do not have the "
                "%s keyword. Check your raw frames/detector to "
                "ensure all is OK. "
                "Will proceed with recipe execution anyway...", SPH_IFS_KEYWORD_BENCH_TEMP);
    }

    /*------------------------------------------------------------------
     -  Now process all sets of wavelength frames separately
     --------------------------------------------------------------------*/

    if (self->static_badpixel_frame) {
        hot_image = cpl_image_load(
                cpl_frame_get_filename(self->static_badpixel_frame),
                CPL_TYPE_INT, 0, 0);
        if (hot_image) {
            badpix = cpl_mask_threshold_image_create(hot_image, 0.5, 10000);
            cpl_image_delete(hot_image);
            hot_image = NULL;
        }
    }
    if (self->master_dark_frame) {
        dark = sph_master_frame_load_(self->master_dark_frame, 0);
        if (dark && badpix == NULL) {
            badpix = sph_master_frame_get_badpixelmask(dark);
        }
        if (!dark || rerr != CPL_ERROR_NONE) {
            sph_error_raise(SPH_ERROR_PREVIOUS_OP_FAILED, __FILE__, __func__,
                    __LINE__, SPH_ERROR_ERROR,
                    "Error when loading master dark.");
            return rerr;
        }
    }

    for (set = 0; set < 6; ++set) {
        lamp_set_frames = sph_ifs_master_detector_flat_extract_lamp_frames(
                self->rawframes, lamp_keys[set]);
        lambda = lamp_lambdas[set];
        if (lamp_set_frames && cpl_frameset_get_size(lamp_set_frames)) {
            if (sph_ifs_master_detector_flat_process_set(self, lamp_set_frames,
                    dark, badpix, &preamp_flat, lambda) != CPL_ERROR_NONE) {
                SPH_ERROR_RAISE_ERR(
                        cpl_error_get_code(),
                        "Could not create flat for frames with lamp %s.", lamp_keys[set]);
                break;
            }
        }
        cpl_frameset_delete(lamp_set_frames);
        lamp_set_frames = NULL;
    }
    cpl_frameset_delete(lamp_set_frames);
    lamp_set_frames = NULL;

    if (dark) {
        sph_master_frame_delete(dark);
        dark = NULL;
    }
    if (badpix) {
        cpl_mask_delete(badpix);
        badpix = NULL;
    }

    SPH_RAISE_CPL;

    //sph_filemanager_clean();
    return (int) cpl_error_get_code();
}
static cpl_frameset*
sph_ifs_master_detector_flat_extract_lamp_frames(cpl_frameset* rawframes,
        const char* tag) {
    const cpl_frame* aframe = NULL;
    cpl_propertylist* pl = NULL;
    cpl_frameset* result = NULL;

    aframe = cpl_frameset_get_first_const(rawframes);
    result = cpl_frameset_new();

    while (aframe) {
        pl = sph_keyword_manager_load_properties(cpl_frame_get_filename(aframe),
                0);
        if (cpl_propertylist_has(pl, tag)) {
            cpl_frameset_insert(result, cpl_frame_duplicate(aframe));
        }
        aframe = cpl_frameset_get_next_const(rawframes);
        cpl_propertylist_delete(pl);
        pl = NULL;
    }
    return result;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Process a single filter set of raw frames
 @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/
static sph_error_code sph_ifs_master_detector_flat_process_set(
        sph_ifs_master_detector_flat* self, cpl_frameset* rawframes,
        sph_master_frame* dark, cpl_mask* badpix,
        sph_master_frame** preamp_flat, double lambda) {
    sph_error_code rerr = CPL_ERROR_NONE;
    cpl_image* hot_image = NULL;
    cpl_mask* nonlin_mask = NULL;
    sph_master_frame* temp_master = NULL;
    cpl_propertylist* pl = NULL;
    cpl_propertylist* plhot = NULL;
    double rms = 0;
    cpl_image* nonlin_image = NULL;
    cpl_vector* lampflux = NULL;
    cpl_vector* lampcounts = NULL;
    cpl_vector* lampflux_stdev = NULL;
    cpl_frame* template_frame = NULL;
    cpl_propertylist* template_plist = NULL;
    char* filename = NULL;
    sph_master_frame* lss_flat = NULL;
    char* tag = NULL;
    char* short_tag = NULL;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

    template_frame = cpl_frameset_get_position(rawframes, 0);
    template_plist = sph_keyword_manager_load_properties(
            cpl_frame_get_filename(template_frame), 0);
    tag = cpl_strdup(SPH_IFS_TAG_DFF_CALIB);

    if (cpl_propertylist_has(template_plist, SPH_IFS_KEYWORD_LAMP1_SWITCH)) {
        if (tag) {
            cpl_free(tag);
            tag = NULL;
        }
        if (short_tag) {
            cpl_free(short_tag);
            short_tag = NULL;
        }
        tag = cpl_strdup(SPH_IFS_TAG_DFF_LONG1_CALIB);
        short_tag = cpl_strdup("l1");
    }
    if (cpl_propertylist_has(template_plist, SPH_IFS_KEYWORD_LAMP2_SWITCH)) {
        if (tag) {
            cpl_free(tag);
            tag = NULL;
        }
        if (short_tag) {
            cpl_free(short_tag);
            short_tag = NULL;
        }
        tag = cpl_strdup(SPH_IFS_TAG_DFF_LONG2_CALIB);
        short_tag = cpl_strdup("l2");
    }
    if (cpl_propertylist_has(template_plist, SPH_IFS_KEYWORD_LAMP3_SWITCH)) {
        if (tag) {
            cpl_free(tag);
            tag = NULL;
        }
        if (short_tag) {
            cpl_free(short_tag);
            short_tag = NULL;
        }
        tag = cpl_strdup(SPH_IFS_TAG_DFF_LONG3_CALIB);
        short_tag = cpl_strdup("l3");
    }
    if (cpl_propertylist_has(template_plist, SPH_IFS_KEYWORD_LAMP4_SWITCH)) {
        if (tag) {
            cpl_free(tag);
            tag = NULL;
        }
        if (short_tag) {
            cpl_free(short_tag);
            short_tag = NULL;
        }
        tag = cpl_strdup(SPH_IFS_TAG_DFF_LONG4_CALIB);
        short_tag = cpl_strdup("l4");
    }
    // What's LAMP5 ?!?!?

    if (cpl_propertylist_has(template_plist, SPH_IFS_KEYWORD_LAMP5_SWITCH)) {
        if (tag) {
            cpl_free(tag);
            tag = NULL;
        }
        if (short_tag) {
            cpl_free(short_tag);
            short_tag = NULL;
        }
        tag = cpl_strdup(SPH_IFS_TAG_DFF_LONGBB_CALIB);
        short_tag = cpl_strdup("l5");
    }

    if (cpl_propertylist_has(template_plist, SPH_IFS_KEYWORD_LAMP6_SWITCH)) {
        if (tag) {
            cpl_free(tag);
            tag = NULL;
        }
        if (short_tag) {
            cpl_free(short_tag);
            short_tag = NULL;
        }
        tag = cpl_strdup(SPH_IFS_TAG_DFF_LONGBB_CALIB);
        short_tag = cpl_strdup("l6");
    }

    cpl_ensure_code( template_frame, CPL_ERROR_ILLEGAL_INPUT);

    pl = cpl_propertylist_new();
    if (self->lambda > 0.0) {
        lambda = self->lambda;
    }
    cpl_propertylist_append_double(pl, SPH_COMMON_KEYWORD_CALIB_LAMBDA, lambda);
    rerr = sph_utils_get_exptime_from_raw(self->rawframes,pl);
    if ( rerr != CPL_ERROR_NONE ){
        SPH_ERR("Missing keyword EXPTIME in rawdata!");
    }
    if (self->robust_fit) {
        if (cpl_frameset_get_size(rawframes) < 5) {
            SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_INPUT,
                    "The robust fitting is enabled. "
                    "For this to work, a minimum of 5 raw frames needs "
                    "to be provided. Either provide more frames or "
                    "switch robust fitting off.");
            return CPL_ERROR_ILLEGAL_INPUT;
        }
    }

    temp_master = sph_create_flat(self->rawframes, self->robust_fit, badpix,
            dark, &hot_image, self->badpix_lowtolerance,
            self->badpix_uptolerance, self->badpix_chisqtolerance,
            &lampflux,
            &lampcounts,
            0,
            &lampflux_stdev);

    if (temp_master) {
        if (self->preamp_flat) {
            *preamp_flat = sph_master_frame_load_(self->preamp_flat, 0);
        } else {
            *preamp_flat = sph_ifs_master_detector_flat_preamp_flat(temp_master,
                    pl);
        }
        if (*preamp_flat) {
            sph_master_frame_divide_master_frame(temp_master, *preamp_flat);
        } else {
            SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
                    "Could not create/load the preamp flat.");
        }

        if (*preamp_flat && self->preamp_flat == NULL) {
            if (short_tag != NULL)
                filename = sph_filemanager_new_filename_from_base(
                        self->pream_outfilename, (const char*) short_tag);
            else
                filename = cpl_strdup(self->pream_outfilename);
            if (self->save_addprod){
                sph_master_frame_save_dfs(*preamp_flat, filename, self->inframes,
                    template_frame, self->inparams, SPH_IFS_TAG_PREAMP_FLAT,
                    SPH_RECIPE_NAME_IFS_MASTER_DFF, SPH_PIPELINE_NAME_IFS, pl);
            }
            cpl_free(filename);
            filename = NULL;
        }
        sph_master_frame_delete(*preamp_flat);
        *preamp_flat = NULL;

        if (self->master_lss_flat) {
            lss_flat = sph_master_frame_load_(self->master_lss_flat, 0);
        } else {
            lss_flat = sph_ifs_master_detector_flat_create_lss_flat(self,
                    temp_master);
        }
        if (lss_flat) {
            sph_master_frame_divide_master_frame(temp_master, lss_flat);
        } else {
            SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
                    "Could not load the LSS flat.");
        }
        if (lss_flat) {
            if (short_tag != NULL)
                filename = sph_filemanager_new_filename_from_base(
                        self->lss_outfilename, (const char*) short_tag);
            else
                filename = cpl_strdup(self->lss_outfilename);
            if(self->save_addprod){
                sph_master_frame_save_dfs(lss_flat, filename, self->inframes,
                    template_frame, self->inparams, SPH_IFS_TAG_LSS_FLAT,
                    SPH_RECIPE_NAME_IFS_MASTER_DFF, SPH_PIPELINE_NAME_IFS, pl);
            }

            sph_master_frame_delete(lss_flat);
            lss_flat = NULL;
            cpl_free(filename);
            filename = NULL;
        }

        cpl_propertylist_append_double(pl, SPH_COMMON_KEYWORD_FLAT_MEAN_COUNT,
                sph_master_frame_get_mean(temp_master, &rms));
        cpl_propertylist_append_double(pl, SPH_COMMON_KEYWORD_FLAT_FPN, rms);
        // Get the RMS of the flat, which is the fitting error.
        // The mean in that image will be our FLAT nonlin. factor
        nonlin_image = sph_master_frame_get_rms(temp_master);
        nonlin_mask = sph_master_frame_get_badpixelmask(temp_master);
        cpl_image_reject_from_mask(nonlin_image, nonlin_mask);
        cpl_propertylist_append_double(pl,
                SPH_COMMON_KEYWORD_FLAT_NONLIN_FACTOR,
                cpl_image_get_mean(nonlin_image));
        cpl_propertylist_append_int(pl, SPH_COMMON_KEYWORD_NUMBER_BADPIXELS,
                sph_master_frame_get_nbads(temp_master));
        if (lampflux) {
            sph_keyword_manager_vector2proplist(pl, lampflux,
                    SPH_COMMON_KEYWORD_FLAT_LAMP_FLUX);
        }
        if (lampflux_stdev) {
            sph_keyword_manager_vector2proplist(pl, lampflux_stdev,
                    SPH_COMMON_KEYWORD_FLAT_LAMP_FLUX_STDEV);
        }
        sph_ifs_master_detector_flat_add_qc(pl, self->inframes, 500, 500,
                                            1500, 1500);

        if (short_tag != NULL)
            filename = sph_filemanager_new_filename_from_base(self->outfilename,
                    (const char*) short_tag);
        else
            filename = cpl_strdup(self->outfilename);
        sph_master_frame_save_dfs(temp_master, filename, self->inframes,
                template_frame, self->inparams, (const char*) tag,
                SPH_RECIPE_NAME_IFS_MASTER_DFF, SPH_PIPELINE_NAME_IFS, pl);
        cpl_free(filename);
        filename = NULL;
        if (hot_image) {
            plhot = cpl_propertylist_new();
            if (lambda > 0.0) {
                cpl_propertylist_append_double(plhot,
                        SPH_COMMON_KEYWORD_CALIB_LAMBDA, lambda);
            }
            cpl_propertylist_append_double(plhot,
                    SPH_COMMON_KEYWORD_NUMBER_BADPIXELS,
                    cpl_image_get_flux(hot_image));
            cpl_propertylist_append_string(plhot, SPH_COMMON_KEYWORD_PRO_CATG,
                    SPH_IFS_TAG_NON_LINEAR_PIXELMAP_CALIB);

            if (short_tag != NULL)
                filename = sph_filemanager_new_filename_from_base(
                        self->badpix_filename, (const char*) short_tag);
            else
                filename = cpl_strdup(self->badpix_filename);
            if (self->save_addprod){
                cpl_dfs_save_image(self->inframes, NULL, self->inparams,
                    self->inframes, template_frame, hot_image,
                    CPL_BPP_8_UNSIGNED, SPH_RECIPE_NAME_IFS_MASTER_DFF, plhot,
                    NULL, SPH_PIPELINE_NAME_IFS, filename);
            }
            cpl_image_delete(hot_image);
            hot_image = NULL;
            cpl_propertylist_delete(plhot);
            plhot = NULL;
            cpl_free(filename);
            filename = NULL;
        }
    }
    cpl_propertylist_delete(template_plist);
    template_plist = NULL;
    cpl_image_delete(nonlin_image);
    nonlin_image = NULL;
    cpl_mask_delete(nonlin_mask);
    nonlin_mask = NULL;
    cpl_propertylist_delete(pl);
    pl = NULL;
    cpl_vector_delete(lampflux);
    lampflux = NULL;
    cpl_vector_delete(lampcounts);
    lampcounts = NULL;
    cpl_vector_delete(lampflux_stdev);
    lampflux_stdev = NULL;
    sph_master_frame_delete(temp_master);
    if (tag) {
        cpl_free(tag);
        tag = NULL;
    }
    if (short_tag) {
        cpl_free(short_tag);
        short_tag = NULL;
    }
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

/*----------------------------------------------------------------------------*/
/**
 @brief  Compute the median lamp counts from the hard-coded windows
 @param  self     The QC propertylist to modify
 @param  inframes The input frame to read from
 @param  llx      Left  x position (1 for leftmost)
 @param  lly      Lower y position (1 for lowest)
 @param  urx      Right x position
 @param  ury      Upper y position
 @return The resulting CPL error code

 */
/*----------------------------------------------------------------------------*/
static cpl_error_code
sph_ifs_master_detector_flat_add_qc(cpl_propertylist* self,
                                    const cpl_frameset* inframes,
                                    cpl_size llx,
                                    cpl_size lly,
                                    cpl_size urx,
                                    cpl_size ury)
{
    char ilabel[] = SPH_COMMON_KEYWORD_QC_LAMPCOUNT;

    if (self == NULL)
        return cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
    if (inframes == NULL)
        return cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
    if (cpl_frameset_get_size(inframes) < 2)
        return cpl_error_set(cpl_func, CPL_ERROR_DATA_NOT_FOUND);
    if (llx < 1)
        return cpl_error_set(cpl_func, CPL_ERROR_ILLEGAL_INPUT);
    if (urx < llx)
        return cpl_error_set(cpl_func, CPL_ERROR_ILLEGAL_INPUT);
    if (lly < 1)
        return cpl_error_set(cpl_func, CPL_ERROR_ILLEGAL_INPUT);
    if (ury < lly)
        return cpl_error_set(cpl_func, CPL_ERROR_ILLEGAL_INPUT);

    for (int iqc = 0; iqc < 2; iqc++) {
        cpl_errorstate prestate = cpl_errorstate_get();
        const cpl_frame* iframe = cpl_frameset_get_position_const(inframes, iqc);
        const char* ifile = cpl_frame_get_filename(iframe);
        cpl_image* iimage = cpl_image_load(ifile, CPL_TYPE_UNSPECIFIED, 0, 0);

        ilabel[SPH_COMMON_KEYWORD_QC_LAMPCOUNT_SZ] = '0' + iqc;

        if (urx <= cpl_image_get_size_x(iimage) && 
            ury <= cpl_image_get_size_y(iimage)) {
            const double imed = cpl_image_get_median_window(iimage, llx, lly,
                                                            urx, ury);
            cpl_propertylist_append_double(self, ilabel, imed);
        } else {
            cpl_msg_warning(cpl_func, "Skipping %s due to too large window: "
                            "%d <= %d, %d <= %d", ilabel,
                            (int)cpl_image_get_size_x(iimage), (int)urx,
                            (int)cpl_image_get_size_y(iimage), (int)ury);
        }

        cpl_image_delete(iimage);

        if (!cpl_errorstate_is_equal(prestate)) {
            /* May fail on invalid upper window limit */
            return cpl_error_set_where(cpl_func);
        }
    }

    return CPL_ERROR_NONE;
}

static sph_master_frame* sph_ifs_master_detector_flat_create_lss_flat(
        sph_ifs_master_detector_flat* self, sph_master_frame* temp_master) {
    int nx = 0;
    int ny = 0;
    cpl_mask* filter_mask = NULL;
    cpl_image* lss_image = NULL;
    sph_master_frame* lss_flat = NULL;

    if (self->smoothl > 0.0) {
        if (self->smooth_method == 0) {
            nx = (int) (self->smoothl * 5.0);
            ny = (int) (self->smoothl * 5.0);
            if (nx > 256 || ny > 256) {
                SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_INPUT,
                        "Smoothing too large. Will not produce smoothed flat.");
            } else {
                if (nx % 2 == 0)
                    nx = nx + 1;
                if (ny % 2 == 0)
                    ny = ny + 1;
                filter_mask = cpl_mask_new(nx, ny);
                cpl_mask_not(filter_mask);
                SPH_ERROR_RAISE_INFO(
                        SPH_ERROR_GENERAL,
                        "Smoothing flat image width median kernel width %f...", self->smoothl);
                sph_master_frame_interpolate_bpix(temp_master);
                lss_image = cpl_image_duplicate(temp_master->image);
                cpl_image_filter_mask(lss_image, temp_master->image,
                        filter_mask, CPL_FILTER_MEDIAN, CPL_BORDER_FILTER);
                SPH_RAISE_CPL;
                lss_flat = sph_master_frame_new_from_cpl_image(lss_image);
                cpl_image_delete(lss_image);
                lss_image = NULL;
                cpl_mask_delete(filter_mask);
                filter_mask = NULL;
            }
        } else if (self->smooth_method == 1) {
            lss_flat = sph_master_frame_duplicate(temp_master);
            sph_master_frame_interpolate_bpix(lss_flat);
            sph_master_frame_smooth(lss_flat, self->smoothl);
        }
    }
    return lss_flat;
}

static sph_master_frame*
sph_ifs_master_detector_flat_preamp_flat(sph_master_frame* flat,
        cpl_propertylist* pl) {
    sph_master_frame* result = NULL;
    int preamp_width = 64;
    int nx = 0;
    int ny = 0;
    double preamp_mean = 0.0;
    double preamp_rms = 0.0;
    double preamp_median = 0.0;
    double preamp_ngoods = 0.0;
    int preamp_cc = 0;
    int npreamps = 0;
    cpl_image* flatim = NULL;
    int llx = 0;
    int urx = 0;
    cpl_image* windowim = NULL;
    char keyname[256];

    cpl_ensure(flat, CPL_ERROR_NULL_INPUT, NULL);
    flatim = sph_master_frame_extract_image(flat, 1);
    cpl_ensure(flatim, CPL_ERROR_ILLEGAL_INPUT, NULL);
    nx = cpl_image_get_size_x(flat->image);
    ny = cpl_image_get_size_y(flat->image);
    if (nx % preamp_width != 0) {
        SPH_ERROR_RAISE_ERR(
                SPH_ERROR_GENERAL,
                "The image size %d is not divisible by the preamp width %d. Cant create preamp correction.", nx, preamp_width);
        return NULL;
    }
    result = sph_master_frame_new(nx, ny);
    npreamps = nx / preamp_width;
    for (preamp_cc = 0; preamp_cc < npreamps; ++preamp_cc) {
        llx = preamp_cc * preamp_width + 1;
        urx = llx + preamp_width - 1;
        preamp_mean = cpl_image_get_mean_window(flatim, llx, 1, urx, ny - 1);
        preamp_median = cpl_image_get_median_window(flatim, llx, 1, urx,
                ny - 1);
        preamp_rms = cpl_image_get_stdev_window(flatim, llx, 1, urx, ny - 1);
        preamp_ngoods = cpl_image_get_absflux_window(flat->ncombmap, llx, 1,
                urx, ny - 1);
        if (cpl_error_get_code() == CPL_ERROR_NONE) {
            windowim = cpl_image_new(preamp_width, ny, CPL_TYPE_DOUBLE);
            cpl_image_add_scalar(windowim, preamp_median);
            cpl_image_copy(result->image, windowim, llx, 1);
            cpl_image_multiply_scalar(windowim, 0.0);
            cpl_image_add_scalar(windowim, preamp_rms);
            cpl_image_copy(result->rmsmap, windowim, llx, 1);
            cpl_image_multiply_scalar(windowim, 0.0);
            cpl_image_add_scalar(windowim, preamp_ngoods);
            cpl_image_copy(result->ncombmap, windowim, llx, 1);
            cpl_image_delete(windowim);
            windowim = NULL;
            sprintf(keyname, "%s%d", SPH_IFS_KEYWORD_PREAMPCORR_MEDIAN,
                    preamp_cc + 1);
            cpl_propertylist_update_double(pl, keyname, preamp_median);
            sprintf(keyname, "%s%d", SPH_IFS_KEYWORD_PREAMPCORR_MEAN,
                    preamp_cc + 1);
            cpl_propertylist_update_double(pl, keyname, preamp_mean);
            sprintf(keyname, "%s%d", SPH_IFS_KEYWORD_PREAMPCORR_RMS,
                    preamp_cc + 1);
            cpl_propertylist_update_double(pl, keyname, preamp_rms);
            SPH_RAISE_CPL;
        } else {
            SPH_RAISE_CPL;
            cpl_image_delete(flatim);
            flatim = NULL;
            sph_master_frame_delete(result);
            result = NULL;
            return NULL;
        }
    }
    cpl_image_delete(flatim);
    flatim = NULL;
    return result;
}

#ifdef SPH_IFS_EXTRA
static
int sph_ifs_master_detector_flat_check_lamp(cpl_propertylist* plist,
        const char* key, double lambda, double *outlambda) {
    double result = 0;
    if (cpl_propertylist_has(plist, key)) {
        if (cpl_propertylist_get_bool(plist, key) == CPL_TRUE) {
            *outlambda = lambda;
            result = 1;
        }
    }
    return result;
}

static
double sph_ifs_master_detector_flat_extrac_wavelength(cpl_frameset* frames) {
    double lambda = 0.0;
    double dummylambda = 0.0;
    int ok = 0;
    cpl_frame* aframe = NULL;
    cpl_propertylist* plist = NULL;

    aframe = cpl_frameset_get_first(frames);
    while (aframe) {
        plist = sph_keyword_manager_load_properties(
                cpl_frame_get_filename(aframe), 0);
        if (plist) {
            ok |= sph_ifs_master_detector_flat_check_lamp(plist,
                    SPH_IFS_KEYWORD_LAMP1_SWITCH, SPH_IFS_LAMP1_LAMBDA,
                    &dummylambda);
            ok |= sph_ifs_master_detector_flat_check_lamp(plist,
                    SPH_IFS_KEYWORD_LAMP2_SWITCH, SPH_IFS_LAMP2_LAMBDA,
                    &dummylambda);
            ok |= sph_ifs_master_detector_flat_check_lamp(plist,
                    SPH_IFS_KEYWORD_LAMP3_SWITCH, SPH_IFS_LAMP3_LAMBDA,
                    &dummylambda);
            ok |= sph_ifs_master_detector_flat_check_lamp(plist,
                    SPH_IFS_KEYWORD_LAMP4_SWITCH, SPH_IFS_LAMP4_LAMBDA,
                    &dummylambda);
            ok |= sph_ifs_master_detector_flat_check_lamp(plist,
                    SPH_IFS_KEYWORD_LAMP5_SWITCH, SPH_IFS_LAMP5_LAMBDA,
                    &dummylambda);
            ok |= sph_ifs_master_detector_flat_check_lamp(plist,
                    SPH_IFS_KEYWORD_LAMP5_SWITCH, SPH_IFS_LAMP6_LAMBDA,
                    &dummylambda);
        }
        if (!ok) {
            SPH_ERROR_RAISE_WARNING(
                    CPL_ERROR_DATA_NOT_FOUND,
                    "I could not find any of the LAMP keywords"
                    " (e.g. %s) in the frame %s. "
                    "Not using the correct input raw frames can"
                    " seriously affect quality of the flat.", SPH_IFS_KEYWORD_LAMP1_SWITCH, cpl_frame_get_filename(aframe));
        } else {
            if (lambda > 0.0 && fabs(dummylambda - lambda) > 0.0001) {
                SPH_ERROR_RAISE_ERR(
                        CPL_ERROR_INCOMPATIBLE_INPUT,
                        "There seem to be different lamps used "
                        "in the input raw frames."
                        "Please only use raw frames as input with one"
                        "lamp setting (e.g. keyword %s).", SPH_IFS_KEYWORD_LAMP1_SWITCH);
            } else {
                if (dummylambda > 0.0) {
                    lambda = dummylambda;
                }
            }
        }
        cpl_propertylist_delete(plist);
        plist = NULL;
        aframe = cpl_frameset_get_next(frames);
    }

    return lambda;
}
#endif

/**@}*/
