/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
// Indent used: indent sph_ifs_science.c -i4 -nut -bli0 -l80 -bad -nprs -psl -npcs -di30 -o tmp.c
/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>
#include "sph_ifs_science.h"
#include "sph_filemanager.h"
#include "sph_create_super_flat.h"
#include "sph_ifs_tags.h"
#include "sph_common_keywords.h"
#include "sph_ifs_keywords.h"
#include "sph_keyword_manager.h"
#include "sph_fft.h"
#include "sph_fits.h"
#include "sph_common_science.h"
#include "sph_ifs_subtract_dark_scaled.h"
#include "sph_crosstalkco.h"
#include "sph_badpixelco.h"
#include "sph_extract_angles.h"
#include "sph_ldt.h"

#include <math.h>
#include <strings.h>
#include <string.h>

const double SPH_IFS_SCIENCE_DR_OFFACC = 0.01;
const double SPH_DITHERX_TOLERANCE = 0.1;
const double SPH_DITHERY_TOLERANCE = 0.1;

const double SPH_IFS_ASTROM_OFFSETANGLE_1 = 0.0;
const double SPH_IFS_ASTROM_OFFSETANGLE_2 = 0.0;


const char* const SPH_IFS_STAR_CENTER_TIME_UT_NAME = "TIME_UT";
const char* const SPH_IFS_STAR_CENTER_CENTX_NAME = "CENTRE_X";
const char* const SPH_IFS_STAR_CENTER_CENTY_NAME = "CENTRE_Y";
const char* const SPH_IFS_STAR_CENTER_DMS_POS_X_NAME = "DMS_POS_X";
const char* const SPH_IFS_STAR_CENTER_DMS_POS_Y_NAME = "DMS_POS_Y";

/*----------------------------------------------------------------------------*/
/**
 * @defgroup A SPHERE API Module
 * @par Synopsis:
 * @code
 * typedef _module_ {
 * } module
 * @endcode
 * @par Desciption:
 *
 * This module provides ...
 */
/*----------------------------------------------------------------------------*/
/**@{*/

static sph_pixel_description_table *
sph_ifs_science_dither_pdt(sph_pixel_description_table * pdt,
        sph_ifs_lenslet_model * lensmodel, double dx, double dy) {
    sph_pixel_description_table *result = NULL;

    result = sph_pixel_description_table_new_shift(pdt, lensmodel, dx, dy);
    return result;
}

static sph_pixel_description_table *
sph_ifs_science_find_matching_pdt(const cpl_frame * frame,
        cpl_frameset * frames_to_search, int create_if_missing) {
    sph_pixel_description_table *pdt = NULL;
    sph_pixel_description_table *pdt_new = NULL;
    cpl_propertylist *plist = NULL;
    cpl_frame *aframe = NULL;
    double dx = 0.0;
    double dy = 0.0;
    double ndx = 0.0;
    double ndy = 0.0;
    double mindist = 10000000000000.0;
    int ff = 0;
    int closest = 0;
    int found_one = 0;
    sph_ifs_lenslet_model *lensmodel = NULL;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;

    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
            "Searching/creating matching PDT...");
    plist = sph_keyword_manager_load_properties(cpl_frame_get_filename(frame),
            0);
    if (plist) {
        dx = cpl_propertylist_get_double(plist, SPH_COMMON_KEYWORD_SPH_DITHERX);
        dy = cpl_propertylist_get_double(plist, SPH_COMMON_KEYWORD_SPH_DITHERY);
        cpl_propertylist_delete(plist);
        plist = NULL;
        if ((dx == 0 || dy == 0) && cpl_error_get_code() != CPL_ERROR_NONE) {
            sph_error_raise(
                    SPH_ERROR_GENERAL,
                    __FILE__,
                    __func__,
                    __LINE__,
                    SPH_ERROR_WARNING,
                    "Could not load the DITHER keywords %s and/or %s "
                            "from science frame. Am assuming zero dither.",
                    sph_keyword_manager_get_keyword(
                            SPH_COMMON_KEYWORD_SPH_DITHERX),
                    sph_keyword_manager_get_keyword(
                            SPH_COMMON_KEYWORD_SPH_DITHERY));
            cpl_error_reset();
            dx = 0;
            dy = 0;
        }
        for (ff = 0; ff < cpl_frameset_get_size(frames_to_search); ++ff) {
            plist = sph_keyword_manager_load_properties(
                    cpl_frame_get_filename(
                            cpl_frameset_get_position(frames_to_search, ff)),
                    0);
            if (plist) {
                ndx = cpl_propertylist_get_double(plist,
                        SPH_COMMON_KEYWORD_SPH_DITHERX);
                ndy = cpl_propertylist_get_double(plist,
                        SPH_COMMON_KEYWORD_SPH_DITHERY);
                if ((ndx == 0 || ndy == 0)
                        && cpl_error_get_code() != CPL_ERROR_NONE) {
                    SPH_RAISE_CPL_RESET;
                    //Possibly should add other error message here...
                } else {
                    if (fabs(ndx - dx) < SPH_DITHERX_TOLERANCE
                            && fabs(ndy - dy) < SPH_DITHERY_TOLERANCE) {
                        pdt = sph_pixel_description_table_load(
                                cpl_frame_get_filename(
                                        cpl_frameset_get_position(
                                                frames_to_search, ff)));
                        closest = ff;
                        mindist = (ndx - dx) * (ndx - dx)
                                + (ndy - dy) * (ndy - dy);
                        found_one = 1;
                        break;
                    } else {
                        if ((ndx - dx) * (ndx - dx) + (ndy - dy) * (ndy - dy)
                                < mindist) {
                            closest = ff;
                            mindist = (ndx - dx) * (ndx - dx)
                                    + (ndy - dy) * (ndy - dy);
                        }
                    }
                }
            }
            cpl_propertylist_delete(plist);
            plist = NULL;
        }
        if (create_if_missing && found_one == 0) {
            pdt_new = sph_pixel_description_table_load(
                    cpl_frame_get_filename(
                            cpl_frameset_get_position(frames_to_search,
                                    closest)));
            plist = sph_keyword_manager_load_properties(
                    cpl_frame_get_filename(
                            cpl_frameset_get_position(frames_to_search,
                                    closest)), 0);
            lensmodel = sph_ifs_lenslet_model_new_from_propertylist(plist);
            ndx = cpl_propertylist_get_double(plist,
                    SPH_COMMON_KEYWORD_SPH_DITHERX);
            ndy = cpl_propertylist_get_double(plist,
                    SPH_COMMON_KEYWORD_SPH_DITHERY);
            if (pdt_new && lensmodel) {
                pdt = sph_ifs_science_dither_pdt(pdt_new, lensmodel, dx - ndx,
                        dy - ndy);
                aframe = sph_filemanager_create_temp_frame("temp_pdt.fits",
                        SPH_IFS_TAG_WAVE_CALIB_CALIB, CPL_FRAME_GROUP_CALIB);
                sph_pixel_description_table_save(pdt,
                        cpl_frame_get_filename(aframe), lensmodel);
                cpl_frameset_insert(frames_to_search, aframe);
                sph_pixel_description_table_delete(pdt_new);
                pdt_new = NULL;
                sph_ifs_lenslet_model_delete(lensmodel);
                lensmodel = NULL;
            }
        }
        cpl_propertylist_delete(plist);
        plist = NULL;
    }

    cpl_propertylist_delete(plist);
    plist = NULL;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;

    return pdt;
}

sph_ldt *
sph_ifs_science_reduce_no_dither_no_adi(cpl_frameset * science_rawframeset,
        sph_collapse_algorithm colalg, sph_ifs_lenslet_model * lensmodel,
        cpl_frame * dark_frame, cpl_frame * static_bpix_frame,
        cpl_frame * wavecal_frame, cpl_frame * preamp_corr_frame,
        cpl_frame * ifu_flat_frame, cpl_frame * ldff1_frame,
        cpl_frame * ldff2_frame, cpl_frame * ldff3_frame,
        cpl_frame * ldff4_frame, cpl_frame * ldffbb_frame,
        cpl_frame * sdffbb_frame) {
    return sph_ifs_science_reduce_dither_no_adi(science_rawframeset, colalg,
            lensmodel, dark_frame, static_bpix_frame, wavecal_frame,
            preamp_corr_frame, ifu_flat_frame, ldff1_frame, ldff2_frame,
            ldff3_frame, ldff4_frame, ldffbb_frame, sdffbb_frame);
}

sph_master_frame *
sph_ifs_science_reduce_and_collapse(cpl_frameset * science_rawframeset,
        cpl_frameset * inframes, cpl_parameterlist * inparams,
        sph_collapse_algorithm colalg, sph_ifs_lenslet_model * lensmodel,
        const cpl_frame * dark_frame, cpl_frame * static_bpix_frame,
        cpl_frame * wavecal_frame, cpl_frame * preamp_frame,
        cpl_frame * ifu_flat_frame, cpl_frame * lenslet_distortion_frame,
        cpl_frame * ldff1_frame, cpl_frame * ldff2_frame,
        cpl_frame * ldff3_frame, cpl_frame * ldff4_frame,
        cpl_frame * ldffbb_frame, cpl_frame * sdffbb_frame,
        const char *recipename, const char *tag) {
    cpl_frameset *uncollapsed_frameset = NULL;
    cpl_frameset *collapsed_frameset = NULL;
    sph_master_frame *result = NULL;

    uncollapsed_frameset = sph_ifs_science_reduce_cubes(science_rawframeset,
            inframes, inparams, lensmodel, dark_frame, static_bpix_frame,
            wavecal_frame, preamp_frame, ifu_flat_frame,
            lenslet_distortion_frame, ldff1_frame, ldff2_frame, ldff3_frame,
            ldff4_frame, ldffbb_frame, sdffbb_frame, recipename, tag, 0, NULL);

    collapsed_frameset = sph_ifs_science_collapse_ldts(uncollapsed_frameset,
            inframes, lenslet_distortion_frame, recipename, tag);

    result = sph_common_science_combine(collapsed_frameset, colalg, 0, 1, 2);
    cpl_frameset_delete(collapsed_frameset);
    cpl_frameset_delete(uncollapsed_frameset);

    return result;
}

sph_ldt *
sph_ifs_science_reduce_dither_no_adi(cpl_frameset * science_rawframeset,
        sph_collapse_algorithm colalg, sph_ifs_lenslet_model * lensmodel,
        cpl_frame * dark_frame, cpl_frame * static_bpix_frame,
        cpl_frame * wavecal_frame, cpl_frame * preamp_corr_frame,
        cpl_frame * ifu_flat_frame, cpl_frame * ldff1_frame,
        cpl_frame * ldff2_frame, cpl_frame * ldff3_frame,
        cpl_frame * ldff4_frame, cpl_frame * ldffbb_frame,
        cpl_frame * sdffbb_frame) {
    cpl_frameset *ldt_frameset = NULL;
    sph_ldt *result = NULL;

    ldt_frameset = sph_ifs_science_reduce_cubes(
            science_rawframeset,
            NULL, // No inframes (so no DFS save)
            NULL, // No inparams (so no DFS save)
            lensmodel, dark_frame, static_bpix_frame, wavecal_frame,
            preamp_corr_frame, ifu_flat_frame,
            NULL, // No lenslet distortion
            ldff1_frame, ldff2_frame, ldff3_frame, ldff4_frame, ldffbb_frame,
            sdffbb_frame, NULL, // No recipe (no DFS save)
            NULL, // No tag (no DFS save)
            0,   // No illumination fraction applied
			NULL ); // No badpixel or crosstalk correction
    cpl_ensure(ldt_frameset, cpl_error_get_code(), NULL);

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL
    result = sph_ifs_science_combine_ldts(ldt_frameset, colalg);

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL
    cpl_ensure(result, cpl_error_get_code(), NULL);
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL

    return result;
}

cpl_frameset *
sph_ifs_science_reduce_cubes(cpl_frameset * science_rawframeset,
        cpl_frameset * inframes, cpl_parameterlist * inparams,
        sph_ifs_lenslet_model * lensmodel, const cpl_frame * dark_frame,
        const cpl_frame * static_bpix_frame, const cpl_frame * wavecal_frame,
        const cpl_frame * preamp_corr_frame, const cpl_frame * ifu_flat_frame,
        const cpl_frame * lenslet_distortion_map, cpl_frame * ldff1_frame,
        cpl_frame * ldff2_frame, cpl_frame * ldff3_frame,
        cpl_frame * ldff4_frame, cpl_frame * ldffbb_frame,
        cpl_frame * sdffbb_frame, const char *recipe, const char *tag,
        int use_illum,
		sph_raw_image_corrector_data* corrector_data) {
    cpl_frameset              *result = NULL;
    sph_master_frame          *sdffbb = NULL;
    sph_master_frame            *dark = NULL;
    cpl_table               *ifu_flat = NULL;
    sph_master_frame     *preamp_flat = NULL;
    cpl_frame               *pdtframe = NULL;
    cpl_frameset         *pdtframeset = NULL;
    sph_ldt                      *ldt = NULL;
    sph_pixel_description_table *pdt  = NULL;
    cpl_frameset *super_flat_frameset = NULL;
    cpl_frame       *super_flat_frame = NULL;
    cpl_image                  *bpixs = NULL;
    cpl_mask                *bpixmask = NULL;
    int                            ff = 0;
    int                            ii = 0;
    cpl_frame               *outframe = NULL;
    cpl_propertylist              *pl = NULL;
    cpl_vector               *parangs = NULL;
    int                       nplanes = 0;
    cpl_image               *rawimage = NULL;
    char                    base[256];
    char                     ext[256];
    char                   *new_fname = NULL;
    char                *fname_no_dir = NULL;
    int                            pp = 0;

    sph_distortion_model *lenslet_distort = NULL;

    if (static_bpix_frame) {
        SPH_ERROR_RAISE_INFO(
                SPH_ERROR_GENERAL,
                "Using extra static badpixel map %s", cpl_frame_get_filename(static_bpix_frame));
        bpixs = cpl_image_load(cpl_frame_get_filename(static_bpix_frame),
                CPL_TYPE_INT, 0, 0);
        if (bpixs == NULL) {
            SPH_ERROR_RAISE_ERR(
                    CPL_ERROR_ILLEGAL_INPUT,
                    "Could not load static badpixel image from %s "
                    "extension %d. Please make sure the input image "
                    "exists and has a badpixel map in the main data section "
                    "(zeroth extension).", cpl_frame_get_filename(static_bpix_frame), 0);
            cpl_ensure(0, CPL_ERROR_ILLEGAL_INPUT, NULL);
        }
        bpixmask = cpl_mask_threshold_image_create(bpixs, 0.5, INT_MAX);
        cpl_image_delete(bpixs);
        bpixs = NULL;
    }
    SPH_RAISE_CPL_RESET;
    if (dark_frame) {
        const char* dark_file = cpl_frame_get_filename(dark_frame);
        const char* dark_tag  = cpl_frame_get_tag(dark_frame);
        dark = sph_master_frame_load_(dark_frame, 0);
        cpl_msg_info(cpl_func, "Will dark subtract with: %s (%s)", dark_file,
                     dark_tag);
    } else {
        cpl_msg_info(cpl_func, "No frame for dark subtraction provided");
    }
    if (preamp_corr_frame) {
        preamp_flat = sph_master_frame_load_(preamp_corr_frame, 0);
        SPH_INFO_MSG("Will be dividing preamp from raw frames.");
    }
    if (lenslet_distortion_map) {
        SPH_ERROR_RAISE_INFO(
                SPH_ERROR_GENERAL,
                "Loading lenslet distortion from %s", cpl_frame_get_filename(lenslet_distortion_map));
        lenslet_distort =
            sph_distortion_model_load(cpl_frame_get_filename(lenslet_distortion_map),
                                                      0,
                                                      SPH_COMMON_KEYWORD_DISTMAP_COEFFX,
                                                      SPH_COMMON_KEYWORD_DISTMAP_COEFFY);
        sph_ifs_lenslet_model_set_lenslet_distortion(lensmodel,
                lenslet_distort);
    }

    pdtframeset = cpl_frameset_new();

    pdtframe = cpl_frame_duplicate(wavecal_frame);
    cpl_frameset_insert(pdtframeset, pdtframe);
    SPH_INFO_MSG("Creating PDTs and super flats for all dither positions...");

    super_flat_frameset = sph_ifs_science_create_super_flats(
            science_rawframeset, ldff1_frame, ldff2_frame, ldff3_frame,
            ldff4_frame, ldffbb_frame, sdffbb_frame, pdtframeset);

    SPH_ERROR_RAISE_INFO(
            SPH_ERROR_GENERAL,
            "Now reducing %d science frames...", (int) cpl_frameset_get_size(science_rawframeset));
    if (ifu_flat_frame) {
        ifu_flat = cpl_table_load(cpl_frame_get_filename(ifu_flat_frame), 4, 0);
        if (ifu_flat == NULL) {
            SPH_ERROR_RAISE_ERR(CPL_ERROR_FILE_IO, "Could not load IFU flat.");
            goto EXIT;
        }
    }

    if (sdffbb_frame)
        sdffbb = sph_master_frame_load_(sdffbb_frame, 0);
    result = cpl_frameset_new();
    for (ff = 0; ff < cpl_frameset_get_size(science_rawframeset); ++ff) {
        cpl_errorstate okstate = cpl_errorstate_get();
        const cpl_frame*
            frame = cpl_frameset_get_position_const(science_rawframeset, ff);


        fname_no_dir = sph_filemanager_remove_dir(
                cpl_frame_get_filename(frame));
        sph_filemanager_split(cpl_frame_get_filename(frame), base, ext);
        nplanes = sph_fits_get_nplanes(cpl_frame_get_filename(frame), 0);
        pl = sph_keyword_manager_load_properties(cpl_frame_get_filename(frame),
                0);
        //
        // put this frame to the angle extraction to determine
        // angle vector for this frame
        // Offsets both 0 to start with
        //

        parangs = sph_extract_angles_from_cube(frame,
                                               SPH_IFS_ASTROM_OFFSETANGLE_1,
                                               SPH_IFS_ASTROM_OFFSETANGLE_2);
        if (parangs == NULL){
            cpl_msg_warning(cpl_func, "Could not determine parallactic angles!"
                            "Using %d zero-values", nplanes);
            cpl_errorstate_dump(okstate, CPL_FALSE, NULL);
            cpl_errorstate_set(okstate);

            parangs = cpl_vector_new(nplanes);
            for (ii=0; ii < nplanes; ++ii) {
                cpl_vector_set(parangs, ii, 0.0);
            }
        }

        for (pp = 0; pp < nplanes; ++pp) {
            rawimage = cpl_image_load(cpl_frame_get_filename(frame),
                    CPL_TYPE_DOUBLE, pp, 0);

            if ( corrector_data && corrector_data->badpixco_apply) {
            	sph_badpixelco_correct_image_in_place(corrector_data,&rawimage);
            }
            if ( corrector_data && corrector_data->crosstalkco_apply) {
            	sph_crosstalkco_correct_image_in_place(corrector_data,&rawimage);
            }

            new_fname = sph_filemanager_get_numbered_filename_new_file(
                    fname_no_dir);
            if (super_flat_frameset) {
                super_flat_frame = cpl_frameset_get_position(
                        super_flat_frameset, ff);
                if (super_flat_frame == NULL) {
                    SPH_ERROR_RAISE_WARNING(
                            SPH_ERROR_GENERAL,
                            "Could not use/obtain super flat for frame %s, so"
                            "this frame is not flat-fielded !", cpl_frame_get_filename(frame));
                    cpl_error_reset();
                }
            }
            pdt = sph_ifs_science_find_matching_pdt(frame, pdtframeset, 1);

            ldt = sph_ifs_science_create_wavelength_cube(rawimage, dark,
                    bpixmask, preamp_flat, sdffbb, super_flat_frame, pdt,
                    lensmodel, ifu_flat, pl, use_illum);

            cpl_frameset* this_frameset_only;
            this_frameset_only = sph_common_science_frameset_extract_raw_single(inframes, frame);
            if (this_frameset_only == NULL) {
                this_frameset_only = cpl_frameset_duplicate(inframes) ;
                SPH_RAISE_CPL_RESET ;
            }

            // PIPE-6081 - note inframes/this_frameset_only the full frameset, not just sci raw
            sph_ldt_save_cube(ldt, new_fname, this_frameset_only, inframes, frame, inparams, tag,
                    recipe, SPH_PIPELINE_NAME_IFS, SPH_IFS_PIXEL_SCALE, SPH_IFS_ASTROM_ROTATIONSIGN * cpl_vector_get(parangs,pp),NULL);
//            cpl_msg_warning(__func__, "Holding at the end of sph_ldt_save_cube, press ENTER to continue");
//            getchar();
            // We're safe to delete this_frameset_only, as it contains cpl_frame duplicates
            if (this_frameset_only != NULL) {
                cpl_frameset_delete(this_frameset_only);
                this_frameset_only = NULL ;
                cpl_msg_debug(__func__, "The special frameset has now been deleted.\n");
            }

            if (inframes == NULL) {
                sph_filemanager_add_tmp_file(new_fname);
            }
            outframe = cpl_frame_new();
            SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
            cpl_frame_set_filename(outframe, new_fname);
            if ( tag ) {
                SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
            	cpl_frame_set_tag(outframe, tag);
                SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
            }
            else {
            	cpl_frame_set_tag(outframe, SPH_IFS_TAG_SCIENCE_RAW);
            }
            cpl_frame_set_group(outframe, CPL_FRAME_GROUP_PRODUCT);
            SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
            cpl_frameset_insert(result, outframe);
            SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
            sph_pixel_description_table_delete(pdt);
            pdt = NULL;
            cpl_image_delete(rawimage);
            rawimage = NULL;
            sph_ldt_delete(ldt);
            ldt = NULL;
            cpl_free(new_fname);
            new_fname = NULL;
        }
        cpl_propertylist_delete(pl);
        pl = NULL;
        cpl_free(fname_no_dir);
        fname_no_dir = NULL;
    }

    EXIT: sph_master_frame_delete(sdffbb);
    sph_master_frame_delete(dark);
    sph_master_frame_delete(preamp_flat);
    cpl_mask_delete(bpixmask);
    cpl_table_delete(ifu_flat);
    cpl_frameset_delete(pdtframeset);
    cpl_frameset_delete(super_flat_frameset);
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    cpl_vector_delete(parangs);

    return result;
}

cpl_frameset *
sph_ifs_science_create_super_flats(cpl_frameset * science_rawframeset,
        cpl_frame * ldff1_frame, cpl_frame * ldff2_frame,
        cpl_frame * ldff3_frame, cpl_frame * ldff4_frame,
        cpl_frame * ldffbb_frame, cpl_frame * sdffbb_frame,
        cpl_frameset * pdtframeset) {
    cpl_frameset *super_flat_frameset = NULL;
    sph_master_frame *ldff1 = NULL;
    sph_master_frame *ldff2 = NULL;
    sph_master_frame *ldff3 = NULL;
    sph_master_frame *ldff4 = NULL;
    sph_master_frame *sdffbb = NULL;
    sph_master_frame *ldffbb = NULL;
    int ff = 0;
    cpl_frame *sciframe = NULL;
    sph_pixel_description_table *pdt = NULL;
    sph_master_frame *super_flat = NULL;
    cpl_frame *super_flat_frame = NULL;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    super_flat_frameset = cpl_frameset_new();

    if (ldff1_frame)
        ldff1 = sph_master_frame_load_(ldff1_frame, 0);
    if (ldff2_frame)
        ldff2 = sph_master_frame_load_(ldff2_frame, 0);
    if (ldff3_frame)
        ldff3 = sph_master_frame_load_(ldff3_frame, 0);
    if (ldff4_frame)
        ldff4 = sph_master_frame_load_(ldff4_frame, 0);
    if (ldffbb_frame)
        ldffbb = sph_master_frame_load_(ldffbb_frame, 0);
    if (sdffbb_frame)
        sdffbb = sph_master_frame_load_(sdffbb_frame, 0);

    for (ff = 0; ff < cpl_frameset_get_size(science_rawframeset); ++ff) {
        sciframe = cpl_frameset_get_position(science_rawframeset, ff);

        cpl_ensure(sciframe, CPL_ERROR_DATA_NOT_FOUND, NULL);

        pdt = sph_ifs_science_find_matching_pdt(sciframe, pdtframeset, 1);

        if (pdt) {
            super_flat = sph_create_super_flat(pdt, ldff1, ldff2, ldff3, ldff4);

            if (super_flat) {
                super_flat_frame = sph_filemanager_create_temp_frame(
                        "super_flat_temp.fits", SPH_IFS_TAG_FLAT_CALIB,
                        CPL_FRAME_GROUP_CALIB);
                sph_master_frame_save(super_flat,
                        cpl_frame_get_filename(super_flat_frame), NULL);
                cpl_frameset_insert(super_flat_frameset, super_flat_frame);
                sph_master_frame_delete(super_flat);
                super_flat = NULL;
            } else {
                SPH_ERROR_RAISE_WARNING(CPL_ERROR_ILLEGAL_OUTPUT,
                        "Could not create a super flat.");
                cpl_error_reset();
            }
            sph_pixel_description_table_delete(pdt);
            pdt = NULL;
        } else {
            cpl_error_reset();
            SPH_ERROR_RAISE_WARNING(
                    SPH_ERROR_GENERAL,
                    "Could not find/create the "
                    "PDT matching the input raw frame %s dithering position."
                    "Please check that the necessary keywords are present."
                    "Will continue anyway, skipping creation of super flat.", cpl_frame_get_filename(sciframe));
        }
    }
    sph_master_frame_delete(ldff1);
    sph_master_frame_delete(ldff2);
    sph_master_frame_delete(ldff3);
    sph_master_frame_delete(ldff4);
    sph_master_frame_delete(sdffbb);
    sph_master_frame_delete(ldffbb);
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    if (cpl_frameset_get_size(super_flat_frameset) == 0) {
        cpl_frameset_delete(super_flat_frameset);
        return NULL;
    }
    return super_flat_frameset;
}

sph_ldt *
sph_ifs_science_create_wavelength_cube(cpl_image * rawimage,
        sph_master_frame * dark, cpl_mask * bpixmask,
        sph_master_frame * preamp_flat, sph_master_frame * sdffbb,
        cpl_frame * super_flat_frame, sph_pixel_description_table * pdt,
        sph_ifs_lenslet_model * lensmodel, cpl_table * ifu_flat,
        cpl_propertylist * props, int use_illum) {
    sph_master_frame *rawscience = NULL;
    sph_master_frame *super_flat = NULL;
    cpl_image *ill = NULL;
    sph_master_frame *ill_master = NULL;
    double ditherx = 0.0;
    double dithery = 0.0;
    sph_ldt *ldt = NULL;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;

    rawscience = sph_master_frame_new_from_cpl_image(rawimage);
    cpl_propertylist_delete(rawscience->properties);
    rawscience->properties = cpl_propertylist_duplicate(props);
    if (dark) {
        sph_ifs_subtract_dark_scaled(rawscience, dark);
    }
    if (bpixmask) {
        sph_master_frame_set_bads_from_mask(rawscience, bpixmask);
    }
    if (preamp_flat) {
        sph_master_frame_divide_master_frame(rawscience, preamp_flat);
        SPH_INFO_MSG("Divided preamp flat.");
    }

    if (super_flat_frame) {
        super_flat = sph_master_frame_load_(super_flat_frame, 0);
        sph_master_frame_divide_master_frame(rawscience, super_flat);
        sph_master_frame_delete(super_flat);
        super_flat = NULL;
        SPH_INFO_MSG("Divided super flat.");
    } else {
        cpl_error_reset();
    }
    if (sdffbb) {
        sph_master_frame_divide_master_frame(rawscience, sdffbb);
        SPH_INFO_MSG("Divided daily BB flat.");
    }

    if (use_illum) {
        if (cpl_propertylist_has(props, SPH_COMMON_KEYWORD_SPH_DITHERX)
                && cpl_propertylist_has(props, SPH_COMMON_KEYWORD_SPH_DITHERY)) {
            SPH_INFO_MSG("Dividing by illumination pattern...");
            ditherx = cpl_propertylist_get_double(props,
                    SPH_COMMON_KEYWORD_SPH_DITHERX);
            dithery = cpl_propertylist_get_double(props,
                    SPH_COMMON_KEYWORD_SPH_DITHERY);

            sph_ifs_lenslet_model_create_pdt_images(lensmodel, NULL, NULL, &ill,
                    NULL, ditherx, dithery);
            ill_master = sph_master_frame_new_from_cpl_image(ill);

            sph_master_frame_divide_master_frame(rawscience, ill_master);
            sph_master_frame_delete(ill_master);
            ill_master = NULL;
            cpl_image_delete(ill);
            ill = NULL;
        } else {
            SPH_INFO_MSG(
                    "Could not find diterhing keywords in raw frame(s) so am not dividing illumination pattern. The result may  contain artefacts (usually a 'wavy' structure).");
        }
    } else {
        SPH_INFO_MSG(
                "Am not dividing illumination pattern. The result may contain artefacts (usually a 'wavy' structure).");
    }SPH_INFO_MSG("Creating science LDT.");
    ldt = sph_ldt_new_from_pdt_master_frame(pdt,
            sph_ifs_lenslet_model_duplicate(lensmodel), rawscience);
    if (ldt) {
        if (ifu_flat) {
            SPH_INFO_MSG("Dividing by IFU flat.");
            sph_ldt_divide_ifu_flat(ldt, ifu_flat);
        }
        //cpl_propertylist_append(ldt->properties,rawscience->properties);
    }
    sph_master_frame_delete(rawscience);
    rawscience = NULL;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;

    return ldt;
}

cpl_frameset*
sph_ifs_science_collapse_ldts(const cpl_frameset* ldtframes,
                              cpl_frameset* inframes,
                              const cpl_frame* lenslet_distortion_frame,
                              const char *recipe,
                              const char *tag) {
    sph_ldt *ldt = NULL;
    int ff = 0;
    cpl_frame *dupframe = NULL;
    cpl_frameset *result = NULL;
    sph_master_frame *collapsed = NULL;
    cpl_parameterlist *params = NULL;
    cpl_propertylist *pl = NULL;

    cpl_ensure(ldtframes, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(recipe, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(tag, CPL_ERROR_NULL_INPUT, NULL);

    result = cpl_frameset_new();
    params = cpl_parameterlist_new();
    for (ff = 0; ff < cpl_frameset_get_size(ldtframes); ++ff) {
        const cpl_frame *aframe =
            cpl_frameset_get_position_const(ldtframes, ff);
        dupframe = sph_filemanager_get_unique_duplicate_frame(aframe);
        cpl_frame_set_tag(dupframe, tag);
        ldt = sph_ldt_load(cpl_frame_get_filename(aframe));
        if (lenslet_distortion_frame) {
            pl = sph_keyword_manager_load_properties(
                    cpl_frame_get_filename(lenslet_distortion_frame), 0);
            ldt->model->lenslet_distortion =
                    sph_distortion_model_new_from_proplist(pl,
                                                              SPH_COMMON_KEYWORD_DISTMAP_COEFFX,
                                                              SPH_COMMON_KEYWORD_DISTMAP_COEFFY);
            cpl_propertylist_delete(pl);
            pl = NULL;
            pl = sph_distortion_model_get_proplist(
                    ldt->model->lenslet_distortion);
        }
        if (ldt) {
            collapsed = sph_ldt_collapse_interpolate(ldt, 0.0, 1.0);
            if (inframes) {
                sph_master_frame_save_dfs(collapsed,
                        cpl_frame_get_filename(dupframe), inframes,
                        cpl_frameset_get_first(inframes), params, tag, recipe,
                        SPH_PIPELINE_NAME_IFS, pl);
            } else {
                sph_master_frame_save(collapsed,
                        cpl_frame_get_filename(dupframe), pl);
                sph_filemanager_add_tmp_file(cpl_frame_get_filename(dupframe));
            }
            cpl_frameset_insert(result, dupframe);
            sph_ldt_delete(ldt);
            ldt = NULL;
            sph_master_frame_delete(collapsed);
            collapsed = NULL;
        }
        cpl_propertylist_delete(pl);
        pl = NULL;
    }
    cpl_parameterlist_delete(params);
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Combine LDTs
 * @param ldtframes     the set of input frames of LDTs
 * @param coll_alg      the collapse algorithm to use
 * @param coll_params   the collapse parameters
 *
 * @return new LDT or NULL on error
 *
 * Creates a new LDT that is the collapse product of all
 * the input LDTs.
 *
 *
 */
/*----------------------------------------------------------------------------*/
sph_ldt *
sph_ifs_science_combine_ldts(cpl_frameset * ldtframes,
        sph_collapse_algorithm coll_alg) {
    sph_ldt *result = NULL;
    sph_ldt *templdt = NULL;
    sph_ifs_lenslet_model *model = NULL;
    int ff = 0;
    int np = 0;
    int pp = 0;
    int nlens = 0;
    cpl_imagelist *imagelist = NULL;
    cpl_imagelist *bpixlist = NULL;
    cpl_imagelist *weightlist = NULL;
    cpl_image *bpixim = NULL;
    cpl_image *wim = NULL;
    cpl_image *vals = NULL;
    sph_master_frame *medim = NULL;
    cpl_propertylist *pl;
    int xx = 0;
    int bpix = 0;
    int nf = 0;

    cpl_ensure(ldtframes, CPL_ERROR_NULL_INPUT, NULL);
    nf = (int) cpl_frameset_get_size(ldtframes);
    cpl_ensure(nf > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(coll_alg == SPH_COLL_ALG_MEDIAN ||
    		   coll_alg == SPH_COLL_ALG_MEAN,
			   CPL_ERROR_ILLEGAL_INPUT, NULL);
    np = sph_fits_get_nplanes(
            cpl_frame_get_filename(cpl_frameset_get_first(ldtframes)), 0);
    cpl_ensure(np > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    pl = sph_keyword_manager_load_properties(
            cpl_frame_get_filename(cpl_frameset_get_first(ldtframes)), 0);
    cpl_ensure(pl, CPL_ERROR_ILLEGAL_INPUT, NULL);

    for (ff = 0; ff < nf; ++ff) {
        cpl_ensure(
                sph_fits_get_nplanes (cpl_frame_get_filename (cpl_frameset_get_position(ldtframes, ff)), 0) == np,
                CPL_ERROR_INCOMPATIBLE_INPUT, NULL);
    }

    model = sph_ifs_lenslet_model_new_from_propertylist(pl);
    result = sph_ldt_new(model);
    nlens = sph_ifs_lenslet_model_get_nlens(model);
    for (pp = 0; pp < np; ++pp) {
        imagelist = cpl_imagelist_new();
        bpixlist = cpl_imagelist_new();
        weightlist = cpl_imagelist_new();
        for (ff = 0; ff < nf; ++ff) {
            templdt = sph_ldt_load(
                    cpl_frame_get_filename(
                            cpl_frameset_get_position(ldtframes, ff)));
            vals = cpl_image_new(nlens, 1, CPL_TYPE_DOUBLE);
            bpixim = cpl_image_new(nlens, 1, CPL_TYPE_INT);
            wim = cpl_image_new(nlens, 1, CPL_TYPE_DOUBLE);
            for (xx = 0; xx < nlens; ++xx) {
                cpl_image_set(vals, xx + 1, 1, templdt->arr[xx]->values[pp]);
                cpl_image_set(bpixim, xx + 1, 1, templdt->arr[xx]->bpix[pp]);
                if (templdt->arr[xx]->rms[pp])
                    cpl_image_set(
                            wim,
                            xx + 1,
                            1,
                            1.0
                                    / (templdt->arr[xx]->rms[pp]
                                            * templdt->arr[xx]->rms[pp]));
                else
                    cpl_image_set(wim, xx + 1, 1, 0.0);
            }
            cpl_imagelist_set(imagelist, vals, ff);
            cpl_imagelist_set(bpixlist, bpixim, ff);
            cpl_imagelist_set(weightlist, wim, ff);
            sph_ldt_delete(templdt);
            templdt = NULL;
        }
        SPH_ERROR_CHECK_STATE_ONERR_GOTO_EXIT
        (sph_common_science_combine_imlists(coll_alg, imagelist, bpixlist,
                weightlist, &medim));
        for (xx = 0; xx < nlens; ++xx) {
            result->arr[xx]->values[pp] = cpl_image_get(medim->image, xx + 1, 1,
                    &bpix);
            result->arr[xx]->bpix[pp] =
                    cpl_image_get(medim->badpixelmap, xx + 1, 1, &bpix) > 0.0 ?
                            1 : 0;
            result->arr[xx]->rms[pp] = cpl_image_get(medim->rmsmap, xx + 1, 1,
                    &bpix);
            result->arr[xx]->ncomb[pp] = cpl_image_get(medim->ncombmap, xx + 1,
                    1, &bpix);
        }
        cpl_imagelist_delete(imagelist);
        imagelist = NULL;
        cpl_imagelist_delete(bpixlist);
        bpixlist = NULL;
        cpl_imagelist_delete(weightlist);
        weightlist = NULL;
        sph_master_frame_delete(medim);
        medim = NULL;
    }
    cpl_propertylist_delete(pl);
    pl = NULL;
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return result;
    EXIT: cpl_imagelist_delete(imagelist);
    imagelist = NULL;
    cpl_imagelist_delete(bpixlist);
    bpixlist = NULL;
    cpl_imagelist_delete(weightlist);
    weightlist = NULL;
    SPH_ERROR_CHECK_STATE_RETURN_NULL;
}

/**@}*/
