/* This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_error.h"
#include "sph_common_science.h"
#include "sph_common_keywords.h"
#include "sph_ifs_tags.h"
#include "sph_ifs_keywords.h"
#include "sph_fitting.h"
#include "sph_keyword_manager.h"
#include "sph_transform.h"
#include "sph_spec_deconv.h"
#include "sph_utils.h"

#include <math.h>

/*-----------------------------------------------------------------------------
 Private function prototypes
 -----------------------------------------------------------------------------*/

static sph_error_code
sph_ifs_spec_deconv_resize_image(cpl_image** im, int nx, int ny);
static cpl_imagelist*
sph_ifs_spec_deconv_fit_imcube(cpl_imagelist* imlist, cpl_imagelist* bpixlist,
        cpl_imagelist* rmslist, cpl_vector* lambdas, int order);
static sph_error_code
sph_ifs_spec_deconv_scale_imcube(cpl_imagelist* imlist, cpl_imagelist* badpixs,
        cpl_imagelist* ncombs, cpl_imagelist* rms, cpl_vector* lambdas,
        double reflambda, double centx, double centy, int inverse_flag,
        sph_transform* transform, double smoothing);
static cpl_error_code
sph_ifs_spec_deconv_save_product(cpl_frameset* inframes,
                                 const cpl_parameterlist* inparams,
                                 const char* recipe,
                                 const cpl_imagelist* imlist,
                                 const char* fname,
                                 cpl_propertylist* pl,
                                 const cpl_frame* current_frame);

/*-----------------------------------------------------------------------------
 Function code
 -----------------------------------------------------------------------------*/
cpl_error_code sph_spec_deconv(cpl_frameset* inframes,
                               const cpl_frameset* current_raw_frameset,
                               const cpl_parameterlist* inparams,
                               const cpl_parameterlist* framecomb_parameterlist,
                               const char* tag,
                               const char* recipe,
                               const char* outfilename,
                               int order,
                               short user_cent,
                               double cx,
                               double cy,
                               double reflambda,
                               double smooth_fwhm,
                               const cpl_frameset* science_frameset) {
    int ff = 0;
    cpl_frame* current_frame = NULL;
    int next = 0;
    cpl_imagelist* imlist = NULL;
    cpl_imagelist* imlist_orig = NULL;
    cpl_imagelist* bpixlist = NULL;
    cpl_imagelist* rmslist = NULL;
    cpl_imagelist* ncomblist = NULL;
    cpl_imagelist* fitimlist = NULL;
    double centx0 = 0.0;
    double centy0 = 0.0;
    double centx = 0.0;
    double centy = 0.0;
    cpl_image* dumim = NULL;
    cpl_vector* lambdas = NULL;
    cpl_propertylist* pl = NULL;
    sph_ifs_lenslet_model* model = NULL;
    char fname[256];
    sph_transform* transform = NULL;

    cpl_ensure_code(science_frameset, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(current_raw_frameset, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(framecomb_parameterlist, CPL_ERROR_NULL_INPUT);

    transform = sph_transform_new_cpl_warp();
    for (ff = 0; ff < cpl_frameset_get_size(science_frameset); ++ff) {
        sprintf(fname, "%s_%03d.fits", outfilename, ff);

        current_frame = cpl_frame_duplicate(
                cpl_frameset_get_position_const(science_frameset, ff));

        pl = sph_keyword_manager_load_properties(
                cpl_frame_get_filename(current_frame), 0);
        model = sph_ifs_lenslet_model_new_from_propertylist(pl);

        if (!model) {
            cpl_propertylist_delete(pl);
            SPH_ERROR_RAISE_ERR(
                    CPL_ERROR_ILLEGAL_INPUT,
                    "Could not load IFS lenslet model from %s. "
                    "Check that this file has the lenslet model "
                    "information stored in the header.", cpl_frame_get_filename(current_frame));
            goto EXIT;
        }

        lambdas = sph_ifs_lenslet_model_get_lambda_vector(model);
        next = cpl_frame_get_nextensions(current_frame);

        cpl_frame_set_tag(current_frame, tag);
        if (next >= 0) {
            imlist_orig = cpl_imagelist_load(
                    cpl_frame_get_filename(current_frame), CPL_TYPE_DOUBLE, 0);
            imlist = cpl_imagelist_duplicate(imlist_orig);
            dumim = cpl_imagelist_get(imlist, 0);

            if (!dumim) {
                SPH_ERROR_RAISE_ERR(
                        CPL_ERROR_ILLEGAL_INPUT,
                        "Could not load image from %s. ", cpl_frame_get_filename(current_frame));
                goto EXIT;
            }

            centx0 = (double) cpl_image_get_size_x(dumim) / 2.0 + 0.5;
            centy0 = (double) cpl_image_get_size_y(dumim) / 2.0 + 0.5;

            if (user_cent) {
                centx = cx;
                centy = cy;
            } else {
                centx = centx0;
                centy = centy0;
            }
        }
        if (next >= 1) {
            bpixlist = cpl_imagelist_load(cpl_frame_get_filename(current_frame),
                    CPL_TYPE_INT, 1);
        }
        if (next >= 2) {
            ncomblist = cpl_imagelist_load(
                    cpl_frame_get_filename(current_frame), CPL_TYPE_DOUBLE, 2);
        }
        if (next >= 3) {
            rmslist = cpl_imagelist_load(cpl_frame_get_filename(current_frame),
                    CPL_TYPE_DOUBLE, 3);
        }

        sph_ifs_spec_deconv_scale_imcube(imlist, bpixlist, ncomblist, rmslist,
                lambdas, reflambda, centx, centy, 0, transform, smooth_fwhm);

        SPH_ERROR_CHECK_STATE_ONERR_GOTO_EXIT;

#ifdef SPH_SAVE_NONDFS
        cpl_imagelist_save(imlist, "scaled_cube.fits", CPL_TYPE_DOUBLE, NULL,
                CPL_IO_CREATE);
#endif
        fitimlist = sph_ifs_spec_deconv_fit_imcube(imlist, bpixlist, rmslist,
                lambdas, order);
#ifdef SPH_SAVE_NONDFS
        cpl_imagelist_save(fitimlist, "fitted_cube.fits", CPL_TYPE_DOUBLE, NULL,
                CPL_IO_CREATE);
#endif

        if (fitimlist) {
            // This is to fix a bug when user chooses maximum of wavelength range as
            // reference (which may be slightly above max(lambdas) )

            const double lambda_max = cpl_vector_get_max(lambdas);
            const double reflam = CPL_MIN(lambda_max, reflambda);

            if (reflam < reflambda)
                cpl_msg_warning(cpl_func, "Reducing maximum wavelength range "
                                "from %g to %g", reflambda, reflam);

            cpl_imagelist_subtract(imlist, fitimlist);

#ifdef SPH_SAVE_NONDFS
            cpl_imagelist_save(imlist, "difference_cube.fits", CPL_TYPE_DOUBLE,
                    NULL, CPL_IO_CREATE);
#endif
            sph_ifs_spec_deconv_scale_imcube(imlist, bpixlist, ncomblist,
                    rmslist, lambdas, reflam, centx0, centy0, 1, transform,
                    -1.0);
            cpl_propertylist_append_string(pl, SPH_COMMON_KEYWORD_PRO_CATG,
                    tag);
            sph_ifs_spec_deconv_save_product(inframes, inparams, recipe, imlist,
                    fname, pl, current_frame);

            SPH_ERROR_CHECK_STATE_ONERR_GOTO_EXIT
        }

        cpl_imagelist_delete(fitimlist);
        fitimlist = NULL;
        cpl_imagelist_delete(imlist);
        imlist = NULL;
        cpl_imagelist_delete(imlist_orig);
        imlist_orig = NULL;
        cpl_imagelist_delete(bpixlist);
        bpixlist = NULL;
        cpl_imagelist_delete(ncomblist);
        ncomblist = NULL;
        cpl_imagelist_delete(rmslist);
        rmslist = NULL;
        cpl_propertylist_delete(pl);
        pl = NULL;
        sph_ifs_lenslet_model_delete(model);
        model = NULL;
        cpl_frame_delete(current_frame);
        current_frame = NULL;
    }
    EXIT: cpl_imagelist_delete(fitimlist);
    fitimlist = NULL;
    cpl_imagelist_delete(imlist);
    imlist = NULL;
    cpl_imagelist_delete(imlist_orig);
    imlist_orig = NULL;
    cpl_imagelist_delete(bpixlist);
    bpixlist = NULL;
    cpl_imagelist_delete(ncomblist);
    ncomblist = NULL;
    cpl_imagelist_delete(rmslist);
    rmslist = NULL;
    cpl_propertylist_delete(pl);
    pl = NULL;
    sph_ifs_lenslet_model_delete(model);
    model = NULL;
    sph_transform_delete(transform);
    transform = NULL;
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

static sph_error_code sph_ifs_spec_deconv_resize_image(cpl_image** im, int nx,
        int ny) {
    cpl_image* tmpim = NULL;

    cpl_ensure_code(im, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(*im, CPL_ERROR_NULL_INPUT);
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;
    if (cpl_image_get_size_x(*im) > nx || cpl_image_get_size_y(*im) > ny) {
        tmpim = cpl_image_extract(*im, 1, 1, nx, ny);
        cpl_image_delete(*im);
        *im = tmpim;
    } else if (cpl_image_get_size_x(*im) < nx
            || cpl_image_get_size_y(*im) < ny) {
        SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_INPUT, "Too small images.");
        return CPL_ERROR_ILLEGAL_INPUT;
    }
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}
static cpl_imagelist*
sph_ifs_spec_deconv_fit_imcube(cpl_imagelist* imlist, cpl_imagelist* bpixlist,
        cpl_imagelist* rmslist, cpl_vector* lambdas, int order) {

    cpl_image* errimage = NULL;
    cpl_imagelist* coeffs = NULL;
    int xx = 0;
    int yy = 0;
    int nx = 0;
    int ny = 0;
    int nplanes = 0;
    int pp = 0;
    cpl_vector* yvals = NULL;
    cpl_vector* xvals = NULL;
    cpl_vector* xvals2 = NULL;
    cpl_vector* errs = NULL;
    double mse = 0.0;
    int bpix = 0;
    int bpix2 = 0;
    double pd = 0.0;
    cpl_polynomial* poly = NULL;
    cpl_imagelist* result = NULL;
    double rmsval = 0.0;
    cpl_size pows[1];
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;

    nx = cpl_image_get_size_x(cpl_imagelist_get(imlist, 0));
    ny = cpl_image_get_size_y(cpl_imagelist_get(imlist, 0));
    errimage = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
    nplanes = cpl_imagelist_get_size(imlist);
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    result = cpl_imagelist_duplicate(imlist);
    SPH_ERROR_ENSURE_GOTO_EXIT(result, cpl_error_get_code());
    cpl_imagelist_multiply_scalar(result, 0.0);
    xvals = cpl_vector_duplicate(lambdas);
    cpl_vector_power(xvals, -1.0);
    // TODO: make the 6 and 3 below and in the loop user parameters!
    // (They come from the fact that a few first and last
    // planes contain bad data and should not be included
    // in the fit)
    xvals2 = cpl_vector_new(nplanes - 6);
    errs = cpl_vector_new(nplanes - 6);
    yvals = cpl_vector_new(nplanes - 6);
    SPH_ERROR_CHECK_STATE_ONERR_GOTO_EXIT;
    // Set default error when no rmslist provided:
    if (rmslist == NULL) {
        for (pp = 0; pp < nplanes - 6; ++pp) {
            cpl_vector_set(errs, pp,
                    cpl_image_get_stdev(cpl_imagelist_get(imlist, pp + 3)));
        }
    }

    coeffs = cpl_imagelist_new();
    for (pp = 0; pp <= order; ++pp) {
        cpl_imagelist_set(coeffs, cpl_image_new(nx, ny, CPL_TYPE_DOUBLE), pp);
    }
    for (yy = 0; yy < ny; ++yy) {
        for (xx = 0; xx < nx; ++xx) {
            for (pp = 0; pp < nplanes - 6; ++pp) {
                cpl_vector_set(xvals2, pp, cpl_vector_get(xvals, pp + 3));
                cpl_vector_set(
                        yvals,
                        pp,
                        cpl_image_get(cpl_imagelist_get(imlist, pp + 3), xx + 1,
                                yy + 1, &bpix));
                if (rmslist) {
                    rmsval = cpl_image_get(cpl_imagelist_get(rmslist, pp + 3),
                            xx + 1, yy + 1, &bpix2);
                    if (rmsval > 0.0) {
                        cpl_vector_set(errs, pp, rmsval);
                    } else {
                        bpix2 = 1;
                    }
                    if (bpix2) {
                        cpl_vector_set(errs, pp, SPH_MASTER_FRAME_BAD_RMS);
                    }
                } else {
                    // Assume poisson error by default
                    //cpl_vector_set(errs, pp, sqrt(fabs(
                    //        cpl_vector_get(yvals, pp))));
                }
                if (bpixlist) {
                    if (cpl_image_get(cpl_imagelist_get(bpixlist, pp + 3),
                            xx + 1, yy + 1, &bpix2) > 0) {
                        cpl_vector_set(errs, pp, SPH_MASTER_FRAME_BAD_RMS);
                    }
                }
                if (bpix) {
                    cpl_vector_set(errs, pp, SPH_MASTER_FRAME_BAD_RMS);
                }

            }
            poly = sph_fitting_fit_poly1d(xvals2, yvals, errs, 0, order, 0, 0.0,
                    &mse);
            if (poly == NULL) {
                cpl_error_reset();
                poly = sph_fitting_fit_poly1d(xvals2, yvals, errs, 0, order, 0,
                        0.0, NULL);
                mse = 0.0;
            }
            for (pp = 0; pp < nplanes; ++pp) {
                pd = cpl_polynomial_eval_1d(poly, cpl_vector_get(xvals, pp),
                        NULL);
                cpl_image_set(cpl_imagelist_get(result, pp), xx + 1, yy + 1,
                        pd);
            }
            for (pp = 0; pp <= order; ++pp) {
                pows[0] = pp;
                cpl_image_set(cpl_imagelist_get(coeffs, pp), xx + 1, yy + 1,
                        cpl_polynomial_get_coeff(poly, pows));
            }
            cpl_image_set(errimage, xx + 1, yy + 1, mse);
            cpl_polynomial_delete(poly);
            poly = NULL;
        }
    }

#ifdef SPH_SAVE_NONDFS
    cpl_imagelist_save(coeffs, "coeffs.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_CREATE);
#endif

//    coeffs = cpl_fit_imagelist_polynomial(means,imlist,0,order,CPL_FALSE,CPL_TYPE_DOUBLE,errimage);
//    ptab = sph_pixel_polyfit_table_new_from_imlist(coeffs,errimage);
//    sph_cube_calculate_polynomial_pixel_fit()
//    sph_pixel_polyfit_table_delete(ptab); ptab = NULL;
//    cpl_image_delete(errimage); errimage = NULL;
//    cpl_imagelist_delete(coeffs); coeffs = NULL;
    EXIT: cpl_image_delete(errimage);
    errimage = NULL;
    cpl_imagelist_delete(coeffs);
    coeffs = NULL;
    cpl_vector_delete(yvals);
    yvals = NULL;
    cpl_vector_delete(xvals);
    xvals = NULL;
    cpl_vector_delete(xvals2);
    xvals2 = NULL;
    cpl_vector_delete(errs);
    errs = NULL;
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Scale the LDT planes according to inverse wavelength
 * @param imcube LDT to scale as imagelist
 * @param lambda       the wavelengths
 * @param reflambda     a reference wavelength
 * @param centx         central x coordinate in FITS coords
 * @param centy         central y coordinate in FITS coords
 * @return scaled LDT
 *
 * This scales the LDT so that all planes are scaled according to their
 * wavelength by a factor of reflambda/lambda
 *
 *
 */
/*----------------------------------------------------------------------------*/
static sph_error_code sph_ifs_spec_deconv_scale_imcube(cpl_imagelist* imlist,
        cpl_imagelist* badpixs, cpl_imagelist* ncombs, cpl_imagelist* rms,
        cpl_vector* lambdas, double reflambda, double centx, double centy,
        int inverse_flag, sph_transform* transform, double smoothing) {
    int plane = 0;
    int nplanes = 0;
    double factor = 0.0;
    cpl_image* im = NULL;
    cpl_image* badpix = NULL;
    cpl_image* ncombmap = NULL;
    cpl_image* rmsmap = NULL;
    cpl_image** pbadpix = NULL;
    cpl_image** pncombmap = NULL;
    cpl_image** prmsmap = NULL;
    sph_fft_operand* fftop;
    int nx = 0;
    int ny = 0;
    sph_error_code rerr = CPL_ERROR_NONE;

    cpl_ensure_code(imlist, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(transform, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(lambdas, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(
            cpl_vector_get_size(lambdas) == cpl_imagelist_get_size(imlist),
            CPL_ERROR_INCOMPATIBLE_INPUT);
    cpl_ensure_code(reflambda > 0, CPL_ERROR_ILLEGAL_INPUT);
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

    if (cpl_vector_get_max(lambdas) - cpl_vector_get_min(lambdas) > 2.0) {
        SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_INPUT,
                "The wavelength range in the input cube is too large. "
                "Accurate results can not be obtained. Check the wavelengths.");
        return CPL_ERROR_ILLEGAL_INPUT;
    }
    if (reflambda > cpl_vector_get_max(lambdas)
            || reflambda < cpl_vector_get_min(lambdas)) {
        SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_INPUT,
                "The reference wavelength %f is not inside the range of "
                "input wavelengths.", reflambda);
        return CPL_ERROR_ILLEGAL_INPUT;
    }

    fftop = sph_fft_operand_new();

    nplanes = cpl_imagelist_get_size(imlist);
    nx = cpl_image_get_size_x(cpl_imagelist_get(imlist, 0));
    ny = cpl_image_get_size_x(cpl_imagelist_get(imlist, 0));
    for (plane = 0; plane < nplanes; ++plane) {
        factor = cpl_vector_get(lambdas, plane);
        SPH_ERROR_ENSURE_GOTO_EXIT(factor > 0.0, CPL_ERROR_DIVISION_BY_ZERO);
        factor = reflambda / factor;
        if (inverse_flag)
            factor = 1.0 / factor;
        im = rmsmap = badpix = ncombmap = NULL;
        prmsmap = pbadpix = pncombmap = NULL;
        im = cpl_image_duplicate(cpl_imagelist_get(imlist, plane));
        SPH_ERROR_ENSURE_GOTO_EXIT(im, CPL_ERROR_NULL_INPUT);
        if (badpixs) {
            badpix = cpl_image_duplicate(cpl_imagelist_get(badpixs, plane));
            pbadpix = &badpix;
        }
        if (rms) {
            rmsmap = cpl_image_duplicate(cpl_imagelist_get(rms, plane));
            prmsmap = &rmsmap;
        }
        if (ncombs) {
            ncombmap = cpl_image_duplicate(cpl_imagelist_get(ncombs, plane));
            pncombmap = &ncombmap;
        }
        rerr = sph_transform_apply_to_images(transform, &im, pbadpix, prmsmap,
                pncombmap, NULL, centx - 0.5, centy - 0.5, 0.0, factor, NULL);
        // The -0.5 comes from the fact that the user interface
        // uses FITS coordinates, but DRH uses a coord system
        // where the bottom left corner of the bottom left pixel
        // has coords 0,0.
        if (rerr) {
            SPH_ERROR_CHECK_STATE_ONERR_GOTO_EXIT;
        }
        rerr = sph_ifs_spec_deconv_resize_image(&im, nx, ny);
        if (badpix) {
            rerr |= sph_ifs_spec_deconv_resize_image(&badpix, nx, ny);
            cpl_imagelist_set(badpixs, badpix, plane);
            SPH_ERROR_CHECK_STATE_ONERR_GOTO_EXIT;
        }
        if (rmsmap) {
            rerr |= sph_ifs_spec_deconv_resize_image(&rmsmap, nx, ny);
            cpl_imagelist_set(rms, rmsmap, plane);
            SPH_ERROR_CHECK_STATE_ONERR_GOTO_EXIT;
        }
        if (ncombmap) {
            rerr |= sph_ifs_spec_deconv_resize_image(&ncombmap, nx, ny);
            cpl_imagelist_set(ncombs, ncombmap, plane);
            SPH_ERROR_CHECK_STATE_ONERR_GOTO_EXIT;
        }
        SPH_ERROR_CHECK_STATE_ONERR_GOTO_EXIT;
        if (smoothing > 0.0) {
            sph_fft_smoothe_image(im, fftop, smoothing);
        }
        cpl_imagelist_set(imlist, im, plane);
        SPH_ERROR_CHECK_STATE_ONERR_GOTO_EXIT;
    }

    EXIT: sph_fft_operand_delete(fftop);
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}
static cpl_error_code
sph_ifs_spec_deconv_save_product(cpl_frameset* inframes,
                                 const cpl_parameterlist* inparams,
                                 const char* recipe,
                                 const cpl_imagelist* imlist,
                                 const char* fname,
                                 cpl_propertylist* pl,
                                 const cpl_frame* current_frame) {
    cpl_frameset* current_frameset = NULL;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

    cpl_ensure_code( cpl_frameset_get_size(inframes) > 0,
                     CPL_ERROR_DATA_NOT_FOUND);

    current_frameset = cpl_frameset_new();
    cpl_frameset_insert(current_frameset, cpl_frame_duplicate(current_frame));
    cpl_propertylist_update_string(pl, SPH_COMMON_KEYWORD_PRO_CATG,
            SPH_IFS_TAG_SPEC_DECONV_CALIB);
    /* Update the header if required */
    sph_utils_update_header(pl);
    cpl_dfs_save_imagelist(inframes, pl, inparams, current_frameset,
            current_frame, imlist, CPL_TYPE_DOUBLE, recipe, pl, NULL,
            SPH_PIPELINE_NAME_IFS, fname);
    cpl_frameset_delete(current_frameset);
    current_frameset = NULL;
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

