/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include "sph_common_keywords.h"
#include "sph_ird_keywords.h"
#include "sph_ird_tags.h"
#include "sph_ird_distortion_map.h"
#include "sph_ird_instrument_model.h"
#include "sph_master_frame.h"
#include "sph_error.h"
#include "sph_keyword_manager.h"
#include "sph_cube.h"
#include "sph_utils.h"
#include "sph_filemanager.h"
#include "sph_distortion_model.h"
#include "sph_framecombination.h"
#include "sph_point_pattern.h"
#include "sph_transform.h"

#include "irplib_fft.h"

#include <cpl.h>
#include <math.h>
#include <assert.h>
#include <string.h>


#define SPH_IRD_DISTORTION_MAP_POINT_LIMIT 20

#ifndef SPH_IRD_DISTORTION_MAP_QC_FILENAME
#define SPH_IRD_DISTORTION_MAP_QC_FILENAME \
    SPH_RECIPE_NAME_IRD_DISTORTION_MAP "_qc" CPL_DFS_FITS
#endif

/*-----------------------------------------------------------------------------
 Error Codes
 -----------------------------------------------------------------------------*/

extern sph_error_code SPH_IRD_DISTORTION_MAP_GENERAL;
extern sph_error_code SPH_IRD_DISTORTION_MAP_FRAMES_MISSING;

/*-----------------------------------------------------------------------------
 Private Functions
 -----------------------------------------------------------------------------*/

static sph_error_code
sph_ird_distortion_map_create_qc_of_input_pattern__(sph_point_pattern* pp,
        sph_ird_instrument_model* model, const char * filename);

static sph_error_code
sph_ird_distortion_map_create_qc_of_distortion__(cpl_image* qcim_l,
        cpl_image* qcim_r, sph_ird_instrument_model* model,
        const cpl_propertylist* qclist,
        const char * filename);

static sph_error_code
sph_ird_distortion_map_create_qc_of_pp__(sph_ird_distortion_map* self,
        sph_ird_instrument_model* model, cpl_image* qcim_l, cpl_image* qcim_r,
        sph_point_pattern* obs_left, sph_point_pattern* obs_right,
        const char * filename);
static sph_error_code
sph_ird_distortion_map_create_qc_of_accuracy__(sph_ird_distortion_map * self,
        cpl_image* qcim_l, cpl_image* qcim_r, sph_point_pattern* ppin,
        const char * filename);
static
cpl_error_code sph_ird_distortion_map_convert__( sph_ird_distortion_map* self );

static sph_master_frame*
create_science(const sph_ird_distortion_map* self, const sph_master_frame* dark,
               const sph_master_frame* flat, const cpl_frameset* rawframes)
    CPL_ATTR_ALLOC;

static void
propertylist_change_prop_name(cpl_propertylist * pl, const char * source_tag,
                              const char * dest_tag);

static
void sph_ird_instrument_qc_eval_dist(cpl_propertylist* self,
                                     const sph_distortion_model* distmap,
                                     cpl_boolean is_left)
    CPL_ATTR_NONNULL;

/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_ird_distortion_map_run Create Master Dark Recipe
 *
 * This module provides the algorithm inplementation for the creation of the
 * master dakr
 *
 * @par Synopsis:
 * @code
 *   #include "sph_ird_distortion_map.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Interpret the command line options and execute the data processing
 @param    frameset   the frames list
 @param    parlist    the parameters list

 @return   the cpl error code of the operation.

 This is the main recipe function for the sph_ird_distortion_map recipe. The error
 code returned is always a cpl error code (to allow maximal compatibility with
 esorex, gasgano, etc.) even if during recipe execution an error in the SPHERE
 API is the cause. In this case (and if the underlying error is not a cpl error)
 the cpl error code is set to the cpl_error_code that matches the failure
 reason best.
 The error from the SPHERE API is still written in the log as usual
 with the more informative and accurate sph_error_code.

 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_ird_distortion_map_run(sph_ird_distortion_map* self) {
    sph_master_frame* dark = NULL;
    sph_master_frame* flat = NULL;
    sph_master_frame* result = NULL;
    sph_master_frame* result_left = NULL;
    sph_master_frame* result_right = NULL;
    sph_point_pattern* point_pattern_left = NULL;
    sph_point_pattern* point_pattern_right = NULL;
    sph_point_pattern* point_pattern = NULL;
    sph_point_pattern* obs_pp_left = NULL;
    sph_point_pattern* obs_pp_right = NULL;
    sph_distortion_model* distmap_left = NULL;
    sph_distortion_model* distmap_right = NULL;
    sph_ird_instrument_model* model = NULL;
    cpl_table               * dist_table_left  = NULL;
    cpl_table               * dist_table_right = NULL;
    cpl_propertylist* plist  = NULL;
    cpl_propertylist* pli    = NULL;
    cpl_propertylist* qclist = NULL;
    cpl_propertylist* tplist = NULL;
    cpl_image* qcim = NULL;
    cpl_image* qcim_l = NULL;
    cpl_image* qcim_r = NULL;
    double opt_cent_l_x = -10000.0;
    double opt_cent_l_y = -10000.0;
    double opt_cent_r_x = -10000.0;
    double opt_cent_r_y = -10000.0;

    /*------------------------------------------------------------------
     -  Check for execution in conversion mode
     --------------------------------------------------------------------*/


    /*------------------------------------------------------------------
     -  Placing master dark in cube structure for later use
     --------------------------------------------------------------------*/


    /*------------------------------------------------------------------
    -  conversion mode
    --------------------------------------------------------------------*/


    if ( self->convert ) {
        return sph_ird_distortion_map_convert__( self );
    }

    /*------------------------------------------------------------------
     -  select the right dark
     --------------------------------------------------------------------*/

    if(self->skybg_fit_frame){
        dark = sph_master_frame_load_(self->skybg_fit_frame,0);
        cpl_msg_info(cpl_func,"Using SKY_BG_FIT frame as background!");
        }
        else {
            if(self->skybg_frame){
                dark = sph_master_frame_load_(self->skybg_frame,0);
                cpl_msg_info(cpl_func,"Using SKY_BG frame as background!");
            }
            else {
                if(self->insbg_fit_frame){
                    dark = sph_master_frame_load_(self->insbg_fit_frame,0);
                    cpl_msg_info(cpl_func,"Using INS_BG_FIT frame as background!");
                }
                else{
                    if(self->insbg_frame){
                        dark = sph_master_frame_load_(self->insbg_frame,0);
                        cpl_msg_info(cpl_func,"Using INS_BG frame as background!");
                    }
                    else {
                        if(self->dark_frame){
                            dark = sph_master_frame_load_(self->dark_frame,0);

                            cpl_msg_info(cpl_func,"Using master dark as background!");
                        }

                    }

                }
            }

        }

    /*------------------------------------------------------------------
    -  Begin recipe operations
    --------------------------------------------------------------------*/


    if (self->flat_frame) {
        flat = sph_master_frame_load_(self->flat_frame,
                0);
        plist = sph_keyword_manager_load_properties(
                cpl_frame_get_filename(self->flat_frame), 0);
        model = sph_ird_instrument_model_new_from_propertylist(plist);
    } else {
        model = sph_ird_instrument_model_new();
        plist = sph_ird_instrument_model_get_as_propertylist(model);
    }
    result = create_science(self, dark, flat, self->rawframes);

    if (self->user_cent) {
        opt_cent_l_x = self->cent_left_x;
        opt_cent_l_y = self->cent_left_y;
        opt_cent_r_x = self->cent_right_x;
        opt_cent_r_y = self->cent_right_y;
    }
    if (result) {
        const cpl_size pows0[2] = {0, 0};
        cpl_mask* left_mask = NULL;
        cpl_mask* right_mask = NULL;
        double avg_cent_l_x, avg_cent_l_y, avg_cent_r_x, avg_cent_r_y;
        double off_centx = 0.0, off_centy = 0.0;
        double det_centx, det_centy;
        double qc_fwhm = 0.0;

        if (!self->point_pattern_frame) {
            cpl_image* imleft = NULL;
            cpl_image* imright = NULL;
            double qc_lfwhm = 0.0;
            double qc_rfwhm = 0.0;

            SPH_INFO_MSG("No point pattern provided. "
            "Cant proceed to create distortion map and will"
            " create point pattern from raw images instead.");
            result_left = sph_ird_instrument_model_extract_left_master_frame(
                    model, result);
            imleft = sph_master_frame_extract_image(result_left, 1);
            result_right = sph_ird_instrument_model_extract_right_master_frame(
                    model, result);
            imright = sph_master_frame_extract_image(result_right, 1);

            point_pattern_left = sph_point_pattern_new_from_image(imleft, NULL,
                                       self->threshold_sigma, 3.0, 3.0, 3.0, 0,
                                                                  &qc_lfwhm);

            point_pattern_right = sph_point_pattern_new_from_image(imright,
                                 NULL, self->threshold_sigma, 3.0, 3.0, 3.0, 0,
                                                                   &qc_rfwhm);

            cpl_image_delete(imleft);
            imleft = NULL;
            cpl_image_delete(imright);
            imright = NULL;
            if (!point_pattern_left && !point_pattern_right) {
                SPH_ERR("Could not create point pattern. Cant proceed.");
                goto EXIT;
            }
            if (!point_pattern_left && point_pattern_right) {
                qc_fwhm = qc_rfwhm;
                cpl_msg_info(cpl_func, "Could not create left point pattern. "
                             "Am using only right point pattern. FWHM=%g",
                             qc_fwhm);
                point_pattern = point_pattern_right;
            }
            if (point_pattern_left && !point_pattern_right) {
                qc_fwhm = qc_lfwhm;
                cpl_msg_info(cpl_func, "Could not create right point pattern. "
                             "Am using only left point pattern. FWHM=%g",
                             qc_fwhm);
                point_pattern = point_pattern_left;
            }
            if (point_pattern_left && point_pattern_right) {
                if (sph_point_pattern_get_size(point_pattern_left)
                        <= sph_point_pattern_get_size(point_pattern_right)) {
                    qc_fwhm = qc_lfwhm;
                    cpl_msg_info(cpl_func, "Am taking the left point pattern "
                                 "as point pattern to save. FWHM=%g",
                                 qc_fwhm);
                    point_pattern = point_pattern_left;
                    sph_point_pattern_delete(point_pattern_right);
                } else {
                    qc_fwhm = qc_rfwhm;
                    cpl_msg_info(cpl_func, "Am taking the right point pattern "
                                 "as point pattern to save. FWHM=%g",
                                 qc_fwhm);
                    point_pattern = point_pattern_right;
                    sph_point_pattern_delete(point_pattern_left);
                }
            }

            sph_point_pattern_centralise_wrt_central_point(point_pattern);
            sph_point_pattern_save_dfs(point_pattern, self->pp_outfilename,
                    self->inframes, NULL, self->inparams,
                    SPH_IRD_TAG_POINT_PATTERN,
                    SPH_RECIPE_NAME_IRD_DISTORTION_MAP, SPH_PIPELINE_NAME_IRDIS,
                    NULL);
        }
        if (point_pattern == NULL) {
            cpl_errorstate prestate = cpl_errorstate_get();
            point_pattern = sph_point_pattern_new_from_table(
                    self->point_pattern_frame, 1);
            if (point_pattern == NULL) {
                cpl_errorstate_dump(prestate, CPL_FALSE,
                                    cpl_errorstate_dump_one_info);
                cpl_errorstate_set(prestate);
                cpl_msg_info(cpl_func, "Loading ASCII point pattern: %s",
                             cpl_frame_get_filename(self->point_pattern_frame));
                point_pattern = sph_point_pattern_load_ascii(
                        cpl_frame_get_filename(self->point_pattern_frame));
            }
        }
        if (!point_pattern) {
            SPH_ERR("Could not load point pattern. Can't proceed.");
            goto EXIT;
        }

        if (sph_point_pattern_get_size(
                point_pattern) < SPH_IRD_DISTORTION_MAP_POINT_LIMIT) {
            SPH_ERR("Point pattern has too few points. Can't proceed.");
            goto EXIT;
        }

        SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
                             "Determining distortion of left FOV.");
        if (!result_left)
            result_left = sph_ird_instrument_model_extract_left_master_frame(
                    model, result);
        qclist = cpl_propertylist_new();

        left_mask = sph_ird_instrument_model_get_mask_leftwin(model);
        right_mask = sph_ird_instrument_model_get_mask_rightwin(model);
        sph_ird_instrument_raw_add_qc_counts(qclist, self->rawframes,
                                             left_mask, right_mask, CPL_TRUE);
        cpl_mask_delete(left_mask);
        cpl_mask_delete(right_mask);

        det_centx = 0.5 * (1 + cpl_image_get_size_x(result_left->image));
        det_centy = 0.5 * (1 + cpl_image_get_size_y(result_left->image));

        cpl_msg_info(cpl_func, "Mean point-radius [pixel]: %g", 0.5 *
                     sqrt((double)(cpl_image_get_size_x(result_left->image) *
                                   cpl_image_get_size_y(result_left->image)) /
                          (double)sph_point_pattern_get_size(point_pattern)));

        cpl_propertylist_update_int(qclist, SPH_IRD_KEYWORD_DISTMAP_LEFT_NPOINTS_IN,
                sph_point_pattern_get_size(point_pattern));

        if (qc_fwhm > 0.0) {
            cpl_msg_warning(cpl_func, "Median FWHM for %d-point-pattern: %g",
                            sph_point_pattern_get_size(point_pattern),
                            qc_fwhm);
            cpl_propertylist_update_double(qclist, SPH_COMMON_KEYWORD_QC_FWHM,
                                           qc_fwhm);
        } else {
            cpl_msg_warning(cpl_func, "No FWHM for %d-point-pattern",
                            sph_point_pattern_get_size(point_pattern));
        }

        distmap_left = sph_distortion_model_from_grid_image(
                result_left, point_pattern, self->polyfit_order,
                self->threshold_sigma,
                self->threshold_length, qclist, 0.0, 0.0,
                &opt_cent_l_x, &opt_cent_l_y,
                &avg_cent_l_x, &avg_cent_l_y,
                &obs_pp_left, &dist_table_left);

        if(distmap_left == NULL){
            SPH_ERR("Unable to calculate the left distortion map, aborting the recipe.");
            goto EXIT;
        }

        propertylist_change_prop_name(qclist, SPH_COMMON_KEYWORD_DISTMAP_NREMOVED,
                                      SPH_IRD_KEYWORD_DISTMAP_LEFT_NREMOVED);

        cpl_propertylist_update_int(qclist,
                                    SPH_IRD_KEYWORD_DISTMAP_LEFT_NPOINTS_OBS,
                                    sph_point_pattern_get_size(obs_pp_left));

        cpl_propertylist_update_double(qclist,
                                       SPH_IRD_KEYWORD_DISTMAP_LEFT_OPTICAL_AXIS_X,
                                       opt_cent_l_x);
        cpl_propertylist_update_double(qclist,
                                       SPH_IRD_KEYWORD_DISTMAP_LEFT_OPTICAL_AXIS_Y,
                                       opt_cent_l_y);

        cpl_propertylist_update_string(qclist,
                                       SPH_DOUBLE_IMAGE_EXTNAME_ADDITION_LEFT,
                                       "LEFT.DISTX");
        cpl_propertylist_update_string(qclist,
                                       SPH_DOUBLE_IMAGE_EXTNAME_ADDITION_RIGHT,
                                       "LEFT.DISTY");

        pli = cpl_propertylist_load(cpl_frame_get_filename
                                    (cpl_frameset_get_first(self->rawframes)),
                                    0);
        sph_utils_simple_copy_singular(pli, qclist);
        cpl_propertylist_delete(pli);
        pli = NULL;

        SPH_ERROR_RAISE_INFO( SPH_ERROR_GENERAL, "Creating QC image by "
                              "applying distortionmap to input image itself.");

        qcim_l = sph_distortion_model_apply(distmap_left, result_left->image,
                                            CPL_KERNEL_DEFAULT,
                                            2 * CPL_KERNEL_DEF_WIDTH,
                                            SPH_COMMON_KEYWORD_QC_LX,
                                            SPH_COMMON_KEYWORD_QC_LY, qclist);
        sph_ird_instrument_qc_eval_dist(qclist, distmap_left, CPL_TRUE);
        sph_distortion_model_save_dfs(distmap_left,
                                      self->outfilename, self->inframes,
                                      NULL, self->inparams,
                                      SPH_IRD_TAG_DISTORTION_MAP_CALIB,
                                      SPH_RECIPE_NAME_IRD_DISTORTION_MAP,
                                      SPH_PIPELINE_NAME_IRDIS, qclist,
                                      cpl_image_get_size_x(result_left->image),
                                      cpl_image_get_size_y(result_left->image),
                                      1.0, 0, NULL,
                                      SPH_IRD_KEYWORD_DISTMAP_LEFT_COEFFX,
                                      SPH_IRD_KEYWORD_DISTMAP_LEFT_COEFFY);

        SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
                             "Left distortion map saved.");


        /* Keep QC keys, need both left and right QC keys for anamorphism */
        cpl_propertylist_erase_regexp(qclist, " QC ", CPL_TRUE);
        SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
                "Determining distortion of right FOV.");
        if (!result_right)
            result_right = sph_ird_instrument_model_extract_right_master_frame(
                    model, result);

        cpl_propertylist_update_int(qclist, SPH_IRD_KEYWORD_DISTMAP_RIGHT_NPOINTS_IN,
                sph_point_pattern_get_size(point_pattern));

        irplib_image_find_shift(result_left->image, result_right->image,
                                &off_centx, &off_centy);

        cpl_msg_info(cpl_func, "Right channel offset from left by (%g, %g)",
                     off_centx, off_centy);

        /* Distortion model uses detector center as Center of Origin */
        off_centx -= opt_cent_l_x - det_centx;
        off_centy -= opt_cent_l_y - det_centy;

        distmap_right = sph_distortion_model_from_grid_image(
                result_right, point_pattern, self->polyfit_order,
                self->threshold_sigma,
                self->threshold_length, qclist, off_centx, off_centy,
                &opt_cent_r_x, &opt_cent_r_y,
                &avg_cent_r_x, &avg_cent_r_y,
                &obs_pp_right, &dist_table_right);

        if(distmap_right == NULL){
            SPH_ERR("Unable to calculate the right distortion map, aborting the recipe.");
            goto EXIT;
        }

        if (self->align_right) {
            const double rdistx =
                cpl_polynomial_get_coeff(distmap_right->polyx, pows0);
            const double rdisty =
                cpl_polynomial_get_coeff(distmap_right->polyy, pows0);

            const double cshiftx = opt_cent_l_x - opt_cent_r_x;
            const double cshifty = opt_cent_l_y - opt_cent_r_y;

            cpl_msg_warning(cpl_func, "Aligning right channel onto left by "
                            "(%g,%g)", cshiftx, cshifty);

            cpl_polynomial_set_coeff(distmap_right->polyx, pows0,
                                     rdistx + cshiftx);
            cpl_polynomial_set_coeff(distmap_right->polyy, pows0,
                                     rdisty + cshifty);

        }

        propertylist_change_prop_name(qclist, SPH_COMMON_KEYWORD_DISTMAP_NREMOVED,
                                      SPH_IRD_KEYWORD_DISTMAP_RIGHT_NREMOVED);

        cpl_propertylist_update_int(qclist,
                                    SPH_IRD_KEYWORD_DISTMAP_RIGHT_NPOINTS_OBS,
                                    sph_point_pattern_get_size(obs_pp_right));

        cpl_propertylist_update_double(qclist,
                                       SPH_IRD_KEYWORD_DISTMAP_RIGHT_OPTICAL_AXIS_X, opt_cent_r_x);
        cpl_propertylist_update_double(qclist,
                                       SPH_IRD_KEYWORD_DISTMAP_RIGHT_OPTICAL_AXIS_Y, opt_cent_r_y);

        cpl_propertylist_update_string(qclist,
                                       SPH_DOUBLE_IMAGE_EXTNAME_ADDITION_LEFT, "RIGHT.DISTX");
        cpl_propertylist_update_string(qclist,
                                       SPH_DOUBLE_IMAGE_EXTNAME_ADDITION_RIGHT, "RIGHT.DISTY");

        SPH_ERROR_RAISE_INFO( SPH_ERROR_GENERAL, "Creating QC image by "
                              "applying distortionmap to input image itself.");

        qcim_r = sph_distortion_model_apply(distmap_right, result_right->image,
                                            CPL_KERNEL_DEFAULT,
                                            CPL_KERNEL_DEF_WIDTH,
                                            SPH_COMMON_KEYWORD_QC_RX,
                                            SPH_COMMON_KEYWORD_QC_RY, qclist);

        sph_ird_instrument_qc_eval_dist(qclist, distmap_right, CPL_FALSE);
        sph_distortion_model_save(distmap_right,
                                             self->outfilename,
                                             cpl_image_get_size_x(result_right->image),
                                             cpl_image_get_size_y(result_right->image), CPL_IO_EXTEND,
                                             qclist, NULL, SPH_IRD_KEYWORD_DISTMAP_RIGHT_COEFFX,
                                             SPH_IRD_KEYWORD_DISTMAP_RIGHT_COEFFY);

        tplist = cpl_propertylist_new();
        cpl_propertylist_append_string(tplist, "EXTNAME",
                                       "Detected point sources - left");

        cpl_table_save(dist_table_left, NULL, tplist, self->outfilename,
                       CPL_IO_EXTEND);
        cpl_propertylist_update_string(tplist, "EXTNAME",
                                       "Detected point sources - right");
        cpl_table_save(dist_table_right, NULL, tplist, self->outfilename,
                       CPL_IO_EXTEND);

        if (self->full_qc) {

            cpl_propertylist_append_string(qclist, CPL_DFS_PRO_CATG,
                                           SPH_IRD_TAG_POINT_PATTERN_QC);
            cpl_dfs_save_propertylist(self->inframes, NULL, self->inparams,
                                      self->inframes, NULL,
                                      SPH_RECIPE_NAME_IRD_DISTORTION_MAP,
                                      qclist, NULL, SPH_PIPELINE_NAME_IRDIS,
                                      SPH_IRD_DISTORTION_MAP_QC_FILENAME);

            sph_ird_distortion_map_create_qc_of_input_pattern__(point_pattern,
                    model, SPH_IRD_DISTORTION_MAP_QC_FILENAME);

            sph_ird_distortion_map_create_qc_of_distortion__(qcim_l, qcim_r,
                    model, qclist, SPH_IRD_DISTORTION_MAP_QC_FILENAME);

            sph_ird_distortion_map_create_qc_of_pp__(self, model, qcim_l,
                    qcim_r, obs_pp_left, obs_pp_right,
                    SPH_IRD_DISTORTION_MAP_QC_FILENAME);

            sph_ird_distortion_map_create_qc_of_accuracy__(self, qcim_l, qcim_r,
                    point_pattern, SPH_IRD_DISTORTION_MAP_QC_FILENAME);

            sph_point_pattern_draw_save(obs_pp_left,  result_left->image,
                                        obs_pp_right, result_right->image,
                                        point_pattern,
                                        qcim_l, opt_cent_l_x, opt_cent_l_y,
                                        qcim_r, self->align_right
                                        ? opt_cent_l_x : opt_cent_r_x,
                                        self->align_right
                                        ? opt_cent_l_y : opt_cent_r_y,
                                        5.0, SPH_IRD_DISTORTION_MAP_QC_FILENAME);
        }
        EXIT: cpl_image_delete(qcim_l);
        qcim_l = NULL;
        cpl_image_delete(qcim_r);
        qcim_r = NULL;
        cpl_image_delete(qcim);
        qcim = NULL;

        cpl_propertylist_delete(qclist);
        qclist = NULL;
        sph_point_pattern_delete(point_pattern);
        point_pattern = NULL;
        sph_master_frame_delete(result);
        result = NULL;
        sph_master_frame_delete(result_left);
        result_left = NULL;
        sph_master_frame_delete(result_right);
        result_right = NULL;
    }

    if (dark) {
        sph_master_frame_delete(dark);
        dark = NULL;
    }
    if (flat) {
        sph_master_frame_delete(flat);
        flat = NULL;
    }
    sph_distortion_model_delete(distmap_left);
    distmap_left = NULL;
    sph_distortion_model_delete(distmap_right);
    distmap_right = NULL;
    sph_ird_instrument_model_delete(model);
    model = NULL;
    cpl_propertylist_delete(plist);
    plist = NULL;
    cpl_propertylist_delete(tplist);
    sph_point_pattern_delete(obs_pp_left);
    sph_point_pattern_delete(obs_pp_right);
    cpl_table_delete(dist_table_left);
    cpl_table_delete(dist_table_right);
    //sph_filemanager_clean();
    return (int) cpl_error_get_code();
}

static sph_error_code sph_ird_distortion_map_create_qc_of_input_pattern__(
        sph_point_pattern* pp, sph_ird_instrument_model* model,
        const char * filename) {
    cpl_image* testim_left = NULL;
    cpl_image* testim_right = NULL;
    cpl_image* testim = NULL;
    sph_point* point = NULL;
    double dx = 0.0;
    double dy = 0.0;
    cpl_propertylist* extlist = cpl_propertylist_new();
    point = sph_point_pattern_get_point(pp,
            sph_point_pattern_get_central_point(pp, &dx, &dy));
    sph_point_pattern_add_offset(pp, model->window_size_y / 2 - point->x,
            model->window_size_y / 2 - point->y);

    cpl_propertylist_append_string(extlist, SPH_COMMON_KEYWORD_EXTNAME,
                                   "Input pattern left");
    testim_left = sph_point_pattern_create_image(pp, model->window_size_y,
            model->window_size_y, 1.5);
    cpl_ensure_code(testim_left != NULL, cpl_error_get_code());

    cpl_image_save(testim_left, filename, CPL_TYPE_DOUBLE,
            extlist, CPL_IO_EXTEND);

    testim_right = sph_point_pattern_create_image(pp, model->window_size_y,
            model->window_size_y, 1.5);
    cpl_ensure_code(testim_right != NULL, cpl_error_get_code());

    cpl_propertylist_set_string(extlist, SPH_COMMON_KEYWORD_EXTNAME,
                                   "Input pattern right");
    cpl_image_save(testim_right, filename, CPL_TYPE_DOUBLE,
            extlist, CPL_IO_EXTEND);

    testim = sph_ird_instrument_model_assemble_image(model, testim_left,
            testim_right);

    cpl_propertylist_set_string(extlist, SPH_COMMON_KEYWORD_EXTNAME,
                                "Input pattern");
    cpl_image_save(testim, filename, CPL_TYPE_DOUBLE, extlist,
            CPL_IO_EXTEND);

    cpl_propertylist_delete(extlist);
    cpl_image_delete(testim_left);
    testim_left = NULL;
    cpl_image_delete(testim_right);
    testim_right = NULL;
    cpl_image_delete(testim);
    testim = NULL;
    sph_point_delete(point);
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

static sph_error_code sph_ird_distortion_map_create_qc_of_distortion__(
        cpl_image* qcim_l, cpl_image* qcim_r, sph_ird_instrument_model* model,
        const cpl_propertylist* qclist,
        const char * filename) {
    cpl_propertylist* extlist = cpl_propertylist_duplicate(qclist);
    cpl_image* qcim = NULL;

    cpl_propertylist_append_string(extlist, SPH_COMMON_KEYWORD_EXTNAME,
                                   "Image");
    if (model && qcim_l && qcim_r) {
        qcim = sph_ird_instrument_model_assemble_image(model, qcim_l, qcim_r);
        cpl_image_save(qcim, filename, CPL_TYPE_DOUBLE, extlist,
                CPL_IO_EXTEND);
        SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL, "Full QC image  saved.");
    }

    if (qcim_l) {
        cpl_propertylist_set_string(extlist, SPH_COMMON_KEYWORD_EXTNAME,
                                    "Image left");
        cpl_image_save(qcim_l, filename, CPL_TYPE_DOUBLE, extlist,
                CPL_IO_EXTEND);
        SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL, "Left QC image  saved.");
    }
    if (qcim_r) {
        cpl_propertylist_set_string(extlist, SPH_COMMON_KEYWORD_EXTNAME,
                                    "Image right");
        cpl_image_save(qcim_r, filename, CPL_TYPE_DOUBLE, extlist,
                CPL_IO_EXTEND);
        SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL, "Right QC image saved.");
    }
    cpl_propertylist_delete(extlist);
    cpl_image_delete(qcim);
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}
static sph_error_code sph_ird_distortion_map_create_qc_of_accuracy__(
        sph_ird_distortion_map * self, cpl_image* qcim_l, cpl_image* qcim_r,
        sph_point_pattern* ppin,
        const char * filename) {
    sph_point_pattern* pp_left_corrected = NULL;
    sph_point_pattern* pp_right_corrected = NULL;
    sph_distortion_map* dmap = NULL;
    sph_dataset* dat = NULL;
    int ii = 0;
    int nn = 0;
    double rr = 0.0;
    cpl_image* im = NULL;
    cpl_image* imdx = NULL;
    cpl_image* imdy = NULL;
    cpl_image* nim = NULL;
    double cx = 512.0;
    double cy = 512.0;
    int bpix = 0;

    cpl_propertylist* extlist = cpl_propertylist_new();

#ifdef SPH_IRD_OPT_CENT
    double opt_cent_l_x;
    double opt_cent_l_y;
    double opt_cent_r_x;
    double opt_cent_r_y;
    if (self->user_cent) {
        opt_cent_l_x = self->cent_left_x;
        opt_cent_l_y = self->cent_left_y;
        opt_cent_r_x = self->cent_right_x;
        opt_cent_r_y = self->cent_right_y;
    }
#endif

    if (qcim_l) {
        pp_left_corrected = sph_point_pattern_new_from_image(qcim_l, NULL,
                                       self->threshold_sigma, 3.0, 3.0, 3.0, 0, NULL);
        sph_point_pattern_centralise_wrt_central_point(pp_left_corrected);
        sph_point_pattern_centralise_wrt_central_point(ppin);

        dmap = sph_distortion_map_new(pp_left_corrected, ppin);
        sph_distortion_map_delete_multi(dmap);

        dat = sph_dataset_new(100, -1.0, 9.5);
        im = cpl_image_new(65, 65, CPL_TYPE_DOUBLE);
        imdx = cpl_image_new(65, 65, CPL_TYPE_DOUBLE);
        imdy = cpl_image_new(65, 65, CPL_TYPE_DOUBLE);
        nim = cpl_image_new(65, 65, CPL_TYPE_DOUBLE);

        sph_distortion_map_remove_offset(dmap, NULL, NULL);

        SPH_RAISE_CPL_RESET;
        for (ii = 0; ii < sph_distortion_map_get_size(dmap); ++ii) {
            sph_distortion_vector dv;
            const cpl_error_code mycode =
                sph_distortion_map_get_one(dmap, ii, &dv);
            assert(!mycode);
            rr = dv.dx * dv.dx + dv.dy * dv.dy;
            rr = sqrt(rr);
            cpl_image_set(im, (dv.x + cx) * 60.0 / 1024.0 + 1,
                    (dv.y + cy) * 60.0 / 1024.0 + 1, rr);
            cpl_image_set(imdx, (dv.x + cx) * 60.0 / 1024.0 + 1,
                    (dv.y + cy) * 60.0 / 1024.0 + 1, dv.dx);
            cpl_image_set(imdy, (dv.x + cx) * 60.0 / 1024.0 + 1,
                    (dv.y + cy) * 60.0 / 1024.0 + 1, dv.dy);

            nn = cpl_image_get(nim, (dv.x + cx) * 60.0 / 1024.0 + 1,
                    (dv.y + cy) * 60.0 / 1024.0 + 1, &bpix);

            cpl_image_set(nim, (dv.x + cx) * 60.0 / 1024.0 + 1,
                    (dv.y + cy) * 60.0 / 1024.0 + 1, nn + 1);
            if (rr > 9.0)
                rr = 9.0;
            sph_dataset_add_value(dat, rr, 1.0);
        }
        SPH_RAISE_CPL_RESET;

#ifdef SPH_SAVE_NONDFS
        sph_dataset_save_ascii(dat, "qc_residuals_left_hist.txt");
#endif

        cpl_image_threshold(nim, -0.5, 0.5, 1.0, 1.0);
        if (cpl_image_divide(im, nim)) {
            (void)cpl_error_set_where(cpl_func);
        } else {
            cpl_image_divide(imdx, nim);
            cpl_image_divide(imdy, nim);

            cpl_propertylist_append_string(extlist, SPH_COMMON_KEYWORD_EXTNAME,
                                           "Residuals left");
            cpl_image_save(im, filename, CPL_TYPE_DOUBLE, extlist,
                           CPL_IO_EXTEND);
            cpl_propertylist_set_string(extlist, SPH_COMMON_KEYWORD_EXTNAME,
                                        "Residuals left X");
            cpl_image_save(imdx, filename, CPL_TYPE_DOUBLE, extlist,
                           CPL_IO_EXTEND);
            cpl_propertylist_set_string(extlist, SPH_COMMON_KEYWORD_EXTNAME,
                                        "Residuals left Y");
            cpl_image_save(imdy, filename, CPL_TYPE_DOUBLE, extlist,
                           CPL_IO_EXTEND);
        }

        sph_point_pattern_delete(pp_left_corrected);
        sph_distortion_map_delete(dmap);
        dmap = NULL;
        sph_dataset_delete(dat);
        dat = NULL;
        cpl_image_delete(im);
        im = NULL;
        cpl_image_delete(imdx);
        imdx = NULL;
        cpl_image_delete(imdy);
        imdy = NULL;
        cpl_image_delete(nim);
        nim = NULL;
        pp_left_corrected = NULL;
    }
    if (qcim_r) {
        SPH_RAISE_CPL_RESET;
        pp_right_corrected = sph_point_pattern_new_from_image(qcim_r, NULL,
                                       self->threshold_sigma, 3.0, 3.0, 3.0, 0, NULL);

        sph_point_pattern_centralise_wrt_central_point(pp_right_corrected);
        sph_point_pattern_centralise_wrt_central_point(ppin);
        dmap = sph_distortion_map_new(pp_right_corrected, ppin);
        sph_distortion_map_delete_multi(dmap);

        dat = sph_dataset_new(100, -1.0, 9.5);

        im = cpl_image_new(65, 65, CPL_TYPE_DOUBLE);
        imdx = cpl_image_new(65, 65, CPL_TYPE_DOUBLE);
        imdy = cpl_image_new(65, 65, CPL_TYPE_DOUBLE);
        nim = cpl_image_new(65, 65, CPL_TYPE_DOUBLE);

        sph_distortion_map_remove_offset(dmap, NULL, NULL);

        for (ii = 0; ii < sph_distortion_map_get_size(dmap); ++ii) {
            sph_distortion_vector dv;
            const cpl_error_code mycode =
                sph_distortion_map_get_one(dmap, ii, &dv);
            assert(!mycode);

            rr = dv.dx * dv.dx + dv.dy * dv.dy;
            rr = sqrt(rr);
            cpl_image_set(im, (dv.x + cx) * 60.0 / 1024.0 + 1,
                    (dv.y + cy) * 60.0 / 1024.0 + 1, rr);
            cpl_image_set(imdx, (dv.x + cx) * 60.0 / 1024.0 + 1,
                    (dv.y + cy) * 60.0 / 1024.0 + 1, dv.dx);
            cpl_image_set(imdy, (dv.x + cx) * 60.0 / 1024.0 + 1,
                    (dv.y + cy) * 60.0 / 1024.0 + 1, dv.dy);
            nn = cpl_image_get(nim, (dv.x + cx) * 60.0 / 1024.0 + 1,
                    (dv.y + cy) * 60.0 / 1024.0 + 1, &bpix);

            cpl_image_set(nim, (dv.x + cx) * 60.0 / 1024.0 + 1,
                    (dv.y + cy) * 60.0 / 1024.0 + 1, nn + 1);
            if (rr > 9.0)
                rr = 9.0;
            sph_dataset_add_value(dat, rr, 1.0);
        }

#ifdef SPH_SAVE_NONDFS
        sph_dataset_save_ascii(dat, "qc_residuals_right_hist.txt");
#endif
        cpl_image_threshold(nim, -0.5, 0.5, 1.0, 1.0);
        if (cpl_image_divide(im, nim)) {
            (void)cpl_error_set_where(cpl_func);
        } else {
            cpl_image_divide(imdx, nim);
            cpl_image_divide(imdy, nim);

            cpl_propertylist_set_string(extlist, SPH_COMMON_KEYWORD_EXTNAME,
                                        "Residuals right");
            cpl_image_save(im, filename, CPL_TYPE_DOUBLE, extlist,
                           CPL_IO_EXTEND);
            cpl_propertylist_set_string(extlist, SPH_COMMON_KEYWORD_EXTNAME,
                                        "Residuals right X");
            cpl_image_save(imdx, filename, CPL_TYPE_DOUBLE, extlist,
                           CPL_IO_EXTEND);
            cpl_propertylist_set_string(extlist, SPH_COMMON_KEYWORD_EXTNAME,
                                        "Residuals right Y");
            cpl_image_save(imdy, filename, CPL_TYPE_DOUBLE, extlist,
                           CPL_IO_EXTEND);
        }

        sph_distortion_map_delete(dmap);
        dmap = NULL;
        sph_dataset_delete(dat);
        dat = NULL;
        cpl_image_delete(im);
        im = NULL;
        cpl_image_delete(nim);
        nim = NULL;
        sph_point_pattern_delete(pp_right_corrected);
        cpl_image_delete(imdx);
        imdx = NULL;
        cpl_image_delete(imdy);
        imdy = NULL;
        pp_right_corrected = NULL;
    }
    cpl_propertylist_delete(extlist);

    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

static sph_error_code sph_ird_distortion_map_create_qc_of_pp__(
        sph_ird_distortion_map * self, sph_ird_instrument_model* model,
        cpl_image* qcim_l, cpl_image* qcim_r, sph_point_pattern* obs_left,
        sph_point_pattern* obs_right,
        const char * filename) {
    sph_point_pattern* pp_left_corrected = NULL;
    sph_point_pattern* pp_right_corrected = NULL;
    cpl_image* imleft = NULL;
    cpl_image* imright = NULL;
    cpl_image* im = NULL;
    cpl_propertylist* extlist = cpl_propertylist_new();
    double qc_fwhm  = 0.0;
    double qc_lfwhm = 0.0;
    double qc_rfwhm = 0.0;

    cpl_propertylist_append_string(extlist, SPH_COMMON_KEYWORD_EXTNAME,
                                   "Point pattern left");

    if (qcim_l) {
        pp_left_corrected = sph_point_pattern_new_from_image(qcim_l, NULL,
                                       self->threshold_sigma, 3.0, 3.0, 3.0, 0,
                                                             &qc_lfwhm);
        sph_point_pattern_centralise_wrt_central_point(pp_left_corrected);
#ifdef SPH_SAVE_NONDFS
        sph_point_pattern_save_ascii(pp_left_corrected,
                "qc_pp_left_corrected.txt");
#endif
        sph_point_pattern_delete(pp_left_corrected);
        pp_left_corrected = NULL;
    }
    if (qcim_r) {
        pp_right_corrected = sph_point_pattern_new_from_image(qcim_r, NULL,
                                       self->threshold_sigma, 3.0, 3.0, 3.0, 0,
                                                              &qc_rfwhm);
        sph_point_pattern_centralise_wrt_central_point(pp_right_corrected);
#ifdef SPH_SAVE_NONDFS
        sph_point_pattern_save_ascii(pp_right_corrected,
                "qc_pp_right_corrected.txt");
#endif
        sph_point_pattern_delete(pp_right_corrected);
        pp_right_corrected = NULL;
    }
    if (obs_left) {
#ifdef SPH_SAVE_NONDFS
        sph_point_pattern_save_ascii(obs_left, "qc_pp_left.txt");
#endif

        SPH_ERROR_RAISE_INFO( SPH_ERROR_GENERAL,
                "QC point pattern (left) saved");
        sph_point_pattern_add_offset(obs_left, model->window_size_y / 2.0,
                model->window_size_y / 2.0);
        imleft = sph_point_pattern_create_image(obs_left, model->window_size_y,
                model->window_size_y, 3.0);

        if (qc_lfwhm > 0.0) {
            cpl_msg_warning(cpl_func, "Left median FWHM for point-pattern: %g",
                            qc_lfwhm);
            cpl_propertylist_update_double(extlist, SPH_COMMON_KEYWORD_QC_FWHM,
                                           qc_lfwhm);
        } else {
            cpl_msg_warning(cpl_func, "No left FWHM for point-pattern");
        }

        cpl_image_save(imleft, filename, CPL_TYPE_DOUBLE, extlist,
                CPL_IO_EXTEND);
    }
    if (obs_right) {
#ifdef SPH_SAVE_NONDFS
        sph_point_pattern_save_ascii(obs_right, "qc_pp_right.txt");
#endif
        SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
                "QC point pattern (right) saved");

        sph_point_pattern_add_offset(obs_right, model->window_size_y / 2.0,
                model->window_size_y / 2.0);
        imright = sph_point_pattern_create_image(obs_right,
                model->window_size_y, model->window_size_y, 3.0);
        cpl_propertylist_set_string(extlist, SPH_COMMON_KEYWORD_EXTNAME,
                                       "Point pattern right");

        if (qc_rfwhm > 0.0) {
            cpl_msg_warning(cpl_func, "Right median FWHM for point-pattern: %g",
                            qc_rfwhm);
            cpl_propertylist_update_double(extlist, SPH_COMMON_KEYWORD_QC_FWHM,
                                           qc_rfwhm);
        } else {
            cpl_propertylist_erase(extlist, SPH_COMMON_KEYWORD_QC_FWHM);
            cpl_msg_warning(cpl_func, "No right FWHM for point-pattern");
        }

        cpl_image_save(imright, filename, CPL_TYPE_DOUBLE, extlist,
                CPL_IO_EXTEND);
    }
    if (imright && imleft) {
        if (qc_rfwhm > 0.0) {
            qc_fwhm = qc_lfwhm > 0.0 ? (qc_rfwhm + qc_lfwhm)/2.0 : qc_rfwhm;
        } else {
            qc_fwhm = qc_lfwhm;
        }
        if (qc_fwhm > 0.0) {
            cpl_msg_warning(cpl_func, "Total median FWHM for point-pattern: %g",
                            qc_fwhm);
            cpl_propertylist_update_double(extlist, SPH_COMMON_KEYWORD_QC_FWHM,
                                           qc_fwhm);
        } else {
            cpl_propertylist_erase(extlist, SPH_COMMON_KEYWORD_QC_FWHM);
            cpl_msg_warning(cpl_func, "No total FWHM for point-pattern");
        }
        im = sph_ird_instrument_model_assemble_image(model, imleft, imright);
        cpl_propertylist_set_string(extlist, SPH_COMMON_KEYWORD_EXTNAME,
                                       "Point pattern full");
        cpl_image_save(im, filename, CPL_TYPE_DOUBLE, extlist,
                CPL_IO_EXTEND);
        cpl_image_delete(im);
        im = NULL;
    }
    cpl_propertylist_delete(extlist);
    cpl_image_delete(imleft);
    cpl_image_delete(imright);
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

static sph_master_frame*
create_science(const sph_ird_distortion_map* self, const sph_master_frame* dark,
               const sph_master_frame* flat, const cpl_frameset* rawframes) {
    sph_master_frame* mframe = NULL;

    mframe = sph_framecombination_master_frame_from_cpl_frameset(rawframes,
            self->coll_alg, self->framecomb_parameterlist);

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        sph_master_frame_delete(mframe);
        mframe = NULL;
    }

    if (!mframe)
        return NULL;

    if (dark)
        sph_master_frame_subtract_master_frame(mframe, dark);
    if (flat)
        sph_master_frame_divide_master_frame(mframe, flat);

    if (sph_master_frame_interpolate_bpix(mframe) == CPL_ERROR_NONE) {
        SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
                "Interpolating badpixels on reduced science image.");
    } else {
        SPH_ERROR_RAISE_WARNING(SPH_ERROR_GENERAL,
                "No interpolation of badpixels possible.");
        cpl_error_reset();
    }
    return mframe;
}

static
cpl_error_code sph_ird_distortion_map_convert__( sph_ird_distortion_map* self )
{
    sph_point_pattern* point_pattern = NULL;

    int mode = 1; // FITS TO ASCII
    if ( !self->point_pattern_frame ) {
        SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_INPUT,
                "In conversion mode, you must provide the point pattern table.");
        return CPL_ERROR_ILLEGAL_INPUT;
    }
    point_pattern = sph_point_pattern_new_from_table(
            self->point_pattern_frame, 1);
    if (point_pattern == NULL) {
        cpl_error_reset();
        mode = 2; // ASCII TO FITS
        point_pattern = sph_point_pattern_load_ascii(
                cpl_frame_get_filename(self->point_pattern_frame));
    }
    if (!point_pattern) {
        SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_INPUT,
                "Could not load point pattern. Cant proceed.");
        return CPL_ERROR_ILLEGAL_INPUT;
    }
    if ( mode == 1 ) {
#ifdef SPH_SAVE_NONDFS
        sph_point_pattern_save_ascii(point_pattern,self->outfilename);
#endif
        SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
    }
    if ( mode == 2 ) {
#ifdef SPH_SAVE_NONDFS
        sph_point_pattern_save(point_pattern,self->outfilename);
#endif
        SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
    }
    return CPL_ERROR_NONE;
}

static void
propertylist_change_prop_name(cpl_propertylist * pl,
                              const char * source_tag,
                              const char * dest_tag)
{
    if(pl == NULL || source_tag == NULL || dest_tag == NULL) return;
    if(!cpl_propertylist_has(pl, source_tag)) return;

    cpl_property * prop = cpl_propertylist_get_property(pl, source_tag);
    cpl_propertylist_erase(pl, dest_tag);
    cpl_property_set_name(prop, dest_tag);
}

/*----------------------------------------------------------------------------*/
/**
 @internal
 @brief    Add QC distortion point parameters to QC propertylist
 @param    self         The QC propertylist to update
 @param    distmap      The distortion map object
 @param    is_left      True iff left, else right
 @return   Nothing

 */
/*----------------------------------------------------------------------------*/
static
void sph_ird_instrument_qc_eval_dist(cpl_propertylist* self,
                                     const sph_distortion_model* distmap,
                                     cpl_boolean is_left)
{
    const double x12[2][2] = {{SPH_COMMON_KEYWORD_QC_DX_1,
                               SPH_COMMON_KEYWORD_QC_DX_2},
                              {SPH_COMMON_KEYWORD_QC_EDX_1,
                               SPH_COMMON_KEYWORD_QC_EDX_2}};
    const double y12[2][2] = {{SPH_COMMON_KEYWORD_QC_DY_1,
                               SPH_COMMON_KEYWORD_QC_DY_2},
                              {SPH_COMMON_KEYWORD_QC_EDY_1,
                               SPH_COMMON_KEYWORD_QC_EDY_2}};
    const size_t keysz[2] = {SPH_COMMON_KEYWORD_QC_DXL_SZ,
                             SPH_COMMON_KEYWORD_QC_DEXL_SZ};
    char qckey1[SPH_COMMON_KEYWORD_QC_DXL_SZ] = SPH_COMMON_KEYWORD_QC_DXL_12;
    char qckey2[SPH_COMMON_KEYWORD_QC_DEXL_SZ] = SPH_COMMON_KEYWORD_QC_DEXL_12;
    char *qckey[2] = {qckey1, qckey2};
    const size_t csz = 80;
    char comment[csz];
    const char * commstart = is_left ? "X-distortion Left at ("
        : "X-distortion Right at (";
    const size_t pcsz = strlen(commstart); /* Start of comment remainder */

    (void)memcpy(comment, commstart, pcsz);

    for (int itype = 0; itype < 2; itype++) {
        /* Addresses where to write the four varying key elements */
        char * pxy = memchr(qckey[itype], 'X', keysz[itype]);
        char * plr = 1 + pxy;
        char * pd1 = 2 + plr;
        char * pd2 = 1 + pd1;

        assert(pxy > qckey[itype]);
        assert(pd2 < qckey[itype] + keysz[itype]);

        *plr = is_left ? 'L' : 'R';

        for (*pd1 = '1'; *pd1 <= '2'; ++*pd1) {
            for (*pd2 = '1'; *pd2 <= '2'; ++*pd2) {
                const double xypos[2] = {x12[itype][*pd1 - '1'],
                                         y12[itype][*pd2 - '1']};
                double xydist[2];
                const int ncsz = snprintf(comment + pcsz, csz - pcsz,
                                          "%.f, %.f)", xypos[0], xypos[1]);

                assert(ncsz + (int)pcsz < (int)csz);

                sph_distortion_model_get_qc_dist(distmap, xypos, xydist);

                /* Change key and comment and set the QC parameter */
                *pxy = *comment = 'X';
                cpl_propertylist_append_double(self, qckey[itype], xydist[0]);
                cpl_propertylist_set_comment(self, qckey[itype], comment);

                *pxy = *comment = 'Y';
                cpl_propertylist_append_double(self, qckey[itype], xydist[1]);
                cpl_propertylist_set_comment(self, qckey[itype], comment);
            }
        }

    }

    if (distmap->polyx != NULL && distmap->polyx != NULL) {
        /* Compute anamorphism */
        const cpl_size pows01y[2] = {0, 1};
        const cpl_size pows10x[2] = {1, 0};

        const double c01y = cpl_polynomial_get_coeff(distmap->polyy, pows01y);
        const double c10x = cpl_polynomial_get_coeff(distmap->polyx, pows10x);

        const double anam = (1.0 + c01y) / (1.0 + c10x);

        cpl_propertylist_append_double(self, is_left ?
                                       SPH_COMMON_KEYWORD_QC_ANAM_LEFT :
                                       SPH_COMMON_KEYWORD_QC_ANAM_RIGHT, anam);
    }
}

/**@}*/
