/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <cpl.h>
#include <math.h>
#include "sph_common_keywords.h"
#include "sph_ird_keywords.h"
#include "sph_time.h"
#include "sph_ird_tags.h"
#include "sph_ird_science_dbi.h"
#include "sph_master_frame.h"
#include "sph_error.h"
#include "sph_cube.h"
#include "sph_utils.h"
#include "sph_ird_instrument_model.h"
#include "sph_framecombination.h"
#include "sph_ird_common_science.h"
#include "sph_keyword_manager.h"
#include "sph_differential_imaging.h"
#include "sph_filemanager.h"
#include "sph_simple_adi.h"
#include "sph_common_science.h"
#include "sph_fits.h"
#include "sph_strehl.h"
/*-----------------------------------------------------------------------------
 Error Codes
 -----------------------------------------------------------------------------*/

extern sph_error_code SPH_IRD_SCIENCE_DBI_GENERAL;
extern sph_error_code SPH_IRD_SCIENCE_DBI_FRAMES_MISSING;

const char* const SPH_IRD_STAR_CENTER_FILE_NAME = "WAFFFRAME";
const char* const SPH_IRD_STAR_CENTER_ANGLE_NAME = "ANGLE";

static sph_master_frame*
sph_ird_science_dbi_process_cubes(sph_ird_science_dbi* self,
                                  cpl_frameset* fcframes,
                                  sph_master_frame** left_out,
                                  sph_master_frame** right_out,
                                  cpl_frameset**,
                                  cpl_frameset**,
                                  cpl_frameset**);

static cpl_error_code
sph_ird_science_dbi_make_template(sph_ird_science_dbi* self);


/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_ird_science_dbi_run
 *
 * This module provides the core IRDIS science recipe for the DBI mode
 *
 * @par Synopsis:
 * @code
 * #include "sph_ird_science_dbi.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Interpret the command line options and execute the data processing
 @param    frameset   the frames list
 @param    parlist    the parameters list

 @return   the cpl error code of the operation.

 This is the main recipe function for the sph_ird_science_dbi recipe. The error
 code returned is always a cpl error code (to allow maximal compatibility with
 esorex, gasgano, etc.) even if during recipe execution an error in the SPHERE
 API is the cause. In this case (and if the underlying error is not a cpl error)
 the cpl error code is set to the cpl_error_code that matches the failure
 reason best.
 The error from the SPHERE API is still written in the log as usual
 with the more informative and accurate sph_error_code.

 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_ird_science_dbi_run(sph_ird_science_dbi* self) {
    sph_transform* transform = NULL;
    sph_master_frame* result = NULL;
    sph_master_frame* result_adi = NULL;
    cpl_propertylist* plist = NULL;
    cpl_propertylist * plist_left = NULL;
    cpl_propertylist * plist_right= NULL;
    cpl_propertylist* pli   = NULL;
    sph_master_frame* leftFOV = NULL;
    sph_master_frame* rightFOV = NULL;
    cpl_frameset* left_frames_out  = NULL;
    cpl_frameset* right_frames_out = NULL;
    cpl_frameset* total_frames_out = NULL;
    cpl_frameset* speckle_frameset = NULL;
    cpl_frameset* adi_frameset     = NULL;
    sph_ird_common_science* sci    = NULL;
    cpl_vector* angles = NULL;
    cpl_frameset* fctable_frames   = NULL;
    int master_dark_present        = 0;





    pli = cpl_propertylist_load(
                    cpl_frame_get_filename(cpl_frameset_get_first(self->rawframes)),
                    0); // read template property list of first frame to copy singular keys later



    /*------------------------------------------------------------------
     -  selecting the right dark
     --------------------------------------------------------------------*/
    if(self->dark_frame)
        master_dark_present = 1;

    if ((self->skybg_fit_frame || self->skybg_frame || self->insbg_fit_frame || self->insbg_frame) && self->dark_frame)
          cpl_frame_delete(self->dark_frame); // delete dark because it's going to be re-assigne below

    if(self->skybg_fit_frame){
           self->dark_frame = cpl_frame_duplicate(self->skybg_fit_frame);
           cpl_msg_info(cpl_func,"Using SKY_BG_FIT frame as background!");
    }
    else {
        if(self->skybg_frame){
            self->dark_frame = cpl_frame_duplicate(self->skybg_frame);
            cpl_msg_info(cpl_func,"Using SKY_BG frame as background!");
               }
        else {
            if(self->insbg_fit_frame){
                self->dark_frame = cpl_frame_duplicate(self->insbg_fit_frame);
                cpl_msg_info(cpl_func,"Using INS_BG_FIT frame as background!");
            }
               else{
                   if(self->insbg_frame){
                       self->dark_frame = cpl_frame_duplicate(self->insbg_frame);
                       cpl_msg_info(cpl_func,"Using INS_BG frame as background!");
                   }
                   else {
                       if(self->dark_frame){
                           if(master_dark_present)
                               cpl_msg_info(cpl_func,"Using master dark as background!");
                       }
                       else {
                           sph_error_raise( SPH_IRD_SCIENCE_DBI_FRAMES_MISSING,
                                __FILE__, __func__, __LINE__ ,
                                SPH_ERROR_ERROR,
                                                    "No dark or background whatsoever supplied! "
                                                    "Must have one of %s, %s, %s, %s, or %s!",
                                                    SPH_IRD_TAG_INS_BG_FIT_CALIB,
                                                    SPH_IRD_TAG_INS_BG_CALIB,
                                                    SPH_IRD_TAG_SKY_BG_FIT_CALIB,
                                                    SPH_IRD_TAG_SKY_BG_CALIB,
                                                    SPH_IRD_TAG_DARK_CALIB);
                           cpl_ensure_code(0, cpl_error_get_code());

                       }

                   }
               }
        }
    }

    if (self->window_size > 0) {
        if (self->distmap_frame) {
            SPH_ERROR_RAISE_ERR(CPL_ERROR_INCOMPATIBLE_INPUT,
                    "It is not possible to use the window mode at the "
                    "same time as applying a distortion map. Remove"
                    " the distortion map from the input or "
                    " set winow_size = 0.");
            cpl_ensure_code(0, cpl_error_get_code());
        }
    }

    if (self->make_template) {
        return sph_ird_science_dbi_make_template(self);
    }
    if (self->fcasciis != NULL) {
        /* FIXME: This type of input is non-standard and should be removed */
        fctable_frames = cpl_frameset_duplicate(self->fcasciis);
        if (fctable_frames == NULL) {
            goto EXIT;
        }
    } else if (self->fcframe != NULL) {
        fctable_frames = sph_ird_common_science_create_fctables(self->rawframes,
                self->fcframe, self->save_interprod, 1);
    } else {
        sph_error_raise( SPH_IRD_SCIENCE_DBI_FRAMES_MISSING,
                         __FILE__, __func__, __LINE__ ,
                         SPH_ERROR_WARNING,
                         "Could not extract a star center calibration frame."
                         "Since this is an optional frame, this is ok "
                         "and this is just a warning. If you intended, "
                         "to use them check that they have the %s tag.",
                         SPH_IRD_TAG_STAR_CENTER_CALIB);
        cpl_error_reset();
    }
    if (self->flag_adi == 0) {  // If ADI flag is NOT set
        result = sph_ird_science_dbi_process_cubes(self, fctable_frames,
                                                   &leftFOV, &rightFOV,
                                                   &left_frames_out,
                                                   &right_frames_out,
                                                   &total_frames_out);
        if (result) {
            cpl_errorstate okstate = cpl_errorstate_get();
            cpl_image * img_left = sph_master_frame_extract_image(leftFOV, 1);
            cpl_image * img_right = sph_master_frame_extract_image(rightFOV, 1);

            plist = cpl_propertylist_new();
            plist_left = cpl_propertylist_new();
            plist_right = cpl_propertylist_new();
            if (self->filter_frame != NULL &&
                sph_strehl_irdis_and_append(plist_left, plist_right,
                                 img_left, img_right,
                                 cpl_frameset_get_first(self->rawframes),
                                 self->filter_frame,
                                 "science_dbi", self->inparams, NULL, NULL)) {
                cpl_msg_warning(cpl_func, "Strehl estimation failed:");
                cpl_errorstate_dump(okstate, CPL_FALSE,
                        cpl_errorstate_dump_one_warning);
                cpl_errorstate_set(okstate);
            }

            cpl_propertylist_append(plist, plist_left);
            cpl_propertylist_append(plist, plist_right);

            cpl_image_delete(img_left);
            cpl_image_delete(img_right);
            sph_utils_simple_copy_singular(pli, plist);
            sph_ird_common_science_put_wcs(plist,512.0,512.0);

            sph_utils_simple_copy_singular(pli, plist_left);
            sph_ird_common_science_put_wcs(plist_left,512.0,512.0);

            sph_utils_simple_copy_singular(pli, plist_right);
            sph_ird_common_science_put_wcs(plist_right,512.0,512.0);

            sph_master_frame_save_dfs(result, self->science_dbi_outfilename,
                    self->inframes, cpl_frameset_get_first(self->rawframes),
                    self->inparams, SPH_IRD_TAG_SCIENCE_DBI_CALIB,
                    SPH_RECIPE_NAME_IRD_SCIENCE_DBI, SPH_PIPELINE_NAME_IRDIS,
                    plist);
        }
    } else {
        sci = sph_ird_common_science_new(self->inframes, self->inparams,
                self->dark_frame, self->flat_frame, self->static_badpixel_frame,
                self->distmap_frame, self->filter_frame, self->flat_frame,
                fctable_frames,
                SPH_RECIPE_NAME_IRD_SCIENCE_DBI, SPH_IRD_TAG_SCIENCE_DBI_CALIB);

        sci->flag_sdi = self->flag_sdi;
        sci->maxr = self->maxr;
        sci->minr = self->minr;
        if (self->window_size > 0) {
            sph_ird_instrument_model_set_windows(sci->irdmodel,
                    self->window_size);
        }
        if (self->transform_method == 0) {
            if (self->filter_method == 0) {
                transform = sph_transform_new_fft(SPH_FFT_FILTER_METHOD_NONE,
                        0.0, 0.0);
            } else if (self->filter_method == 1) {
                transform = sph_transform_new_fft(SPH_FFT_FILTER_METHOD_TH,
                        self->filter_radius, 0.0);
            } else if (self->filter_method == 2) {
                transform = sph_transform_new_fft(SPH_FFT_FILTER_METHOD_FERMI,
                        self->filter_radius, self->fermi_temp);
            } else if (self->filter_method == 3) {
                transform = sph_transform_new_fft(SPH_FFT_FILTER_METHOD_BUTTER,
                        self->butter_pass, self->butter_stop);
            }
        } else if (self->transform_method == 1) {
            transform = sph_transform_new_cpl_warp();
        }
        if (transform)
            sph_ird_common_science_set_transform(sci, transform);
        sph_ird_common_science_process_cubes(sci, self->rawframes,
                fctable_frames, &left_frames_out, &right_frames_out,
                &total_frames_out,
                                      SPH_IRD_TAG_SCIENCE_DBI_CALIB_LEFT_CUBE,
                                      SPH_IRD_TAG_SCIENCE_DBI_CALIB_RIGHT_CUBE,
                                      SPH_IRD_TAG_SCIENCE_DBI_CALIB_TOTAL_CUBE);
        SPH_ERROR_CHECK_STATE_ONERR_GOTO_EXIT;
        angles = sph_common_science_extract_angles_from_fctables(
                fctable_frames);
        adi_frameset = sph_simple_adi_process_cubes(total_frames_out,
                self->rawframes, angles, sci->transform, sci->irdmodel, 0, 1, 2,
                self->fullset);
        SPH_ERROR_CHECK_STATE_ONERR_GOTO_EXIT;
        if (adi_frameset) {
            result_adi = sph_common_science_combine(adi_frameset,
                    self->coll_alg, 0, 1, 2);
            cpl_frameset_delete(adi_frameset);
            adi_frameset = NULL;
        }
        SPH_ERROR_CHECK_STATE_ONERR_GOTO_EXIT;

        if (result_adi) {
            plist = cpl_propertylist_new();
            sph_utils_simple_copy_singular(pli, plist);
            sph_ird_common_science_put_wcs(plist,512.0,512.0);

            sph_master_frame_save_dfs(result_adi, self->science_dbi_outfilename,
                    self->inframes, cpl_frameset_get_first(self->rawframes),
                    self->inparams, SPH_IRD_TAG_SCIENCE_DBI_CALIB,
                    SPH_RECIPE_NAME_IRD_SCIENCE_DBI, SPH_PIPELINE_NAME_IRDIS,
                    plist);
        }
        SPH_ERROR_CHECK_STATE_ONERR_GOTO_EXIT;
        adi_frameset = sph_simple_adi_process_cubes(left_frames_out,
                self->rawframes, angles, sci->transform, sci->irdmodel, 0, 1, 2,
                self->fullset);
        SPH_ERROR_CHECK_STATE_ONERR_GOTO_EXIT;
        if (adi_frameset) {
            leftFOV = sph_common_science_combine(adi_frameset, self->coll_alg,
                    0, 1, 2);
            cpl_frameset_delete(adi_frameset);
            adi_frameset = NULL;
        }
        SPH_ERROR_CHECK_STATE_ONERR_GOTO_EXIT;
        adi_frameset = sph_simple_adi_process_cubes(right_frames_out,
                self->rawframes, angles, sci->transform, sci->irdmodel, 0, 1, 2,
                self->fullset);
        SPH_ERROR_CHECK_STATE_ONERR_GOTO_EXIT;
        if (adi_frameset) {
            rightFOV = sph_common_science_combine(adi_frameset, self->coll_alg,
                    0, 1, 2);
            cpl_frameset_delete(adi_frameset);
            adi_frameset = NULL;
        }
        SPH_ERROR_CHECK_STATE_ONERR_GOTO_EXIT;

        cpl_vector_delete(angles);
        angles = NULL;
        cpl_frameset_delete(speckle_frameset);
        speckle_frameset = NULL;
        sph_master_frame_delete(result_adi);
        result_adi = NULL;
    }

    if (self->save_interprod) {
        if (total_frames_out != NULL) {
            cpl_frameset_join(self->inframes, total_frames_out);
        }
        if (left_frames_out != NULL) {
            cpl_frameset_join(self->inframes, left_frames_out);
        }
        if (right_frames_out != NULL) {
            cpl_frameset_join(self->inframes, right_frames_out);
        }
    }
    if (self->save_addprod) {
        if (leftFOV) {
            sph_master_frame_save_dfs(leftFOV, self->science_dbi_outfilename_left,
                self->inframes, cpl_frameset_get_first(self->rawframes),
                self->inparams, SPH_IRD_TAG_SCIENCE_DBI_CALIB_LEFT,
                SPH_RECIPE_NAME_IRD_SCIENCE_DBI, SPH_PIPELINE_NAME_IRDIS,
                plist_left);
        }
        if (rightFOV) {
            sph_master_frame_save_dfs(rightFOV, self->science_dbi_outfilename_right,
                self->inframes, cpl_frameset_get_first(self->rawframes),
                self->inparams, SPH_IRD_TAG_SCIENCE_DBI_CALIB_RIGHT,
                SPH_RECIPE_NAME_IRD_SCIENCE_DBI, SPH_PIPELINE_NAME_IRDIS,
                plist_right);
        }
    }
    EXIT: sph_master_frame_delete(result);
    result = NULL;
    cpl_frameset_delete(total_frames_out);
    total_frames_out = NULL;
    cpl_frameset_delete(left_frames_out);
    left_frames_out = NULL;
    cpl_frameset_delete(right_frames_out);
    right_frames_out = NULL;
    cpl_propertylist_delete(plist);
    cpl_propertylist_delete(plist_right);
    cpl_propertylist_delete(plist_left);
    sph_ird_common_science_delete(sci);
    sci = NULL;
    sph_master_frame_delete(leftFOV);
    leftFOV = NULL;
    sph_master_frame_delete(rightFOV);
    rightFOV = NULL;
    sph_filemanager_delete(self->save_interprod);
    cpl_frameset_delete(fctable_frames);
    fctable_frames = NULL;
    cpl_propertylist_delete(pli);
    pli = NULL;
    return (int) cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Process raw frames in DBI mode
 * @param self            ?
 * @param fctable_frames  ?
 * @param left_out        ?
 * @param right_out       ?
 * @param pleft_frameset  Created frameset of intermediate products, left
 * @param pright_frameset Created frameset of intermediate products, right
 * @param ptotal_frameset Created frameset of intermediate products, total
 *
 * @return a new double image or NULL
 *
 * The main IRDIS science reduction in DBI mode.
 * The rawframes are processed and a double image is created that
 * contains the left optical path in the iframe and the right optical
 * path image as the pframe.
 * The propertylist pl is changed on output, so that essential
 * keywords (QC keywords relevant to the reduction)
 * are added to it.
 *
 */
/*----------------------------------------------------------------------------*/
static sph_master_frame*
sph_ird_science_dbi_process_cubes(sph_ird_science_dbi* self,
                                  cpl_frameset* fctable_frames,
                                  sph_master_frame** left_out,
                                  sph_master_frame** right_out,
                                  cpl_frameset** pleft_frameset,
                                  cpl_frameset** pright_frameset,
                                  cpl_frameset** ptotal_frameset) {
    sph_master_frame* result = NULL;
    sph_ird_common_science* sci = NULL;
    sph_transform* transform = NULL;

    if (self->transform_method == 0) {
        if (self->filter_method == 0) {
            transform = sph_transform_new_fft(SPH_FFT_FILTER_METHOD_NONE, 0.0,
                    0.0);
        } else if (self->filter_method == 1) {
            transform = sph_transform_new_fft(SPH_FFT_FILTER_METHOD_TH,
                    self->filter_radius, 0.0);
        } else if (self->filter_method == 2) {
            transform = sph_transform_new_fft(SPH_FFT_FILTER_METHOD_FERMI,
                    self->filter_radius, self->fermi_temp);
        } else if (self->filter_method == 3) {
            transform = sph_transform_new_fft(SPH_FFT_FILTER_METHOD_BUTTER,
                    self->butter_pass, self->butter_stop);
        }
    } else if (self->transform_method == 1) {
        transform = sph_transform_new_cpl_warp();
    }
    sci = sph_ird_common_science_new(self->inframes, self->inparams,
            self->dark_frame, self->flat_frame, self->static_badpixel_frame,
            self->distmap_frame, self->filter_frame, self->flat_frame,
            fctable_frames,
            SPH_RECIPE_NAME_IRD_SCIENCE_DBI, SPH_IRD_TAG_SCIENCE_DBI_CALIB);

    SPH_ERROR_ENSURE_GOTO_EXIT( sci, CPL_ERROR_NULL_INPUT);
    sci->minr = self->minr;
    sci->maxr = self->maxr;
    sci->flag_sdi = self->flag_sdi;
    sci->maxr = self->maxr;
    sci->minr = self->minr;

    if (self->window_size > 0) {
        sph_ird_instrument_model_set_windows(sci->irdmodel, self->window_size);
    }

    if (transform)
        sph_ird_common_science_set_transform(sci, transform);

    sph_ird_common_science_process_cubes(sci, self->rawframes, fctable_frames,
            pleft_frameset, pright_frameset, ptotal_frameset,
                                      SPH_IRD_TAG_SCIENCE_DBI_CALIB_LEFT_CUBE,
                                      SPH_IRD_TAG_SCIENCE_DBI_CALIB_RIGHT_CUBE,
                                      SPH_IRD_TAG_SCIENCE_DBI_CALIB_TOTAL_CUBE);

    SPH_ERROR_CHECK_STATE_ONERR_GOTO_EXIT;
    result = sph_common_science_combine(*ptotal_frameset, self->coll_alg, 0, 1,
                                        -1);
    if (left_out) {
        *left_out = sph_common_science_combine(*pleft_frameset, self->coll_alg,
                                               0, 1, -1);
    }
    if (right_out) {
        *right_out = sph_common_science_combine(*pright_frameset,
                                                self->coll_alg, 0, 1, -1);
    }
EXIT: sph_ird_common_science_delete(sci);
    sci = NULL;
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return result;
}

static cpl_error_code sph_ird_science_dbi_make_template(
        sph_ird_science_dbi* self) {
    cpl_frame* current_frame = NULL;
    sph_fctable* tmptab = NULL;
    char* tmpfilename = NULL;
    int ff = 0;

    current_frame = cpl_frameset_get_first(self->rawframes);
    while (current_frame) {
        tmptab = sph_fctable_create_fctable(current_frame, SPH_FCTAB_DOUBLE);
        if (!tmptab) {
            cpl_ensure_code(0, CPL_ERROR_ILLEGAL_OUTPUT);
        }
        tmpfilename = sph_fctable_construct_filename(
                cpl_frameset_get_position(self->rawframes, ff), 1);

        sph_fctable_save_ascii(tmptab, tmpfilename);
        cpl_free(tmpfilename);
        tmpfilename = NULL;
        current_frame = cpl_frameset_get_next(self->rawframes);
        sph_fctable_delete(tmptab);
        tmptab = NULL;
        ff++;
    }

    return CPL_ERROR_NONE;
}



/**@}*/

