/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <cpl.h>
#include <math.h>
#include "sph_common_keywords.h"
#include "sph_ird_keywords.h"

#include "sph_ird_tags.h"
#include "sph_ird_science_spectroscopy.h"
#include "sph_ird_instrument_model.h"
#include "sph_master_frame.h"
#include "sph_error.h"
#include "sph_keyword_manager.h"
#include "sph_cube.h"
#include "sph_utils.h"
#include "sph_filemanager.h"
#include "sph_ird_common_science.h"
#include "sph_common_science.h"

/*-----------------------------------------------------------------------------
 Error Codes
 -----------------------------------------------------------------------------*/

extern sph_error_code SPH_IRD_SCIENCE_SPECTROSCOPY_GENERAL;
extern sph_error_code SPH_IRD_SCIENCE_SPECTROSCOPY_FRAMES_MISSING;

/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_ird_science_spectroscopy_run Create Master Dark Recipe
 *
 * This module provides the algorithm inplementation for the creation of the
 * master dakr
 *
 * @par Synopsis:
 * @code
 *   #include "sph_ird_science_spectroscopy.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/
/**@{*/

static sph_error_code sph_ird_science_spectroscopy_apply_atm(
        sph_ird_science_spectroscopy* self, sph_master_frame* result,
        sph_master_frame* atmframe) {

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;cpl_ensure_code(self,
            CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(result, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(atmframe, CPL_ERROR_NULL_INPUT);

    sph_master_frame_subtract_master_frame(result, atmframe);

    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Interpret the command line options and execute the data processing
 @param    frameset   the frames list
 @param    parlist    the parameters list

 @return   the cpl error code of the operation.

 This is the main recipe function for the sph_ird_science_spectroscopy recipe. The error
 code returned is always a cpl error code (to allow maximal compatibility with
 esorex, gasgano, etc.) even if during recipe execution an error in the SPHERE
 API is the cause. In this case (and if the underlying error is not a cpl error)
 the cpl error code is set to the cpl_error_code that matches the failure
 reason best.
 The error from the SPHERE API is still written in the log as usual
 with the more informative and accurate sph_error_code.

 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_ird_science_spectroscopy_run(
        sph_ird_science_spectroscopy* self) {
    sph_master_frame* result_left = NULL;
    sph_master_frame* result_right = NULL;
    cpl_propertylist* plist = NULL;
    cpl_propertylist* pli   = NULL;
    sph_ird_instrument_model* model = NULL;
    sph_master_frame* atmframe = NULL;
    sph_ird_common_science* sci = NULL;
    cpl_frameset* left_frames = NULL;
    cpl_frameset* right_frames = NULL;
    char* fname_left = NULL;
    char* fname_right = NULL;
    int master_dark_present = 0;

    pli = cpl_propertylist_load(
    	                    cpl_frame_get_filename(cpl_frameset_get_first(self->rawframes)),
    	                    0); // read template property list of first frame to copy singular keys later

    cpl_ensure(pli,cpl_error_get_code(),0);



    /*------------------------------------------------------------------
     -  selecting the right dark
     --------------------------------------------------------------------*/
    if(self->master_dark_frame)
    	master_dark_present = 1;

    if ((self->skybg_fit_frame || self->skybg_frame || self->insbg_fit_frame || self->insbg_frame) && self->master_dark_frame)
	      cpl_frame_delete(self->master_dark_frame); // delete dark because it's going to be re-assigne below

    if(self->skybg_fit_frame){
       	self->master_dark_frame = cpl_frame_duplicate(self->skybg_fit_frame);
       	cpl_msg_info(cpl_func,"Using SKY_BG_FIT frame as background!");
    }
    else {
    	if(self->skybg_frame){
    		self->master_dark_frame = cpl_frame_duplicate(self->skybg_frame);
    		cpl_msg_info(cpl_func,"Using SKY_BG frame as background!");
           	}
    	else {
    		if(self->insbg_fit_frame){
    			self->master_dark_frame = cpl_frame_duplicate(self->insbg_fit_frame);
    			cpl_msg_info(cpl_func,"Using INS_BG_FIT frame as background!");
    		}
           	else{
           		if(self->insbg_frame){
           			self->master_dark_frame = cpl_frame_duplicate(self->insbg_frame);
           		    cpl_msg_info(cpl_func,"Using INS_BG frame as background!");
           		}
           		else {
           			if(self->master_dark_frame){
           				if(master_dark_present)
           					cpl_msg_info(cpl_func,"Using master dark as background!");
           			}
           			else {
           				sph_error_raise( SPH_IRD_SCIENCE_SPECTROSCOPY_FRAMES_MISSING,
                                __FILE__, __func__, __LINE__ ,
                                SPH_ERROR_ERROR,
           				                         "No dark or background whatsoever supplied! "
           				                         "Must have one of %s, %s, %s, %s, or %s!",
           				                         SPH_IRD_TAG_INS_BG_FIT_CALIB,
           				                         SPH_IRD_TAG_INS_BG_CALIB,
           				                         SPH_IRD_TAG_SKY_BG_FIT_CALIB,
           				                         SPH_IRD_TAG_SKY_BG_CALIB,
           				                         SPH_IRD_TAG_DARK_CALIB);
           				cpl_ensure_code(0, cpl_error_get_code());

           			}

           		}
           	}
    	}
    }





    if (self->atmospheric_frame) {
        atmframe = sph_master_frame_load_(self->atmospheric_frame, 0);
    }

    sci = sph_ird_common_science_new(self->inframes, self->inparams,
            self->master_dark_frame, self->master_flat_frame,
            self->static_badpixel_frame, NULL, NULL, self->master_flat_frame,
            NULL,
            SPH_RECIPE_NAME_IRD_SCIENCE_SPECTROSCOPY,
            SPH_IRD_TAG_SPECTRA_RESOLUTION_CALIB);

    if (sph_ird_common_science_process_cubes(sci, self->rawframes, NULL,
            &left_frames, &right_frames, NULL,
                                             NULL, NULL, NULL) == CPL_ERROR_NONE) {
        result_left = sph_common_science_combine(left_frames, self->coll_alg, 0,
                1, -1);
        result_right = sph_common_science_combine(right_frames, self->coll_alg,
                0, 1, -1);
        fname_left = sph_filemanager_new_filename_from_base(self->outfilename,
                "left");
        fname_right = sph_filemanager_new_filename_from_base(self->outfilename,
                "right");
        if (result_left) {
            plist = sph_ird_instrument_model_get_as_propertylist(model);
            if(!plist){
            	plist=cpl_propertylist_new();
            }
            sph_utils_simple_copy_singular(pli, plist);
            if (atmframe) {
                if (sph_ird_science_spectroscopy_apply_atm(self, result_left,
                        atmframe) != CPL_ERROR_NONE) {
                    SPH_ERROR_RAISE_ERR(SPH_ERROR_GENERAL,
                            "Could not apply the atmospheric calibration");
                }
            }
            sph_master_frame_interpolate_bpix(result_left);
            /* FIXME: Set WCS to dummy (pixel) value for now */
            sph_utils_reset_wcs_12d(plist);
            sph_master_frame_save_dfs(result_left, fname_left, self->inframes,
                    NULL, self->inparams,
                    SPH_IRD_TAG_SCIENCE_SPECTROSCOPY_CALIB_LEFT,
                    SPH_RECIPE_NAME_IRD_SCIENCE_SPECTROSCOPY,
                    SPH_PIPELINE_NAME_IRDIS, plist);

            sph_master_frame_delete(result_left);
            result_left = NULL;
        }
        if (result_right) {
            plist = sph_ird_instrument_model_get_as_propertylist(model);
            if(!plist){
            	plist=cpl_propertylist_new();
            }
            sph_utils_simple_copy_singular(pli, plist);


            if (atmframe) {
                if (sph_ird_science_spectroscopy_apply_atm(self, result_right,
                        atmframe) != CPL_ERROR_NONE) {
                    SPH_ERROR_RAISE_ERR(SPH_ERROR_GENERAL,
                            "Could not apply the atmospheric calibration");
                }
            }
            sph_master_frame_interpolate_bpix(result_right);
            /* FIXME: Set WCS to dummy (pixel) value for now */
            sph_utils_reset_wcs_12d(plist);
            sph_master_frame_save_dfs(result_right, fname_right, self->inframes,
                    NULL, self->inparams,
                    SPH_IRD_TAG_SCIENCE_SPECTROSCOPY_CALIB_RIGHT,
                    SPH_RECIPE_NAME_IRD_SCIENCE_SPECTROSCOPY,
                    SPH_PIPELINE_NAME_IRDIS, plist);

            sph_master_frame_delete(result_right);
            result_right = NULL;
        }
    }

    if (atmframe) {
        sph_master_frame_delete(atmframe);
        atmframe = NULL;
    }
    sph_filemanager_clean();
    if(pli){
    	cpl_propertylist_delete(pli);
    	pli = NULL;
    }
    cpl_propertylist_delete(plist);
    plist = NULL;
    sph_master_frame_delete(result_left);
    result_left = NULL;
    sph_master_frame_delete(result_right);
    result_right = NULL;
    sph_ird_instrument_model_delete(model);
    model = NULL;
    sph_ird_common_science_delete(sci);
    cpl_frameset_delete(left_frames);
    cpl_frameset_delete(right_frames);
    cpl_free(fname_left);
    cpl_free(fname_right);
    return (int) cpl_error_get_code();
}

/**@}*/
