/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <libgen.h>
#include <cpl.h>
#include <math.h>
#include "sph_zpl_science_p1.h"
#include "sph_zpl_common_preproc.h"
#include "sph_zpl_exposure.h"
#include "sph_zpl_keywords.h"
#include "sph_zpl_tags.h"
#include "sph_zpl_utils.h"
#include "sph_zpl_framegroups.h"
#include "sph_quad_image.h"
#include "sph_triple_image.h"
#include "sph_double_image.h"
#include "sph_master_frame.h"
#include "sph_keywordnode.h"
#include "sph_error.h"
#include "sph_utils.h"
#include "sph_fits.h"
#include "sph_common_keywords.h"
#include "sph_common_science.h"
#include "sph_filemanager.h"
#include "sph_transform.h"
#include "sph_zpl_science_pol_product.h"
#include "sph_strehl.h"
#include "sph_zpl_science_pol_utils.h"
#include "sph_zpl_subtract_dark_scaled.h"

/*-----------------------------------------------------------------------------
 Error Codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_zpl_science_p1_run Reduce science frames of the Q and U observations in the polarization P1 mode
 *
 * This module provides the algorithm implementation for the creation of the
 * reduced polarimetric science observations (in P1 mode) and the creation of the corresponding
 * mueller matrix elements (not implemented yet)
 *
 * @par Synopsis:
 * @code
 *   #include "sph_zpl_science_p1_run.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/
/**@{*/

#define  SPH_ZPL_STOKES_Q					0
#define  SPH_ZPL_STOKES_U					1
#define  SPH_ZPL_STOKES_NONE				2

#define SPH_SAVE_NONDFS                     0



/*-------------------------------------------------------------------------------
 * The Structure Definition
 *
 * This structure contains the members of the sph_zpl_science_p1_cunit that are
 * to reduce data for the one zimpol channel (camera). This structure is used as
 * an interface between a sphere "standard" recipe structure and re-designed
 * zimpol "_run" function which treats both zimpol channel.
 *
 * ------------------------------------------------------------------------------
 */
typedef struct _sph_zpl_science_p1_cunit {
    cpl_frameset* inframes; /* The recipe input frames */

    cpl_frameset* current_raw_frameset; /* The frameset for one product */

    cpl_parameterlist* inparams; /* The recipe input parameters */

    cpl_parameterlist* framecomb_parameterlist; /* The recipe input parameters */

    char* science_p1_outfilename; /* The parameter of zpl.science_p1.outfilename_q_cam1 */

    char* science_p1_plus_outfilename; /* The parameter of zpl.science_p1_plus_q_cam1.outfilename */

    char* science_p1_minus_outfilename; /* The parameter of zpl.science_p1_minus_q_cam1.outfilename */

    short subtract_overscan; /* The parameter of zpl.science_p1.subtract_overscan */

    short save_interprod; /* The parameter of zpl.science_p1.save_interprod */

    short keep_intermediate; /* The parameter of zpl.science_p1.keep_intermediate */

    int coll_alg; /* The parameter of zpl.science_p1.coll_alg */

    double filter_radius; /* The parameter of zpl.science_p1.filter_radius */

    double center_xoffset; /* The parameter of zpl.science_p1.center_xoffset_cam1 or _cam1 */

    double center_yoffset; /* The parameter of zpl.science_p1.center_yoffset_cam1 or cam2 */

    short star_center_iframe; /* The parameter of zpl.science_p1.star_center_iframe */

    cpl_frameset* rawframes; /* The ZPL_SCIENCE_P1_RAW frames */

    cpl_frameset* preproc_frames; /* The ZPL_SCIENCE_P1_PREPROC_CAM1 frames */

    cpl_frame* master_bias_frame; /* The ZPL_MASTER_BIAS_CAM1 frames */

    cpl_frame* master_dark_frame; /* The ZPL_MASTER_DARK_CAM1 frames */

    cpl_frame* intensity_flat_frame; /* The ZPL_INT_FLAT_FIELD_CAM1 frames */

    cpl_frame* intensity_flat_frame_master; /* The ZPL_INT_FLAT_FIELD_MASTER_CAM1 frames */

    cpl_frame* polarization_flat_frame; /* The ZPL_POL_FLAT_FIELD_CAM1 frames */

    cpl_frame* modem_efficiency_frame; /* SPH_ZPL_TAG_MODEM_EFF_CALIB_CAM1 or _CAM2 */

    cpl_frameset* fctable_frames; /* The ZPL_CENTER_TABLE frames */

    cpl_frame*  star_center_frame; /* The ZPL_STAR_CENTER_POL frames */

    sph_zpl_science_pol_product * science_di; /* The ZPL_SCIENCE_P1_REDUCED product */

    sph_zpl_science_pol_product * science_di_plus; /* The ZPL_SCIENCE_P1_PLUS_REDUCED product */

    sph_zpl_science_pol_product * science_di_minus; /* The ZPL_SCIENCE_P1_MINUS_REDUCED product */

    char* eso_pro_catg;

    char* eso_pro_catg_plus;

    char* eso_pro_catg_minus;

    sph_zpl_framegroups* fgs;

    sph_zpl_framegroups*  fgsraw;

//  sph_triple_image*               science_ti                            ; /* The ZPL_SCIENCE_XMATRIX_ELEMENTS product */

} sph_zpl_science_p1_cunit;

/*-----------------------------------------------------------------------------
 Private function prototypes
 -----------------------------------------------------------------------------*/
static sph_zpl_science_p1_cunit* _sph_zpl_science_p1_cunit_create(
        sph_zpl_science_p1* self, int camera, short int stoke);
static sph_error_code _sph_zpl_science_p1_cunit_delete(
        sph_zpl_science_p1_cunit* cunit);
static cpl_error_code _sph_zpl_science_p1_cunit_run(
        sph_zpl_science_p1_cunit* cunit);

static cpl_error_code save_products_with_qc_pars(sph_zpl_science_p1 * rec);

//static cpl_error_code _sph_zpl_science_p1_create_fctable_frames ()

/*----------------------------------------------------------------------------*/
/**
 @brief    Interpret the command line options and execute the data processing
 @param    frameset   the frames list
 @param    parlist    the parameters list

 @return   the cpl error code of the operation.

 This is the main recipe function for the sph_zpl_science_p1 recipe
 (polarimetric mode). The error  code returned is always a cpl error code
 (to allow maximal compatibility with  esorex, gasgano, etc.) even if during
 recipe execution an error in the SPHERE API is the cause. In this case
 (and if the underlying error is not a cpl error)  the cpl error code is set
 to the cpl_error_code that matches the failure reason best.
 The error from the SPHERE API is still written in the log as usual
 with the more informative and accurate sph_error_code.

 */
/*----------------------------------------------------------------------------*/

cpl_error_code sph_zpl_science_p1_run(sph_zpl_science_p1* self) {
    sph_zpl_science_p1_cunit* cunit_q = NULL;
    sph_zpl_science_p1_cunit* cunit_u = NULL;
    cpl_error_code recipe_error = CPL_ERROR_NONE;

    SPH_INFO_MSG("Starting sph_zpl_science_p1_run...");
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE
    if (self == NULL) {
        sph_error_raise(CPL_ERROR_NULL_INPUT, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Null input pointer.");
        return cpl_error_get_code();
    }

    if (self->preproc_frames_cam1 == NULL && self->preproc_frames_cam2 == NULL && self->preproc_frames == NULL) {

        if (self->rawframes) {
            self->preproc_frames_cam1 = cpl_frameset_new();
            self->preproc_frames_cam2 = cpl_frameset_new();
            SPH_INFO_MSG("Start pre-processing...")
            //recipe_error = sph_zpl_master_dark_preproc( self );
            recipe_error = sph_zpl_common_preproc_recipe(self->rawframes,
                    self->outfilename_cam1, self->outfilename_cam2,
                    self->preproc_frames_cam1, self->preproc_frames_cam2);
        } else {
            SPH_ERR(
                    "There are neither raw frames nor pre-processed ones! Please verify that the tags are set correctly!");
            return sph_error_get_last_code();
        }
        if (recipe_error) {
            SPH_ERR("Pre-processing step is failed.");
            cpl_frameset_delete(self->preproc_frames_cam1); self->preproc_frames_cam1 = NULL;
            cpl_frameset_delete(self->preproc_frames_cam2); self->preproc_frames_cam2 = NULL;
            return (int) recipe_error;
        }


        if ( sph_zpl_utils_check_format( self->rawframes ) == 1) {
            SPH_INFO_MSG("New style format of the raw data...")
            if ( cpl_frameset_is_empty( self->preproc_frames_cam1 ) && !cpl_frameset_is_empty( self->preproc_frames_cam2 ) ) {
                self->preproc_frames = cpl_frameset_duplicate( self->preproc_frames_cam2 );
                cpl_frameset_delete( self-> preproc_frames_cam1 ); self->preproc_frames_cam1 = NULL;
                cpl_frameset_delete( self-> preproc_frames_cam2 ); self->preproc_frames_cam2 = NULL;
            } else if ( !cpl_frameset_is_empty( self->preproc_frames_cam1 ) && cpl_frameset_is_empty( self->preproc_frames_cam2 )){
                self->preproc_frames = cpl_frameset_duplicate( self->preproc_frames_cam1 );
                cpl_frameset_delete( self-> preproc_frames_cam1 ); self->preproc_frames_cam1 = NULL;
                cpl_frameset_delete( self-> preproc_frames_cam2 ); self->preproc_frames_cam2 = NULL;
            } else if ( !cpl_frameset_is_empty( self->preproc_frames_cam1 ) && !cpl_frameset_is_empty( self->preproc_frames_cam2 )){
                SPH_WARNING("Rawdata set contains fits files from camera-1 and camera-2. Please, make sure that you provide calibrations for both cameras."
                        "The products will be created individually for each camera (off-line pipeline extended support!")
            } else {
                SPH_ERR("Pre-processing step is failed.");
                cpl_frameset_delete(self->preproc_frames_cam1); self->preproc_frames_cam1 = NULL;
                cpl_frameset_delete(self->preproc_frames_cam2); self->preproc_frames_cam2 = NULL;
                return (int) recipe_error;
            }
        } else if ( sph_zpl_utils_check_format( self->rawframes ) == 0) {
            SPH_INFO_MSG("Old style format of the raw data...")
        } else {
            SPH_ERR("Not supported raw data format or new format and old format are mixed in the input dataset. "
                    "Please, make sure that you use either old or new style of the rawdata format! Stop recipe...");
            cpl_frameset_delete(self->preproc_frames_cam1); self->preproc_frames_cam1 = NULL;
            cpl_frameset_delete(self->preproc_frames_cam2); self->preproc_frames_cam2 = NULL;
            return (int) recipe_error;
        }
    }
    sph_error_reset();
    SPH_RAISE_CPL_RESET; //reset of the error system after pre-processing

#ifndef SPH_SAVE_NONDFS
    if (cpl_fits_get_mode() != CPL_FITS_START_CACHING) {
        /* Work-around that happens to fix the temporary file bug (PIPE-6794) */
        cpl_errorstate prestate = cpl_errorstate_get();
        if (cpl_fits_set_mode(CPL_FITS_START_CACHING)) {
            cpl_msg_warning(cpl_func, "Could not start CPL FITS caching");
            cpl_errorstate_dump(prestate, CPL_FALSE, NULL);
            cpl_errorstate_set(prestate);
        } else {
            cpl_msg_info(cpl_func, "Starting CPL FITS caching");
        }
    }
#endif

    SPH_INFO_MSG("*********** [AFTER PREPROC] ***********");
    //reduce data for the camera-1
    //create cunit structure
    if (self->preproc_frames_cam1) {
        SPH_INFO_MSG("Create cunit structures for the camera-1...");

        SPH_INFO_MSG("Camera-1 stokes Q...")
        cunit_q = _sph_zpl_science_p1_cunit_create(self,
                SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID, SPH_ZPL_STOKES_Q);
        if (cunit_q) {

            recipe_error = _sph_zpl_science_p1_cunit_run(cunit_q);
            if (recipe_error != CPL_ERROR_NONE) {
                SPH_ERR(
                        "reduction for the camera-1 for the stokes Q is failed!");
            } else {
                //put the output product in the main self structure
                if (cunit_q->science_di)
                    self->science_di_q_cam1 = sph_zpl_science_pol_product_duplicate(
                            cunit_q->science_di);
                if (cunit_q->science_di_minus)
                    self->science_di_q_minus_cam1 = sph_zpl_science_pol_product_duplicate(
                            cunit_q->science_di_minus);
                if (cunit_q->science_di_plus)
                    self->science_di_q_plus_cam1 = sph_zpl_science_pol_product_duplicate(
                            cunit_q->science_di_plus);
            }
        } else {
            SPH_WARNING(
                    "Creation of the cunit_q structure for the camera-1 (stokes Q) is failed...trying for the stokes U")
        }

        SPH_INFO_MSG("Camera-1 stokes U...")
        //reset an error log system to reduce data for the U frames of the camera-1
        sph_error_reset();
        SPH_RAISE_CPL_RESET;
        cunit_u = _sph_zpl_science_p1_cunit_create(self,
                SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID, SPH_ZPL_STOKES_U);
        if (cunit_u) {

            recipe_error = _sph_zpl_science_p1_cunit_run(cunit_u);
            if (recipe_error != CPL_ERROR_NONE) {
                SPH_ERR(
                        "reduction for the camera-1 for the stokes U is failed!");
            } else {
                //put the output product in the main self structure
                if (cunit_u->science_di)
                    self->science_di_u_cam1 = sph_zpl_science_pol_product_duplicate(
                            cunit_u->science_di);
                if (cunit_u->science_di_minus)
                    self->science_di_u_minus_cam1 = sph_zpl_science_pol_product_duplicate(
                            cunit_u->science_di_minus);
                if (cunit_u->science_di_plus)
                    self->science_di_u_plus_cam1 = sph_zpl_science_pol_product_duplicate(
                            cunit_u->science_di_plus);
            }
        } else {
            SPH_WARNING(
                    "Creation of the cunit_u structure for the camera-1 (stokes U) is failed!")
        }

        if (cunit_u == NULL && cunit_q == NULL) {
            SPH_INFO_MSG(
                    "I will try to create cunit, considering all input frames from the camera-1 as Qplus frames...");
            //reset an error log system to set up all frames as Q frames of the camera-1
            sph_error_reset();
            SPH_RAISE_CPL_RESET;
            cunit_q = _sph_zpl_science_p1_cunit_create(self,
                    SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID, SPH_ZPL_STOKES_NONE);

            if (cunit_q) {

                recipe_error = _sph_zpl_science_p1_cunit_run(cunit_q);
                if (recipe_error != CPL_ERROR_NONE) {
                    SPH_ERR(
                            "reduction for the camera-1 for the stokes Q is failed anyway!");
                } else {
                    //put the output product in the main self structure
                    if (cunit_q->science_di)
                        self->science_di_q_cam1 = sph_zpl_science_pol_product_duplicate(
                                cunit_q->science_di);
                    if (cunit_q->science_di_minus)
                        self->science_di_q_minus_cam1 =
                        		sph_zpl_science_pol_product_duplicate(
                                        cunit_q->science_di_minus);
                    if (cunit_q->science_di_plus)
                        self->science_di_q_plus_cam1 =
                        		sph_zpl_science_pol_product_duplicate(
                                        cunit_q->science_di_plus);
                }
            } else {
                SPH_WARNING(
                        "Creation of the cunit_q structure for the camera-1 (stokes Q) is failed anyway...trying camera-2")
            }
        }

    } else {
        SPH_WARNING(
                "No pre-processed data found for the camera-1 -> nothing to reduce..trying for the camera-2");
    }
    if (cunit_q) {
        _sph_zpl_science_p1_cunit_delete(cunit_q);
        cunit_q = NULL;
    }
    if (cunit_u) {
        _sph_zpl_science_p1_cunit_delete(cunit_u);
        cunit_u = NULL;
    }

    //reset an error log system to reduce data for camera-2
    sph_error_reset();
    SPH_RAISE_CPL_RESET;

    //reduce data for the camera-2
    //create cunit structure
    if (self->preproc_frames_cam2) {
        SPH_INFO_MSG("Create cunit structure for the camera-2...");

        SPH_INFO_MSG("Camera-2 stokes Q...")
        cunit_q = _sph_zpl_science_p1_cunit_create(self,
                SPH_ZPL_KEYWORD_VALUE_CAMERA2_ID, SPH_ZPL_STOKES_Q);

        if (cunit_q) {
            recipe_error = _sph_zpl_science_p1_cunit_run(cunit_q);
            if (recipe_error != CPL_ERROR_NONE) {
                SPH_ERR(
                        "reduction for the camera-2 is for the stokes Q is failed!");
            } else {
                //put the output product in the main self structure
                if (cunit_q->science_di)
                    self->science_di_q_cam2 = sph_zpl_science_pol_product_duplicate(
                            cunit_q->science_di);
                if (cunit_q->science_di_minus)
                    self->science_di_q_minus_cam2 = sph_zpl_science_pol_product_duplicate(
                            cunit_q->science_di_minus);
                if (cunit_q->science_di_plus)
                    self->science_di_q_plus_cam2 = sph_zpl_science_pol_product_duplicate(
                            cunit_q->science_di_plus);

            }
        } else {
            SPH_WARNING(
                    "Creation of the cunit_q structure for the camera-2 (stokes Q) is failed...trying for the stokes U")
        }

        SPH_INFO_MSG("Camera-2 stokes U...")
        //reset an error log system to reduce data for U frames of the camera-2
        sph_error_reset();
        SPH_RAISE_CPL_RESET;
        cunit_u = _sph_zpl_science_p1_cunit_create(self,
                SPH_ZPL_KEYWORD_VALUE_CAMERA2_ID, SPH_ZPL_STOKES_U);

        if (cunit_u) {
            recipe_error = _sph_zpl_science_p1_cunit_run(cunit_u);
            if (recipe_error != CPL_ERROR_NONE) {
                SPH_ERR(
                        "reduction for the camera-2 is for the stokes U is failed!");
            } else {
                //put the output product in the main self structure
                if (cunit_u->science_di)
                    self->science_di_u_cam2 = sph_zpl_science_pol_product_duplicate(
                            cunit_u->science_di);
                if (cunit_u->science_di_minus)
                    self->science_di_u_minus_cam2 = sph_zpl_science_pol_product_duplicate(
                            cunit_u->science_di_minus);
                if (cunit_u->science_di_plus)
                    self->science_di_u_plus_cam2 = sph_zpl_science_pol_product_duplicate(
                            cunit_u->science_di_plus);
            }
        } else {
            SPH_WARNING(
                    "Creation of the cunit_u structure for the camera-2 (stokes U) is failed!")
        }

        if (cunit_q == NULL && cunit_u == NULL) {
            SPH_INFO_MSG(
                    "I will try to create cunit, considering all input frames from the camera-2 as Qplus frames...");
            //reset an error log system to set up all frames as Q frames of the camera-2
            sph_error_reset();
            SPH_RAISE_CPL_RESET;

            cunit_q = _sph_zpl_science_p1_cunit_create(self,
                    SPH_ZPL_KEYWORD_VALUE_CAMERA2_ID, SPH_ZPL_STOKES_NONE);
            if (cunit_q) {

                recipe_error = _sph_zpl_science_p1_cunit_run(cunit_q);
                if (recipe_error != CPL_ERROR_NONE) {
                    SPH_ERR(
                            "reduction for the camera-2 for the stokes Q is failed anyway!");
                } else {
                    //put the output product in the main self structure
                    if (cunit_q->science_di)
                        self->science_di_q_cam2 = sph_zpl_science_pol_product_duplicate(
                                cunit_q->science_di);
                    if (cunit_q->science_di_minus)
                        self->science_di_q_minus_cam2 =
                        		sph_zpl_science_pol_product_duplicate(
                                        cunit_q->science_di_minus);
                    if (cunit_q->science_di_plus)
                        self->science_di_q_plus_cam2 =
                        		sph_zpl_science_pol_product_duplicate(
                                        cunit_q->science_di_plus);
                }
            } else {
                SPH_WARNING(
                        "Creation of the cunit_q structure for the camera-1 (stokes Q) is failed anyway...trying camera-2")
            }
        }

    } else {
        SPH_WARNING(
                "No pre-processed data found for the camera-2 -> nothing to reduce.");
    }

    if (cunit_q) {
        _sph_zpl_science_p1_cunit_delete(cunit_q);
        cunit_q = NULL;
    }
    if (cunit_u) {
        _sph_zpl_science_p1_cunit_delete(cunit_u);
        cunit_u = NULL;
    }



    //reset the error log system to avoid a recipe abort in the case if some products were not created
    //for the new style format data (indifferent to the cameras))
    sph_error_reset();
    SPH_RAISE_CPL_RESET;

    if (self->preproc_frames) {
        SPH_INFO_MSG("Create cunit structure for the new style format...");

        SPH_INFO_MSG("Indifferent camera stokes Q...")
        cunit_q = _sph_zpl_science_p1_cunit_create(self,
                SPH_ZPL_KEYWORD_VALUE_CAMERA_INDEFERENT_ID, SPH_ZPL_STOKES_Q);
        if (cunit_q) {

            recipe_error = _sph_zpl_science_p1_cunit_run(cunit_q);
            if (recipe_error != CPL_ERROR_NONE) {
                SPH_ERR(
                        "reduction for the newstyle pre-processed data stokes Q is failed!");
            } else {
                //put the output product in the main self structure
                if (cunit_q->science_di)
                    self->science_di_q = sph_zpl_science_pol_product_duplicate(
                            cunit_q->science_di);
                if (cunit_q->science_di_minus)
                    self->science_di_q_minus = sph_zpl_science_pol_product_duplicate(
                            cunit_q->science_di_minus);
                if (cunit_q->science_di_plus)
                    self->science_di_q_plus = sph_zpl_science_pol_product_duplicate(
                            cunit_q->science_di_plus);
            }
        } else {
            SPH_WARNING(
                    "Creation of the cunit_q structure for the new style format (stokes Q) is failed...trying for the stokes U")
        }

        SPH_INFO_MSG("Indifferent camera  stokes U...")
        //reset an error log system to reduce data for the U frames of the camera-1
        sph_error_reset();
        SPH_RAISE_CPL_RESET;
        cunit_u = _sph_zpl_science_p1_cunit_create(self,
                SPH_ZPL_KEYWORD_VALUE_CAMERA_INDEFERENT_ID, SPH_ZPL_STOKES_U);
        if (cunit_u) {

            recipe_error = _sph_zpl_science_p1_cunit_run(cunit_u);
            if (recipe_error != CPL_ERROR_NONE) {
                SPH_ERR(
                        "reduction for the newstyle pre-processed data stokes U is failed!");
            } else {
                //put the output product in the main self structure
                if (cunit_u->science_di)
                    self->science_di_u = sph_zpl_science_pol_product_duplicate(
                            cunit_u->science_di);
                if (cunit_u->science_di_minus)
                    self->science_di_u_minus = sph_zpl_science_pol_product_duplicate(
                            cunit_u->science_di_minus);
                if (cunit_u->science_di_plus)
                    self->science_di_u_plus = sph_zpl_science_pol_product_duplicate(
                            cunit_u->science_di_plus);
            }
        } else {
            SPH_WARNING(
                    "Creation of the cunit_u structure for the newstyle format (stokes U) is failed!")
        }

        if (cunit_u == NULL && cunit_q == NULL) {
            SPH_INFO_MSG(
                    "I will try to create cunit, considering all input frames from the camera-1 as Qplus frames...");
            //reset an error log system to set up all frames as Q frames of the camera-1
            sph_error_reset();
            SPH_RAISE_CPL_RESET;
            cunit_q = _sph_zpl_science_p1_cunit_create(self,
                    SPH_ZPL_KEYWORD_VALUE_CAMERA_INDEFERENT_ID, SPH_ZPL_STOKES_NONE);

            if (cunit_q) {

                recipe_error = _sph_zpl_science_p1_cunit_run(cunit_q);
                if (recipe_error != CPL_ERROR_NONE) {
                    SPH_ERR(
                            "reduction for the  the newstyle pre-processed data stokes Q is failed anyway!");
                } else {
                    //put the output product in the main self structure
                    if (cunit_q->science_di)
                        self->science_di_q = sph_zpl_science_pol_product_duplicate(
                                cunit_q->science_di);
                    if (cunit_q->science_di_minus)
                        self->science_di_q_minus =
                        		sph_zpl_science_pol_product_duplicate(
                                        cunit_q->science_di_minus);
                    if (cunit_q->science_di_plus)
                        self->science_di_q_plus =
                        		sph_zpl_science_pol_product_duplicate(
                                        cunit_q->science_di_plus);
                }
            } else {
                SPH_WARNING(
                        "Creation of the cunit_q structure for the newstyle format data (stokes Q) is failed anyway...trying camera-2")
            }
        }

    } else {
        SPH_WARNING(
                "No pre-processed data found for the new style format -> nothing to reduce.");
    }

    const cpl_error_code err = save_products_with_qc_pars(self);

    if(err){

        sph_error_raise(
                SPH_ERROR_GENERAL,
                __FILE__,
                __func__,
                __LINE__,
                SPH_ERROR_ERROR,
                "Couldn't save science products\n"
                        "CPL error code is: %d", err);
    }

    if (cunit_q) {
        _sph_zpl_science_p1_cunit_delete(cunit_q);
        cunit_q = NULL;
    }
    if (cunit_u) {
        _sph_zpl_science_p1_cunit_delete(cunit_u);
        cunit_u = NULL;
    }

    //reset an error log system to reduce data for camera-2
    sph_error_reset();
    SPH_RAISE_CPL_RESET;

    //post-condistions
    if ((self->science_di_q_cam1 == NULL && self->science_di_q_cam2 == NULL
            && self->science_di_q_minus_cam1 == NULL
            && self->science_di_q_minus_cam2 == NULL
            && self->science_di_q_plus_cam1 == NULL
            && self->science_di_q_plus_cam2 == NULL
            && self->science_di_u_cam1 == NULL && self->science_di_u_cam2
            && self->science_di_u_minus_cam1 == NULL
            && self->science_di_u_minus_cam2
            && self->science_di_u_plus_cam1 == NULL
            && self->science_di_u_plus_cam2)
            && self->science_di_q == NULL
            && self->science_di_q_minus == NULL
            && self->science_di_q_plus == NULL
            && self->science_di_u == NULL
            && self->science_di_u_minus
            && self->science_di_u_plus ) {
        SPH_ERR(" No reduced frame outputs have been created !!!");
    }


    if (!self->keep_intermediate && self->rawframes != NULL) {
        SPH_INFO_MSG("Unliking intermediate data!");
        if (self->preproc_frames_cam1) sph_utils_frames_unlink(self->preproc_frames_cam1);
        if (self->preproc_frames_cam2) sph_utils_frames_unlink(self->preproc_frames_cam2);
        if (self->preproc_frames) sph_utils_frames_unlink(self->preproc_frames);
    }


    SPH_INFO_MSG("sph_zpl_science_p1_run...End");
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
}

sph_zpl_science_p1_cunit* _sph_zpl_science_p1_cunit_create(
        sph_zpl_science_p1* self, int camera_id, short int stoke) {
    sph_zpl_science_p1_cunit* result = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    if (self == NULL) {
        sph_error_raise(CPL_ERROR_NULL_INPUT, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Null input pointer.");
        return NULL;
    }

    result = cpl_calloc(1, sizeof(sph_zpl_science_p1_cunit));

    if (result == NULL) {
        SPH_ERR( "Could not allocate the structure for run unit.");
        return result;
    }

    //copy and duplicate some elements of the initial "sph_zpl_science_p1" structure into the cunit structure
    result->inframes = self->inframes;
    result->inparams = self->inparams;
    result->framecomb_parameterlist = self->framecomb_parameterlist;
    result->subtract_overscan = self->subtract_overscan;
    result->save_interprod = self->save_interprod;
    result->keep_intermediate = self->keep_intermediate;
    result->coll_alg = self->coll_alg;
    result->star_center_iframe = self->star_center_iframe;


    result->rawframes = self->rawframes;
    result->fctable_frames = self->fctable_frames;

    if (camera_id == SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID) {

        result->center_xoffset = self->center_xoffset_cam1;
        result->center_yoffset = self->center_yoffset_cam1;

        result->preproc_frames = cpl_frameset_duplicate(
                self->preproc_frames_cam1);
        if (stoke == SPH_ZPL_STOKES_Q) {
            result->fgs = sph_zpl_framegroups_create_Q(result->preproc_frames);
            if (!result->fgs) {
                SPH_WARNING(
                        "Neither Qplus nor Qminus frames found, sph_zpl_framegroups_create returns NULL!");
                _sph_zpl_science_p1_cunit_delete(result);
                result = NULL;
                return result;
            } else {
                SPH_INFO_MSG("CAM1 Q stokes: fgs is not NULL");
                result->science_p1_outfilename = cpl_strdup(
                        self->science_p1_outfilename_q_cam1);
                result->science_p1_plus_outfilename = cpl_strdup(
                        self->science_p1_plus_q_outfilename_cam1);
                result->science_p1_minus_outfilename = cpl_strdup(
                        self->science_p1_minus_q_outfilename_cam1);
                result->eso_pro_catg = cpl_strdup(
                        SPH_ZPL_TAG_SCIENCE_P1_Q_CALIB_CAM1); //TAG for the double image science product CAM1
                result->eso_pro_catg_plus = cpl_strdup(
                        SPH_ZPL_TAG_SCIENCE_P1_QPLUS_CALIB_CAM1); //TAG for the double image science plus product CAM1
                result->eso_pro_catg_minus = cpl_strdup(
                        SPH_ZPL_TAG_SCIENCE_P1_QMINUS_CALIB_CAM1); //TAG for the double image science plus product CAM1

                //group raw frames on QPLUS and QMINUS, which is needed for save_dfs
                result->fgsraw = sph_zpl_framegroups_create_Q(self->rawframes);
                if (!result->fgsraw) {
                     SPH_WARNING(
                             "Neither Qplus nor Qminus frames found, sph_zpl_framegroups_create_Q raw returns NULL!");
                     _sph_zpl_science_p1_cunit_delete(result);
                     result = NULL;
                     return result;
                }
            }
        } else if (stoke == SPH_ZPL_STOKES_U) {
            result->fgs = sph_zpl_framegroups_create_U(result->preproc_frames);
            if (!result->fgs) {
                SPH_WARNING(
                        "Neither Uplus nor Uminus frames found, sph_zpl_framegroups_create returns NULL!");
                _sph_zpl_science_p1_cunit_delete(result);
                result = NULL;
                return result;
            } else {
                SPH_INFO_MSG("CAM1 U stokes: fgs is not NULL");
                result->science_p1_outfilename = cpl_strdup(
                        self->science_p1_outfilename_u_cam1);
                result->science_p1_plus_outfilename = cpl_strdup(
                        self->science_p1_plus_u_outfilename_cam1);
                result->science_p1_minus_outfilename = cpl_strdup(
                        self->science_p1_minus_u_outfilename_cam1);
                result->eso_pro_catg = cpl_strdup(
                        SPH_ZPL_TAG_SCIENCE_P1_U_CALIB_CAM1); //TAG for the double image science product CAM1
                result->eso_pro_catg_plus = cpl_strdup(
                        SPH_ZPL_TAG_SCIENCE_P1_UPLUS_CALIB_CAM1); //TAG for the double image science plus product CAM1
                result->eso_pro_catg_minus = cpl_strdup(
                        SPH_ZPL_TAG_SCIENCE_P1_UMINUS_CALIB_CAM1); //TAG for the double image science plus product CAM1

                //group raw frames on UPLUS and UMINUS, which is needed for save_dfs
                result->fgsraw = sph_zpl_framegroups_create_U(self->rawframes);
                if (!result->fgsraw) {
                     SPH_WARNING(
                             "Neither Uplus nor Uminus frames found, sph_zpl_framegroups_create_U raw returns NULL!");
                     _sph_zpl_science_p1_cunit_delete(result);
                     result = NULL;
                     return result;
                }

            }
        } else { //set all to the Q stokes (usually if stoke == SPH_ZPL_STOKES_NONE )
            SPH_INFO_MSG("CAM1: Setting all frames as Qplus frames.")
            result->fgs = sph_zpl_framegroups_new();
            rerr = sph_zpl_framegroups_set_framesplus(result->fgs,
                    self->preproc_frames_cam1);
            if (rerr != CPL_ERROR_NONE) {
                SPH_ERR(
                        "Can't set plus frames into the framegroups, sph_zpl_framegroups_frames returns NULL!");
                _sph_zpl_science_p1_cunit_delete(result);
                result = NULL;
                return result;
            }
            result->science_p1_outfilename = cpl_strdup(
                    self->science_p1_outfilename_q_cam1);
            result->science_p1_plus_outfilename = cpl_strdup(
                    self->science_p1_plus_q_outfilename_cam1);
            result->science_p1_minus_outfilename = cpl_strdup(
                    self->science_p1_minus_q_outfilename_cam1);
            result->eso_pro_catg = cpl_strdup(
                    SPH_ZPL_TAG_SCIENCE_P1_Q_CALIB_CAM1); //TAG for the double image science product CAM1
            result->eso_pro_catg_plus = cpl_strdup(
                    SPH_ZPL_TAG_SCIENCE_P1_QPLUS_CALIB_CAM1); //TAG for the double image science plus product CAM1
            result->eso_pro_catg_minus = cpl_strdup(
                    SPH_ZPL_TAG_SCIENCE_P1_QMINUS_CALIB_CAM1); //TAG for the double image science plus product CAM1

            //group raw frames on QPLUS and QMINUS, which is needed for save_dfs
            result->fgsraw = sph_zpl_framegroups_create_Q(self->rawframes);
            if (!result->fgsraw) {
                 SPH_WARNING(
                         "Neither Qplus nor Qminus frames found, sph_zpl_framegroups_create_Q raw returns NULL!");
                 _sph_zpl_science_p1_cunit_delete(result);
                 result = NULL;
                 return result;
            }
        }

        if (self->master_bias_frame_cam1) {
            result->master_bias_frame = cpl_frame_duplicate(
                    self->master_bias_frame_cam1);
        }
        if (self->master_dark_frame_cam1) {
            result->master_dark_frame = cpl_frame_duplicate(
                    self->master_dark_frame_cam1);
        }
        if (self->intensity_flat_frame_cam1) {
            result->intensity_flat_frame = cpl_frame_duplicate(
                    self->intensity_flat_frame_cam1);
        }
        if (self->intensity_flat_frame_master_cam1) {
            result->intensity_flat_frame_master = cpl_frame_duplicate(
                    self->intensity_flat_frame_master_cam1);
        }
        if (self->polarization_flat_frame_cam1) {
            result->polarization_flat_frame = cpl_frame_duplicate(
                    self->polarization_flat_frame_cam1);
        }
        if (self->modem_efficiency_frame_cam1) {
            result->modem_efficiency_frame = cpl_frame_duplicate(
                    self->modem_efficiency_frame_cam1);
        }
        if (self->star_center_frame_cam1) {
            result->star_center_frame = cpl_frame_duplicate(
                    self->star_center_frame_cam1);
        }

    } else if (camera_id == SPH_ZPL_KEYWORD_VALUE_CAMERA2_ID) {
        result->center_xoffset = self->center_xoffset_cam2;
        result->center_yoffset = self->center_yoffset_cam2;

        result->preproc_frames = cpl_frameset_duplicate(
                self->preproc_frames_cam2);
        if (stoke == SPH_ZPL_STOKES_Q) {
            result->fgs = sph_zpl_framegroups_create_Q(result->preproc_frames);
            if (!result->fgs) {
                SPH_WARNING(
                        "Neither Qplus nor Qminus frames found, sph_zpl_framegroups_create returns NULL!");
                _sph_zpl_science_p1_cunit_delete(result);
                result = NULL;
                return result;
            } else {
                SPH_INFO_MSG("CAM2: Q stokes: fgs is not NULL");
                result->science_p1_outfilename = cpl_strdup(
                        self->science_p1_outfilename_q_cam2);
                result->science_p1_plus_outfilename = cpl_strdup(
                        self->science_p1_plus_q_outfilename_cam2);
                result->science_p1_minus_outfilename = cpl_strdup(
                        self->science_p1_minus_q_outfilename_cam2);
                result->eso_pro_catg = cpl_strdup(
                        SPH_ZPL_TAG_SCIENCE_P1_Q_CALIB_CAM2); //TAG for the double image science product CAM1
                result->eso_pro_catg_plus = cpl_strdup(
                        SPH_ZPL_TAG_SCIENCE_P1_QPLUS_CALIB_CAM2); //TAG for the double image science plus product CAM1
                result->eso_pro_catg_minus = cpl_strdup(
                        SPH_ZPL_TAG_SCIENCE_P1_QMINUS_CALIB_CAM2); //TAG for the double image science plus product CAM1

                //group raw frames on QPLUS and QMINUS, which is needed for save_dfs
                result->fgsraw = sph_zpl_framegroups_create_Q(self->rawframes);
                if (!result->fgsraw) {
                     SPH_WARNING(
                             "Neither Qplus nor Qminus frames found, sph_zpl_framegroups_create_Q raw returns NULL!");
                     _sph_zpl_science_p1_cunit_delete(result);
                     result = NULL;
                     return result;
                }
            }
        } else if (stoke == SPH_ZPL_STOKES_U) {
            result->fgs = sph_zpl_framegroups_create_U(result->preproc_frames);
            if (!result->fgs) {
                SPH_WARNING(
                        "Neither Uplus nor Uminus frames found, sph_zpl_framegroups_create returns NULL!");
                _sph_zpl_science_p1_cunit_delete(result);
                result = NULL;
                return result;
            } else {
                SPH_INFO_MSG("CAM2 U stokes: fgs is not NULL");
                result->science_p1_outfilename = cpl_strdup(
                        self->science_p1_outfilename_u_cam2);
                result->science_p1_plus_outfilename = cpl_strdup(
                        self->science_p1_plus_u_outfilename_cam2);
                result->science_p1_minus_outfilename = cpl_strdup(
                        self->science_p1_minus_u_outfilename_cam2);
                result->eso_pro_catg = cpl_strdup(
                        SPH_ZPL_TAG_SCIENCE_P1_U_CALIB_CAM2); //TAG for the double image science product CAM1
                result->eso_pro_catg_plus = cpl_strdup(
                        SPH_ZPL_TAG_SCIENCE_P1_UPLUS_CALIB_CAM2); //TAG for the double image science plus product CAM1
                result->eso_pro_catg_minus = cpl_strdup(
                        SPH_ZPL_TAG_SCIENCE_P1_UMINUS_CALIB_CAM2); //TAG for the double image science plus product CAM1

                //group raw frames on UPLUS and UMINUS, which is needed for save_dfs
                result->fgsraw = sph_zpl_framegroups_create_U(self->rawframes);
                if (!result->fgsraw) {
                     SPH_WARNING(
                             "Neither Uplus nor Uminus frames found, sph_zpl_framegroups_create_Q raw returns NULL!");
                     _sph_zpl_science_p1_cunit_delete(result);
                     result = NULL;
                     return result;
                }

            }
        } else { //set all to the Q stokes
            result->fgs = sph_zpl_framegroups_new();
            rerr = sph_zpl_framegroups_set_framesplus(result->fgs,
                    self->preproc_frames_cam2);
            if (rerr != CPL_ERROR_NONE) {
                SPH_ERR(
                        "Can't set plus frames into the framegroups, sph_zpl_framegroups_frames returns NULL!");
                _sph_zpl_science_p1_cunit_delete(result);
                result = NULL;
                return result;
            }SPH_INFO_MSG("CAM2: Setting all frames as Qplus frames.")
            result->science_p1_outfilename = cpl_strdup(
                    self->science_p1_outfilename_q_cam2);
            result->science_p1_plus_outfilename = cpl_strdup(
                    self->science_p1_plus_q_outfilename_cam2);
            result->science_p1_minus_outfilename = cpl_strdup(
                    self->science_p1_minus_q_outfilename_cam2);
            result->eso_pro_catg = cpl_strdup(
                    SPH_ZPL_TAG_SCIENCE_P1_Q_CALIB_CAM2); //TAG for the double image science product CAM1
            result->eso_pro_catg_plus = cpl_strdup(
                    SPH_ZPL_TAG_SCIENCE_P1_QPLUS_CALIB_CAM2); //TAG for the double image science plus product CAM1
            result->eso_pro_catg_minus = cpl_strdup(
                    SPH_ZPL_TAG_SCIENCE_P1_QMINUS_CALIB_CAM2); //TAG for the double image science plus product CAM1

            //group raw frames on QPLUS and QMINUS, which is needed for save_dfs
            result->fgsraw = sph_zpl_framegroups_create_Q(self->rawframes);
            if (!result->fgsraw) {
                 SPH_WARNING(
                         "Neither Qplus nor Qminus frames found, sph_zpl_framegroups_create_Q raw returns NULL!");
                 _sph_zpl_science_p1_cunit_delete(result);
                 result = NULL;
                 return result;
            }

        }

        if (self->master_bias_frame_cam2) {
            result->master_bias_frame = cpl_frame_duplicate(
                    self->master_bias_frame_cam2);
        }
        if (self->master_dark_frame_cam2) {
            result->master_dark_frame = cpl_frame_duplicate(
                    self->master_dark_frame_cam2);
        }
        if (self->intensity_flat_frame_cam2) {
            result->intensity_flat_frame = cpl_frame_duplicate(
                    self->intensity_flat_frame_cam2);
        }
        if (self->intensity_flat_frame_master_cam2) {
            result->intensity_flat_frame_master = cpl_frame_duplicate(
                    self->intensity_flat_frame_master_cam2);
        }
        if (self->polarization_flat_frame_cam2) {
            result->polarization_flat_frame = cpl_frame_duplicate(
                    self->polarization_flat_frame_cam2);
        }
        if (self->modem_efficiency_frame_cam2) {
            result->modem_efficiency_frame = cpl_frame_duplicate(
                    self->modem_efficiency_frame_cam2);
        }
        if (self->star_center_frame_cam2) {
            result->star_center_frame = cpl_frame_duplicate(
                    self->star_center_frame_cam2);
        }

    } else if (camera_id == SPH_ZPL_KEYWORD_VALUE_CAMERA_INDEFERENT_ID) {

                result->center_xoffset = self->center_xoffset_cam1;
                result->center_yoffset = self->center_yoffset_cam1;


                result->preproc_frames = cpl_frameset_duplicate(
                        self->preproc_frames);
                if (stoke == SPH_ZPL_STOKES_Q) {
                    result->fgs = sph_zpl_framegroups_create_Q(result->preproc_frames);
                    if (!result->fgs) {
                        SPH_WARNING(
                                "Neither Qplus nor Qminus frames found, sph_zpl_framegroups_create returns NULL!");
                        _sph_zpl_science_p1_cunit_delete(result);
                        result = NULL;
                        return result;
                    } else {
                        SPH_INFO_MSG("CAM2: Q stokes: fgs is not NULL");
                        result->science_p1_outfilename = cpl_strdup(
                                self->science_p1_outfilename_q);
                        result->science_p1_plus_outfilename = cpl_strdup(
                                self->science_p1_plus_q_outfilename);
                        result->science_p1_minus_outfilename = cpl_strdup(
                                self->science_p1_minus_q_outfilename);
                        result->eso_pro_catg = cpl_strdup(
                                SPH_ZPL_TAG_SCIENCE_P1_Q_CALIB); //TAG for the double image science product
                        result->eso_pro_catg_plus = cpl_strdup(
                                SPH_ZPL_TAG_SCIENCE_P1_QPLUS_CALIB); //TAG for the double image science plus product
                        result->eso_pro_catg_minus = cpl_strdup(
                                SPH_ZPL_TAG_SCIENCE_P1_QMINUS_CALIB); //TAG for the double image science plus product
                    }
                } else if (stoke == SPH_ZPL_STOKES_U) {
                    result->fgs = sph_zpl_framegroups_create_U(result->preproc_frames);
                    if (!result->fgs) {
                        SPH_WARNING(
                                "Neither Uplus nor Uminus frames found, sph_zpl_framegroups_create returns NULL!");
                        _sph_zpl_science_p1_cunit_delete(result);
                        result = NULL;
                        return result;
                    } else {
                        SPH_INFO_MSG("CAM2 U stokes: fgs is not NULL");
                        result->science_p1_outfilename = cpl_strdup(
                                self->science_p1_outfilename_u);
                        result->science_p1_plus_outfilename = cpl_strdup(
                                self->science_p1_plus_u_outfilename);
                        result->science_p1_minus_outfilename = cpl_strdup(
                                self->science_p1_minus_u_outfilename);
                        result->eso_pro_catg = cpl_strdup(
                                SPH_ZPL_TAG_SCIENCE_P1_U_CALIB); //TAG for the double image science product
                        result->eso_pro_catg_plus = cpl_strdup(
                                SPH_ZPL_TAG_SCIENCE_P1_UPLUS_CALIB); //TAG for the double image science plus product
                        result->eso_pro_catg_minus = cpl_strdup(
                                SPH_ZPL_TAG_SCIENCE_P1_UMINUS_CALIB); //TAG for the double image science plus product
                    }
                } else { //set all to the Q stokes
                    result->fgs = sph_zpl_framegroups_new();
                    rerr = sph_zpl_framegroups_set_framesplus(result->fgs,
                            self->preproc_frames);
                    if (rerr != CPL_ERROR_NONE) {
                        SPH_ERR(
                                "Can't set plus frames into the framegroups, sph_zpl_framegroups_frames returns NULL!");
                        _sph_zpl_science_p1_cunit_delete(result);
                        result = NULL;
                        return result;
                    }SPH_INFO_MSG("CAM Indifferent: Setting all frames as Qplus frames.")
                    result->science_p1_outfilename = cpl_strdup(
                            self->science_p1_outfilename_q);
                    result->science_p1_plus_outfilename = cpl_strdup(
                            self->science_p1_plus_q_outfilename);
                    result->science_p1_minus_outfilename = cpl_strdup(
                            self->science_p1_minus_q_outfilename);
                    result->eso_pro_catg = cpl_strdup(
                            SPH_ZPL_TAG_SCIENCE_P1_Q_CALIB); //TAG for the double image science product CAM1
                    result->eso_pro_catg_plus = cpl_strdup(
                            SPH_ZPL_TAG_SCIENCE_P1_QPLUS_CALIB); //TAG for the double image science plus product CAM1
                    result->eso_pro_catg_minus = cpl_strdup(
                            SPH_ZPL_TAG_SCIENCE_P1_QMINUS_CALIB); //TAG for the double image science plus product CAM1
                }

                if (self->master_bias_frame) {
                    result->master_bias_frame = cpl_frame_duplicate(
                            self->master_bias_frame);
                }
                if (self->master_dark_frame) {
                    result->master_dark_frame = cpl_frame_duplicate(
                            self->master_dark_frame);
                }
                if (self->intensity_flat_frame) {
                    result->intensity_flat_frame = cpl_frame_duplicate(
                            self->intensity_flat_frame);
                }
                if (self->intensity_flat_frame_master) {
                    result->intensity_flat_frame_master = cpl_frame_duplicate(
                            self->intensity_flat_frame_master);
                }
                if (self->polarization_flat_frame) {
                    result->polarization_flat_frame = cpl_frame_duplicate(
                            self->polarization_flat_frame);
                }
                if (self->modem_efficiency_frame) {
                    result->modem_efficiency_frame = cpl_frame_duplicate(
                            self->modem_efficiency_frame);
                }
                if (self->star_center_frame) {
                    result->star_center_frame = cpl_frame_duplicate(
                            self->star_center_frame);
                }

    } else {
        sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Wrong camera id = %d (it must be 1 or 2 for oldstyle format, 0 for newstyle format)",
                camera_id);
        _sph_zpl_science_p1_cunit_delete(result);
        result = NULL;
    }

    if (sph_error_get_last_code() != CPL_ERROR_NONE) {
        sph_error_raise(
                SPH_ERROR_GENERAL,
                __FILE__,
                __func__,
                __LINE__,
                SPH_ERROR_ERROR,
                "Post condition error check shows the following error %d, please, check ",
                sph_error_get_last_code());
        _sph_zpl_science_p1_cunit_delete(result);
        result = NULL;
    }

    return result;

}

sph_error_code _sph_zpl_science_p1_cunit_delete(sph_zpl_science_p1_cunit* cunit) {
    sph_error_code rerr = CPL_ERROR_NONE;

    if (cunit == NULL) {
        sph_error_raise(CPL_ERROR_NULL_INPUT, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Null input pointer.");
        return rerr;
    }

    /* Code to delete recipe pointers GENERATED DO NOT EDIT */

    if (cunit->preproc_frames != NULL) {
        cpl_frameset_delete(cunit->preproc_frames);
        cunit->preproc_frames = NULL;
    }
    if (cunit->science_di) {
    	sph_zpl_science_pol_product_delete(cunit->science_di);
        cunit->science_di = NULL;
    }
    if (cunit->science_di_plus) {
    	sph_zpl_science_pol_product_delete(cunit->science_di_plus);
        cunit->science_di_plus = NULL;
    }
    if (cunit->science_di_minus) {
    	sph_zpl_science_pol_product_delete(cunit->science_di_minus);
        cunit->science_di_minus = NULL;
    }

    if (cunit->science_p1_outfilename)
        cpl_free(cunit->science_p1_outfilename);
    if (cunit->science_p1_plus_outfilename)
        cpl_free(cunit->science_p1_plus_outfilename);
    if (cunit->science_p1_minus_outfilename)
        cpl_free(cunit->science_p1_minus_outfilename);
    if (cunit->eso_pro_catg)
        cpl_free(cunit->eso_pro_catg);
    if (cunit->eso_pro_catg_plus)
        cpl_free(cunit->eso_pro_catg_plus);
    if (cunit->eso_pro_catg_minus)
        cpl_free(cunit->eso_pro_catg_minus);
    if (cunit->fgs)
        sph_zpl_framegroups_delete(cunit->fgs);
    if (cunit->fgsraw)
        sph_zpl_framegroups_delete(cunit->fgsraw);

    if (cunit->preproc_frames)
        cpl_frameset_delete(cunit->preproc_frames);

    if (cunit->master_bias_frame)
        cpl_frame_delete(cunit->master_bias_frame);
    if (cunit->master_dark_frame)
        cpl_frame_delete(cunit->master_dark_frame);
    if (cunit->intensity_flat_frame)
        cpl_frame_delete(cunit->intensity_flat_frame);
    if (cunit->intensity_flat_frame_master)
        cpl_frame_delete(cunit->intensity_flat_frame_master);
    if (cunit->polarization_flat_frame)
        cpl_frame_delete(cunit->polarization_flat_frame);
    if (cunit->modem_efficiency_frame)
        cpl_frame_delete(cunit->modem_efficiency_frame);
    if (cunit->star_center_frame)
        cpl_frame_delete(cunit->star_center_frame);


    if (cunit->current_raw_frameset) {
        cpl_frameset_delete(cunit->current_raw_frameset);
        cunit->current_raw_frameset = NULL;
    }
    sph_polygon_free_all();
    cpl_free(cunit);
    return rerr;
}

static void sph_zpl_science_p1_delete__(sph_quad_image* master_bias_quadimage,
        sph_quad_image* master_dark_quadimage,
        sph_quad_image* intensity_flat_quadimage,
        sph_master_frame* intensity_flat_masterframe,
        sph_double_image* polarization_flat_doubleimage,
        sph_master_frame* modem_efficiency_masterframe) {
    if (master_bias_quadimage) {
        sph_quad_image_delete(master_bias_quadimage);

    }
    if (master_dark_quadimage) {
        sph_quad_image_delete(master_dark_quadimage);

    }
    if (intensity_flat_quadimage) {
        sph_quad_image_delete(intensity_flat_quadimage);

    }
    if (intensity_flat_masterframe) {
        sph_master_frame_delete(intensity_flat_masterframe);

    }
    if (polarization_flat_doubleimage) {
        sph_double_image_delete(polarization_flat_doubleimage);

    }
    if (modem_efficiency_masterframe) {
        sph_master_frame_delete(modem_efficiency_masterframe);

    }
}

static cpl_frame* sph_zpl_science_p1_calibrate_transform_frame__(
        const char* filename, cpl_frameset* inframes, sph_fctable* fctable,
        sph_quad_image* master_bias_quadimage,
        sph_quad_image* master_dark_quadimage,
        sph_quad_image* intensity_flat_quadimage,
        sph_master_frame* intensity_flat_masterframe,
        sph_double_image* polarization_flat_doubleimage,
        sph_master_frame* modem_efficiency_masterframe ) {

    sph_zpl_exposure* zplexp = NULL;
    sph_cube* dicube = NULL;
    cpl_frame* result = NULL;

    int iplane = 0;
    int rerr = CPL_ERROR_NONE;
    sph_transform* transform = NULL;

    cpl_ensure(inframes, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(filename, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(fctable, CPL_ERROR_NULL_INPUT, NULL);

    transform = sph_transform_new_default();
    if (!transform) {
        SPH_ERR("sph_transform_new_dafault() constructor returns NULL...");
        return NULL;
    }
    dicube = sph_cube_new(filename);
    if (!dicube) {
        SPH_ERR("sph_cube_new constructor returns NULL...");
        sph_transform_delete(transform);
        return NULL;
    }

    zplexp = sph_common_science_get_next_zpl_exposure(inframes);
    if (!zplexp) {
        SPH_ERR(
                "no zplexp can be loaded, sph_common_science_get_next_zpl_exposure returns NULL...");
        sph_transform_delete(transform);
        sph_cube_delete(dicube);
        return NULL;
    }

    while (zplexp) {
        sph_double_image* stokes = NULL;
        sph_quad_image* curquadimage = 
            sph_quad_image_new_from_zpl_exposures(zplexp);

        if (!curquadimage) {
            SPH_ERR(
                    "can't create quad image from the zplexp, sph_quad_image_new_from_zpl_exposures constructor returns NULL...");
            sph_transform_delete(transform);
            sph_cube_delete(dicube);
            sph_zpl_exposure_delete(zplexp);
            return NULL;

        }

        if (master_bias_quadimage) {
            rerr = sph_quad_image_subtract_quad_image(curquadimage,
                    master_bias_quadimage);
            if (rerr != CPL_ERROR_NONE) {
                sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR,
                        "Error is raised by subtracting master bias:\n"
                                "cpl error code is: %d", cpl_error_get_code());

                sph_zpl_exposure_delete(zplexp);
                sph_quad_image_delete(curquadimage);
                sph_cube_delete(dicube);
                sph_transform_delete(transform);
                return NULL;
            }
            SPH_INFO_MSG("Subtracted master bias.")
        }

        if (master_dark_quadimage) {
            rerr = sph_zpl_subtract_dark_quad_image_scaled(curquadimage,
                    master_dark_quadimage);
            if (rerr != CPL_ERROR_NONE) {
                sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR,
                        "Error is raised by subtracting dark:\n"
                                "cpl error code is: %d", cpl_error_get_code());
                sph_zpl_exposure_delete(zplexp);
                sph_quad_image_delete(curquadimage);
                sph_cube_delete(dicube);
                sph_transform_delete(transform);
                return NULL;
            }
            SPH_INFO_MSG("Subtracted master dark.")
        }

        if (intensity_flat_masterframe) {
            rerr = sph_quad_image_divide_master_frame(curquadimage,
                    intensity_flat_masterframe);
            if (rerr != CPL_ERROR_NONE) {
                sph_error_raise(
                        SPH_ERROR_GENERAL,
                        __FILE__,
                        __func__,
                        __LINE__,
                        SPH_ERROR_ERROR,
                        "Error is raised by applying (dividing) intensity flat (master frame):\n"
                                "cpl error code is: %d", cpl_error_get_code());
                sph_zpl_exposure_delete(zplexp);
                sph_quad_image_delete(curquadimage);
                sph_cube_delete(dicube);
                sph_transform_delete(transform);
                return NULL;
            }
            SPH_INFO_MSG("Applied intensity flat field (master frame).")
        } else if (intensity_flat_quadimage) {
            rerr = sph_quad_image_divide_quad_image(curquadimage,
                    intensity_flat_quadimage);
            if (rerr != CPL_ERROR_NONE) {
                sph_error_raise(
                        SPH_ERROR_GENERAL,
                        __FILE__,
                        __func__,
                        __LINE__,
                        SPH_ERROR_ERROR,
                        "Error is raised by applying (dividing) intensity flat (quad image):\n"
                                "cpl error code is: %d", cpl_error_get_code());
                sph_zpl_exposure_delete(zplexp);
                sph_quad_image_delete(curquadimage);
                sph_cube_delete(dicube);
                sph_transform_delete(transform);
                return NULL;
            }
            SPH_INFO_MSG("Applied intensity flat field (quad image).")

        }

        stokes =
                sph_zpl_utils_calculate_stokes_param_double_image_from_quad_image(
                        curquadimage);
        if (!stokes) {
            SPH_ERR("stokes parameters is NULL");
            sph_zpl_exposure_delete(zplexp);
            sph_quad_image_delete(curquadimage);
            sph_cube_delete(dicube);
            sph_transform_delete(transform);
            return NULL;
        }SPH_INFO_MSG("Calculated stokes.");

        if (polarization_flat_doubleimage) {
            rerr = sph_zpl_utils_apply_polarization_flat(stokes,
                    polarization_flat_doubleimage);
            if (rerr != CPL_ERROR_NONE) {
                sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR,
                        "Error is raised by applying polarization flat:\n"
                                "sph error code is: %d",
                        sph_error_get_last_code());
                sph_zpl_exposure_delete(zplexp);
                sph_quad_image_delete(curquadimage);
                sph_cube_delete(dicube);
                sph_transform_delete(transform);
                sph_double_image_delete(stokes);
                return NULL;
            }SPH_INFO_MSG("Applied polarization flat.");

        }

        if (modem_efficiency_masterframe) {
            rerr = sph_zpl_utils_apply_modem_efficiency(stokes,
                    modem_efficiency_masterframe);
            if (rerr != CPL_ERROR_NONE) {
                sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR,
                        "Error is raised by applying modem efficiency:\n"
                                "sph error code is: %d",
                        sph_error_get_last_code());
                sph_zpl_exposure_delete(zplexp);
                sph_quad_image_delete(curquadimage);
                sph_cube_delete(dicube);
                sph_transform_delete(transform);
                sph_double_image_delete(stokes);
                return NULL;
            }SPH_INFO_MSG("Applied modem effieciency.");
        }

        //quality check (calculate statistical moments for the basic polarization)
        rerr = sph_double_image_quality_check(stokes);
        if (rerr) {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "sph_double_image_quality_check returns cpl error.\n"
                            "cpl error code is: %d", rerr);
            sph_zpl_exposure_delete(zplexp);
            sph_quad_image_delete(curquadimage);
            sph_cube_delete(dicube);
            sph_transform_delete(transform);
            sph_double_image_delete(stokes);
            return NULL;
        }

        //if stokes double image is not squared expand it by factor 2 in y direction
        if (!sph_double_image_is_squared(stokes)) {
            SPH_INFO_MSG(
                    "double image is not squared, it will be linear interpolated in y-direction, "
                    "the total flux will be preserved (a small flux lost is due to the border lines)!")
            rerr = sph_double_image_interpolate_y_zpl_polarimetry( stokes, CPL_TRUE );
            if (rerr) {
                sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR,
                        "sph_doble_image_interpolate_y returns cpl error.\n"
                                "cpl error code is: %d", rerr);
                sph_zpl_exposure_delete(zplexp);
                sph_quad_image_delete(curquadimage);
                sph_cube_delete(dicube);
                sph_transform_delete(transform);
                sph_double_image_delete(stokes);
                return NULL;
            }
        }

        //check if it is squared
        if (!sph_double_image_is_squared(stokes)) {
            SPH_ERR("doubleimage is still not squared, "
                    "check the size of the input data...stop processing.\n");

            sph_zpl_exposure_delete(zplexp);
            sph_quad_image_delete(curquadimage);
            sph_cube_delete(dicube);
            sph_transform_delete(transform);
            sph_double_image_delete(stokes);
            return NULL;

        }
/*
        SPH_INFO_MSG("sph_transform_apply_to_doubleimage...")
        rerr = sph_transform_apply_to_doubleimage(transform, stokes, NULL,
                sph_fctable_get_centre_x_element(fctable, iplane),
                sph_fctable_get_centre_y_element(fctable, iplane),
                sph_fctable_get_angle(fctable, iplane), 1.0);
*/

        SPH_INFO_MSG("sph_transform_apply_to_doubleimage...")
        rerr = sph_transform_apply_to_doubleimage_doublemode(transform, stokes, NULL,
                    sph_fctable_get_centre_x_left_element( fctable, iplane ),
                    sph_fctable_get_centre_y_left_element( fctable, iplane),
                    sph_fctable_get_centre_x_right_element( fctable, iplane ),
                    sph_fctable_get_centre_y_right_element( fctable, iplane),
                    sph_fctable_get_angle(fctable, iplane), 1);

        if (rerr) {
            sph_error_raise(
                    SPH_ERROR_GENERAL,
                    __FILE__,
                    __func__,
                    __LINE__,
                    SPH_ERROR_ERROR,
                    "sph_transform_apply_to_doubleimage function returns cpl error.\n"
                            "cpl error code is: %d", rerr);
            sph_zpl_exposure_delete(zplexp);
            sph_quad_image_delete(curquadimage);
            sph_cube_delete(dicube);
            sph_transform_delete(transform);
            sph_double_image_delete(stokes);
            return NULL;
        }

        SPH_INFO_MSG(
                "append the transformed stokes double image to the cube...");
//        rerr = sph_double_image_save_dfs(
//                stokes,
//                cpl_sprintf("ST_%s", filename),
//                inframes, cpl_frameset_get_first(inframes),
//                NULL,
//                "GENERIC_TAG",
//                SPH_RECIPE_NAME_ZPL_SCIENCE_P1, SPH_PIPELINE_NAME_ZIMPOL,
//                NULL);
//        if (rerr) {
//            sph_error_raise(
//                    SPH_ERROR_GENERAL,
//                    __FILE__,
//                    __func__,
//                    __LINE__,
//                    SPH_ERROR_ERROR,
//                    "can't save the individual stokes image: %d", rerr);
//            sph_zpl_exposure_delete(zplexp);
//            sph_quad_image_delete(curquadimage);
//            sph_cube_delete(dicube);
//            sph_transform_delete(transform);
//            sph_double_image_delete(stokes);
//            return NULL;
//        }
        rerr = sph_cube_append_double_image(dicube, stokes, 1);
        if (rerr) {
            sph_error_raise(
                    SPH_ERROR_GENERAL,
                    __FILE__,
                    __func__,
                    __LINE__,
                    SPH_ERROR_ERROR,
                    "can't append the transformed stokes double image to the cube: "
                            "sph_cube_append_double_image function returns cpl error.\n"
                            "cpl error code is: %d", rerr);
            sph_zpl_exposure_delete(zplexp);
            sph_quad_image_delete(curquadimage);
            sph_cube_delete(dicube);
            sph_transform_delete(transform);
            sph_double_image_delete(stokes);
            return NULL;
        }

        sph_double_image_delete(stokes);
        sph_quad_image_delete(curquadimage);
        sph_zpl_exposure_delete(zplexp);
        zplexp = sph_common_science_get_next_zpl_exposure(inframes);
        iplane++;
    }

    rerr = sph_cube_finalise_file(dicube->filename);
#ifndef SPH_SAVE_NONDFS
    /* Avoiding creating this non-dfs file is too cumbersome, instead use rm */
    if (dicube->filename != NULL)
        (void)remove(dicube->filename);
#endif
    if (rerr) {
        sph_error_raise(
                SPH_ERROR_GENERAL,
                __FILE__,
                __func__,
                __LINE__,
                SPH_ERROR_ERROR,
                "can't finalize output cube for the filename %s: sph_cube_finalise_file function returns cpl error.\n"
                        "cpl error code is: %d", dicube->filename, rerr);
        sph_cube_delete(dicube);
        sph_transform_delete(transform);
        return NULL;
    }

    sph_cube_delete(dicube);

    result = cpl_frame_new();
    cpl_frame_set_filename(result, filename);
    cpl_frame_set_tag(result, SPH_ZPL_TAG_SCIENCE_P1_CALIB_TMP);
    sph_transform_delete(transform);

    return result;

}

static cpl_frameset* sph_zpl_science_p1_calibrate_frames__(
        sph_zpl_science_p1_cunit* self, cpl_frameset* inframes,
        sph_quad_image* master_bias_quadimage,
        sph_quad_image* master_dark_quadimage,
        sph_quad_image* intensity_flat_quadimage,
        sph_master_frame* intensity_flat_masterframe,
        sph_double_image* polarization_flat_doubleimage,
        sph_master_frame* modem_efficiency_masterframe) {

    cpl_frameset* results = NULL;
    const cpl_frame* curframe = NULL;
    cpl_propertylist*   pl = NULL;
    cpl_propertylist*   pl_star_center = NULL;
    cpl_image* image = NULL;
    double clx = 512.0;
    double cly = 512.0;
    double crx = 512.0;
    double cry = 512.0;

    cpl_ensure( self, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(inframes, CPL_ERROR_NULL_INPUT, NULL);

    SPH_INFO_MSG("Calibrate cubes...")

    if ( self->star_center_frame ){
        pl_star_center = cpl_propertylist_load( cpl_frame_get_filename( self->star_center_frame), 0);
    }

    results = cpl_frameset_new();
    curframe = cpl_frameset_get_first_const(inframes);
    //set up a middle of the image
    image = cpl_image_load(cpl_frame_get_filename( curframe ), CPL_TYPE_DOUBLE, 0, 0  );
    if ( image != NULL ){
        clx = (double) (cpl_image_get_size_x( image ))/ 2.0;
        cly = crx = cry = clx;
        cpl_image_delete( image ); image = NULL;
    } else {
        SPH_ERR("Can't load image from the given filename!");
    }


    //curframe = cpl_frameset_get_first( ovsc_subtract_frames );

    while (curframe) {
        cpl_frameset* ovsc_subtract_frames = NULL;
        char* outfilename = NULL;
        char* procatg = NULL;
        sph_fctable* fctable = NULL;
        cpl_frame* calframe = NULL;
        cpl_frameset* curframes = cpl_frameset_new();

        cpl_frameset_insert(curframes, cpl_frame_duplicate(curframe));

        SPH_ERROR_RAISE_INFO(
                SPH_ERROR_WARNING,
                "The filename of the current frame: %s", cpl_frame_get_filename( curframe ));
        if (self->fctable_frames) {
            SPH_INFO_MSG("Searching a corresponding fctable frame...")
            fctable = sph_fctable_find_fctable(curframe, self->fctable_frames);
            if (fctable == NULL)
                SPH_INFO_MSG(
                        "The fctable not found in the given fctable input frameset!")
        }

        if (fctable == NULL) {
            SPH_INFO_MSG("Creating fctable for the current frame");
            fctable = sph_fctable_create_fctable_new(curframe, SPH_FCTAB_DOUBLE);

            pl = cpl_propertylist_load(cpl_frame_get_filename(curframe), 0);

            //degrees HIERARCH ESO INS4 DROT2 POSANG = 180.0000 / Position angle [deg].
            //So this can be read from header?
            //FIXME

            if ( cpl_propertylist_has( pl, SPH_COMMON_KEYWORD_DROT2_MODE) ){
                const char*   drot2_mode = 
                  cpl_propertylist_get_string(pl, SPH_COMMON_KEYWORD_DROT2_MODE);

                  if (!strcmp(drot2_mode, SPH_COMMON_KEYWORD_VALUE_DROT2_MODE_NO_DEROT)){
                         sph_fctable_add_angle_scalar( fctable, SPH_ZPL_KEYWORD_P1_NO_DEROT_ANGLE_OFFSET);
                  }
            }
            if (pl) {cpl_propertylist_delete(pl); pl = NULL;}

            if ( self->center_xoffset != 0.0 || self->center_yoffset != 0.0 ) {
                          SPH_INFO_MSG("Applying offset from the center of image.")
                          sph_fctable_add_center_offset_double_mode ( fctable,
                                  self->center_xoffset, self->center_yoffset,
                                  self->center_xoffset, self->center_yoffset);
            }

            if ( pl_star_center ) {
                if ( cpl_propertylist_has( pl_star_center, SPH_ZPL_STAR_CENTER_IFRAME_XCOORD) &&
                     cpl_propertylist_has( pl_star_center, SPH_ZPL_STAR_CENTER_IFRAME_YCOORD) &&
                     cpl_propertylist_has( pl_star_center, SPH_ZPL_STAR_CENTER_PFRAME_XCOORD) &&
                     cpl_propertylist_has( pl_star_center, SPH_ZPL_STAR_CENTER_PFRAME_YCOORD) ){

                    SPH_INFO_MSG("Applying star center calibration...");
                    if ( !self->star_center_iframe ){
                       SPH_INFO_MSG("Center coordinates of IFRAME and PFRAME are used to de-rotate double image.");
                       sph_fctable_add_center_offset_double_mode ( fctable,
                         - clx + cpl_propertylist_get_double( pl_star_center, SPH_ZPL_STAR_CENTER_IFRAME_XCOORD),
                         - cly + cpl_propertylist_get_double( pl_star_center, SPH_ZPL_STAR_CENTER_IFRAME_YCOORD),
                         - crx + cpl_propertylist_get_double( pl_star_center, SPH_ZPL_STAR_CENTER_PFRAME_XCOORD),
                         - cry + cpl_propertylist_get_double( pl_star_center, SPH_ZPL_STAR_CENTER_PFRAME_YCOORD) );
                    } else {
                        SPH_INFO_MSG("Center coordinates of IFRAME  are used to de-rotate double image.");
                        sph_fctable_add_center_offset_double_mode ( fctable,
                          - clx + cpl_propertylist_get_double( pl_star_center, SPH_ZPL_STAR_CENTER_IFRAME_XCOORD),
                          - cly + cpl_propertylist_get_double( pl_star_center, SPH_ZPL_STAR_CENTER_IFRAME_YCOORD),
                          - crx + cpl_propertylist_get_double( pl_star_center, SPH_ZPL_STAR_CENTER_IFRAME_XCOORD),
                          - cry + cpl_propertylist_get_double( pl_star_center, SPH_ZPL_STAR_CENTER_IFRAME_YCOORD) );
                    }
                }
            }

            if (self->save_interprod) {
                char* fctable_name =
                    sph_fctable_construct_filename(curframe, 1);;
                sph_fctable_save_ascii(fctable, fctable_name);
                cpl_free(fctable_name);
            }
        }

        if (fctable == NULL) {
            SPH_ERR("fctable has null pointer");
            if (results) {
                cpl_frameset_delete(results);
                results = NULL;
            }
            if (curframes) {
                cpl_frameset_delete(curframes);
                curframes = NULL;
            }
            return NULL;
        }

        SPH_INFO_MSG("Creating filename for the intermediate products...");
        // PIPE-7143 - Let's turn this into a sensible filename
        cpl_propertylist* curframe_pl = cpl_propertylist_load_regexp(
                cpl_frame_get_filename(curframe), 0,
                cpl_sprintf("^.*(%s|%s).*$",
                            SPH_ZPL_KEYWORD_POL_STOKES_NAME,
                            SPH_ZPL_KEYWORD_DET_CHIP_INDEX
                            ),
                0
                );
        if (cpl_propertylist_get_size(curframe_pl) != (cpl_size)2) {
            cpl_msg_warning(__func__, "Unable to determine STOKES NAME and CAM for %s - "
                                      "resorting to generic PRO CATG",
                                      cpl_frame_get_filename(curframe));
            procatg = cpl_sprintf("%s", "ZPL_SCIENCE_P1_CALIBRATED");
        } else {
            const char* stokes_name = cpl_propertylist_get_string(curframe_pl, SPH_ZPL_KEYWORD_POL_STOKES_NAME);
            procatg = cpl_sprintf("ZPL_SCIENCE_P1_CALIBRATED_%s_CAM%d",
                    stokes_name,
                    cpl_propertylist_get_int(curframe_pl, SPH_ZPL_KEYWORD_DET_CHIP_INDEX)
                    );
            sph_utils_uppercase(procatg + 26); /* Convert from stokes_name */
        }
        cpl_propertylist_delete(curframe_pl);
        char base[1024] ;
        char ext[256] ;
        const char* cur_name = sph_filemanager_get_basename(cpl_frame_get_filename(curframe));
        sph_filemanager_split(cur_name, base, ext);
//        cpl_msg_info(__func__, "Split into %s and %s", base, ext);
        outfilename = cpl_sprintf("%s_%s", procatg, base);
        cur_name = NULL ;
        SPH_ERROR_RAISE_INFO( SPH_ERROR_INFO,
                "Intermediate outfilename = %s", outfilename);
//        getchar();

        if (self->subtract_overscan) {
            SPH_INFO_MSG("Subtract overscans from rawframes...")
            ovsc_subtract_frames = sph_zpl_utils_subtract_overscans(curframes);
            if (!ovsc_subtract_frames) {
                SPH_ERROR_RAISE_WARNING(
                        SPH_ERROR_WARNING,
                        "Overscans could not be subtracted, trying to proceed further "
                        "without subtraction...")
                ovsc_subtract_frames = cpl_frameset_duplicate(curframes);
                sph_error_reset(); //reset is needed because sph_zpl_utils_subtract_overscans raises error when it returns NULL
            }
        } else {
            SPH_ERROR_RAISE_WARNING(
                    SPH_ERROR_WARNING,
                    "Overscans could not be subtracted, trying to proceed further "
                    "without subtraction...")
            ovsc_subtract_frames = cpl_frameset_duplicate(curframes);
        }

        char* outfilename_tmp = cpl_sprintf("tmp_%s", outfilename);
        calframe = sph_zpl_science_p1_calibrate_transform_frame__(
                outfilename_tmp, ovsc_subtract_frames, fctable,
                master_bias_quadimage, master_dark_quadimage,
                intensity_flat_quadimage, intensity_flat_masterframe,
                polarization_flat_doubleimage, modem_efficiency_masterframe );
        if (!calframe) {
            SPH_ERR(
                    "can't calibrate the given frame(plane), the sph_zpl_science_p1_calibrate_transform_frame returns NULL");
            if (results) {
                cpl_frameset_delete(results);
                results = NULL;
            }
            if (curframes) {
                cpl_frameset_delete(curframes);
                curframes = NULL;
            }
            if (ovsc_subtract_frames) {
                if (!self->keep_intermediate)  sph_utils_frames_unlink(ovsc_subtract_frames);
                cpl_frameset_delete(ovsc_subtract_frames);
                ovsc_subtract_frames = NULL;
            }
            if (fctable) {
                sph_fctable_delete(fctable);
                fctable = NULL;
            }
            return NULL;
        }

        // PIPE-7143 outfilename points to our interprod cube, so we can apply
        // PRO.CATG to it here
        if (self->save_interprod) {
//            cpl_property* pcatg_prop = cpl_property_new(SPH_COMMON_KEYWORD_PRO_CATG,
//                                                        CPL_TYPE_STRING);
//            cpl_property_set_string(pcatg_prop, procatg);
//            sph_fits_update_property(outfilename,
//                                     pcatg_prop,
//                                     0);
//            cpl_property_delete(pcatg_prop); pcatg_prop = NULL ;

            cpl_propertylist* applist = cpl_propertylist_new();
            cpl_propertylist_append_string(applist,
                                           SPH_COMMON_KEYWORD_PRO_CATG,
                                           procatg);

//            cpl_msg_info(__func__, "self->inframes:");
//            cpl_frameset_dump(self->inframes, stdout);
//            cpl_msg_info(__func__, "self->inparams:");
//            cpl_parameterlist_dump(self->inparams, stdout);
//             Must set CPL_FRAME_GROUP_CALIB for all frames
            cpl_frameset_iterator* it = cpl_frameset_iterator_new(inframes);
            cpl_frame* fr = cpl_frameset_iterator_get(it);
            while (fr) {
                cpl_frame_set_group(fr, CPL_FRAME_GROUP_CALIB);
                cpl_frameset_iterator_advance(it, 1);
                fr = cpl_frameset_iterator_get(it);
            }
            cpl_frameset_iterator_delete(it); it = NULL;
//            cpl_msg_info(__func__, "inframes:");
//            cpl_frameset_dump(inframes, stdout);
            // Now, see if we can save a DFS product here
//            cpl_msg_info(__func__, "first:");
//            cpl_msg_info(__func__, "applist:");
//            cpl_propertylist_dump(applist, stdout);
            cpl_errorstate prestate = cpl_errorstate_get();
            cpl_error_code rerr = cpl_dfs_save_propertylist(
                    self->inframes, NULL,
                    self->inparams, inframes,
                    NULL, SPH_RECIPE_NAME_ZPL_SCIENCE_P1,
                    applist, NULL,
                    SPH_PIPELINE_NAME_ZIMPOL,
                    outfilename
                    );
            if (rerr != CPL_ERROR_NONE) {
                cpl_msg_warning(__func__, "DFS FAIL: %d, %s @ L%i, %s",
                                rerr, cpl_error_get_file(),
                                cpl_error_get_line(),
                                cpl_error_get_message());
                cpl_msg_warning(__func__, "Previous error state:");
                cpl_errorstate_dump(prestate, CPL_FALSE, NULL);
//                getchar();
            }

            // Now need to append the different panels to the DFS product
            cpl_imagelist* cube_im ;
            for (cpl_size i=0; i <= cpl_frame_get_nextensions(calframe); i++) {
                cube_im = cpl_imagelist_load(cpl_frame_get_filename(calframe),
                                             CPL_TYPE_UNSPECIFIED,
                                             i);
                prestate = cpl_errorstate_get();
                rerr = cpl_imagelist_save(cube_im, outfilename,
                                          CPL_TYPE_UNSPECIFIED, NULL,
                                          CPL_IO_EXTEND);
                if (rerr != CPL_ERROR_NONE) {
                    cpl_msg_warning(__func__, "IMSAVE FAIL: %d, %s @ L%i, %s",
                                    rerr, cpl_error_get_file(),
                                    cpl_error_get_line(),
                                    cpl_error_get_message());
                    cpl_errorstate_dump(prestate, CPL_FALSE, NULL);
//                    getchar();
                }
            }
            cpl_imagelist_delete(cube_im) ; cube_im = NULL ;

            cpl_propertylist_delete(applist) ; applist = NULL;
        }


        cpl_frameset_insert(results, calframe);
        if (!self->keep_intermediate) {
            sph_utils_frames_unlink(ovsc_subtract_frames);
        }
        cpl_frameset_delete(ovsc_subtract_frames);
        cpl_frameset_delete(curframes);
        sph_fctable_delete(fctable);
        cpl_free(outfilename_tmp); outfilename_tmp = NULL;
        cpl_free(outfilename); outfilename = NULL;
        cpl_free(procatg) ; procatg = NULL ;

        //curframe = cpl_frameset_get_next(ovsc_subtract_frames);
        curframe = cpl_frameset_get_next_const(inframes);
    }

    cpl_propertylist_delete( pl_star_center );

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return results;

}

cpl_error_code _sph_zpl_science_p1_cunit_run(sph_zpl_science_p1_cunit* self) {
    int recipe_error = CPL_ERROR_NONE;
    sph_quad_image* master_bias_quadimage = NULL;
    sph_quad_image* master_dark_quadimage = NULL;
    sph_quad_image* intensity_flat_quadimage = NULL;
    sph_master_frame* intensity_flat_masterframe = NULL;
    sph_double_image* polarization_flat_doubleimage = NULL;
    sph_master_frame* modem_efficiency_masterframe = NULL;
    sph_master_frame* imframe = NULL;
    sph_master_frame* pmframe = NULL;

    cpl_frameset* calframes = NULL;
    double xpix = -9999.0;
    double ypix = -9999.0;


    SPH_INFO_MSG("Starting static _sph_zpl_science_p1_cunit_run...");
//    getchar();
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE

    if (!self->preproc_frames) {
        SPH_ERR("no preproc frames are set up!")
//        getchar();
        return sph_error_get_last_code();
    }
    if (!self->fgs) {
        SPH_ERR("no framegroups found in the cunit!")
//        getchar();
        return sph_error_get_last_code();
    }

    cpl_msg_info(__func__, "Completed cunit_run sanity checks, press ENTER to continue");
//    getchar();

    //if master bias exists load it
    if (self->master_bias_frame) {
        SPH_INFO_MSG("Loading master bias...\n");
        master_bias_quadimage = sph_quad_image_load(
                cpl_frame_get_filename(self->master_bias_frame), 0);
    }

    cpl_msg_info(__func__, "Loaded master_bias_frame");
//    getchar();

    //if master dark exists load it
    if (self->master_dark_frame) {
        SPH_INFO_MSG("Loading master dark...\n");
        master_dark_quadimage = sph_quad_image_load(
                cpl_frame_get_filename(self->master_dark_frame), 0);
    }
    cpl_msg_info(__func__, "Loaded master_dark_frame");
//    getchar();

    if (self->intensity_flat_frame_master) {
        //if intensity flat field master frame exists load
        SPH_INFO_MSG("Loading intensity flat field masterframe...\n");
        intensity_flat_masterframe = sph_master_frame_load_(
                self->intensity_flat_frame_master, 0);
    } else {
        //if intensity flat field quad image exists load
        if (self->intensity_flat_frame) {
            SPH_INFO_MSG("Loading intensity flat field quad image...\n");
            intensity_flat_quadimage = sph_quad_image_load(
                    cpl_frame_get_filename(self->intensity_flat_frame), 0);
        }
    }

    cpl_msg_info(__func__, "Loaded intensity_flat_frame");
//    getchar();

    //if polarization flat field exists load it
    if (self->polarization_flat_frame) {
        SPH_INFO_MSG("Loading polarization flat field...\n");
        polarization_flat_doubleimage = sph_double_image_load(
                cpl_frame_get_filename(self->polarization_flat_frame), 0);
    }
    cpl_msg_info(__func__, "Loaded polarization_flat_frame");
//    getchar();

    //if modulation/demodulation efficiency exists load it
    if (self->modem_efficiency_frame) {
        SPH_INFO_MSG("Loading modulation/demodulation efficiency...\n");
        modem_efficiency_masterframe = sph_master_frame_load_(
                self->modem_efficiency_frame, 0);
    }
    cpl_msg_info(__func__, "Loaded modem_efficiency_frame");
//    getchar();

    if (self->fgs->framesplus != NULL && self->fgs->framesminus == NULL) {
        SPH_WARNING(
                "Only Plus frames were found, the performance can be reduced!");
//        getchar();
        SPH_ERROR_RAISE_WARNING(
                CPL_ERROR_ILLEGAL_INPUT,
                "It is usually needed two polarization stokes string value (Qplus/Uplus) and (Qminus/Uminus) "
                "in the header input. Make sure the %s keyword with the value = %s(%s) is presented in the "
                "input frames.", SPH_ZPL_KEYWORD_POL_STOKES_NAME, SPH_ZPL_KEYWORD_VALUE_POL_STOKES_QMINUS, SPH_ZPL_KEYWORD_VALUE_POL_STOKES_UMINUS);
//        getchar();

        //calibrate and transform
        SPH_INFO_MSG("Creating calframes...");
//        getchar();
        calframes = sph_zpl_science_p1_calibrate_frames__(self,
                self->fgs->framesplus,
                master_bias_quadimage,
                master_dark_quadimage, intensity_flat_quadimage,
                intensity_flat_masterframe, polarization_flat_doubleimage,
                modem_efficiency_masterframe);
//        cpl_frameset_dump(calframes, stdout);
//        cpl_msg_info(__func__, "Press ENTER to continue... (L1863)");
//        getchar();

        if (!calframes) {
            SPH_ERR("Can't calibrate given frames...");
            sph_zpl_science_p1_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
        }

        //combine iframe and pframe
        imframe = sph_common_science_combine(calframes, self->coll_alg,
                SPH_MASTER_FRAME_IMAGE_EXT, SPH_MASTER_FRAME_BPIX_EXT, -1);
        if (!imframe) {
            SPH_ERR(
                    "Can't combine calframes, imframe from sph_common_science_combine is NULL...");
            sph_zpl_science_p1_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            if (calframes) {
                cpl_frameset_delete(calframes);
                calframes = NULL;
            }
            SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
        }

        pmframe = sph_common_science_combine(calframes, self->coll_alg,
                SPH_MASTER_FRAME_IMAGE_EXT + 4, SPH_MASTER_FRAME_BPIX_EXT + 4,
                -1);
        if (!pmframe) {
            SPH_ERR("Can't combine calframes, pmframe is NULL...");
            sph_zpl_science_p1_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            if (calframes) {
                cpl_frameset_delete(calframes);
                calframes = NULL;
            }
            if (imframe) {
                sph_master_frame_delete(imframe);
                imframe = NULL;
            }
            SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
        }

        // PIPE-7143 - if no longer needed (save as DFS)
//        if ( !self->save_interprod ) {
            sph_utils_frames_unlink( calframes );
//        }
//        cpl_frameset_dump(calframes, stdout);
//        cpl_msg_info(__func__, "Press ENTER to continue... (L1914)");
//        getchar();

        cpl_frameset_delete(calframes);  calframes = NULL;

        //create double image from iframe and pframe master frames
        //input master frames are not duplicated, so don't delete them after creation of the double image
        self->science_di_plus = sph_zpl_science_pol_product_create(imframe,
                pmframe);
        if (!self->science_di_plus) {
            SPH_ERR(
                    "Can't create double image from iframe and pframe master frames, self->science_di_plus is NULL...");
            sph_zpl_science_p1_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            if (imframe) {
                sph_master_frame_delete(imframe);
                imframe = NULL;
            }
            if (pmframe) {
                sph_master_frame_delete(pmframe);
                pmframe = NULL;
            }
            SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
        }

        //calculate quality parameters and save it as a double image
        recipe_error = sph_double_image_quality_check(self->science_di_plus->image);
        if (recipe_error) {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "sph_double_image_quality_check returns cpl error.\n"
                            "cpl error code is: %d", recipe_error);
            sph_zpl_science_p1_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            return recipe_error;
        }

        // reference pixels for WCS
        xpix = cpl_image_get_size_x(self->science_di_plus->image->iframe->image)/2.;
        ypix = cpl_image_get_size_y(self->science_di_plus->image->iframe->image)/2.;

        if (self->rawframes) {

            cpl_propertylist * pl = sph_zpl_utils_get_camera_header(self->preproc_frames,
                    self->fgsraw->framesplus);
            //create WCS keywords and add to pl
            cpl_propertylist * pl_wcs = sph_zpl_utils_astrometry_create_wcs_pl(
                    (const cpl_frame*) cpl_frameset_get_first(self->fgsraw->framesplus),
                    (const double) xpix, (const double) ypix);
            if ( pl_wcs && pl){
                cpl_propertylist_append( pl, pl_wcs );
                cpl_propertylist_delete( pl_wcs ); pl_wcs = NULL;
            }

            sph_zpl_science_pol_product_set_filename(self->science_di_plus, self->science_p1_plus_outfilename);
            sph_zpl_science_pol_product_set_tag(self->science_di_plus, self->eso_pro_catg_plus);
            sph_zpl_science_pol_product_set_inherit_frame(self->science_di_plus, cpl_frameset_get_first(self->fgsraw->framesplus));
            sph_zpl_science_pol_product_wrap_header(self->science_di_plus, pl);

        } else if (self->preproc_frames) {

            cpl_propertylist * pl_wcs = sph_zpl_utils_astrometry_create_wcs_pl(
                    (const cpl_frame*) cpl_frameset_get_first(self->fgs->framesplus),
                    (const double) xpix, (const double) ypix);

            sph_zpl_science_pol_product_set_filename(self->science_di_plus, self->science_p1_plus_outfilename);
            sph_zpl_science_pol_product_set_tag(self->science_di_plus, self->eso_pro_catg_plus);
            sph_zpl_science_pol_product_set_inherit_frame(self->science_di_plus, cpl_frameset_get_first(self->fgs->framesplus));
            sph_zpl_science_pol_product_wrap_header(self->science_di_plus, pl_wcs);

        } else {
            SPH_ERR("Neither raw frames nor pre-processed frames provided!");
            recipe_error = sph_error_get_last_code();
        }

        if (recipe_error) {
            sph_error_raise(
                    SPH_ERROR_GENERAL,
                    __FILE__,
                    __func__,
                    __LINE__,
                    SPH_ERROR_ERROR,
                    "Couldn't save science_di product: sph_double_image_save_dfs returns cpl error.\n"
                            "cpl error code is: %d", recipe_error);
        }

        sph_zpl_science_p1_delete__(master_bias_quadimage,
                master_dark_quadimage, intensity_flat_quadimage,
                intensity_flat_masterframe, polarization_flat_doubleimage,
                modem_efficiency_masterframe);

        SPH_INFO_MSG("sph_zpl_science_p1...End");
        SPH_ERROR_CHECK_STATE_RETURN_ERRCODE

    } else if (self->fgs->framesplus == NULL && self->fgs->framesminus != NULL) {
        SPH_WARNING(
                "Only Minus frames were found, the performance can be reduced!");
        SPH_ERROR_RAISE_WARNING(
                CPL_ERROR_ILLEGAL_INPUT,
                "It is usually needed two polarization stokes string value (Qplus/Uplus) and (Qminus/Uminus) "
                "in the header input. Make sure the %s keyword with the value = %s(%s) is presented in the "
                "input frames.", SPH_ZPL_KEYWORD_POL_STOKES_NAME, SPH_ZPL_KEYWORD_VALUE_POL_STOKES_QMINUS, SPH_ZPL_KEYWORD_VALUE_POL_STOKES_UMINUS);

        //calibrate and transform MINUS frames(cubes)
        calframes = sph_zpl_science_p1_calibrate_frames__(self,
                self->fgs->framesminus,
                master_bias_quadimage,
                master_dark_quadimage, intensity_flat_quadimage,
                intensity_flat_masterframe, polarization_flat_doubleimage,
                modem_efficiency_masterframe);
//        cpl_frameset_dump(calframes, stdout);
//        cpl_msg_info(__func__, "Press ENTER to continue... (L2027)");
//        getchar();

        if (!calframes) {
            SPH_ERR("Can't calibrate given MINUS frames...");
            sph_zpl_science_p1_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
        }

        //combine MINUS frames(cubes)
        imframe = sph_common_science_combine(calframes, self->coll_alg,
                SPH_MASTER_FRAME_IMAGE_EXT, SPH_MASTER_FRAME_BPIX_EXT, -1);
        if (!imframe) {
            SPH_ERR("Can't combine calframes, imframe is NULL...");
            sph_zpl_science_p1_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            if (calframes) {
                cpl_frameset_delete(calframes);
                calframes = NULL;
            }
            SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
        }

        pmframe = sph_common_science_combine(calframes, self->coll_alg,
                SPH_MASTER_FRAME_IMAGE_EXT + 4, SPH_MASTER_FRAME_BPIX_EXT, -1);
        if (!pmframe) {
            SPH_ERR("Can't combine calframes, pmframe is NULL...");
            sph_zpl_science_p1_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            if (calframes) {
                cpl_frameset_delete(calframes);
                calframes = NULL;
            }
            if (imframe) {
                sph_master_frame_delete(imframe);
                imframe = NULL;
            }
            SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
        }

        // PIPE-7143 - if no longer needed (save as DFS)
//        if ( !self->save_interprod ) {
            sph_utils_frames_unlink( calframes );
//        }
//        cpl_frameset_dump(calframes, stdout);
//        cpl_msg_info(__func__, "Press ENTER to continue... (L2076)");
//        getchar();

        if (calframes) {
            cpl_frameset_delete(calframes);
            calframes = NULL;
        }

        //create calibrated and transformed double image for MINUS frames
        self->science_di_minus = sph_zpl_science_pol_product_create(
                imframe, pmframe);
        if (!self->science_di_minus) {
            SPH_ERR(
                    "Can't create double image from iframe and pframe of the MINUS master frames, self->science_di_minus is NULL...");
            sph_zpl_science_p1_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            if (imframe) {
                sph_master_frame_delete(imframe);
                imframe = NULL;
            }
            if (pmframe) {
                sph_master_frame_delete(pmframe);
                pmframe = NULL;
            }
            SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
        }

        //MINUS: calculate quality parameters and save it as a double image
        recipe_error = sph_double_image_quality_check(self->science_di_minus->image);
        if (recipe_error) {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "sph_double_image_quality_check returns cpl error.\n"
                            "cpl error code is: %d", recipe_error);
            sph_zpl_science_p1_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            return recipe_error;
        }

        // reference pixels for WCS
        xpix = cpl_image_get_size_x(self->science_di_minus->image->iframe->image)/2.;
        ypix = cpl_image_get_size_y(self->science_di_minus->image->iframe->image)/2.;
        if (self->rawframes) {

            cpl_propertylist * pl = sph_zpl_utils_get_camera_header(self->preproc_frames,
                    self->fgsraw->framesminus);
            cpl_propertylist * pl_wcs = sph_zpl_utils_astrometry_create_wcs_pl(
                    (const cpl_frame*) cpl_frameset_get_first(self->fgsraw->framesminus),
                    (const double) xpix, (const double) ypix);
            if ( pl_wcs && pl){
                cpl_propertylist_append( pl, pl_wcs );
                cpl_propertylist_delete( pl_wcs ); pl_wcs = NULL;
            }

            sph_zpl_science_pol_product_set_filename(self->science_di_minus, self->science_p1_minus_outfilename);
            sph_zpl_science_pol_product_set_tag(self->science_di_minus, self->eso_pro_catg_minus);
            sph_zpl_science_pol_product_set_inherit_frame(self->science_di_minus, cpl_frameset_get_first(self->fgsraw->framesplus));
            sph_zpl_science_pol_product_wrap_header(self->science_di_minus, pl);

        } else if (self->preproc_frames) {

            cpl_propertylist * pl_wcs = sph_zpl_utils_astrometry_create_wcs_pl(
                    (const cpl_frame*) cpl_frameset_get_first(self->fgs->framesminus),
                    (const double) xpix, (const double) ypix);

            sph_zpl_science_pol_product_set_filename(self->science_di_minus, self->science_p1_minus_outfilename);
            sph_zpl_science_pol_product_set_tag(self->science_di_minus, self->eso_pro_catg_minus);
            sph_zpl_science_pol_product_set_inherit_frame(self->science_di_minus, cpl_frameset_get_first(self->fgs->framesplus));
            sph_zpl_science_pol_product_wrap_header(self->science_di_minus, pl_wcs);

        } else {
            SPH_ERR("Neither raw frames nor pre-processed frames provided!");
            recipe_error = sph_error_get_last_code();
        }

        if (recipe_error) {
            sph_error_raise(
                    SPH_ERROR_GENERAL,
                    __FILE__,
                    __func__,
                    __LINE__,
                    SPH_ERROR_ERROR,
                    "Couldn't save science_di_minus product: sph_double_image_save_dfs returns cpl error.\n"
                            "cpl error code is: %d", recipe_error);
            sph_zpl_science_p1_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            return recipe_error;
        }

        SPH_INFO_MSG("sph_zpl_science_p1...End");
        SPH_ERROR_CHECK_STATE_RETURN_ERRCODE

    } else {
        SPH_INFO_MSG("PLUS and MINUS frames found!");
        //calibrate and transform PLUS frames(cubes)
        calframes = sph_zpl_science_p1_calibrate_frames__(self,
                self->fgs->framesplus,
                master_bias_quadimage,
                master_dark_quadimage, intensity_flat_quadimage,
                intensity_flat_masterframe, polarization_flat_doubleimage,
                modem_efficiency_masterframe);
//        cpl_frameset_dump(calframes, stdout);
//        cpl_msg_info(__func__, "Press ENTER to continue... (L2182)");
//        getchar();
        if (!calframes) {
            SPH_ERR("Can't calibrate given PLUS frames...");
            sph_zpl_science_p1_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
        }

        //combine PLUS frames(cubes): imframe
        imframe = sph_common_science_combine(calframes, self->coll_alg,
                SPH_MASTER_FRAME_IMAGE_EXT, SPH_MASTER_FRAME_BPIX_EXT, -1);
        if (!imframe) {
            SPH_ERR("Can't combine calframes, imframe is NULL...");
            sph_zpl_science_p1_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            if (calframes) {
                cpl_frameset_delete(calframes);
                calframes = NULL;
            }
            SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
        }

        //combine PLUS frames(cubes): pmframe
        pmframe = sph_common_science_combine(calframes, self->coll_alg,
                SPH_MASTER_FRAME_IMAGE_EXT + 4, SPH_MASTER_FRAME_BPIX_EXT, -1);
        if (!pmframe) {
            SPH_ERR("Can't combine calframes, pmframe is NULL...");
            sph_zpl_science_p1_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            if (calframes) {
                cpl_frameset_delete(calframes);
                calframes = NULL;
            }
            if (imframe) {
                sph_master_frame_delete(imframe);
                imframe = NULL;
            }
            SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
        }

        // PIPE-7143 - if no longer needed (save as DFS)
        //        if ( !self->save_interprod ) {
                    sph_utils_frames_unlink( calframes );
        //        }
//        cpl_frameset_dump(calframes, stdout);
//        cpl_msg_info(__func__, "Press ENTER to continue... (2292)");
//        getchar();
        if (calframes) {
//            // PIPE-7143 Can't delete here, need it for the MINUS frames
            calframes = NULL;
        }

        //create calibrated and transfrmed double image for PLUS frames
        self->science_di_plus = sph_zpl_science_pol_product_create(imframe,
                pmframe);
        if (!self->science_di_plus) {
            SPH_ERR(
                    "Can't create double image from iframe and pframe PLUS master frames, self->science_di_plus is NULL...");
            sph_zpl_science_p1_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            if (imframe) {
                sph_master_frame_delete(imframe);
                imframe = NULL;
            }
            if (pmframe) {
                sph_master_frame_delete(pmframe);
                pmframe = NULL;
            }
            SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
        }

        //calibrate and transform MINUS frames(cubes)
        calframes = sph_zpl_science_p1_calibrate_frames__(self,
                self->fgs->framesminus,
                master_bias_quadimage,
                master_dark_quadimage, intensity_flat_quadimage,
                intensity_flat_masterframe, polarization_flat_doubleimage,
                modem_efficiency_masterframe);
//        cpl_frameset_dump(calframes, stdout);
//        cpl_msg_info(__func__, "Press ENTER to continue... (L2270)");
//        getchar();
        if (!calframes) {
            SPH_ERR("Can't calibrate given MINUS frames...");
            sph_zpl_science_p1_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
        }

        //combine MINUS frames(cubes)
        imframe = sph_common_science_combine(calframes, self->coll_alg,
                SPH_MASTER_FRAME_IMAGE_EXT, SPH_MASTER_FRAME_BPIX_EXT, -1);
        if (!imframe) {
            SPH_ERR("Can't combine calframes, imframe is NULL...");
            sph_zpl_science_p1_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            if (calframes) {
                cpl_frameset_delete(calframes);
                calframes = NULL;
            }
            SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
        }

        pmframe = sph_common_science_combine(calframes, self->coll_alg,
                SPH_MASTER_FRAME_IMAGE_EXT + 4, SPH_MASTER_FRAME_BPIX_EXT, -1);
        if (!pmframe) {
            SPH_ERR("Can't combine calframes, pmframe is NULL...");
            sph_zpl_science_p1_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            if (calframes) {
                cpl_frameset_delete(calframes);
                calframes = NULL;
            }
            if (imframe) {
                sph_master_frame_delete(imframe);
                imframe = NULL;
            }
            SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
        }

        // PIPE-7143 - if no longer needed (save as DFS)
//        if ( !self->save_interprod ) {
            sph_utils_frames_unlink( calframes );
//        }
//        cpl_frameset_dump(calframes, stdout);
//        cpl_msg_info(__func__, "Press ENTER to continue... (L2317)");
//        getchar();

        if (calframes) {
            cpl_frameset_delete(calframes);
            calframes = NULL;
        }

        //create calibrated and transformed double image for MINUS frames
        self->science_di_minus = sph_zpl_science_pol_product_create(
                imframe, pmframe);
        if (!self->science_di_minus) {
            SPH_ERR(
                    "Can't create double image from iframe and pframe of the MINUS master frames, self->science_di_minus is NULL...");
            sph_zpl_science_p1_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            if (imframe) {
                sph_master_frame_delete(imframe);
                imframe = NULL;
            }
            if (pmframe) {
                sph_master_frame_delete(pmframe);
                pmframe = NULL;
            }
            SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
        }

        if (!(self->science_di_minus && self->science_di_plus)) {
            SPH_ERROR_RAISE_ERR(
                    SPH_ERROR_GENERAL,
                    " sciece_di_minus and/or self->science_di_plus have(has) null pointer(s)!\n");
            SPH_RAISE_CPL;
            sph_zpl_science_p1_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            return (int) cpl_error_get_code();
        }

        //PLUS: calculate quality parameters and save it as a double image
        recipe_error = sph_double_image_quality_check(self->science_di_plus->image);
        if (recipe_error) {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "sph_double_image_quality_check returns cpl error.\n"
                            "cpl error code is: %d", recipe_error);
            sph_zpl_science_p1_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            return recipe_error;
        }

        // reference pixels for WCS
        xpix = cpl_image_get_size_x(self->science_di_plus->image->iframe->image)/2.;
        ypix = cpl_image_get_size_y(self->science_di_plus->image->iframe->image)/2.;

        if (self->rawframes) {

            cpl_propertylist * pl = sph_zpl_utils_get_camera_header(self->fgs->framesplus,
                    self->fgsraw->framesplus);
            //create WCS keywords and add to pl
            cpl_propertylist * pl_wcs = sph_zpl_utils_astrometry_create_wcs_pl(
                    (const cpl_frame*) cpl_frameset_get_first(self->fgsraw->framesplus),
                    (const double) xpix, (const double) ypix);
            if ( pl_wcs && pl){
                cpl_propertylist_append( pl, pl_wcs );
                cpl_propertylist_delete( pl_wcs ); pl_wcs = NULL;
            }

            sph_zpl_science_pol_product_set_filename(self->science_di_plus, self->science_p1_plus_outfilename);
            sph_zpl_science_pol_product_set_tag(self->science_di_plus, self->eso_pro_catg_plus);
            sph_zpl_science_pol_product_set_inherit_frame(self->science_di_plus, cpl_frameset_get_first(self->fgsraw->framesplus));
            sph_zpl_science_pol_product_wrap_header(self->science_di_plus, pl);

         } else if (self->preproc_frames) {

        	cpl_propertylist * pl_wcs = sph_zpl_utils_astrometry_create_wcs_pl(
                     (const cpl_frame*) cpl_frameset_get_first(self->fgs->framesplus),
                     (const double) xpix, (const double) ypix);


            sph_zpl_science_pol_product_set_filename(self->science_di_plus, self->science_p1_plus_outfilename);
            sph_zpl_science_pol_product_set_tag(self->science_di_plus, self->eso_pro_catg_plus);
            sph_zpl_science_pol_product_set_inherit_frame(self->science_di_plus, cpl_frameset_get_first(self->fgs->framesplus));
            sph_zpl_science_pol_product_wrap_header(self->science_di_plus, pl_wcs);

        } else {
            SPH_ERR("Neither raw frames nor pre-processed frames provided!");
            recipe_error = sph_error_get_last_code();
        }

        if (recipe_error) {
            sph_error_raise(
                    SPH_ERROR_GENERAL,
                    __FILE__,
                    __func__,
                    __LINE__,
                    SPH_ERROR_ERROR,
                    "Couldn't save science_di_plus product: sph_double_image_save_dfs returns cpl error.\n"
                            "cpl error code is: %d", recipe_error);
            sph_zpl_science_p1_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            return recipe_error;
        }

        //MINUS: calculate quality parameters and save it as a double image
        recipe_error = sph_double_image_quality_check(self->science_di_minus->image);
        if (recipe_error) {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "sph_double_image_quality_check returns cpl error.\n"
                            "cpl error code is: %d", recipe_error);
            sph_zpl_science_p1_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            return recipe_error;
        }

        // reference pixels for WCS
        xpix = cpl_image_get_size_x(self->science_di_minus->image->iframe->image)/2.;
        ypix = cpl_image_get_size_y(self->science_di_minus->image->iframe->image)/2.;

        if (self->rawframes) {

            cpl_propertylist * pl = sph_zpl_utils_get_camera_header(self->fgs->framesminus,
                    self->fgsraw->framesminus);
            cpl_propertylist * pl_wcs = sph_zpl_utils_astrometry_create_wcs_pl(
                    (const cpl_frame*) cpl_frameset_get_first(self->fgsraw->framesminus),
                    (const double) xpix, (const double) ypix);
            if ( pl_wcs && pl){
                cpl_propertylist_append( pl, pl_wcs );
                cpl_propertylist_delete( pl_wcs ); pl_wcs = NULL;
            }

            sph_zpl_science_pol_product_set_filename(self->science_di_minus, self->science_p1_minus_outfilename);
            sph_zpl_science_pol_product_set_tag(self->science_di_minus, self->eso_pro_catg_minus);
            sph_zpl_science_pol_product_set_inherit_frame(self->science_di_minus, cpl_frameset_get_first(self->fgsraw->framesminus));
            sph_zpl_science_pol_product_wrap_header(self->science_di_minus, pl);

        } else if (self->preproc_frames) {

            cpl_propertylist * pl_wcs = sph_zpl_utils_astrometry_create_wcs_pl(
                    (const cpl_frame*) cpl_frameset_get_first(self->fgs->framesminus),
                    (const double) xpix, (const double) ypix);

            sph_zpl_science_pol_product_set_filename(self->science_di_minus, self->science_p1_minus_outfilename);
            sph_zpl_science_pol_product_set_tag(self->science_di_minus, self->eso_pro_catg_minus);
            sph_zpl_science_pol_product_set_inherit_frame(self->science_di_minus, cpl_frameset_get_first(self->fgs->framesminus));
            sph_zpl_science_pol_product_wrap_header(self->science_di_minus, pl_wcs);

        } else {
            SPH_ERR("Neither raw frames nor pre-processed frames provided!");
            recipe_error = sph_error_get_last_code();
        }

        if (recipe_error) {
            sph_error_raise(
                    SPH_ERROR_GENERAL,
                    __FILE__,
                    __func__,
                    __LINE__,
                    SPH_ERROR_ERROR,
                    "Couldn't save science_di_minus product: sph_double_image_save_dfs returns cpl error.\n"
                            "cpl error code is: %d", recipe_error);
            sph_zpl_science_p1_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            return recipe_error;
        }

        //combine +Q(+U) and -Q(+U) double images:
        // I = [I(+Q) + I(-Q)]/2.  (the same for +U and -U)
        // P = [P(+Q) - P(-Q)]/2.  (the same for +U and -U)
        //first, duplicate science_di_plus into science_di and then combine it with the science_di_minus
        self->science_di = sph_zpl_science_pol_product_duplicate(self->science_di_plus);
        if (!self->science_di) {
            SPH_ERR(
                    "Can't duplicate science_di_plus to science_di, self->science_di is NULL...");
            sph_zpl_science_p1_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
        }

        recipe_error = sph_double_image_combine_double_image(self->science_di->image,
                self->science_di_minus->image);
        if (recipe_error) {
            sph_error_raise(
                    SPH_ERROR_GENERAL,
                    __FILE__,
                    __func__,
                    __LINE__,
                    SPH_ERROR_ERROR,
                    "Couldn't combine plus and minus product science product: sph_double_image_combine_double_image returns cpl error.\n"
                            "cpl error code is: %d", recipe_error);
            sph_zpl_science_p1_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            return recipe_error;
        }

        //calculate quality parameters and save it as a double image
        recipe_error = sph_double_image_quality_check(self->science_di->image);
        if (recipe_error) {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "sph_double_image_quality_check returns cpl error.\n"
                            "cpl error code is: %d", recipe_error);
            sph_zpl_science_p1_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            return recipe_error;
        }

        // reference pixels for WCS
        xpix = cpl_image_get_size_x(self->science_di->image->iframe->image)/2.;
        ypix = cpl_image_get_size_y(self->science_di->image->iframe->image)/2.;

        if (self->rawframes) {
            SPH_INFO_MSG(
                    "Save science polarimetry P1 as a dfs product: "
                    "header info taken from the first raw frame.");
            cpl_propertylist * pl = sph_zpl_utils_get_camera_header(self->preproc_frames,
                    self->fgsraw->framesplus);
            //create WCS keywords and add to pl
            cpl_propertylist * pl_wcs = sph_zpl_utils_astrometry_create_wcs_pl(
                    (const cpl_frame*) cpl_frameset_get_first(self->fgsraw->framesplus),
                    (const double) xpix, (const double) ypix);
            if ( pl_wcs && pl){
                cpl_propertylist_append( pl, pl_wcs );
                cpl_propertylist_delete( pl_wcs ); pl_wcs = NULL;
            }

            sph_zpl_science_pol_product_set_filename(self->science_di, self->science_p1_outfilename);
            sph_zpl_science_pol_product_set_tag(self->science_di, self->eso_pro_catg);
            sph_zpl_science_pol_product_set_inherit_frame(self->science_di, cpl_frameset_get_first(self->fgsraw->framesplus));
            sph_zpl_science_pol_product_wrap_header(self->science_di, pl);

        } else if (self->preproc_frames) {

            cpl_propertylist * pl_wcs = sph_zpl_utils_astrometry_create_wcs_pl(
                    (const cpl_frame*) cpl_frameset_get_first(self->fgs->framesplus),
                    (const double) xpix, (const double) ypix);

            sph_zpl_science_pol_product_set_filename(self->science_di, self->science_p1_outfilename);
            sph_zpl_science_pol_product_set_tag(self->science_di, self->eso_pro_catg);
            sph_zpl_science_pol_product_set_inherit_frame(self->science_di, cpl_frameset_get_first(self->fgs->framesplus));
            sph_zpl_science_pol_product_wrap_header(self->science_di, pl_wcs);

        } else {
            SPH_ERR("Neither raw frames nor pre-processed frames provided!");
            recipe_error = sph_error_get_last_code();
        }

        if (recipe_error) {
            sph_error_raise(
                    SPH_ERROR_GENERAL,
                    __FILE__,
                    __func__,
                    __LINE__,
                    SPH_ERROR_ERROR,
                    "Couldn't save science_di product: sph_double_image_save_dfs returns cpl error.\n"
                            "cpl error code is: %d", recipe_error);
        }

        sph_zpl_science_p1_delete__(master_bias_quadimage,
                master_dark_quadimage, intensity_flat_quadimage,
                intensity_flat_masterframe, polarization_flat_doubleimage,
                modem_efficiency_masterframe);

        SPH_INFO_MSG("sph_zpl_science_p1_run...End");
        SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
    }
}



static cpl_error_code try_to_save_product(const sph_zpl_science_pol_product * prod,
		sph_zpl_science_p1 * recipe){

	if(prod == NULL) return CPL_ERROR_NONE;
	return sph_double_image_save_dfs(prod->image,
            prod->out_filename, recipe->inframes,
			prod->inherit_frame,
			recipe->inparams,
			prod->tag, SPH_RECIPE_NAME_ZPL_SCIENCE_P1,
            SPH_PIPELINE_NAME_ZIMPOL, prod->header);
}

typedef enum{
	SPH_ZPL_SCIENCE_CAMERA_P1_CAM1,
	SPH_ZPL_SCIENCE_CAMERA_P1_CAM2,
	SPH_ZPL_SCIENCE_CAMERA_P1_UNDEFINED
}sph_zpl_science_camera;

static cpl_boolean _sph_zpl_science_p1_plist_contains_value(const cpl_propertylist * list,
		const char * key, const char * expected_value){

	if(!cpl_propertylist_has(list, key)) return CPL_FALSE;
	const char * content = cpl_propertylist_get_string(list, key);

	return strcmp(content, expected_value) == 0;
}

static cpl_boolean
_sph_zpl_science_p1_is_ao_enabled_from_list(const cpl_propertylist * list){

	const char * ao_keywords[] = {SPH_PHOT_ZPL_AOS_HOLOOP,
			SPH_PHOT_ZPL_AOS_PUPLOOP, SPH_PHOT_ZPL_AOS_TTLOOP};

	const char * expected_ao_value = "CLOSED_LOOP";
	const char * expected_value_for_IRLOOP = "OPEN_LOOP";

	if(!_sph_zpl_science_p1_plist_contains_value(list,
			SPH_PHOT_ZPL_AOS_IRLOOP, expected_value_for_IRLOOP)){

		cpl_msg_warning(cpl_func, "%s was not set to %s", SPH_PHOT_ZPL_AOS_IRLOOP, expected_value_for_IRLOOP);
	}

	cpl_size sz = sizeof(ao_keywords) / sizeof(ao_keywords[0]);

	for(cpl_size i = 0; i < sz; ++i){
		if(!_sph_zpl_science_p1_plist_contains_value(list,
				ao_keywords[i], expected_ao_value)) return CPL_FALSE;
	}

	return CPL_TRUE;
}

static cpl_boolean
_sph_zpl_science_p1_is_ao_enabled(const cpl_frame * raw_frame){

	cpl_propertylist * list = cpl_propertylist_load(cpl_frame_get_filename(raw_frame), 0);
	cpl_boolean to_ret = _sph_zpl_science_p1_is_ao_enabled_from_list(list);
	cpl_propertylist_delete(list);
	return to_ret;
}

static double sph_master_frame_integrate(const sph_master_frame * image,
		sph_zpl_science_p1 * rec, const sph_zpl_science_camera camera,
		cpl_propertylist * pl_for_strehl_pars, cpl_boolean calc_strehl){

	if(image == NULL) return NAN;

	double flx = 0.0;
	const cpl_boolean is_cam1 = camera == SPH_ZPL_SCIENCE_CAMERA_P1_CAM1;
	if(!_sph_zpl_science_p1_is_ao_enabled(cpl_frameset_get_first(rec->rawframes)) ||
			rec->filter_frame == NULL || !calc_strehl){

		if(rec->filter_frame)
			cpl_msg_info(cpl_func, "AO is disabled, Strehl ratio computation will be skipped");
		else
			cpl_msg_info(cpl_func, "Filters table not provided, only flux will be calculated");

		double bkg = 0.0;

		const cpl_error_code fail = sph_strehl_disabled_ao_flux_zimpol(
			image->image, cpl_frameset_get_first(rec->rawframes), rec->inparams,
			SPH_RECIPE_NAME_ZPL_SCIENCE_P1, is_cam1, &flx, &bkg);

		if(fail){
			cpl_msg_warning(cpl_func, "Strehl flux estimation for disabled AO failed: %s",
					cpl_error_get_message());
			cpl_error_reset();
			return NAN;
		} else {
			cpl_msg_info(cpl_func, "Strehl flux estimation for disabled AO successful!");
		}

		if(pl_for_strehl_pars)
			sph_strehl_fill_qc_pars_zimpol_ao_disabled(pl_for_strehl_pars, flx, bkg);
	}
	else{

		sph_strehl_qc_pars qc_out = {0};

		cpl_msg_info(cpl_func, "AO is enabled, Strehl ratio computation will be performed");

		cpl_error_code fail = sph_strehl_zimpol(
			image->image, cpl_frameset_get_first(rec->rawframes), rec->filter_frame,
			SPH_RECIPE_NAME_ZPL_SCIENCE_P1, rec->inparams, is_cam1, &qc_out);

		if(fail){
			cpl_msg_warning(cpl_func, "Strehl estimation failed: %s", cpl_error_get_message());
			cpl_error_reset();
			return NAN;
		} else {
			cpl_msg_info(cpl_func, "Strehl estimation successful!");
		}
		flx = qc_out.star_flux;

		if(pl_for_strehl_pars)
			sph_strehl_fill_qc_pars_zimpol(pl_for_strehl_pars, &qc_out);

	}
	return flx;
}



static char * get_key(const char * qc_name,
		const sph_zpl_science_camera camera){

	if(camera == SPH_ZPL_SCIENCE_CAMERA_P1_CAM1)
			return cpl_sprintf("%s %s", qc_name, "CAM1");
	if(camera == SPH_ZPL_SCIENCE_CAMERA_P1_CAM2)
			return cpl_sprintf("%s %s", qc_name, "CAM2");

	return cpl_strdup(qc_name);
}

static void append_par(cpl_propertylist * plist, const char * qc_name,
		const sph_zpl_science_camera camera, const double val){

	if(!plist || !qc_name || strlen(qc_name) == 0) return;

	if(!isfinite(val))
	{
		cpl_msg_warning(cpl_func, "Unable to write parameter %s: the value is invalid", qc_name);
		return;
	}
	char * key = get_key(qc_name, camera);

	cpl_propertylist_update_double(plist, key, val);

	cpl_free(key);
}


typedef enum{
	SPH_ZPL_SCIENCE_STAR_TYPE_NON_STD_POL,
	SPH_ZPL_SCIENCE_STAR_TYPE_STD_POL_ZERO,
	SPH_ZPL_SCIENCE_STAR_TYPE_STD_POL_HIGH
}sph_zpl_science_star_type;

static cpl_frameset * get_input_frames(cpl_frameset * raws,
		cpl_frameset * preproc){

	if(raws) return raws;

	return preproc;
}

static cpl_boolean try_get_ra_dec(const cpl_frame * f, double * ra, double * dec){

	cpl_propertylist * pl = cpl_propertylist_load(cpl_frame_get_filename(f), 0);
	if(!pl) return CPL_FALSE;

	const cpl_boolean has_props = cpl_propertylist_has( pl, SPH_COMMON_KEYWORD_WCS_RA) &&
			cpl_propertylist_has(pl, SPH_COMMON_KEYWORD_WCS_DEC);

	if (has_props) {
		*ra = cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_WCS_RA);
		*dec = cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_WCS_DEC);
	}

	return has_props && isfinite(*ra) && isfinite(*dec);
}

static cpl_boolean has_table_string_column(const cpl_table * tb, const char * name){

	if(!cpl_table_has_column(tb, name)) return CPL_FALSE;

	if(cpl_table_get_column_type(tb, name) != CPL_TYPE_STRING) return CPL_FALSE;
	return CPL_TRUE;
}

static cpl_boolean has_table_double_column(const cpl_table * tb, const char * name){

	if(!cpl_table_has_column(tb, name)) return CPL_FALSE;

	if(cpl_table_get_column_type(tb, name) != CPL_TYPE_DOUBLE) return CPL_FALSE;
	return CPL_TRUE;
}

static inline double
cpl_table_get_double_or_NAN(const cpl_table * tb, const char * clmn_name, const cpl_size i){
	int rej = 0;

	const double data = cpl_table_get_double(tb, clmn_name, i, &rej);
	if(rej)
		return NAN;
	return data;
}


static cpl_boolean try_extract_expected_from_catg(cpl_frame * catg_f,
		const char * filter, const double ra_sci, const double dec_sci,
		double * expected_p, double * expected_theta){

	const double max_dist_acceptable = 5.0 / 60.0; //5 arcmins

	if(!catg_f || !filter || strlen(filter) < 1) return CPL_FALSE;

	const char * allowed[] = { "V", "N_I", "N_R" };  //allowed filters
	size_t iter = 0, nelems = sizeof(allowed) / sizeof(allowed[0]);
	for(; iter < nelems; ++iter) if(!strcmp(allowed[iter], filter)) break;
	if(iter >= nelems) return CPL_FALSE;  //filter not allowed

	const int mark_invalid_values = 1;
	const char * fname = cpl_frame_get_filename(catg_f);
	cpl_table * tb = cpl_table_load(fname, 1, mark_invalid_values);

	if(!tb) return CPL_FALSE;

	const char * name_clmn_name = "HD";
	const char * ra_clmn_name = "RA";
	const char * dec_clmn_name = "DEC";
	char p_clmn_name[] = "p_?";
	char theta_clmn_name[] = "theta_?";

	//replace '?' with filter char (last char of one of the following: V, N_I, N_R)
	theta_clmn_name[strlen(theta_clmn_name) - 1] = p_clmn_name[strlen(p_clmn_name) - 1]
					= filter[strlen(filter) - 1];

	const cpl_boolean clmns_ok = has_table_double_column(tb, ra_clmn_name) &&
					has_table_double_column(tb, dec_clmn_name) &&
					has_table_double_column(tb, p_clmn_name) &&
					has_table_double_column(tb, theta_clmn_name) &&
					has_table_string_column(tb, name_clmn_name);

	if(!clmns_ok){
		cpl_table_delete(tb);
		return CPL_FALSE;
	}

	cpl_size best_idx = -1;
	double best_distance = INFINITY;
	for(cpl_size i = 0; i < cpl_table_get_nrow(tb); ++i){

		const double ra = cpl_table_get_double(tb, ra_clmn_name, i, NULL);
		const double dec = cpl_table_get_double(tb, dec_clmn_name, i, NULL);

		const double new_distance = sph_ulils_great_circle_dist(ra_sci, dec_sci, ra, dec);

		if(new_distance < best_distance && new_distance < max_dist_acceptable){
			best_idx = i;
			best_distance = new_distance;
		}
	}

	const cpl_boolean found = best_idx >= 0;
	if(found){
		*expected_p = cpl_table_get_double_or_NAN(tb, p_clmn_name, best_idx);
		*expected_theta = cpl_table_get_double_or_NAN(tb, theta_clmn_name, best_idx);
		const char * name = cpl_table_get_string(tb, name_clmn_name, best_idx);
		cpl_msg_info(cpl_func, "QC for high polarization, standard star %s", name);
	} else {
		cpl_msg_warning(cpl_func, "QC for high polarization, standard star not found");
	}

	cpl_table_delete(tb);
	return found;
}

static cpl_boolean try_get_expected_p_theta(sph_zpl_science_p1 * rec,
		const char * filter, double * p, double * theta){

	cpl_frame * catg = rec->high_polarization_std_stars_catalog;

	if(catg == NULL){
		cpl_msg_warning(cpl_func, "High polarization catalog for "
				"standard stars not provided, QC offset parameters will not be produced");
		return CPL_FALSE;
	}

	const cpl_frameset * frames = get_input_frames(rec->rawframes,
			rec->preproc_frames);
	if(frames == NULL || cpl_frameset_get_size(frames) == 0) return CPL_FALSE;

	const cpl_frame * sci_fr = cpl_frameset_get_position_const(frames, 0);

	if(sci_fr == NULL) return CPL_FALSE;

	double ra = NAN;
	double dec = NAN;

	if(!try_get_ra_dec(sci_fr, &ra, &dec)) return CPL_FALSE;

	return try_extract_expected_from_catg(catg, filter, ra, dec, p, theta);
}


const char * correction_table_filter_clmn = "Filter";
const char * correction_table_d_theta_clmn = "d_Theta";
const char * correction_table_d_p_clmn = "d_P";

static inline cpl_boolean
is_corr_table_correct(const cpl_table * tb){

	if(!tb) return CPL_FALSE;

	return has_table_string_column(tb, correction_table_filter_clmn) &&
		 has_table_double_column(tb, correction_table_d_theta_clmn) &&
		 has_table_double_column(tb, correction_table_d_p_clmn);
}


static inline cpl_boolean
try_get_corrections_from_tb(const cpl_table * corr_tb,
			const char * filter, double * t_corr, double * p_corr){

	if(!is_corr_table_correct(corr_tb) || filter == NULL){
			return CPL_FALSE;
	}

	for(cpl_size i = 0; i < cpl_table_get_nrow(corr_tb); ++i){
		const char * filter_this = cpl_table_get_string(corr_tb, correction_table_filter_clmn, i);
		if(filter_this != NULL && !strcmp(filter_this, filter)){
			*t_corr = cpl_table_get_double_or_NAN(corr_tb, correction_table_d_theta_clmn, i);
			*p_corr = cpl_table_get_double_or_NAN(corr_tb, correction_table_d_p_clmn, i);
			return CPL_TRUE;
		}
	}

	cpl_msg_warning(cpl_func, "Unable to find correction values for filter %s, "
			"QC parameters for Polarimetric standard stars will not be written", filter);

	return CPL_FALSE;
}



static inline cpl_boolean
try_get_corrections(const cpl_frame * raw_f, const sph_zpl_science_camera camera,
		const cpl_frame * correction_table_f, const char ** pfilter,
		double * t_corr, double * p_corr){

	if(raw_f == NULL || correction_table_f == NULL || pfilter == NULL) return CPL_FALSE;

	cpl_propertylist * rawplist = cpl_propertylist_load(cpl_frame_get_filename(raw_f), 0);

	if(!rawplist) return CPL_FALSE;

	sph_arm arm = SPH_ARM_ZIMPOL1;
	if(camera == SPH_ZPL_SCIENCE_CAMERA_P1_CAM2)
		arm = SPH_ARM_ZIMPOL2;

	double pscale;
	const char * key;

	sph_strehl_read_filter_and_scale(rawplist, arm, &pscale, &key, pfilter);

	cpl_table * corr_tb = cpl_table_load(cpl_frame_get_filename(correction_table_f), 1, 0);


	const cpl_boolean success = try_get_corrections_from_tb(corr_tb,
			*pfilter, t_corr, p_corr);

	cpl_propertylist_delete(rawplist);
	cpl_table_delete(corr_tb);
	return success;
}

static inline cpl_boolean has_double_property(const cpl_propertylist * plist, const char * name){

	return plist != NULL && cpl_propertylist_has(plist, name) &&
				cpl_propertylist_get_type(plist, name) == CPL_TYPE_DOUBLE;

}

static inline
cpl_boolean try_get_parallatic(const cpl_frame * f0, const cpl_frame * f1, double * parallax){

	const char * start = "ESO TEL PARANG START";
	const char * end = "ESO TEL PARANG END";

	cpl_propertylist * plist0 = cpl_propertylist_load(cpl_frame_get_filename(f0), 0);
	cpl_propertylist * plist1 = cpl_propertylist_load(cpl_frame_get_filename(f1), 0);

	if((plist0 == NULL) || (plist1 == NULL))
		return CPL_FALSE;

	const cpl_boolean success = has_double_property(plist0, start) &&
			has_double_property(plist1, end);
	if(success){
		const double str = cpl_propertylist_get_double(plist0, start);
		const double nd = cpl_propertylist_get_double(plist1, end);
		*parallax = (str + nd) / 2.0;
	}
	cpl_propertylist_delete(plist0);
	cpl_propertylist_delete(plist1);
	return success && isfinite(*parallax);
}

static void add_polarimetry_qc_pars(sph_zpl_science_pol_product * q_prod,
		sph_zpl_science_pol_product * u_prod,
		const sph_zpl_science_camera camera,
		const sph_zpl_science_star_type pol_type,
		sph_zpl_science_p1 * rec){

	if(rec->std_star_pol_correction == NULL) {
		cpl_msg_warning(cpl_func, "No correction table provided, QC parameters "
				"for Polarimetric Standard Stars will not be produced");
		return;
	}
	if(camera == SPH_ZPL_SCIENCE_CAMERA_P1_UNDEFINED) return;

	double theta_corr = .0;
	double p_corr = .0;
	const char * filter = NULL;

	if(!try_get_corrections(cpl_frameset_get_position_const(rec->rawframes, 0), camera,
				rec->std_star_pol_correction, &filter, &theta_corr, &p_corr) || !filter){
		cpl_msg_warning(cpl_func, "Unable to extract corrections for P and THETA, "
				"aborting Polarimetric Std Stars QC generation");
		return;
	}

	//immediately make a local copy of filter: further calls to sph_strehl_*
	//functions (indirectly called below) can change what it points to
	char copy[32] = { '\0' };
	strncpy(copy, filter, sizeof(copy)-1);  //WARNING: filter truncated if longer than copy buffer
	filter = copy;

	double theta_parallatic = .0;
	cpl_size n_raw_frames = cpl_frameset_get_size(rec->rawframes);
	if(!try_get_parallatic(cpl_frameset_get_position_const(rec->rawframes, 0),
			       cpl_frameset_get_position_const(rec->rawframes, n_raw_frames-1),
			       &theta_parallatic)){
		cpl_msg_warning(cpl_func, "Unable to extract parallatic angle, "
				"aborting Polarimetric Std Stars QC generation");
		return;
	}

	if(q_prod == NULL || q_prod->image == NULL) return;
	if(u_prod == NULL || u_prod->image == NULL) return;

	const double u_p = sph_master_frame_integrate(u_prod->image->pframe, rec, camera, NULL, CPL_FALSE);
	const double u_i = sph_master_frame_integrate(u_prod->image->iframe, rec, camera, u_prod->header, CPL_TRUE);

	const double q_p = sph_master_frame_integrate(q_prod->image->pframe, rec, camera, NULL, CPL_FALSE);
	const double q_i = sph_master_frame_integrate(q_prod->image->iframe, rec, camera, q_prod->header, CPL_TRUE);

	const double u_uncorr = u_p / u_i;
	const double q_uncorr = q_p / q_i;

	const double corr_u = p_corr / 100.0 * sin(2.0 * CPL_MATH_RAD_DEG * (theta_corr + theta_parallatic));
	const double corr_q = p_corr / 100.0 * cos(2.0 * CPL_MATH_RAD_DEG * (theta_corr + theta_parallatic));

	const double u = 100.0 * (u_uncorr - corr_u);
	const double q = 100.0 * (q_uncorr - corr_q);

	const double p = hypot(q, u);
	// Determine the atan solution in radians
	const double rad = atan2(u, q);
	// Normalise so that the angle is positive before converting to degrees, and halving
	const double theta = (rad >= 0 ? rad : rad + CPL_MATH_2PI) * CPL_MATH_DEG_RAD / 2.0;

	append_par(q_prod->header, SPH_ZPL_KEYWORD_ESO_QC_POL_Q, camera, q);
	append_par(q_prod->header, SPH_ZPL_KEYWORD_ESO_QC_POL_P, camera, p);
	append_par(q_prod->header, SPH_ZPL_KEYWORD_ESO_QC_POL_THETA, camera, theta);

	append_par(u_prod->header, SPH_ZPL_KEYWORD_ESO_QC_POL_U, camera, u);
	append_par(u_prod->header, SPH_ZPL_KEYWORD_ESO_QC_POL_P, camera, p);
	append_par(u_prod->header, SPH_ZPL_KEYWORD_ESO_QC_POL_THETA, camera, theta);

	if(pol_type != SPH_ZPL_SCIENCE_STAR_TYPE_STD_POL_HIGH) return;

	double expected_p = NAN;
	double expected_theta = NAN;

	if(!try_get_expected_p_theta(rec, filter, &expected_p, &expected_theta)){
		cpl_msg_warning(cpl_func, "Unable to extract expected Theta and P from Catalog, "
				"Offset QC parameters for polarimetry will not be written");
		return;
	}

	if(isfinite(expected_p)){
		const double p_off = p - expected_p;
		append_par(u_prod->header, SPH_ZPL_KEYWORD_ESO_QC_POL_P_OFFSET, camera, p_off);
	} else {
		cpl_msg_warning(cpl_func, "Expected P unavailable, P OFFSET will be not created");
	}

	if(isfinite(expected_theta)){
		double theta_off = theta - expected_theta;
		if (theta_off < -90) {
			theta_off += 180;
		}
		if (theta_off > 90) {
			theta_off -= 180;
		}
		append_par(u_prod->header, SPH_ZPL_KEYWORD_ESO_QC_POL_THETA_OFFSET, camera, theta_off);
	} else {
		cpl_msg_warning(cpl_func, "Expected THETA unavailable, THETA OFFSET will be not created");
	}
}

static inline
sph_zpl_science_star_type
translate_sph_zpl_science_star_type_from_string(const char * s){

	if(s == NULL)
		return SPH_ZPL_SCIENCE_STAR_TYPE_NON_STD_POL;

	if(!strcmp(s, "STD,POL0"))
		return SPH_ZPL_SCIENCE_STAR_TYPE_STD_POL_ZERO;

	if(!strcmp(s, "STD,POLHIGH"))
		return SPH_ZPL_SCIENCE_STAR_TYPE_STD_POL_HIGH;

	return SPH_ZPL_SCIENCE_STAR_TYPE_NON_STD_POL;
}

static sph_zpl_science_star_type
get_polarimetric_std_star_type(const cpl_frameset * raw_frames){

	if(raw_frames == NULL) return SPH_ZPL_SCIENCE_STAR_TYPE_NON_STD_POL;

	if(cpl_frameset_get_size(raw_frames) == 0) return SPH_ZPL_SCIENCE_STAR_TYPE_NON_STD_POL;

	const cpl_frame * f1 = cpl_frameset_get_position_const(raw_frames, 0);

	if(f1 == NULL) return SPH_ZPL_SCIENCE_STAR_TYPE_NON_STD_POL;

	const char * fname = cpl_frame_get_filename(f1);
	cpl_propertylist * header = cpl_propertylist_load(fname, 0);

	if(header == NULL){
		cpl_error_reset();
		return SPH_ZPL_SCIENCE_STAR_TYPE_NON_STD_POL;
	}

	sph_zpl_science_star_type pol_type = SPH_ZPL_SCIENCE_STAR_TYPE_NON_STD_POL;

	const cpl_boolean has_prop_and_is_string = cpl_propertylist_has(header, SPH_COMMON_KEYWORD_DPR_TYPE) &&
			cpl_propertylist_get_type(header, SPH_COMMON_KEYWORD_DPR_TYPE) == CPL_TYPE_STRING;

	if(has_prop_and_is_string){
		const char * type = cpl_propertylist_get_string(header, SPH_COMMON_KEYWORD_DPR_TYPE);
		pol_type = translate_sph_zpl_science_star_type_from_string(type);
	}

	cpl_propertylist_delete(header);
	return pol_type;
}

static cpl_error_code save_products_with_qc_pars(sph_zpl_science_p1 * rec){

	cpl_error_code err = CPL_ERROR_NONE;

	const sph_zpl_science_star_type std_star_pol_type =
			get_polarimetric_std_star_type(get_input_frames(rec->rawframes,
			rec->preproc_frames));

	if(std_star_pol_type != SPH_ZPL_SCIENCE_STAR_TYPE_NON_STD_POL){
		add_polarimetry_qc_pars(rec->science_di_q_cam1, rec->science_di_u_cam1,
				SPH_ZPL_SCIENCE_CAMERA_P1_CAM1, std_star_pol_type, rec);
		add_polarimetry_qc_pars(rec->science_di_q_cam2, rec->science_di_u_cam2,
				SPH_ZPL_SCIENCE_CAMERA_P1_CAM2, std_star_pol_type, rec);
	}

	err |= try_to_save_product(rec->science_di_q, rec);
	err |= try_to_save_product(rec->science_di_q_cam1, rec);
	err |= try_to_save_product(rec->science_di_q_cam2, rec);

	err |= try_to_save_product(rec->science_di_q_minus, rec);
	err |= try_to_save_product(rec->science_di_q_minus_cam1, rec);
	err |= try_to_save_product(rec->science_di_q_minus_cam2, rec);

	err |= try_to_save_product(rec->science_di_q_plus, rec);
	err |= try_to_save_product(rec->science_di_q_plus_cam1, rec);
	err |= try_to_save_product(rec->science_di_q_plus_cam2, rec);

	err |= try_to_save_product(rec->science_di_u, rec);
	err |= try_to_save_product(rec->science_di_u_cam1, rec);
	err |= try_to_save_product(rec->science_di_u_cam2, rec);

	err |= try_to_save_product(rec->science_di_u_minus, rec);
	err |= try_to_save_product(rec->science_di_u_minus_cam1, rec);
	err |= try_to_save_product(rec->science_di_u_minus_cam2, rec);

	err |= try_to_save_product(rec->science_di_u_plus, rec);
	err |= try_to_save_product(rec->science_di_u_plus_cam1, rec);
	err |= try_to_save_product(rec->science_di_u_plus_cam2, rec);

	return err;
}


/**@}*/

