/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_create_super_flat.h"

#include "sph_pixel_polyfit.h"
#include "sph_common_keywords.h"

/*-----------------------------------------------------------------------------
                              Private function declarations
 -----------------------------------------------------------------------------*/

static sph_pixel_polyfit* sph_create_flat_fit(const sph_master_frame*,
                                              const sph_master_frame*,
                                              const sph_master_frame*,
                                              const sph_master_frame*,
                                              int)
    CPL_ATTR_ALLOC;

/*-----------------------------------------------------------------------------
                              Function definitions
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 @brief Create flat field polynomial pixel fit.
 @param longdff1       A long DFF 1 master frame
 @param longdff2       A long DFF 2 master frame
 @param longdff3       A long DFF 3 master frame
 @param longdff4       A long DFF 4 master frame, or NULL
 @param order          The order of the fit
 @return The created object or NULL on error

 Description: Create a polynomial fit to the wavelength dependence of the
 flat field for all pixels. The resulting frame is a sph_pixel_polyfit object.
 The calculated flat field fit is normalised (divided by the long broad band flat).

 */
/*----------------------------------------------------------------------------*/

static sph_pixel_polyfit*
sph_create_flat_fit(const sph_master_frame* longdff1,
                    const sph_master_frame* longdff2,
                    const sph_master_frame* longdff3,
                    const sph_master_frame* longdff4,
                    int order) {
    sph_pixel_polyfit* fittab = NULL;
    cpl_vector* lambdas = NULL;
    cpl_imagelist* imlist = NULL;
    cpl_imagelist* imcoeffs = NULL;
    cpl_image* errimage = NULL;

    cpl_ensure( longdff1 && longdff2 && longdff3, CPL_ERROR_NULL_INPUT, NULL);

    cpl_ensure( cpl_error_get_code() == CPL_ERROR_NONE, cpl_error_get_code(),
            NULL);

    cpl_ensure(
            cpl_propertylist_has(longdff1->properties,SPH_COMMON_KEYWORD_CALIB_LAMBDA),
            CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(
            cpl_propertylist_has(longdff2->properties,SPH_COMMON_KEYWORD_CALIB_LAMBDA),
            CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(
            cpl_propertylist_has(longdff3->properties,SPH_COMMON_KEYWORD_CALIB_LAMBDA),
            CPL_ERROR_ILLEGAL_INPUT, NULL);
    if (longdff4) {
        cpl_ensure(
                cpl_propertylist_has(longdff4->properties,SPH_COMMON_KEYWORD_CALIB_LAMBDA),
                CPL_ERROR_ILLEGAL_INPUT, NULL);
        lambdas = cpl_vector_new(4);
    } else
        lambdas = cpl_vector_new(3);
    cpl_vector_set(
            lambdas,
            0,
            cpl_propertylist_get_double(longdff1->properties,
                    SPH_COMMON_KEYWORD_CALIB_LAMBDA));
    cpl_vector_set(
            lambdas,
            1,
            cpl_propertylist_get_double(longdff2->properties,
                    SPH_COMMON_KEYWORD_CALIB_LAMBDA));
    cpl_vector_set(
            lambdas,
            2,
            cpl_propertylist_get_double(longdff3->properties,
                    SPH_COMMON_KEYWORD_CALIB_LAMBDA));
    if (longdff4) {
        cpl_vector_set(
                lambdas,
                3,
                cpl_propertylist_get_double(longdff4->properties,
                        SPH_COMMON_KEYWORD_CALIB_LAMBDA));
    }
    imlist = cpl_imagelist_new();
    cpl_imagelist_set(imlist, sph_master_frame_extract_image(longdff1, 1), 0);
    cpl_imagelist_set(imlist, sph_master_frame_extract_image(longdff2, 1), 1);
    cpl_imagelist_set(imlist, sph_master_frame_extract_image(longdff3, 1), 2);
    if (longdff4) {
        cpl_imagelist_set(imlist, sph_master_frame_extract_image(longdff4, 1),
                3);
    }

    errimage = cpl_image_new(cpl_image_get_size_x(cpl_imagelist_get(imlist, 0)),
            cpl_image_get_size_y(cpl_imagelist_get(imlist, 0)),
            CPL_TYPE_DOUBLE);
    imcoeffs = cpl_fit_imagelist_polynomial(lambdas, imlist, 0, order,
            CPL_FALSE, CPL_TYPE_DOUBLE, errimage);
    cpl_ensure(cpl_error_get_code()== CPL_ERROR_NONE, cpl_error_get_code(),
            NULL);
    fittab = sph_pixel_polyfit_wrap(imcoeffs, errimage);
#ifdef SPH_SAVE_NONDFS
    sph_pixel_polyfit_table_save_cube(fittab, "fittab.fits");
#endif
    cpl_imagelist_delete(imlist);
    cpl_vector_delete(lambdas);
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return fittab;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Create a master frame with a wavelength dependent flat field.
 @param self           The pixel description table
 @param longdff1       A long DFF 1 master frame
 @param longdff2       A long DFF 2 master frame
 @param longdff3       A long DFF 3 master frame
 @param longdff4       A long DFF 4 master frame, or NULL
 @return The created master frame or NULL on error

 Description: Create a wavelength dependent flat field from a PDT, a set of
 long DFF. Each pixel value is assigned a flat field value
 that is calculated from the wavelength associated with that pixel from the PDT
 and the value calculated from the pixel fit.

 */
/*----------------------------------------------------------------------------*/
sph_master_frame*
sph_create_super_flat(const sph_pixel_description_table* self,
                      const sph_master_frame* longdff1,
                      const sph_master_frame* longdff2,
                      const sph_master_frame* longdff3,
                      const sph_master_frame* longdff4)
{
    sph_pixel_polyfit* pft;
    sph_master_frame* flat;
    const double* perr;
    double * pimg;
    double * prms;
    double * pncomb;
    int    * pbad;

    cpl_ensure(!cpl_error_get_code(), cpl_error_get_code(), NULL);
    cpl_ensure(self     != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(longdff1 != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(longdff2 != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(longdff3 != NULL, CPL_ERROR_NULL_INPUT, NULL);

    pft = sph_create_flat_fit(longdff1, longdff2, longdff3, longdff4, 2);

    if (!pft) {
        return NULL;
    }

    flat = sph_master_frame_new(self->nx, self->ny);
    if (!flat) {
        sph_pixel_polyfit_delete(pft);
        return NULL;
    }

    perr = pft->error ? cpl_image_get_data_double_const(pft->error) : NULL;

    pimg   = cpl_image_get_data_double(flat->image);
    prms   = cpl_image_get_data_double(flat->rmsmap);
    pncomb = cpl_image_get_data_double(flat->ncombmap);
    pbad   = cpl_image_get_data_int(flat->badpixelmap);

    for (int yy = 0; yy < self->ny; ++yy) {
        for (int xx = 0; xx < self->nx; ++xx) {
            const sph_pixel_descriptor* pdescr = 
                sph_pixel_description_table_get_descriptor_const(self, xx, yy);

            if (pdescr && pdescr->specid) {
                cpl_size k = pft->order;

                pimg[xx + yy * self->nx] = pft->pfit[k--][xx + yy * self->nx];

                while (k >= 0) {
                    pimg[xx + yy * self->nx] =
                        pimg[xx + yy * self->nx] * pdescr->wavelength +
                        pft->pfit[k--][xx + yy * self->nx];
                }

                pncomb[xx + yy * self->nx] = 1.0;
                pbad  [xx + yy * self->nx] = 0;
                prms  [xx + yy * self->nx] =
                    perr != NULL ? perr[xx + yy * self->nx] : 0.0;

            } else {
                pimg  [xx + yy * self->nx] = 0.0;
                pncomb[xx + yy * self->nx] = 0.0;
                pbad  [xx + yy * self->nx] = 1;
                prms  [xx + yy * self->nx] = 0.0; /* SPH_MASTER_FRAME_BAD_RMS */
            }
        }
    }

    sph_pixel_polyfit_delete(pft);

    return flat;
}
