/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/
#include "sph_utils.h"
#include "sph_error.h"
#include "sph_keyword_manager.h"
#include "sph_distortion_map.h"
#include "sph_distortion_model.h"
#include "sph_double_image.h"
#include "sph_fitting.h"
#include "sph_common_keywords.h"
#include <cpl.h>
#include <string.h>
#include <assert.h>
/*----------------------------------------------------------------------------*/
/**
 * @defgroup A SPHERE API Module
 * @par Synopsis:
 * @code
 * typedef _module_ {
 * } module
 * @endcode
 * @par Desciption:
 *
 * This module provides ...
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*-----------------------------------------------------------------------------
 Definition of Functions
 -----------------------------------------------------------------------------*/

static cpl_error_code
sph_distortion_model_fit_distortion_map_1(sph_distortion_model*,
                                          const sph_distortion_map*,
                                          cpl_size, cpl_boolean,
                                          double*,
                                          double*);

static sph_distortion_model*
sph_distortion_model_new_empty(void) CPL_ATTR_ALLOC;

static sph_distortion_model*
sph_distortion_model_new_empty(void)
{
    return (sph_distortion_model*)
        cpl_calloc(1, sizeof(sph_distortion_model));
}


sph_distortion_model*
sph_distortion_model_new(const cpl_polynomial* polyx,
                         const cpl_polynomial* polyy)
{
    sph_distortion_model* self = sph_distortion_model_new_empty();

    self->polyx =
        polyx ? cpl_polynomial_duplicate(polyx) : cpl_polynomial_new(2);

    self->polyy =
        polyy ? cpl_polynomial_duplicate(polyy) : cpl_polynomial_new(2);

    self->workv = cpl_vector_wrap(2, self->workxy);

    return self;
}

cpl_error_code
sph_distortion_model_set_coeff_x(
        sph_distortion_model* self,
        cpl_size order_x,
        cpl_size order_y,
        double value)
{
    const cpl_size pows[2] = {order_x, order_y};

    return cpl_polynomial_set_coeff(self->polyx, pows, value) 
        ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;
}

cpl_error_code
sph_distortion_model_set_coeff_y(
        sph_distortion_model* self,
        cpl_size order_x,
        cpl_size order_y,
        double value)
{
    const cpl_size pows[2] = {order_x, order_y};

    return cpl_polynomial_set_coeff(self->polyy, pows, value) 
        ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;
}

cpl_error_code
sph_distortion_model_add_coeff_x(
        sph_distortion_model* self,
        cpl_size order_x,
        cpl_size order_y,
        double value)
{
    const cpl_size pows[2] = {order_x, order_y};

    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);

    value += cpl_polynomial_get_coeff(self->polyx, pows);

    return cpl_polynomial_set_coeff(self->polyx, pows, value) 
        ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;
}

cpl_error_code
sph_distortion_model_add_coeff_y(
        sph_distortion_model* self,
        cpl_size order_x,
        cpl_size order_y,
        double value)
{
    const cpl_size pows[2] = {order_x, order_y};

    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);

    value += cpl_polynomial_get_coeff(self->polyy, pows);

    return cpl_polynomial_set_coeff(self->polyx, pows, value) 
        ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Transform a point pattern via a distortion model
 * @param self  The distortion model used for the transformation
 * @param pp    The point pattern to transform
 * @return On success the created point pattern, on NULL on error
 * @see cpl_image_warp()
 * @note FIXME: Distortion offset ignored for now
 */
/*----------------------------------------------------------------------------*/
sph_point_pattern*
sph_distortion_model_apply_pp(const sph_distortion_model* self,
                              const sph_point_pattern* pp)
{
    const cpl_size      np     = sph_point_pattern_get_size(pp);
    sph_point_pattern * result;

    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(pp   != NULL, CPL_ERROR_NULL_INPUT, NULL);

    result = sph_point_pattern_new_(np);

    for (cpl_size i = 0; i < np; i++) {
        const double* xyi = sph_point_pattern_get(pp, i);
        double distxy[2];

        sph_distortion_model_get_dist(self, xyi, distxy);

        sph_point_pattern_add_point(result,
                                    xyi[0] - distxy[0],
                                    xyi[1] - distxy[1]);

        if (distxy[0] * distxy[0] + distxy[1] * distxy[1] > 49.0)
            cpl_msg_warning(cpl_func, "Ref-point distorted source(%d): "
                            "(%g,%g) - (%g,%g) = (%g,%g)",
                            (int)i, xyi[0], xyi[1], distxy[0], distxy[1],
                            xyi[0] - distxy[0], xyi[1] - distxy[1]);
    }

    return result;
}

sph_distortion_model*
sph_distortion_model_new_from_proplist(const cpl_propertylist* proplist,
                                                  const char* c_x_tag,
                                                  const char* c_y_tag) {
    sph_distortion_model* result = NULL;
    const cpl_size n = cpl_propertylist_get_size(proplist);
    char* form_x =
        cpl_sprintf("%s %%" CPL_SIZE_FORMAT "_%%" CPL_SIZE_FORMAT, c_x_tag);
    char* form_y =
        cpl_sprintf("%s %%" CPL_SIZE_FORMAT "_%%" CPL_SIZE_FORMAT, c_y_tag);
    cpl_size i;

    cpl_ensure(proplist != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(c_x_tag  != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(c_y_tag  != NULL, CPL_ERROR_NULL_INPUT, NULL);

    result = sph_distortion_model_new(NULL, NULL);

    for (i = 0; i < n; i++) {
        const cpl_property *p = cpl_propertylist_get_const(proplist, i);
        const char *key = cpl_property_get_name(p);

        if (!strcmp(key, "NAXIS1")) {
            const int value = cpl_property_get_int(p);
            const double offset = 0.5 + 0.5 * value;

            result->offx = offset;

        } else if (!strcmp(key, "NAXIS2")) {
            const int value = cpl_property_get_int(p);
            const double offset = 0.5 + 0.5 * value;

            result->offy = offset;

        } else if (cpl_property_get_type(p) == CPL_TYPE_DOUBLE) {
            const double value = cpl_property_get_double(p);
            cpl_size pows[2];

            assert(key != NULL);

CPL_DIAG_PRAGMA_PUSH_IGN(-Wformat-nonliteral);
            if (sscanf(key, form_x, pows, pows+1) == 2) {
                cpl_polynomial_set_coeff(result->polyx, pows, value);
            } else if (sscanf(key, form_y, pows, pows+1) == 2) {
                cpl_polynomial_set_coeff(result->polyy, pows, value);
            }
CPL_DIAG_PRAGMA_POP;
        }

    }

    cpl_free(form_x);
    cpl_free(form_y);

    cpl_msg_info(cpl_func, "Distortion %d-degree X-model (%s) has center: %g",
                 (int)cpl_polynomial_get_degree(result->polyx), c_x_tag,
                 result->offx);
    cpl_msg_info(cpl_func, "Distortion %d-degree Y-model (%s) has center: %g",
                 (int)cpl_polynomial_get_degree(result->polyy), c_y_tag,
                 result->offy);

    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Load a distortion map from a FITS header
 * @param filename      The name of the FITS file, type IRD_DISTORTION_MAP
 * @param ext           The extension number (0 for left, 8 for right)
 * @param c_x_tag       The FITS key X-pattern
 * @param c_x_tag       The FITS key Y-pattern
 * @return a new polyonomial distortion model
 *
 */
/*----------------------------------------------------------------------------*/
sph_distortion_model*
sph_distortion_model_load(const char* filename,
                          int ext,
                          const char* c_x_tag,
                          const char* c_y_tag) {
    sph_distortion_model* result = NULL;
    char* regexp = cpl_sprintf("%s|%s|NAXIS1|NAXIS2", c_x_tag, c_y_tag);
    cpl_propertylist* proplist = NULL;

    proplist = cpl_propertylist_load_regexp(filename, ext, regexp, 0);
    result = sph_distortion_model_new_from_proplist(proplist,
                                                    c_x_tag,
                                                    c_y_tag);
    cpl_propertylist_delete(proplist);
    cpl_free(regexp);
    if (result == NULL) (void)cpl_error_set_where(cpl_func);

    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Load a distortion map and optical axis from a FITS header
 * @param filename      The name of the FITS file, type IRD_DISTORTION_MAP
 * @param ext           The extension number (0 for left, 8 for right)
 * @param c_x_tag       The FITS key X-pattern
 * @param c_x_tag       The FITS key Y-pattern
 * @param o_x_key       The FITS key X-optical axis key
 * @param o_y_key       The FITS key Y-optical axis key
 * @param popt_cent_x   On return, the optical X-axis
 * @param popt_cent_y   On return, the optical Y-axis
 * @param palign_right  On return, non-zero if the optical axes are aligned
 * @return a new polyonomial distortion model
 * @see sph_distortion_model_load()
 *
 */
/*----------------------------------------------------------------------------*/
sph_distortion_model*
sph_distortion_model_load_opt(const char* filename,
                              int ext,
                              const char* c_x_tag,
                              const char* c_y_tag,
                              const char* o_x_key,
                              const char* o_y_key,
                              double * popt_cent_x,
                              double * popt_cent_y,
                              int * palign_right) {
    sph_distortion_model* result = NULL;
    char* regexp = cpl_sprintf(palign_right != NULL ?
                               "%s|%s|NAXIS1|NAXIS2|%s|%s|"
                               "ESO PRO REC.+ PARAM.+ (NAME|VALUE)" :
                               "%s|%s|NAXIS1|NAXIS2|%s|%s",
                               c_x_tag, c_y_tag, o_x_key, o_y_key);

    /**
     * o_x_key and o_y_key where stored in extension x up till 0.51.0.
     * They are stored in extension 0 from 0.52.0 onwards. It is therefore
     * necessary to check both.
     */
    cpl_propertylist* proplist = NULL;
    proplist = cpl_propertylist_load_regexp(filename, ext, regexp, 0);

    cpl_propertylist* proplist_primary = NULL;
    proplist_primary = cpl_propertylist_load_regexp(filename, 0, regexp, 0);

    result = sph_distortion_model_new_from_proplist(proplist,
                                                    c_x_tag,
                                                    c_y_tag);

    if (cpl_propertylist_has(proplist, o_x_key)) {
        *popt_cent_x = cpl_propertylist_get_double(proplist, o_x_key);
        *popt_cent_y = cpl_propertylist_get_double(proplist, o_y_key);
    } else {
        *popt_cent_x = cpl_propertylist_get_double(proplist_primary, o_x_key);
        *popt_cent_y = cpl_propertylist_get_double(proplist_primary, o_y_key);
    }

    if (palign_right != NULL) {
        const cpl_size nprops = cpl_propertylist_get_size(proplist);

        *palign_right = 0;

        for (cpl_size i = 0; i < nprops; i++) {
            const cpl_property * myname =
                cpl_propertylist_get_const(proplist, i);

            if (cpl_property_get_size(myname) == 31 &&
                memcmp(cpl_property_get_string(myname),
                       "ird.distortion_map.align_right", 31) == 0) {
                /* Found the parameter name key */
                const char * mynamekey = cpl_property_get_name(myname);
                const size_t keylen = strlen(mynamekey);

                if (keylen > 5 &&
                    memcmp(mynamekey + keylen - 5, " NAME", 5) == 0) {
                    const cpl_property * myval;
                    char * mykey = memcpy(cpl_malloc(keylen + 2), mynamekey,
                                          keylen - 4);
                    memcpy(mykey + keylen - 4, "VALUE", 6);

                    myval =
                        cpl_propertylist_get_property_const(proplist, mykey);

                    if (myval != NULL &&
                        cpl_property_get_size(myval) >= 5 &&
                        memcmp(cpl_property_get_string(myval),
                               "true", 4) == 0) {
                        cpl_msg_info(cpl_func, "Distortion map has optical "
                                     "axis of right channel aligned into left");
                        *palign_right = 1;
                    }
                    cpl_free(mykey);
                }
                break;
            }
        }
    }

    cpl_propertylist_delete(proplist);
    cpl_propertylist_delete(proplist_primary);
    cpl_free(regexp);
    if (result == NULL) (void)cpl_error_set_where(cpl_func);

    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create a polynomial model of the distortion from image of grid
 * @param mframe        The frame containing gridpoints
 * @param ref_pp        The reference point pattern
 * @param polyfit_order The polynomial order of the fit for both in x and y
 * @param threshold_sigma  The sigma of threshold for point detection
 * @param threshold_length The distortion threshold [pixel]
 * @param qclist           The quality control propertylist, to append to
 * @param off_centx    The X-offset from the detector center
 * @param off_centy    The Y-offset from the detector center
 * @param popt_centx   Iff non-NULL: positive: use as center, negative: set to center
 * @param popt_centy   Iff non-NULL: positive: use as center, negative: set to center
 * @param pavg_centx   The combined X-centroid of the observed patterns
 * @param pavg_centy   The combined Y-centroid of the observed patterns
 * @param pobs_pp      Iff non-NULL, set to extracted, observed point pattern
 * @param pdist_table  A row set for each detected source w. its correction
 * @return a new polyonomial distortion model
 * @see sph_distortion_model_from_grid_image_general()
 *
 *
 */
/*----------------------------------------------------------------------------*/
sph_distortion_model*
sph_distortion_model_from_grid_image(
        const sph_master_frame* mframe,
        const sph_point_pattern* ref_pp,
        int polyfit_order,
        double threshold_sigma,
        double threshold_length,
        cpl_propertylist* qclist,
        double off_centx,
        double off_centy,
        double * popt_centx,
        double * popt_centy,
        double * pavg_centx,
        double * pavg_centy,
        sph_point_pattern** pobs_pp,
        cpl_table ** pdist_table)
{
    sph_distortion_model* self =
        sph_distortion_model_from_grid_image_general
        (mframe, ref_pp, polyfit_order, threshold_sigma, threshold_length,
         qclist, off_centx, off_centy, popt_centx, popt_centy,
         pavg_centx, pavg_centy, pobs_pp, pdist_table, 0);

    if (self == NULL) (void)cpl_error_set_where(cpl_func);

    return self;
}    
/*----------------------------------------------------------------------------*/
/**
 * @brief Create a polynomial model of the distortion from image of grid
 * @param mframe        The frame containing gridpoints
 * @param ref_pp        The reference point pattern
 * @param polyfit_order The polynomial order of the fit for both in x and y
 * @param threshold_sigma  The sigma of threshold for point detection
 * @param threshold_length The distortion threshold [pixel]
 * @param qclist           The quality control propertylist, to append to
 * @param off_centx    The X-offset from the detector center
 * @param off_centy    The Y-offset from the detector center
 * @param popt_centx Iff non-NULL: positive: use as center, negative: set to center
 * @param popt_centy Iff non-NULL: positive: use as center, negative: set to center
 * @param pavg_centx   The averaged X-center of the observed patterns
 * @param pavg_centy   The averaged Y-center of the observed patterns
 * @param pobs_pp Iff  non-NULL, set to extracted, observed point pattern
 * @param pdist_table  A row set for each detected source w. its correction
 * @param use_median   Iff zero use point pattern central point, not mean center
 * @return A new polynomial distortion model
 * @note The use_median option should not be used
 * @see sph_distortion_model_from_grid_image()
 *
 * The observed point pattern shares its center of origin with the reference one
 *
 *
 */
/*----------------------------------------------------------------------------*/
sph_distortion_model*
sph_distortion_model_from_grid_image_general
    (const sph_master_frame* mframe,
     const sph_point_pattern* ref_pp,
     int polyfit_order,
     double threshold_sigma,
     double threshold_length,
     cpl_propertylist* qclist,
     double off_centx,
     double off_centy,
     double * popt_centx,
     double * popt_centy,
     double * pavg_centx,
     double * pavg_centy,
     sph_point_pattern** pobs_pp,
     cpl_table ** pdist_table,
     int use_median)
{
    sph_distortion_model* result = NULL;
    cpl_image*                       testim;
    sph_point_pattern*               obs_pp = NULL;
    sph_point_pattern*               trans_pp = NULL;
    sph_point*                       point = NULL;
    sph_point*                       optical_axis = NULL;
    sph_distortion_map             * dmap = NULL;
    double                           chiX, chiY;
    double                           detcentxy[2];
    const cpl_boolean                do_fixpoint = CPL_TRUE;
    const cpl_size                   ntry = 1;
    const double                     try_length = threshold_length / ntry;
    const cpl_size                   refsz = sph_point_pattern_get_size(ref_pp);
    int                              nremoved = -1; /* Ensure at least one pass */
    cpl_size                         nx, ny;

    cpl_ensure(ref_pp != NULL, CPL_ERROR_NULL_INPUT, NULL);

    testim = sph_master_frame_extract_image(mframe, 1);
    nx = cpl_image_get_size_x(testim);
    ny = cpl_image_get_size_y(testim);
    detcentxy[0] = 0.5 * (nx + 1);
    detcentxy[1] = 0.5 * (ny + 1);

    obs_pp = sph_point_pattern_new_from_image(testim, NULL, threshold_sigma,
                                              3.0, 3.0, 3.0, refsz, NULL);
    cpl_image_delete(testim);

    if (pobs_pp) *pobs_pp = obs_pp;

    cpl_ensure(obs_pp != NULL, cpl_error_get_code(), NULL);

    sph_point_pattern_get_centre(obs_pp, pavg_centx, pavg_centy);

    if (popt_centx != NULL && popt_centy != NULL &&
        *popt_centx > 0.0 && *popt_centy > 0.0) {
        optical_axis = sph_point_new(*popt_centx, *popt_centy);
    } else if (use_median != 0) {
        optical_axis = sph_point_new(*pavg_centx, *pavg_centy);
    } else {
        cpl_size icenter = -1;

        if (off_centx != 0.0 || off_centy != 0.0) {
            /* Assume that the central hole in the mask is the hole placed
               closest to the detector center, as detected previously */
            const double offcentxy[2] = {detcentxy[0] - off_centx,
                                         detcentxy[1] - off_centy};
            icenter = sph_point_pattern_find_closest(obs_pp, offcentxy);
        }

        if (icenter < 0) {
            /* Assume that the central hole in the mask is the hole placed
               closest to the detector center */
            icenter = sph_point_pattern_find_closest(obs_pp, detcentxy);
            if (icenter < 0) {
                /* Should not fail, but neverthless use this as fallback */
                icenter = sph_point_pattern_get_central_point(obs_pp, NULL, NULL);
            }
        }

        optical_axis = sph_point_pattern_get_point(obs_pp, icenter);
    }

    cpl_msg_info(cpl_func, "As optical axis is used the central point of the "
                 "%d-point observed pattern: (%g, %g) (%g,%g)",
                 sph_point_pattern_get_size(obs_pp),
                 optical_axis->x, optical_axis->y, off_centx, off_centy);

    /* Find center of reference pattern. For IRDIS: (0,0) */
    if (use_median == 0) {
        point = sph_point_pattern_get_point(ref_pp,
                                     sph_point_pattern_get_central_point(ref_pp,
                                                                         NULL,
                                                                         NULL));
    } else {
        double doffx0 = 0.0;
        double doffy0 = 0.0;
        sph_point_pattern_get_centre(ref_pp, &doffx0, &doffy0);
        point = sph_point_new(doffx0, doffy0);
    }

    cpl_msg_info(cpl_func, "Central point in %d-point reference pattern: "
                 "(%g, %g)", sph_point_pattern_get_size(ref_pp),
                 point->x, point->y);

    /* Transform center of origin of observed pattern to central point of
       reference pattern. Reference and observed patterns now each have their
       central point at the same (0,0) coordinate. */
    sph_point_pattern_add_offset(obs_pp,
                                 point->x - optical_axis->x,
                                 point->y - optical_axis->y);
    cpl_msg_info(cpl_func, "Shifted observed pattern CoO with (%g,%g), "
                 "detector center is now at (%g,%g)",
                 point->x - optical_axis->x,
                 point->y - optical_axis->y,
                 detcentxy[0] + point->x - optical_axis->x,
                 detcentxy[1] + point->y - optical_axis->y);

    if (popt_centx != NULL && popt_centy != NULL) {
        /* FIXME:  (point->x, point->y) should be (0,0) but is not for IFS */
        *popt_centx = optical_axis->x;
        *popt_centy = optical_axis->y;
    }

#ifdef SPH_POINT_PATTERN_SORT
    /* Sorting the order of the detected objects (from detector center towards
       edge) will cause the less distorted points to be fitted first. This
       will how the mean X/Y-sigma changes through out the fitting of the points.
       Ideally, this should make no difference to the fitted solution. */
    sph_point_pattern_sort(obs_pp);
#endif

    for (cpl_size itry = 0; itry < ntry; itry++) {
        const sph_point_pattern * use_pp   = trans_pp ? trans_pp : ref_pp;
        const int                 nremprev = nremoved;
        int                       nmulti;

        sph_distortion_map_delete(dmap);
        dmap = sph_distortion_map_new(obs_pp, use_pp);

        nmulti = sph_distortion_map_delete_multi(dmap);

        nremoved = sph_distortion_map_threshold(dmap, try_length,
                                                itry + 1 == ntry);

        cpl_msg_info(cpl_func, "From distortion map removed %d of %d vectors "
                     " above distortion %g (in try %d/%d)", nremoved,
                     sph_distortion_map_get_size(dmap) + nremoved,
                     threshold_length, 1+(int)itry, (int)ntry);

        if (nremoved == nremprev) break;

        if (sph_distortion_map_get_size(dmap) == 0) {
            sph_distortion_model_delete(result);
            sph_point_pattern_delete(trans_pp);
            sph_distortion_map_delete(dmap);
            sph_point_delete(point);
            sph_point_delete(optical_axis);
            sph_point_pattern_delete(obs_pp);
            if (pobs_pp != NULL) *pobs_pp = NULL;
            return NULL;
        }

        if (use_pp != ref_pp)
            sph_distortion_map_add(dmap, ref_pp);

        sph_distortion_model_delete(result);
        result = sph_distortion_model_new(NULL, NULL);
        /* The fitting uses the center of origin from the reference pattern */
        result->offx = optical_axis->x - point->x;
        result->offy = optical_axis->y - point->y;

        sph_distortion_model_fit_distortion_map_1(result, dmap,
                                                  polyfit_order,
                                                  do_fixpoint,
                                                  &chiX, &chiY);

        if (nmulti == 0 && nremoved == 0) break;

        sph_point_pattern_delete(trans_pp);
        trans_pp = sph_distortion_model_apply_pp(result, ref_pp);
    }

    *pdist_table = sph_distortion_map_table_create(dmap, result, ref_pp,
                                                   detcentxy[0],
                                                   detcentxy[1]);

    sph_distortion_map_delete(dmap);
    sph_point_pattern_delete(trans_pp);

    if (qclist) cpl_propertylist_update_int(qclist,
                                            SPH_COMMON_KEYWORD_DISTMAP_NREMOVED,
                                            nremoved);

    /* Change CoO to detector center, for both distortion correction and for
       observed pattern (if applicable) */
    sph_polynomial_distortion_set_offset(result, detcentxy[0], detcentxy[1]);

    if (pobs_pp == NULL) {
        sph_point_pattern_delete(obs_pp);
    } else {
        cpl_msg_info(cpl_func, "Shifting observed pattern CoO to (0,0) with "
                     "(%g,%g)",
                     -(detcentxy[0] + point->x - optical_axis->x),
                     -(detcentxy[1] + point->y - optical_axis->y));
        sph_point_pattern_add_offset(obs_pp,
                     -(detcentxy[0] + point->x - optical_axis->x),
                     -(detcentxy[1] + point->y - optical_axis->y));
    }

    sph_point_delete(point);
    sph_point_delete(optical_axis);

    return result;
}

sph_error_code
sph_distortion_model_save_dfs(
        sph_distortion_model* self,
        const char* outfilename,
        cpl_frameset* allframes,
        cpl_frame* template_frame,
        cpl_parameterlist* params,
        const char* tag,
        const char* recipe,
        const char* pipename,
        cpl_propertylist* qclist,
        int nx,
        int ny,
        double pixscale,
        int change_scale,
        const char* postfix,
        const char* c_x_tag,
        const char* c_y_tag)
{
    cpl_propertylist     * proplist       = NULL;
    sph_double_image     * double_image   = NULL;
    sph_distortion_model * dmap_duplicate = NULL;

    if(!c_x_tag)
        c_x_tag = SPH_COMMON_KEYWORD_DISTMAP_COEFFX;

    if(!c_y_tag)
        c_y_tag = SPH_COMMON_KEYWORD_DISTMAP_COEFFY;


    proplist = sph_distortion_model_get_proplist_override_tags(self, c_x_tag, c_y_tag);

    if ( change_scale ) {
        self = dmap_duplicate =
            sph_distortion_model_new_from_proplist(proplist,
                                                   c_x_tag,
                                                   c_y_tag);
        sph_distortion_model_scale_units(self, 1.0 / pixscale);
        cpl_propertylist_update_double( proplist,
                SPH_COMMON_KEYWORD_DISTMAP_PIXSCALE, pixscale);
    }
    double_image = sph_double_image_new(nx, ny);
    sph_distortion_model_fill_image_xy(self,
                                       double_image->iframe->image,
                                       double_image->pframe->image);
    if ( qclist ) {
        if ( double_image->qclist )
            cpl_propertylist_append(double_image->qclist,qclist);
        else
            double_image->qclist = cpl_propertylist_duplicate(qclist);
        if ( postfix != NULL )
            sph_utils_append_to_propertynames(double_image->qclist,postfix);
    }


    if ( postfix != NULL )
        sph_utils_append_to_propertynames(proplist,postfix);
    sph_double_image_save_dfs(double_image,
            outfilename,allframes,
            template_frame,
            params,
            tag,
            recipe,
            pipename,
            proplist);

    sph_double_image_delete(double_image);
    cpl_propertylist_delete(proplist);
    sph_distortion_model_delete(dmap_duplicate);
    return CPL_ERROR_NONE;
}
sph_error_code sph_distortion_model_save(
        const sph_distortion_model* self,
        const char* filename,
        int nx,
        int ny, unsigned mode,
        const cpl_propertylist* pl,
        const char* postfix,
        const char* c_x_tag,
        const char* c_y_tag)
{
    cpl_propertylist* proplist = NULL;
    sph_double_image* double_image = NULL;

    proplist = sph_distortion_model_get_proplist_override_tags(self, c_x_tag, c_y_tag);
    double_image = sph_double_image_new(nx,ny);
    sph_distortion_model_fill_image_xy(self,
                                                  double_image->iframe->image,
                                                  double_image->pframe->image);

    if ( pl ) {
        if ( double_image->qclist )
            cpl_propertylist_append(double_image->qclist,pl);
        else
            double_image->qclist = cpl_propertylist_duplicate(pl);
        if ( postfix != NULL )
            sph_utils_append_to_propertynames(double_image->qclist,postfix);
    }
    if ( postfix != NULL )
        sph_utils_append_to_propertynames(proplist,postfix);
    sph_double_image_save(double_image,filename,proplist,mode);
    sph_double_image_delete(double_image);
    cpl_propertylist_delete(proplist);
    return CPL_ERROR_NONE;
}

sph_point_pattern*
sph_distortion_model_create_point_pattern(const sph_distortion_model* self,
                                          int nx, int ny, int np)
{
    const double       dx = nx / (np + 1.0);
    const double       dy = ny / (np + 1.0);
    sph_point_pattern* pp = sph_point_pattern_new_(np * np);
    const double offsetx = self->offx;
    const double offsety = self->offy;


    for (int yy = 0; yy < np; yy++) {
        for (int xx = 0; xx < np; xx++) {
            // adding 0.5 below to go to FITS (CPL) pixel coords.
            // (note: offsetx should also be in CPL coords.)
            const double posxy[2] = {dx * ( xx + 1.0 ) + 0.5 - offsetx,
                                     dy * ( yy + 1.0 ) + 0.5 - offsety};
            double distxy[2];


            sph_distortion_model_get_dist(self, posxy, distxy);

            distxy[0] += posxy[0];
            distxy[1] += posxy[1];

            sph_point_pattern_add_point(pp, distxy[0], distxy[1]);
        }
    }

    return pp;
}

cpl_polynomial*
sph_distortion_model_get_poly_x( sph_distortion_model* self ) {
    return self->polyx;
}

cpl_polynomial*
sph_distortion_model_get_poly_y( sph_distortion_model* self ) {
    return self->polyy;
}

cpl_propertylist* sph_distortion_model_get_proplist_override_tags
    (const sph_distortion_model* self,
     const char * coeff_x_tag,
     const char * coeff_y_tag)
{
    cpl_size maxdeg;
    cpl_propertylist* proplist;

    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(cpl_polynomial_get_dimension(self->polyx) == 2,
               CPL_ERROR_TYPE_MISMATCH, NULL);
    cpl_ensure(cpl_polynomial_get_dimension(self->polyy) == 2,
               CPL_ERROR_TYPE_MISMATCH, NULL);

    maxdeg = cpl_polynomial_get_degree(self->polyx);

    cpl_ensure(maxdeg == cpl_polynomial_get_degree(self->polyy),
               CPL_ERROR_INCOMPATIBLE_INPUT, NULL);

    if(!coeff_x_tag)
        coeff_x_tag = SPH_COMMON_KEYWORD_DISTMAP_COEFFX;

    if(!coeff_y_tag)
        coeff_y_tag = SPH_COMMON_KEYWORD_DISTMAP_COEFFY;

    proplist = cpl_propertylist_new();

    for (cpl_size yy = 0; yy <= maxdeg; yy++) {
        for (cpl_size xx = 0; xx <= maxdeg; xx++) {
            const cpl_size pows[2] = {xx, yy};
            const double xcoeff = cpl_polynomial_get_coeff(self->polyx, pows);
            const double ycoeff = cpl_polynomial_get_coeff(self->polyy, pows);

            sph_keyword_manager_update_double2(proplist, coeff_x_tag, xx, yy,
                                               xcoeff);
            sph_keyword_manager_update_double2(proplist, coeff_y_tag, xx, yy,
                                               ycoeff);
        }
    }

    return proplist;
}

cpl_propertylist*
sph_distortion_model_get_proplist( const sph_distortion_model* self ) {
    return sph_distortion_model_get_proplist_override_tags(self, NULL, NULL);
}


/*----------------------------------------------------------------------------*/
/**
 * @brief Compute the distortion in X and Y at the given position
 * @param self    The distortion model to use
 * @param xypos   The (X,Y) position
 * @param xydist  The X- and Y-distortion
 * @return Nothing
 * @note No NULL input!
 */
/*----------------------------------------------------------------------------*/
inline void sph_distortion_model_get_dist(const sph_distortion_model* self,
                                          const double xypos[],
                                          double xydist[])
{
    if (self->polyx != NULL || self->polyy != NULL) {
        double * posvxy;

        CPL_DIAG_PRAGMA_PUSH_IGN(-Wcast-qual);
        posvxy = (double*)self->workxy;
        CPL_DIAG_PRAGMA_POP;

        (void)memcpy(posvxy, xypos, 2 * sizeof(*posvxy));
    }

    xydist[0] = self->polyx == NULL ? 0.0
        : cpl_polynomial_eval(self->polyx, self->workv);

    xydist[1] = self->polyy == NULL ? 0.0
        : cpl_polynomial_eval(self->polyy, self->workv);
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Compute the truncated distortion in X and Y at the given position
 * @param self    The distortion model to use
 * @param xypos   The (X,Y) position
 * @param xydist  The X- and Y-distortion
 * @return Nothing
 * @note No NULL input!
 */
/*----------------------------------------------------------------------------*/
void sph_distortion_model_get_qc_dist(const sph_distortion_model* self,
                                      const double xypos[],
                                      double xydist[])
{
    const cpl_size pows00[2] = {0, 0};
    const cpl_size pows10[2] = {1, 0};
    const cpl_size pows01[2] = {0, 1};
    const cpl_size pows11[2] = {1, 1};

    xydist[0] = 0.0;
    if (self->polyx != NULL) {
        xydist[0] += cpl_polynomial_get_coeff(self->polyx, pows00);
        xydist[0] += cpl_polynomial_get_coeff(self->polyx, pows10) * xypos[0];
        xydist[0] += cpl_polynomial_get_coeff(self->polyx, pows01) * xypos[1];
        xydist[0] += cpl_polynomial_get_coeff(self->polyx, pows11) * xypos[0]
            * xypos[1];
    }

    xydist[1] = 0.0;
    if (self->polyy != NULL) {
        xydist[1] += cpl_polynomial_get_coeff(self->polyy, pows00);
        xydist[1] += cpl_polynomial_get_coeff(self->polyy, pows10) * xypos[0];
        xydist[1] += cpl_polynomial_get_coeff(self->polyy, pows01) * xypos[1];
        xydist[1] += cpl_polynomial_get_coeff(self->polyy, pows11) * xypos[0]
            * xypos[1];
    }
}

/*----------------------------------------------------------------------------*/
/**
 * @internal
 * @brief Fill X- and Y-image with distortion corrections
 * @param self  The distortion model to fill from
 * @param imgx  The X-image of type double to fill
 * @param imgy  The Y-image of type double to fill
 * @return On success CPL_ERROR_NONE or else the relevant CPL error
 * @note Both images must have identical sizes
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
sph_distortion_model_fill_image_xy(const sph_distortion_model* self,
                                   cpl_image* imgx,
                                   cpl_image* imgy)
{

    const cpl_size nx = cpl_image_get_size_x(imgx);
    const cpl_size ny = cpl_image_get_size_y(imgx);
    double * pimgx;
    double * pimgy;
    double   distmax_x = 0.0, distmax_y = 0.0, sqdistmax_xy = 0.0;
    cpl_size distmax_x_posx  = 0, distmax_x_posy  = 0;
    cpl_size distmax_y_posx  = 0, distmax_y_posy  = 0;
    cpl_size distmax_xy_posx = 0, distmax_xy_posy = 0;

    cpl_ensure_code(self        != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(cpl_polynomial_get_dimension(self->polyx) == 2,
                    CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(cpl_polynomial_get_dimension(self->polyy) == 2,
                    CPL_ERROR_ILLEGAL_INPUT);

    cpl_ensure_code(cpl_image_get_size_x(imgy) == nx,
                    CPL_ERROR_INCOMPATIBLE_INPUT);
    cpl_ensure_code(cpl_image_get_size_y(imgy) == ny,
                    CPL_ERROR_INCOMPATIBLE_INPUT);

    pimgx = cpl_image_get_data_double(imgx);
    pimgy = cpl_image_get_data_double(imgy);

    cpl_ensure_code(pimgx != NULL, CPL_ERROR_UNSUPPORTED_MODE);
    cpl_ensure_code(pimgy != NULL, CPL_ERROR_UNSUPPORTED_MODE);

    for (cpl_size j = 1; j <= ny; j++) {
        for (cpl_size i = 1; i <= nx; i++) {
            const double posxy[2] = {(double)i - self->offx,
                                     (double)j - self->offy};
            double distxy[2];

            sph_distortion_model_get_dist(self, posxy, distxy);

            pimgx[nx * (j - 1) + (i - 1)] = distxy[0];
            pimgy[nx * (j - 1) + (i - 1)] = distxy[1];

            if (fabs(distxy[0]) > fabs(distmax_x)) {
                distmax_x = distxy[0];
                distmax_x_posx = i;
                distmax_x_posy = j;
            }
            if (fabs(distxy[1]) > fabs(distmax_y)) {
                distmax_y = distxy[1];
                distmax_y_posx = i;
                distmax_y_posy = j;
            }
            if (distxy[0] * distxy[0] + distxy[1] * distxy[1] > sqdistmax_xy) {
                sqdistmax_xy = distxy[0] * distxy[0] + distxy[1] * distxy[1];
                distmax_xy_posx = i;
                distmax_xy_posy = j;
            }
        }
    }
    cpl_image_accept_all(imgx);
    cpl_image_accept_all(imgy);

    cpl_msg_info(cpl_func, "Maximum X-distortion at pixel (%d, %d): |%g|",
                 (int)distmax_x_posx, (int)distmax_x_posy, distmax_x);
    cpl_msg_info(cpl_func, "Maximum Y-distortion at pixel (%d, %d): |%g|",
                 (int)distmax_y_posx, (int)distmax_y_posy, distmax_y);
    cpl_msg_info(cpl_func, "Maximum distortion at pixel (%d, %d): %g",
                 (int)distmax_xy_posx, (int)distmax_xy_posy, sqrt(sqdistmax_xy));

    return CPL_ERROR_NONE;
}


sph_error_code
sph_polynomial_distortion_set_offset(sph_distortion_model* self,
                                     double offx,
                                     double offy)
{
    if (self == NULL) {
        return cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
    } else {
        const double dx = offx - self->offx;
        const double dy = offy - self->offy;
        double       posxy[2] = {0.0, 0.0};
        double       xycen0[2];
        double       xycen1[2];

        sph_distortion_model_get_dist(self, posxy, xycen0);

        cpl_polynomial_shift_1d(self->polyx, 0, dx);
        cpl_polynomial_shift_1d(self->polyy, 0, dx);
        cpl_polynomial_shift_1d(self->polyx, 1, dy);
        cpl_polynomial_shift_1d(self->polyy, 1, dy);

        posxy[0] = -dx;
        posxy[1] = -dy;

        sph_distortion_model_get_dist(self, posxy, xycen1);

        cpl_msg_info(cpl_func,"Move X/Y-distortion fix-point from (%g, %g) to "
                     "(%g, %g): (%g, %g) w. residual (%g, %g)",
                     self->offx, self->offy, offx, offy, xycen1[0], xycen1[1],
                     xycen1[0] - xycen0[0], xycen1[1] - xycen0[1]);

        self->offx = offx;
        self->offy = offy;
    }
    return CPL_ERROR_NONE;
}

sph_distortion_map*
sph_distortion_model_fit_distortion_map(
        sph_distortion_model* self,
        const sph_distortion_map* dmap,
        int maxdegree,
        double* pchiX,
        double* pchiY) {

    const int ndmap = sph_distortion_map_get_size(dmap);
    cpl_vector* xvals;
    cpl_vector* yvals;
    cpl_vector* zvalsX;
    cpl_vector* zvalsY;
    sph_distortion_map* newdistmap = NULL;
    const double* pxvals  = sph_distortion_map_get_data_x(dmap);
    const double* pyvals  = sph_distortion_map_get_data_y(dmap);
    const double* pdxvals = sph_distortion_map_get_data_dx(dmap);
    const double* pdyvals = sph_distortion_map_get_data_dy(dmap);

    CPL_DIAG_PRAGMA_PUSH_IGN(-Wcast-qual);
    xvals  = cpl_vector_wrap(ndmap, (double*)pxvals);
    yvals  = cpl_vector_wrap(ndmap, (double*)pyvals);
    zvalsX = cpl_vector_wrap(ndmap, (double*)pdxvals);
    zvalsY = cpl_vector_wrap(ndmap, (double*)pdyvals);
    CPL_DIAG_PRAGMA_POP;

    cpl_polynomial_delete(self->polyx);
    cpl_polynomial_delete(self->polyy);
    self->polyx = sph_fitting_fit_poly2d(xvals, yvals, zvalsX, pchiX,
                                         maxdegree, maxdegree);
    self->polyy = sph_fitting_fit_poly2d(xvals, yvals, zvalsY, pchiY,
                                         maxdegree, maxdegree);

    newdistmap = sph_distortion_map_new(NULL, NULL);

    for (int cc = 0; cc < ndmap; ++cc) {
        const double posxy[2] = {pxvals[cc], pyvals[cc]};
        double distxy[2];

        sph_distortion_model_get_dist(self, posxy, distxy);

        sph_distortion_map_append_one(newdistmap,
                                      pdxvals[cc], pdyvals[cc],
                                      distxy[0] - pdxvals[cc],
                                      distxy[1] - pdyvals[cc]);
    }

    (void)cpl_vector_unwrap(xvals);
    (void)cpl_vector_unwrap(yvals);
    (void)cpl_vector_unwrap(zvalsX);
    (void)cpl_vector_unwrap(zvalsY);

    return newdistmap;
}


static cpl_error_code
sph_distortion_model_fit_distortion_map_1(sph_distortion_model* self,
                                                     const sph_distortion_map* dmap,
                                                     cpl_size maxdeg,
                                                     cpl_boolean do_fixpoint,
                                                     double* pchiX,
                                                     double* pchiY)
{

    const int ndmap = sph_distortion_map_get_size(dmap);
    const cpl_size mindeg = (do_fixpoint && maxdeg > 0) ? 1 : 0;
    cpl_vector* zvalsX;
    cpl_vector* zvalsY;

    const double* pxvals  = sph_distortion_map_get_data_x(dmap);
    const double* pyvals  = sph_distortion_map_get_data_y(dmap);
    const double* pdxvals = sph_distortion_map_get_data_dx(dmap);
    const double* pdyvals = sph_distortion_map_get_data_dy(dmap);
    double      * pxypos = cpl_malloc(2 * (size_t)ndmap * sizeof(*pxypos));
    cpl_matrix  * xypos  = cpl_matrix_wrap(2, ndmap, pxypos);
    cpl_error_code code;

    CPL_DIAG_PRAGMA_PUSH_IGN(-Wcast-qual);
    zvalsX = cpl_vector_wrap(ndmap, (double*)pdxvals);
    zvalsY = cpl_vector_wrap(ndmap, (double*)pdyvals);
    CPL_DIAG_PRAGMA_POP;

    (void)memcpy(pxypos, pxvals, (size_t)ndmap * sizeof(*pxypos));
    (void)memcpy(pxypos + (size_t)ndmap, pyvals,
                 (size_t)ndmap * sizeof(*pxypos));

    code = sph_fitting_fit_poly2d_1(self->polyx, xypos, zvalsX, pchiX,
                                    mindeg, maxdeg);
    if (!code) code =
        sph_fitting_fit_poly2d_1(self->polyy, xypos, zvalsY, pchiY,
                                 mindeg, maxdeg);

    cpl_matrix_delete(xypos);
    cpl_vector_unwrap(zvalsX);
    cpl_vector_unwrap(zvalsY);

    return code ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;
}

cpl_image*
sph_distortion_model_apply(
        const sph_distortion_model* self,
        const cpl_image* image,
        cpl_kernel kerneltype,
        double radin,
        const char * qckeyx,
        const char * qckeyy,
        cpl_propertylist* qclist)
{
    cpl_image* result = cpl_image_new(cpl_image_get_size_x(image),
                                      cpl_image_get_size_y(image),
                                      cpl_image_get_type(image));

    sph_distortion_model_apply_1(result, self, image,
                                 kerneltype, radin,
                                 qckeyx, qckeyy, qclist);

    return result;
}

cpl_error_code
sph_distortion_model_apply_1(cpl_image* result,
                             const sph_distortion_model* self,
                             const cpl_image* image,
                             cpl_kernel kerneltype,
                             double radin,
                             const char * qckeyx,
                             const char * qckeyy,
                             cpl_propertylist* qclist)
{
    cpl_vector*  profile = NULL;
    const double rad     = radin > 0.0 ? radin : 10.0;

    cpl_ensure_code(image       != NULL,  CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(self        != NULL,  CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(self->polyx != NULL,  CPL_ERROR_DATA_NOT_FOUND);
    cpl_ensure_code(self->polyy != NULL,  CPL_ERROR_DATA_NOT_FOUND);
    cpl_ensure_code(radin       < 2048.0, CPL_ERROR_ILLEGAL_INPUT);

    profile = cpl_vector_new(CPL_KERNEL_DEF_SAMPLES * (int)radin);
    if (profile != NULL) {
        const cpl_size nx    = cpl_image_get_size_x(image);
        const cpl_size ny    = cpl_image_get_size_y(image);
        const size_t   bufsz = (size_t)(nx * ny)
            * cpl_type_get_sizeof(CPL_TYPE_DOUBLE);
        double * imgdata = cpl_malloc(2 * bufsz);
        cpl_image* dxim = cpl_image_wrap_double(nx, ny, imgdata);
        cpl_image* dyim = cpl_image_wrap_double(nx, ny, imgdata + nx * ny);

        sph_distortion_model_fill_image_xy(self, dxim, dyim);

        cpl_vector_fill_kernel_profile(profile, kerneltype, radin);

        cpl_image_warp(result, image, dxim, dyim, profile, rad, profile, rad);
        cpl_vector_delete(profile);

        if (qclist != NULL && qckeyx != NULL && qckeyy != NULL ) {
            const double xstdev = cpl_image_get_stdev(dxim);
            const double ystdev = cpl_image_get_stdev(dyim);

            cpl_propertylist_append_double(qclist, qckeyx, xstdev);
            cpl_propertylist_append_double(qclist, qckeyy, ystdev);
        }

        cpl_image_delete(dxim);
        cpl_image_unwrap(dyim);
    }

    return cpl_error_get_code() ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Transform a polygon via a distortion model
 * @param self  The distortion model used for the transformation
 * @param poly  The polygon to transform in-place
 * @return On success CPL_ERROR_NONE or else the relevant CPL error
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_distortion_model_apply_poly(const sph_distortion_model* self,
                                               sph_polygon* poly)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(poly != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(self->polyx != NULL, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(self->polyy != NULL, CPL_ERROR_ILLEGAL_INPUT);

    for (polygon_sz i = 0; i < poly->npoints; i++) {
        double distxy[2];

        sph_distortion_model_get_dist(self, (const double*)(poly->points + i),
                                      distxy);

        (void)sph_point_shift(poly->points + i, distxy[0], distxy[1]);
    }

    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Scale the units of the distortion map
 * @param self   The distortion map with units to scale
 * @param scale  the factor to scale the units by
 *
 * @return error code
 *
 * This will scale the UNITS of the distortion map by the given
 * factor. So for example if the units of the distortion were in pixels
 * and a factor of 0.5 is applied, the distortion is then in units of 0.5 pixels.
 *
 */
/*----------------------------------------------------------------------------*/

sph_error_code
sph_distortion_model_scale_units(
        sph_distortion_model* self,
        double scale)
{
    cpl_size degx, degy;

    cpl_ensure_code( self, CPL_ERROR_NULL_INPUT );
    cpl_ensure_code( scale > 0.0, CPL_ERROR_ILLEGAL_INPUT );

    cpl_ensure_code( cpl_polynomial_get_dimension(self->polyx) == 2,
                     CPL_ERROR_TYPE_MISMATCH );
    cpl_ensure_code( cpl_polynomial_get_dimension(self->polyy) == 2,
                     CPL_ERROR_TYPE_MISMATCH );

    degx = cpl_polynomial_get_degree(self->polyx);
    degy = cpl_polynomial_get_degree(self->polyy);

    cpl_ensure_code( degx == degy, CPL_ERROR_INCOMPATIBLE_INPUT );

    if (scale != 1.0) {
        double factor = scale;
        for (cpl_size degree = 0; degree <= degx; degree++) {
            for (cpl_size power_x = 0; power_x <= degree; power_x++) {
                const cpl_size power_y = degree - power_x;
                const cpl_size pows[2] = {power_x, power_y};

                const double xcoeff =
                    cpl_polynomial_get_coeff(self->polyx, pows);
                const double ycoeff =
                    cpl_polynomial_get_coeff(self->polyy, pows);

                if (xcoeff != 0.0)
                    cpl_polynomial_set_coeff( self->polyx, pows,
                                              xcoeff * factor);
                if (ycoeff != 0.0)
                    cpl_polynomial_set_coeff( self->polyy, pows,
                                              ycoeff * factor);
            }
            factor /= scale;
        }
    }
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

void
sph_distortion_model_delete(sph_distortion_model* self) {
    if (self != NULL) {
        cpl_polynomial_delete(self->polyx);
        cpl_polynomial_delete(self->polyy);
        (void)cpl_vector_unwrap(self->workv);
        cpl_free(self);
    }
}


/**@}*/
