/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_fctable.h"

#include "sph_fits.h"
#include "sph_common_keywords.h"
#include "sph_keyword_manager.h"
#include "sph_filemanager.h"
#include "sph_time.h"
#include "sph_skycalc.h"
#include "sph_coord.h"

#include <math.h>
#include <strings.h>
#include <string.h>

/*-----------------------------------------------------------------------------
                            Private Function prototypes
 -----------------------------------------------------------------------------*/

static int sph_fctable_getline__(FILE* f, char line[], int max)
    CPL_ATTR_NONNULL;

/*-----------------------------------------------------------------------------
                                Function code
 -----------------------------------------------------------------------------*/

/* Read one line from standard input, */
/* copying it to line array (but no more than max chars). */
/* Does not place terminating \n in line array. */
/* Returns line length, or 0 for empty line, or EOF for end-of-file. */
static int sph_fctable_getline__(FILE* f, char line[], int max)
{
    int nch = 0;
    int c;
    max--;          /* leave room for '\0' */

    while((c = fgetc(f)) != EOF)
    {
        if (c == '\n')
            break;

        if (nch < max)
            line[nch++] = (char)c;
    }

    if (c == EOF && nch == 0)
        return EOF;

    line[nch] = '\0';
    return nch;
}


/*----------------------------------------------------------------------------*/
/**
 * @defgroup A SPHERE API Module
 * @par Synopsis:
 * @code
 * typedef _module_ {
 * } module
 * @endcode
 * @par Desciption:
 *
 * This module provides functionality for apertures, extending the functionality
 * as it exists for cpl_apertures.
 */
/*----------------------------------------------------------------------------*/
/**@{*/
static
void
sph_fctable_increase_size__(sph_fctable* self) {
    int n = 0;

    n = sph_fctable_get_size(self) + 1;

    cpl_vector_set_size(self->times,n+1);
    cpl_vector_set_size(self->centresL_x,n+1);
    cpl_vector_set_size(self->centresL_y,n+1);
    cpl_vector_set_size(self->angles,n+1);
    cpl_vector_set_size(self->angles_err,n+1);
    if ( self->mode == SPH_FCTAB_DOUBLE ) {
        cpl_vector_set_size(self->centresR_x,n+1);
        cpl_vector_set_size(self->centresR_y,n+1);
    }
    return;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new table of field centres and angles
 * @param
 *
 * @return new pointer
 *
 * Creates a new table of field centres and angles.
 *
 */
/*----------------------------------------------------------------------------*/
sph_fctable*  sph_fctable_new(void) {
    sph_fctable*        fctab = NULL;

    fctab = calloc(1,sizeof(sph_fctable));
    fctab->angles = cpl_vector_new(1);
    fctab->times = cpl_vector_new(1);
    fctab->angles_err = cpl_vector_new(1);
    fctab->centresL_x = cpl_vector_new(1);
    fctab->centresL_x_err = cpl_vector_new(1);
    fctab->centresL_y = cpl_vector_new(1);
    fctab->centresL_y_err = cpl_vector_new(1);
    fctab->centresR_x = cpl_vector_new(1);
    fctab->centresR_x_err = cpl_vector_new(1);
    fctab->centresR_y = cpl_vector_new(1);
    fctab->centresR_y_err = cpl_vector_new(1);
    fctab->mode = SPH_FCTAB_UNDEFINED;
    return fctab;
}

int sph_fctable_get_size(const sph_fctable* self) {
    cpl_ensure(self,CPL_ERROR_ILLEGAL_INPUT,-1);
    return (int)cpl_vector_get_size(self->centresL_x)-1;
}

sph_error_code
sph_fctable_add_row_single(sph_fctable* self,
        double mjd,
        double cx, double cy, double angle) {
    int         n = 0;
    cpl_ensure_code(self,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(self->mode == SPH_FCTAB_SINGLE ||
            self->mode == SPH_FCTAB_UNDEFINED,
            CPL_ERROR_ILLEGAL_INPUT );
    n = sph_fctable_get_size(self);
    cpl_vector_set(self->times,n,mjd);
    cpl_vector_set(self->angles,n,angle);
    cpl_vector_set(self->angles_err,n,0.0);
    cpl_vector_set(self->centresL_x,n,cx);
    cpl_vector_set(self->centresL_y,n,cy);
    self->mode = SPH_FCTAB_SINGLE;
    sph_fctable_increase_size__(self);
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

sph_error_code
sph_fctable_add_row_double(
        sph_fctable* self,
        double mjd,
        double clx, double cly,
        double crx, double cry, double angle) {
    int         n = 0;
    cpl_ensure_code(self,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(self->mode == SPH_FCTAB_DOUBLE ||
            self->mode == SPH_FCTAB_UNDEFINED,
            CPL_ERROR_ILLEGAL_INPUT );
    n = sph_fctable_get_size(self);
    cpl_vector_set(self->times,n,mjd);
    cpl_vector_set(self->angles,n,angle);
    cpl_vector_set(self->angles_err,n,0.0);
    cpl_vector_set(self->centresL_x,n,clx);
    cpl_vector_set(self->centresL_y,n,cly);
    cpl_vector_set(self->centresR_x,n,crx);
    cpl_vector_set(self->centresR_y,n,cry);
    self->mode = SPH_FCTAB_DOUBLE;
    sph_fctable_increase_size__(self);
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

char*
sph_fctable_construct_filename( const cpl_frame* inframe, int txtflag ) {
    char*                       tmpfilename = NULL;
    char*                       tmpfilename2 = NULL;

    cpl_ensure(inframe,CPL_ERROR_NULL_INPUT,NULL);
    tmpfilename2 = sph_filemanager_new_filename_from_base(
             cpl_frame_get_filename(inframe), "fctable");
     tmpfilename = sph_filemanager_remove_dir(tmpfilename2);
     cpl_free(tmpfilename2); tmpfilename2 = NULL;
     if (txtflag) {
         tmpfilename2 = sph_filemanager_filename_new_ext(
                 tmpfilename, "txt");
         cpl_free(tmpfilename); tmpfilename = NULL;
         return tmpfilename2;
     }
     return tmpfilename;
}

sph_error_code
sph_fctable_save_ascii( const sph_fctable* self, const char* filename ) {
    FILE*       f = NULL;
    char        line[1024];
    int         n = 0;
    int         ii = 0;
    char        timechars[256];

    cpl_ensure_code( filename, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( self, CPL_ERROR_NULL_INPUT);

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

    n = sph_fctable_get_size(self);
    cpl_ensure_code( n > 0, CPL_ERROR_ILLEGAL_INPUT);

    f = fopen( filename, "w" );
    cpl_ensure_code( f, CPL_ERROR_FILE_IO);

    fprintf(f,"# Field centre and angles table\n");
    fprintf(f,"# \n");
    fprintf(f,"# This table gives the field centres and rotation angles\n");
    fprintf(f,"# for a set of frames.\n");
    fprintf(f,"# Format is: centres in pixel coordinates (0,0) at lower left\n");
    fprintf(f,"#            angles in degrees ccw wrt. x-axis.\n");
    fprintf(f,"# \n");
    if ( self->mode == SPH_FCTAB_SINGLE ) {
        fprintf(f,"#  Time-UT   Centre x  Centre y   Angle (deg)\n");
    }
    else {
        fprintf(f,"# Time-UT   Centre Left x    Left y   Right x   Right y   Angle (deg)\n");
    }
    for (ii = 0; ii < n; ++ii) {
        sph_time_mjd_to_iso8601string(
                cpl_vector_get(
                        self->times,ii), timechars);
        if ( self->mode == SPH_FCTAB_SINGLE ) {
            sprintf(line,"%s %9.2f %9.2f %12.6f",
                    timechars,
                    cpl_vector_get(self->centresL_x,ii),
                    cpl_vector_get(self->centresL_y,ii),
                    cpl_vector_get(self->angles,ii) );
        }
        else if ( self->mode == SPH_FCTAB_DOUBLE ) {
            sprintf(line,"%s  %9.2f %9.2f %9.2f %9.2f %12.6f",
                    timechars,
                    cpl_vector_get(self->centresL_x,ii),
                    cpl_vector_get(self->centresL_y,ii),
                    cpl_vector_get(self->centresR_x,ii),
                    cpl_vector_get(self->centresR_y,ii),
                    cpl_vector_get(self->angles,ii) );
        }
        fprintf(f,"%s\n",line);
    }
    fclose(f); f = NULL;

    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

cpl_table*
sph_fctable_get_as_cpl_table(const sph_fctable* self) {
    cpl_table*      result          = NULL;
    int             n                = 0;
    int             ii               = 0;
    char        timechars[256];

    cpl_ensure(self,CPL_ERROR_NULL_INPUT,NULL);
    n = sph_fctable_get_size(self);
    cpl_ensure(n > 0, CPL_ERROR_ILLEGAL_INPUT,NULL);
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;

    result = cpl_table_new(n);
    cpl_ensure(result,cpl_error_get_code(),NULL);
    if ( self->mode == SPH_FCTAB_DOUBLE ) {
        cpl_table_new_column(result,"TIME_UT",CPL_TYPE_STRING);
        cpl_table_new_column(result,"CENTRE_LEFT_X",CPL_TYPE_DOUBLE);
        cpl_table_new_column(result,"CENTRE_LEFT_Y",CPL_TYPE_DOUBLE);
        cpl_table_new_column(result,"CENTRE_RIGHT_X",CPL_TYPE_DOUBLE);
        cpl_table_new_column(result,"CENTRE_RIGHT_Y",CPL_TYPE_DOUBLE);
    }
    else if ( self->mode == SPH_FCTAB_SINGLE ) {
        cpl_table_new_column(result,"TIME_UT",CPL_TYPE_STRING);
        cpl_table_new_column(result,"CENTRE_X",CPL_TYPE_DOUBLE);
        cpl_table_new_column(result,"CENTRE_Y",CPL_TYPE_DOUBLE);
    }
    cpl_table_new_column(result,"ANGLE",CPL_TYPE_DOUBLE);
    for (ii = 0; ii < n; ++ii) {
        sph_time_mjd_to_iso8601string(
                cpl_vector_get(
                        self->times,ii), timechars);
        cpl_table_set_string(result,"TIME_UT",ii,timechars);
        if ( self->mode == SPH_FCTAB_DOUBLE ) {
            cpl_table_set_double(result,"CENTRE_LEFT_X",ii,cpl_vector_get(self->centresL_x,ii));
            cpl_table_set_double(result,"CENTRE_LEFT_Y",ii,cpl_vector_get(self->centresL_y,ii));
            cpl_table_set_double(result,"CENTRE_RIGHT_X",ii,cpl_vector_get(self->centresR_x,ii));
            cpl_table_set_double(result,"CENTRE_RIGHT_Y",ii,cpl_vector_get(self->centresR_y,ii));
        }
        else if ( self->mode == SPH_FCTAB_SINGLE ) {
            cpl_table_set_double(result,"CENTRE_X",ii,cpl_vector_get(self->centresL_x,ii));
            cpl_table_set_double(result,"CENTRE_Y",ii,cpl_vector_get(self->centresL_y,ii));
        }
        cpl_table_set_double(result,"ANGLE",ii,cpl_vector_get(self->angles,ii));
    }
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return result;
}


sph_fctable*
sph_fctable_load_ascii(const char* filename) {
    FILE*       f = NULL;
    char       line[256];
    double      clx = 0.0;
    double      cly = 0.0;
    double      crx = 0.0;
    double      cry = 0.0;
    double      angle = 0.0;
    int         nmatches = 0;
    int         linesize = 0;
    int         year = 0;
    int         month = 0;
    int         day = 0;
    int         hour = 0;
    int         minute   =   0;
    double      second      = 0.0;
    double      mjd = 0.0;
    sph_fctab_mode__    mm = SPH_FCTAB_UNDEFINED;
    sph_fctable*    self = NULL;
    cpl_ensure(filename,CPL_ERROR_NULL_INPUT,NULL);
    f = fopen(filename,"r");
    cpl_ensure(f,CPL_ERROR_FILE_NOT_FOUND,NULL);
    self = sph_fctable_new();
    while ( (linesize = sph_fctable_getline__(f,line,256)) != EOF ) {
        if ( line[0]== '#' || linesize == 0 ) {
            ;
        }
        else {
            nmatches = sscanf(line,
                    "%4d-%2d-%2dT%2d:%2d:%lf %lf %lf %lf %lf %lf",
                    &year,
                    &month,
                    &day,
                    &hour,
                    &minute,
                    &second,
                    &clx,&cly,&crx,&cry,&angle);
            if ( nmatches == 9 ) {
                sph_time_mjd_from_iso8601(
                        &mjd,year,month,day,hour,minute,second);
                if ( mm == SPH_FCTAB_UNDEFINED ||
                        mm == SPH_FCTAB_SINGLE ) {
                    mm = SPH_FCTAB_SINGLE;
                    angle = crx;
                    sph_fctable_add_row_single(self,
                            mjd,clx,cly,angle);
                }
                else {
                    SPH_ERROR_ENSURE_GOTO_EXIT(0,CPL_ERROR_BAD_FILE_FORMAT);
                }
            }
            else if ( nmatches == 11 ) {
                sph_time_mjd_from_iso8601(
                        &mjd,year,month,day,hour,minute,second);
                if ( mm == SPH_FCTAB_UNDEFINED ||
                        mm == SPH_FCTAB_DOUBLE ) {
                    mm = SPH_FCTAB_DOUBLE;
                    sph_fctable_add_row_double(self,
                            mjd,clx,cly,crx,cry,angle);
                }
                else {
                    SPH_ERROR_ENSURE_GOTO_EXIT(0,CPL_ERROR_BAD_FILE_FORMAT);
                }
            }
            else {
                SPH_ERROR_ENSURE_GOTO_EXIT(0,CPL_ERROR_BAD_FILE_FORMAT);
            }
        }
    }
    fclose(f); f = NULL;

    return self;
EXIT:
    sph_fctable_delete(self); self = NULL;
    return NULL;
}

sph_fctable*
sph_fctable_load_fits(const char* filename) {
    sph_fctable*    self    = NULL;
    cpl_table*      tab     = NULL;
    cpl_size        next    = 0;
    unsigned int    ee      = 0;
    int             rr      = 0;
    int             bp      = 0;
    double          mjd     = 0.0;

    cpl_ensure(filename,CPL_ERROR_NULL_INPUT,NULL);

    next = cpl_fits_count_extensions(filename);
    for (ee = 0; ee < next + 1; ++ee) {
        tab = cpl_table_load( filename, (int)ee, 0 );
        if ( tab == NULL ) {
            cpl_error_reset();
        }
        else {
            break;
        }
    }

    cpl_ensure(tab,CPL_ERROR_FILE_IO,NULL);
    self = sph_fctable_new();
    if ( strcasecmp(cpl_table_get_column_name(tab),
                "TIME_UT") != 0 ) {
        sph_fctable_delete(self);self = NULL;
        cpl_ensure(self,CPL_ERROR_BAD_FILE_FORMAT,NULL);
    }
    if ( strcasecmp(cpl_table_get_column_name(NULL),
            "CENTRE_LEFT_X") == 0 ) {
        self->mode = SPH_FCTAB_DOUBLE;
    }
    else if ( strcasecmp(cpl_table_get_column_name(NULL),
                "CENTRE_Y") == 0 ) {
        self->mode = SPH_FCTAB_SINGLE;
    }
    else {
        sph_fctable_delete(self);self = NULL;
        cpl_ensure(self,CPL_ERROR_BAD_FILE_FORMAT,NULL);
    }

    for (rr = 0; rr < cpl_table_get_nrow(tab); ++rr) {
        if ( self->mode == SPH_FCTAB_DOUBLE ) {
            sph_time_mjd_from_string(&mjd,
                    cpl_table_get_string(tab,"TIME_UT",rr));

            sph_fctable_add_row_double(self,mjd,
                    cpl_table_get_double(tab,"CENTRE_LEFT_X",rr,&bp),
                    cpl_table_get_double(tab,"CENTRE_LEFT_Y",rr,&bp),
                    cpl_table_get_double(tab,"CENTRE_RIGHT_X",rr,&bp),
                    cpl_table_get_double(tab,"CENTRE_RIGHT_Y",rr,&bp),
                    cpl_table_get_double(tab,"ANGLE",rr,&bp) );
        }
        else {
            sph_time_mjd_from_string(&mjd,
                    cpl_table_get_string(tab,"TIME_UT",rr));
            sph_fctable_add_row_single(self,mjd,
                    cpl_table_get_double(tab,"CENTRE_X",rr,&bp),
                    cpl_table_get_double(tab,"CENTRE_Y",rr,&bp),
                    cpl_table_get_double(tab,"ANGLE",rr,&bp) );
        }
    }
    cpl_table_delete(tab); tab = NULL;
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return self;
}

sph_fctable*
sph_fctable_load_fits_IRDIS(const char* filename) {
    sph_fctable*    self    = NULL;
    cpl_table*      tab     = NULL;
    cpl_size    	next    = 0;
    unsigned int    ee      = 0;
    int             rr      = 0;
    int             bp      = 0;
    double          mjd     = 0.0;

    cpl_ensure(filename,CPL_ERROR_NULL_INPUT,NULL);

    next = cpl_fits_count_extensions(filename);
    for (ee = 0; ee < next + 1; ++ee) {
        tab = cpl_table_load( filename, (int)ee, 0 );
        if ( tab == NULL ) {
            cpl_error_reset();
        }
        else {
            break;
        }
    }

    cpl_ensure(tab,CPL_ERROR_FILE_IO,NULL);
    self = sph_fctable_new();
    if ( strcasecmp(cpl_table_get_column_name(tab),
                "TIME_UT") != 0 ) {
        sph_fctable_delete(self);self = NULL;
        cpl_ensure(self,CPL_ERROR_BAD_FILE_FORMAT,NULL);
    }
    if ( strcasecmp(cpl_table_get_column_name(NULL),
            "CENTRE_LEFT_X") == 0 ) {
        self->mode = SPH_FCTAB_DOUBLE;
    }
    else if ( strcasecmp(cpl_table_get_column_name(NULL),
                "CENTRE_Y") == 0 ) {
        self->mode = SPH_FCTAB_SINGLE;
    }
    else {
        sph_fctable_delete(self);self = NULL;
        cpl_ensure(self,CPL_ERROR_BAD_FILE_FORMAT,NULL);
    }

    for (rr = 0; rr < cpl_table_get_nrow(tab); ++rr) {
        if ( self->mode == SPH_FCTAB_DOUBLE ) {
            sph_time_mjd_from_string(&mjd,
                    cpl_table_get_string(tab,"TIME_UT",rr));

            sph_fctable_add_row_double(self,mjd,
                    cpl_table_get_double(tab,"CENTRE_LEFT_X",rr,&bp),
                    cpl_table_get_double(tab,"CENTRE_LEFT_Y",rr,&bp),
                    cpl_table_get_double(tab,"CENTRE_RIGHT_X",rr,&bp),
                    cpl_table_get_double(tab,"CENTRE_RIGHT_Y",rr,&bp), 0.0);
        }
        else {
            sph_time_mjd_from_string(&mjd,
                    cpl_table_get_string(tab,"TIME_UT",rr));
            sph_fctable_add_row_single(self,mjd,
                    cpl_table_get_double(tab,"CENTRE_X",rr,&bp),
                    cpl_table_get_double(tab,"CENTRE_Y",rr,&bp), 0.0);
        }
    }
    cpl_table_delete(tab); tab = NULL;
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return self;
}


cpl_vector*
sph_fctable_get_times(const sph_fctable* self) {
    cpl_vector* vector = NULL;
    int         n = 0;
    cpl_ensure(self,CPL_ERROR_NULL_INPUT,NULL);
    n = sph_fctable_get_size(self);
    cpl_ensure(n>0,CPL_ERROR_ILLEGAL_INPUT,NULL);
    vector = cpl_vector_duplicate(self->times);
    cpl_vector_set_size(vector,n);
    return vector;
}

cpl_vector*
sph_fctable_get_angles(const sph_fctable* self) {
    cpl_vector* vector = NULL;
    int         n = 0;
    cpl_ensure(self,CPL_ERROR_NULL_INPUT,NULL);
    n = sph_fctable_get_size(self);
    cpl_ensure(n>0,CPL_ERROR_ILLEGAL_INPUT,NULL);
    vector = cpl_vector_duplicate(self->angles);
    cpl_vector_set_size(vector,n);
    return vector;
}


double
sph_fctable_get_time(const sph_fctable* self, int idx) {
    cpl_ensure(self, CPL_ERROR_NULL_INPUT, -1.0);
	return cpl_vector_get(self->times, idx);
}

double
sph_fctable_get_angle(const sph_fctable* self, int idx) {
    cpl_ensure(self, CPL_ERROR_NULL_INPUT, -1.0);
	return cpl_vector_get(self->angles, idx);
}

double sph_fctable_get_centre_x_element(const sph_fctable* self, int idx){
	return sph_fctable_get_centre_x_left_element(self, idx);
}

double
sph_fctable_get_centre_y_element(const sph_fctable* self, int idx){
	return sph_fctable_get_centre_y_left_element(self, idx);
}

cpl_vector* sph_fctable_get_centre_x(const sph_fctable* self){
	return sph_fctable_get_centre_x_left(self);
}

cpl_vector* sph_fctable_get_centre_y(const sph_fctable* self){
	return sph_fctable_get_centre_y_left(self);
}

double
sph_fctable_get_centre_x_left_element(const sph_fctable* self, int idx){
    cpl_ensure(self, CPL_ERROR_NULL_INPUT, -1.0);
    return cpl_vector_get(self->centresL_x, idx);
}

double
sph_fctable_get_centre_y_left_element(const sph_fctable* self, int idx){
    cpl_ensure(self, CPL_ERROR_NULL_INPUT, -1.0);
    return cpl_vector_get(self->centresL_y, idx);
}

double
sph_fctable_get_centre_x_right_element(const sph_fctable* self, int idx){
    cpl_ensure(self, CPL_ERROR_NULL_INPUT, -1.0);
    return cpl_vector_get(self->centresR_x, idx);
}

double
sph_fctable_get_centre_y_right_element(const sph_fctable* self, int idx){
    cpl_ensure(self, CPL_ERROR_NULL_INPUT, -1.0);
    return cpl_vector_get(self->centresR_y, idx);
}


cpl_vector*
sph_fctable_get_centre_x_left(const sph_fctable* self) {
    cpl_vector* vector = NULL;
    int         n = 0;
    cpl_ensure(self,CPL_ERROR_NULL_INPUT,NULL);
    n = sph_fctable_get_size(self);
    cpl_ensure(n>0,CPL_ERROR_ILLEGAL_INPUT,NULL);
    vector = cpl_vector_duplicate(self->centresL_x);
    cpl_vector_set_size(vector,n);
    return vector;
}

cpl_vector*
sph_fctable_get_centre_y_left(const sph_fctable* self) {
    cpl_vector* vector = NULL;
    int         n = 0;
    cpl_ensure(self,CPL_ERROR_NULL_INPUT,NULL);
    n = sph_fctable_get_size(self);
    cpl_ensure(n>0,CPL_ERROR_ILLEGAL_INPUT,NULL);
    vector = cpl_vector_duplicate(self->centresL_y);
    cpl_vector_set_size(vector,n);
    return vector;
}

cpl_vector*
sph_fctable_get_centre_x_right(const sph_fctable* self) {
    cpl_vector* vector = NULL;
    int         n = 0;
    cpl_ensure(self,CPL_ERROR_NULL_INPUT,NULL);
    n = sph_fctable_get_size(self);
    cpl_ensure(n > 0 ,CPL_ERROR_ILLEGAL_INPUT,NULL);
    if ( self->mode != SPH_FCTAB_DOUBLE ) return NULL;
    vector = cpl_vector_duplicate(self->centresR_x);
    cpl_vector_set_size(vector,n);
    return vector;
}

cpl_vector*
sph_fctable_get_centre_y_right(const sph_fctable* self) {
    cpl_vector* vector = NULL;
    int         n = 0;
    cpl_ensure(self,CPL_ERROR_NULL_INPUT,NULL);
    n = sph_fctable_get_size(self);
    cpl_ensure(n>0,CPL_ERROR_ILLEGAL_INPUT,NULL);
    if ( self->mode != SPH_FCTAB_DOUBLE ) return NULL;
    vector = cpl_vector_duplicate(self->centresR_y);
    cpl_vector_set_size(vector,n);
    return vector;
}

void sph_fctable_delete(sph_fctable* self) {
    if ( self ) {
        cpl_vector_delete(self->times);
        cpl_vector_delete(self->angles);
        cpl_vector_delete(self->angles_err);
        cpl_vector_delete(self->centresL_x);
        cpl_vector_delete(self->centresL_x_err);
        cpl_vector_delete(self->centresL_y);
        cpl_vector_delete(self->centresL_y_err);
        cpl_vector_delete(self->centresR_x);
        cpl_vector_delete(self->centresR_x_err);
        cpl_vector_delete(self->centresR_y);
        cpl_vector_delete(self->centresR_y_err);
        free(self);
    }
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Set the center information for a specific row
 * @param self 		pointer to self
 * @param row	    the row number
 * @param clx0	x coordinate of the center of the left image of the frames
 * @param cly0	y coordinate of the center of the left image of the frames
 * @param crx0	x coordinate of the center of the right image of the frames
 * @param cry0	y coordinate of the center of the right image of the frames
 *
 *
 * @return error code
 *
 * Set the centre information for a specific row in the fctable
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_fctable_set_cent_double(
		sph_fctable* self,
		int row,
		double clx, double cly, double crx, double cry)
{

	cpl_ensure_code( self, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code( row >=0, CPL_ERROR_ILLEGAL_INPUT);
	cpl_ensure_code(
			cpl_vector_set( self->centresL_x, row, clx) == CPL_ERROR_NONE,
			cpl_error_get_code() );
	cpl_ensure_code(
			cpl_vector_set( self->centresL_y, row, cly) == CPL_ERROR_NONE,
			cpl_error_get_code() );
	cpl_ensure_code(
			cpl_vector_set( self->centresR_x, row, crx) == CPL_ERROR_NONE,
			cpl_error_get_code() );
	cpl_ensure_code(
			cpl_vector_set( self->centresR_y, row, cry) == CPL_ERROR_NONE,
			cpl_error_get_code() );
	return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Set the angle information for a specific row
 * @param self 		pointer to self
 * @param row	    the row number
 * @param angle     the angle
 * @param angle_err     the angle error
 *
 * @return error code
 *
 * Set the centre information for a specific row in the fctable
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_fctable_set_angle(
		sph_fctable* self,
		int row,
		double angle,
		double angle_err)
{

	cpl_ensure_code( self, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code( row >=0, CPL_ERROR_ILLEGAL_INPUT);
	cpl_ensure_code(
			cpl_vector_set( self->angles, row, angle) == CPL_ERROR_NONE,
			cpl_error_get_code() );
	cpl_ensure_code(
			cpl_vector_set( self->angles_err, row, angle_err) == CPL_ERROR_NONE,
			cpl_error_get_code() );
	return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Fill all raw with a given angle information
 * @param self      pointer to self
 * @param angle     the angle
 * @param angle_err     the angle error
 *
 * @return error code
 *
 * Set the centre information for a specific row in the fctable
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_fctable_fill_angle(sph_fctable* self,
                                      double angle,
                                      double angle_err)
{

    cpl_ensure_code( self, CPL_ERROR_NULL_INPUT);

    return cpl_vector_fill( self->angles, angle)
        || cpl_vector_fill( self->angles_err, angle_err)
        ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Add the angle to the angle vector of the fctable
 * @param self      pointer to self
 * @param angle     the angle to be added
 *
 * @return error code
 * Add a scalar angle to each element of the fctable angle vector.
  */
/*----------------------------------------------------------------------------*/
sph_error_code sph_fctable_add_angle_scalar( sph_fctable* self, double angle )
{
    cpl_ensure_code( self, CPL_ERROR_NULL_INPUT);

    return cpl_vector_add_scalar( self->angles, angle)
        ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Add the center ofsett to the center coordinates  of the fctable
 * @param self      pointer to self
 * @param cx        the x-offset to be added
 * @param cy        the y-offset to be added
 *
 * @return error code
 * Add a scalar angle to each element of the fctable angle vector.
  */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_fctable_add_center_offset_single_mode( sph_fctable* self, double cx, double cy )
{
    cpl_ensure_code( self, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( self->mode == SPH_FCTAB_SINGLE, CPL_ERROR_ILLEGAL_INPUT );
    cpl_ensure_code( cx >=-512.0 && cx <512.0, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code( cy >=-512.0 && cy <512.0, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code( cpl_vector_add_scalar( self->centresL_x, cx) == CPL_ERROR_NONE, cpl_error_get_code() );
    cpl_ensure_code( cpl_vector_add_scalar( self->centresL_y, cy) == CPL_ERROR_NONE, cpl_error_get_code() );
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Add the center ofsett to the center coordinates  of the fctable (double mode)
 * @param self      pointer to self
 * @param clx        the x-offset to be added
 * @param cly        the y-offset to be added
 * @param crx        the x-offset to be added
 * @param cry        the y-offset to be added
 *
 * @return error code
 * Add a scalar angle to each element of the fctable angle vector.
  */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_fctable_add_center_offset_double_mode( sph_fctable* self, double clx, double cly, double crx, double cry )
{
    cpl_ensure_code( self, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( self->mode == SPH_FCTAB_DOUBLE, CPL_ERROR_ILLEGAL_INPUT );
    cpl_ensure_code( clx >=-512.0 && clx <512.0, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code( cly >=-512.0 && cly <512.0, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code( crx >=-512.0 && crx <512.0, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code( cry >=-512.0 && cry <512.0, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code( cpl_vector_add_scalar( self->centresL_x, clx) == CPL_ERROR_NONE, cpl_error_get_code() );
    cpl_ensure_code( cpl_vector_add_scalar( self->centresL_y, cly) == CPL_ERROR_NONE, cpl_error_get_code() );
    cpl_ensure_code( cpl_vector_add_scalar( self->centresR_x, crx) == CPL_ERROR_NONE, cpl_error_get_code() );
    cpl_ensure_code( cpl_vector_add_scalar( self->centresR_y, cry) == CPL_ERROR_NONE, cpl_error_get_code() );
    return CPL_ERROR_NONE;
}



/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new field centre table
 * @param raw_science_frames the list of raw science frames
 * @param field_center_frame a field centre table input frame (maybe NULL)
 * @param use_dither   flag to control if dithering info should be used
 * @param clx0	x coordinate of the center of the left image of the frames
 * @param cly0	y coordinate of the center of the left image of the frames
 * @param crx0	x coordinate of the center of the right image of the frames
 * @param cry0	y coordinate of the center of the right image of the frames
 *
 *
 * @return new field centre table or NULL on error.
 *
 * Create a field centre table. This may either be a straight reading in
 * of the field centre table from the field_center frame if one is provided
 * or a default table is constructed for the input list of raw frames.
 * In the latter case, if the use_dither flag is not set, the table returned
 * is simply filled with the pixel centers as determined from the instrument
 * model (the central pixel for each frame would usually then be at 512 in x and y).
 * If the use_dither flag is set to 1, the dithering is taken into
 * account when constructing the default table.
 *
 */
/*----------------------------------------------------------------------------*/
#define SIZE_SKYINPARR 7
sph_fctable*
sph_fctable_create_fctable( const cpl_frame* raw_science_frame, int mode )
{
    sph_error_code        rerr 	  = CPL_ERROR_NONE;
    sph_fctable*          fctab   = NULL;
    sph_skycalc_input	  skyinp[SIZE_SKYINPARR];
    sph_skycalc_output	  skyout;
    cpl_propertylist*	  pl 	  = NULL;
    cpl_image*			  image	  = NULL;
    double                crx 	  = 0.0;
    double                cry	  = 0.0;
    double                clx 	  = 0.0;
    double                cly 	  = 0.0;
    double				  xdith	  = 0.0;
    double				  ydith	  = 0.0;
    int                   ff 	  = 0;
    cpl_vector*           timev   = NULL;
    int                   nplanes = 0;
    cpl_errorstate        okstate = cpl_errorstate_get();
    const char*           drot2_mode = NULL;

    //Minimal init of 'skyinp'
    for (int i=0; i<SIZE_SKYINPARR; i++) {
        skyinp[i].command = '\0';
        skyinp[i].param_string[0] = '\0';
    }

    cpl_ensure( raw_science_frame, CPL_ERROR_NULL_INPUT, NULL );


    fctab = sph_fctable_new();
    if ( !fctab ) {
        SPH_ERR("Could not create centre positions table.");
        return NULL;
    }

    if ( mode == SPH_FCTAB_SINGLE || mode == SPH_FCTAB_DOUBLE ){
    	fctab->mode = mode;
    } else {
    	fctab->mode = SPH_FCTAB_SINGLE;
    }

    nplanes = sph_fits_get_nplanes( cpl_frame_get_filename(
            raw_science_frame), 0 );

    if ( fctab->mode == SPH_FCTAB_SINGLE ){
        timev = sph_time_extract_times_zpl( raw_science_frame );
    } else {
        timev = sph_time_extract_times_from_frame( raw_science_frame  );
    }

    //SPH_ERROR_ENSURE_GOTO_EXIT( timev != NULL, rerr );
    if ( timev == NULL ) {
        cpl_msg_warning(cpl_func, "The time interpolation failed, "
                "proceed further without de-rotation!");
        cpl_errorstate_dump( okstate, CPL_FALSE,
                             cpl_errorstate_dump_one_warning);
        cpl_errorstate_set( okstate );
    	timev = cpl_vector_new( nplanes );
        cpl_vector_fill( timev, 0.0);
    }

    //pl = cpl_propertylist_load( cpl_frame_get_filename(raw_science_frame), 0);
    pl = sph_keyword_manager_load_properties(cpl_frame_get_filename(raw_science_frame), 0);

    if ( fctab->mode == SPH_FCTAB_SINGLE ) {
    	SPH_INFO_MSG("fctab->mode == SPH_FCTAB_SINGLE");
    	image = cpl_image_load(cpl_frame_get_filename( raw_science_frame ), CPL_TYPE_DOUBLE, 0, 0  );
    	if ( image != NULL ){
    		clx = (double) (cpl_image_get_size_x( image ))/ 2.0;
    		cly = clx;
    		cpl_image_delete( image );
    	} else {
    		clx = 0.0; cly = 0.0;
    	}
    } else {
        SPH_INFO_MSG("fctab->mode == SPH_FCTAB_DOUBLE");
    	clx = 0.0; cly = 0.0;
    	crx = 0.0; cry = 0.0;
    }

    if ( pl ) {

      //dithering
      if ( cpl_propertylist_has( pl, SPH_COMMOM_KEYWORD_CHIP_INDEX) ) {

    	  if ( cpl_propertylist_get_int( pl, SPH_COMMOM_KEYWORD_CHIP_INDEX )
    			  == SPH_COMMON_KEYWORD_VALUE_CHIP_INDEX_CAM1 ) {
    		  //cam1
    		  if ( cpl_propertylist_has ( pl, SPH_COMMON_KEYWORD_CAM1_DITHERING_X) ){
    			  xdith = cpl_propertylist_get_double( pl, SPH_COMMON_KEYWORD_CAM1_DITHERING_X ) +
    			          SPH_COMMON_KEYWORD_VALUE_CAM1_POS0_X;
    		  } else {
       		   SPH_ERROR_RAISE_WARNING(SPH_ERROR_WARNING, "%s keyword (which sets up xdith for cam1) not found!\n",SPH_COMMON_KEYWORD_CAM1_DITHERING_X );
    		  }
    		  if ( cpl_propertylist_has ( pl, SPH_COMMON_KEYWORD_CAM1_DITHERING_Y) ){
    			  ydith = cpl_propertylist_get_double( pl, SPH_COMMON_KEYWORD_CAM1_DITHERING_Y ) +
    			          SPH_COMMON_KEYWORD_VALUE_CAM1_POS0_Y;
    		  } else {
       		   SPH_ERROR_RAISE_WARNING(SPH_ERROR_WARNING, "%s keyword (which sets up ydith for cam1) not found!\n",SPH_COMMON_KEYWORD_CAM1_DITHERING_Y );
    		  }
    	  } else if ( cpl_propertylist_get_int( pl, SPH_COMMOM_KEYWORD_CHIP_INDEX )
        			  == SPH_COMMON_KEYWORD_VALUE_CHIP_INDEX_CAM2 ) {
    		  //cam2
    		  if ( cpl_propertylist_has ( pl, SPH_COMMON_KEYWORD_CAM2_DITHERING_X) ){
    			  xdith = cpl_propertylist_get_double( pl, SPH_COMMON_KEYWORD_CAM2_DITHERING_X ) +
    			          SPH_COMMON_KEYWORD_VALUE_CAM2_POS0_X;
    		  } else {
       		   SPH_ERROR_RAISE_WARNING(SPH_ERROR_WARNING, "%s keyword (which sets up xdith for cam2) not found!\n",SPH_COMMON_KEYWORD_CAM2_DITHERING_X );
    		  }
    		  if ( cpl_propertylist_has ( pl, SPH_COMMON_KEYWORD_CAM2_DITHERING_Y) ){
    			  ydith = cpl_propertylist_get_double( pl, SPH_COMMON_KEYWORD_CAM2_DITHERING_Y ) +
    			          SPH_COMMON_KEYWORD_VALUE_CAM2_POS0_Y;
    		  } else {
       		   SPH_ERROR_RAISE_WARNING(SPH_ERROR_WARNING, "%s keyword (which sets up ydith for cam2) not found!\n",SPH_COMMON_KEYWORD_CAM2_DITHERING_Y );
    		  }

    	  } else {
      		   SPH_ERROR_RAISE_WARNING(SPH_ERROR_WARNING, "Chip index has a wrong value: %d (it must be 1 or 2)!\n",  cpl_propertylist_get_int( pl, SPH_COMMOM_KEYWORD_CHIP_INDEX) );
    	  }
      }
    }


    //setup sph_skycalc and create first raw
    if ( cpl_propertylist_has( pl, SPH_COMMON_KEYWORD_RA ) &&
    		cpl_propertylist_has( pl, SPH_COMMON_KEYWORD_DEC ) &&
    		(cpl_vector_get(timev, 0) != 0.0 ) )
    {
    	sph_coord_initialise_skycalc(pl,skyinp,cpl_vector_get(timev,0), &skyout);
    } else {
        if (cpl_propertylist_has( pl, SPH_COMMON_KEYWORD_RA ) &&
            cpl_propertylist_has( pl, SPH_COMMON_KEYWORD_DEC ) ) {
            SPH_ERROR_RAISE_WARNING(SPH_ERROR_WARNING,
                "Failed to calculate paralactic angle (PA)!"
                " Please, verify if the %s and %s keywords are presented in the fits header.\n",
                SPH_COMMON_KEYWORD_RA,  SPH_COMMON_KEYWORD_DEC);
        } else {
            SPH_WARNING("Failed to calculate paralactic angle (PA)! "
                    "Time vector was not set correctly.")
        }
        SPH_WARNING("Paralactic angles are set to 0.0. The recipe will proceed anyway without "
                "de-rotation.")
    	skyout.pa = 0.0;
    }


    if ( fctab->mode == SPH_FCTAB_SINGLE ){
        if ( cpl_propertylist_has( pl, SPH_COMMON_KEYWORD_DROT2_MODE) ){
            drot2_mode = cpl_propertylist_get_string(pl, SPH_COMMON_KEYWORD_DROT2_MODE);
             if (!strcmp(drot2_mode, SPH_COMMON_KEYWORD_VALUE_DROT2_MODE_NO_DEROT)){
                SPH_INFO_MSG("STAT mode (nothing war rotated) mode was detected!");
                rerr = sph_fctable_add_row_single( fctab, cpl_vector_get( timev, 0 ),
                                                          clx - xdith, cly - ydith, -skyout.pa + skyout.alt);
            } else if (!strcmp(drot2_mode, SPH_COMMON_KEYWORD_VALUE_DROT2_MODE_PUPIL_STAB)){
                SPH_INFO_MSG("Pupil- stabilized mode was detected!");
                rerr = sph_fctable_add_row_single( fctab, cpl_vector_get( timev, 0 ),
                                                          clx - xdith, cly - ydith, -skyout.pa);
            } else if (!strcmp(drot2_mode, SPH_COMMON_KEYWORD_VALUE_DROT2_MODE_FIELD_STAB)) {
                SPH_INFO_MSG("Field- stabilized mode was detected! All PAs are set to 0.0.");
                rerr = sph_fctable_add_row_single( fctab, cpl_vector_get( timev, 0 ),
                                                          clx - xdith, cly - ydith, 0.0);
            } else {
                SPH_ERROR_RAISE_ERR(SPH_ERROR_ERROR,
                    "Not allowed value of the %s keyword is presented in the fits header!\n",
                    SPH_COMMON_KEYWORD_DROT2_MODE);
            }
        } else {
            SPH_ERR("ESO INS4 DROT2 MODE keyword doesn't exist! Stop... ");
        }

    } else {
    	rerr = sph_fctable_add_row_double( fctab, cpl_vector_get( timev, 0 ),
    												  clx, cly, crx, cry, skyout.pa );
   	}


    SPH_ERROR_ENSURE_GOTO_EXIT( rerr == CPL_ERROR_NONE, rerr );

    for (ff = 1; ff < nplanes; ++ff) {

        if ( cpl_propertylist_has( pl, SPH_COMMON_KEYWORD_RA ) &&
        		cpl_propertylist_has( pl, SPH_COMMON_KEYWORD_DEC ) &&
        		(cpl_vector_get(timev, 0) != 0.0 ))
        {
        	sph_coord_do_skycalc(skyinp,
        			  	  	  	  cpl_vector_get(timev,ff),
        			  	  	  	  &skyout);
        } else {
        	skyout.pa = 0.0;
        }


        if ( fctab->mode == SPH_FCTAB_SINGLE ){
            if ( drot2_mode ){
                if (!strcmp(drot2_mode, SPH_COMMON_KEYWORD_VALUE_DROT2_MODE_NO_DEROT)){
                    rerr = sph_fctable_add_row_single( fctab, cpl_vector_get( timev, ff ),
        												  clx - xdith, cly - ydith, -skyout.pa + skyout.alt);
                } else if (!strcmp(drot2_mode, SPH_COMMON_KEYWORD_VALUE_DROT2_MODE_PUPIL_STAB)) {

                    rerr = sph_fctable_add_row_single( fctab, cpl_vector_get( timev, ff ),
                                                              clx - xdith, cly - ydith, -skyout.pa );
                }else if (!strcmp(drot2_mode, SPH_COMMON_KEYWORD_VALUE_DROT2_MODE_FIELD_STAB)) {
                    rerr = sph_fctable_add_row_single( fctab, cpl_vector_get( timev, 0 ),
                                                              clx - xdith, cly - ydith, 0.0);
                } else {
                    rerr = sph_fctable_add_row_single( fctab, cpl_vector_get( timev, 0 ),
                                                             clx - xdith, cly - ydith, 0.0);
                }
            } else {
                rerr = sph_fctable_add_row_single( fctab, cpl_vector_get( timev, ff ),
                                                          clx - xdith, cly - ydith, 0.0 );
            }

        } else {
        	rerr = sph_fctable_add_row_double( fctab, cpl_vector_get( timev, ff ),
        												  clx, cly, crx, cry, skyout.pa );
       	}

        SPH_ERROR_ENSURE_GOTO_EXIT( rerr == CPL_ERROR_NONE, rerr );

    }


    cpl_propertylist_delete(pl); pl = NULL;
    cpl_vector_delete(timev); timev = NULL;
    return fctab;

EXIT:
    if (pl) cpl_propertylist_delete(pl); pl = NULL;
    cpl_vector_delete(timev); timev = NULL;
    sph_fctable_delete(fctab); fctab = NULL;
    return NULL;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new field centre table ( modified sph_fctable_create_fctable)
 * @param raw_science_frames the list of raw science frames
 * @param field_center_frame a field centre table input frame (maybe NULL)
 * @param use_dither   flag to control if dithering info should be used
 * @param clx0  x coordinate of the center of the left image of the frames
 * @param cly0  y coordinate of the center of the left image of the frames
 * @param crx0  x coordinate of the center of the right image of the frames
 * @param cry0  y coordinate of the center of the right image of the frames
 *
 *
 * @return new field centre table or NULL on error.
 *
 * Create a field centre table. This may either be a straight reading in
 * of the field centre table from the field_center frame if one is provided
 * or a default table is constructed for the input list of raw frames.
 * In the latter case, if the use_dither flag is not set, the table returned
 * is simply filled with the pixel centers as determined from the instrument
 * model (the central pixel for each frame would usually then be at 512 in x and y).
 * If the use_dither flag is set to 1, the dithering is taken into
 * account when constructing the default table.
 *
 */
/*----------------------------------------------------------------------------*/
sph_fctable*
sph_fctable_create_fctable_new( const cpl_frame* raw_science_frame, int mode )
{
    sph_error_code        rerr    = CPL_ERROR_NONE;
    sph_fctable*          fctab   = NULL;
    sph_skycalc_input     skyinp[7];
    sph_skycalc_output    skyout;
    cpl_propertylist*     pl      = NULL;
    cpl_image*            image   = NULL;
    double                crx     = 0.0;
    double                cry     = 0.0;
    double                clx     = 0.0;
    double                cly     = 0.0;
    double                xdith   = 0.0;
    double                ydith   = 0.0;
    int                   ff      = 0;
    cpl_vector*           timev   = NULL;
    int                   nplanes = 0;
    cpl_errorstate        okstate = cpl_errorstate_get();
    const char*           drot2_mode = NULL;
    cpl_boolean           zimpol = CPL_FALSE;


    cpl_ensure( raw_science_frame, CPL_ERROR_NULL_INPUT, NULL );


    fctab = sph_fctable_new();
    if ( !fctab ) {
        SPH_ERR("Could not create centre positions table.");
        return NULL;
    }

    if ( mode == SPH_FCTAB_SINGLE || mode == SPH_FCTAB_DOUBLE ){
        fctab->mode = mode;
    } else {
        fctab->mode = SPH_FCTAB_SINGLE;
    }

    //pl = cpl_propertylist_load( cpl_frame_get_filename(raw_science_frame), 0);
    pl = sph_keyword_manager_load_properties(cpl_frame_get_filename(raw_science_frame), 0);
    if ( !pl ) {
        SPH_ERROR_RAISE_ERR(SPH_ERROR_ERROR,
                "Can't load header info from the filename = %s \n", cpl_frame_get_filename(raw_science_frame));
        goto EXIT;
    }

    //check if the sub-system is zimpol
    if ( cpl_propertylist_has( pl, SPH_COMMON_KEYWORD_DET_SYSTEM_ID) ){
        const char* det_system = 
            cpl_propertylist_get_string(pl, SPH_COMMON_KEYWORD_DET_SYSTEM_ID);

        if ( !strcmp(det_system, SPH_COMMON_KEYWORD_VALUE_DET_ID_ZIMPOL)) {
            SPH_INFO_MSG("The sub-system is ZIMPOL...");
            zimpol = CPL_TRUE;
        } else {
            zimpol = CPL_FALSE;
        }
    } else {
        SPH_WARNING("Keyword (ESO DET DEV1 ID) not found! "
                "The star center coordinates for ZIMPOL might be wrong!");
    }


    nplanes = sph_fits_get_nplanes( cpl_frame_get_filename(
            raw_science_frame), 0 );


    if ( fctab->mode == SPH_FCTAB_SINGLE ){
        timev = sph_time_extract_times_zpl( raw_science_frame );
    } else if ( fctab->mode == SPH_FCTAB_DOUBLE && zimpol ){
        SPH_INFO_MSG("fctab->mode == SPH_FCTAB_DOUBLE for ZIMPOL sub-system.");
        timev = sph_time_extract_times_zpl( raw_science_frame ); //zimpol
    } else {
        SPH_INFO_MSG("fctab->mode == SPH_FCTAB_DOUBLE");
        timev = sph_time_extract_times_from_frame( raw_science_frame  ); //irdis
    }

    //SPH_ERROR_ENSURE_GOTO_EXIT( timev != NULL, rerr );
    if ( timev == NULL ) {
        cpl_msg_warning(cpl_func, "The time interpolation failed, "
                "proceed further without de-rotation!");
        cpl_errorstate_dump( okstate, CPL_FALSE,
                             cpl_errorstate_dump_one_warning);
        cpl_errorstate_set( okstate );
        timev = cpl_vector_new( nplanes );
        cpl_vector_fill( timev, 0.0);
    }


    if ( fctab->mode == SPH_FCTAB_SINGLE ) {
        SPH_INFO_MSG("fctab->mode == SPH_FCTAB_SINGLE");
        image = cpl_image_load(cpl_frame_get_filename( raw_science_frame ), CPL_TYPE_DOUBLE, 0, 0  );
        if ( image != NULL ){
            clx = (double) (cpl_image_get_size_x( image ))/ 2.0;
            cly = clx;
            cpl_image_delete( image ); image = NULL;
        } else {
            clx = 0.0; cly = 0.0;
        }
    } else if ( fctab->mode == SPH_FCTAB_DOUBLE && zimpol ) {
        SPH_INFO_MSG("fctab->mode == SPH_FCTAB_DOUBLE for ZIMPOL sub-system (set initial coords).");
        image = cpl_image_load(cpl_frame_get_filename( raw_science_frame ), CPL_TYPE_DOUBLE, 0, 0  );
        if ( image != NULL ){
            clx = (double) (cpl_image_get_size_x( image ))/ 2.0;
            cly = crx = cry = clx;
            cpl_image_delete( image ); image = NULL;
        } else {
            clx = 0.0; cly = 0.0;
            crx = 0.0; cry = 0.0;
        }
    } else {
        SPH_INFO_MSG("fctab->mode == SPH_FCTAB_DOUBLE");
        clx = 0.0; cly = 0.0;
        crx = 0.0; cry = 0.0;
    }

    if ( pl && zimpol ) {

      //dithering
      if ( cpl_propertylist_has( pl, SPH_COMMOM_KEYWORD_CHIP_INDEX) ) {

          if ( cpl_propertylist_get_int( pl, SPH_COMMOM_KEYWORD_CHIP_INDEX )
                  == SPH_COMMON_KEYWORD_VALUE_CHIP_INDEX_CAM1 ) {
              //cam1
              if ( cpl_propertylist_has ( pl, SPH_COMMON_KEYWORD_CAM1_DITHERING_X) ){
                  xdith = cpl_propertylist_get_double( pl, SPH_COMMON_KEYWORD_CAM1_DITHERING_X ) +
                          SPH_COMMON_KEYWORD_VALUE_CAM1_POS0_X;
              } else {
               SPH_ERROR_RAISE_WARNING(SPH_ERROR_WARNING, "%s keyword (which sets up xdith for cam1) not found!\n",SPH_COMMON_KEYWORD_CAM1_DITHERING_X );
              }
              if ( cpl_propertylist_has ( pl, SPH_COMMON_KEYWORD_CAM1_DITHERING_Y) ){
                  ydith = cpl_propertylist_get_double( pl, SPH_COMMON_KEYWORD_CAM1_DITHERING_Y ) +
                          SPH_COMMON_KEYWORD_VALUE_CAM1_POS0_Y;
                  ydith=-ydith;
              } else {
               SPH_ERROR_RAISE_WARNING(SPH_ERROR_WARNING, "%s keyword (which sets up ydith for cam1) not found!\n",SPH_COMMON_KEYWORD_CAM1_DITHERING_Y );
              }
          } else if ( cpl_propertylist_get_int( pl, SPH_COMMOM_KEYWORD_CHIP_INDEX )
                      == SPH_COMMON_KEYWORD_VALUE_CHIP_INDEX_CAM2 ) {
              //cam2
              if ( cpl_propertylist_has ( pl, SPH_COMMON_KEYWORD_CAM2_DITHERING_X) ){
                  xdith = cpl_propertylist_get_double( pl, SPH_COMMON_KEYWORD_CAM2_DITHERING_X ) +
                          SPH_COMMON_KEYWORD_VALUE_CAM2_POS0_X;
                  xdith=-xdith;
              } else {
               SPH_ERROR_RAISE_WARNING(SPH_ERROR_WARNING, "%s keyword (which sets up xdith for cam2) not found!\n",SPH_COMMON_KEYWORD_CAM2_DITHERING_X );
              }
              if ( cpl_propertylist_has ( pl, SPH_COMMON_KEYWORD_CAM2_DITHERING_Y) ){
                  ydith = cpl_propertylist_get_double( pl, SPH_COMMON_KEYWORD_CAM2_DITHERING_Y ) +
                          SPH_COMMON_KEYWORD_VALUE_CAM2_POS0_Y;
                  ydith=-ydith;
              } else {
               SPH_ERROR_RAISE_WARNING(SPH_ERROR_WARNING, "%s keyword (which sets up ydith for cam2) not found!\n",SPH_COMMON_KEYWORD_CAM2_DITHERING_Y );
              }

          } else {
               SPH_ERROR_RAISE_WARNING(SPH_ERROR_WARNING, "Chip index has a wrong value: %d (it must be 1 or 2)!\n",  cpl_propertylist_get_int( pl, SPH_COMMOM_KEYWORD_CHIP_INDEX) );
          }
      }
    }

    //setup sph_skycalc and create first raw
    if ( cpl_propertylist_has( pl, SPH_COMMON_KEYWORD_RA ) &&
            cpl_propertylist_has( pl, SPH_COMMON_KEYWORD_DEC ) &&
            (cpl_vector_get(timev, 0) != 0.0 ) )
    {
        sph_coord_initialise_skycalc(pl,skyinp,cpl_vector_get(timev,0), &skyout);
    } else {
        if (cpl_propertylist_has( pl, SPH_COMMON_KEYWORD_RA ) &&
            cpl_propertylist_has( pl, SPH_COMMON_KEYWORD_DEC ) ) {
            SPH_ERROR_RAISE_WARNING(SPH_ERROR_WARNING,
                "Failed to calculate paralactic angle (PA)!"
                " Please, verify if the %s and %s keywords are presented in the fits header.\n",
                SPH_COMMON_KEYWORD_RA,  SPH_COMMON_KEYWORD_DEC);
        } else {
            SPH_WARNING("Failed to calculate paralactic angle (PA)! "
                    "Time vector was not set correctly.")
        }
        SPH_WARNING("Paralactic angles are set to 0.0. The recipe will proceed anyway without "
                "de-rotation.")
        skyout.pa = 0.0;
    }


    if ( fctab->mode == SPH_FCTAB_SINGLE){
        SPH_INFO_MSG("fctab->mode == SPH_FCTAB_SINGLE (set coords + dithering & PA).");

        if ( cpl_propertylist_has( pl, SPH_COMMON_KEYWORD_DROT2_MODE) ){
            drot2_mode = 
                cpl_propertylist_get_string(pl, SPH_COMMON_KEYWORD_DROT2_MODE);

             if (!strcmp(drot2_mode, SPH_COMMON_KEYWORD_VALUE_DROT2_MODE_NO_DEROT)){
                SPH_INFO_MSG("STAT mode (nothing war rotated) mode was detected!");
                rerr = sph_fctable_add_row_single( fctab, cpl_vector_get( timev, 0 ),
                                                          clx - xdith, cly - ydith, -skyout.pa + skyout.alt);
            } else if (!strcmp(drot2_mode, SPH_COMMON_KEYWORD_VALUE_DROT2_MODE_PUPIL_STAB)){
                SPH_INFO_MSG("Pupil- stabilized mode was detected!");
                rerr = sph_fctable_add_row_single( fctab, cpl_vector_get( timev, 0 ),
                                                          clx - xdith, cly - ydith, -skyout.pa);
            } else if (!strcmp(drot2_mode, SPH_COMMON_KEYWORD_VALUE_DROT2_MODE_FIELD_STAB)) {
                SPH_INFO_MSG("Field- stabilized mode was detected! All PAs are set to 0.0.");
                rerr = sph_fctable_add_row_single( fctab, cpl_vector_get( timev, 0 ),
                                                          clx - xdith, cly - ydith, 0.0);
            } else {
                SPH_ERROR_RAISE_ERR(SPH_ERROR_ERROR,
                    "Not allowed value of the %s keyword is presented in the fits header!\n",
                    SPH_COMMON_KEYWORD_DROT2_MODE);
            }
        } else {
            SPH_ERR("ESO INS4 DROT2 MODE keyword doesn't exist! Stop... ");
        }

    } else if ( fctab->mode == SPH_FCTAB_DOUBLE && zimpol ) {
        SPH_INFO_MSG("fctab->mode == SPH_FCTAB_DOUBLE for ZIMPOL sub-system (set coords + dithering & PA).");

        if ( cpl_propertylist_has( pl, SPH_COMMON_KEYWORD_DROT2_MODE) ) {
            drot2_mode = 
                cpl_propertylist_get_string(pl, SPH_COMMON_KEYWORD_DROT2_MODE);
             if (!strcmp(drot2_mode, SPH_COMMON_KEYWORD_VALUE_DROT2_MODE_NO_DEROT)){
                SPH_INFO_MSG("STAT mode (nothing war rotated) mode was detected!");
                rerr = sph_fctable_add_row_double( fctab, cpl_vector_get( timev, 0 ),
                                                          clx - xdith, cly - ydith,
                                                          crx - xdith, cry - ydith,
                                                          -skyout.pa + skyout.alt);
            } else if (!strcmp(drot2_mode, SPH_COMMON_KEYWORD_VALUE_DROT2_MODE_PUPIL_STAB)){
                SPH_INFO_MSG("Pupil- stabilized mode was detected!");
                rerr = sph_fctable_add_row_double( fctab, cpl_vector_get( timev, 0 ),
                                                          clx - xdith, cly - ydith,
                                                          crx - xdith, cry - ydith,
                                                          -skyout.pa);
            } else if (!strcmp(drot2_mode, SPH_COMMON_KEYWORD_VALUE_DROT2_MODE_FIELD_STAB)) {
                SPH_INFO_MSG("Field- stabilized mode was detected! All PAs are set to 0.0.");
                rerr = sph_fctable_add_row_double( fctab, cpl_vector_get( timev, 0 ),
                                                          clx - xdith, cly - ydith,
                                                          crx - xdith, cry - ydith,
                                                          0.0);
            } else {
                SPH_ERROR_RAISE_ERR(SPH_ERROR_ERROR,
                    "Not allowed value of the %s keyword is presented in the fits header!\n",
                    SPH_COMMON_KEYWORD_DROT2_MODE);
            }
        } else {
            SPH_ERR("ESO INS4 DROT2 MODE keyword doesn't exist! Stop... ");
        }

    } else {
        SPH_INFO_MSG("fctab->mode == SPH_FCTAB_DOUBLE");
        rerr = sph_fctable_add_row_double( fctab, cpl_vector_get( timev, 0 ),
                                                      clx, cly, crx, cry, skyout.pa );
    }


    SPH_ERROR_ENSURE_GOTO_EXIT( rerr == CPL_ERROR_NONE, rerr );

    for (ff = 1; ff < nplanes; ++ff) {

        if ( cpl_propertylist_has( pl, SPH_COMMON_KEYWORD_RA ) &&
                cpl_propertylist_has( pl, SPH_COMMON_KEYWORD_DEC ) &&
                (cpl_vector_get(timev, 0) != 0.0 ))
        {
            sph_coord_do_skycalc(skyinp,
                                  cpl_vector_get(timev,ff),
                                  &skyout);
        } else {
            skyout.pa = 0.0;
        }


        if ( fctab->mode == SPH_FCTAB_SINGLE ){
            if ( drot2_mode ){
                if (!strcmp(drot2_mode, SPH_COMMON_KEYWORD_VALUE_DROT2_MODE_NO_DEROT)){
                    rerr = sph_fctable_add_row_single( fctab, cpl_vector_get( timev, ff ),
                                                          clx - xdith, cly - ydith, -skyout.pa + skyout.alt);
                } else if (!strcmp(drot2_mode, SPH_COMMON_KEYWORD_VALUE_DROT2_MODE_PUPIL_STAB)) {

                    rerr = sph_fctable_add_row_single( fctab, cpl_vector_get( timev, ff ),
                                                              clx - xdith, cly - ydith, -skyout.pa );
                }else if (!strcmp(drot2_mode, SPH_COMMON_KEYWORD_VALUE_DROT2_MODE_FIELD_STAB)) {
                    rerr = sph_fctable_add_row_single( fctab, cpl_vector_get( timev, ff ),
                                                              clx - xdith, cly - ydith, 0.0);
                } else {
                    rerr = sph_fctable_add_row_single( fctab, cpl_vector_get( timev, ff ),
                                                             clx - xdith, cly - ydith, 0.0);
                }
            } else {
                rerr = sph_fctable_add_row_single( fctab, cpl_vector_get( timev, ff ),
                                                          clx - xdith, cly - ydith, 0.0 );
            }

        } else if ( fctab->mode == SPH_FCTAB_DOUBLE && zimpol ) {
            SPH_INFO_MSG("fctab->mode == SPH_FCTAB_DOUBLE for ZIMPOL sub-system (set coords).");

            if ( drot2_mode ) {
                if (!strcmp(drot2_mode, SPH_COMMON_KEYWORD_VALUE_DROT2_MODE_NO_DEROT)){
                    //SPH_INFO_MSG("STAT mode (nothing war rotated) mode was detected!");
                    rerr = sph_fctable_add_row_double( fctab, cpl_vector_get( timev, ff ),
                                                              clx - xdith, cly - ydith,
                                                              crx - xdith, cry - ydith,
                                                              -skyout.pa + skyout.alt);
                } else if (!strcmp(drot2_mode, SPH_COMMON_KEYWORD_VALUE_DROT2_MODE_PUPIL_STAB)){
                    //SPH_INFO_MSG("Pupil- stabilized mode was detected!");
                    rerr = sph_fctable_add_row_double( fctab, cpl_vector_get( timev, ff ),
                                                              clx - xdith, cly - ydith,
                                                              crx - xdith, cry - ydith,
                                                              -skyout.pa);
                } else if (!strcmp(drot2_mode, SPH_COMMON_KEYWORD_VALUE_DROT2_MODE_FIELD_STAB)) {
                    //SPH_INFO_MSG("Field- stabilized mode was detected! All PAs are set to 0.0.");
                    rerr = sph_fctable_add_row_double( fctab, cpl_vector_get( timev, ff ),
                                                              clx - xdith, cly - ydith,
                                                              crx - xdith, cry - ydith,
                                                              0.0);
                } else {
                    rerr = sph_fctable_add_row_double( fctab, cpl_vector_get( timev, ff ),
                                                       clx, cly, crx, cry, 0.0 );
                }
            } else {
                SPH_ERR("ESO INS4 DROT2 MODE keyword doesn't exist! Stop... ");
            }

        } else {
            rerr = sph_fctable_add_row_double( fctab, cpl_vector_get( timev, ff ),
                                                          clx, cly, crx, cry, skyout.pa );
        }

        SPH_ERROR_ENSURE_GOTO_EXIT( rerr == CPL_ERROR_NONE, rerr );

    }


    cpl_propertylist_delete(pl); pl = NULL;
    cpl_vector_delete(timev); timev = NULL;
    return fctab;

EXIT:
    if (pl) cpl_propertylist_delete(pl); pl = NULL;
    cpl_vector_delete(timev); timev = NULL;
    sph_fctable_delete(fctab); fctab = NULL;
    return NULL;
}



/*----------------------------------------------------------------------------*/
/**
 * @brief Find a new field centre table
 * @param science_frame the science frame to find corr. fctable for
 * @param field_center_frames list of field centre frames to search
 *
 * @return new field centre table or NULL on error.
 *
 * Searches the field center list of frames for a field center that corresponds
 * to the input science frame.
 *
 * Matching is done on the filename for txt files and on the PRO FIRST RAW FRAME
 * for FITS files.
 *
 * If no corresponding frame is found NULL is returned.
 */
/*----------------------------------------------------------------------------*/
sph_fctable*
sph_fctable_find_fctable(
        const cpl_frame* raw_science_frame,
        const cpl_frameset* star_centre_calib_frameset
        )
{
    sph_fctable*    field_centre_tab = NULL;
    int             ff = 0;
    const cpl_frame*      star_centre_calib_frame = NULL;
    cpl_propertylist*   pl = NULL;
    char           base[256];
    char           ext[256];
    char           base_raw[256];
    char*          fname = NULL;
    char*          fname2 = NULL;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;

    cpl_ensure(raw_science_frame,CPL_ERROR_NULL_INPUT,NULL);
    cpl_ensure(star_centre_calib_frameset,CPL_ERROR_NULL_INPUT,NULL);

    fname = sph_filemanager_remove_dir(
            cpl_frame_get_filename(raw_science_frame)
            );
    sph_filemanager_split(fname,base_raw,ext);
    for (ff = 0; ff < cpl_frameset_get_size(star_centre_calib_frameset); ++ff) {
        const cpl_frame*      current_frame = 
            cpl_frameset_get_position_const(star_centre_calib_frameset, ff);
        fname2 = sph_filemanager_remove_dir(cpl_frame_get_filename(current_frame));
        sph_filemanager_split(fname2,base,ext);
        if ( strcasecmp(ext,"txt") != 0 ) {
            pl = sph_keyword_manager_load_properties(
                    cpl_frame_get_filename(current_frame),
                    0);
            if ( cpl_propertylist_has(pl,
                    SPH_COMMON_KEYWORD_PRO_FIRST_RAW_FRAME) ) {
                sph_filemanager_split(
                        cpl_propertylist_get_string(pl,
                        SPH_COMMON_KEYWORD_PRO_FIRST_RAW_FRAME) ,
                        base, ext);
                if ( strncasecmp(base,base_raw,
                        strlen(base_raw)) == 0 ) {
                    star_centre_calib_frame = current_frame;
                }
            }
            else {
                SPH_ERROR_RAISE_ERR( CPL_ERROR_BAD_FILE_FORMAT,
                        "The star center frame %s is missing "
                        "the %s keyword!",
                        cpl_frame_get_filename(current_frame),
                        SPH_COMMON_KEYWORD_PRO_FIRST_RAW_FRAME);
                break;
            }
        }
        if ( star_centre_calib_frame == NULL ) {
            if ( strncasecmp(base,base_raw,strlen(base_raw)) == 0 ) {
                star_centre_calib_frame = current_frame;
            }
        }
        cpl_free(fname2); fname2 = NULL;
    }
    cpl_free(fname); fname = NULL;
    if ( star_centre_calib_frame ) {
        field_centre_tab = sph_fctable_load_ascii(
                cpl_frame_get_filename( star_centre_calib_frame ));
        if ( field_centre_tab ) {
            return field_centre_tab;
        }
        else {
            cpl_error_reset();
            field_centre_tab = sph_fctable_load_fits(
                    cpl_frame_get_filename( star_centre_calib_frame ));
        }
    }

    return NULL;
}

/**@}*/
