/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
//#include <config.h>
#endif

/* for mkstemp */
#ifndef _XOPEN_SOURCE
#define _XOPEN_SOURCE 500
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/
#include "sph_error.h"
#include "sph_filemanager.h"
#include <cpl.h>
#include <strings.h>
#include <string.h>
#include <stdlib.h>

static sph_filemanager* sphfileman = NULL;
#define SPH_FILEMANAGER_MAX_TMP_FILES 10000

/*----------------------------------------------------------------------------*/
/**
 * @defgroup A SPHERE API Module
 * @par Synopsis:
 * @code
 * typedef _module_ {
 * } module
 * @endcode
 * @par Desciption:
 *
 * This module provides ...
 */
/*----------------------------------------------------------------------------*/
/**@{*/
/*----------------------------------------------------------------------------*/
/**
 * @brief Create new filemaneger singleton
 * @param
 *
 * @return pointer to filemanager
 *
 * Creates a new filemanager and returns pointer to it. Implemented
 * as singleton.
 *
 */
/*----------------------------------------------------------------------------*/
sph_filemanager*
sph_filemanager_new(void) {
    int ii;
    if ( !sphfileman ) {
        sphfileman = cpl_calloc(1,sizeof(sph_filemanager) );
        sphfileman->workdir = cpl_calloc(2,sizeof(char));
        sphfileman->tmpfiles = cpl_calloc( SPH_FILEMANAGER_MAX_TMP_FILES, sizeof(char*));
        for ( ii = 0 ; ii < SPH_FILEMANAGER_MAX_TMP_FILES; ++ii ) {
            sphfileman->tmpfiles[ii] = cpl_calloc( 256, sizeof(char));
        }
        sprintf( sphfileman->workdir, "." );
    }
    return sphfileman;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Split a string at / and return name without path
 * @param filename to split
 * @return filename without path
 *
 * Splits the input string at the last / character and
 * returns the trailing part.
 *
 *----------------------------------------------------------------------------*/
char*
sph_filemanager_remove_dir( const char* filename ) {
    unsigned int     cc      = 0;
    int     isext   = 1;
    unsigned int     splitpos = strlen(filename);
    char*   result  = cpl_calloc(strlen(filename)+1, sizeof(char));

    for (cc = strlen(filename); cc > 0; --cc) {
        if ( filename[cc - 1] == '/' && isext ) {
            splitpos = cc;
            isext = 0;
        }
    }

    if ( splitpos == strlen(filename) ) {
        (void)strcpy(result,filename);
        return result;
    }
    strncpy( result, &filename[splitpos], strlen(filename) - splitpos + 1);

    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Checks if a filename is a fits
 * @param filename to test
 * @return 1 if fits or 0 if not
 *
 */
/*----------------------------------------------------------------------------*/
int sph_filemanager_is_fits( const char * filename ) {
	char			base[256];
	char			ext[256];

    sph_filemanager_split(filename,base,ext);
    if ( strcasecmp(ext,"fits") == 0 ) {
    	return 1;
    }
	return 0;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Find and return the basename part of a filename
 * @param self  Filename to search
 * @return The basename part of the string, or NULL on NULL input
 * @note The return value may be the whole string, or the empty string, or NULL
 * @see basename()
 *
 *
 */
/*----------------------------------------------------------------------------*/
const char * sph_filemanager_get_basename(const char* self)
{
    const char * p = self ? strrchr(self, '/') : NULL;

    return p ? p + 1 : self;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief Split a string at .
 * @param filename to split
 * @param base  base part, before . (output)
 * @param ext   extension part, after . (output)
 * @return error code
 *
 * Splits the input string at the last . character and places the
 * results in base and ext (without the .).
 * If no , is in the input string, a copy of input is returned in the base.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_filemanager_split( const char* filename, char* base, char* ext ) {
    unsigned int     cc      = 0;
    int     isext   = 1;
    unsigned int     splitpos = strlen(filename);
    base[0] = '\0';
    ext[0] = '\0';
    for (cc = strlen(filename); cc > 0; --cc) {
        if ( filename[cc - 1] == '.' && isext ) {
            splitpos = cc;
            isext = 0;
        }
    }

    if ( splitpos == strlen(filename) ) {
        // No . found, so return filename as base
        strncpy(base,filename,splitpos);
        base[splitpos]='\0';
        return CPL_ERROR_NONE;
    }
    if ( splitpos > 1) {
        strncpy(base,filename,splitpos - 1);
        base[splitpos - 1]='\0';
    }
    strncpy( ext, &filename[splitpos], strlen(filename) - splitpos + 1);
    ext[strlen(filename) - splitpos + 1]='\0';
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create a filename using base and number
 * @param filename      the filename
 * @param ii        a number
 *
 * @return new filename or NULL
 *
 * Creates a new filename based on the filename and a number.
 * If the input filename contains a . the filename is split first,
 * so an input of "myfile.fits",1 results in in "myfile_00001.fits"
 *
 */
/*----------------------------------------------------------------------------*/
char*
sph_filemanager_get_numbered_filename( const char* filename, int ii ) {
    char*    name;
    char*    filebase = cpl_calloc(strlen(filename) + 1,sizeof(char));
    char*    fileext = cpl_calloc(strlen(filename) + 1,sizeof(char));

    cpl_ensure( filename, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure( ii < 10000, CPL_ERROR_ILLEGAL_INPUT, NULL);

    sph_filemanager_split( filename, filebase, fileext );
    name = cpl_sprintf("%s_%05d.%s",filebase,ii,fileext);
    cpl_free(fileext); fileext = NULL;
    cpl_free(filebase); filebase = NULL;
    return name;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new unique duplicate of a frame.
 * @param the input frame to duplicate
 *
 * @return new duplicate frame
 *
 * This function creates a new duplicate frame that is identical to the
 * input except for the filename which is a unique numbered filename, with
 * the input frames filename as a base.
 * @see sph_filemanager_get_numbered_filename_new_file
 *
 */
/*----------------------------------------------------------------------------*/
cpl_frame*
sph_filemanager_get_unique_duplicate_frame( const cpl_frame* inframe ) {
    cpl_frame* result = NULL;
    char*      newname = NULL;
    char*      fname = NULL;

    cpl_ensure(inframe, CPL_ERROR_NULL_INPUT, NULL);
    result = cpl_frame_duplicate(inframe);
    cpl_ensure(result,cpl_error_get_code(),NULL);
    fname = sph_filemanager_remove_dir(cpl_frame_get_filename(inframe));
    newname = sph_filemanager_get_numbered_filename_new_file(fname);
    cpl_frame_set_filename(result,newname);
    cpl_free(newname);
    cpl_free(fname);
    return result;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief Create a filename using base making it unique
 * @param filename      the filename
 *
 * @return new filename or NULL
 *
 * Creates a new filename based on the filename and a number.
 * If the input filename contains a . the filename is split first,
 * so an input of "myfile.fits",1 results in "myfile_00001.fits"
 * Contrary to sph_filemanager_get_numbered_filename this function
 * automatically checks which files already exists in the current
 * working directory and set the number to be the minimum which does
 * not yet exist. For example calling it with "myfile.fits" in a
 * directory already containing "myfile_00000.fits" and "myfile_00001.fits"
 * results in "myfile_00002.fits"
 */
/*----------------------------------------------------------------------------*/
char*
sph_filemanager_get_numbered_filename_new_file( const char* filename ) {
    char*    name;
    char*    tmpn = cpl_calloc(strlen(filename) + 10,sizeof(char));
    char*    filebase = cpl_calloc(strlen(filename) + 1,sizeof(char));
    char*    fileext = cpl_calloc(strlen(filename) + 1,sizeof(char));
    FILE*   filep = NULL;
    int     ii = 0;

    sph_filemanager_split( filename, filebase, fileext );
    sprintf(tmpn,"%s_%05d.%s",filebase,ii,fileext);
    filep = fopen(tmpn,"r");
    while ( filep && ii < 10001 ) {
        fclose(filep); filep = NULL;
        sprintf(tmpn,"%s_%05d.%s",filebase,ii,fileext);
        filep = fopen(tmpn,"r");
        if ( filep ) ii++;
    }
    if ( ii >= 10000 ) {
        SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_INPUT,
                "Too many files with name base %s already.", filebase );
    }
    name = cpl_sprintf("%s_%05d.%s",filebase,ii,fileext);
    cpl_free(tmpn); tmpn = NULL;
    cpl_free(filebase); filebase = NULL;
    cpl_free(fileext); fileext = NULL;
    return name;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create new filename based on base part using an inserted string
 * @param filename      the filename to create a variance of
 * @param insertion     the insertion string
 * @return new filename
 *
 * Creates a new filename that is the same as the input except the the
 * insertion string is inserted with a preceding "_" before the .
 * So, for example, "myfile.fits" becomes "mystile_first.fits" when
 * insertion is "first".
 *
 */
/*----------------------------------------------------------------------------*/
char*
sph_filemanager_new_filename_from_base( const char* filename,
                                        const char* insert) {
    char* base = cpl_calloc(strlen(filename) + 1, sizeof(char));
    char* ext = cpl_calloc(strlen(filename) + 1, sizeof(char));
    char* result = NULL;

    if (sph_filemanager_split(filename,base,ext) != CPL_ERROR_NONE) {
        (void)cpl_error_set(cpl_func, CPL_ERROR_ILLEGAL_INPUT);
    } else {
        result = cpl_sprintf("%s_%s.%s",base,insert,ext);
        if ( result[strlen(result)-1] == '.') {
            result[strlen(result)-1] = '\0';
        }
    }
    cpl_free(base);
    cpl_free(ext);
    return result;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief Create new filename as copy with new extension (after .)
 * @param filename      the filename to create a variance of
 * @param insertion     the new ext
 * @return new filename
 *
 * Creates a new filename that is the same as the input except the the
 * extension string (after the ,) is changed .
 * So, for example, "myfile.fits" becomes "myfile.txt" when
 * insertion is "txt".
 *
 */
/*----------------------------------------------------------------------------*/
char*
sph_filemanager_filename_new_ext( const char* filename, const char* ext) {
    char*        base = cpl_calloc(strlen(filename) + 1, sizeof(char));
    char*        oext = cpl_calloc(strlen(filename) + 1, sizeof(char));
    char*       result = NULL;

    if (sph_filemanager_split(filename,base,oext) != CPL_ERROR_NONE) {
        (void)cpl_error_set(cpl_func, CPL_ERROR_ILLEGAL_INPUT);
    } else {
        result = cpl_sprintf("%s.%s",base,ext);
        if ( result[strlen(result)-1] == '.') {
            result[strlen(result)-1] = '\0';
        }
    }

    cpl_free( base );
    cpl_free( oext );
    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create a temporary filename
 * @param base      the base of the filename
 *
 * @return a new temporary filename
 *
 * A new filename is created using the input as base and adding
 * the six new random unique characters using mkstemp.
 *
 * Should in future use sph_filemanager_get_numbered_file instead.
 *
 * The new file is not added to the filemaneger temporary file list.
 * @deprecated
 *
 */
/*----------------------------------------------------------------------------*/
char*
sph_filemanager_get_tmp_filename( const char* filename ) {
    char*    name = NULL;

    if ( !sphfileman ) sphfileman = sph_filemanager_new();

    if ( sphfileman != NULL) {
        char*    tmpn = cpl_calloc(strlen(filename) + 10,sizeof(char));
        char*    base = cpl_calloc(strlen(filename) + 1,sizeof(char));
        char*    ext = cpl_calloc(strlen(filename) + 1,sizeof(char));
        int      fp;

        sph_filemanager_split(filename,base,ext);
        sprintf( tmpn, "%s.XXXXXX", base );
        fp = mkstemp( tmpn );
        close( fp );
        unlink(tmpn);
        name = cpl_sprintf( "%s.%s", tmpn, ext );
        cpl_free(tmpn);
        cpl_free(base);
        cpl_free(ext);
    }
    return name;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Add a new filename to temporary file lists
 * @param base      the base of the filename
 *
 * The file is added to the temporary file lists that will be cleaned
 * when the filemanager is destroyed or cleaned.
 *
 */
/*----------------------------------------------------------------------------*/
void
sph_filemanager_add_tmp_file( const char* filename ) {
    int ii;
    if ( !sphfileman ) sphfileman = sph_filemanager_new();
    if ( !sphfileman ) return;
    if ( sphfileman->tmpfiles == NULL ) {
        sphfileman->tmpfiles = cpl_calloc( SPH_FILEMANAGER_MAX_TMP_FILES, sizeof(char*));
        for ( ii = 0 ; ii < SPH_FILEMANAGER_MAX_TMP_FILES; ++ii ) {
            sphfileman->tmpfiles[ii] = cpl_calloc( 256, sizeof(char));
        }
    }
    (void)strcpy(sphfileman->tmpfiles[sphfileman->ntmpfiles],filename);
    sphfileman->ntmpfiles++;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new temporary frame
 * @param filename base
 * @param  tag
 * @param group
 *
 * @return new cpl_frame or NULL
 *
 * Creates a new cpl_frame wich will be added to the list of temporary
 * frames, so the file will be deleted once the filemanager is destroyed or
 * cleaned.
 * The cpl_frame returned should be deleted by the calling code.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_frame*    sph_filemanager_create_temp_frame( const char* filebase,
        const char* tag, cpl_frame_group group ) {
    cpl_frame*        frame    = NULL;
    char*            template = NULL;
    char*             str = cpl_calloc(strlen(filebase) + 10, sizeof(char));
    char*             base = cpl_calloc(strlen(filebase) + 10, sizeof(char));
    char*             ext = cpl_calloc(strlen(filebase) + 10, sizeof(char));
    if ( !sphfileman ) sphfileman = sph_filemanager_new();
    if ( !sphfileman ) return NULL;
    frame = cpl_frame_new();
    sph_filemanager_split(filebase,base,ext);
    if ( strlen(ext) == 0 ) {
        sprintf(str,"%s.fits",filebase);
    }
    else {
        (void)strcpy(str,filebase);
    }
    if ( frame ) {
        cpl_frame_set_tag( frame, tag);
        cpl_frame_set_group( frame, group);
        template = sph_filemanager_get_tmp_filename(str);
        cpl_frame_set_filename( frame, template );
        sph_filemanager_add_tmp_file(template);
        cpl_free(template); template = NULL;
    }
    cpl_free(str); str = NULL;
    cpl_free(base); base = NULL;
    cpl_free(ext); ext = NULL;
    return frame;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Clean temporary files
 *
 * @return
 *
 * Deletes all temporary files
 *
 */
/*----------------------------------------------------------------------------*/
void sph_filemanager_clean(void) {
    int ii;
    if ( !sphfileman ) return;
    if ( sphfileman->tmpfiles ) {
        while (sphfileman->ntmpfiles) {
            unlink(sphfileman->tmpfiles[sphfileman->ntmpfiles-1]);
            sphfileman->ntmpfiles--;
        }
        for ( ii = 0 ; ii < SPH_FILEMANAGER_MAX_TMP_FILES; ++ii ) {
            cpl_free(sphfileman->tmpfiles[ii]);
        }
        cpl_free(sphfileman->tmpfiles);sphfileman->tmpfiles=NULL;
    }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Delete the singleton filemanager
 *
 * @param   noclean if set, does not delete temporary files
 *
 *
 * Deletes the filemanager singleton and if noclean is zero also
 * delete all temporary files from filesystem.
 */
/*----------------------------------------------------------------------------*/
void
sph_filemanager_delete( int noclean ) {
    int ii;
    if (sphfileman) {
        if ( sphfileman->workdir ) cpl_free(    sphfileman->workdir);
        if ( noclean ) {
            if ( sphfileman->tmpfiles ) {
                while (sphfileman->ntmpfiles) {
                    sphfileman->ntmpfiles--;
                }
                for ( ii = 0 ; ii < SPH_FILEMANAGER_MAX_TMP_FILES; ++ii ) {
                    cpl_free(sphfileman->tmpfiles[ii]);
                }
                cpl_free(sphfileman->tmpfiles);sphfileman->tmpfiles=NULL;
            }
        }
        else {
            sph_filemanager_clean();
        }
        cpl_free(sphfileman);sphfileman = NULL;
    }
}
/**@}*/
