/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#include <longnam.h>
#include <fitsio.h>
#include <cpl.h>
#include "sph_error.h"
#include "sph_fits.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_fits  Easy FITS access for reading and writing.
 *
 * @par Description:
 *
 * This module contains routines to handle FITS in- and output. It contains
 * routines that perform essentially the tasks in CFITSIO but are easier to use.
 * Also contained are routines for FITS access that are particular about the
 * sphere DRH types and objects.
 *
 */
/*----------------------------------------------------------------------------*/


/**@{*/

sph_error_code SPH_FITS_GENERAL             = SPH_FITS_ERR_START + 0;
sph_error_code SPH_FITS_ILLEGAL_EXTENSION   = SPH_FITS_ERR_START + 1;
sph_error_code SPH_FITS_ILLEGAL_TYPE        = SPH_FITS_ERR_START + 2;
sph_error_code SPH_FITS_ILLEGAL_PLANE       = SPH_FITS_ERR_START + 3;


/*----------------------------------------------------------------------------*/
/**
 @brief    Get the number of planes of a FITS file
 @param    self      The name of the FITS file
 @param    extnum    The extension of the fitsfile to check ( 0 being HDU)
 @return   number of planes, -1 on error.

 Returns the number of planes in a FITS file. If the FITS file has NAXIS
 equal 1 or 2, the function returns 1.

 */
/*----------------------------------------------------------------------------*/
int sph_fits_get_nplanes(const char* self, int extnum )
{
    cpl_propertylist* pl     = NULL;
    int               result = 0;
    int               naxis;

    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, -1);

    pl = cpl_propertylist_load_regexp(self, extnum, "^NAXIS", 0);
    if (pl == NULL) {
        (void)cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                    "extnum=%d in file=%s", extnum, self);
        return -1;
    }

    naxis = cpl_propertylist_get_int(pl, "NAXIS");
    if (cpl_error_get_code()) {
        (void)cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                    "extnum=%d in file=%s", extnum, self);
        result = -1;
    } else if (naxis  == 3) {
        result = cpl_propertylist_get_int(pl, "NAXIS3");
        if (cpl_error_get_code()) {
            /* Malformed FITS file */
            (void)cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                        "extnum=%d in file=%s", extnum, self);
            result = -1;
        }
    } else if (naxis > 3) {
        (void)cpl_error_set_message(cpl_func, CPL_ERROR_UNSUPPORTED_MODE,
                                    "NAXIS=%d, extnum=%d in file=%s",
                                    naxis, extnum, self);
        result = -1;
    } else if (naxis < 1) {
        (void)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                    "NAXIS=%d, extnum=%d in file=%s",
                                    naxis, extnum, self);
        result = -1;
    } else {
        result = 1;
    }

    cpl_propertylist_delete(pl);

    return result;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Check if a given file is a FITS cube
 @param    self      The name of the FITS file
 @param    extnum    The extension of the fitsfile to check ( 0 being HDU)
 @return   0 if not a cube, 1 if a cube, -1 on error.

 Check if a FITS file with the given name is a cube with NAXIS equal
 to three.

 */
/*----------------------------------------------------------------------------*/
int sph_fits_test_iscube( const char* self, int extnum )
{
    cpl_propertylist* pl     = NULL;
    int               result = 0;
    int               naxis;

    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, -1);

    pl = cpl_propertylist_load_regexp(self, extnum, "^NAXIS$", 0);
    if (pl == NULL) {
        (void)cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                    "extnum=%d in file=%s", extnum, self);
        return -1;
    }

    naxis = cpl_propertylist_get_int(pl, "NAXIS");
    if (cpl_error_get_code()) {
        (void)cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                    "extnum=%d in file=%s", extnum, self);
        result = -1;
    } else {
        result = naxis == 3 ? 1 : 0;
    }

    cpl_propertylist_delete(pl);

    return result;
}



/*----------------------------------------------------------------------------*/
/**
 @brief    Update a property in the header of a fits extension
 @param    filename     The FITS filename
 @param    prop         The property to update
 @param    extnum       The extension of the fitsfile to update ( 0 being HDU)

 @return   0 if all ok, error code otherwise.

 Update a specific property in a FITS header. The routine updates the value
 of the keywords corresponding to the property and creates it if
 it does not already exist.

 The extnum is a number from 0 - N-1, where N is the total number of extensions
 (including the HDU) of the file. 0 refers to the HDU.

 In CPL properties are key-value pairs
 that are represented as keywords in FITS files. See the CPL documentation for more
 details.

 */
/*----------------------------------------------------------------------------*/

int sph_fits_update_property(const char* filename, const cpl_property* prop,
                             int extnum)
{
    int         rerr        = 0;
    int         fstatus     = 0;
    int         maxhdus     = 0;
    int         hdutype     = 0;
    cpl_type    type        = 0;
    const char* propname    = NULL;
    const char* propcomment;
    fitsfile*   fitshandle  = NULL;

    if (cpl_fits_get_mode() == CPL_FITS_START_CACHING &&
        cpl_fits_set_mode(CPL_FITS_RESTART_CACHING) != CPL_ERROR_NONE) {
        return (int)cpl_error_set_where(cpl_func);
    }

    rerr = ffopen            ( &fitshandle, filename, READWRITE , &fstatus );
    rerr = fits_get_num_hdus ( fitshandle, &maxhdus, &fstatus );

    if ( maxhdus < extnum ) {
        sph_error_raise( SPH_FITS_ILLEGAL_EXTENSION, __FILE__, __func__, __LINE__, SPH_ERROR_ERROR,
                         "Only have %d extensions in file %s, so cant "
                         "go to extension %d!",
                         maxhdus, filename, extnum
                        );
        return SPH_FITS_ILLEGAL_EXTENSION;
    }

    rerr = fits_movabs_hdu(fitshandle, extnum + 1, &hdutype, &fstatus);

    type = cpl_property_get_type(prop);

    if ( type == CPL_TYPE_INT ) {
        int ival    = cpl_property_get_int(prop);

        propname    = cpl_property_get_name(prop);
        propcomment = cpl_property_get_comment(prop);
        rerr        = fits_update_key(fitshandle, TINT,
                                      propname, (void*)&ival,
                                      propcomment, &fstatus);
    }
    if ( type == CPL_TYPE_FLOAT ) {
        float fval  = cpl_property_get_float(prop);

        propname    = cpl_property_get_name(prop);
        propcomment = cpl_property_get_comment(prop);
        rerr        = fits_update_key(fitshandle, TFLOAT,
                                      propname, (void*)&fval,
                                      propcomment, &fstatus);
    }
    if ( type == CPL_TYPE_DOUBLE ) {
        double dval = cpl_property_get_double(prop);

        propname    = cpl_property_get_name(prop);
        propcomment = cpl_property_get_comment(prop);
        rerr        = fits_update_key(fitshandle, TDOUBLE,
                                      propname, (void*)&dval,
                                      propcomment, &fstatus);
    }
    if ( type == CPL_TYPE_STRING ) {
        const char* sval = cpl_property_get_string(prop);

        propname    = cpl_property_get_name(prop);
        propcomment = cpl_property_get_comment(prop);
CPL_DIAG_PRAGMA_PUSH_IGN(-Wcast-qual);
        rerr        = fits_update_key(fitshandle, TSTRING,
                                      propname, (void*)sval,
                                      propcomment, &fstatus);
CPL_DIAG_PRAGMA_POP;
    }
    if ( type == CPL_TYPE_CHAR ) {
        /* Copied from CPL: */
        /*
         * Character properties should be represented as a single
         * character string, not as its numerical equivalent.
         */

        char cval[2] = {cpl_property_get_char(prop), '\0'};

        propname    = cpl_property_get_name(prop);
        propcomment = cpl_property_get_comment(prop);
        rerr        = fits_update_key(fitshandle, TSTRING,
                                      propname, (void*)&cval,
                                      propcomment, &fstatus);
    }


    rerr = fits_close_file( fitshandle, &fstatus );
    if ( rerr ) {
        fits_report_error( stdout, fstatus );
        sph_error_raise( SPH_FITS_GENERAL, __FILE__, __func__, __LINE__,
                         SPH_ERROR_ERROR,
                         "Could not update property %s"
                         "A FITS I/O error with CFITSIO code %d occured. ",
                         propname, fstatus
                        );
        return SPH_FITS_GENERAL;
    }

    return rerr;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Update the timestamp in the header of a fits extension
 @param    filename     The FITS filename
 @param    prop         The keyword to update with the timestamp
 @param    extnum       The extension of the fitsfile to update ( 0 being HDU)

 @return   0 if all ok, error code otherwise.

 Update the timestamp in a FITS header. The routine updates the value
 of the given keyword with the current system time -- it creates it if
 it does not already exist.

 The extnum is a number from 0 - N-1, where N is the total number of extensions
 (including the HDU) of the file. 0 refers to the HDU.

 In CPL properties are key-value pairs
 that are represented as keywords in FITS files. See the CPL documentation for more
 details.

 */
/*----------------------------------------------------------------------------*/
int sph_fits_update_timestamp(const char* filename, const char* timekeyword,
                              int extnum) {
    int         rerr        = 0;
    int         fstatus     = 0;
    int         maxhdus     = 0;
    int         hdutype     = 0;
    char*       propcomment = NULL;
    char*       cval        = NULL;
    fitsfile*   fitshandle  = NULL;
    int         timeref     = 0;

    if (cpl_fits_get_mode() == CPL_FITS_START_CACHING &&
        cpl_fits_set_mode(CPL_FITS_RESTART_CACHING) != CPL_ERROR_NONE) {
        return (int)cpl_error_set_where(cpl_func);
    }

    rerr = ffopen            ( &fitshandle, filename, READWRITE , &fstatus );
    rerr = fits_get_num_hdus ( fitshandle, &maxhdus, &fstatus );

    if ( maxhdus < extnum ) {
        sph_error_raise( SPH_FITS_ILLEGAL_EXTENSION, __FILE__, __func__, __LINE__, SPH_ERROR_ERROR,
                         "Only have %d extensions in file %s, so cant "
                         "go to extension %d!",
                         maxhdus, filename, extnum
                        );
        return SPH_FITS_ILLEGAL_EXTENSION;
    }

    rerr = fits_movabs_hdu(fitshandle, extnum + 1, &hdutype, &fstatus);

    if ( ! rerr ) {
        const char* propname    = timekeyword;
        rerr        = fits_get_system_time( cval, &timeref, &fstatus );
        rerr        |= fits_update_key(fitshandle, TSTRING,
                                      propname, cval,
                                      propcomment, &fstatus);
    }
    else {
        sph_error_raise( SPH_FITS_ILLEGAL_EXTENSION, __FILE__, __func__, __LINE__, SPH_ERROR_ERROR,
                         "Only have %d extensions in file %s, so cant "
                         "go to extension %d!",
                         maxhdus, filename, extnum
                        );
        return SPH_FITS_ILLEGAL_EXTENSION;
    }
    rerr = fits_close_file( fitshandle, &fstatus );

    if ( rerr ) {
        fits_report_error( stdout, fstatus );
        sph_error_raise( SPH_FITS_GENERAL, __FILE__, __func__, __LINE__,
                         SPH_ERROR_ERROR,
                         "Could not update date"
                         "A FITS I/O error with CFITSIO code %d occured. ",
                         fstatus
                        );
        return SPH_FITS_GENERAL;
    }

    return rerr;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Update a FITS table in a fits extension
 @param    filename     The FITS filename
 @param    table        The cpl_table to update
 @param    extnum       The extension of the fitsfile to update ( 0 being HDU)

 @return   0 if all ok, error code otherwise.

 Update a specific table in a FITS extension.

 The extnum is a number from 0 - N-1, where N is the total number of extensions
 (including the HDU) of the file. 0 refers to the HDU.

 See the CPL documentation for more details on the cpl_table.

 */
/*----------------------------------------------------------------------------*/
int sph_fits_update_table( const char* filename, cpl_table* table, int extnum ) {
    int         rerr        = 0;
    int         fstatus     = 0;
    int         maxhdus     = 0;
    int         hdutype     = 0;
    long        nrows       = 0;
    int         ncols       = 0;
    fitsfile*   fitshandle  = NULL;
    int         cc          = 0;
    int         type        = 0;
    const char* colname     = NULL;
    void*       coldata     = NULL;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

    if (cpl_fits_get_mode() == CPL_FITS_START_CACHING &&
        cpl_fits_set_mode(CPL_FITS_RESTART_CACHING) != CPL_ERROR_NONE) {
        return (int)cpl_error_set_where(cpl_func);
    }

    rerr = ffopen        ( &fitshandle, filename, READWRITE , &fstatus );

    rerr = fits_get_num_hdus( fitshandle, &maxhdus, &fstatus );

    if ( maxhdus < extnum ) {
        sph_error_raise( SPH_FITS_ILLEGAL_EXTENSION, __FILE__, __func__,
                        __LINE__, SPH_ERROR_ERROR,
                        "Only have %d extensions in file %s, so cant "
                        "go to extension %d!",
                        maxhdus, filename, extnum);
        return SPH_FITS_ILLEGAL_EXTENSION;
    }

    rerr = fits_movabs_hdu( fitshandle, extnum + 1, &hdutype, &fstatus );
    rerr = fits_get_num_cols( fitshandle, &ncols, &fstatus);
    rerr = fits_get_num_rows( fitshandle, &nrows, &fstatus);

    if ( rerr ) {
        sph_error_raise( SPH_FITS_GENERAL,__FILE__, __func__,
                        __LINE__,SPH_ERROR_ERROR,
                        "Error reading table from %s extension %d\n",
                        filename, extnum);
        return SPH_FITS_GENERAL;
    }

    ncols = cpl_table_get_ncol(table);
    nrows = cpl_table_get_nrow(table);

    for ( cc = 0; cc < ncols ;cc++)
    {
        if ( cc == 0 )  colname = cpl_table_get_column_name(table);
        else            colname = cpl_table_get_column_name(NULL);

        if ( cpl_table_get_column_type(table,colname)  == CPL_TYPE_FLOAT )
        {
            type = TFLOAT;
            coldata = (void*)cpl_table_get_data_float(table, colname);

            if ( !coldata ) {
                sph_error_raise( (int)cpl_error_get_code(), __FILE__, __func__,
                                 __LINE__,SPH_ERROR_ERROR,
                                 "Could not access column data.\n"
                                 "CPL code: %d => %s", cpl_error_get_code(),
                                 cpl_error_get_message());
                return (int)cpl_error_get_code();
            }
        }
        if ( cpl_table_get_column_type(table,colname)  == CPL_TYPE_INT )
        {
            type    = TINT;
            coldata = (void*)cpl_table_get_data_int(table, colname);

            if ( !coldata ) {
                sph_error_raise( (int)cpl_error_get_code(),__FILE__, __func__,
                                 __LINE__,SPH_ERROR_ERROR,
                                 "Could not access column data.\n"
                                 "CPL code: %d => %s", cpl_error_get_code(),
                                 cpl_error_get_message()
                                );

                return (int)cpl_error_get_code();
            }

        }
        if ( cpl_table_get_column_type(table,colname)  == CPL_TYPE_DOUBLE )
        {
            type    = TDOUBLE;
            coldata = (void*)cpl_table_get_data_double(table, colname);

            if ( !coldata ) {
                sph_error_raise( (int)cpl_error_get_code(),__FILE__, __func__,
                                __LINE__,SPH_ERROR_ERROR,
                                "Could not access column data.\n"
                                "CPL code: %d => %s", cpl_error_get_code(),
                                 cpl_error_get_message());
                return (int)cpl_error_get_code();
            }
        }

        rerr = fits_write_col(fitshandle, type, cc+1, 1,
                       1, nrows, coldata, &fstatus);
    }
    rerr = fits_close_file( fitshandle, &fstatus );
    if ( rerr ) {
        fits_report_error( stdout, fstatus );
        sph_error_raise( SPH_FITS_GENERAL, __FILE__, __func__, __LINE__,
                         SPH_ERROR_ERROR,
                         "Could not update table"
                         "A FITS I/O error with CFITSIO code %d occured. ",
                         fstatus
                        );
        return SPH_FITS_GENERAL;
    }
    return rerr;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Update a an image in a FITS extension
 @param    filename     The FITS filename
 @param    image        The cpl_image to update
 @param    plane        The plane
 @param    extnum       The extension of the fitsfile to update ( 0 being HDU)

 @return   0 if all ok, error code otherwise.

 Update a specific image in a FITS extension.

 The extnum is a number from 0 - N-1, where N is the total number of extensions
 (including the HDU) of the file. 0 refers to the HDU.

 The plane is a the 3rd pixel dimension and goes from 0 - N-1, where N is the
 number of pixels in the z-axis.

 See the CPL documentation for more details on the cpl_image.

 */
/*----------------------------------------------------------------------------*/
int sph_fits_update_image(const char* filename,
                          cpl_image* image,
                          int plane,
                          int extnum)
{
    int         rerr        = 0;
    int         fstatus     = 0;
    int         maxhdus     = 0;
    int         hdutype     = 0;
    char*       propcomment = NULL;
    int         ival        = 0;
    fitsfile*   fitshandle  = NULL;
    int         datatype    = 0;
    long        fpixel[3]   = { 1, 1, 0 };
    long        nelements   = 0;
    int         bpp         = 0;
    int         naxis       = 0;

    cpl_ensure_code(filename,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(image,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(plane > 0, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(extnum >= 0, CPL_ERROR_ILLEGAL_INPUT);

    if (cpl_fits_get_mode() == CPL_FITS_START_CACHING &&
        cpl_fits_set_mode(CPL_FITS_RESTART_CACHING) != CPL_ERROR_NONE) {
        return (int)cpl_error_set_where(cpl_func);
    }

    rerr = ffopen        ( &fitshandle, filename, READWRITE , &fstatus );
    rerr = fits_get_num_hdus( fitshandle, &maxhdus, &fstatus );

    if ( maxhdus < extnum ) {
        cpl_msg_error(__FILE__,
                      "Only have %d extensions in file %s, so cant "
                      "go to extension %d!",
                      maxhdus, filename, extnum);
        return 1;
    }

    rerr = fits_movabs_hdu( fitshandle, extnum +1, &hdutype, &fstatus );
    rerr = fits_read_key( fitshandle, TINT, "NAXIS", (void*)&naxis,
            propcomment, &fstatus );

    rerr = fits_read_key( fitshandle, TINT, "NAXIS1", (void*)&ival,
                               propcomment, &fstatus );
    rerr        = fits_read_key( fitshandle, TINT, "NAXIS2", (void*)&ival,
                               propcomment, &fstatus );
    if ( naxis > 2 ) {
        rerr        = fits_read_key( fitshandle, TINT, "NAXIS3", (void*)&ival,
                propcomment, &fstatus );
    }
    else {
        ival = 0;
    }

    if ( rerr ) {
        ival = 0;
        rerr = 0;
    }

    if ( plane > ival ) {
        sph_error_raise( SPH_FITS_ILLEGAL_PLANE, __FILE__, NULL,
                           __LINE__,SPH_ERROR_ERROR,
                           "Only have %d planes in file %s, so cant "
                           "write to plane %d!",
                           ival, filename, plane );
        return SPH_FITS_ILLEGAL_PLANE;
    }

    if ( cpl_image_get_type(image) == CPL_TYPE_FLOAT ) {
        datatype = TFLOAT;
        bpp = FLOAT_IMG;
    }

    if ( cpl_image_get_type(image) == CPL_TYPE_INT ) {
        datatype = TINT;
        bpp = LONG_IMG;
    }
    if ( cpl_image_get_type(image) == CPL_TYPE_DOUBLE ) {
        datatype = TDOUBLE;
        bpp = DOUBLE_IMG;
    }

    if ( bpp == 0 ) {
        sph_error_raise( SPH_FITS_ILLEGAL_TYPE, __FILE__, __func__, __LINE__,
                         SPH_ERROR_ERROR,
                         "Could not update image"
                         "The type %d provided is not supported. ",
                         cpl_image_get_type(image)
                        );
        return SPH_FITS_ILLEGAL_TYPE;
    }


    nelements = cpl_image_get_size_x(image) * cpl_image_get_size_y(image);

    fpixel[2] = plane;

    rerr = fits_write_pix( fitshandle, datatype, fpixel,
                         nelements, (void*)cpl_image_get_data(image), &fstatus);

    rerr = fits_close_file( fitshandle, &fstatus );
    if ( rerr ) {
        fits_report_error( stdout, fstatus );
        sph_error_raise( SPH_FITS_GENERAL, __FILE__, __func__, __LINE__,
                         SPH_ERROR_ERROR,
                         "Could not update image"
                         "A FITS I/O error with CFITSIO code %d occured. ",
                         fstatus
                        );
        return SPH_FITS_GENERAL;
    }
    return rerr;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Append an image to a FITS extension, extending a data cube
 @param    filename     The FITS filename
 @param    image        The cpl_image to update
 @param    extnum       The extension of the fitsfile to update ( 0 being HDU)

 @return   0 if all ok, error code otherwise.

 Append a specific image to a FITS extension. This adds another plane to that
 image extension -- in case that the FITS extension has only two axes defined,
 an SPH_FITS_ILLEGAL_PLANE error is returned.

 The extnum is a number from 0 - N-1, where N is the total number of extensions
 (including the HDU) of the file. 0 refers to the HDU.

 See the CPL documentation for more details on the cpl_image.

 */
/*----------------------------------------------------------------------------*/
int sph_fits_append_image(const char* filename, cpl_image* image, int extnum) {
    int         rerr        = 0;
    int         fstatus     = 0;
    int         maxhdus     = 0;
    int         hdutype     = 0;
    char*       propcomment = NULL;
    int         ival        = 0;
    fitsfile*   fitshandle  = NULL;
    int         datatype    = 0;
    long        fpixel[3]   = { 1, 1, 0 };
    long        nelements   = 0;
    int         bpp         = 0;
    long        naxes[3]    = { 0, 0, 0};
    int         bitpix      = 0;

    if (cpl_fits_get_mode() == CPL_FITS_START_CACHING &&
        cpl_fits_set_mode(CPL_FITS_RESTART_CACHING) != CPL_ERROR_NONE) {
        return (int)cpl_error_set_where(cpl_func);
    }

    rerr = ffopen        ( &fitshandle, filename, READWRITE , &fstatus );
    rerr = fits_get_num_hdus( fitshandle, &maxhdus, &fstatus );
    if ( maxhdus < extnum ) {
        sph_error_raise((int)cpl_error_get_code(),__FILE__, NULL,
                        __LINE__,SPH_ERROR_ERROR,
                      "Only have %d extensions in file %s, so cant "
                      "go to extension %d!",
                      maxhdus, filename, extnum);
        return 1;
    }
    rerr = fits_movabs_hdu( fitshandle, extnum +1, &hdutype, &fstatus );
    rerr = fits_read_key( fitshandle, TINT, "BITPIX", (void*)&ival,
                               propcomment, &fstatus );
    bitpix = ival;

    rerr = fits_read_key( fitshandle, TINT, "NAXIS1", (void*)&ival,
                               propcomment, &fstatus );
    naxes[0] = ival;
    rerr = fits_read_key( fitshandle, TINT, "NAXIS2", (void*)&ival,
                               propcomment, &fstatus );
    if ( rerr ) {
        fits_report_error( stdout, fstatus );
        sph_error_raise( SPH_FITS_GENERAL, __FILE__, __func__, __LINE__,
                         SPH_ERROR_ERROR,
                         "Could not append image"
                         "A FITS I/O error with CFITSIO code %d occured. ",
                         fstatus
                        );
        return SPH_FITS_GENERAL;
    }

    naxes[1] = ival;
    rerr = fits_read_key( fitshandle, TINT, "NAXIS3", (void*)&ival,
                               propcomment, &fstatus );
        if ( rerr ) {
        fits_report_error( stdout, fstatus );
        sph_error_raise( SPH_FITS_ILLEGAL_PLANE, __FILE__, __func__, __LINE__,
                         SPH_ERROR_ERROR,
                         "Could not append image"
                         "The FTIS image extension %d in %s has no third axis. ",
                         extnum, filename
                        );
        return SPH_FITS_ILLEGAL_PLANE;
    }

    ival = ival + 1;
    naxes[2] = ival;

    if ( cpl_image_get_type(image) == CPL_TYPE_FLOAT ) {
        datatype = TFLOAT;
        bpp = FLOAT_IMG;
    }

    if ( cpl_image_get_type(image) == CPL_TYPE_INT ) {
        datatype = TINT;
        bpp = LONG_IMG;
    }
    if ( cpl_image_get_type(image) == CPL_TYPE_DOUBLE ) {
        datatype = TDOUBLE;
        bpp = DOUBLE_IMG;
    }

    if ( bpp == 0 ) {
        sph_error_raise( SPH_FITS_ILLEGAL_TYPE, __FILE__, __func__, __LINE__,
                         SPH_ERROR_ERROR,
                         "Could not append image"
                         "The type %d provided is not supported. ",
                         cpl_image_get_type(image)
                        );
        return SPH_FITS_ILLEGAL_TYPE;
    }
    if ( bpp != bitpix ) {
        sph_error_raise( SPH_FITS_ILLEGAL_TYPE, __FILE__, __func__, __LINE__,
                         SPH_ERROR_ERROR,
                         "Could not append image"
                         "The existing file has bitpix %d "
                         "but the image provided gets %d. ",
                         bitpix, bpp
                        );
        return SPH_FITS_ILLEGAL_TYPE;
    }

    nelements = cpl_image_get_size_x(image) * cpl_image_get_size_y(image);

    fpixel[2] = ival;

    rerr = fits_resize_img( fitshandle, bpp, 3, naxes, &fstatus);
    rerr = fits_write_pix ( fitshandle, datatype, fpixel,
                            nelements,
                            (void*)cpl_image_get_data(image),
                            &fstatus);

    rerr = fits_close_file( fitshandle, &fstatus );
    if ( rerr ) {
        fits_report_error( stdout, fstatus );
        sph_error_raise( SPH_FITS_GENERAL, __FILE__, __func__, __LINE__,
                         SPH_ERROR_ERROR,
                         "Could not append image"
                         "A FITS I/O error with CFITSIO code %d occured. ",
                         fstatus
                        );
        return SPH_FITS_GENERAL;
    }
    return rerr;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Create a datacube FITS extension.
 @param    filename     The FITS filename
 @param    cpltype      The cpl_type of the cube

 @return   0 if all ok, error code otherwise.

 Append a new image extension to a FITS file, with 3 data axis. The type of the
 new FITS image extension can currently only be CPL_FLOAT.
 If a nonvalid type is passed the function return SPH_FITS_ILLEGAL_TYPE

 See the CPL documentation for more details on the cpl_image.

 */
/*----------------------------------------------------------------------------*/
int sph_fits_create_cube_extension(const char* filename, cpl_type cpltype) {
    int         rerr        = 0;
    int         fstatus     = 0;
    int         maxhdus     = 0;
    int         hdutype     = 0;
    char*       propcomment = NULL;
    int         ival        = 0;
    fitsfile*   fitshandle  = NULL;
    int         bpp         = 0;
    long        naxes[3]    = { 0, 0, 0};

    if (cpl_fits_get_mode() == CPL_FITS_START_CACHING &&
        cpl_fits_set_mode(CPL_FITS_RESTART_CACHING) != CPL_ERROR_NONE) {
        return (int)cpl_error_set_where(cpl_func);
    }

    rerr = ffopen        ( &fitshandle, filename, READWRITE , &fstatus );
    rerr = fits_get_num_hdus( fitshandle, &maxhdus, &fstatus );
    rerr = fits_movabs_hdu( fitshandle, 1, &hdutype, &fstatus );
    rerr = fits_read_key( fitshandle, TINT, "NAXIS1", (void*)&ival,
                               propcomment, &fstatus );
    naxes[0] = ival;
    rerr = fits_read_key( fitshandle, TINT, "NAXIS2", (void*)&ival,
                               propcomment, &fstatus );
    naxes[1] = ival;
    naxes[2] = 0;

    if ( cpltype == CPL_TYPE_FLOAT ) {
        bpp = FLOAT_IMG;
    }
    if ( cpltype == CPL_TYPE_DOUBLE) {
        bpp = DOUBLE_IMG;
    }
    if ( cpltype == CPL_TYPE_INT ) {
        bpp = LONG_IMG;
    }

    if ( bpp == 0 ) {
        sph_error_raise( SPH_FITS_ILLEGAL_TYPE, __FILE__, __func__, __LINE__,
                         SPH_ERROR_ERROR,
                         "Could not create cube"
                         "The type %d provided is not supported. ",
                         cpltype
                        );
        return SPH_FITS_ILLEGAL_TYPE;
    }
    rerr = fits_create_img(fitshandle, bpp, 3, naxes, &fstatus);

    rerr = fits_close_file( fitshandle, &fstatus );

    if ( rerr ) {
        fits_report_error( stdout, fstatus );
        sph_error_raise( SPH_FITS_GENERAL, __FILE__, __func__, __LINE__,
                         SPH_ERROR_ERROR,
                         "Could not create cube"
                         "A FITS I/O error with CFITSIO code %d occured. ",
                         fstatus
                        );
        return SPH_FITS_GENERAL;
    }

    return rerr;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Create a datacube FITS file.
 @param    filename     The FITS filename to create
 @param    nx           The pixel size in x
 @param    ny           The pixel size in y
 @param    nz           The number of planes
 @param    CPL_BPP      The BPP of the cube

 @return   0 if all ok, error code otherwise.

 Create a new FITS file to contain a data cube, with 3 data axis in the HDU.
 If a nonvalid type is passed the function return SPH_FITS_ILLEGAL_TYPE

 */
/*----------------------------------------------------------------------------*/
int sph_fits_create_cube(const char* filenamein, int nx, int ny, int nz, cpl_type_bpp bpp) {
    int         rerr        = 0;
    int         fstatus     = 0;
    fitsfile*   fitshandle  = NULL;
    long        naxes[3]    = { 0, 0, 0};
    char        filename[256];

    sprintf(filename, "!%s",filenamein);
    rerr = fits_create_file( &fitshandle, filename, &fstatus );
    naxes[0] = nx;
    naxes[1] = ny;
    naxes[2] = nz;

    if ( bpp == 0 ) {
        sph_error_raise( SPH_FITS_ILLEGAL_TYPE, __FILE__, __func__, __LINE__,
                         SPH_ERROR_ERROR,
                         "Could not create cube"
                         "The type %d provided is not supported. ",
                         bpp
                        );
        return SPH_FITS_ILLEGAL_TYPE;
    }

    rerr = fits_create_img(fitshandle, bpp, 3, naxes, &fstatus);

    rerr = fits_close_file( fitshandle, &fstatus );

    if ( rerr ) {
        fits_report_error( stdout, fstatus );
        sph_error_raise( SPH_FITS_GENERAL, __FILE__, __func__, __LINE__,
                         SPH_ERROR_ERROR,
                         "Could not create cube"
                         "A FITS I/O error with CFITSIO code %d occured. ",
                         fstatus
                        );
        return SPH_FITS_GENERAL;
    }
    return rerr;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Create a datacube FITS file.
 @param    filename     The FITS filename to create
 @param    nx           The pixel size in x
 @param    ny           The pixel size in y
 @param    cpltype      The cpl_type of the cube

 @return   0 if all ok, error code otherwise.

 Create a new FITS file to contain a data cube, with 3 data axis in the HDU.
 The type of the FITS image HDU can currently only be CPL_FLOAT.
 If a nonvalid type is passed the function return SPH_FITS_ILLEGAL_TYPE

 */
/*----------------------------------------------------------------------------*/
int sph_fits_create_cube_head(const char* filenamein, int nx, int ny,
                              cpl_type cpltype) {
    int         rerr        = 0;
    int         fstatus     = 0;
    fitsfile*   fitshandle  = NULL;
    int         bpp         = 0;
    long        naxes[3]    = { 0, 0, 0};
    char        filename[256];

    sprintf(filename, "!%s",filenamein);
    rerr = fits_create_file( &fitshandle, filename, &fstatus );
    naxes[0] = nx;
    naxes[1] = ny;
    naxes[2] = 0;

    if ( cpltype == CPL_TYPE_FLOAT ) {
        bpp = FLOAT_IMG;
    }

    if ( cpltype == CPL_TYPE_DOUBLE) {
        bpp = DOUBLE_IMG;
    }
    if ( cpltype == CPL_TYPE_INT) {
        bpp = LONG_IMG;
    }

    if ( bpp == 0 ) {
        sph_error_raise( SPH_FITS_ILLEGAL_TYPE, __FILE__, __func__, __LINE__,
                         SPH_ERROR_ERROR,
                         "Could not create cube"
                         "The type %d provided is not supported. ",
                         cpltype
                        );
        return SPH_FITS_ILLEGAL_TYPE;
    }

    rerr = fits_create_img(fitshandle, bpp, 3, naxes, &fstatus);

    rerr = fits_close_file( fitshandle, &fstatus );

    if ( rerr ) {
        fits_report_error( stdout, fstatus );
        sph_error_raise( SPH_FITS_GENERAL, __FILE__, __func__, __LINE__,
                         SPH_ERROR_ERROR,
                         "Could not create cube"
                         "A FITS I/O error with CFITSIO code %d occured. ",
                         fstatus
                        );
        return SPH_FITS_GENERAL;
    }
    return rerr;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Update the checksum in the header of a fits
 @param    filename     The FITS filename

 @return   0 if all ok, error code otherwise.

 Update the checksum in a FITS header.

 */
/*----------------------------------------------------------------------------*/
int sph_fits_update_checksum( const char* filename ) {
    int         rerr        = 0;
    int         fstatus     = 0;
    fitsfile*   fitshandle  = NULL;
    char       md5[32];
    unsigned long   datamd5 = 0;
    unsigned long   hdusum  = 0;

    if (cpl_fits_get_mode() == CPL_FITS_START_CACHING &&
        cpl_fits_set_mode(CPL_FITS_RESTART_CACHING) != CPL_ERROR_NONE) {
        return (int)cpl_error_set_where(cpl_func);
    }

    rerr = ffopen            ( &fitshandle, filename, READWRITE , &fstatus );
    rerr = fits_write_chksum ( fitshandle, &fstatus );
    rerr = fits_get_chksum( fitshandle, &datamd5, &hdusum, &fstatus);
    fits_encode_chksum(hdusum, 1, md5);

    rerr = fits_update_key( fitshandle, TSTRING,
                            "DATAMD5", (void*)md5,
                            "MD5 data signature", &fstatus);
    rerr = fits_close_file( fitshandle, &fstatus );

    if ( rerr ) {
        fits_report_error( stdout, fstatus );
        sph_error_raise( SPH_FITS_GENERAL, __FILE__, __func__, __LINE__,
                         SPH_ERROR_ERROR,
                         "Could not update date"
                         "A FITS I/O error with CFITSIO code %d occured. ",
                         fstatus
                        );
        return SPH_FITS_GENERAL;
    }

    return rerr;
}


/**@}*/
