/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#include "sph_framecombination.h"

#include "sph_error.h"
#include "sph_image_grid.h"
#include "sph_common_keywords.h"
#include "sph_fft.h"
#include "sph_smart_imagelist.h"
#include "sph_keyword_manager.h"
#include "sph_utils.h"
#include "sph_filemanager.h"

#include "irplib_utils.h"

#include <cpl.h>

#include <strings.h>
#include <assert.h>

sph_error_code SPH_FRAMECOMBINATION_GENERAL = SPH_FRAMECOMBINATION_ERR_START + 0;
sph_error_code SPH_FRAMECOMBINATION_DOUBLE_IMAGE_BAD_TYPE = SPH_FRAMECOMBINATION_ERR_START + 1;
sph_error_code SPH_FRAMECOMBINATION_DOUBLE_IMAGE_NO_TYPE = SPH_FRAMECOMBINATION_ERR_START + 2;
sph_error_code SPH_FRAMECOMBINATION_QUAD_IMAGE_BAD_TYPE = SPH_FRAMECOMBINATION_ERR_START + 3;
sph_error_code SPH_FRAMECOMBINATION_QUAD_IMAGE_NO_TYPE = SPH_FRAMECOMBINATION_ERR_START + 4;
sph_error_code SPH_FRAMECOMBINATION_PREPROC_ZPL_EXP_BAD_TYPE = SPH_FRAMECOMBINATION_ERR_START + 5;
sph_error_code SPH_FRAMECOMBINATION_PREPROC_ZPL_EXP_NO_TYPE = SPH_FRAMECOMBINATION_ERR_START + 6;

static
cpl_error_code sph_framecombination_master_frame_collapse_get_params_clean_mean
               (const cpl_parameterlist*, int*, int*) CPL_ATTR_NONNULL;

/*----------------------------------------------------------------------------*/
/**
 * @brief Collapse series of cubes individually
 * @param rawframes     frameset containing the cubes
 * @param collalg       the collapse algorithm
 * @param params        the collapse parameters
 * @return new frameset of master frames
 *
 * This function takes all the frames in rawframes, which should all be
 * cubes, and collapses them
 *
 *
 */
/*----------------------------------------------------------------------------*/
cpl_frameset*
sph_framecombination_collapse_cubes(cpl_frameset* rawframes,
        sph_collapse_algorithm collalg,
        cpl_parameterlist* framecomb_parameterlist,
        const char* recname,
        const char* tagname) {
    int             ff          = 0;
    cpl_frame*      frame       = NULL;
    cpl_frameset*   collapsed_frameset = NULL;
    sph_master_frame*       combined    = NULL;
    cpl_frame*      tmpframe        = NULL;
    cpl_frameset*   subframeset     = NULL;
    cpl_propertylist*    pl = NULL;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    cpl_ensure(rawframes,CPL_ERROR_NULL_INPUT,NULL);
    collapsed_frameset = cpl_frameset_new();
    for (ff = 0; ff < cpl_frameset_get_size(rawframes); ++ff) {
        frame = cpl_frameset_get_position(rawframes,ff);
        pl = sph_keyword_manager_load_properties(cpl_frame_get_filename(frame),0);
        subframeset = cpl_frameset_new();
        cpl_frameset_insert(subframeset,cpl_frame_duplicate(frame));
        tmpframe = sph_filemanager_create_temp_frame(recname,tagname,
                CPL_FRAME_GROUP_CALIB);
        combined = sph_framecombination_master_frame_from_frameset(
                subframeset,collalg,framecomb_parameterlist,0);
        sph_master_frame_save(combined,
                cpl_frame_get_filename(tmpframe),pl);
        SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,"Collaped cube in file %s.",cpl_frame_get_filename(frame));
        sph_master_frame_delete(combined); combined = NULL;
        cpl_frameset_insert(collapsed_frameset,tmpframe);
        cpl_frameset_delete(subframeset); subframeset = NULL;
        cpl_propertylist_delete(pl); pl = NULL;
    }

    return collapsed_frameset;
}

/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_framecombination Module for frame combination algorithms
 * @par Description:
 *
 * The sph_framecombination module provides a set of functions to combine
 * frames in various way using a variety of methods. The routines in this
 * module are used by several recipes in the Sphere reduction pipeline.
 *
 *
 * @par Synopsis:
 * No specific associated data structure.
 */
/*----------------------------------------------------------------------------*/
/**@{*/
static
cpl_vector*
sph_framecombination_get_cplwarp_radial_profile( void ) {
    cpl_vector*    xprofile    = NULL;
    xprofile = cpl_vector_new(CPL_KERNEL_DEF_SAMPLES);

    cpl_vector_fill_kernel_profile( xprofile, CPL_KERNEL_TANH, CPL_KERNEL_DEF_WIDTH );

    return xprofile;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Get the clean mean algorithm parameters from the parlist.
  @param    parlist        input parameterlist that should contain the necessary
                          parameters
  @param    reject_low    pointer to the reject_low variable to fill
  @param    reject_high    pointer to the reject_high variable to fill
  @return   the error code

  @note     A CPL error is raised in case there was a problem.

  Extract the reject_low and reject_high parameters from the parlist. The routine checks
  for all possible errors and raises CPL errors if something goes wrong, e.g. the
  parameters are not in the parlist.
 */
/*----------------------------------------------------------------------------*/
static
cpl_error_code sph_framecombination_master_frame_collapse_get_params_clean_mean
               (const cpl_parameterlist* parlist,
                int* reject_low,
                int* reject_high)
{
    const cpl_parameter*  param =
        cpl_parameterlist_find_const(parlist, "clean_mean.reject_low");

    if (param == NULL) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                     "The clean mean algorithm to be used when "
                                     "collapsing the imagelist needs "
                                     "parameters, missing: %s\n",
                                     "clean_mean.reject_low");
    }

    *reject_low = cpl_parameter_get_int( param );

    if (cpl_error_get_code()) {
        return cpl_error_set_where(cpl_func);
    }


    param = cpl_parameterlist_find_const(parlist, "clean_mean.reject_high");
    if (param == NULL) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                     "The clean mean algorithm to be used when "
                                     "collapsing the imagelist needs "
                                     "parameters, missing: %s\n",
                                     "clean_mean.reject_high");
    }

    *reject_high = cpl_parameter_get_int( param );

    if (cpl_error_get_code()) {
        return cpl_error_set_where(cpl_func);
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    (Re)calculate the master frame by collapsing the given image list
            using the clean mean algorithm.
  @param    self        the master image.
  @param    imagelist   input imagelist used to collapse
  @return   CPL_ERROR_NONE or the relevant CPL error on failure
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code
sph_framecombination_master_frame_collapse_cleanmean(sph_master_frame* self,
                                                     const cpl_imagelist*imlist,
                                                     int reject_low,
                                                     int reject_high)
{
    const cpl_image*   im0        = cpl_imagelist_get_const(imlist, 0);
    const size_t       nx         = cpl_image_get_size_x(im0);
    const size_t       ny         = cpl_image_get_size_y(im0);
    const size_t       nxy        = nx * ny;
    const size_t       nz         = cpl_imagelist_get_size(imlist);
    const double**     pimg;
    const cpl_binary** pbpm;
    cpl_vector*        allval;
    double*            pallval;
    cpl_vector*        okwrap = NULL;
    size_t             okprev = 0;
    const size_t       nreject = reject_high + reject_low;
    double*            pmimg;
    double*            pmrms;
    double*            pmcomb;
    int*               pmbad;

    cpl_ensure_code(self   != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(imlist != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(nz     >  0,    CPL_ERROR_DATA_NOT_FOUND);
    cpl_ensure_code(cpl_image_get_type(im0) == CPL_TYPE_DOUBLE,
                    CPL_ERROR_UNSUPPORTED_MODE);

    if ( nreject > nz - 1 ) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                                     "Rejecting too many images: reject_low "
                                     "+ reject_high = %d + %d = %d > %d - 1",
                                     reject_low, reject_high, (int)nreject,
                                     (int)nz);
    }

    if (self->image == NULL ||
        cpl_image_get_size_x(self->image) != (cpl_size)nx ||
        cpl_image_get_size_y(self->image) != (cpl_size)ny ||
        cpl_image_get_type(self->image) != CPL_TYPE_DOUBLE) {

        cpl_image_delete(self->image);
        self->image = cpl_image_wrap_double(nx, ny,
                                            cpl_malloc(nxy * sizeof(double)));
    }
    if (self->rmsmap == NULL ||
        cpl_image_get_size_x(self->rmsmap) != (cpl_size)nx ||
        cpl_image_get_size_y(self->rmsmap) != (cpl_size)ny ||
        cpl_image_get_type(self->rmsmap) != CPL_TYPE_DOUBLE) {

        cpl_image_delete(self->rmsmap);
        self->rmsmap = cpl_image_wrap_double(nx, ny,
                                            cpl_malloc(nxy * sizeof(double)));
    }
    if (self->ncombmap == NULL ||
        cpl_image_get_size_x(self->ncombmap) != (cpl_size)nx ||
        cpl_image_get_size_y(self->ncombmap) != (cpl_size)ny ||
        cpl_image_get_type(self->ncombmap) != CPL_TYPE_DOUBLE) {

        cpl_image_delete(self->ncombmap);
        self->ncombmap = cpl_image_wrap_double(nx, ny,
                                               cpl_malloc(nxy * sizeof(double)));
    }
    if (self->badpixelmap == NULL ||
        cpl_image_get_size_x(self->badpixelmap) != (cpl_size)nx ||
        cpl_image_get_size_y(self->badpixelmap) != (cpl_size)ny ||
        cpl_image_get_type(self->badpixelmap) != CPL_TYPE_INT) {

        cpl_image_delete(self->badpixelmap);
        self->badpixelmap = cpl_image_wrap_int(nx, ny,
                                               cpl_malloc(nxy * sizeof(int)));
    }
    pmimg  = cpl_image_get_data_double(self->image);
    pmrms  = cpl_image_get_data_double(self->rmsmap);
    pmcomb = cpl_image_get_data_double(self->ncombmap);
    pmbad  = cpl_image_get_data_int(self->badpixelmap);

    allval = cpl_vector_new(nz);
    pallval = cpl_vector_get_data(allval);

    pimg = (const double**)    cpl_malloc(nz * sizeof(*pimg));
    pbpm = (const cpl_binary**)cpl_malloc(nz * sizeof(*pbpm));

    for (size_t k = 0; k < nz; k++) {
        const cpl_image* imi = cpl_imagelist_get_const(imlist, k);
        const cpl_mask * bpmi = cpl_image_get_bpm_const(imi);

        pimg[k] = cpl_image_get_data_double_const(imi);
        pbpm[k] = bpmi ? cpl_mask_get_data_const(bpmi) : NULL;
    }

    for (size_t i = 0; i < nx * ny; i++) {
        size_t nok = 0;
        size_t ibad, ireject;

        for (size_t k = 0; k < nz; k++) {
            if (pbpm[k] == NULL || !pbpm[k][i]) {
                pallval[nok++] = pimg[k][i];
            }
        }
        ibad = nz - nok;
        /* The first pixel(s) to be rejected are any bad ones */
        ireject = nreject < ibad ? 0 : nreject - ibad;

        if (nok <= ireject) {
            /* All good values were rejected */
            pmimg[i]  = 0.0;
            pmcomb[i] = 0.0;
            pmbad[i]  = CPL_BINARY_1;
            pmrms[i]  = SPH_MASTER_FRAME_BAD_RMS;
        } else {
            double       oksum    = 0.0;
            double       weight   = 0.0;
            /* Half of pre-rejected bad pixels are from lower part */
            const size_t ireject_low  = (size_t)reject_low < ibad / 2 ? 0 :
                reject_low - ibad / 2;
            /* The remainder from upper part */
            const size_t ireject_high =
                (size_t)reject_high < (ibad - (reject_low - ireject_low)) ? 0 :
                (size_t)reject_high - (ibad - (reject_low - ireject_low));
            cpl_boolean isok = CPL_FALSE;


            if (ireject_low > 0 || ireject_high > 0) {
                if (nok != okprev) {
                    (void)cpl_vector_unwrap(okwrap);
                    okwrap = cpl_vector_wrap(nok, pallval);
                    okprev = nok;
                }

                if (ireject_low > 0)
                    irplib_vector_get_kth(okwrap, ireject_low - 1);
                if (ireject_high > 0)
                    irplib_vector_get_kth(okwrap, nok - ireject_high);
            }

            for (size_t j = ireject_low; j < nok - ireject_high; j++) {
                oksum += pallval[j];
                isok = CPL_TRUE;
            }
            assert(isok);

            oksum /= nok - ireject;

            for (size_t j = ireject_low; j < nok - ireject_high; j++) {
                weight += (pallval[j] - oksum) * (pallval[j] - oksum);
            }

            if ( (double)(nok - ireject) > 1.5) {
                /* the correction of 1.5 below (and in the line before)
                   is an attempt to get a good high performance
                   unbiased estimator for sigma. See also:
                   http://en.wikipedia.org/wiki/Unbiased_estimation_of_standard_deviation
                */
                weight /= (double)(nok - ireject) - 1.5;

            } else {
                weight /= (double)(nok - ireject);
            }

            pmimg[i]  = oksum;
            pmcomb[i] = (double)(nok - ireject);
            pmbad[i]  = 0;
            pmrms[i]  = sqrt(weight);
        }
    }

    (void)cpl_vector_unwrap(okwrap);
    cpl_free(pimg);
    cpl_free(pbpm);
    cpl_vector_delete(allval);

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    (Re)calculate the master frame by collapsing the given image list
            to a median master image.
  @param    self        the master image.
  @param    imagelist   input imagelist used to collapse
  @return   the error code
  @note     A CPL error is raised in case there was a problem.

  Calculate the new sph_master_frame by collapsing the image_list.
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code
sph_framecombination_master_frame_collapse_median(sph_master_frame* self,
                                                  const cpl_imagelist* imlist )
{
    const cpl_image*   im0        = cpl_imagelist_get_const(imlist, 0);
    const size_t       nx         = cpl_image_get_size_x(im0);
    const size_t       ny         = cpl_image_get_size_y(im0);
    const size_t       nxy        = nx * ny;
    const size_t       nz         = cpl_imagelist_get_size(imlist);
    const double**     pimg;
    const cpl_binary** pbpm;
    cpl_vector*        allval;
    double*            pallval;
    cpl_vector*        okwrap = NULL;
    size_t             okprev;
    double*            pmimg;
    double*            pmrms;
    double*            pmcomb;
    int*               pmbad;

    cpl_ensure_code(self   != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(imlist != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(nz     >  0,    CPL_ERROR_DATA_NOT_FOUND);
    cpl_ensure_code(cpl_image_get_type(im0) == CPL_TYPE_DOUBLE,
                    CPL_ERROR_UNSUPPORTED_MODE);

    if (self->image == NULL ||
        cpl_image_get_size_x(self->image) != (cpl_size)nx ||
        cpl_image_get_size_y(self->image) != (cpl_size)ny ||
        cpl_image_get_type(self->image) != CPL_TYPE_DOUBLE) {

        cpl_image_delete(self->image);
        self->image = cpl_image_wrap_double(nx, ny,
                                            cpl_malloc(nxy * sizeof(double)));
    }
    if (self->rmsmap == NULL ||
        cpl_image_get_size_x(self->rmsmap) != (cpl_size)nx ||
        cpl_image_get_size_y(self->rmsmap) != (cpl_size)ny ||
        cpl_image_get_type(self->rmsmap) != CPL_TYPE_DOUBLE) {

        cpl_image_delete(self->rmsmap);
        self->rmsmap = cpl_image_wrap_double(nx, ny,
                                            cpl_malloc(nxy * sizeof(double)));
    }
    if (self->ncombmap == NULL ||
        cpl_image_get_size_x(self->ncombmap) != (cpl_size)nx ||
        cpl_image_get_size_y(self->ncombmap) != (cpl_size)ny ||
        cpl_image_get_type(self->ncombmap) != CPL_TYPE_DOUBLE) {

        cpl_image_delete(self->ncombmap);
        self->ncombmap = cpl_image_wrap_double(nx, ny,
                                               cpl_malloc(nxy * sizeof(double)));
    }
    if (self->badpixelmap == NULL ||
        cpl_image_get_size_x(self->badpixelmap) != (cpl_size)nx ||
        cpl_image_get_size_y(self->badpixelmap) != (cpl_size)ny ||
        cpl_image_get_type(self->badpixelmap) != CPL_TYPE_INT) {

        cpl_image_delete(self->badpixelmap);
        self->badpixelmap = cpl_image_wrap_int(nx, ny,
                                               cpl_malloc(nxy * sizeof(int)));
    }
    pmimg  = cpl_image_get_data_double(self->image);
    pmrms  = cpl_image_get_data_double(self->rmsmap);
    pmcomb = cpl_image_get_data_double(self->ncombmap);
    pmbad  = cpl_image_get_data_int(self->badpixelmap);

    allval = cpl_vector_new(nz);
    pallval = cpl_vector_get_data(allval);
    okwrap = allval;
    okprev = nz;

    pimg = (const double**)    cpl_malloc(nz * sizeof(*pimg));
    pbpm = (const cpl_binary**)cpl_malloc(nz * sizeof(*pbpm));

    for (size_t k = 0; k < nz; k++) {
        const cpl_image* imi = cpl_imagelist_get_const(imlist, k);
        const cpl_mask * bpmi = cpl_image_get_bpm_const(imi);

        pimg[k] = cpl_image_get_data_double_const(imi);
        pbpm[k] = bpmi ? cpl_mask_get_data_const(bpmi) : NULL;
    }

    for (size_t i = 0; i < nx * ny; i++) {
        size_t nok = 0;

        for (size_t k = 0; k < nz; k++) {
            if (pbpm[k] == NULL || !pbpm[k][i]) {
                pallval[nok++] = pimg[k][i];
            }
        }

        if (nok == 0) {
            /* All good values were rejected */
            pmimg[i]  = 0.0;
            pmcomb[i] = 0.0;
            pmbad[i]  = CPL_BINARY_1;
            pmrms[i]  = SPH_MASTER_FRAME_BAD_RMS;
        } else {
            double median;
            double weight = 0.0;

            if (nok != okprev) {
                if (okwrap != allval) (void)cpl_vector_unwrap(okwrap);
                okwrap = nok == nz ? allval : cpl_vector_wrap(nok, pallval);
                okprev = nok;
            }

            median = cpl_vector_get_median(okwrap);

            for (size_t j = 0; j < nok; j++) {
                weight += (pallval[j] - median) * (pallval[j] - median);
            }

            pmimg[i]  = median;
            pmcomb[i] = (double)nok;
            pmbad[i]  = 0;
            pmrms[i]  = sqrt(weight / (double)nok);
        }
    }

    if (okwrap != allval) 
        (void)cpl_vector_unwrap(okwrap);
    cpl_free(pimg);
    cpl_free(pbpm);
    cpl_vector_delete(allval);

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    (Re)calculate the master frame by collapsing the given image list.
  @param    self        the master image.
  @param    imagelist    input imagelist used to collapse
  @return   the error code

  @note     A CPL error is raised in case there was a problem.

  Calaculate the new sph_master_frame by collapsing the image_list. The algorithm used
  is one of the collapse_algorithm enum algorithms.
  The parameters allparlist gives the parameters used for the algorithm. In the case
  of the CLEAN_MEAN algorithm, the parameters "clean_mean.reject_low" and "clean_mean.
  reject_high" need to be provided.
 */
/*----------------------------------------------------------------------------*/
static int sph_framecombination_master_frame_collapse( sph_master_frame* self,
                                                       const cpl_imagelist* imlist,
                                                       sph_collapse_algorithm  collapsealgorithm)
{
    int                     rerr             = cpl_error_get_code();
    int                    reject_low        = 0;
    int                 reject_high        = 0;

    cpl_ensure_code( self, CPL_ERROR_NULL_INPUT );
    cpl_ensure_code( imlist, CPL_ERROR_NULL_INPUT );

    if ( rerr != CPL_ERROR_NONE ) {
        sph_error_raise( SPH_ERROR_PREVIOUS_OP_FAILED, __FILE__,
                         __func__, __LINE__,
                         SPH_ERROR_ERROR,
                         "Master frame"
                         "Could not collapse frame. "
                         "CPL says: %s\n", cpl_error_get_message()
                      );
        return rerr;
    }
    if ( collapsealgorithm == SPH_COLL_ALG_MEAN )
    {
        sph_framecombination_master_frame_collapse_cleanmean ( self, imlist,
                                                    0,
                                                    0
                                                     );
        if ( self->image == NULL ) {
            sph_error_raise( (int)cpl_error_get_code(), __FILE__,
                             __func__, __LINE__,
                             SPH_ERROR_ERROR,
                             "Master frame"
                             "The image could not be collapsed. "
                             );
            return (int)cpl_error_get_code();
        }
    }
    else if ( collapsealgorithm == SPH_COLL_ALG_MEDIAN )
    {
        sph_framecombination_master_frame_collapse_median ( self,
                imlist );
        if ( self->image == NULL ) {
            sph_error_raise( (int)cpl_error_get_code(), __FILE__,
                             __func__, __LINE__,
                             SPH_ERROR_ERROR,
                             "Master frame"
                             "The image could not be collapsed. "
                             );
            return (int)cpl_error_get_code();
        }
    }
    else if ( collapsealgorithm == SPH_COLL_ALG_CLEAN_MEAN )
    {
        rerr = sph_framecombination_master_frame_collapse_get_params_clean_mean( self->alg_parlist,
                                                         &reject_low,
                                                         &reject_high );

        if ( rerr ) {
            return cpl_error_set_where(cpl_func);
        }

        sph_framecombination_master_frame_collapse_cleanmean ( self, imlist,
                                              reject_low,
                                              reject_high);
        if ( self->image == NULL ) {
            return cpl_error_set_where(cpl_func);
        }

    }
    else {
        sph_error_raise( SPH_MASTER_FRAME_NO_ALGORITHM, __FILE__,
                         __func__, __LINE__,
                         SPH_ERROR_ERROR,
                         "Master frame"
                         "The algorithm to be used when "
                         "collapsing the imagelist was "
                         "not set or the one set is invalid !\n");
        return SPH_MASTER_FRAME_NO_ALGORITHM;
    }
    if ( self->qclist )
        cpl_propertylist_delete( self->qclist );
    self->qclist = cpl_propertylist_new();
    rerr = sph_master_frame_quality_check( self );
    if ( rerr ) {
        if ( self-> qclist ) {
            cpl_propertylist_delete( self->qclist );
            self->qclist = NULL;
        }
        if ( self->image ) {
            cpl_image_delete( self->image );
            self->image = NULL;
        }
        return rerr;
    }


    if ( rerr != CPL_ERROR_NONE ) {
        sph_error_raise( (int)cpl_error_get_code(), __FILE__,
                         __func__, __LINE__,
                         SPH_ERROR_ERROR,
                         "Master frame"
                         "Could not collapse frame. "
                         "CPL says: %s\n", cpl_error_get_message()
        );
        if ( self-> qclist ) {
            cpl_propertylist_delete( self->qclist );
            self->qclist = NULL;
        }
        if ( self->image ) {
            cpl_image_delete( self->image );
            self->image = NULL;
        }
        return rerr;
    }
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Calculate the master frame by collapsing the images of the given
            extension from the input frame (cube)
  @param    inframe        input frame (cube)
  @param    extension    input extension to collapse images
  @param    alg            input collapse algorithm
  @param    parlist        input parameter list to collapse images

  @return   master frame

  Calculate the new sph_master_frame by collapsing the images from the given
  extension of the input cpl_frame (cube).
 */
/*----------------------------------------------------------------------------*/
sph_master_frame*
sph_framecombination_new_master_frame_from_cpl_frame_multi_ext( cpl_frame* inframe, int extension,
                                        const sph_collapse_algorithm alg,
                                        cpl_parameterlist* parlist )
{
    int                         set             = 0;
    sph_smart_imagelist*        smart_imlist    = NULL;
    sph_master_frame*           master_frame    = NULL;
    sph_master_frame**          master_frame_l  = NULL;
    cpl_imagelist*              image_list      = NULL;


    smart_imlist = sph_smart_imagelist_create_manyplanes( smart_imlist, inframe, extension );

    master_frame_l = cpl_calloc( smart_imlist->nregions, sizeof(sph_master_frame*) );
    for ( set = 0; set < smart_imlist->nregions; ++set ) {
        image_list = sph_smart_imagelist_get_set_manyplanes( smart_imlist, set );
        master_frame_l[set] = sph_framecombination_master_frame_new_collapse( image_list,
                                                      alg,
                                                      parlist );
        cpl_imagelist_delete( image_list );
        image_list = NULL;
    }

    master_frame = sph_smart_imagelist_reassemble_master_frame( smart_imlist, master_frame_l );

    for ( set = 0; set < smart_imlist->nregions; ++set ) {
        sph_master_frame_delete( master_frame_l[set] );
    }
    sph_smart_imagelist_delete( smart_imlist );
    cpl_free( master_frame_l );

    return master_frame;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Calculate the quad image by collapsing the images (ZPL EXP TYPE)
            of each  extension of the input frame(s) (cube(s))
  @param    inframes       input frames (cubes) of ZPL EXP IMG TYE
  @param    alg            input collapse algorithm
  @param    parlist        input parameter list to collapse images

  @return   quad image

  Create a new sph_double_image by collapsing the images from the given
  cpl_frameset (cubes) of the zpl exposure type. This function uses
  sph_framecombination_master_frame_from_frameset
 */
/*----------------------------------------------------------------------------*/
sph_quad_image*
sph_framecombination_quad_image_from_frameset( cpl_frameset* inframes,
                                        const sph_collapse_algorithm alg,
                                        cpl_parameterlist*    parlist ){
    cpl_frame*                curframe        = NULL;
    cpl_propertylist*        plist            = NULL;
    sph_error_code          rerr            = CPL_ERROR_NONE;
    sph_quad_image*            result            = NULL;
    sph_master_frame*        mf_zero_odd        = NULL;
    sph_master_frame*        mf_zero_even    = NULL;
    sph_master_frame*        mf_pi_odd        = NULL;
    sph_master_frame*        mf_pi_even        = NULL;
    const char*            czType            = NULL;

    // type verification block -- begin
    // this verification of the format can be either omitted (commented) or extended if other formats are going to be used
    sph_keyword_manager_new();

    curframe = cpl_frameset_get_first( inframes );

    while ( curframe ) {
        plist = sph_keyword_manager_load_properties( cpl_frame_get_filename(curframe), 0 );
        if ( plist ) {
            czType = cpl_propertylist_get_string( plist, SPH_COMMON_KEYWORD_SPH_TYPE );
            if ( czType ) {
                if ( strcasecmp( czType, SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_PREPROC_ZPL_EXP ) !=0 ) {
                    rerr = SPH_FRAMECOMBINATION_PREPROC_ZPL_EXP_BAD_TYPE;
                }
            }
            else {
                rerr =  SPH_FRAMECOMBINATION_PREPROC_ZPL_EXP_NO_TYPE;
            }
        }
        cpl_propertylist_delete( plist );
        curframe = cpl_frameset_get_next( inframes );
    }
    cpl_error_reset( );
    if ( rerr != CPL_ERROR_NONE ) {
        sph_error_raise( rerr, __FILE__, __func__, __LINE__, SPH_ERROR_ERROR,
                         "There was a problem with the type of some "
                         "or all of the input frames. "
                         "Make sure they are of %s type and the value of "
                         "the %s keyword is set correspondingly!",
                         SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_PREPROC_ZPL_EXP,
                         sph_keyword_manager_get_keyword( SPH_COMMON_KEYWORD_SPH_TYPE ) );
        return NULL;
    }
    // type verification block -- end

    //zero phase
    //combine zero->iframe (planes from the first extension of zpl exp)
    mf_zero_odd = sph_framecombination_master_frame_from_frameset( inframes, alg, parlist, 0);
    //combine zero->pframe (planes from the second extension of zpl exp)
    mf_zero_even = sph_framecombination_master_frame_from_frameset( inframes, alg, parlist, 1);

    //pi phase
    //combine pi->iframe (planes from the third extension of zpl exp)
    mf_pi_odd = sph_framecombination_master_frame_from_frameset( inframes, alg, parlist, 2);
    //combine pi->pframe (planes from the forth extension of zpl exp)
    mf_pi_even = sph_framecombination_master_frame_from_frameset( inframes, alg, parlist, 3);

    if (   mf_zero_odd && mf_zero_even
        && mf_pi_odd   && mf_pi_even      ) {
        result = sph_quad_image_new_from_master_frames( mf_zero_odd, mf_zero_even, mf_pi_odd, mf_pi_even);
    } else {
        SPH_ERR("Quad image frame combination was failed: null poniter(s) from master frames combination is returned")
    }

    return result;


}




/*----------------------------------------------------------------------------*/
/**
  @brief    Calculate the quad image by collapsing the images (ZPL EXP TYPE)
            of each  extension of the input frames (cubes)
  @param    inframes       input frames (cubes) of ZPL EXP TYE
  @param    alg            input collapse algorithm
  @param    parlist        input parameter list to collapse images

  @return   quad image

  Calculate the new sph_quad_image by collapsing the images from the given
  cpl_frameset (cubes) of the zpl exposure type.
 */
/*----------------------------------------------------------------------------*/
sph_quad_image*
sph_framecombination_quad_image_new_from_cpl_frameset_multi_cubes( cpl_frameset* inframes,
                                        const sph_collapse_algorithm alg,
                                        cpl_parameterlist* parlist ){
    sph_quad_image*             result          = NULL;
    sph_quad_image*             curqimage        = NULL;
    cpl_frame*                  curframe        = NULL;
    cpl_propertylist*           plist           = NULL;
    sph_error_code              rerr               = CPL_ERROR_NONE;
    const char*                 czType            = NULL;
    int                         iframe_cube        = 0;

    sph_master_frame*            mf_zero_odd        = NULL;
    sph_master_frame*            mf_zero_even    = NULL;
    sph_master_frame*            mf_pi_odd        = NULL;
    sph_master_frame*            mf_pi_even        = NULL;

    if ( !inframes  ) {
        SPH_NO_SELF;
        return NULL;
    }

    sph_keyword_manager_new();

    curframe = cpl_frameset_get_first( inframes );

    while ( curframe ) {
        plist = sph_keyword_manager_load_properties( cpl_frame_get_filename(curframe), 0 );
        if ( plist ) {
            czType = cpl_propertylist_get_string( plist, SPH_COMMON_KEYWORD_SPH_TYPE );
            if ( czType ) {
                if ( strcasecmp( czType, SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_PREPROC_ZPL_EXP ) !=0 ) {
                    rerr = SPH_FRAMECOMBINATION_PREPROC_ZPL_EXP_BAD_TYPE;
                }
            }
            else {
                rerr =  SPH_FRAMECOMBINATION_PREPROC_ZPL_EXP_NO_TYPE;
            }
            cpl_propertylist_delete(plist); plist = NULL;
        }
        curframe = cpl_frameset_get_next( inframes );
    }
    cpl_error_reset( );
    if ( rerr != CPL_ERROR_NONE ) {
        sph_error_raise( rerr, __FILE__, __func__, __LINE__, SPH_ERROR_ERROR,
                         "There was a problem with the type of some "
                         "or all of the input frames. "
                         "Make sure they are of %s type and the value of "
                         "the %s keyword is set correspondingly!",
                         SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_PREPROC_ZPL_EXP,
                         sph_keyword_manager_get_keyword( SPH_COMMON_KEYWORD_SPH_TYPE ) );
        return NULL;
    }


    iframe_cube = 0;
    curframe = cpl_frameset_get_first( inframes );
    while ( curframe ) {
        sph_error_raise( SPH_FRAMECOMBINATION_GENERAL,
            __FILE__, __func__, __LINE__,
            SPH_ERROR_INFO,  " filename of the current frame to combine: %s", cpl_frame_get_filename ( curframe ));
        //printf( "filename of the current frame to combine: %s, iframe_cube= %d\n", cpl_frame_get_filename ( curframe ), iframe_cube );

           mf_zero_odd = sph_framecombination_new_master_frame_from_cpl_frame_multi_ext( curframe, 0,  alg,  parlist );
          mf_zero_even = sph_framecombination_new_master_frame_from_cpl_frame_multi_ext( curframe, 1,  alg,  parlist );
         mf_pi_odd = sph_framecombination_new_master_frame_from_cpl_frame_multi_ext( curframe, 2,  alg,  parlist );
          mf_pi_even = sph_framecombination_new_master_frame_from_cpl_frame_multi_ext( curframe, 3,  alg,  parlist );
          iframe_cube++;

          if (iframe_cube == 1) {
              result = sph_quad_image_new_from_master_frames(mf_zero_odd, mf_zero_even, mf_pi_odd, mf_pi_even);
          } else {
             curqimage = sph_quad_image_new_from_master_frames(mf_zero_odd, mf_zero_even, mf_pi_odd, mf_pi_even);
             rerr = sph_quad_image_add_quad_image( result, curqimage);
             if ( rerr != CPL_ERROR_NONE ) {
                SPH_ERR("Could not add current quad image to the result")
             }
             sph_quad_image_delete( curqimage );
         }
        curframe = cpl_frameset_get_next( inframes );
    }
    cpl_ensure(result != NULL,CPL_ERROR_ILLEGAL_OUTPUT,NULL);
    rerr = sph_quad_image_divide_double( result, (double) iframe_cube );
    if ( rerr != CPL_ERROR_NONE ) {
           SPH_ERR("Could not divide quad image by number of frames")
    }

    return result;
}


sph_framecombination*
sph_framecombination_new(void) {
    sph_framecombination*    result = NULL;
    result = cpl_calloc( 1, sizeof(sph_framecombination));
    return result;
}

cpl_image*
sph_framecombination_apply_gimros( cpl_image* im, double degrees, double dx, double dy ) {
    sph_image_grid*        grid0    = NULL;
    sph_image_grid*        grid    = NULL;
    cpl_image*            result    = NULL;
    sph_error_code        rerr    = CPL_ERROR_NONE;
    if ( !im ) return NULL;
    grid0 = sph_image_grid_new_from_image( im, -0.5, 0.5, -0.5, 0.5);
    grid = sph_image_grid_new_from_image( im, -0.5, 0.5, -0.5, 0.5);
    if ( !grid || !grid0 ) return NULL;
    rerr = sph_image_grid_rotate_and_shift( grid, 0.0, 0.0, degrees, dx, dy );
    sph_image_grid_map_to_target( grid, grid0 );
    if ( rerr == CPL_ERROR_NONE ) {
        result = sph_image_grid_extract_image(grid0);
    }
    sph_image_grid_delete(grid); grid = NULL;
    sph_image_grid_delete(grid0); grid0 = NULL;
    return result;
}
cpl_image*
sph_framecombination_apply_cplwarp( sph_framecombination_cplwarp_params* params, cpl_image* im, double degrees, double dx, double dy ) {
    const cpl_size nx = cpl_image_get_size_x(im);
    const cpl_size ny = cpl_image_get_size_y(im);
    cpl_image*            result    = NULL;
    cpl_vector*            xprofile = NULL;
    cpl_vector*            yprofile = NULL;
    double                 xradius    = 0.0;
    double                 yradius    = 0.0;
    int                    xx        = 0;
    int                    yy        = 0;
    double                ddx        = 0.0;
    double                ddy        = 0.0;
    double                 angle    = degrees * CPL_MATH_RAD_DEG;
    sph_error_code        rerr    = CPL_ERROR_NONE;

    if ( !im ) return NULL;
    if ( !params ) return NULL;
    if ( !params->dx || !params->dy) {
        const double cx = nx / 2.0;
        const double cy = ny / 2.0;

        if ( params->dx ) cpl_image_delete( params->dx );
        if ( params->dy ) cpl_image_delete( params->dy );
        params->dx = cpl_image_new( nx, ny, CPL_TYPE_DOUBLE );
        params->dy = cpl_image_new( nx, ny, CPL_TYPE_DOUBLE );
        for ( yy = 0; yy < ny; ++yy ) {
            for ( xx = 0; xx < nx; ++xx ) {
                ddx = cos( angle ) * ( (double)xx - cx ) -
                        sin( angle ) * ( (double)yy - cy );
                ddy = cos( angle ) * ( (double)yy - cy ) +
                            sin( angle ) * ( (double)xx - cx );
                ddx = ( ddx - (double)xx + cx ) + dx;
                ddy = ( ddy - (double)yy + cy ) + dy;
                cpl_image_set( params->dx, xx + 1, yy + 1, ddx );
                cpl_image_set( params->dy, xx + 1, yy + 1, ddy );
            }
        }
    }
    result = cpl_image_new( nx, ny, CPL_TYPE_DOUBLE );

    xprofile = sph_framecombination_get_cplwarp_radial_profile( );
    yprofile = sph_framecombination_get_cplwarp_radial_profile( );
    xradius = CPL_KERNEL_DEF_WIDTH;
    yradius = CPL_KERNEL_DEF_WIDTH;
    rerr = cpl_image_warp( result, im, params->dx, params->dy, xprofile, xradius, yprofile, yradius );

    cpl_vector_delete( xprofile ); xprofile = NULL;
    cpl_vector_delete( yprofile ); yprofile = NULL;
    if ( rerr != CPL_ERROR_NONE ) {
        SPH_RAISE_CPL;
    }
    return result;
}
cpl_image*
sph_framecombination_apply_gslfft( cpl_image* imin, double degrees, double dx, double dy, double top_hat_radius ) {
    cpl_image*            result    = NULL;
    sph_fft*            sphfft    = NULL;
    sph_error_code        rerr    = CPL_ERROR_NONE;
    cpl_image*            im        = cpl_image_duplicate( imin );

    SPH_RAISE_CPL_RESET;

    sphfft = sph_fft_new( SPH_FFT_GSL_RADIX2 );
    if ( degrees > 270.0 ) {
        cpl_image_turn( im, -3 );
        degrees = degrees - 270.0;
    }
    if ( degrees > 180.0 ) {
        cpl_image_turn( im, -2 );
        degrees = degrees - 180.0;
    }
    if ( degrees > 90.0 ) {
        cpl_image_turn( im, -1 );
        degrees = degrees - 90.0;
    }
    if ( degrees < -270.0 ) {
        cpl_image_turn( im, 3 );
        degrees = degrees + 270.0;
    }
    if ( degrees < -180.0 ) {
        cpl_image_turn( im, 2 );
        degrees = degrees + 180.0;
    }
    if ( degrees < -90.0 ) {
        cpl_image_turn( im, 1 );
        degrees = degrees + 90.0;
    }
    sphfft->top_hat_radius = top_hat_radius;
    rerr = sph_fft_rotate_prepare( sphfft, im, -degrees, dx, dy );
    if ( rerr != CPL_ERROR_NONE ) {
        sph_fft_delete(sphfft);
        return NULL;
    }
    rerr = sph_fft_rotate(sphfft);
    if ( rerr != CPL_ERROR_NONE ) {
        sph_fft_delete(sphfft);
        return NULL;
    }
    result = sph_fft_rotate_complex2image(sphfft);
    sph_fft_delete(sphfft);
    cpl_image_delete( im ); im = NULL;
    SPH_RAISE_CPL_RESET;
    return result;
}

cpl_image*
sph_framecombination_rotate_image( sph_framecombination* self, cpl_image* im, double degrees) {
    cpl_image*    result        = NULL;
    if ( !self ) return NULL;
    if ( self->method == SPH_FRAMECOMBINATION_METHOD_GIMROS ) {
        result = sph_framecombination_apply_gimros( im, degrees, 0.0, 0.0 );
    }
    if ( self->method == SPH_FRAMECOMBINATION_METHOD_CPLWARP ) {
      result = sph_framecombination_apply_cplwarp( (sph_framecombination_cplwarp_params*)self->pmethod,
                im,
                degrees, 0.0, 0.0 );
    }
    if ( self->method == SPH_FRAMECOMBINATION_METHOD_GSLFFT ) {
        result = sph_framecombination_apply_gslfft( im, degrees, 0.0, 0.0, self->tr_radius );
    }
    return result;
}
sph_ldt*
sph_framecombination_rotate_ldt( sph_framecombination* self, sph_ldt* ldtin, double degrees) {
    sph_ldt*    result        = NULL;
    if ( !self ) return NULL;
    if ( self->method == SPH_FRAMECOMBINATION_METHOD_GIMROS ) {
        result = sph_ldt_new_rotate_gimros( ldtin, -degrees, 0.1 );
    }
    else {
        sph_error_raise( SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Unsupported rotation method for LDT."
                        "Am going for tea instead (and wont rotate)." );
    }
    if ( cpl_error_get_code() != CPL_ERROR_NONE ) {
      SPH_RAISE_CPL
    }
    return result;
}
cpl_image*
sph_framecombination_shift_image( sph_framecombination* self, cpl_image* im, double dx, double dy) {
    cpl_image*    result        = NULL;
    if ( !self ) return NULL;
    if ( self->method == SPH_FRAMECOMBINATION_METHOD_GIMROS ) {
        result = sph_framecombination_apply_gimros( im, 0.0, dx, dy );
    }
    if ( self->method == SPH_FRAMECOMBINATION_METHOD_CPLWARP ) {
        result = sph_framecombination_apply_cplwarp( (sph_framecombination_cplwarp_params*)self->pmethod,
                im,
                0.0, dx, dy );
    }
    if ( self->method == SPH_FRAMECOMBINATION_METHOD_GSLFFT ) {
        result = sph_framecombination_apply_gslfft(
                im, 0.0, dx, dy, self->tr_radius );
    }
    return result;
}

void sph_framecombination_set_method_gimros( sph_framecombination* self ) {
    if ( !self ) return;
    self->method = SPH_FRAMECOMBINATION_METHOD_GIMROS;
    self->pmethod = NULL;
}
void sph_framecombination_set_method_cplwarp( sph_framecombination* self, cpl_kernel kernel ) {
    sph_framecombination_cplwarp_params* params = NULL;
    if ( !self ) return;

    self->method = SPH_FRAMECOMBINATION_METHOD_CPLWARP;
    self->pmethod = cpl_calloc( 1, sizeof(sph_framecombination_cplwarp_params));
    params = (sph_framecombination_cplwarp_params*)self->pmethod;
    params->kernel = kernel;
}

void sph_framecombination_set_method_gslfft( sph_framecombination* self, double filter_radius ) {
    if ( !self ) return;

    self->method = SPH_FRAMECOMBINATION_METHOD_GSLFFT;
    self->pmethod = NULL; /* FIXME: Had zero-size via empty/unused struct... */
    self->tr_radius = filter_radius;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Combines frames, overwriting files
 * @param self
 * @param inframes the input list of frames
 *
 * @return a new master frame with the combined and collapsed frame or NULL
 *
 * This function performs framecombination including rotation and shifting.
 *
 */
/*----------------------------------------------------------------------------*/
sph_master_frame*
sph_framecombination_combine_master_frames_overwrite( sph_framecombination* self,
                cpl_frameset* frames, sph_collapse_algorithm collalg, cpl_parameterlist* colparams ) {
    sph_master_frame*    result        = NULL;
    sph_master_frame*    current        = NULL;
    sph_master_frame*    current_res    = NULL;
    cpl_image*            dumbad        = NULL;
    cpl_frame*            frame        = NULL;
    double                dx            = 0.0;
    double                 dy            = 0.0;
    double                 angle        = 0.0;
    double                angle0        = -500.0;
    int                    ii            = 0;
    char                dummyfilename[256];

    if ( !self || !frames ) {
        return NULL;
    }
    for (ii = 0; ii < cpl_frameset_get_size( frames ); ++ii) {
        frame = cpl_frameset_get_position( frames, ii );
        current = sph_master_frame_load_(frame , 0 );
        if ( current ) {
            dx = 0.0; dy = 0.0;angle = 0.0;
            current_res = sph_master_frame_new_empty();
            if (  !self->ignoredither && cpl_propertylist_has( current->properties, SPH_COMMON_KEYWORD_SPH_DITHERX ) ) {
                dx = -cpl_propertylist_get_double( current->properties, SPH_COMMON_KEYWORD_SPH_DITHERX );
            }
            if (  !self->ignoredither && cpl_propertylist_has( current->properties, SPH_COMMON_KEYWORD_SPH_DITHERY ) ) {
                dy = -cpl_propertylist_get_double( current->properties, SPH_COMMON_KEYWORD_SPH_DITHERY );
            }
            if ( dx != 0 || dy != 0  )
            {
                current_res->image = sph_framecombination_shift_image( self, current->image, dx, dy );
                dumbad = cpl_image_cast( current->badpixelmap, CPL_TYPE_DOUBLE );
                current_res->badpixelmap = sph_framecombination_shift_image( self, dumbad, dx, dy );
                cpl_image_delete( dumbad );
                cpl_image_add_scalar( current_res->badpixelmap,1.0);
                cpl_image_threshold(current_res->badpixelmap,0.0,100.0,0.0,100.0);
                dumbad = cpl_image_cast( current_res->badpixelmap, CPL_TYPE_INT );
                cpl_image_delete( current_res->badpixelmap );
                current_res->badpixelmap = dumbad;
                current_res->ncombmap = sph_framecombination_shift_image( self, current->ncombmap, dx, dy );
                current_res->rmsmap = sph_framecombination_shift_image( self, current->rmsmap, dx, dy );
                cpl_propertylist_delete(current_res->properties);
                current_res->properties = cpl_propertylist_duplicate( current->properties );
                sph_master_frame_delete( current );
                current = current_res;
                current_res = sph_master_frame_new_empty();
                SPH_RAISE_CPL_RESET;
            }
            sprintf( dummyfilename, "%s.old", cpl_frame_get_filename( frame) );
            sph_master_frame_save( current, dummyfilename, NULL );
            SPH_RAISE_CPL_RESET;
            if ( self->subtract_mframe ) {
                sph_master_frame_subtract_master_frame( current, self->subtract_mframe );
            }
            if ( !self->ignoreang && cpl_propertylist_has( current->properties, SPH_COMMON_KEYWORD_SPH_POSANG ) ) {
                angle = cpl_propertylist_get_double( current->properties, SPH_COMMON_KEYWORD_SPH_POSANG );
                if ( angle0 < -400.0 ) {
                    angle0 = angle;
                }
            }
            if ( !self->ignoreang && fabs(angle0 - angle) > 0.001 ) {
                current_res->image = sph_framecombination_rotate_image( self, current->image, angle0-angle );
                dumbad = cpl_image_cast( current->badpixelmap, CPL_TYPE_DOUBLE );
                current_res->badpixelmap = sph_framecombination_rotate_image( self, dumbad, angle0-angle );
                cpl_image_delete( dumbad );
                cpl_image_threshold(current_res->badpixelmap,0.0,100.0,0.0,100.0);
                dumbad = cpl_image_cast( current_res->badpixelmap, CPL_TYPE_INT );
                cpl_image_delete( current_res->badpixelmap );
                current_res->badpixelmap = dumbad;
                current_res->ncombmap = sph_framecombination_rotate_image( self, current->ncombmap, angle0-angle );
                current_res->rmsmap = sph_framecombination_rotate_image( self, current->rmsmap, angle0-angle );
                cpl_propertylist_delete(current_res->properties);
                current_res->properties = cpl_propertylist_duplicate( current->properties );
                sph_master_frame_delete( current );
                current = current_res;
                current_res = sph_master_frame_new_empty();
                SPH_RAISE_CPL_RESET;
            }
            sph_master_frame_save( current, cpl_frame_get_filename( frame), NULL );
            SPH_RAISE_CPL_RESET;
            sph_master_frame_delete( current_res ); current_res = NULL;
            sph_master_frame_delete( current ); current = NULL;
        }
    }
    result = sph_framecombination_master_frame_from_master_frameset( frames, collalg, colparams );
    return result;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Combines frames, overwriting files
 * @param self
 * @param inframes the input list of frames
 *
 * @return a new ldt with the combined and collapsed frames or NULL
 *
 * This function performs framecombination including rotation and shifting.
 *
 */
/*----------------------------------------------------------------------------*/
sph_ldt*
sph_framecombination_combine_ldt_overwrite( sph_framecombination* self,
                cpl_frameset* frames ) {
    sph_ldt*    result        = NULL;
    sph_ldt*    current        = NULL;
    sph_ldt*    current_res    = NULL;
    cpl_frame*            frame        = NULL;
    double                 angle        = 0.0;
    double                angle0        = -500.0;
    int                    ii            = 0;
    char                dummyfilename[256];

    if ( !self || !frames ) {
        return NULL;
    }
    for (ii = 0; ii < cpl_frameset_get_size( frames ); ++ii) {
        frame = cpl_frameset_get_position( frames, ii );
        current = sph_ldt_load( cpl_frame_get_filename( frame ) );
        if ( current ) {
            angle = 0.0;
            if ( self->subtract_ldt ) {
                sph_ldt_subtract( current, self->subtract_ldt );
            }
            if ( !self->ignoreang && cpl_propertylist_has( current->properties, SPH_COMMON_KEYWORD_SPH_POSANG ) ) {
              angle = cpl_propertylist_get_double( current->properties, SPH_COMMON_KEYWORD_SPH_POSANG );
              if ( angle0 < -400.0 ) {
                angle0 = angle;
              }
            }
            if ( !self->ignoreang && fabs(angle0 - angle) > 0.001 ) {
              sprintf( dummyfilename, "%s.old", cpl_frame_get_filename( frame) );
              sph_ldt_save( current, dummyfilename, CPL_IO_CREATE );
              current_res = sph_framecombination_rotate_ldt( self, current, angle0-angle );
              sprintf( dummyfilename, "%s.rotated", cpl_frame_get_filename( frame) );
              sph_ldt_save( current_res, dummyfilename, CPL_IO_CREATE );
              sph_ldt_delete( current );
              current = current_res; current_res = NULL;
            }

            sph_ldt_save( current,cpl_frame_get_filename( frame ), CPL_IO_CREATE  );
            if ( result == NULL ) {
              result = sph_ldt_load( cpl_frame_get_filename( frame ) );
            }
            else {
                sph_ldt_add( result, current );
                sprintf( dummyfilename, "%s.added", cpl_frame_get_filename( frame) );
                sph_ldt_save( result, dummyfilename, CPL_IO_CREATE );
            }
            sph_ldt_delete( current ); current = NULL;
        }
    }
    return result;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Constructor function for sph_master_frame from collapsing imagelist.
  @param    imagelist    input imagelist used to collapse
  @param    collapse_alg    algorithm to use for collapsing
  @param    allparlist    the list of parameters controlling the collapsing algorithm
  @return   a pointer to the newly created sph_master_frame or NULL if
              unsuccessfull.
  @note     A CPL error is raised in case there was a problem.

  Construct a new sph_master_frame by collapsing the image_list. The algorithm used
  is one of the collalg enum algorithms. Currently only CLEAN_MEAN is implemented.
  The parameters allparlist gives the parameters used for the algorithm. In the case
  of the CLEAN_MEAN algorithm, the parameters "clean_mean.reject_low" and "clean_mean.
  reject_high" need to be provided.
  Note that the imagelist is not required by the master_frame after this function is
  calles and therefore can be safely deleted using a cpl_delete_imagelist call.

 */
/*----------------------------------------------------------------------------*/
sph_master_frame* sph_framecombination_master_frame_new_collapse(cpl_imagelist* imagelist,
                                            const sph_collapse_algorithm collapse_alg,
                                            const cpl_parameterlist* allgparlist)
{
    sph_master_frame* spmi = NULL;
    int rerr = cpl_error_get_code();

    cpl_ensure( imagelist, CPL_ERROR_NULL_INPUT, NULL );

    if ( rerr != CPL_ERROR_NONE ) {
        return NULL;
    }
    spmi = sph_master_frame_new_empty();
    if ( spmi == NULL )
    {
        sph_error_raise( SPH_ERROR_MEMORY_FAIL, __FILE__,
                         __func__, __LINE__,
                         SPH_ERROR_ERROR, "Could not allocate master frame" );
        return NULL;
    }

    spmi->alg_parlist = allgparlist;
    rerr = sph_framecombination_master_frame_collapse( spmi, imagelist, collapse_alg );
    if ( rerr ) {
        (void)cpl_error_set_where(cpl_func);
        sph_master_frame_delete( spmi );
        spmi = NULL;
    }
    return spmi;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Create master frame from frameset of master frames
 *
 * @param inframes  the input cpl_frameset to combine
 * @param alg       the collapse algorithm to use
 * @param parlist   the input parameterlist for the collapse algorithm
 *
 * @return pointer to the newly created master frame or NULL in case of error.
 *
 * This function creates a new sph_master_frame from a set of sph_master_frames,
 * given as a cpl_frameset. The function uses the sph_smart_imagelist module to combine
 * frames in a way that protects from memory overload problems by dividing the
 * frames of the input frameset into smaller sub-frames before collapsing these
 * and subsequently reassembling them into a single larger sph_master_frame.
 *
 * The collapse algorithm and parameters are passed directly onwards to the
 * sph_master_frame__new_collapse algorithm -- see that function for a description
 * of these parameters.
 *
 */
/*----------------------------------------------------------------------------*/
sph_master_frame*
sph_framecombination_master_frame_from_master_frameset( cpl_frameset* inframes,
                                        const sph_collapse_algorithm alg,
                                        cpl_parameterlist* parlist )
{
    sph_master_frame*           master_frame1    = NULL;
    sph_master_frame*           master_frame2    = NULL;
    sph_master_frame*           master_frame3    = NULL;
    sph_master_frame*           master_frame4    = NULL;

    master_frame1 = sph_framecombination_master_frame_from_frameset(inframes,alg,parlist,0);

    if ( master_frame1 ) {
        master_frame2 = sph_framecombination_master_frame_from_frameset(inframes,alg,parlist,1);
        master_frame3 = sph_framecombination_master_frame_from_frameset(inframes,alg,parlist,2);
        master_frame4 = sph_framecombination_master_frame_from_frameset(inframes,alg,parlist,3);
        if ( master_frame2 && master_frame3 && master_frame4 ) {
            cpl_image_delete(master_frame1->badpixelmap); master_frame1->badpixelmap = NULL;
            cpl_image_delete(master_frame1->ncombmap); master_frame1->ncombmap = NULL;
            cpl_image_delete(master_frame1->rmsmap); master_frame1->rmsmap = NULL;
            master_frame1->badpixelmap = cpl_image_cast(master_frame2->image,CPL_TYPE_INT);
            master_frame1->ncombmap = cpl_image_duplicate(master_frame3->image);
            master_frame1->rmsmap = cpl_image_duplicate(master_frame4->image);
        }
        sph_master_frame_delete(master_frame2);master_frame2=NULL;
        sph_master_frame_delete(master_frame3);master_frame3=NULL;
        sph_master_frame_delete(master_frame4);master_frame4=NULL;
    }
    return master_frame1;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create master frame from cpl_frameset
 *
 * @param inframes  the input cpl_frameset to combine
 * @param alg       the collapse algorithm to use
 * @param parlist   the input parameterlist for the collapse algorithm
 *
 * @return pointer to the newly created master frame or NULL in case of error.
 *
 * This function creates a new sph_master_frame from a set of frames, given as
 * a cpl_frameset. The function uses the sph_smart_imagelist module to combine
 * frames in a way that protects from memory overload problems by dividing the
 * frames of the input frameset into smaller sub-frames before collapsing these
 * and subsequently reassembling them into a single larger sph_master_frame.
 *
 * The collapse algorithm and parameters are passed directly onwards to the
 * sph_master_frame__new_collapse algorithm -- see that function for a description
 * of these parameters.
 *
 */
/*----------------------------------------------------------------------------*/
sph_master_frame*
sph_framecombination_master_frame_from_frameset(const cpl_frameset* inframes,
                                                const sph_collapse_algorithm alg,
                                                const cpl_parameterlist* parlist,
                                                int extension )
{
    sph_master_frame*    master_frame = NULL;
    sph_smart_imagelist* smart_imlist =
        sph_smart_imagelist_create(inframes, extension);

    if (smart_imlist == NULL) {
        (void)cpl_error_set_where(cpl_func);
    } else {
        int                set;
        sph_master_frame** master_frame_l =
            cpl_calloc( smart_imlist->nregions,
                        sizeof(sph_master_frame*) );

        smart_imlist->extension = extension;

        for (set = 0; set < smart_imlist->nregions; ++set) {
            cpl_imagelist* image_list =
                sph_smart_imagelist_get_set(smart_imlist, set);

            master_frame_l[set] =
                sph_framecombination_master_frame_new_collapse(image_list,
                                                               alg,
                                                               parlist);
            cpl_imagelist_delete(image_list);
            if (master_frame_l[set] == NULL) break;
        }

        if (set == smart_imlist->nregions) {
            master_frame =
                sph_smart_imagelist_reassemble_master_frame(smart_imlist,
                                                            master_frame_l);
        }

        for (; set-- > 0;) {
            sph_master_frame_delete( master_frame_l[set] );
        }

        if (master_frame) {
            sph_smart_imagelist_get_raw_median_properties
                (smart_imlist, master_frame->properties);
        }

        sph_smart_imagelist_delete(smart_imlist);
        cpl_free(master_frame_l);
    }

    return master_frame;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create master frame from cpl_frameset
 *
 * @param inframes  the input cpl_frameset to combine
 * @param alg       the collapse algorithm to use
 * @param parlist   the input parameterlist for the collapse algorithm
 *
 * @return pointer to the newly created master frame or NULL in case of error.
 *
 * This function creates a new sph_master_frame from a set of frames, given as
 * a cpl_frameset. The function uses the sph_smart_imagelist module to combine
 * frames in a way that protects from memory overload problems by dividing the
 * frames of the input frameset into smaller sub-frames before collapsing these
 * and subsequently reassembling them into a single larger sph_master_frame.
 *
 * The collapse algorithm and parameters are passed directly onwards to the
 * sph_master_frame__new_collapse algorithm -- see that function for a description
 * of these parameters.
 *
 */
/*----------------------------------------------------------------------------*/
sph_master_frame*
sph_framecombination_master_frame_from_cpl_frameset(const cpl_frameset* inframes,
                                                    const sph_collapse_algorithm alg,
                                                    const cpl_parameterlist* parlist)
{
    sph_master_frame* master_frame =
        sph_framecombination_master_frame_from_frameset(inframes, alg, parlist,
                                                        0);

    if (master_frame == NULL) (void)cpl_error_set_where(cpl_func);

    return master_frame;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief Create weighted mean master frame from the master frames
 *
 * @param inframes  the input cpl_frameset with master frames to combine
 *
 * @return pointer to the newly created master frame or NULL in case of error.
 *
 * This function creates a new sph_master_frame from a set of master frames,
 * given as a cpl_frameset.
 *
 */
/*----------------------------------------------------------------------------*/
sph_master_frame*
sph_framecombination_master_frame_new_mean_weight_from_cpl_framest( const cpl_frameset* inframes )
{
    sph_master_frame*        result         = NULL;
    sph_master_frame*        mframe        = NULL;
    sph_master_frame*        mframe_copy = NULL;
    const cpl_frame*         curframe;
    int                        nframes        = 0;

    if ( !inframes ) {
        SPH_ERR("No input frames (inframes is a null pointer)")
        return NULL;
    }

    // verification whether frames are master frames can be done later

    curframe = cpl_frameset_get_first_const( inframes );
    while ( curframe ){
        sph_error_raise( SPH_FRAMECOMBINATION_GENERAL,
            __FILE__, __func__, __LINE__,
            SPH_ERROR_INFO,  " filename of the current master frame to combine: %s", cpl_frame_get_filename ( curframe ));
        //printf( "filename of the current master frame to combine: %s, cpl_frame_get_filename ( curframe ) );

        mframe = sph_master_frame_load_(curframe , 0 );
        if ( !mframe ){
            sph_error_raise( SPH_FRAMECOMBINATION_GENERAL,
                __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR,  "load master frame to combine is failed: %s", cpl_frame_get_filename ( curframe ));
            return NULL;
        }
        nframes++;
        mframe_copy = sph_master_frame_duplicate( mframe );
        cpl_image_multiply( mframe_copy->rmsmap, mframe_copy->rmsmap );
        cpl_image_divide( mframe_copy->image, mframe_copy->rmsmap );

        if ( nframes == 1){
            //result = sph_master_frame_duplicate( mframe_copy );
            result = sph_master_frame_new( cpl_image_get_size_x( mframe->image),
                    cpl_image_get_size_y( mframe->image));
            if ( !result ){
                SPH_ERR("could not create master frame")
                return NULL;
            }
        }
        cpl_image_add( result->image, mframe_copy->image );
        cpl_image_add( result->badpixelmap, mframe_copy->badpixelmap);
        cpl_image_add( result->ncombmap, mframe_copy->ncombmap);
        sph_utils_cpl_image_fill_double( mframe->rmsmap, 1.0);
        cpl_image_divide( mframe->rmsmap, mframe_copy->rmsmap );
        cpl_image_add( result->rmsmap, mframe->rmsmap );

        //delete master frames
        sph_master_frame_delete( mframe );
        sph_master_frame_delete( mframe_copy );

        curframe = cpl_frameset_get_next_const( inframes );
    }
    cpl_image_divide( result->image, result->rmsmap );
    cpl_image_threshold( result->badpixelmap, 0, 1, 0, 1);
    cpl_image_power( result->rmsmap, 0.5);


    if (cpl_error_get_code() != CPL_ERROR_NONE ){
        SPH_RAISE_CPL;
        return NULL;
    }

    return result;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Calculate the double image by collapsing the images (ZPL EXP IMG TYPE)
            of each  extension of the input frame(s) (cube(s))
  @param    inframes       input frames (cubes) of ZPL EXP IMG TYE
  @param    alg            input collapse algorithm
  @param    parlist        input parameter list to collapse images

  @return   double image

  Create a new sph_double_image by collapsing the images from the given
  cpl_frameset (cubes) of the zpl exposure type. This function uses
  sph_framecombination_master_frame_from_frameset
 */
/*----------------------------------------------------------------------------*/



sph_double_image*
sph_framecombination_double_image_from_frameset( cpl_frameset* inframes,
                                        const sph_collapse_algorithm alg,
                                        cpl_parameterlist*    parlist ){
    sph_double_image*        result        = NULL;
    sph_master_frame*        mf_iframe    = NULL;
    sph_master_frame*        mf_pframe    = NULL;
    sph_error_code          rerr        = CPL_ERROR_NONE;
    const cpl_frame*        curframe    = NULL;
    cpl_propertylist*       plist       = NULL;
    const char*            czType      = NULL;



    if ( !inframes ) {
        SPH_NO_SELF;
        return NULL;
    }



    // type verification block -- begin
    // this verification of the format can be either omitted (commented) or extended if other formats are going to be used
    sph_keyword_manager_new();

    curframe = cpl_frameset_get_first_const( inframes );

    while ( curframe ) {
        plist = sph_keyword_manager_load_properties( cpl_frame_get_filename(curframe), 0 );
        if ( plist ) {
            czType = cpl_propertylist_get_string( plist, SPH_COMMON_KEYWORD_SPH_TYPE );
            if ( czType ) {
                if ( strcasecmp( czType, SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_PREPROC_ZPL_EXP_IMG ) !=0 ) {
                    rerr = SPH_FRAMECOMBINATION_PREPROC_ZPL_EXP_BAD_TYPE;
                }
            }
            else {
                rerr =  SPH_FRAMECOMBINATION_PREPROC_ZPL_EXP_NO_TYPE;
            }
        }
        cpl_propertylist_delete( plist );
        curframe = cpl_frameset_get_next_const( inframes );
    }
    cpl_error_reset( );
    if ( rerr != CPL_ERROR_NONE ) {
        sph_error_raise( rerr, __FILE__, __func__, __LINE__, SPH_ERROR_ERROR,
                         "There was a problem with the type of some "
                         "or all of the input frames. "
                         "Make sure they are of %s type and the value of "
                         "the %s keyword is set correspondingly!",
                         SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_PREPROC_ZPL_EXP_IMG,
                         sph_keyword_manager_get_keyword( SPH_COMMON_KEYWORD_SPH_TYPE ) );
        return NULL;
    }
    // type verification block -- end

    // combine iframe (planes from the first extention)
    mf_iframe = sph_framecombination_master_frame_from_frameset( inframes, alg, parlist, 0);
    // combine iframe (planes from the second extention)
    mf_pframe = sph_framecombination_master_frame_from_frameset( inframes, alg, parlist, 1);

    if ( mf_iframe && mf_pframe ){
        result = sph_double_image_new_from_master_frames( mf_iframe, mf_pframe);
    } else {
        SPH_ERR("Double image frame combination was failed: null poniter(s) from master frames combination is returned")
    }

    return result;

}




/*----------------------------------------------------------------------------*/
/**
 * @brief Create a double image from cpl_frameset
 *
 * @param self  the input cpl_frameset to combine
 * @param alg       the collapse algorithm to use
 * @param parlist   the input parameterlist for the collapse algorithm
 *
 * @return pointer to the newly created double_image or NULL in case of error.
 *
 * This function creates a new sph_double_image from a set of frames, given as
 * a cpl_frameset. The function uses the sph_smart_imagelist module to combine
 * frames in a way that protects from memory overload problems by dividing the
 * frames of the input frameset into smaller sub-frames before collapsing these
 * and subsequently reassembling them into a single larger sph_double_image.
 *
 * The collapse algorithm and parameters are passed directly onwards to the
 * sph_master_frame_new_collapse algorithm -- see that function for a description
 * of these parameters.
 *
 */
/*----------------------------------------------------------------------------*/
sph_double_image*
sph_framecombination_double_image_new_from_cpl_frameset( cpl_frameset* self,
                                        const sph_collapse_algorithm alg,
                                        const cpl_parameterlist* parlist )
{
    sph_double_image*           result      = NULL;
    sph_smart_imagelist*        smart_imlist = NULL;
    const cpl_frame*            curframe    = NULL;
    sph_error_code              rerr        = CPL_ERROR_NONE;
    const char*                       czType      = NULL;
    int                         set         = 0;
    cpl_imagelist*              image_list  = NULL;
    sph_master_frame**          master_frame_l  = NULL;

    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);

    sph_keyword_manager_new();

    result = sph_double_image_new_empty();

    curframe = cpl_frameset_get_first_const( self );

    while ( curframe ) {
        cpl_propertylist* plist =
        sph_keyword_manager_load_properties( cpl_frame_get_filename(curframe), 0);
        if ( plist ) {
            czType = cpl_propertylist_get_string( plist, SPH_COMMON_KEYWORD_SPH_TYPE );
            if ( czType ) {
                if ( strcasecmp( czType, SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_DOUBLE_IMAGE ) !=0 ) {
                    rerr = SPH_FRAMECOMBINATION_DOUBLE_IMAGE_BAD_TYPE;

                }
            }
            else {
                rerr = SPH_FRAMECOMBINATION_DOUBLE_IMAGE_NO_TYPE;
            }
            cpl_propertylist_delete(plist); plist = NULL;
        }
        curframe = cpl_frameset_get_next_const( self );
    }

    cpl_error_reset( );
    if ( rerr != CPL_ERROR_NONE ) {
        sph_error_raise( rerr, __FILE__, __func__, __LINE__, SPH_ERROR_ERROR,
                         "There was a problem with the type of some "
                         "or all of the input frames. "
                         "Make sure they are of %s type and the value of "
                         "the %s keyword is set correspondingly!",
                         SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_DOUBLE_IMAGE,
                         sph_keyword_manager_get_keyword( SPH_COMMON_KEYWORD_SPH_TYPE ) );
        return NULL;
    }

    smart_imlist = sph_smart_imagelist_create( self,0 );

    master_frame_l = cpl_calloc( smart_imlist->nregions, sizeof(sph_master_frame*) );

    smart_imlist->extension = 0;//index of iframe sph_double_image

    for ( set = 0; set < smart_imlist->nregions; ++set ) {
        image_list = sph_smart_imagelist_get_set( smart_imlist, set );
        master_frame_l[set] = sph_framecombination_master_frame_new_collapse( image_list,
                                                      alg,
                                                      parlist );
        cpl_imagelist_delete( image_list );
        image_list = NULL;
    }

    result->iframe = sph_smart_imagelist_reassemble_master_frame( smart_imlist, master_frame_l );

    for ( set = 0; set < smart_imlist->nregions; ++set ) {
        sph_master_frame_delete( master_frame_l[set] );
    }

    smart_imlist->extension = 4;//index of pframe sph_double_image

    for ( set = 0; set < smart_imlist->nregions; ++set ) {
        image_list = sph_smart_imagelist_get_set( smart_imlist, set );
        master_frame_l[set] = sph_framecombination_master_frame_new_collapse( image_list,
                                                      alg,
                                                      parlist );
        cpl_imagelist_delete( image_list );
        image_list = NULL;
    }

    result->pframe = sph_smart_imagelist_reassemble_master_frame( smart_imlist, master_frame_l );

    for ( set = 0; set < smart_imlist->nregions; ++set ) {
        sph_master_frame_delete( master_frame_l[set] );
    }
    cpl_free(master_frame_l); master_frame_l = NULL;
    sph_smart_imagelist_delete(smart_imlist); smart_imlist = NULL;
    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create a combined triple image from cpl_frameset of the triple images
 *
 * @param self  the input cpl_frameset (of TRIPLE_IMAGE_TYPE) to combine
 * @param alg       the collapse algorithm to use
 * @param parlist   the input parameterlist for the collapse algorithm
 *
 * @return pointer to the newly created quad image or NULL in case of error.
 *
 * This function creates a new combined sph_double_image from a set of frames
 * (given as a cpl_frameset) of the TRIPLE_IMAGE_TYPE. The existing rms images
 * in the triple image are ignored (not used to combine images), new ones are
 * calculated direct from a collapse algorithm.
 * The collapse algorithm and parameters are passed directly onwards to the
 * sph_framecombination_master_frame_from_frameset and to the
 * sph_master_frame_new_collapse algorithm -- see these functions for a description
 * of these parameters.
 *
 */
/*----------------------------------------------------------------------------*/
sph_triple_image*
sph_framecombination_triple_image_new_from_cpl_frameset( cpl_frameset* self,
                                                         const sph_collapse_algorithm alg,
                                                         const cpl_parameterlist* parlist)
{
    sph_triple_image*        result        = NULL;
    sph_master_frame*        mframe1       = NULL;
    sph_master_frame*        mframe2       = NULL;
    sph_master_frame*        mframe3       = NULL;
    //sph_master_frame*        mframe1_badpx = NULL;
    //sph_master_frame*        mframe2_badpx = NULL;
    //sph_master_frame*        mframe3_badpx = NULL;

    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);

    mframe1 = sph_framecombination_master_frame_from_frameset( self, alg, parlist, 0); // 0-ext: image of the first master frame
    //mframe1_badpx = sph_framecombination_master_frame_from_frameset( self, alg, parlist, 1); // 1-ext: badpixelmap of the first master frame
    mframe2 = sph_framecombination_master_frame_from_frameset( self, alg, parlist, 4); // 4-ext: image of the second master frame
    //mframe2_badpx = sph_framecombination_master_frame_from_frameset( self, alg, parlist, 5); // 5-ext: badpixelmap of the second master frame
    mframe3 = sph_framecombination_master_frame_from_frameset( self, alg, parlist, 8); // 8-ext: image of the third master frame
    //mframe3_badpx = sph_framecombination_master_frame_from_frameset( self, alg, parlist, 9); // 9-ext: badpixelmap of the second master frame

    result = sph_triple_image_new_from_master_frames( mframe1, mframe2, mframe3 );

    return result;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief Create a quad image from cpl_frameset of quad image TYPE (combining)
 *
 * @param self     The input cpl_frameset (of QUAD_IMAGE_TYPE) to combine
 * @param alg      The collapse algorithm to use
 * @param parlist  The input parameterlist for the collapse algorithm
 *
 * @return pointer to the newly created quad image or NULL in case of error.
 *
 * This function creates a new sph_quad_image from a set of frames, given as
 * a cpl_frameset. The function uses the sph_smart_imagelist module to combine
 * frames in a way that protects from memory overload problems by dividing the
 * frames of the input frameset into smaller sub-frames before collapsing these
 * and subsequently reassembling them into a single larger sph_quad_image.
 *
 * The collapse algorithm and parameters are passed directly onwards to the
 * sph_master_frame_new_collapse algorithm -- see that function for a description
 * of these parameters.
 *
 */
/*----------------------------------------------------------------------------*/
sph_quad_image* sph_framecombination_quad_image_new_from_cpl_frameset
(cpl_frameset* self,
 const sph_collapse_algorithm alg,
 const cpl_parameterlist* parlist) {

    sph_quad_image*             result          = NULL;
    sph_smart_imagelist*        smart_imlist    = NULL;
    const cpl_frame*            curframe        = NULL;
    sph_error_code              rerr            = CPL_ERROR_NONE;
    const char*                 czType          = NULL;
    int                         set             = 0;
    cpl_imagelist*              image_list      = NULL;
    sph_master_frame**          master_frame_l  = NULL;

    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);

    sph_keyword_manager_new();

    result = sph_quad_image_new_empty();

    curframe = cpl_frameset_get_first_const( self );

    while ( curframe ) {
        cpl_propertylist* plist =
            sph_keyword_manager_load_properties( cpl_frame_get_filename(curframe), 0 );
        if ( plist ) {
            czType = cpl_propertylist_get_string( plist, SPH_COMMON_KEYWORD_SPH_TYPE );
            if ( czType ) {
                if ( strcasecmp( czType, SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_QUAD_IMAGE ) !=0 ) {
                    rerr = SPH_FRAMECOMBINATION_QUAD_IMAGE_BAD_TYPE;
                }
            }
            else {
                rerr =  SPH_FRAMECOMBINATION_QUAD_IMAGE_NO_TYPE;
            }
        }
        curframe = cpl_frameset_get_next_const( self );
        cpl_propertylist_delete(plist);
    }
    cpl_error_reset( );
    if ( rerr != CPL_ERROR_NONE ) {
        sph_error_raise( rerr, __FILE__, __func__, __LINE__, SPH_ERROR_ERROR,
                         "There was a problem with the type of some "
                         "or all of the input frames. "
                         "Make sure they are of %s type and the value of "
                         "the %s keyword is set correspondingly!",
                         SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_QUAD_IMAGE,
                         sph_keyword_manager_get_keyword( SPH_COMMON_KEYWORD_SPH_TYPE ) );
        return NULL;
    }

    smart_imlist = sph_smart_imagelist_create( self,0 );
    cpl_ensure(smart_imlist != NULL,cpl_error_get_code(),NULL);
    master_frame_l = cpl_calloc( smart_imlist->nregions, sizeof(sph_master_frame*) );


    // zero- phase images
    smart_imlist->extension = 0;//index of "zero(phase) image iframe" sph_quad_image


    for ( set = 0; set < smart_imlist->nregions; ++set ) {
        image_list = sph_smart_imagelist_get_set( smart_imlist, set );
        master_frame_l[set] = sph_framecombination_master_frame_new_collapse( image_list,
                                                      alg,
                                                      parlist );
        cpl_imagelist_delete( image_list );
        image_list = NULL;
    }

    result->zero_image->iframe = sph_smart_imagelist_reassemble_master_frame( smart_imlist, master_frame_l );

    for ( set = 0; set < smart_imlist->nregions; ++set ) {
        sph_master_frame_delete( master_frame_l[set] );
    }

    smart_imlist->extension = 4;//index of "zero(phase) image pframe" sph_quad_image

    for ( set = 0; set < smart_imlist->nregions; ++set ) {
        image_list = sph_smart_imagelist_get_set( smart_imlist, set );
        master_frame_l[set] = sph_framecombination_master_frame_new_collapse( image_list,
                                                      alg,
                                                      parlist );
        cpl_imagelist_delete( image_list );
        image_list = NULL;
    }

    result->zero_image->pframe = sph_smart_imagelist_reassemble_master_frame( smart_imlist, master_frame_l );

    for ( set = 0; set < smart_imlist->nregions; ++set ) {
        sph_master_frame_delete( master_frame_l[set] );
    }

    // pi- phase images

    smart_imlist->extension = 8;//index of "pi(phase) image iframe" sph_quad_image

    for ( set = 0; set < smart_imlist->nregions; ++set ) {
        image_list = sph_smart_imagelist_get_set( smart_imlist, set );
        master_frame_l[set] = sph_framecombination_master_frame_new_collapse( image_list,
                                                      alg,
                                                      parlist );
        cpl_imagelist_delete( image_list );
        image_list = NULL;
    }

    result->pi_image->iframe = sph_smart_imagelist_reassemble_master_frame( smart_imlist, master_frame_l );

    for ( set = 0; set < smart_imlist->nregions; ++set ) {
        sph_master_frame_delete( master_frame_l[set] );
    }

    smart_imlist->extension = 12;//index of "pi(phase) image pframe" sph_quad_image

    for ( set = 0; set < smart_imlist->nregions; ++set ) {
        image_list = sph_smart_imagelist_get_set( smart_imlist, set );
        master_frame_l[set] = sph_framecombination_master_frame_new_collapse( image_list,
                                                      alg,
                                                      parlist );
        cpl_imagelist_delete( image_list );
        image_list = NULL;
    }

    result->pi_image->pframe = sph_smart_imagelist_reassemble_master_frame( smart_imlist, master_frame_l );

    for ( set = 0; set < smart_imlist->nregions; ++set ) {
        sph_master_frame_delete( master_frame_l[set] );
    }
    cpl_free(master_frame_l); master_frame_l = NULL;
    sph_smart_imagelist_delete(smart_imlist); smart_imlist = NULL;
    return result;
}


void
sph_framecombination_delete( sph_framecombination* self ) {

    if (self != NULL) {
        if ( self->pmethod) {
            if ( self->method == SPH_FRAMECOMBINATION_METHOD_CPLWARP ) {
                sph_framecombination_cplwarp_params* cplparams
                    = (sph_framecombination_cplwarp_params*)self->pmethod;
                cpl_image_delete(cplparams->dx);
                cpl_image_delete(cplparams->dy);
            }
            cpl_free(self->pmethod);
        }
        cpl_free(self);
    }
}

/**@}*/
