/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>
#include "sph_error.h"
#include "sph_master_frame.h"
#include "sph_framecombination.h"
#include "sph_fits.h"
#include "sph_common_science.h"
#include "sph_filemanager.h"
#include "sph_fits.h"
#include "sph_shared_irdifs.h"
#include "sph_common_keywords.h"
#include <assert.h>

static cpl_frame*
sph_irdifs_master_dark_skip_copy(cpl_frame* frame, int nskip) {
    cpl_frame* result = NULL;
    int pp = 0;
    int nplanes = 0;
    cpl_image* tmpimage = NULL;
    cpl_propertylist* pl = NULL;
    char* tmpstr = NULL;
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;cpl_ensure(frame,
            CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(nskip >= 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    result = cpl_frame_duplicate(frame);
    tmpstr = sph_filemanager_get_tmp_filename("DARK_SKIP");
    cpl_frame_set_filename(result, tmpstr);
    cpl_free(tmpstr);
    tmpstr = NULL;
    tmpimage = cpl_image_load(cpl_frame_get_filename(frame), CPL_TYPE_INT, 0,
            0);
    if (!tmpimage) {
        SPH_ERROR_RAISE_ERR(
                CPL_ERROR_FILE_IO,
                "Could not read plane %d in %s", 0, cpl_frame_get_filename(frame));
        cpl_frame_delete(result);
        return NULL;
    }

    cpl_image_delete(tmpimage);
    tmpimage = NULL;

    sph_filemanager_add_tmp_file(cpl_frame_get_filename(result));

    nplanes = sph_fits_get_nplanes(cpl_frame_get_filename(frame), 0);
    pl = cpl_propertylist_load(cpl_frame_get_filename(frame), 0);
    for (pp = nskip; pp < nplanes; ++pp) {
        tmpimage = cpl_image_load(cpl_frame_get_filename(frame), CPL_TYPE_INT,
                pp, 0);
        if (tmpimage) {
            sph_cube_append_image(cpl_frame_get_filename(result), tmpimage, pl,
                    0);
            cpl_image_delete(tmpimage);
            tmpimage = NULL;
        } else {
            SPH_ERROR_RAISE_ERR(
                    CPL_ERROR_FILE_IO,
                    "Could not read plane %d in %s", pp, cpl_frame_get_filename(frame));
            cpl_frame_delete(result);
            return NULL;
        }
    }
    cpl_propertylist_delete(pl);
    pl = NULL;
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return result;

}
/*----------------------------------------------------------------------------*/
/**
 * @defgroup A SPHERE API Module
 * @par Synopsis:
 * @code
 * typedef _module_ {
 * } module
 * @endcode
 * @par Desciption:
 *
 * This module provides ...
 */
/*----------------------------------------------------------------------------*/
/**@{*/

sph_master_frame*
sph_irdifs_master_dark_create(cpl_frameset* rawframes,
        sph_collapse_algorithm coll_alg, cpl_parameterlist* framecomb,
        double sigma_clip, double smoothing, double min_accept,
        double max_accept, int nskip, cpl_image** static_bpix) {
    sph_master_frame* result = NULL;
    sph_master_frame* result_duplicate = NULL;
    double rms = 0.0;
    /*------------------------------------------------------------------
     -  Sorting the frames into framesets of equal DIT setup
     --------------------------------------------------------------------*/

    if (nskip == 0) {
        result = sph_framecombination_master_frame_from_cpl_frameset(rawframes,
                coll_alg, framecomb);
    } else {
        cpl_frameset* nframeset = cpl_frameset_new();
        for (int ff = 0; ff < cpl_frameset_get_size(rawframes); ++ff) {
            cpl_frame* frame = cpl_frameset_get_position(rawframes, ff);
            if (sph_fits_test_iscube(cpl_frame_get_filename(frame), 0) == 1) {
                const int nplanes =
                    sph_fits_get_nplanes(cpl_frame_get_filename(frame), 0);
                if (nplanes > nskip) {
                    cpl_frame* nframe =
                        sph_irdifs_master_dark_skip_copy(frame, nskip);
                    SPH_ERROR_RAISE_INFO(
                            SPH_ERROR_GENERAL,
                            "Copying cube in file %s, "
                            "skipping the first %d planes "
                            "so only %d planes are copied.", cpl_frame_get_filename(frame), nskip, nplanes-nskip);
                    cpl_frameset_insert(nframeset, nframe);
                }
            }
        }
        result = sph_framecombination_master_frame_from_cpl_frameset(nframeset,
                coll_alg, framecomb);
        cpl_frameset_delete(nframeset);
        nframeset = NULL;
    }

    if (result == NULL) {
        sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Could not create the dark -- "
                        "Please check the previous error messages."
                        "Error code was: %d", cpl_error_get_code());

        return NULL;
    }

    sph_master_frame_mask_tolerance(result, min_accept, max_accept);

    result_duplicate = sph_master_frame_duplicate(result);
    sph_master_frame_quality_check(result);
    if (smoothing > 0) {
        sph_master_frame* smooth_dark = sph_master_frame_duplicate(result);
        sph_master_frame_interpolate_bpix(smooth_dark);
        sph_master_frame_smooth(smooth_dark, smoothing);
        sph_master_frame_subtract_master_frame(result_duplicate, smooth_dark);
        sph_master_frame_delete(smooth_dark);
        smooth_dark = NULL;
    }
    sph_master_frame_quality_check(result_duplicate);
    sph_master_frame_get_mean(result_duplicate, &rms);
    sph_master_frame_mask_sigma(result_duplicate, sigma_clip);
    sph_master_frame_quality_check(result_duplicate);
    sph_master_frame_get_mean(result_duplicate, &rms);

    sph_master_frame_set_bads_from_image(result, result_duplicate->badpixelmap);
    sph_master_frame_delete(result_duplicate);
    result_duplicate = NULL;
    sph_master_frame_quality_check(result);

    if (static_bpix) {
        *static_bpix = sph_master_frame_get_badpixelmap(result);

        if (*static_bpix == NULL) {
            sph_error_raise(
                    SPH_ERROR_ERROR,
                    __FILE__,
                    __func__,
                    __LINE__,
                    SPH_ERROR_ERROR,
                    "Could not find any badpixel map plane in the master dark.");
        }
    }

    return result;
}



/*----------------------------------------------------------------------------*/
/**
 @brief  Compute the median of each pixel's RON (Read-Out-Noise)
 @param  self  The set of frames, only primary HDU read
 @param  nimg  The number of image planes required, e.g. 5
 @param  nlast The last image plane to read, e.g. 10
 @param  llx   Left  x position (1 for leftmost)
 @param  lly   Lower y position (1 for lowest)
 @param  urx   Right x position
 @param  ury   Upper y position
 @return The RON, or negative on error or on too few images
 @note   On error a cpl_error_code is set

 In the normal case the first input frame has a cube with more than nimg
 planes, in which case the nimg planes are taken from that cube. However, if the
 first frame has less than nimg frames the first nimg available planes are used.

 */
/*----------------------------------------------------------------------------*/
double sph_irdifs_compute_ron(const cpl_frameset* self,
                              cpl_size nimg,
                              cpl_size nlast,
                              cpl_size llx,
                              cpl_size lly,
                              cpl_size urx,
                              cpl_size ury)
{
    const cpl_frame* frame    = cpl_frameset_get_first_const(self);
    cpl_image*       img_mean = NULL; /* The running mean */
    cpl_image*       img_varn = NULL; /* The running variance times n */
    cpl_size         iframe   = 0;    /* Counter of frames */
    cpl_size         iimg     = 0;    /* Counter of images */
    double           median   = -1.0; /* Return -1 on insufficient images */

    cpl_ensure(self  != NULL,  CPL_ERROR_NULL_INPUT,         -2.0);
    cpl_ensure(nimg  >= 2,     CPL_ERROR_ILLEGAL_INPUT,      -3.0);
    cpl_ensure(nlast >= nimg,  CPL_ERROR_INCOMPATIBLE_INPUT, -4.0);
    cpl_ensure(llx   >= 1,     CPL_ERROR_ILLEGAL_INPUT,      -5.0);
    cpl_ensure(lly   >= 1,     CPL_ERROR_ILLEGAL_INPUT,      -6.0);
    cpl_ensure(urx   >= llx,   CPL_ERROR_INCOMPATIBLE_INPUT, -7.0);
    cpl_ensure(ury   >= lly,   CPL_ERROR_INCOMPATIBLE_INPUT, -8.0);

    while (frame) {
        const char * file = cpl_frame_get_filename(frame);
        if (file == NULL) {
            frame = NULL;
        } else {
            cpl_propertylist * plist = cpl_propertylist_load_regexp(file, 0,
                                                                    "NAXIS3", 0);
            const cpl_size iplanes = plist == NULL ? 0 : 
                (cpl_propertylist_has(plist, "NAXIS3") ?
                 cpl_propertylist_get_int(plist, "NAXIS3") : 1);
            const cpl_size ifirst = iplanes <= nimg ? 0 : (iplanes - nimg);
            cpl_size       i;

            cpl_propertylist_delete(plist);
            iframe++;

            for (i = ifirst; i < iplanes && iimg < nimg; i++) {
                cpl_image * img =
                    cpl_image_load_window(file, CPL_TYPE_UNSPECIFIED,
                                          i, 0, llx, lly, urx, ury);

                if (img == NULL) {
                    (void)cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                "image=%d/%d", (int)i+1,
                                                (int)iplanes);
                    break;
                } else {
                    iimg++;
                    if (img_mean == NULL) {
                        assert(img_varn == NULL);
                        assert(iimg      == 1);

                        img_mean = cpl_image_duplicate(img); /* mean at n = 1 */

                    } else {
                        cpl_image_subtract(img, img_mean); /* mean deviation */
                        cpl_image_multiply_scalar(img, 1.0/(double)iimg);
                        cpl_image_add(img_mean, img);

                        if (img_varn == NULL) {
                            assert(iimg == 2);
                            /* Variance at n = 2 */
                            img_varn = cpl_image_power_create(img, 2.0);
                            /* Correct for extra division with n, above */
                            cpl_image_multiply_scalar(img_varn, (double)
                                                      iimg*(double)(iimg-1));
                        } else {
                            assert(iimg > 2);
                            cpl_image_power(img, 2.0);
                            /* Correct for extra division with n, above */
                            cpl_image_multiply_scalar(img, (double)
                                                      iimg*(double)(iimg-1));
                            cpl_image_add(img_varn, img);
                        }
                    }
                    cpl_image_delete(img);
                }
            }
            frame = (i < iplanes && iimg < nimg) ? NULL
                : cpl_frameset_get_next_const(self);
        }
    }

    if (cpl_error_get_code()) {
        (void)cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                    "frame=%d/%d", (int)iframe,
                                    (int)cpl_frameset_get_size(self));
    } else if (iimg < nimg) {
        cpl_msg_warning(cpl_func, "No RON computed from frameset with %d "
                        "frame(s) and %d < %d images",
                        (int)cpl_frameset_get_size(self), (int)iimg, (int)nimg);
    } else {
        cpl_image_multiply_scalar(img_varn, 1.0/(double)(iimg-1));
        cpl_image_power(img_varn, 0.5);
        median = cpl_image_get_median(img_varn);
    }

    cpl_image_delete(img_mean);
    cpl_image_delete(img_varn);

    return median;
}


/*----------------------------------------------------------------------------*/
/**
 @brief  Compute the persistance from the hard-coded windows
 @param  self     The QC propertylist to modify
 @param  mframe   The master frame to read from, not (really) modified
 @param  plist    Propertylist w. DIT + SPH_COMMON_KEYWORD_203_TEMP, or NULL
 @param  lollx      Left  x position (1 for leftmost)
 @param  lolly      Lower y position (1 for lowest)
 @param  lourx      Right x position
 @param  loury      Upper y position
 @param  hillx      Left  x position (1 for leftmost)
 @param  hilly      Lower y position (1 for lowest)
 @param  hiurx      Right x position
 @param  hiury      Upper y position
 @return The persistance, or undefined on error

 */
/*----------------------------------------------------------------------------*/
void sph_irdifs_qc_persistance(cpl_propertylist* self,
                               sph_master_frame* mframe,
                               const cpl_propertylist* plist,
                               cpl_size lollx,
                               cpl_size lolly,
                               cpl_size lourx,
                               cpl_size loury,
                               cpl_size hillx,
                               cpl_size hilly,
                               cpl_size hiurx,
                               cpl_size hiury)
{
    if (mframe == NULL) {
        (void)cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
    } else {
        cpl_errorstate prestate = cpl_errorstate_get();
        /* Ignore the bad pixel map (per PIPE-10969), avoid image duplication */
        cpl_mask * bpm = cpl_image_unset_bpm(mframe->image);
        const cpl_image* img = mframe->image;
        /* FIXME: Verify against image size ? */
        const double medianhi =
            cpl_image_get_median_window(img, hillx, hilly, hiurx, hiury);
        const double medianlo =
            cpl_image_get_median_window(img, lollx, lolly, lourx, loury);

        if (!cpl_errorstate_is_equal(prestate)) {
            cpl_msg_warning(cpl_func, "%d X %d - image persistance not computed: "
                            "%d %d %d %d %d %d %d %d",
                            (int)cpl_image_get_size_x(img),
                            (int)cpl_image_get_size_y(img),
                            (int)lollx, (int)lolly, (int)lourx, (int)loury,
                            (int)hillx, (int)hilly, (int)hiurx, (int)hiury);
            cpl_errorstate_dump(prestate, CPL_FALSE,
                                cpl_errorstate_dump_one_warning);
            cpl_errorstate_set(prestate);
        } else {
            const double persist = medianhi - medianlo;

            cpl_msg_info(cpl_func, "Image persistance: %g - %g = %g",
                         medianhi, medianlo, persist);

            cpl_propertylist_append_double(self, SPH_QC_PERSIST, persist);

            if (plist != NULL) {
                if (!cpl_propertylist_has(plist, SPH_COMMON_KEYWORD_SEQ1DIT)) {
                    cpl_msg_warning(cpl_func, "Propertylist does not have "
                                    SPH_COMMON_KEYWORD_SEQ1DIT);
                } else if (!cpl_propertylist_has(plist,
                                                 SPH_COMMON_KEYWORD_203_TEMP)) {
                    cpl_msg_warning(cpl_func, "Propertylist does not have "
                                    SPH_COMMON_KEYWORD_203_TEMP);
                } else {
                    const double dit =
                        cpl_propertylist_get_double(plist,
                                                    SPH_COMMON_KEYWORD_SEQ1DIT);
                    const double t203 =
                        cpl_propertylist_get_double(plist,
                                                    SPH_COMMON_KEYWORD_203_TEMP);

                    if (dit <= 0.0) {
                        cpl_msg_warning(cpl_func, "Value of "
                                        SPH_COMMON_KEYWORD_SEQ1DIT " is not "
                                        "positive: %g", dit);
                    } else if (t203 <= 0.0) {
                        cpl_msg_warning(cpl_func, "Value of "
                                        SPH_COMMON_KEYWORD_203_TEMP " is not "
                                        "positive: %g", t203);
                    } else {
                        const double npersist = persist * 60.0 / ( dit * t203);
                        cpl_msg_info(cpl_func, "Persistance scaled w. %g * %g "
                                     "[ADU/C/minute]: %g", dit, t203, npersist);
                        cpl_propertylist_append_double(self,
                                                       SPH_QC_PERSIST_NORM,
                                                       npersist);
                    }
                }
            }
        }
        (void)cpl_image_set_bpm(mframe->image, bpm); /* Reinstate bpm */
    }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Compute and add QC.BACKGROUND.RATE to the QC propertylist
 * @param self      The QC propertylist to modify
 * @param other     The propertylist to read from
 * @param mediankey The propertylist with the median value to use
 * @return Nothing
 *
 */
/*----------------------------------------------------------------------------*/
void sph_irdifs_bg_qc_rate(cpl_propertylist* self,
                           const cpl_propertylist* other,
                           const char * mediankey)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    const double median =
        cpl_propertylist_get_double(cpl_propertylist_has(self, mediankey) ?
                                    self : other, mediankey);
    const double dit
        = cpl_propertylist_get_double(cpl_propertylist_has
                                      (self, SPH_COMMON_KEYWORD_SEQ1DIT) ?
                                      self : other, SPH_COMMON_KEYWORD_SEQ1DIT);

    if (!cpl_errorstate_is_equal(prestate) || dit <= 0. ||
        cpl_propertylist_update_double(self,
                                       SPH_COMMON_KEYWORD_QC_BACKGROUND_RATE,
                                       median / dit)) {
        cpl_msg_warning(cpl_func, "Could not compute "
                        SPH_COMMON_KEYWORD_QC_BACKGROUND_RATE ": %g / %g",
                        median, dit);
        cpl_errorstate_dump(prestate, CPL_FALSE,
                            cpl_errorstate_dump_one_warning);
        cpl_errorstate_set(prestate);
    }
}

/**@}*/
