/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_psfcube.h"

#include <math.h>
#include <assert.h>

#ifndef SPH_PSFCUBE_POSTOLERANCE
#define SPH_PSFCUBE_POSTOLERANCE 0.0001
#endif

/*----------------------------------------------------------------------------*/
/**
 * @defgroup A SPHERE API Module
 * @par Synopsis:
 * @code
 * typedef _module_ {
 * } module
 * @endcode
 * @par Desciption:
 *
 * This module provides functionality for apertures, extending the functionality
 * as it exists for cpl_apertures.
 */
/*----------------------------------------------------------------------------*/
/**@{*/
/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new PSF cube
 * @param imlistin      the input imagelist
 * @param xin           the input xpositions
 * @param yin           the input ypositions
 *
 * @return a new sph_psfcube or NULL on error.
 *
 * This function creates a new sph_psfcube from the input list of images and the
 * input position vectors in x and y. The size of bot the x and y position vectors
 * must be the same as the number of images in the imagelist.
 * The imagelist and the vectors are "owned" by the psf_cube and should not
 * be deleted elsewhere in the program.
 *
 */
/*----------------------------------------------------------------------------*/
sph_psfcube* sph_psfcube_new( cpl_imagelist* imlistin,
                              cpl_vector* xin,
                              cpl_vector* yin) {
    sph_psfcube*   psfcube = NULL;

    cpl_ensure( imlistin != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure( xin      != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure( yin      != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure( cpl_vector_get_size(xin) == cpl_vector_get_size(yin),
                CPL_ERROR_INCOMPATIBLE_INPUT, NULL);
    cpl_ensure( cpl_imagelist_get_size(imlistin) == cpl_vector_get_size(yin),
                CPL_ERROR_INCOMPATIBLE_INPUT, NULL);

    psfcube = cpl_malloc(sizeof(*psfcube));

    psfcube->imlist = imlistin;
    psfcube->xpos = xin;
    psfcube->ypos = yin;
    return psfcube;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Get the size (number of images) in the cube
 * @param self
 *
 * @return the number of images
 *
 */
/*----------------------------------------------------------------------------*/

int sph_psfcube_get_size( const sph_psfcube* self) {
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, -1);
    return cpl_vector_get_size(self->xpos);
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Get the pixel size (along one edge) of the cube
 * @param self
 *
 * @return the side length in pixels
 *
 */
/*----------------------------------------------------------------------------*/

int sph_psfcube_get_npix_side( const sph_psfcube* self) {
    const cpl_image* img = NULL;

    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, -1);

    img = cpl_imagelist_get_const(self->imlist, 0);
    cpl_ensure(img != NULL,   CPL_ERROR_ILLEGAL_INPUT, -2);

    return cpl_image_get_size_x(img);
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Return an image from the cube
 * @param dx        the position in x of the image to return
 * @param dy        the position in y of the image to return
 *
 * @return image or NULL if none found
 *
 * This function returns the image with the given position in x and y if such
 * an image exists in the cube.
 *
 *
 */
/*----------------------------------------------------------------------------*/
const cpl_image*
sph_psfcube_get_exact( const sph_psfcube* self, double dx, double dy) {
    cpl_size    nz;
    cpl_size    ind     = -1;
    int         errc    = 1;

    cpl_ensure(self         != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(self->imlist != NULL, CPL_ERROR_NULL_INPUT, NULL);
    nz = cpl_imagelist_get_size(self->imlist);
    cpl_ensure(nz > 0, CPL_ERROR_DATA_NOT_FOUND, NULL);

    for (cpl_size ii = 0; ii < nz; ++ii) {
        if ( fabs(cpl_vector_get(self->xpos,ii)-dx) < SPH_PSFCUBE_POSTOLERANCE ) {
            if ( fabs(cpl_vector_get(self->ypos,ii)-dy) < SPH_PSFCUBE_POSTOLERANCE ) {
                if ( ind == -1 ) {
                    ind = ii;
                    errc = 0;
                }
                else {
                    errc = 2; // Set to 2 to signifiy double solution
                }
            }
        }
    }
    if ( errc == 2 ) {
        SPH_ERROR_RAISE_WARNING( CPL_ERROR_ILLEGAL_OUTPUT,
                                 "Found two matches in PSF cube.");
        return cpl_imagelist_get_const(self->imlist, ind);
    }
    if ( errc == 1 ) {
        SPH_ERROR_RAISE_ERR( CPL_ERROR_DATA_NOT_FOUND,
                             "Found no matches in PSF cube.");
        return NULL;
    }
    return cpl_imagelist_get_const(self->imlist, ind);
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Return an image from the cube
 * @param dx        the position in x of the image to return
 * @param dy        the position in y of the image to return
 *
 * @return image or NULL if none found
 *
 * This function returns the image closest to the given position
 * in x and y if such an image exists in the cube.
 *
 *
 */
/*----------------------------------------------------------------------------*/
const cpl_image*
sph_psfcube_get( const sph_psfcube* self, double dx, double dy) {
    cpl_size    nz;
    cpl_size    ind     = -1;
    double      cminx   = DBL_MAX;
    double      cminy   = DBL_MAX;

    cpl_ensure(self         != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(self->imlist != NULL, CPL_ERROR_NULL_INPUT, NULL);

    nz = cpl_imagelist_get_size(self->imlist);
    cpl_ensure(nz > 0, CPL_ERROR_DATA_NOT_FOUND, NULL);

    for (cpl_size ii = 0; ii < nz; ++ii) {
        if ( fabs(cpl_vector_get(self->xpos, ii)-dx) < cminx ) {
            if ( fabs(cpl_vector_get(self->ypos, ii)-dy) < cminy ) {
                ind = ii;
                cminx = fabs(cpl_vector_get(self->xpos,ii)-dx);
                cminy = fabs(cpl_vector_get(self->xpos,ii)-dx);
            }
        }
    }

    assert( ind < nz);

    return ind < 0 ? NULL : cpl_imagelist_get_const(self->imlist, ind);
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Insert a new image into cube
 * @param self
 * @param im    the image to insert
 * @param dx    the x position of the new image
 * @param dy    the y postion of the new image
 *
 * @return error code
 *
 * This function inserts the new image into the imagelist as maintained by
 * the cube. It stores the image at the specified position. If the position
 * in x and y are already occupied by an image, an error code
 * CPL_ERROR_ILLEGAL_INPUT is returned.
 *
 *
 */
/*----------------------------------------------------------------------------*/

sph_error_code
sph_psfcube_insert( sph_psfcube* self, cpl_image* im, double dx, double dy) {
    int         nels    = 0;

    cpl_ensure_code( self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( im   != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( sph_psfcube_get_exact(self,dx,dy) == NULL,
                     CPL_ERROR_ILLEGAL_INPUT);
    sph_error_unset_last(); // The last error was caused by the get call as it found
                            // nothing, but not getting anything back does not
                            // constitute an error here, but is what we wanted.
    nels = cpl_imagelist_get_size(self->imlist);

    cpl_vector_set_size(self->xpos,nels+1);
    cpl_vector_set_size(self->ypos,nels+1);
    cpl_vector_set(self->xpos,nels,dx);
    cpl_vector_set(self->ypos,nels,dy);
    cpl_imagelist_set(self->imlist,im,nels);

    return cpl_error_set_where(cpl_func);
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Delete the psf_cube
 *
 * Deletes the psf_cube and all associated resources.
 *
 */
/*----------------------------------------------------------------------------*/
void
sph_psfcube_delete(sph_psfcube* self) {
    if ( self ) {
        cpl_imagelist_delete(self->imlist);
        cpl_vector_delete(self->xpos);
        cpl_vector_delete(self->ypos);
        cpl_free(self);
    }
}

/**@}*/
