/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifndef SPH_NO_USE_HDRL
/* Use HDRL by default, but provide an option to turn it off */
#define HDRL_USE_EXPERIMENTAL
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_strehl.h"
#include "sph_common_keywords.h"
#include "sph_phot.h"
#include "irplib_strehl.h"
#include "irplib_plugin.h"
#include "irplib_utils.h"

#ifdef HDRL_USE_EXPERIMENTAL
#include "hdrl.h"
#endif

#include <assert.h>
#include <string.h>
#include <math.h>

/*-----------------------------------------------------------------------------
                                   Define
 -----------------------------------------------------------------------------*/

#ifndef SPH_STREHL_COLUMN_KEY
#define SPH_STREHL_COLUMN_KEY "filter_key"
#endif

#ifndef SPH_STREHL_COLUMN_ID
#define SPH_STREHL_COLUMN_ID "filter_id"
#endif

#ifndef SPH_STREHL_COLUMN_LAMBDA_LEFT
#define SPH_STREHL_COLUMN_LAMBDA_LEFT "lambda_left"
#endif

#ifndef SPH_STREHL_COLUMN_LAMBDA_RIGHT
#define SPH_STREHL_COLUMN_LAMBDA_RIGHT "lambda_right"
#endif

#ifndef SPH_STREHL_COLUMN_WIDTH_LEFT
#define SPH_STREHL_COLUMN_WIDTH_LEFT "width_left"
#endif

#ifndef SPH_STREHL_COLUMN_WIDTH_RIGHT
#define SPH_STREHL_COLUMN_WIDTH_RIGHT "width_right"
#endif

#ifndef SPH_STREHL_COLUMN_LAMBDA_CAM1
#define SPH_STREHL_COLUMN_LAMBDA_CAM1 "lambda_cam1"
#endif

#ifndef SPH_STREHL_COLUMN_LAMBDA_CAM2
#define SPH_STREHL_COLUMN_LAMBDA_CAM2 "lambda_cam2"
#endif

#ifndef SPH_STREHL_COLUMN_WIDTH_CAM1
#define SPH_STREHL_COLUMN_WIDTH_CAM1 "width_cam1"
#endif

#ifndef SPH_STREHL_COLUMN_WIDTH_CAM2
#define SPH_STREHL_COLUMN_WIDTH_CAM2 "width_cam2"
#endif

#ifndef SPH_STREHL_BORDER
#define SPH_STREHL_BORDER 256
#endif


/* Modified from fitsio.h */
#define SPH_FLEN_KEYWORD 72  /* max length of a keyword (HIERARCH convention) */

/*-----------------------------------------------------------------------------
                                   Function declarations
 -----------------------------------------------------------------------------*/

static cpl_error_code
sph_strehl_compute_core(const sph_arm             arm,
                        const cpl_image         * image,
                        const cpl_propertylist  * rawplist,
                        const char              * label_lambda,
                        const char              * label_width,
                        const cpl_frame         * filterframe,
                        const char              * armname,
                        const char              * recipe,
                        const cpl_parameterlist * parlist,
                        sph_strehl_qc_pars      * qc_out_pars);

static
cpl_error_code sph_strehl_compute(const cpl_image         * image,
                                  sph_arm                   arm,
                                  const char              * loadregexp,
                                  const cpl_frame         * rawframe,
                                  const cpl_frame         * filterframe,
                                  const char              * label1_lambda,
                                  const char              * label1_width,
                                  const char              * armname,
                                  const char              * recipe,
                                  const cpl_parameterlist * parlist,
                                  sph_strehl_qc_pars      * qc_out_pars);


static cpl_error_code sph_strehl_read_parlist(const cpl_parameterlist * self,
                                              const char * armname,
                                              const char * recipe,
                                              const double star_r_default,
                                              const double bg_r1_default,
                                              const double bg_r2_default,
                                              double * pstar_r,
                                              double * pbg_r1,
                                              double * pbg_r2);

static cpl_error_code
sph_strehl_read_parlist_star_r(const cpl_parameterlist * self,
                                              const char * armname,
                                              const char * recipe,
                                              double * pstar_r);

static cpl_boolean sph_strehl_is_neutral_density(const char * str);

static cpl_error_code
sph_strehl_get_star_flux_ao_disabled(const cpl_image * im, const cpl_parameterlist * parlist,
        const cpl_propertylist * proplist, const char * arm_name, const sph_arm arm,
        const char * rec_name, double * flx, double *bg);



static
cpl_error_code sph_strehl_fill_flux_bg_qc(cpl_propertylist * self,
        const char * key_star_flux, double star_flux,
        const char * key_star_bg, double star_bg);

static cpl_error_code sph_strehl_read_scale(const cpl_propertylist * self,
        const sph_arm arm, double * ppscale);

static cpl_error_code
sph_strehl_irdis_internal(const cpl_image         * image,
                          const cpl_frame         * rawframe,
                          const cpl_frame         * filterframe,
                          const char              * recipe,
                          const cpl_parameterlist * parlist,
                          const char              * lambda_label,
                          const char              * width_label,
                          sph_strehl_qc_pars      * qc_out_pars);

static cpl_error_code
sph_strehl_zimpol_internal(const cpl_image         * image,
                           const sph_arm             arm,
                           const cpl_frame         * rawframe,
                           const cpl_frame         * filterframe,
                           const char              * recipe,
                           const cpl_parameterlist * parlist,
                           const char              * lambda_label,
                           const char              * width_label,
                           sph_strehl_qc_pars      * qc_out_pars);

static cpl_error_code
sph_strehl_disabled_ao_flux_internal(
                                     const cpl_image         * image_cam,
                                           const cpl_frame         * rawframe,
                                     const cpl_parameterlist * parlist,
                                     const char              * recipe,
                                     const sph_arm                arm,
                                     double                  * flux,
                                     double                  * background);

static cpl_error_code
sph_strehl_fill_qc_pars(cpl_propertylist * self,
                        const sph_strehl_qc_pars * pars,
                        const char * key_strehl,
                        const char * key_strehl_err,
                        const char * key_pos_x,
                        const char * key_pos_y,
                        const char * key_sigma,
                        const char * key_star_flux,
                        const char * key_star_peak,
                        const char * key_star_bg,
                        const char * key_star_bg_noise);

static cpl_error_code
sph_strehl_zimpol_1(const cpl_image         * image1,
                    const cpl_frame         * rawframe,
                    const cpl_frame         * filterframe,
                    const char              * recipe,
                    const cpl_parameterlist * parlist,
                    sph_strehl_qc_pars      * qc_out_pars);

static cpl_error_code
sph_strehl_zimpol_2(const cpl_image         * image2,
                    const cpl_frame         * rawframe,
                    const cpl_frame         * filterframe,
                    const char              * recipe,
                    const cpl_parameterlist * parlist,
                    sph_strehl_qc_pars      * qc_out_pars);


/*-----------------------------------------------------------------------------
                                   Function definitions
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 @brief Compute the star flux in the case of disabled adaptive optics for ZIMPOL 1/2
 @param image_cam   Image to process from camera 1/2
 @param rawframe    The input raw frame
 @param parlist     Recipe parameter list
 @param recipe      Recipe name (for parameter access)
 @param is_arm_1    CPL_TRUE if ZIMPOL_ARM1 arm
 @param flux        Calculated flux
 @param background  Calculated background
 @return CPL_ERROR_NONE or the relevant CPL error code on error
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
sph_strehl_disabled_ao_flux_zimpol(const cpl_image         * image_cam,
                                   const cpl_frame         * rawframe,
                                   const cpl_parameterlist * parlist,
                                   const char              * recipe,
                                   const cpl_boolean         is_arm_1,
                                   double                  * flux,
                                   double                    * background){

    const sph_arm arm = is_arm_1 ? SPH_ARM_ZIMPOL1 : SPH_ARM_ZIMPOL2;
    return sph_strehl_disabled_ao_flux_internal(image_cam, rawframe,
                parlist, recipe, arm, flux, background);
}

/*----------------------------------------------------------------------------*/
/**
 @brief Compute the star flux in the case of disabled adaptive optics for IRDIS
 @param image_cam   Image to process from camera left/right
 @param rawframe    The input raw frame
 @param parlist     Recipe parameter list
 @param recipe      Recipe name (for parameter access)
 @param flux        Calculated flux
 @param background  Calculated background
 @return CPL_ERROR_NONE or the relevant CPL error code on error
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
sph_strehl_disabled_ao_flux_irdis(const cpl_image         * image_cam,
                                  const cpl_frame         * rawframe,
                                  const cpl_parameterlist * parlist,
                                  const char              * recipe,
                                  double                  * flux,
                                  double                  * background){

    return sph_strehl_disabled_ao_flux_internal(image_cam, rawframe,
                parlist, recipe, SPH_ARM_IRDIS, flux, background);
}


/*----------------------------------------------------------------------------*/
/**
 @internal
 @brief Compute the star flux in the case of disabled adaptive optics for ZIMPOL/IRDIS
 @param image_cam   Image to process from camera 1 or 2
 @param rawframe    The input raw frame
 @param parlist     Recipe parameter list
 @param recipe      Recipe name (for parameter access)
 @param arm         Arm of ZIMPOL (ZIMPOL 1/2) or IRDIS
 @param flux        Calculated flux
 @param background  Calculated background
 @return   CPL_ERROR_NONE or the relevant CPL error code on error
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code
sph_strehl_disabled_ao_flux_internal(
                                     const cpl_image         * image_cam,
                                           const cpl_frame         * rawframe,
                                     const cpl_parameterlist * parlist,
                                     const char              * recipe,
                                     const sph_arm                arm,
                                     double                  * flux,
                                     double                  * background){



    cpl_ensure_code(flux       != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(background != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(recipe     != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(parlist    != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(rawframe   != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(image_cam  != NULL, CPL_ERROR_NULL_INPUT);

    const char  * filename = cpl_frame_get_filename(rawframe);

    cpl_ensure_code(filename      != NULL, CPL_ERROR_NULL_INPUT);
    cpl_propertylist * list = cpl_propertylist_load(filename, 0);

    const char * arm_name = "zpl";
    if(arm == SPH_ARM_IRDIS){
        arm_name = "ird";
    }

    const cpl_error_code err1 =
            sph_strehl_get_star_flux_ao_disabled(image_cam, parlist,
                    list, arm_name, arm, recipe,
                    flux, background);

    cpl_propertylist_delete(list);

    if(err1) return cpl_error_set_where(cpl_func);

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Compute the Strehl ratio of the brightest object in the IRDIS left image
 @param image        Image to process
 @param rawframe     The input raw frame
 @param filterframe  An input calibration frame with the filter wavelengths
 @param recipe       Recipe name (for parameter access)
 @param parlist      Recipe parameters
 @param qc_out_pars  The output qc parameters. Undefined in case of error.
 @return   CPL_ERROR_NONE or the relevant CPL error code on error
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
sph_strehl_irdis_left(const cpl_image         * image,
                      const cpl_frame         * rawframe,
                      const cpl_frame         * filterframe,
                      const char              * recipe,
                      const cpl_parameterlist * parlist,
                      sph_strehl_qc_pars      * qc_out_pars) {

    return sph_strehl_irdis_internal(image, rawframe,
            filterframe, recipe, parlist,
            SPH_STREHL_COLUMN_LAMBDA_LEFT,
            SPH_STREHL_COLUMN_WIDTH_LEFT, qc_out_pars);
}

/*----------------------------------------------------------------------------*/
/**
 @brief Compute the Strehl ratio of the brightest object in the IRDIS right image
 @param image        Image to process
 @param rawframe     The input raw frame
 @param filterframe  An input calibration frame with the filter wavelengths
 @param recipe       Recipe name (for parameter access)
 @param parlist      Recipe parameters
 @param qc_out_pars  The output qc parameters. Undefined in case of error.
 @return   CPL_ERROR_NONE or the relevant CPL error code on error
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
sph_strehl_irdis_right(const cpl_image         * image,
                       const cpl_frame         * rawframe,
                       const cpl_frame         * filterframe,
                       const char              * recipe,
                       const cpl_parameterlist * parlist,
                       sph_strehl_qc_pars      * qc_out_pars) {

    return sph_strehl_irdis_internal(image, rawframe,
            filterframe, recipe, parlist,
            SPH_STREHL_COLUMN_LAMBDA_RIGHT,
            SPH_STREHL_COLUMN_WIDTH_RIGHT, qc_out_pars);
}

/*----------------------------------------------------------------------------*/
/**
 @internal
 @brief Compute the Strehl ratio of the brightest object in the IRDIS image
 @param image        Image to process
 @param rawframe     The input raw frame
 @param filterframe  An input calibration frame with the filter wavelengths
 @param recipe       Recipe name (for parameter access)
 @param parlist      Recipe parameters
 @param lambda_label The column label of the filter lambda for the image
 @param width_label  The column label of the filter width for the image
 @param qc_out_pars  The output qc parameters. Undefined in case of error.
 @return   CPL_ERROR_NONE or the relevant CPL error code on error
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code
sph_strehl_irdis_internal(const cpl_image         * image,
                          const cpl_frame         * rawframe,
                          const cpl_frame         * filterframe,
                          const char              * recipe,
                          const cpl_parameterlist * parlist,
                          const char              * lambda_label,
                          const char              * width_label,
                          sph_strehl_qc_pars      * qc_out_pars) {

    const cpl_error_code code =
        sph_strehl_compute(image, SPH_ARM_IRDIS,
                           SPH_STREHL_PSCALE      "|"
                           SPH_STREHL_FILTER_IRD1 "|"
                           SPH_STREHL_FILTER_IRD2,
                           rawframe, filterframe,
                           lambda_label, width_label,
                           "ird", recipe, parlist, qc_out_pars);

    return code ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Compute the Strehl ratio of the brightest object in the ZIMPOL 2 image
 @param image        Image to process
 @param rawframe     The input raw frame
 @param filterframe  An input calibration frame with the filter wavelengths
 @param recipe       Recipe name (for parameter access)
 @param parlist      Recipe parameters
 @param is_arm_1    CPL_TRUE if ZIMPOL_ARM1 arm
 @param qc_out_pars  The output qc parameters. Undefined in case of error.
 @return   CPL_ERROR_NONE or the relevant CPL error code on error
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
sph_strehl_zimpol(const cpl_image         * image,
                  const cpl_frame         * rawframe,
                  const cpl_frame         * filterframe,
                  const char              * recipe,
                  const cpl_parameterlist * parlist,
                  const cpl_boolean          is_arm1,
                  sph_strehl_qc_pars      * qc_out_pars){

    if(is_arm1)
        return sph_strehl_zimpol_1(image, rawframe, filterframe, recipe, parlist, qc_out_pars);
    return sph_strehl_zimpol_2(image, rawframe, filterframe, recipe, parlist, qc_out_pars);
}

/*----------------------------------------------------------------------------*/
/**
 @internal
 @brief Compute the Strehl ratio of the brightest object in the ZIMPOL 1 image
 @param image        Image to process
 @param rawframe     The input raw frame
 @param filterframe  An input calibration frame with the filter wavelengths
 @param recipe       Recipe name (for parameter access)
 @param parlist      Recipe parameters
 @param qc_out_pars  The output qc parameters. Undefined in case of error.
 @return   CPL_ERROR_NONE or the relevant CPL error code on error
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code
sph_strehl_zimpol_1(const cpl_image         * image,
                    const cpl_frame         * rawframe,
                    const cpl_frame         * filterframe,
                    const char              * recipe,
                    const cpl_parameterlist * parlist,
                    sph_strehl_qc_pars      * qc_out_pars) {

    return sph_strehl_zimpol_internal(image, SPH_ARM_ZIMPOL1,
            rawframe, filterframe, recipe, parlist,
            SPH_STREHL_COLUMN_LAMBDA_CAM1,
            SPH_STREHL_COLUMN_WIDTH_CAM1, qc_out_pars);
}

/*----------------------------------------------------------------------------*/
/**
 @internal
 @brief Compute the Strehl ratio of the brightest object in the ZIMPOL 2 image
 @param image        Image to process
 @param rawframe     The input raw frame
 @param filterframe  An input calibration frame with the filter wavelengths
 @param recipe       Recipe name (for parameter access)
 @param parlist      Recipe parameters
 @param qc_out_pars  The output qc parameters. Undefined in case of error.
 @return   CPL_ERROR_NONE or the relevant CPL error code on error
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code
sph_strehl_zimpol_2(const cpl_image         * image,
                    const cpl_frame         * rawframe,
                    const cpl_frame         * filterframe,
                    const char              * recipe,
                    const cpl_parameterlist * parlist,
                    sph_strehl_qc_pars      * qc_out_pars) {

    return sph_strehl_zimpol_internal(image, SPH_ARM_ZIMPOL2,
            rawframe, filterframe, recipe, parlist,
            SPH_STREHL_COLUMN_LAMBDA_CAM2,
            SPH_STREHL_COLUMN_WIDTH_CAM2, qc_out_pars);
}

/*----------------------------------------------------------------------------*/
/**
 @internal
 @brief Compute the Strehl ratio of the brightest object in the ZIMPOL image
 @param image       Image to process from camera
 @param arm         Arm of ZIMPOL (ZIMPOL 1/2)
 @param rawframe    The input raw frame
 @param filterframe An input calibration frame with the filter wavelengths
 @param recipe      Recipe name (for parameter access)
 @param parlist     Recipe parameters
 @param lambda_label The column label of the filter lambda for the image
 @param width_label  The column label of the filter width for the image
 @param qc_out_pars  The output qc parameters. Undefined in case of error.
 @return   CPL_ERROR_NONE or the relevant CPL error code on error
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code
sph_strehl_zimpol_internal(const cpl_image         * image,
                           const sph_arm             arm,
                           const cpl_frame         * rawframe,
                           const cpl_frame         * filterframe,
                           const char              * recipe,
                           const cpl_parameterlist * parlist,
                           const char              * lambda_label,
                           const char              * width_label,
                           sph_strehl_qc_pars      * qc_out_pars){

    const cpl_error_code code =
            sph_strehl_compute(image, arm,
                               SPH_STREHL_PSCALE      "|"
                               SPH_STREHL_FILTER_ZPL0 "|"
                               SPH_STREHL_FILTER_ZPL1 "|"
                               SPH_STREHL_FILTER_ZPL2,
                               rawframe, filterframe,
                               lambda_label, width_label,
                               "zpl", recipe, parlist,
                               qc_out_pars);

    return code ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 @internal
 @brief Compute the Strehl ratio of the brightest object in the 2 ZIMPOL images
 @param image         Image to process
 @param arm           The SPHERE arm (IRDIS or ZIMPOL1/2)
 @param loadregexp    The FITS cards to load
 @param rawframe      The input raw frame
 @param filterframe   An input calibration frame with the filter wavelengths
 @param label_lambda  The column label of the filter lambda for image 1
 @param label_width   The column label of the filter width for image 1
 @param armname       The name of the arm, "ird" or "zpl"
 @param recipe        Recipe name (for parameter access)
 @param parlist       Recipe parameters
 @param qc_out_pars  The output qc parameters. Undefined in case of error.
 @return   CPL_ERROR_NONE or the relevant CPL error code on error
 @note If the parameter star_r is non-positive, the estimation is disabled

 */
/*----------------------------------------------------------------------------*/
static
cpl_error_code sph_strehl_compute(const cpl_image         * image,
                                  sph_arm                   arm,
                                  const char              * loadregexp,
                                  const cpl_frame         * rawframe,
                                  const cpl_frame         * filterframe,
                                  const char              * label_lambda,
                                  const char              * label_width,
                                  const char              * armname,
                                  const char              * recipe,
                                  const cpl_parameterlist * parlist,
                                  sph_strehl_qc_pars      * qc_out_pars) {

    cpl_ensure_code(rawframe      != NULL, CPL_ERROR_NULL_INPUT);

    double star_r = 0.0; /* Used to check star_r */
    cpl_errorstate     okstate  = cpl_errorstate_get();
    const char       * filename = cpl_frame_get_filename(rawframe);
    cpl_propertylist * plist;

    cpl_ensure_code(image         != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(loadregexp    != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(parlist       != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(filename      != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(filterframe   != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(label_lambda  != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(label_width   != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(armname       != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(recipe        != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(parlist       != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(qc_out_pars   != NULL, CPL_ERROR_NULL_INPUT);

    if(sph_strehl_read_parlist_star_r(parlist, armname, recipe, &star_r))
        return cpl_error_set_where(cpl_func);

    if (star_r < 0.0) {
        cpl_msg_info(cpl_func, "Strehl ratio estimation disabled: "
                     "star_r = %g <= 0.0", star_r);
        return CPL_ERROR_NONE;
    }

    plist = cpl_propertylist_load_regexp(filename, 0, loadregexp, CPL_FALSE);

    if (!cpl_errorstate_is_equal(okstate)) return cpl_error_set_where(cpl_func);

    assert( plist != NULL);

    if (sph_strehl_compute_core(arm, image, plist,
                           label_lambda, label_width,
                           filterframe, armname, recipe,
                           parlist,qc_out_pars)) {
        cpl_propertylist_delete(plist);
        return cpl_error_set_where(cpl_func);
    }

    cpl_propertylist_delete(plist);

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Read the pixel scale from the header value, or fall back to a default
 @param self    Propertylist of raw input frame
 @param arm     The SPHERE arm (IRDIS or ZIMPOL)
 @param ppscale The pointer to the pixel scale [Arcseconds/pixel]
 @return   CPL_ERROR_NONE or the relevant CPL error code on error

*/
/*----------------------------------------------------------------------------*/
static
cpl_error_code sph_strehl_read_scale(const cpl_propertylist * self,
                                     const sph_arm arm,
                                     double * ppscale) {

    const cpl_errorstate okstate = cpl_errorstate_get();

    cpl_ensure_code(ppscale != NULL, CPL_ERROR_NULL_INPUT);

    *ppscale = cpl_propertylist_get_double(self, SPH_STREHL_PSCALE);

    if (cpl_errorstate_is_equal(okstate)) {
        /* Convert header's Pixel Scale from [mas/pixel] to [as/pixel] */
        *ppscale *= 1e-3;
        if (arm == SPH_ARM_ZIMPOL1 || arm == SPH_ARM_ZIMPOL2) {
            *ppscale *= 0.5; /* Additional ZIMPOL factor... */
        }
        cpl_msg_info(cpl_func, "Pixel scale from " SPH_STREHL_PSCALE
                     " key [as/pixel]= %f", *ppscale);
    } else if (arm == SPH_ARM_IRDIS) {
        *ppscale = SPH_STREHL_PSCALE_IRDIS;

        cpl_msg_warning(cpl_func, "Using default IRDIS pixel scale "
                        "[Arcseconds/pixel]: "
                        IRPLIB_STRINGIFY(SPH_STREHL_PSCALE_IRDIS));
        cpl_errorstate_dump(okstate, CPL_FALSE,
                            cpl_errorstate_dump_one_warning);
        cpl_errorstate_set(okstate);
    } else if (arm == SPH_ARM_ZIMPOL1 || arm == SPH_ARM_ZIMPOL2) {
        *ppscale = SPH_STREHL_PSCALE_ZIMPOL;

        cpl_msg_warning(cpl_func, "Using default ZIMPOL pixel scale "
                        "[Arcseconds/pixel]: "
                        IRPLIB_STRINGIFY(SPH_STREHL_PSCALE_ZIMPOL));
        cpl_errorstate_dump(okstate, CPL_FALSE,
                            cpl_errorstate_dump_one_warning);
        cpl_errorstate_set(okstate);
    } else {
        return cpl_error_set_where(cpl_func);
    }
    return CPL_ERROR_NONE;
}


/*----------------------------------------------------------------------------*/
/**
 @brief Read header values from raw frame for IRDIS and ZIMPOL
 @param self    Propertylist of raw input frame
 @param arm     The SPHERE arm (IRDIS or ZIMPOL)
 @param ppscale The pixel scale
 @param pkey    The filter key name
 @param pfilter The filter name
 @return   CPL_ERROR_NONE or the relevant CPL error code on error
 @note The filter name is only valid while the propertylist exists

 For IRDIS the light passes through first a broad band filter, with the
 actual filter indicated by 'ESO INS1 FILT NAME'-card. Next, it passes
 though a narrow band filter, with the actual filter indicated by
 'ESO INS1 OPTI2 NAME'-card. The card may have the value 'CLEAR', indicating
 that the second aperture is clear, with no filter. In this case the actual
 filter is the broad band filter.

 For ZIMPOL there is also a shared filter and for each of the two cameras a
 camera specific filter. For ZIMPOL we first check the camera specific filter
 and if it is OPEN, we use the characteristics of the shared filter.
*/
/*----------------------------------------------------------------------------*/
cpl_error_code sph_strehl_read_filter_and_scale(
                                    const cpl_propertylist * self,
                                    sph_arm                  arm,
                                    double                 * ppscale,
                                    const char            ** pkey,
                                    const char            ** pfilter)
{
    cpl_errorstate okstate = cpl_errorstate_get();
    const char * strval;

    cpl_ensure_code(self    != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(pkey    != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(pfilter != NULL, CPL_ERROR_NULL_INPUT);

    if(sph_strehl_read_scale(self, arm, ppscale))
        return cpl_error_set_where(cpl_func);

    if (arm == SPH_ARM_IRDIS) {

        strval = cpl_propertylist_get_string(self, SPH_STREHL_FILTER_IRD2);
        if (!cpl_errorstate_is_equal(okstate)) {
            return cpl_error_set_where(cpl_func);
        }

        assert(strval != NULL);

        if (strcmp(strval, "CLEAR") == 0) {
            strval = cpl_propertylist_get_string(self,
                                                 SPH_STREHL_FILTER_IRD1);
            if (!cpl_errorstate_is_equal(okstate)) {
                return cpl_error_set_where(cpl_func);
            }

            assert(strval != NULL);

            *pkey = SPH_STREHL_FILTER_KEY_IRD1;
        } else {
            *pkey = SPH_STREHL_FILTER_KEY_IRD2;
        }

    } else if (arm == SPH_ARM_ZIMPOL1 || arm == SPH_ARM_ZIMPOL2) {

        strval = cpl_propertylist_get_string(self, SPH_STREHL_FILTER_ZPL0);
        if (!cpl_errorstate_is_equal(okstate)) {
            return cpl_error_set_where(cpl_func);
        }

        assert(strval != NULL);

        if (strcmp(strval, "OPEN") == 0 || sph_strehl_is_neutral_density(strval)) {
            const char * key = arm == SPH_ARM_ZIMPOL1 ? SPH_STREHL_FILTER_ZPL1
                                                      : SPH_STREHL_FILTER_ZPL2;

            strval = cpl_propertylist_get_string(self, key);
            if (!cpl_errorstate_is_equal(okstate))
                return cpl_error_set_where(cpl_func);

            assert(strval != NULL);

            *pkey = arm == SPH_ARM_ZIMPOL1 ? SPH_STREHL_FILTER_KEY_ZPL1
                                           : SPH_STREHL_FILTER_KEY_ZPL2;
        } else {
            *pkey = SPH_STREHL_FILTER_KEY_ZPL0;
        }
    } else {
        return cpl_error_set_message(cpl_func, CPL_ERROR_UNSUPPORTED_MODE,
                                     "Unknown arm: %d", arm);
    }

    *pfilter = strval;

    cpl_msg_info(cpl_func, "Strehl estimation with filter %s = %s "
                 "and pixel scale: %g", *pkey, *pfilter, *ppscale);

    return CPL_ERROR_NONE;

}

/*----------------------------------------------------------------------------*/
/**
 @brief Find (approximately) the star to analyze
 @param self         The image to process
 @param ppos_x       The X-position on success otherwise undefined
 @param ppos_y       The Y-position on success otherwise undefined
 @param psigma       The detection level
 @return   CPL_ERROR_NONE or the relevant CPL error code on error
*/
/*----------------------------------------------------------------------------*/
static cpl_error_code sph_strehl_find_star(const cpl_image * self,
                                           double          * ppos_x,
                                           double          * ppos_y,
                                           double          * psigma)
{

    const cpl_size nx = cpl_image_get_size_x(self);
    const cpl_size ny = cpl_image_get_size_y(self);

    /* Sigma-levels for detection of a bright star. */
    double          psigmas[] = {5.0, 2.0, 1.0, 0.5};
    const int       nsigmas = (int)(sizeof(psigmas)/sizeof(double));
    cpl_size        isigma;
    cpl_vector    * sigmas = NULL;
    cpl_apertures * apert = NULL;
    int             iflux;

    cpl_ensure_code(self   != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(ppos_x != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(ppos_y != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(psigma != NULL, CPL_ERROR_NULL_INPUT);

    /* Detect a bright star around the center */
    cpl_msg_info(cpl_func, "---> Detecting a bright star using %d sigma-levels "
                 "ranging from %g down to %g", nsigmas, psigmas[0],
                 psigmas[nsigmas-1]);
    sigmas = cpl_vector_wrap(nsigmas, psigmas);
    apert = cpl_apertures_extract_window(self, sigmas,
                                         1  + SPH_STREHL_BORDER,
                                         1  + SPH_STREHL_BORDER,
                                         nx - SPH_STREHL_BORDER,
                                         ny - SPH_STREHL_BORDER,
                                         &isigma);
    cpl_vector_unwrap(sigmas);

    if (apert == NULL) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                     "No star in %u X %u with "
                                     IRPLIB_STRINGIFY(SPH_STREHL_BORDER)
                                     " border at sigma=%g", (unsigned)nx,
                                     (unsigned)ny, psigmas[nsigmas-1]);
    }

    if (irplib_apertures_find_max_flux(apert, &iflux, 1)) {
        cpl_apertures_delete(apert);
        return cpl_error_set_where(cpl_func);
    }

    *ppos_x = cpl_apertures_get_centroid_x(apert, iflux);
    *ppos_y = cpl_apertures_get_centroid_y(apert, iflux);

    *psigma = psigmas[isigma];

    cpl_msg_info(cpl_func, "---> Detected the star at %g sigma among %d "
                 "aperture(s) at position (%g, %g)", *psigma,
                 (int)cpl_apertures_get_size(apert),
                 *ppos_x, *ppos_y);

    cpl_apertures_delete(apert);

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Read filter values from filter frame
 @param self         Filter frame to read from
 @param label_lambda The column label of the filter lambda
 @param label_width  The column label of the filter width
 @param key          The key name
 @param filter       The filter name
 @param plam         Central wavelength [micron]
 @param pdlam        Filter bandwidth [micron]
 @return   CPL_ERROR_NONE or the relevant CPL error code on error
*/
/*----------------------------------------------------------------------------*/
static cpl_error_code sph_strehl_read_filter(const cpl_frame * self,
                                             const char      * label_lambda,
                                             const char      * label_width,
                                             const char      * key,
                                             const char      * filter,
                                             double          * plam,
                                             double          * pdlam)
{

    cpl_errorstate okstate  = cpl_errorstate_get();
    const char   * filename = cpl_frame_get_filename(self);
    cpl_table    * table    = cpl_table_load(filename, 1, 0);
    const cpl_size nrow     = cpl_table_get_nrow(table);
    cpl_size       nsel;


    cpl_ensure_code(self         != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(label_lambda != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(label_width  != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(key          != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(filter       != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(plam         != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(pdlam        != NULL, CPL_ERROR_NULL_INPUT);

    if (table == NULL) return cpl_error_set_where(cpl_func);

    /* Select rows not matching the filter key (initially all selected) */
    nsel = cpl_table_and_selected_string(table, SPH_STREHL_COLUMN_KEY,
                                         CPL_NOT_EQUAL_TO, key);
    if (nsel < 0) return cpl_error_set_where(cpl_func);

    /* Select rows not matching the filter id */
    nsel = cpl_table_or_selected_string(table, SPH_STREHL_COLUMN_ID,
                                         CPL_NOT_EQUAL_TO, filter);
    if (nsel < 0) return cpl_error_set_where(cpl_func);

    nsel = nrow - nsel; /* Matching rows */

    if (nsel < 1) return
        cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND, "file=%s. "
                              "key=%s. filter=%s. rows=%u", filename, key,
                              filter, (unsigned)nrow);
    /*
     * The following is a workaround as CPL_NOT_EQUAL_TO can apparently not
     * distinguish between the strings 'B_H' and 'B_Hcmp2'
     */
    if (cpl_table_erase_selected(table)) return cpl_error_set_where(cpl_func);

    if (cpl_table_get_nrow(table) > 1) {
        cpl_msg_warning(cpl_func, "file=%s. key=%s. filter=%s: %u of %u rows "
                        "match, selecting first match!", filename, key, filter,
                        (unsigned)nsel, (unsigned)nrow);
        cpl_table_and_selected_window(table, 1, 9999);
        if (cpl_table_erase_selected(table))
            return cpl_error_set_where(cpl_func);
        cpl_msg_info(cpl_func, "Selected filter: %s",
                     cpl_table_get_string(table,"filter_id",0));
    }

    /*
     * End workaround
     */


    *plam  = cpl_table_get_double(table, label_lambda, 0, NULL);
    *pdlam = cpl_table_get_double(table, label_width,  0, NULL);

    cpl_table_delete(table);

    if (!cpl_errorstate_is_equal(okstate)) return cpl_error_set_where(cpl_func);

    if (*plam <= 0.0) return
        cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT, "file=%s. key="
                              "%s. filter=%s. %u row(s). lambda=%g <= 0. width="
                              "%g", filename, key, filter, (unsigned)nrow,
                              *plam, *pdlam);
    if (*pdlam <= 0.0) return
        cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT, "file=%s. key="
                              "%s. filter=%s. %u row(s). lambda=%g. width=%g <="
                              " 0", filename, key, filter, (unsigned)nrow,
                              *plam, *pdlam);

    return CPL_ERROR_NONE;
}


/*----------------------------------------------------------------------------*/
/**
 @brief Read  star radius parameter from parameter list
 @param self            Parameterlist to read parameters from
 @param armname         The name of the arm, "ird" or "zpl"
 @param recipe          Recipe name (for parameter access)
 @param pstar_r         Star radius
 @return   CPL_ERROR_NONE or the relevant CPL error code on error
*/
/*----------------------------------------------------------------------------*/
static cpl_error_code
sph_strehl_read_parlist_star_r(const cpl_parameterlist * self,
                               const char * armname,
                               const char * recipe,
                               double * pstar_r){

    cpl_ensure_code(pstar_r != NULL, CPL_ERROR_NULL_INPUT);

    const cpl_errorstate okstate = cpl_errorstate_get();

    *pstar_r = irplib_parameterlist_get_double(self, armname, recipe, "star_r");
    if (!cpl_errorstate_is_equal(okstate)) {
        *pstar_r = IRPLIB_STREHL_STAR_RADIUS;
        cpl_msg_warning(cpl_func, "Recipe parameter 'star_r' missing, using def"
                        "ault: " IRPLIB_STRINGIFY(IRPLIB_STREHL_STAR_RADIUS));
        cpl_errorstate_set(okstate);
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Read parameters from parameter list, override with default if the value
          extracted from the parameter list is 0.0
 @param self            Parameterlist to read parameters from
 @param armname         The name of the arm, "ird" or "zpl"
 @param recipe          Recipe name (for parameter access)
 @param star_r_default  Star radius default
 @param bg_r1_default   Background internal radius default
 @param bg_r2_default   Background external radius default
 @param pstar_r         Star radius
 @param pbg_r1          Background internal radius
 @param pbg_r2          Background external radius
 @return   CPL_ERROR_NONE or the relevant CPL error code on error

*/
/*----------------------------------------------------------------------------*/
static cpl_error_code
sph_strehl_read_parlist(const cpl_parameterlist * self,
                        const char * armname,
                        const char * recipe,
                        const double star_r_default,
                        const double bg_r1_default,
                        const double bg_r2_default,
                        double * pstar_r,
                        double * pbg_r1,
                        double * pbg_r2){

    cpl_ensure_code(self    != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(armname != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(recipe  != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(pstar_r != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(pbg_r1  != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(pbg_r2  != NULL, CPL_ERROR_NULL_INPUT);

    if(sph_strehl_read_parlist_star_r(self, armname, recipe, pstar_r))
        return cpl_error_set_where(cpl_func);

    cpl_errorstate okstate = cpl_errorstate_get();

    *pbg_r1 = irplib_parameterlist_get_double(self, armname, recipe, "bg_r1");
    if (!cpl_errorstate_is_equal(okstate)) {
        *pbg_r1 = IRPLIB_STREHL_BACKGROUND_R1;
        cpl_msg_warning(cpl_func, "Recipe parameter 'bg_r1' missing, using def"
                        "ault: "IRPLIB_STRINGIFY(IRPLIB_STREHL_BACKGROUND_R1));
        cpl_errorstate_set(okstate);
    }

    *pbg_r2 = irplib_parameterlist_get_double(self, armname, recipe, "bg_r2");
    if (!cpl_errorstate_is_equal(okstate)) {
       *pbg_r2 = IRPLIB_STREHL_BACKGROUND_R2;
        cpl_msg_warning(cpl_func, "Recipe parameter 'bg_r2' missing, using def"
                        "ault: "IRPLIB_STRINGIFY(IRPLIB_STREHL_BACKGROUND_R2));
        cpl_errorstate_set(okstate);
    }

    *pstar_r = *pstar_r == 0.0 ? star_r_default : *pstar_r;
    *pbg_r1 = *pbg_r1 == 0.0 ? bg_r1_default : *pbg_r1;
    *pbg_r2 = *pbg_r2 == 0.0 ? bg_r2_default : *pbg_r2;

    return CPL_ERROR_NONE;

}

/*----------------------------------------------------------------------------*/
/**
 @brief Compute the Strehl ratio of the brightest object in the image
 @param arm          The SPHERE arm (IRDIS or ZIMPOL)
 @param image        Image to process
 @param rawplist     The propertylist of the input raw frame
 @param label_lambda The column label of the filter lambda
 @param label_width  The column label of the filter width
 @param filterframe  An input calibration frame with the filter wavelengths
 @param armname      The name of the arm, "ird" or "zpl"
 @param recipe       Recipe name (for parameter access)
 @param parlist      Recipe parameters
 @param qc_out_pars  The output qc parameters. Undefined in case of error.
 @return   CPL_ERROR_NONE or the relevant CPL error code on error

 */
/*----------------------------------------------------------------------------*/
static cpl_error_code
sph_strehl_compute_core(const sph_arm             arm,
                        const cpl_image         * image,
                        const cpl_propertylist  * rawplist,
                        const char              * label_lambda,
                        const char              * label_width,
                        const cpl_frame         * filterframe,
                        const char              * armname,
                        const char              * recipe,
                        const cpl_parameterlist * parlist,
                        sph_strehl_qc_pars      * qc_out_pars)
{

    /* From IRPLIB: Determined empirically by C. Lidman for Strehl
     * error computation */
    const double strehl_error_coefficient = CPL_MATH_PI * 0.007 / 0.0271;

    const double star_radius_AO_enabled_default = 2.0;
    const double star_bg_radius_AO_enabled_internal_default = 2.0;
    const double star_bg_radius_AO_enabled_external_default = 3.0;

    double star_r = 0.0, bg_r1 = 0.0, bg_r2 = 0.0;
    double lam = 0.0, dlam = 0.0, pscale = 0.0, pos_x = 0.0, pos_y = 0.0;
    double sigma = 0.0;
    const char * key    = NULL;
    const char * filter = NULL;

    hdrl_parameter   * hdrl_param;
    hdrl_strehl_result hdrl_strehl;
    cpl_image        * strehl_img;
    hdrl_image       * hdrl_img;
    double             mx_r, star_radius;
    cpl_size           ipos_x, ipos_y;

    cpl_ensure_code(image       != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(rawplist    != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(filterframe != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(armname     != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(recipe      != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(parlist     != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(qc_out_pars != NULL, CPL_ERROR_NULL_INPUT);

    if (sph_strehl_read_parlist(parlist, armname, recipe, star_radius_AO_enabled_default,
            star_bg_radius_AO_enabled_internal_default,
            star_bg_radius_AO_enabled_external_default,
            &star_r, &bg_r1, &bg_r2))
        return cpl_error_set_where(cpl_func);

    if (sph_strehl_read_filter_and_scale(rawplist, arm, &pscale, &key, &filter))
        return cpl_error_set_where(cpl_func);

    if (sph_strehl_read_filter(filterframe, label_lambda, label_width, key,
                               filter, &lam, &dlam))
        return cpl_error_set_where(cpl_func);

    if (sph_strehl_find_star(image, &pos_x, &pos_y, &sigma))
        return cpl_error_set_where(cpl_func);

#ifdef SPH_STREHL_CHECK_AGAINST_IRPLIB

    double strehl, strehl_err, star_bg, star_peak, star_flux, psf_peak, psf_flux, bg_noise;

    cpl_image * imagecast = cpl_image_cast(image, CPL_TYPE_FLOAT);

    code = irplib_strehl_compute(imagecast,
                                 IRPLIB_STREHL_M1, IRPLIB_STREHL_M2, lam, dlam,
                                 pscale, IRPLIB_STREHL_BOX_SIZE, pos_x, pos_y,
                                 star_r, bg_r1, bg_r2,
                                 -1, -1,
                                 &strehl,
                                 &strehl_err,
                                 &star_bg,
                                 &star_peak,
                                 &star_flux,
                                 &psf_peak,
                                 &psf_flux,
                                 &bg_noise);

    cpl_msg_info(cpl_func, "IRPLIB - Strehl ratio %c: %g, err=%g, (x,y)=(%g,%g), "
                 "sigma=%g, star flux/peak/background: %g/%g/%g, bg-noise=%g",
                 idx, strehl, strehl_err, pos_x, pos_y, sigma,
                 star_flux, star_peak, star_bg, bg_noise);

    cpl_image_delete(imagecast);

#endif

    /* Radius of star w. area of interest - converted to pixels */
    mx_r = CPL_MAX(CPL_MAX(star_r, bg_r1), bg_r2);
    star_radius = mx_r / pscale;
    ipos_x = CPL_MAX((cpl_size)(pos_x - star_radius), 1); /* Round down */
    ipos_y = CPL_MAX((cpl_size)(pos_y - star_radius), 1);

    strehl_img = cpl_image_extract(image, ipos_x, ipos_y,
                                   CPL_MIN((cpl_size)ceil(pos_x + star_radius),
                                           cpl_image_get_size_x(image)),
                                   CPL_MIN((cpl_size)ceil(pos_y + star_radius),
                                           cpl_image_get_size_y(image)));

    hdrl_img = hdrl_image_create(strehl_img, NULL);
    cpl_image_delete(strehl_img);

    /* HDRL uses SI (m) and the mirror dimensions are given as radii */
    hdrl_param =
        hdrl_strehl_parameter_create(lam * 1e-6,
                                     IRPLIB_STREHL_M1 * 0.5,
                                     IRPLIB_STREHL_M2 * 0.5,
                                     pscale, pscale,
                                     star_r, bg_r1, bg_r2);

    if (hdrl_param == NULL) {
        hdrl_image_delete(hdrl_img);
        return cpl_error_get_code() ? cpl_error_set_where(cpl_func)
            : cpl_error_set(cpl_func, CPL_ERROR_UNSPECIFIED);
    } else if (cpl_error_get_code()) {
        hdrl_image_delete(hdrl_img);
        hdrl_parameter_delete(hdrl_param);
        return cpl_error_set_where(cpl_func);
    } else {

        hdrl_strehl = hdrl_strehl_compute(hdrl_img, hdrl_param);

        hdrl_image_delete(hdrl_img);
        hdrl_parameter_delete(hdrl_param);

        if (cpl_error_get_code()) {
            return cpl_error_set_where(cpl_func);
        } else {

            const double sph_bkg_noise = hdrl_strehl.computed_background_error
                * sqrt((double)hdrl_strehl.nbackground_pixels);
            const double star_radius_pix = star_r / pscale;
            const double sph_strehl_error = strehl_error_coefficient *
                sph_bkg_noise * pscale * star_radius_pix
                * star_radius_pix / hdrl_strehl.star_flux.data;

            /* If ipos_x/y is 1, then 1 must be subtracted */
            const double hdrl_pos_x = hdrl_strehl.star_x + (double)(ipos_x - 1);
            const double hdrl_pos_y = hdrl_strehl.star_y + (double)(ipos_y - 1);

            *qc_out_pars = (sph_strehl_qc_pars){
                hdrl_strehl.strehl_value.data, sph_strehl_error,
                hdrl_pos_x, hdrl_pos_y,
                sigma, hdrl_strehl.star_flux.data,
                hdrl_strehl.star_peak.data, hdrl_strehl.star_background.data,
                sph_bkg_noise
            };
        }
    }

    return CPL_ERROR_NONE;
}

static double get_radius_internal_default_pix_disabled_ao(const sph_arm arm){
    if(arm == SPH_ARM_IRDIS)
        return 277.0; /*For IRDIS*/

    return 500.0; /*For ZIMPOL*/
}

static cpl_error_code
sph_strehl_get_star_flux_ao_disabled(const cpl_image * im,
        const cpl_parameterlist * parlist, const cpl_propertylist * proplist,
        const char * arm_name, const sph_arm arm, const char * rec_name,
        double * flx, double *bg){

    double radius_internal = 0.0;
    double radius_external = 0.0;
    double radius_star = 0.0;
    double scale = 0.0;
    *flx = 0.0;

    if(sph_strehl_read_scale(proplist, arm, &scale))
        return cpl_error_set_where(cpl_func);

    const double radius_internal_default_pix = get_radius_internal_default_pix_disabled_ao(arm);
    const double radius_external_default_pix =
            CPL_MIN(cpl_image_get_size_x(im), cpl_image_get_size_y(im)) / 2.0;


    const double star_radius_AO_disabled_default = radius_internal_default_pix * scale;
    const double star_bg_radius_AO_disabled_internal_default = radius_internal_default_pix * scale;
    const double star_bg_radius_AO_disabled_external_default = radius_external_default_pix * scale;


    if(sph_strehl_read_parlist(parlist, arm_name, rec_name,
            star_radius_AO_disabled_default,
            star_bg_radius_AO_disabled_internal_default,
            star_bg_radius_AO_disabled_external_default,
            &radius_star, &radius_internal, &radius_external))
        return cpl_error_set_where(cpl_func);


    cpl_ensure_code(radius_internal < radius_external, CPL_ERROR_ILLEGAL_INPUT);

    radius_internal /= scale;
    radius_external /= scale;
    radius_star /= scale;

    double pos_x = 0, pos_y = 0, sigma = 0;
    if(sph_strehl_find_star(im, &pos_x, &pos_y, &sigma)){
        return cpl_error_set_where(cpl_func);
    }

    /* Measure the background in the candidate image */
    const double star_bg = irplib_strehl_ring_background(im, pos_x, pos_y,
                                             radius_internal, radius_external,
                                             IRPLIB_BG_METHOD_AVER_REJ);
    if (cpl_error_get_code()) {
        return cpl_error_set_where(cpl_func);
    }

    /* Measure the flux on the candidate image */
    *flx = irplib_strehl_disk_flux(im, pos_x, pos_y, radius_star, star_bg);
    *bg = star_bg;

    return cpl_error_get_code();
}

static cpl_boolean
sph_strehl_is_neutral_density(const char * str){
    if(str == NULL) return CPL_FALSE;

    return strncmp(str, "ND", 2) == 0;
}

/*----------------------------------------------------------------------------*/
/**
@brief Insert the QC parameters into the propertylist for IRDIS cameras
@param plist_left   Propertylist to append the left QC parameters to
@param plist_right  Propertylist to append the right QC parameters to
@param img_left     Image to process from camera left
@param img_right    Image to process from camera right
@param raw_frame    The input raw frame
@param filter_frame An input calibration frame with the filter wavelengths
@param recipe       Recipe name (for parameter access)
@param parlist         Struct containing the QC pars
@return CPL_ERROR_NONE or the relevant CPL error code on error
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
sph_strehl_irdis_and_append(cpl_propertylist  * plist_left,
                            cpl_propertylist  * plist_right,
                            const cpl_image   * img_left,
                            const cpl_image   * img_right,
                            const cpl_frame   * raw_frame,
                            const cpl_frame   * filter_frame,
                            const char        * recipe,
                            cpl_parameterlist * parlist,
                            double            * flux_left,
                            double            * flux_right){

    sph_strehl_qc_pars pars_left = {0, 0, 0, 0, 0, 0, 0, 0, 0};
    sph_strehl_qc_pars pars_right= {0, 0, 0, 0, 0, 0, 0, 0, 0};

    if(sph_strehl_irdis_left(img_left, raw_frame, filter_frame,
            recipe, parlist, &pars_left))
        return cpl_error_set_where(cpl_func);

    if(sph_strehl_irdis_right(img_right, raw_frame, filter_frame,
            recipe, parlist, &pars_right))
        return cpl_error_set_where(cpl_func);

    if(sph_strehl_fill_qc_pars_irdis_left(plist_left, &pars_left))
        return cpl_error_set_where(cpl_func);

    if(sph_strehl_fill_qc_pars_irdis_right(plist_right, &pars_right))
        return cpl_error_set_where(cpl_func);

    if(flux_left) *flux_left = pars_left.star_flux;
    if(flux_right) *flux_right = pars_right.star_flux;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
@brief Insert the QC parameters into the propertylist for IRDIS-LEFT CAMERA
@param self       Propertylist to append QC parameters to
@param pars       Struct containing the QC pars
@return   CPL_ERROR_NONE or the relevant CPL error code on error
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
sph_strehl_fill_qc_pars_irdis_left(cpl_propertylist * self,
                                     const sph_strehl_qc_pars * qc_pars){

    const char *  SPH_IRD_STREHL_QC_STREHL_1          = "ESO QC STREHL LEFT";
    const char *  SPH_IRD_STREHL_QC_STREHL_1_ERR      = "ESO QC STREHL LEFT ERROR";
    const char *  SPH_IRD_STREHL_QC_STREHL_1_POSX     = "ESO QC STREHL LEFT POSX";
    const char *  SPH_IRD_STREHL_QC_STREHL_1_POSY     = "ESO QC STREHL LEFT POSY";
    const char *  SPH_IRD_STREHL_QC_STREHL_1_SIGMA    = "ESO QC STREHL LEFT SIGMA";
    const char *  SPH_IRD_STREHL_QC_STREHL_1_FLUX     = "ESO QC STREHL LEFT FLUX";
    const char *  SPH_IRD_STREHL_QC_STREHL_1_PEAK     = "ESO QC STREHL LEFT PEAK";
    const char *  SPH_IRD_STREHL_QC_STREHL_1_BKG      = "ESO QC STREHL LEFT BACKGROUND";
    const char *  SPH_IRD_STREHL_QC_STREHL_1_BKGNOISE = "ESO QC STREHL LEFT BACKGROUND NOISE";

    return sph_strehl_fill_qc_pars(self, qc_pars, SPH_IRD_STREHL_QC_STREHL_1,
            SPH_IRD_STREHL_QC_STREHL_1_ERR,
            SPH_IRD_STREHL_QC_STREHL_1_POSX,
            SPH_IRD_STREHL_QC_STREHL_1_POSY,
            SPH_IRD_STREHL_QC_STREHL_1_SIGMA,
            SPH_IRD_STREHL_QC_STREHL_1_FLUX,
            SPH_IRD_STREHL_QC_STREHL_1_PEAK,
            SPH_IRD_STREHL_QC_STREHL_1_BKG,
            SPH_IRD_STREHL_QC_STREHL_1_BKGNOISE);
}

/*----------------------------------------------------------------------------*/
/**
@brief Insert the QC parameters into the propertylist for IRDIS-RIGHT CAMERA
@param self       Propertylist to append QC parameters to
@param pars       Struct containing the QC pars
@return   CPL_ERROR_NONE or the relevant CPL error code on error
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
sph_strehl_fill_qc_pars_irdis_right(cpl_propertylist * self,
                                      const sph_strehl_qc_pars * qc_pars){

    const char *  SPH_IRD_STREHL_QC_STREHL_2          = "ESO QC STREHL RIGHT";
    const char *  SPH_IRD_STREHL_QC_STREHL_2_ERR      = "ESO QC STREHL RIGHT ERROR";
    const char *  SPH_IRD_STREHL_QC_STREHL_2_POSX     = "ESO QC STREHL RIGHT POSX";
    const char *  SPH_IRD_STREHL_QC_STREHL_2_POSY     = "ESO QC STREHL RIGHT POSY";
    const char *  SPH_IRD_STREHL_QC_STREHL_2_SIGMA    = "ESO QC STREHL RIGHT SIGMA";
    const char *  SPH_IRD_STREHL_QC_STREHL_2_FLUX     = "ESO QC STREHL RIGHT FLUX";
    const char *  SPH_IRD_STREHL_QC_STREHL_2_PEAK     = "ESO QC STREHL RIGHT PEAK";
    const char *  SPH_IRD_STREHL_QC_STREHL_2_BKG      = "ESO QC STREHL RIGHT BACKGROUND";
    const char *  SPH_IRD_STREHL_QC_STREHL_2_BKGNOISE = "ESO QC STREHL RIGHT BACKGROUND NOISE";

    return sph_strehl_fill_qc_pars(self, qc_pars, SPH_IRD_STREHL_QC_STREHL_2,
            SPH_IRD_STREHL_QC_STREHL_2_ERR,
            SPH_IRD_STREHL_QC_STREHL_2_POSX,
            SPH_IRD_STREHL_QC_STREHL_2_POSY,
            SPH_IRD_STREHL_QC_STREHL_2_SIGMA,
            SPH_IRD_STREHL_QC_STREHL_2_FLUX,
            SPH_IRD_STREHL_QC_STREHL_2_PEAK,
            SPH_IRD_STREHL_QC_STREHL_2_BKG,
            SPH_IRD_STREHL_QC_STREHL_2_BKGNOISE);

}

const char *  SPH_IRD_STREHL_QC_STREHL_FLUX     = SPH_STREHL_QC_STREHL_FLUX;
const char *  SPH_IRD_STREHL_QC_STREHL_BKG      = SPH_STREHL_QC_STREHL_BKG;
/*----------------------------------------------------------------------------*/
/**
@brief Insert the QC parameters into the propertylist for ZIMPOL
@param self       Propertylist to append QC parameters to
@param pars       Struct containing the QC pars
@return   CPL_ERROR_NONE or the relevant CPL error code on error
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
sph_strehl_fill_qc_pars_zimpol(cpl_propertylist * self,
                               const sph_strehl_qc_pars * qc_pars){

    const char *  SPH_IRD_STREHL_QC_STREHL          =  SPH_STREHL_QC_STREHL;
    const char *  SPH_IRD_STREHL_QC_STREHL_ERR      =  SPH_STREHL_QC_STREHL_ERR;
    const char *  SPH_IRD_STREHL_QC_STREHL_POSX     =  SPH_STREHL_QC_STREHL_POSX;
    const char *  SPH_IRD_STREHL_QC_STREHL_POSY     =  SPH_STREHL_QC_STREHL_POSY;
    const char *  SPH_IRD_STREHL_QC_STREHL_SIGMA    =  SPH_STREHL_QC_STREHL_SIGMA;
    const char *  SPH_IRD_STREHL_QC_STREHL_PEAK     =  SPH_STREHL_QC_STREHL_PEAK;
    const char *  SPH_IRD_STREHL_QC_STREHL_BKGNOISE =  SPH_STREHL_QC_STREHL_BKGNOISE;

    return sph_strehl_fill_qc_pars(self, qc_pars, SPH_IRD_STREHL_QC_STREHL,
            SPH_IRD_STREHL_QC_STREHL_ERR,
            SPH_IRD_STREHL_QC_STREHL_POSX,
            SPH_IRD_STREHL_QC_STREHL_POSY,
            SPH_IRD_STREHL_QC_STREHL_SIGMA,
            SPH_IRD_STREHL_QC_STREHL_FLUX,
            SPH_IRD_STREHL_QC_STREHL_PEAK,
            SPH_IRD_STREHL_QC_STREHL_BKG,
            SPH_IRD_STREHL_QC_STREHL_BKGNOISE);
}

/*----------------------------------------------------------------------------*/
/**
@internal
@brief Insert the QC parameters into the propertylist
@param self       Propertylist to append QC parameters to
@param pars                 Struct containing the QC pars
@param key_strehl            QC parameter name
@param key_strehl_err        QC parameter name
@param key_pos_x            QC parameter name
@param key_pos_y            QC parameter name
@param key_sigma            QC parameter name
@param key_star_flux        QC parameter name
@param key_star_peak        QC parameter name
@param key_star_bg            QC parameter name
@param key_star_bg_noise    QC parameter name
@return   CPL_ERROR_NONE or the relevant CPL error code on error
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code
sph_strehl_fill_qc_pars(cpl_propertylist * self,
                        const sph_strehl_qc_pars * pars,
                        const char * key_strehl,
                        const char * key_strehl_err,
                        const char * key_pos_x,
                        const char * key_pos_y,
                        const char * key_sigma,
                        const char * key_star_flux,
                        const char * key_star_peak,
                        const char * key_star_bg,
                        const char * key_star_bg_noise)
{

    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(pars != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(key_strehl != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(key_strehl_err != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(key_pos_x != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(key_pos_y != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(key_sigma != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(key_star_flux != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(key_star_peak != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(key_star_bg != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(key_star_bg_noise != NULL, CPL_ERROR_NULL_INPUT);

    cpl_msg_info(cpl_func, "Strehl ratio %s: %g, err=%g, (x,y)=(%g,%g), "
                 "sigma=%g, star flux/peak/background: %g/%g/%g, bg-noise=%g",
                 key_strehl, pars->strehl, pars->strehl_error, pars->star_x,
                 pars->star_y, pars->sigma, pars->star_flux, pars->star_peak,
                 pars->background, pars->background_noise);

    if (cpl_propertylist_update_double(self, key_strehl, pars->strehl))
        return cpl_error_set_where(cpl_func);

    if (cpl_propertylist_update_double(self, key_strehl_err,
            pars->strehl_error))
        return cpl_error_set_where(cpl_func);

    if (cpl_propertylist_update_double(self, key_pos_x, pars->star_x))
        return cpl_error_set_where(cpl_func);

    if (cpl_propertylist_update_double(self, key_pos_y, pars->star_y))
        return cpl_error_set_where(cpl_func);

    if (cpl_propertylist_update_double(self, key_sigma, pars->sigma))
        return cpl_error_set_where(cpl_func);

    if(sph_strehl_fill_flux_bg_qc(self,  key_star_flux, pars->star_flux,
            key_star_bg, pars->background))
        return cpl_error_set_where(cpl_func);

    if (cpl_propertylist_update_double(self, key_star_peak, pars->star_peak))
        return cpl_error_set_where(cpl_func);

    if (cpl_propertylist_update_double(self, key_star_bg_noise,
            pars->background_noise))
        return cpl_error_set_where(cpl_func);

    return CPL_ERROR_NONE;
}


/*----------------------------------------------------------------------------*/
/**
@brief Insert the QC parameters into the propertylist for ZPL cameras
@param self       Propertylist to append QC parameters to
@param flux          The STREHL FLUX
@param background The STREHL BACKGROUND
@return   CPL_ERROR_NONE or the relevant CPL error code on error
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
sph_strehl_fill_qc_pars_zimpol_ao_disabled(cpl_propertylist * self,
                               const double flux,
                               const double background){

    return sph_strehl_fill_flux_bg_qc(self, SPH_IRD_STREHL_QC_STREHL_FLUX,
            flux, SPH_IRD_STREHL_QC_STREHL_BKG, background);
}

/*----------------------------------------------------------------------------*/
/**
 @internal
 @brief Insert the QC flux and background parameters into the propertylist
 @param self       Propertylist to append QC parameters to
 @param idx        The QC index ('1' for first)
 @param star_flux  The QC parameter
 @param star_bg    The QC parameter
 @return   CPL_ERROR_NONE or the relevant CPL error code on error
 */
/*----------------------------------------------------------------------------*/
static
cpl_error_code sph_strehl_fill_flux_bg_qc(cpl_propertylist * self,
        const char * key_star_flux, double star_flux,
        const char * key_star_bg, double star_bg){

    if (cpl_propertylist_update_double(self, key_star_bg, star_bg))
        return cpl_error_set_where(cpl_func);

    if (cpl_propertylist_update_double(self, key_star_flux, star_flux))
        return cpl_error_set_where(cpl_func);

    return CPL_ERROR_NONE;
}
