/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include "sph_triple_image.h"
#include "sph_smart_imagelist.h"
#include "sph_error.h"
#include "sph_common_keywords.h"
#include "sph_keyword_manager.h"
#include "sph_utils.h"

#include <cpl.h>
#include <math.h>
#include <strings.h>

/*-----------------------------------------------------------------------------
                            Private function prototypes
 -----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
                            Static variables
 -----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
                                Function code
 -----------------------------------------------------------------------------*/
sph_error_code SPH_TRIPLE_IMAGE_GENERAL         = SPH_TRIPLE_IMAGE_ERR_START + 0;
sph_error_code SPH_TRIPLE_IMAGE_BAD_ALGORITHM   = SPH_TRIPLE_IMAGE_ERR_START + 1;
sph_error_code SPH_TRIPLE_IMAGE_NO_ALGORITHM    = SPH_TRIPLE_IMAGE_ERR_START + 2;
sph_error_code SPH_TRIPLE_IMAGE_BAD_TYPE        = SPH_TRIPLE_IMAGE_ERR_START + 3;
sph_error_code SPH_TRIPLE_IMAGE_NO_TYPE         = SPH_TRIPLE_IMAGE_ERR_START + 4;


/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_triple_image  A structure to store triple image products
 *
 * This structure represents a data product with three values for each pixel.
 * It is used to store polarization data either (I, Q, U) stock parameter
 * or elements of the Mueller matrix
 * Apart from the normal value field, it contains, just like sph_master_frame,
 * the following extensions:
 * @par Descirption:
 * <ol>
 * <li> RMSMAP. A map of the same dimension and number of pixels as the
 *      image giving the inverse of the RMS of the image pixel</li>
 * <li> NCombmap. A map of the same dimension and number of pixels as the
 *      image giving the number of pixels that were used to obtain the image</li>
 * <li> Badpixelmap. A map of the same dimension and number of pixels as the
 *      image flagging pixels that were impossible to reduce from the raw frames
 *      either because all input raw pixels at that position were bad or because
 *      the statistics is not calculable.</li>
 * <li> Quality Control Parameter list. A list of quality control parameters.
 *      What quality parameters are listed depends on the recipe used to create
 *      the frame.
 * </ol>
 * Each of these extensions appear triple now, one for each of the three images.
 *
 *
 * @par Synopsis:
 * @code
 * typedef struct _sph_triple_image_
 * {
 *    sph_master_frame*         iframe;
 *    sph_master_frame*        qframe;
 *    sph_master_frame*        uframe;
 *    cpl_propertylist*        properties; //list of the keywords
 *    cpl_propertylist*        qclist; //list of the quality control parameters
 * } sph_triple_image;
 * @endcode
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Constructor function for sph_triple_image.
  @return   a pointer to the newly created sph_triple_image or NULL if
              unsuccessful.
  @note     A CPL error is raised in case there was a problem.

  Construct a new sph_triple_image
 */
/*----------------------------------------------------------------------------*/
sph_triple_image* sph_triple_image_new_empty(void) {
    sph_triple_image* spmi = cpl_calloc(1, sizeof(sph_triple_image));
    return spmi;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Constructor function for sph_triple_image.
  @param    nx        the pixel size in x
  @param    ny         the pixel size in y
  @return   a pointer to the newly created sph_triple_image or NULL if
              unsuccessful.
  @note     A CPL error is raised in case there was a problem.

  Construct a new sph_triple_image
 */
/*----------------------------------------------------------------------------*/
sph_triple_image* sph_triple_image_new( int nx, int ny ) {
    sph_triple_image* spmi;

    cpl_ensure(nx > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(ny > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);

    spmi = sph_triple_image_new_empty();

    spmi->qclist = NULL;
    spmi->properties = NULL;
    spmi->iframe = sph_master_frame_new(nx, ny);
    spmi->qframe = sph_master_frame_new(nx, ny);
    spmi->uframe = sph_master_frame_new(nx, ny);

    if ( spmi->iframe == NULL || spmi->qframe == NULL || spmi->uframe == NULL )
    {
        sph_error_raise( SPH_ERROR_MEMORY_FAIL, __FILE__,
                         __func__, __LINE__,
                         SPH_ERROR_ERROR, "Could not allocate triple image" );
        cpl_free(spmi);
        return NULL;
    }
    return spmi;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Constructor function for sph_triple_image.
  @return   a pointer to the newly created sph_triple_image or NULL if
              unsuccessful.
  @param    mf_i        a pointer to the first(i) master frame
  @param    mf_q        a pointer to the second(q) master frame
  @param    mf_u        a pointer to the third(u) master frame

  @note Input master frames are not duplicated, so don't delete them
  @note A CPL error is raised in case there was a problem.

  Construct a new sph_triple_image
 */
/*----------------------------------------------------------------------------*/
sph_triple_image*
sph_triple_image_new_from_master_frames( sph_master_frame* mf_i,
                                         sph_master_frame* mf_q,
                                         sph_master_frame* mf_u)
{
    sph_triple_image* spmi = NULL;

    if ( mf_i == NULL || mf_q == NULL || mf_q == NULL ){
        SPH_ERR("Could not create triple image: input master frame(s) has(have) null pointer(s).")
        return NULL;
    }
    spmi = cpl_calloc( 1, sizeof(sph_triple_image) );

    spmi->properties = NULL;
    spmi->qclist = NULL;
    spmi->iframe =  mf_i;
    spmi->qframe =  mf_q;
    spmi->uframe =  mf_u;

    return spmi;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Copy of the I/(X_IQ/X~_QQ) sph_master_frame from the given sph_triple_image.

  @param self   the sph_triple_image frame which I/(X_IQ/X~QQ) sph_master_frame is to
                retrieve from

  @return a pointer to the newly copied I/(X_IQ/X~QQ) sph_master_frame or NULL if
            unsuccessfully.

  @note     A CPL error is raised in case there was a problem.

  Create/return a new copied  sph_master_frame as copy of the I/(X_IQ/X~QQ)
  sph_master_frame  from the sph_triple_image.

 */
/*----------------------------------------------------------------------------*/
sph_master_frame*
sph_triple_image_get_copy_i_master_frame( sph_triple_image* self )
{
    return self->iframe ? sph_master_frame_duplicate(self->iframe) : NULL;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Copy of the Q/(X_QQ/X~_UQ) sph_master_frame from the given sph_triple_image.

  @param self   the sph_triple_image frame which Q/(X_QQ/X~UQ) sph_master_frame is to
                retrieve from

  @return a pointer to the newly copied Q/(X_QQ/X~UQ) sph_master_frame or NULL if
            unsuccessfully.

  @note     A CPL error is raised in case there was a problem.

  Create/return a new copied  sph_master_frame as copy of the U/(X_QQ/X~UQ)
  sph_master_frame  from the sph_triple_image.

 */
/*----------------------------------------------------------------------------*/
sph_master_frame*
sph_triple_image_get_copy_q_master_frame( sph_triple_image* self )
{
    return self->qframe ? sph_master_frame_duplicate(self->qframe) : NULL;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Copy of the U/(X_UQ/X~_VQ) sph_master_frame from the given sph_triple_image.

  @param self   the sph_triple_image frame which U/(X_UQ/X~VQ) sph_master_frame is to
                retrieve from

  @return a pointer to the newly copied U/(X_UQ/X~VQ) sph_master_frame or NULL if
            unsuccessfully.

  @note     A CPL error is raised in case there was a problem.

  Create/return a new copied  sph_master_frame as copy of the U/(X_UQ/X~VQ)
  sph_master_frame  from the sph_triple_image.

 */
/*----------------------------------------------------------------------------*/
sph_master_frame*
sph_triple_image_get_copy_u_master_frame( sph_triple_image* self )
{
    return self->uframe ? sph_master_frame_duplicate(self->uframe) : NULL;
}




/*----------------------------------------------------------------------------*/
/**
  @brief    Get iframe sph_master_frame from the given sph_triple_image.

  @param self   the sph_triple_image frame which iframe sph_master_frame is
                to retrieve from
  @return a pointer to the iframe  sph_master_frame or NULL if
            unsuccessfully.

  @note     A CPL error is raised in case there was a problem.

  Return an iframe sph_master_frame from the sph_triple_image without
  duplicate.
 */
/*----------------------------------------------------------------------------*/
sph_master_frame* sph_triple_image_get_iframe_master_frame( sph_triple_image* self ) {

    return self->iframe;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get x_iq (iframe) sph_master_frame from the given sph_triple_image.

  @param self   the sph_triple_image frame which x_iq sph_master_frame is
                to retrieve from
  @return a pointer to the iframe  sph_master_frame or NULL if
            unsuccessfully.

  @note     A CPL error is raised in case there was a problem.

  Return an x_iq(iframe) sph_master_frame from the sph_triple_image without
  duplicate. This function is only more convenient to use (name-meaning)
  when triple image keeps the elements of the "X" mueller matrix for the
  +Q/-Q observations.
 */
/*----------------------------------------------------------------------------*/
sph_master_frame* sph_triple_image_get_x_iq_master_frame( sph_triple_image* self ) {

    return self->iframe;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Get xt_qq (iframe) sph_master_frame from the given sph_triple_image.

  @param self   the sph_triple_image frame which xt_qq sph_master_frame is
                to retrieve from
  @return a pointer to the iframe  sph_master_frame or NULL if
            unsuccessfully.

  @note     A CPL error is raised in case there was a problem.

  Return an xt_qq(iframe) sph_master_frame from the sph_triple_image without
  duplicate.This function is only more convenient to use (name-meaning)
  when triple image keeps the elements of the "X" mueller matrix for the
  +U/-U observations.
 */
/*----------------------------------------------------------------------------*/
sph_master_frame* sph_triple_image_get_xt_qq_master_frame( sph_triple_image* self ) {
    return self->iframe;
}



/*----------------------------------------------------------------------------*/
/**
  @brief    Get qframe sph_master_frame from the given sph_triple_image.

  @param self   the sph_triple_image frame which qframe sph_master_frame is
                to retrieve from
  @return a pointer to the qframe  sph_master_frame or NULL if
            unsuccessfully.

  @note     A CPL error is raised in case there was a problem.

  Return an qframe sph_master_frame from the sph_triple_image without
  duplicate.
 */
/*----------------------------------------------------------------------------*/
sph_master_frame* sph_triple_image_get_qframe_master_frame( sph_triple_image* self ) {

    return self->qframe;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get x_qq (qframe) sph_master_frame from the given sph_triple_image.

  @param self   the sph_triple_image frame which x_qq sph_master_frame is
                to retrieve from
  @return a pointer to the qframe  sph_master_frame or NULL if
            unsuccessfully.

  @note     A CPL error is raised in case there was a problem.

  Return an x_qq(iframe) sph_master_frame from the sph_triple_image without
  duplicate. This function is only more convenient to use (name-meaning)
  when triple image keeps the elements of the "X" mueller matrix for the
  +Q/-Q observations.
*/
/*----------------------------------------------------------------------------*/
sph_master_frame* sph_triple_image_get_x_qq_master_frame( sph_triple_image* self ) {

    return self->qframe;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get xt_uq (qframe) sph_master_frame from the given sph_triple_image.

  @param self   the sph_triple_image frame which xt_qq sph_master_frame is
                to retrieve from
  @return a pointer to the iframe  sph_master_frame or NULL if
            unsuccessfully.

  @note     A CPL error is raised in case there was a problem.

  Return an xt_uq(iframe sph_master_frame from the sph_triple_image without
  duplicate.This function is only more convenient to use (name-meaning)
  when triple image keeps the elements of the "X~" mueller matrix for the
  +U/-U observations.
 */
/*----------------------------------------------------------------------------*/
sph_master_frame* sph_triple_image_get_xt_uq_master_frame( sph_triple_image* self ) {

    return self->qframe;
}



/*----------------------------------------------------------------------------*/
/**
  @brief    Get uframe sph_master_frame from the given sph_triple_image.

  @param self   the sph_triple_image frame which uframe sph_master_frame is
                to retrieve from
  @return a pointer to the uframe  sph_master_frame or NULL if
            unsuccessfully.

  @note     A CPL error is raised in case there was a problem.

  Return an uframe sph_master_frame from the sph_triple_image without
  duplicate.
 */
/*----------------------------------------------------------------------------*/
sph_master_frame* sph_triple_image_get_uframe_master_frame( sph_triple_image* self ) {
    return self->uframe;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get x_uq (uframe) sph_master_frame from the given sph_triple_image.

  @param self   the sph_triple_image frame which x_uq sph_master_frame is
                to retrieve from
  @return a pointer to the qframe  sph_master_frame or NULL if
            unsuccessfully.

  @note     A CPL error is raised in case there was a problem.

  Return an x_uq(qframe) sph_master_frame from the sph_triple_image without
  duplicate. This function is only more convenient to use (name-meaning)
  when triple image keeps the elements of the "X" mueller matrix for the
  +Q/-Q observations.
*/
/*----------------------------------------------------------------------------*/
sph_master_frame* sph_triple_image_get_x_uq_master_frame( sph_triple_image* self ) {

    return self->uframe;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get xt_vq (uframe) sph_master_frame from the given sph_triple_image.

  @param self   the sph_triple_image frame which xt_vq sph_master_frame is
                to retrieve from
  @return a pointer to the uframe  sph_master_frame or NULL if
            unsuccessfully.

  @note     A CPL error is raised in case there was a problem.

  Return an xt_vq (uframe) sph_master_frame from the sph_triple_image without
  duplicate.This function is only more convenient to use (name-meaning)
  when triple image keeps the elements of the "X~" mueller matrix for the
  +U/-U observations.
 */
/*----------------------------------------------------------------------------*/
sph_master_frame* sph_triple_image_get_xt_vq_master_frame( sph_triple_image* self ) {
    return self->uframe;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief        add a triple image from another triple image
 *
 * @param        self        the triple image to add to
 * @param        trimage        the triple image to add
 *
 * @return        error code of the operation
 *
 * This function adds triple image "trimage" to self. Self is changed
 * in place.
 *
 */
/*----------------------------------------------------------------------------*/

sph_error_code sph_triple_image_add_triple_image( sph_triple_image* self,
                                                  sph_triple_image* trimage){
    sph_error_code    rerr    = CPL_ERROR_NONE;

    rerr |= sph_master_frame_add_master_frame( self->iframe, trimage->iframe );
    rerr |= sph_master_frame_add_master_frame( self->qframe, trimage->qframe );
    rerr |= sph_master_frame_add_master_frame( self->uframe, trimage->uframe );


    if ( rerr != CPL_ERROR_NONE ) {
        SPH_ERR("add triple image is failed.")
    }

    return rerr ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief        subtract a triple image from another triple image
 *
 * @param        self        the triple image to subtract from
 * @param        trimage        the triple image to subtract
 *
 * @return        error code of the operation
 *
 * This function subtracts trimage image "trimage" from  self. Self is changed
 * in place.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_triple_image_subtract_triple_image( sph_triple_image* self,
                                                       sph_triple_image* trimage){
    sph_error_code    rerr    = CPL_ERROR_NONE;

    rerr |= sph_master_frame_subtract_master_frame( self->iframe, trimage->iframe );
    rerr |= sph_master_frame_subtract_master_frame( self->qframe, trimage->qframe );
    rerr |= sph_master_frame_subtract_master_frame( self->uframe, trimage->uframe );

    if ( rerr != CPL_ERROR_NONE ) {
        SPH_ERR("subtract triple image is failed.")
    }

    return rerr ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief        divide triple image by scalar double value
 *
 * @param        self        the triple image to divide by   (dividend)
 * @param        value        the triple value (divisor)
 *
 * @return        error code of the operation
 *
 * This function divides triple image from self by double "value".
 * Self is changed in place.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_triple_image_divide_double( sph_triple_image* self,
                                               double value){
    sph_error_code    rerr    = CPL_ERROR_NONE;

    rerr |= sph_master_frame_divide_double( self->iframe, value );
    rerr |= sph_master_frame_divide_double( self->qframe, value );
    rerr |= sph_master_frame_divide_double( self->uframe, value );

    if ( rerr != CPL_ERROR_NONE ) {
        SPH_ERR("division of the triple image by double value is failed.")
    }

    return rerr ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;
}



/*----------------------------------------------------------------------------*/
/**
  @brief    Perform a quality control on the triple image

  @param    self        the triple image
  @return   the error code
  @note     A CPL error is raised in case there was a problem.

  Performs a quality control calculation on the triple image. This means, that
  certain statistics are calculated and added to the qclist first to the each
  master frame of the quad image, and then put it into the main qclist of the

  Currently these statistics are: Mean, median and rms of the image.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_triple_image_quality_check( sph_triple_image* self ){
    int            rerr            = CPL_ERROR_NONE;

    //IFRAME
    rerr = sph_master_frame_quality_check( self->iframe );
    if ( rerr != CPL_ERROR_NONE ){
        SPH_ERR("cpl error is raised by calculating quality parameters for the "
                "triple->iframe master frame.")
        return     rerr;
    }
    if ( self->qclist == NULL ) {
        self->qclist = cpl_propertylist_new();
    }
    //Mean
    if ( cpl_propertylist_has( self->qclist, SPH_COMMON_KEYWORD_QC_MEAN_TRIPLEIMAGE_IFRAME ) ){
        rerr = cpl_propertylist_update_double( self->qclist, SPH_COMMON_KEYWORD_QC_MEAN_TRIPLEIMAGE_IFRAME,
                cpl_propertylist_get_double( self->iframe->qclist, SPH_COMMON_KEYWORD_QC_MEANMASTERFRAME ) );
    } else {
        rerr = cpl_propertylist_append_double( self->qclist, SPH_COMMON_KEYWORD_QC_MEAN_TRIPLEIMAGE_IFRAME,
                cpl_propertylist_get_double( self->iframe->qclist, SPH_COMMON_KEYWORD_QC_MEANMASTERFRAME ) );
    }
    if ( rerr != CPL_ERROR_NONE ){
        SPH_ERR("cpl error is raised by appending/updating QC MEAN of TRIPLE IMAGE IFRAME.")
        SPH_RAISE_CPL;
        return     rerr;
    }
    //Median
    if ( cpl_propertylist_has( self->qclist, SPH_COMMON_KEYWORD_QC_MEDIAN_TRIPLEIMAGE_IFRAME ) ){
        rerr = cpl_propertylist_update_double( self->qclist, SPH_COMMON_KEYWORD_QC_MEDIAN_TRIPLEIMAGE_IFRAME,
                cpl_propertylist_get_double( self->iframe->qclist, SPH_COMMON_KEYWORD_QC_MEDIANMASTERFRAME ) );
    } else {
        rerr = cpl_propertylist_append_double( self->qclist, SPH_COMMON_KEYWORD_QC_MEDIAN_TRIPLEIMAGE_IFRAME,
                cpl_propertylist_get_double( self->iframe->qclist, SPH_COMMON_KEYWORD_QC_MEDIANMASTERFRAME ) );
    }
    if ( rerr != CPL_ERROR_NONE ){
        SPH_ERR("cpl error is raised by appending/updating QC MEDIAN of TRIPLE IMAGE IFRAME.")
        SPH_RAISE_CPL;
        return     rerr;
    }
    //RMS
    if ( cpl_propertylist_has( self->qclist, SPH_COMMON_KEYWORD_QC_RMS_TRIPLEIMAGE_IFRAME ) ){
        rerr = cpl_propertylist_update_double( self->qclist, SPH_COMMON_KEYWORD_QC_RMS_TRIPLEIMAGE_IFRAME,
                cpl_propertylist_get_double( self->iframe->qclist, SPH_COMMON_KEYWORD_QC_RMSMASTERFRAME ) );
    } else {
        rerr = cpl_propertylist_append_double( self->qclist, SPH_COMMON_KEYWORD_QC_RMS_TRIPLEIMAGE_IFRAME,
                cpl_propertylist_get_double( self->iframe->qclist, SPH_COMMON_KEYWORD_QC_RMSMASTERFRAME ) );
    }
    if ( rerr != CPL_ERROR_NONE ){
        SPH_ERR("cpl error is raised by appending/updating QC RMS of TRIPLE IMAGE IFRAME.")
        SPH_RAISE_CPL;
        return     rerr;
    }

    //QFRAME
    rerr = sph_master_frame_quality_check( self->qframe );
    if ( rerr != CPL_ERROR_NONE ){
        SPH_ERR("cpl error is raised by calculating quality parameters for the "
                "triple->qframe master frame.")
        return     rerr;
    }
    //Mean
    if ( cpl_propertylist_has( self->qclist, SPH_COMMON_KEYWORD_QC_MEAN_TRIPLEIMAGE_QFRAME ) ){
        rerr = cpl_propertylist_update_double( self->qclist, SPH_COMMON_KEYWORD_QC_MEAN_TRIPLEIMAGE_QFRAME,
                cpl_propertylist_get_double( self->qframe->qclist, SPH_COMMON_KEYWORD_QC_MEANMASTERFRAME ) );
    } else {
        rerr = cpl_propertylist_append_double( self->qclist, SPH_COMMON_KEYWORD_QC_MEAN_TRIPLEIMAGE_QFRAME,
                cpl_propertylist_get_double( self->qframe->qclist, SPH_COMMON_KEYWORD_QC_MEANMASTERFRAME ) );
    }
    if ( rerr != CPL_ERROR_NONE ){
        SPH_ERR("cpl error is raised by appending/updating QC MEAN of TRIPLE IMAGE QFRAME.")
        SPH_RAISE_CPL;
        return     rerr;
    }
    //Median
    if ( cpl_propertylist_has( self->qclist, SPH_COMMON_KEYWORD_QC_MEDIAN_TRIPLEIMAGE_QFRAME ) ){
        rerr = cpl_propertylist_update_double( self->qclist, SPH_COMMON_KEYWORD_QC_MEDIAN_TRIPLEIMAGE_QFRAME,
                cpl_propertylist_get_double( self->qframe->qclist, SPH_COMMON_KEYWORD_QC_MEDIANMASTERFRAME ) );
    } else {
        rerr = cpl_propertylist_append_double( self->qclist, SPH_COMMON_KEYWORD_QC_MEDIAN_TRIPLEIMAGE_QFRAME,
                cpl_propertylist_get_double( self->qframe->qclist, SPH_COMMON_KEYWORD_QC_MEDIANMASTERFRAME ) );
    }
    if ( rerr != CPL_ERROR_NONE ){
        SPH_ERR("cpl error is raised by appending/updating QC MEDIAN of TRIPLE IMAGE QFRAME.")
        SPH_RAISE_CPL;
        return     rerr;
    }
    //RMS
    if ( cpl_propertylist_has( self->qclist, SPH_COMMON_KEYWORD_QC_RMS_TRIPLEIMAGE_QFRAME ) ){
        rerr = cpl_propertylist_update_double( self->qclist, SPH_COMMON_KEYWORD_QC_RMS_TRIPLEIMAGE_QFRAME,
                cpl_propertylist_get_double( self->qframe->qclist, SPH_COMMON_KEYWORD_QC_RMSMASTERFRAME ) );
    } else {
        rerr = cpl_propertylist_append_double( self->qclist, SPH_COMMON_KEYWORD_QC_RMS_TRIPLEIMAGE_QFRAME,
                cpl_propertylist_get_double( self->qframe->qclist, SPH_COMMON_KEYWORD_QC_RMSMASTERFRAME ) );
    }
    if ( rerr != CPL_ERROR_NONE ){
        SPH_ERR("cpl error is raised by appending/updating QC RMS of TRIPLE IMAGE QFRAME.")
        SPH_RAISE_CPL;
        return     rerr;
    }

    //UFRAME
    rerr = sph_master_frame_quality_check( self->uframe );
    if ( rerr != CPL_ERROR_NONE ){
        SPH_ERR("cpl error is raised by calculating quality parameters for the "
                "triple->uframe master frame.")
        return     rerr;
    }
    //Mean
    if ( cpl_propertylist_has( self->qclist, SPH_COMMON_KEYWORD_QC_MEAN_TRIPLEIMAGE_UFRAME ) ){
        rerr = cpl_propertylist_update_double( self->qclist, SPH_COMMON_KEYWORD_QC_MEAN_TRIPLEIMAGE_UFRAME,
                cpl_propertylist_get_double( self->uframe->qclist, SPH_COMMON_KEYWORD_QC_MEANMASTERFRAME ) );
    } else {
        rerr = cpl_propertylist_append_double( self->qclist, SPH_COMMON_KEYWORD_QC_MEAN_TRIPLEIMAGE_UFRAME,
                cpl_propertylist_get_double( self->uframe->qclist, SPH_COMMON_KEYWORD_QC_MEANMASTERFRAME ) );
    }
    if ( rerr != CPL_ERROR_NONE ){
        SPH_ERR("cpl error is raised by appending/updating QC MEAN of TRIPLE IMAGE UFRAME.")
        SPH_RAISE_CPL;
        return     rerr;
    }
    //Median
    if ( cpl_propertylist_has( self->qclist, SPH_COMMON_KEYWORD_QC_MEDIAN_TRIPLEIMAGE_UFRAME ) ){
        rerr = cpl_propertylist_update_double( self->qclist, SPH_COMMON_KEYWORD_QC_MEDIAN_TRIPLEIMAGE_UFRAME,
                cpl_propertylist_get_double( self->uframe->qclist, SPH_COMMON_KEYWORD_QC_MEDIANMASTERFRAME ) );
    } else {
        rerr = cpl_propertylist_append_double( self->qclist, SPH_COMMON_KEYWORD_QC_MEDIAN_TRIPLEIMAGE_UFRAME,
                cpl_propertylist_get_double( self->uframe->qclist, SPH_COMMON_KEYWORD_QC_MEDIANMASTERFRAME ) );
    }
    if ( rerr != CPL_ERROR_NONE ){
        SPH_ERR("cpl error is raised by appending/updating QC MEDIAN of TRIPLE IMAGE UFRAME.")
        SPH_RAISE_CPL;
        return     rerr;
    }
    //RMS
    if ( cpl_propertylist_has( self->qclist, SPH_COMMON_KEYWORD_QC_RMS_TRIPLEIMAGE_UFRAME ) ){
        rerr = cpl_propertylist_update_double( self->qclist, SPH_COMMON_KEYWORD_QC_RMS_TRIPLEIMAGE_UFRAME,
                cpl_propertylist_get_double( self->uframe->qclist, SPH_COMMON_KEYWORD_QC_RMSMASTERFRAME ) );
    } else {
        rerr = cpl_propertylist_append_double( self->qclist, SPH_COMMON_KEYWORD_QC_RMS_TRIPLEIMAGE_UFRAME,
                cpl_propertylist_get_double( self->uframe->qclist, SPH_COMMON_KEYWORD_QC_RMSMASTERFRAME ) );
    }
    if ( rerr != CPL_ERROR_NONE ){
        SPH_ERR("cpl error is raised by appending/updating QC RMS of TRIPLE IMAGE UFRAME.")
        SPH_RAISE_CPL;
        return     rerr;
    }





    return rerr;

}

/*----------------------------------------------------------------------------*/
/**
   @brief Copy constructor function for sph_triple_image

   @param    self   the double image to duplicate

   @return   a pointer to the newly created sph_triple_image or NULL if
            unsuccessfully.
   @note     A CPL error is raised in case there was a problem.

  Construct a new sph_triple_image as copy of the input triple image.
 */
/*----------------------------------------------------------------------------*/
sph_triple_image* sph_triple_image_duplicate( const sph_triple_image* self) {
    sph_triple_image*  new_tripleimage = NULL;

    if (self == NULL) {
        sph_error_raise( SPH_TRIPLE_IMAGE_GENERAL, __FILE__,
                                 __func__, __LINE__,
                                 SPH_ERROR_WARNING, "An initial triple image to copy has a null pointer."
                                         "Nothing to copy." );
        return NULL;
    }

    //create an empty triple image
    new_tripleimage = sph_triple_image_new_empty();
    if (new_tripleimage == NULL) {
           sph_error_raise( SPH_ERROR_MEMORY_FAIL, __FILE__,
                                 __func__, __LINE__,
                                 SPH_ERROR_ERROR, "Could not allocate memory for the triple image" );
                return NULL;
    }

    //duplicate properties and qclist if exists
    if ( self->properties != NULL ){
        new_tripleimage->properties = cpl_propertylist_duplicate( self->properties );
    }
    if ( self->qclist != NULL ){
        new_tripleimage->qclist = cpl_propertylist_duplicate( self->qclist );
    }

    //duplicate iframe, qframe and uframe  as single master frames
    //iframe
    if (self->iframe !=NULL){
        new_tripleimage->iframe = sph_master_frame_duplicate(self->iframe);
    } else {
        sph_error_raise( SPH_TRIPLE_IMAGE_GENERAL, __FILE__,
                                 __func__, __LINE__,
                                 SPH_ERROR_WARNING, "iframe from the initial triple image to copy has a null pointer." );
    }

    //qframe
    if (self->qframe != NULL){
        new_tripleimage->qframe = sph_master_frame_duplicate(self->qframe);
    } else {
        sph_error_raise( SPH_TRIPLE_IMAGE_GENERAL, __FILE__,
                                 __func__, __LINE__,
                                 SPH_ERROR_WARNING, "qframe from the initial triple image to copy has a null pointer." );
    }

    //uframe
    if (self->uframe != NULL){
        new_tripleimage->uframe = sph_master_frame_duplicate(self->uframe);
    } else {
        sph_error_raise( SPH_TRIPLE_IMAGE_GENERAL, __FILE__,
                                 __func__, __LINE__,
                                 SPH_ERROR_WARNING, "uframe from the initial triple image to copy has a null pointer." );
    }

    return new_tripleimage;
}




/*----------------------------------------------------------------------------*/
/**
 * @brief        Save the triple image
 *
 * @param        self        the triple_image to save
 * @param       czFilename  the filename to save under
 * @param        pl            a propertylist to append to main header (can be NULL)
 * @return        the error code
 *
 * Save the triple image as a FITS file. This saves the triple image as a FITS
 * file with 12 extensions, one each for the image, one for the badpixel map, one
 * for the number of pixels map and one for the rmsmap. First come all extension
 * for frame I, then for frame Q and U.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_triple_image_save( sph_triple_image* self,
                                      const char* czFilename,
                                      cpl_propertylist* pli )
{
    sph_error_code            rerr            = CPL_ERROR_NONE;
    cpl_propertylist*        mainpl            = NULL;
    cpl_propertylist*         extpl            = NULL;

    if ( !self ) {
        return cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
    }
    if ( !self->iframe || !self->qframe || !self->uframe ) {
        return cpl_error_set(cpl_func, CPL_ERROR_ILLEGAL_INPUT);
    }

    extpl = cpl_propertylist_new(); // not really used for the moment; it might be used later
    if ( pli )
    {
        mainpl = sph_keyword_manager_trans_keywords( pli );
    }
    else {
        mainpl = cpl_propertylist_new();
    }

    //add qclist created to the main property list
    if ( self->qclist ) {
        rerr = cpl_propertylist_append( mainpl, self->qclist );
    }

    rerr |= cpl_propertylist_update_string( mainpl, SPH_COMMON_KEYWORD_SPH_TYPE, SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_TRIPLE_IMAGE );

    /* Update the header if required */
    sph_utils_update_header(mainpl);

    rerr |= cpl_image_save( self->iframe->image, czFilename, CPL_TYPE_FLOAT, mainpl, CPL_IO_DEFAULT );
    rerr |= cpl_image_save( self->iframe->badpixelmap, czFilename, CPL_BPP_8_UNSIGNED, extpl, CPL_IO_EXTEND );
    rerr |= cpl_image_save( self->iframe->ncombmap, czFilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );
    rerr |= cpl_image_save( self->iframe->rmsmap, czFilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );
    rerr |= cpl_image_save( self->qframe->image, czFilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );
    rerr |= cpl_image_save( self->qframe->badpixelmap, czFilename, CPL_BPP_8_UNSIGNED, extpl, CPL_IO_EXTEND );
    rerr |= cpl_image_save( self->qframe->ncombmap, czFilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );
    rerr |= cpl_image_save( self->qframe->rmsmap, czFilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );
    rerr |= cpl_image_save( self->uframe->image, czFilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );
    rerr |= cpl_image_save( self->uframe->badpixelmap, czFilename, CPL_BPP_8_UNSIGNED, extpl, CPL_IO_EXTEND );
    rerr |= cpl_image_save( self->uframe->ncombmap, czFilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );
    rerr |= cpl_image_save( self->uframe->rmsmap, czFilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );

    cpl_propertylist_delete( mainpl );
    cpl_propertylist_delete( extpl );

    return rerr ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief        Update property list of the quad image DFS product
 *              by SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_QUAD_IMAGE
 *
 * @param       czFilename  the filename to load and save triple image with
 *                 correctly updated SPH_COMMON_KEYWORD_SPH_TYPE
 * @return        the error code
 *
 * @note This function is used as a final step to save quad image as a
 * DFS product (see function sph_quad_image_save_dfs). It updates
 * SPH_COMMON_KEYWORD_SPH_TYPE by SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_TRIPLE_IMAGE.
 * It is necessary because cpl_dsf_save_image updates the triple image property
 * list by the property list from the first input frames.
 *
 */
/*----------------------------------------------------------------------------*/
static sph_error_code sph_triple_image_update_prod_type( const char* czFilename )
{
    sph_error_code       rerr              = CPL_ERROR_NONE;
    sph_triple_image*    trimage            = NULL;
    cpl_propertylist*    pli               = NULL;
    cpl_propertylist*    pli_regexp        = NULL;
    int                    plane             = 0;

    trimage = sph_triple_image_load(  czFilename, plane);

    if ( !trimage ){
        SPH_ERR("triple image is not loaded.")
        return sph_error_get_last_code();
    }

    pli = cpl_propertylist_load_regexp( czFilename, 0, ".*QC.*", 1);
    pli_regexp = cpl_propertylist_new();
    rerr = cpl_propertylist_copy_property_regexp( pli_regexp, pli, ".*COMMENT.*", 1 );

    rerr = sph_triple_image_save( trimage, czFilename, pli_regexp);
    if (rerr != CPL_ERROR_NONE ) {
        SPH_ERR("Couldn't  save the current triple image")
    }
    sph_triple_image_delete( trimage );
    return rerr;

}


/*----------------------------------------------------------------------------*/
/**
 * @brief        Save the master frame as DFS product
 *
 * @param        self        the triple image to save
 * @param       czFilename  the filename to save under
 * @param       allframes    the frameset used to construct the product
 * @param       usedframes    the frameset used to construct the product
 * @param       template_frame  the frame to use as template for header
 * @param       param       the parameterlist used for product
 * @param       tag         the tag of the product
 * @param       recipe      the recipe creating this product
 * @param       pipename    the pipeline creating this product
 * @param        pl            a propertylist to append to main header (can be NULL)
 * @return        the error code
 *
 * Save the master frame as a FITS file according to ESO DFS.
 * This saves the master frame as a FITS file with 16 extensions (4 master frames ).
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_triple_image_save_dfs(sph_triple_image* self,
                                         const char* outfilename,
                                         cpl_frameset* allframes,
                                         cpl_frame* template_frame,
                                         cpl_parameterlist* params,
                                         const char* tag,
                                         const char* recipe,
                                         const char* pipename,
                                         cpl_propertylist* pli)
{
    sph_error_code            rerr            = CPL_ERROR_NONE;
    cpl_propertylist*        mainpl            = NULL;
    cpl_propertylist*         extpl            = NULL;

    if ( !self ) {
        return cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
    }

    if ( !self->iframe || !self->qframe || !self->uframe ) {
        return cpl_error_set(cpl_func, CPL_ERROR_ILLEGAL_INPUT);
    }

    extpl = cpl_propertylist_new(); // not really used for the moment; it might be used later

    if ( pli ) {
        mainpl = sph_keyword_manager_trans_keywords( pli );
    }
    else {
        mainpl = cpl_propertylist_new();
        SPH_INFO_MSG("No property list presents, a new one is created")
    }

    if ( cpl_propertylist_has( mainpl, SPH_COMMON_KEYWORD_SPH_TYPE )){
        rerr = cpl_propertylist_update_string( mainpl, SPH_COMMON_KEYWORD_SPH_TYPE, SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_TRIPLE_IMAGE );
    } else {
        rerr = cpl_propertylist_append_string( mainpl, SPH_COMMON_KEYWORD_SPH_TYPE, SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_TRIPLE_IMAGE  );
    }

    /*
     * Attention: SPH_COMMON_KEYWORD_SPH_TYPE will be changed by cpl_dfs_save_image function.
     * Unfortunately this keyword will be later updated automatically in cpl_dfs_save_image.
     * In order to avoid this problem  a function, called sph_triple_image_update_prod_type is used
     * as soon as the dfs product is created (see below)
     *
     */

    //add qclist created to the main property list
    if ( self->qclist ) {
        rerr = cpl_propertylist_append( mainpl, self->qclist );
        SPH_INFO_MSG("QC parameter list was added to the property list")
    }

    if ( rerr != CPL_ERROR_NONE ){
        return cpl_error_set_where(cpl_func);
    }
    rerr = cpl_propertylist_update_string( mainpl, SPH_COMMON_KEYWORD_PRO_CATG, tag);

    //printf( "SPH_COMMON_KEYWORD_SPH_TYPE = %s \n", SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_TRIPLE_IMAGE);
    sph_utils_remove_wcs_3d(mainpl);
    /* FIXME: Set WCS to dummy (pixel) value for now */
    sph_utils_reset_wcs_12d(mainpl);

    /* Make any final header updates */
    sph_utils_update_header(mainpl);

    rerr |= cpl_dfs_save_image( allframes, NULL, params,
            allframes,
            template_frame,
            self->iframe->image,
            CPL_TYPE_FLOAT, recipe, mainpl, ".*ESO DRS PC PROD TYPE.*", pipename, outfilename );

    if ( rerr == CPL_ERROR_NONE ){
        //first image has already been already saved in cpl_dfs_save_image
        //rerr = cpl_image_save( self->iframe->image, outfilename, CPL_TYPE_FLOAT, mainpl, CPL_IO_DEFAULT );
        rerr |= cpl_image_save( self->iframe->badpixelmap, outfilename, CPL_BPP_8_UNSIGNED, extpl, CPL_IO_EXTEND );
        rerr |= cpl_image_save( self->iframe->ncombmap, outfilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );
        rerr |= cpl_image_save( self->iframe->rmsmap, outfilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );
        rerr |= cpl_image_save( self->qframe->image, outfilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );
        rerr |= cpl_image_save( self->qframe->badpixelmap, outfilename, CPL_BPP_8_UNSIGNED, extpl, CPL_IO_EXTEND );
        rerr |= cpl_image_save( self->qframe->ncombmap, outfilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );
        rerr |= cpl_image_save( self->qframe->rmsmap, outfilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );
        rerr |= cpl_image_save( self->uframe->image, outfilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );
        rerr |= cpl_image_save( self->uframe->badpixelmap, outfilename, CPL_BPP_8_UNSIGNED, extpl, CPL_IO_EXTEND );
        rerr |= cpl_image_save( self->uframe->ncombmap, outfilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );
        rerr |= cpl_image_save( self->uframe->rmsmap, outfilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );

    }

    cpl_propertylist_delete( mainpl );
    cpl_propertylist_delete( extpl );

    // Attention: Update of the created dfs quad image product by the
    // correct SPH_COMMON_KEYWORD_SPH_TYPE  keyword
    rerr |= sph_triple_image_update_prod_type( outfilename );

    return rerr ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;

}



/*----------------------------------------------------------------------------*/
/**
 * @brief load a triple image from a file
 *
 * @param    filename        the filename to load from
 * @param     plane            the plane to laod from ( 0..Nplanes)
 *
 * This function creates a new triple image and loads the data from a FITS
 * file with at least one plane and at least 12 extensions (the first 12
 * extensions are read). This function does not check the format is actually
 * a sph_triple_image.
 */
/*----------------------------------------------------------------------------*/
sph_triple_image*
sph_triple_image_load( const char* czFilename, int plane ) {
    sph_triple_image*        result        = NULL;
    cpl_propertylist*        pl_eso        = NULL;
    cpl_error_code            rerr        = CPL_ERROR_NONE;


    result = sph_triple_image_new_empty();

    if ( !result ) {
        return NULL;
    }

    pl_eso = cpl_propertylist_load_regexp( czFilename, 0, ".*ESO.*", 0 );

    if ( pl_eso != NULL ){
        //initializing and loading properties, qclist for the newly created quad image
        result->properties = cpl_propertylist_new();
        rerr = cpl_propertylist_copy_property_regexp( result->properties, pl_eso, ".*QC.*", 1 );
        if ( rerr != CPL_ERROR_NONE){
            sph_error_raise( SPH_TRIPLE_IMAGE_GENERAL,
                __FILE__, __func__, __LINE__,
                SPH_ERROR_WARNING,  "Copy regexp properties for the triple image is failed" );
            rerr = CPL_ERROR_NONE;
            cpl_error_reset();
            cpl_propertylist_delete( result->properties ); result->properties = NULL;
        }
        result->qclist = cpl_propertylist_new();
        rerr = cpl_propertylist_copy_property_regexp( result->qclist, pl_eso, ".*QC.*", 0 );
        if ( rerr != CPL_ERROR_NONE){
            sph_error_raise( SPH_TRIPLE_IMAGE_GENERAL,
                __FILE__, __func__, __LINE__,
                SPH_ERROR_WARNING,  "Copy regexp qclist for the triple image is failed" );
            rerr = CPL_ERROR_NONE;
            cpl_error_reset();
            cpl_propertylist_delete( result->qclist ); result->qclist = NULL;
        }
        cpl_propertylist_delete( pl_eso );
    } else {
        SPH_RAISE_CPL;
    }

    result->iframe = sph_master_frame_new_empty();
    result->qframe = sph_master_frame_new_empty();
    result->uframe = sph_master_frame_new_empty();

    if ( !result->iframe || !result->qframe || !result->uframe ) {
        sph_triple_image_delete( result );
        return NULL;
    }

    //iframe
    result->iframe->image = cpl_image_load( czFilename, CPL_TYPE_DOUBLE, plane, 0 );
    if ( !result->iframe->image ) {
        sph_triple_image_delete( result );
        return NULL;
    }
    result->iframe->badpixelmap = cpl_image_load( czFilename, CPL_TYPE_INT, plane, 1 );
    if ( !result->iframe->badpixelmap ) {
        sph_triple_image_delete( result );
        return NULL;
    }
    result->iframe->ncombmap = cpl_image_load( czFilename, CPL_TYPE_FLOAT, plane, 2 );
    if ( !result->iframe->ncombmap ) {
        sph_triple_image_delete( result );
        return NULL;
    }
    result->iframe->rmsmap = cpl_image_load( czFilename, CPL_TYPE_FLOAT, plane, 3 );
    if ( !result->iframe->rmsmap ) {
        sph_triple_image_delete( result );
        return NULL;
    }

    //qframe
    result->qframe->image = cpl_image_load( czFilename, CPL_TYPE_DOUBLE, plane, 4 );
    if ( !result->qframe->image ) {
        sph_triple_image_delete( result );
        return NULL;
    }
    result->qframe->badpixelmap = cpl_image_load( czFilename, CPL_TYPE_INT, plane, 5 );
    if ( !result->qframe->badpixelmap ) {
        sph_triple_image_delete( result );
        return NULL;
    }
    result->qframe->ncombmap = cpl_image_load( czFilename, CPL_TYPE_FLOAT, plane, 6 );
    if ( !result->qframe->ncombmap ) {
        sph_triple_image_delete( result );
        return NULL;
    }
    result->qframe->rmsmap = cpl_image_load( czFilename, CPL_TYPE_FLOAT, plane, 7 );
    if ( !result->qframe->rmsmap ) {
        sph_triple_image_delete( result );
        return NULL;
    }

    //uframe
    result->uframe->image = cpl_image_load( czFilename, CPL_TYPE_DOUBLE, plane, 8 );
    if ( !result->uframe->image ) {
        sph_triple_image_delete( result );
        return NULL;
    }
    result->uframe->badpixelmap = cpl_image_load( czFilename, CPL_TYPE_INT, plane, 9 );
    if ( !result->uframe->badpixelmap ) {
        sph_triple_image_delete( result );
        return NULL;
    }
    result->uframe->ncombmap = cpl_image_load( czFilename, CPL_TYPE_FLOAT, plane, 10 );
    if ( !result->uframe->ncombmap ) {
        sph_triple_image_delete( result );
        return NULL;
    }
    result->uframe->rmsmap = cpl_image_load( czFilename, CPL_TYPE_FLOAT, plane, 11 );
    if ( !result->uframe->rmsmap ) {
        sph_triple_image_delete( result );
        return NULL;
    }

    return result;


}
/*----------------------------------------------------------------------------*/
/**
  @brief    delete the sph_triple_image and release resources
  @param    self The triple image to delete

 */
/*----------------------------------------------------------------------------*/
void sph_triple_image_delete(sph_triple_image* self) {
    if (self) {
        sph_master_frame_delete(self->iframe);
        sph_master_frame_delete(self->qframe);
        sph_master_frame_delete(self->uframe);
        cpl_propertylist_delete(self->properties);
        cpl_propertylist_delete(self->qclist);
        cpl_free(self);
    }
}

/**@}*/
