/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */
#include "sph_test_irdis_model.h"
#include "sph_ird_keywords.h"
#include "sph_test_pupilimage_creator.h"
#include "strings.h"
#include "sph_pixel_description_table.h"
#include "sph_filemanager.h"
#include "sph_test_ngc_ir_simulator.h"
#include <stdlib.h>
/*----------------------------------------------------------------------------*/
/**
 @brief Applies the IRDIS optical path to an image

 @param self    the irdis instrument model to use

 @return a new image of the image processed

 Description:
 The input image, which must be either of the subwindow size or a dimension 2
 larger in both directions is processed into a left and right
 image and the full detector image is returned.
 A vlaue of NULL is returned if the image does not have the correct size

 */
/*----------------------------------------------------------------------------*/
static cpl_image*
sph_test_irdis_model_apply_opticalpath(sph_test_irdis_model* self,
        cpl_image* animage) {
    cpl_image* resamp = NULL;
    cpl_image* result = NULL;
    if (!self) {
        return NULL;
    }

    if (cpl_image_get_size_x(animage) == 2 * self->nx
            && cpl_image_get_size_y(animage) == 2 * self->ny) {
        result = cpl_image_new(self->det_pixels_x, self->det_pixels_y,
                cpl_image_get_type(animage));
        resamp = cpl_image_rebin(animage, 1, 1, 2, 2);
    } else if (cpl_image_get_size_x(animage) == self->nx
            && cpl_image_get_size_y(animage) == self->ny) {
        result = cpl_image_new(self->det_pixels_x, self->det_pixels_y,
                cpl_image_get_type(animage));
        resamp = cpl_image_duplicate(animage);
    } else {
        SPH_ERR("Bad size");
        return NULL;
    }
    cpl_image_copy(result, resamp, 1, self->win_miny + 1);
    cpl_image_copy(result, resamp, self->win_minx + 1, self->win_miny + 1);
    cpl_image_delete(resamp);
    resamp = NULL;

    return result;
}
static cpl_image*
sph_test_irdis_model_set_image_left(sph_test_irdis_model* self,
        cpl_image* bigim, cpl_image* animage) {
    cpl_image* resamp = NULL;
    if (!self) {
        return NULL;
    }

    if (cpl_image_get_size_x(animage) == 2 * self->nx
            && cpl_image_get_size_y(animage) == 2 * self->ny) {
        resamp = cpl_image_rebin(animage, 1, 1, 2, 2);
    } else if (cpl_image_get_size_x(animage) == self->nx
            && cpl_image_get_size_y(animage) == self->ny) {
        resamp = cpl_image_duplicate(animage);
    } else {
        SPH_ERR("Bad size");
        return NULL;
    }
    cpl_image_copy(bigim, resamp, 1, self->win_miny + 1);
    cpl_image_delete(resamp);
    resamp = NULL;

    return bigim;
}
static cpl_image*
sph_test_irdis_model_set_image_right(sph_test_irdis_model* self,
        cpl_image* bigim, cpl_image* animage) {
    cpl_image* resamp = NULL;
    if (!self) {
        return NULL;
    }

    if (cpl_image_get_size_x(animage) == 2 * self->nx
            && cpl_image_get_size_y(animage) == 2 * self->ny) {
        resamp = cpl_image_rebin(animage, 1, 1, 2, 2);
    } else if (cpl_image_get_size_x(animage) == self->nx
            && cpl_image_get_size_y(animage) == self->ny) {
        resamp = cpl_image_duplicate(animage);
    } else {
        SPH_ERR("Bad size");
        return NULL;
    }
    cpl_image_copy(bigim, resamp, self->win_minx + 1, self->win_miny + 1);
    cpl_image_delete(resamp);
    resamp = NULL;

    return bigim;
}
static cpl_image*
sph_test_irdis_model_process_wav(sph_test_irdis_model* self, cpl_image* impupil) {
    cpl_image* result = NULL;
    sph_pixel_description_table* pdt = NULL;
    sph_pixel_descriptor* descr = NULL;
    int xx = 0;
    int yy = 0;
    double y0 = 0;
    double val = 0.0;
    int bpix = 0;
    if (!self) {
        return NULL;
    }
    cpl_test_error(CPL_ERROR_NONE);
    pdt = sph_test_irdis_model_create_pdt_single_window(self);
    result = cpl_image_new(pdt->nx, pdt->ny, CPL_TYPE_DOUBLE);
    for (xx = 0; xx < pdt->nx; ++xx) {
        for (yy = 0; yy < pdt->ny; ++yy) {
            descr = sph_pixel_description_table_get_descriptor(pdt, xx, yy);
            if (descr) {
                if (descr->specid) {
                    y0 = (descr->wavelength - self->spec_minlambda)
                            / (self->spec_maxlambda - self->spec_minlambda);
                    if (y0 < 0.0)
                        y0 = y0 + 0.00001;
                    if (y0 > 1.0 || y0 < 0.0) {
                        SPH_ERROR_RAISE_WARNING( SPH_ERROR_GENERAL,
                                "There is something wrong "
                                "with the spectra ranges: %f.", y0);
                        val = 0.0;
                    } else {
                        y0 = y0 * (cpl_image_get_size_y(impupil) - 1);
                        val = cpl_image_get(impupil, xx + 1, (int) y0 + 1,
                                &bpix);
                    }
                } else
                    val = 0.0;
                cpl_image_set(result, xx + 1, yy + 1, val);
            } else
                cpl_image_set(result, xx + 1, yy + 1, 0.0);
        }
    }
    cpl_test_error(CPL_ERROR_NONE);
    sph_pixel_description_table_delete(pdt);
    pdt = NULL;
    return result;
}
/*----------------------------------------------------------------------------*/
/**
 @brief Apply the circular DBI mask

 @param the instrument model
 @param the (double) image to apply the mask to

 @return none

 Description:
 Applies a DBI mask to both the left and right part of a dual image.
 Use this function after a call to apply_opticalpath

 */
/*----------------------------------------------------------------------------*/
void sph_test_irdis_model_apply_dbi_mask(sph_test_irdis_model* self,
        cpl_image* animage) {
    int xx = 0;
    int yy = 0;
    double lsquare = 0.0;
    double att = 0.0;
    double value = 0.0;
    int bpix = 0;
    cpl_image* dupimage = NULL;
    if (!self) {
        return;
    }

    dupimage = cpl_image_duplicate(animage);

    for (xx = 0; xx < cpl_image_get_size_x(animage); ++xx) {
        for (yy = 0; yy < cpl_image_get_size_y(animage); ++yy) {
            att = 0.0;
            lsquare = (xx - self->centre_left_x) * (xx - self->centre_left_x)
                    + (yy - self->centre_left_y) * (yy - self->centre_left_y);
            if (lsquare < self->radius_left * self->radius_left)
                att = 1.0;
            lsquare = (xx - self->centre_right_x) * (xx - self->centre_right_x)
                    + (yy - self->centre_right_y) * (yy - self->centre_right_y);
            if (lsquare < self->radius_right * self->radius_right)
                att = 1.0;
            value = cpl_image_get(dupimage, xx + 1, yy + 1, &bpix);
            cpl_image_set(animage, xx + 1, yy + 1, value * att);
        }
    }
    cpl_image_delete(dupimage);
    dupimage = NULL;
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Apply the circular CI mask

 @param the instrument model
 @param the (double) image to apply the mask to

 @return none

 Description:
 Applies a CI mask to bot the left and right part of a dual image.
 Use this function after a call to apply_opticalpath

 */
/*----------------------------------------------------------------------------*/
void sph_test_irdis_model_apply_ci_mask(sph_test_irdis_model* self,
        cpl_image* animage) {
    int xx = 0;
    int yy = 0;
    double lsquare = 0.0;
    double att = 0.0;
    double value = 0.0;
    int bpix = 0;
    cpl_image* dupimage = NULL;
    if (!self) {
        return;
    }
    if (!self) {
        return;
    }

    dupimage = cpl_image_duplicate(animage);

    for (xx = 0; xx < cpl_image_get_size_x(animage); ++xx) {
        for (yy = 0; yy < cpl_image_get_size_y(animage); ++yy) {
            att = 0.0;
            lsquare = (xx - self->centre_left_x) * (xx - self->centre_left_x)
                    + (yy - self->centre_left_y) * (yy - self->centre_left_y);
            if (lsquare < self->radius_left * self->radius_left)
                att = 1.0;
            lsquare = (xx - self->centre_right_x) * (xx - self->centre_right_x)
                    + (yy - self->centre_right_y) * (yy - self->centre_right_y);
            if (lsquare < self->radius_right * self->radius_right)
                att = 1.0;
            value = cpl_image_get(dupimage, xx + 1, yy + 1, &bpix);
            cpl_image_set(animage, xx + 1, yy + 1, value * att);
            if (att == 0.0) {
                cpl_image_reject(animage, xx + 1, yy + 1);
            }
        }
    }
    cpl_image_delete(dupimage);
    dupimage = NULL;
    return;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new default model
 *
 *
 * @return pointer to the new default model
 *
 */
/*----------------------------------------------------------------------------*/
sph_test_irdis_model*
sph_test_irdis_model_new(sph_test_ngc_ir_simulator* ngc) {
    sph_test_irdis_model* result = NULL;

    result = cpl_calloc(1, sizeof(sph_test_irdis_model));
    result->insmode = SPH_TEST_IRDIS_INSMODE_CI;
    result->det_pixels_x = ngc->det_size_x;
    result->det_pixels_y = ngc->det_size_y;
    result->nx = ngc->det_size_x / 2; // this is the size of the
    result->ny = ngc->det_size_x / 2; // sub-windows not the full detector!
    result->win_miny = 0;
    result->win_minx = result->nx;
    result->centre_left_x = result->ny / 2;
    result->centre_left_y = result->ny / 2 + result->win_miny;
    result->centre_right_x = result->win_minx + result->nx / 2;
    result->centre_right_y = result->ny / 2 + result->win_miny;
    result->radius_left = result->nx / 2 - 2;
    result->radius_right = result->ny / 2 - 2;
    result->specwin_minx = 20;
    result->specwin_miny = 63;
    result->specwin_maxx = result->nx - 20;
    result->specwin_maxy = result->ny - 63;
    result->insmode = SPH_TEST_IRDIS_INSMODE_CI;
    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new small flat model
 *
 *
 * The model created has all sigmas set to zero, linearity of 1, no wavelength
 * flat dependence, no dead, hot or crazy pixels, read out noise of zero and
 * 2 common filters, 2 dual filters and 2 nd filters. Common and dual filters
 * have wavelengths of 1.0 and 1.7 micrometers, and nd filters a transparancy
 * of 0.8 and 0.5.
 *
 * @return pointer to the new model
 *
 */
/*----------------------------------------------------------------------------*/
sph_test_irdis_model*
sph_test_irdis_model_new_constvalue(sph_test_ngc_ir_simulator* ngc) {
    sph_test_irdis_model* result = sph_test_irdis_model_new(ngc);

    result->spec_dl_over_dx = 0.0;
    result->spec_minlambda = 0.95;
    result->spec_maxlambda = 2.5;
    result->insmode = SPH_TEST_IRDIS_INSMODE_CI;
//    result->common_filters_names[0] = SPH_IRD_KEYWORD_VALUE_COMMON_FILTER_NAME1;
//    result->common_filters_names[1] = SPH_IRD_KEYWORD_VALUE_COMMON_FILTER_NAME2;
//    result->dual_filters_names[0] = SPH_IRD_KEYWORD_VALUE_DUAL_FILTER_NAME1;
//    result->dual_filters_names[1] = SPH_IRD_KEYWORD_VALUE_DUAL_FILTER_NAME2;
//    result->nd_filters_names[0] = SPH_IRD_KEYWORD_VALUE_ND_FILTER_NAME1;
//    result->nd_filters_names[1] = SPH_IRD_KEYWORD_VALUE_ND_FILTER_NAME2;
    return result;
}

void sph_test_irdis_model_set_subwindow_size(sph_test_irdis_model* self, int nx,
        int ny) {
    self->nx = nx;
    self->ny = ny;
}

void sph_test_irdis_model_set_window_pos(sph_test_irdis_model* self, int minx,
        int miny) {
    self->win_minx = minx;
    self->win_miny = miny;
}
void sph_test_irdis_model_set_left_illumcircle_centre(
        sph_test_irdis_model* self, double x, double y) {
    self->centre_left_x = x;
    self->centre_left_y = y;
}

void sph_test_irdis_model_set_left_illumcircle_radius(
        sph_test_irdis_model* self, double rad) {
    self->radius_left = rad;
}

void sph_test_irdis_model_set_right_illumcircle_centre(
        sph_test_irdis_model* self, double x, double y) {
    self->centre_right_x = x;
    self->centre_right_y = y;
}

void sph_test_irdis_model_set_right_illumcircle_radius(
        sph_test_irdis_model* self, double rad) {
    self->radius_right = rad;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Processes a single image
 * @param self      the irdis model
 * @param impupil      the image to process
 *
 * @return the resulting image or NULL on error
 *
 * The functions processes the input image, applying opitcal paths and masks
 * depending on the mode. In all imaging modes the input image must have the size
 * of a quarter of the detector (e.g. 1024x1024) for the spectral modes it must
 * have the same size as the spectral window with the y axis being the wavelength
 * direction.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_image*
sph_test_irdis_model_process_image(sph_test_irdis_model* self,
        cpl_image* impupil) {
    cpl_image* imexit = NULL;

    imexit = sph_test_irdis_model_process_image_left_right(self, impupil, NULL);
    return imexit;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Processes a single image
 * @param self      the irdis model
 * @param impupil      the image to get left from
 *
 * @return the resulting image or NULL on error
 *
 * The functions returns the left side of the input image only.
 */
/*----------------------------------------------------------------------------*/
cpl_image*
sph_test_irdis_model_get_image_left(sph_test_irdis_model* self, cpl_image* inim) {
    cpl_image* imexit = NULL;

    imexit = cpl_image_extract(inim, 1, self->win_miny + 1, self->win_minx,
            self->win_miny + self->ny);

    return imexit;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Processes a single image
 * @param self      the irdis model
 * @param impupil      the image to get left from
 *
 * @return the resulting image or NULL on error
 *
 * The functions returns the left side of the input image only.
 */
/*----------------------------------------------------------------------------*/
cpl_image*
sph_test_irdis_model_get_image_right(sph_test_irdis_model* self,
        cpl_image* inim) {
    cpl_image* imexit = NULL;

    imexit = cpl_image_extract(inim, self->nx + 1, self->win_miny + 1,
            self->nx * 2, self->win_miny + self->ny);

    return imexit;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Processes a single image
 * @param self      the irdis model
 * @param impupil      the image to process left
 * @param impupil      the right to process left (may be null)
 *
 * @return the resulting image or NULL on error
 *
 * The functions processes the input image, applying opitcal paths and masks
 * depending on the mode. In all imaging modes the input image must have the size
 * of a quarter of the detector (e.g. 1024x1024) for the spectral modes it must
 * have the same size as the spectral window with the y axis being the wavelength
 * direction.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_image*
sph_test_irdis_model_process_image_left_right(sph_test_irdis_model* self,
        cpl_image* impupil, cpl_image* imright) {
    cpl_image* imexit = NULL;
    cpl_image* dumim = NULL;

    if (self->insmode != SPH_TEST_IRDIS_INSMODE_SPEC_LRS
            && self->insmode != SPH_TEST_IRDIS_INSMODE_SPEC_MRS) {
        if (imright == NULL) {
            imexit = sph_test_irdis_model_apply_opticalpath(self, impupil);
        } else {
            imexit = cpl_image_new(self->det_pixels_x, self->det_pixels_y,
                    CPL_TYPE_DOUBLE);
            sph_test_irdis_model_set_image_left(self, imexit, impupil);
            sph_test_irdis_model_set_image_right(self, imexit, imright);
        }
        if (imexit) {
            if (self->insmode == SPH_TEST_IRDIS_INSMODE_CI)
                sph_test_irdis_model_apply_ci_mask(self, imexit);
            else if (self->insmode == SPH_TEST_IRDIS_INSMODE_DBI)
                sph_test_irdis_model_apply_ci_mask(self, imexit);
        }
    } else {
        dumim = sph_test_irdis_model_process_wav(self, impupil);
        if (!dumim) {
            SPH_ERR("Could not process wav");
        }
        SPH_RAISE_CPL_RESET;
        imexit = sph_test_irdis_model_apply_opticalpath(self, dumim);
        if (!imexit) {
            SPH_ERR("Could not apply optical path");
        }
        SPH_RAISE_CPL_RESET;
        cpl_image_delete(dumim);
        dumim = NULL;
    }
    return imexit;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Procees the frames to apply the optical path and masks

 @param instrument model
 @param frameset of single (one side only) images to process

 @return nothing

 Description:
 Proceses all input frames to apply the optical path of IRDIS (creating two
 images) and masking the correct modes. This function should be used to turn
 an input image into a realistic image just before it "hits" the detector.
 The input frames are modified to be set to a new filename, with the ending
 .digested
 */
/*----------------------------------------------------------------------------*/
void sph_test_irdis_model_digest_frames(sph_test_irdis_model* self,
                                        cpl_frameset* frames) {
    if (!self)
        return;
    if (!frames)
        return;
    self->frameset = frames;
    for (int ii = 0; ii < cpl_frameset_get_size(self->frameset); ++ii) {
        cpl_frame* frame = cpl_frameset_get_position(self->frameset, ii);
        if (frame) {
            if (strcasecmp(cpl_frame_get_tag(frame),
                           SPH_TEST_PUPILIMAGE_TAG) == 0) {
                const char * oldname    = cpl_frame_get_filename(frame);
                cpl_propertylist* pl    = cpl_propertylist_load(oldname, 0);
                cpl_image*      impupil = cpl_image_load(oldname,
                                                         CPL_TYPE_DOUBLE, 0, 0);
                if (impupil) {
                    char* filename = sph_filemanager_get_tmp_filename
                        ("irdis_model_digest.fits");
                    cpl_image* imexit = sph_test_irdis_model_process_image
                        (self, impupil);

                    cpl_image_save(imexit, filename, CPL_TYPE_DOUBLE, pl,
                                   CPL_IO_DEFAULT);
                    cpl_frame_set_filename(frame, filename);

                    cpl_image_delete(imexit);
                    cpl_image_delete(impupil);
                    cpl_free(filename);
                }
            }
        }
    }
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Crete the linear response (flat) as an image
 * @param self the ngc_simulator
 * @param ndit the number of dits
 * @param exptime the exposure time in seconds
 *
 * @return image or NULL
 *
 * Returns the detector flat as an image. Specifically, this
 * means that the linear coefficient of the response function
 * is returned as a function of pixel coordinate (i.e. as an image)
 * The ndit and exptime parameters are currently ignored.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_image*
sph_test_irdis_model_create_instrument_flat_image(sph_test_irdis_model* self,
        int ndit, double exptime) {
    cpl_image* image = cpl_image_new(self->nx, self->ny, CPL_TYPE_DOUBLE);
    int xx = 0;
    int yy = 0;

    if (ndit > 0 && exptime > 0.0) {
        for (xx = 0; xx < self->nx; ++xx) {
            for (yy = 0; yy < self->ny; ++yy) {
                cpl_image_set(image, xx + 1, yy + 1,
                              cpl_vector_get_mean(self->nd_filter_attenuation));
            }
        }
    }

    //cpl_image_set( image, 100, 100, 10000.0 );
    sph_test_irdis_model_apply_opticalpath(self, image);
    sph_test_irdis_model_apply_ci_mask(self, image);
    return image;
}
/*----------------------------------------------------------------------------*/
/**
 @brief Sets the common filter

 @param filteid        the ID of the filter
 @param filtername    a string of the name of the filter
 @param lambdas        a vector with the wavelengths of the transmission curve
 @param transmiss   a vector with the transmissions of the transmission curve
 @return nothing

 Description:
 Sets the filter curve for the common filter.
 Note that the passed vectors now get
 managed by the sph_test_irdis_model instance and should not be deleted.
 */
/*----------------------------------------------------------------------------*/
void sph_test_irdis_model_set_common_filter(sph_test_irdis_model* self,
        int filterid, const char* filtername, cpl_vector* lambdas,
        cpl_vector* transmissions) {
    self->common_filter_lambdas = lambdas;
    self->common_filter_name = filtername;
    self->common_filter_transmission = transmissions;
    self->current_common_filterid = filterid;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Sets the common filter

 @param filteid        the ID of the filter
 @param filtername    a string of the name of the filter
 @param lambdas        a vector with the wavelengths of the transmission curve
 @param transmiss   a vector with the transmissions of the transmission curve
 @return nothing

 Description:
 Sets the filter curve for the common filter.
 Note that the passed vectors now get
 managed by the sph_test_irdis_model instance and should not be deleted.
 */
/*----------------------------------------------------------------------------*/
void sph_test_irdis_model_set_left_filter(sph_test_irdis_model* self,
        int filterid, const char* filtername, cpl_vector* lambdas,
        cpl_vector* transmissions) {
    self->dual_filter_lambdas_left = lambdas;
    self->dual_filter_name_left = filtername;
    self->dual_filter_transmission_left = transmissions;
    self->current_dual_filterid_left = filterid;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Sets the common filter

 @param filteid        the ID of the filter
 @param filtername    a string of the name of the filter
 @param lambdas        a vector with the wavelengths of the transmission curve
 @param transmiss   a vector with the transmissions of the transmission curve
 @return nothing

 Description:
 Sets the filter curve for the common filter.
 Note that the passed vectors now get
 managed by the sph_test_irdis_model instance and should not be deleted.
 */
/*----------------------------------------------------------------------------*/
void sph_test_irdis_model_set_right_filter(sph_test_irdis_model* self,
        int filterid, const char* filtername, cpl_vector* lambdas,
        cpl_vector* transmissions) {
    self->dual_filter_lambdas_right = lambdas;
    self->dual_filter_name_right = filtername;
    self->dual_filter_transmission_right = transmissions;
    self->current_dual_filterid_right = filterid;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Sets the common filter

 @param filteid        the ID of the filter
 @param filtername    a string of the name of the filter
 @param lambdas        a vector with the wavelengths of the transmission curve
 @param transmiss   a vector with the transmissions of the transmission curve
 @return nothing

 Description:
 Sets the filter curve for the common filter.
 Note that the passed vectors now get
 managed by the sph_test_irdis_model instance and should not be deleted.
 */
/*----------------------------------------------------------------------------*/
void sph_test_irdis_model_set_nd_filter(sph_test_irdis_model* self,
        int filterid, const char* filtername, cpl_vector* lambdas,
        cpl_vector* transmissions) {
    self->nd_filter_lambdas = lambdas;
    self->nd_filter_name = filtername;
    self->nd_filter_attenuation = transmissions;
    self->current_nd_filterid = filterid;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Set the mode of IRDIS

 @param self the irdis model
 @param mode the mode to set to

 Description:
 Set the mode of IRDIS.

 */
/*----------------------------------------------------------------------------*/

void sph_test_irdis_model_set_mode(sph_test_irdis_model* self,
        sph_test_irdis_insmode mode) {
    self->insmode = mode;
    if (mode == SPH_TEST_IRDIS_INSMODE_SPEC_LRS) {
        self->spec_dl_over_dx = 0.0;
        self->spec_minlambda = 0.95;
        self->spec_maxlambda = 2.5;
        // The strange numbers make it 20 and 63 and 898 pixels
        // for 1024 windows.
        // and correspondingly smaller for 128 windows.
        self->specwin_minx = (int) self->nx * 0.0195312 + 1;
        self->specwin_miny = (int) self->ny * 0.06152343 + 1;
        self->specwin_maxx = self->specwin_minx + (int) self->nx * 0.87695312
                + 1;
        self->specwin_maxy = self->ny - self->specwin_miny;
    }
    if (mode == SPH_TEST_IRDIS_INSMODE_SPEC_MRS) {
        self->spec_dl_over_dx = 0.0;
        self->spec_minlambda = 0.95;
        self->spec_maxlambda = 2.5;
        self->specwin_minx = (int) self->nx * 0.0195312 + 1;
        self->specwin_miny = (int) self->ny * 0.06152343 + 1;
        self->specwin_maxx = self->specwin_minx + (int) self->nx * 0.876953125
                + 1;
        self->specwin_maxy = self->ny - self->specwin_miny;
    }
}
/*----------------------------------------------------------------------------*/
/**
 @brief Create a PDT with the size of a subwindow.

 @param self        the irdis model

 @return a new PDT or NULL in case of error

 Description:
 This calculates a new model PDT for one side of the optical path.
 It has non-zero values only in the spectral window as defined by
 the model.
 */
/*----------------------------------------------------------------------------*/
sph_pixel_description_table*
sph_test_irdis_model_create_pdt_single_window(sph_test_irdis_model* self) {
    sph_pixel_description_table* pdt = NULL;
    sph_pixel_descriptor* descr = NULL;
    double lambda = 0.0;
    double lam0 = 0.0;
    double dl = 0.0;
    cpl_image* labims = NULL;
    int xx = 0;
    int yy = 0;

    dl = self->spec_maxlambda - self->spec_minlambda;
    dl = dl / (self->specwin_maxy - self->specwin_miny);

    labims = cpl_image_new(self->nx, self->ny, CPL_TYPE_INT);

    for (xx = self->specwin_minx; xx <= self->specwin_maxx; ++xx) {
        for (yy = self->specwin_miny; yy <= self->specwin_maxy; ++yy) {
            cpl_image_set(labims, xx + 1, yy + 1, 1);
        }
    }
    cpl_image_save(labims, "labims.fits", CPL_TYPE_FLOAT, NULL, CPL_IO_DEFAULT);
    pdt = sph_pixel_description_table_new_from_labels(labims, 0.0, 0.0);
    for (yy = self->specwin_miny; yy <= self->specwin_maxy; ++yy) {
        lam0 = dl * (yy - self->specwin_miny) + self->spec_minlambda;
        for (xx = self->specwin_minx; xx <= self->specwin_maxx; ++xx) {
            descr = sph_pixel_description_table_get_descriptor(pdt, xx, yy);
            if (descr->specid == 1) {
                lambda = lam0
                        + (xx - self->specwin_minx) * self->spec_dl_over_dx;
                descr->wavelength = lambda;
                descr->dlambda = dl;
                descr->ddlambda = 0.0;
                descr->illumination = 1.0;
                descr->lensid = 1;
                descr->value = 0.0;
            }
        }
    }
    cpl_image_delete(labims);
    labims = NULL;
    return pdt;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Create a PDT for the whole detector.

 @param self        the irdis model

 @return a new PDT or NULL in case of error

 Description:
 This calculates a new model PDT for the entire detector
 */
/*----------------------------------------------------------------------------*/
sph_pixel_description_table*
sph_test_irdis_model_create_pdt(sph_test_irdis_model* self) {
    sph_pixel_description_table* pdt = NULL;
    sph_pixel_descriptor* descr = NULL;
    double lambda = 0.0;
    double lam0 = 0.0;
    double dl = 0.0;
    cpl_image* labims = NULL;
    int xx = 0;
    int yy = 0;

    dl = self->spec_maxlambda - self->spec_minlambda;
    dl = dl / (self->specwin_maxy - self->specwin_miny);

    labims = cpl_image_new(self->det_pixels_x, self->det_pixels_y,
            CPL_TYPE_INT);

    for (xx = self->specwin_minx; xx <= self->specwin_maxx; ++xx) {
        for (yy = self->specwin_miny + self->win_miny;
                yy <= self->specwin_maxy + self->win_miny; ++yy) {
            cpl_image_set(labims, xx + 1, yy + 1, 1);
        }
    }
    for (xx = self->specwin_minx + self->win_minx;
            xx <= self->specwin_maxx + self->win_minx; ++xx) {
        for (yy = self->specwin_miny + self->win_miny;
                yy <= self->specwin_maxy + self->win_miny; ++yy) {
            cpl_image_set(labims, xx + 1, yy + 1, 2);
        }
    }
    cpl_image_save(labims, "labims.fits", CPL_TYPE_FLOAT, NULL, CPL_IO_DEFAULT);
    pdt = sph_pixel_description_table_new_from_labels(labims, 0.0, 0.0);
    for (yy = 0; yy < pdt->ny; ++yy) {
        lam0 = dl * (yy - self->specwin_miny - self->win_miny)
                + self->spec_minlambda;
        for (xx = 0; xx < pdt->nx; ++xx) {
            descr = sph_pixel_description_table_get_descriptor(pdt, xx, yy);
            if (descr->specid == 1) {
                lambda = lam0
                        + (xx - self->specwin_minx) * self->spec_dl_over_dx;
            }
            if (descr->specid == 2) {
                lambda = lam0
                        + (xx - self->specwin_minx - self->win_minx)
                                * self->spec_dl_over_dx;
            }
            if (descr->specid == 1 || descr->specid == 2) {
                descr->wavelength = lambda;
                descr->dlambda = dl;
                descr->ddlambda = 0.0;
                descr->illumination = 1.0;
                descr->lensid = descr->specid;
                descr->value = 0.0;
            }
        }
    }
    cpl_image_delete(labims);
    labims = NULL;
    return pdt;
}

void sph_test_irdis_model_delete(sph_test_irdis_model* self) {
    if (self) {
        cpl_free(self);
    }
}

