/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#include "sph_test_ngc_ir_simulator.h"
#include "sph_test_image_tools.h"
#include "sph_ird_tags.h"
#include "sph_ird_keywords.h"

#include "sph_common_keywords.h"
#include "sph_keyword_manager.h"
#include "sph_master_frame.h"
#include "sph_framecombination.h"
#include "sph_test.h"
#include "sph_fits.h"
#include <gsl/gsl_rng.h>
#include <gsl/gsl_randist.h>
#include <time.h>
#include <math.h>

static
cpl_frame*
sph_test_ngc_ir_simulator_create_raw_cube_frame__(
        const char* namein, const char* tag, int ii);
static
void
sph_test_ngc_ir_simulator_save_im_in_fitscube__(
        cpl_image* im,
        cpl_frame* frame,
        cpl_propertylist* pl,
        int pp);
static
int
sph_test_ngc_ir_simulator_advance_dithering__(
        sph_test_ngc_ir_simulator* self );

static
void
sph_test_ngc_ir_simulator_digest_image__(
        sph_test_ngc_ir_simulator* self,
        cpl_image* animage );

static
void
sph_test_ngc_ir_simulator_image_add_bias_and_dark__( sph_test_ngc_ir_simulator* self, cpl_image* animage, double exptime );

static
void
sph_test_ngc_ir_simulator_image_apply_flat__( sph_test_ngc_ir_simulator* self, cpl_image* animage );

static
void
sph_test_ngc_ir_simulator_image_add_noise__( sph_test_ngc_ir_simulator* self, cpl_image* animage );

static
char* sph_test_ngc_ir_simulator_create_raw_cube_filename__(const char*, int ii);

/*----------------------------------------------------------------------------*/
/**
 * @brief Creates a new ngc_simulator factory
 *
 * @param nx        detector pixel size in x
 * @param ny        detector pixel size in y
 *
 * @return the new NGC simulator
 *
 * The mode is set to a default value. All other values/flags are set to zero/
 * false. It also initializes a random number generator from gsl as gsl_rng_taus
 * with seed 1.
 *
 * NOTE: the seed is always the same !!
 * Change it with gsl_rng_set( result->pRNG, seed)
 */
/*----------------------------------------------------------------------------*/
sph_test_ngc_ir_simulator*
sph_test_ngc_ir_simulator_new(int nx, int ny) {
    sph_test_ngc_ir_simulator* result = NULL;

    result = cpl_calloc(1, sizeof(sph_test_ngc_ir_simulator));
    result->add_ron_noise = 0;
    result->pRNG = gsl_rng_alloc(gsl_rng_taus);
    gsl_rng_set(result->pRNG, 1);
    result->det_linear_factor = 1.0;
    result->det_size_x = nx;
    result->det_size_y = ny;
    result->lambda_ramp_linear = 1.0;
    result->mean_dark = 1.0;
    result->mean_bias = 50.0;
    result->win_minx = 0;
    result->win_maxx = result->det_size_x;
    result->win_miny = 0;
    result->win_maxy = result->det_size_y;
    result->exptime = 1.3;
    result->timedelay = -1;
    result->waveflat_fwhm = 2.0;
    result->waveflat_norm = 1.4;
    result->waveflat_meanl = 1.5;
    result->waveflat_factor = 1.0;
    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new set of raw dark frames
 * @param self          the ngc_simulator
 * @param inframe       the input frame
 * @param nframes       the number of frames
 * @param ndit          the number of DITS per frame
 * @param filename      the base filename (including .fits)
 * @param pl            propertylist to add (maybe NULL)
 * @param tag           tag of frames
 * @return pointer to a new frameset
 *
 * This creates a set of raw dark frames, taken with the given exposure time.
 * The function creates darks for all the currently set read out mode.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_frameset*
sph_test_ngc_ir_simulator_create_raw_frameset(sph_test_ngc_ir_simulator* self,
        cpl_frame* inframe, int nframes, int ndit, const char* base_filename,
        cpl_propertylist* pl, const char* tag) {
    cpl_frameset* result = NULL;
    cpl_frameset* inframeset = NULL;
    int ff = 0;
    inframeset = cpl_frameset_new();
    for (ff = 0; ff < nframes; ++ff) {
        cpl_frameset_insert(inframeset, cpl_frame_duplicate(inframe));
    }
    result = sph_test_ngc_ir_simulator_create_raw_frameset_many(self,
            inframeset, ndit, base_filename, pl, tag);
    cpl_frameset_delete(inframeset);
    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new set of raw frames
 * @param self          the ngc_simulator
 * @param inframes      the frameset of input pupil images / cubes
 * @param ndit          the number of dits per output cube
 * @param filename      the base filename
 * @param pl            propertylist to append (maybe NULL)
 * @param tag           tag of the products
 *
 * @return pointer to a new frameset
 *
 * This creates a set of raw frames, taken with the given exposure time.
 * The function creates darks for all the currently set read out mode.
 *
 * @see sph_test_ngc_ir_simulator_raw_cube_from_frame
 */
/*----------------------------------------------------------------------------*/
cpl_frameset*
sph_test_ngc_ir_simulator_create_raw_frameset_many(
        sph_test_ngc_ir_simulator* self, cpl_frameset* inframes, int ndit,
        const char* filename, cpl_propertylist* pl, const char* tag) {
    cpl_frameset* result = cpl_frameset_new();
    cpl_frame* aframe = NULL;
    cpl_frame* inframe = NULL;
    cpl_propertylist* pl2 = NULL;
    int ff = 0;
    int nframes = 0;
    time_t ztime;
    int nplanes = 0;
    if (!pl) {
        pl2 = cpl_propertylist_new();
        pl = pl2;
    }

    nframes = cpl_frameset_get_size(inframes);
    for (ff = 0; ff < nframes; ++ff) {
        inframe = cpl_frameset_get_position(inframes, ff);
        if (ndit <= 0) {
            nplanes = sph_fits_get_nplanes(cpl_frame_get_filename(inframe), 0);
        } else
            nplanes = ndit;
        aframe = sph_test_ngc_ir_simulator_raw_cube_from_frame(self, inframe,
                nplanes, filename, ff, pl, tag);
        sph_test_ngc_ir_simulator_advance_dithering__(self);
        if (self->timedelay > 0) {
            ztime = time(NULL);
            while (abs(time(NULL) - ztime) < self->timedelay) {
                ;
            }
        }
        cpl_frameset_insert(result, aframe);
    }
    if (pl2)
        cpl_propertylist_delete(pl2);
    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new raw image from input image
 * @param self  the NGC simulator
 * @param inimage  the image to process
 * @param dx        the dither in x
 * @param dy        the dither in y
 *
 * @return new image or NULL
 *
 * The function creates a new raw image from the input image, applying the
 * flat field of the NGC, adding bias and dark and adding noise. The exposure
 * time is taken from self->exptime.
 * If dx and dy are not zero the image is dithered before the NGC is applied.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_image*
sph_test_ngc_ir_simulator_create_raw_image(sph_test_ngc_ir_simulator* self,
        cpl_image* inimage, double dx, double dy) {
    sph_framecombination* framecomb = NULL;
    cpl_image* image = NULL;
    cpl_image* dumflat = NULL;
    sph_framecombination* comb = NULL;
    time_t ztime;

    cpl_test_error(CPL_ERROR_NONE);
    comb = sph_framecombination_new();
    sph_framecombination_set_method_gslfft(comb, 0.0);
    if (inimage) {
        if (dx != 0 || dy != 0) {
            framecomb = sph_framecombination_new();
            sph_framecombination_set_method_gslfft(framecomb, 0.0);
            image = sph_framecombination_shift_image(framecomb, inimage, dx,
                    dy);
        }
        if (!image) {
            image = cpl_image_duplicate(inimage);
        }
        if (image) {
            if ((dx || dy) && self->flatim) {
                dumflat = cpl_image_duplicate(self->flatim);
                cpl_image_delete(self->flatim);
                self->flatim = NULL;
                SPH_RAISE_CPL_RESET;
                self->flatim = sph_framecombination_shift_image(comb, dumflat,
                        dx, dy);
                SPH_RAISE_CPL_RESET;
            }
            SPH_RAISE_CPL_RESET;
            sph_test_ngc_ir_simulator_image_apply_flat__(self, image);
            SPH_RAISE_CPL_RESET;
            sph_test_ngc_ir_simulator_image_add_bias_and_dark__(self, image,
                    self->exptime);
            sph_test_ngc_ir_simulator_image_add_noise__(self, image);
            if ((dx || dy) && dumflat && self->flatim) {
                cpl_image_delete(self->flatim);
                self->flatim = NULL;
                self->flatim = cpl_image_duplicate(dumflat);
                cpl_image_delete(dumflat);
                dumflat = NULL;
                SPH_RAISE_CPL_RESET;
            }
            if (self->timedelay > 0) {
                ztime = time(NULL);
                while (abs(time(NULL) - ztime) < self->timedelay) {
                    ;
                }
            }
        }
    }
    sph_framecombination_delete(comb);
    comb = NULL;
    cpl_test_error(CPL_ERROR_NONE);
    return image;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new raw cube from a single input frame
 * @param  self         the NGC simulator
 * @param  inframe      the input frame (pupilimage(s))
 * @param  nplanes      the number of planes / DITs
 * @param  filename     the base filename (including .fits)
 * @param  ff           the number of the resulting frame in the series
 * @param  addprops     propertylist to append (maybe NULL)
 * @param  tag          the tag of the product frames
 *
 * @return new frame for created cube
 *
 * This function creates a raw cube from a single input frame containing
 * either a single pupilimage or a cube (of nplanes planes) of pupilimages.
 *
 *
 */
/*----------------------------------------------------------------------------*/
cpl_frame*
sph_test_ngc_ir_simulator_raw_cube_from_frame(sph_test_ngc_ir_simulator* self,
        cpl_frame* inframe, int nplanes, const char* filename, int ff,
        cpl_propertylist* addprops, const char* tag) {
    cpl_frame* result = NULL;
    cpl_propertylist* pl = NULL;
    int ninplanes = 0;
    cpl_image* im = NULL;
    int pp = 0;

    cpl_ensure(self, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(inframe, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(filename, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(tag, CPL_ERROR_NULL_INPUT, NULL);
    ninplanes = sph_fits_get_nplanes(cpl_frame_get_filename(inframe), 0);
    cpl_ensure( ninplanes == 1 || ninplanes == nplanes, CPL_ERROR_ILLEGAL_INPUT,
            NULL);
    cpl_test_error(CPL_ERROR_NONE);

    pl = cpl_propertylist_new();
    cpl_propertylist_append_double(pl, SPH_COMMON_KEYWORD_EXPTIME,
            self->exptime);
    cpl_propertylist_append_int(pl, SPH_COMMON_KEYWORD_NREADS, 1);
    cpl_propertylist_append_double(pl, SPH_COMMON_KEYWORD_SPH_DITHERX,
            self->current_dx);
    cpl_propertylist_append_double(pl, SPH_COMMON_KEYWORD_SPH_DITHERY,
            self->current_dy);
    cpl_propertylist_append(pl, addprops);

    result = sph_test_ngc_ir_simulator_create_raw_cube_frame__(filename, tag,
            ff);
    if (ninplanes == 1)
        im = cpl_image_load(cpl_frame_get_filename(inframe), CPL_TYPE_DOUBLE, 0,
                0);

    for (pp = 0; pp < nplanes; ++pp) {
        if (im == NULL) {
            im = cpl_image_load(cpl_frame_get_filename(inframe),
                    CPL_TYPE_DOUBLE, pp, 0);
            cpl_ensure(im, CPL_ERROR_FILE_IO, NULL);
        }

        sph_test_ngc_ir_simulator_digest_image__(self, im);

        sph_test_ngc_ir_simulator_save_im_in_fitscube__(im, result, pl, pp);

        if (ninplanes != 1) {
            cpl_image_delete(im);
            im = NULL;
        }
    }

    cpl_image_delete(im);
    im = NULL;
    cpl_propertylist_delete(pl);
    pl = NULL;
    cpl_test_error(CPL_ERROR_NONE);
    return result;
}

gsl_rng*
sph_test_ngc_ir_simulator_get_prng(sph_test_ngc_ir_simulator* self) {
    return self->pRNG;
}

void sph_test_ngc_ir_simulator_set_detector_size(
        sph_test_ngc_ir_simulator* self, int nx, int ny) {
    self->det_size_x = nx;
    self->det_size_y = ny;
    self->win_minx = 0;
    self->win_maxx = self->det_size_x;
    self->win_miny = 0;
    self->win_maxy = self->det_size_y;
}

void sph_test_ngc_ir_simulator_set_readout_window(
        sph_test_ngc_ir_simulator* self, int left, int bottom, int right,
        int top) {
    self->win_minx = left;
    self->win_miny = bottom;
    self->win_maxx = right;
    self->win_maxy = top;
}
void sph_test_ngc_ir_simulator_set_bias(sph_test_ngc_ir_simulator* self,
        double mean, double sigma) {
    self->mean_bias = mean;
    self->sigma_bias = sigma;
}
void sph_test_ngc_ir_simulator_set_dark(sph_test_ngc_ir_simulator* self,
        double mean, double sigma) {
    self->mean_dark = mean;
    self->sigma_dark = sigma;
}

void sph_test_ngc_ir_simulator_set_readout_mode(sph_test_ngc_ir_simulator* self,
        sph_test_rom mode) {
    self->rom = mode;
}
void sph_test_ngc_ir_simulator_set_readout_noise(
        sph_test_ngc_ir_simulator* self, double ron) {
    self->ron = ron;
}
void sph_test_ngc_ir_simulator_set_exptime(sph_test_ngc_ir_simulator* self,
        double exptime) {
    self->exptime = exptime;
}
void sph_test_ngc_ir_simulator_set_num_deadpixels(
        sph_test_ngc_ir_simulator* self, int n) {
    self->ndead = n;
}
void sph_test_ngc_ir_simulator_set_num_hotpixels(
        sph_test_ngc_ir_simulator* self, int n) {
    self->nhot = n;
}
void sph_test_ngc_ir_simulator_set_num_crazypixels(
        sph_test_ngc_ir_simulator* self, int n) {
    self->ncrazy = n;
}
void sph_test_ngc_ir_simulator_set_flat_linear_coeff(
        sph_test_ngc_ir_simulator* self, double c) {
    self->det_linear_factor = c;
}
void sph_test_ngc_ir_simulator_set_flat_quadratic_coeff(
        sph_test_ngc_ir_simulator* self, double c) {
    self->det_qaudratic_factor = c;
}
void sph_test_ngc_ir_simulator_set_dont_reset(sph_test_ngc_ir_simulator* self,
        int yesorno) {
    self->dont_reset = yesorno;
}
void sph_test_ngc_ir_simulator_set_save_singlecube(
        sph_test_ngc_ir_simulator* self, int yesorno) {
    self->singlecube = yesorno;
}
void sph_test_ngc_ir_simulator_set_add_noise(sph_test_ngc_ir_simulator* self,
        int addnoise, int addpoisson) {
    self->add_ron_noise = addnoise;
    self->poisson = addpoisson;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Crete the dark as an image
 * @param self the ngc_simulator
 *
 * @return image or NULL
 *
 * Returns the dark current per second as an image.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_image*
sph_test_ngc_ir_simulator_create_dark_image(sph_test_ngc_ir_simulator* self) {
    cpl_image* image = cpl_image_new(self->det_size_x, self->det_size_y,
            CPL_TYPE_FLOAT);
    int xx = 0;
    int yy = 0;

    for (xx = 0; xx < self->det_size_x; ++xx) {
        for (yy = 0; yy < self->det_size_y; ++yy) {
            cpl_image_set(image, xx + 1, yy + 1, self->mean_dark);
        }
    }

    return image;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Crete the static badpixels as an image
 * @param self the ngc_simulator
 *
 * @return image or NULL
 *
 * Returns the bads as an image.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_image*
sph_test_ngc_ir_simulator_create_static_badpixel_image(
        sph_test_ngc_ir_simulator* self) {
    cpl_image* image = NULL;
    int xx = 0;
    int yy = 0;

    image = cpl_image_new(self->det_size_x, self->det_size_y, CPL_TYPE_INT);
    for (xx = 0; xx < self->det_size_x; ++xx) {
        for (yy = 0; yy < self->det_size_y; ++yy) {
            cpl_image_set(image, xx + 1, yy + 1, 0);
        }
    }

    return image;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Crete the bias as an image
 * @param self the ngc_simulator
 *
 * @return image or NULL
 *
 * Returns the bias as an image.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_image*
sph_test_ngc_ir_simulator_create_bias_image(sph_test_ngc_ir_simulator* self) {
    cpl_image* image = cpl_image_new(self->det_size_x, self->det_size_y,
            CPL_TYPE_FLOAT);
    int xx = 0;
    int yy = 0;

    for (xx = 0; xx < self->det_size_x; ++xx) {
        for (yy = 0; yy < self->det_size_y; ++yy) {
            cpl_image_set(image, xx + 1, yy + 1, self->mean_bias);
        }
    }

    return image;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Copy input image and store for use as flat
 * @param self ngc model
 * @param flat the flat field to use
 *
 * Copies the image and stores it for use as flat.
 * Very useful when creating wavelength dependent flats: create a wavelength
 * dependend flat using sph_test_ngc_ir_simulator_create_flat_wdep and then
 * set it as flat using set_flat.
 */
/*----------------------------------------------------------------------------*/
void sph_test_ngc_ir_simulator_set_flat(sph_test_ngc_ir_simulator* self,
        const cpl_image* flat) {
    if (self->flatim) {
        cpl_image_delete(self->flatim);
        self->flatim = NULL;
    }
    self->flatim = cpl_image_duplicate(flat);
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Unset the flat
 * @param self   ngc simulator
 *
 * Unsets the flat.
 *
 */
/*----------------------------------------------------------------------------*/
void sph_test_ngc_ir_simulator_unset_flat(sph_test_ngc_ir_simulator* self) {
    if (self->flatim) {
        cpl_image_delete(self->flatim);
        self->flatim = NULL;
    }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Get the flat response as function of detector position and wavelength
 * @param self the ngc simulator
 * @param x the x position on detector
 * @param y the y position on detector
 * @param lambda the wavelength in microns
 *
 * @return
 *
 */
/*----------------------------------------------------------------------------*/
double sph_test_ngc_ir_simulator_get_response(sph_test_ngc_ir_simulator* self,
        double x, double y, double lambda) {
    double g = 0.0;
    double f = 0.0;
    double min = 1.0 - self->waveflat_factor * lambda * 0.5;
    double max = 1.0 + self->waveflat_factor * lambda * 0.5;
    double fwhm = self->waveflat_fwhm;
    double sigma = fwhm / 2.3458;
    double norm = self->waveflat_norm;
    double meanl = self->waveflat_meanl;

    g = (x + y) / ((double) self->det_size_x + (double) self->det_size_y)
            * (max - min) + min;
    if (self->waveflat_factor == 0)
        f = 1.0;
    else
        f = norm
                * exp(
                        -1.0 * (lambda - meanl) * (lambda - meanl)
                                / (2.0 * sigma * sigma));

    if (f > 1.0)
        f = 1.0;
    return g * f;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Crete a wavelength dependend flat field as an image
 * @param self the ngc_simulator
 * @param lambda the wavelength
 * @return image or NULL
 *
 * Returns the detector flat as an image. The flat is created
 * as a wavelength dependend flat using the
 * sph_test_ngc_ir_simulator_get_response(x,y,lambda) function.
 */
/*----------------------------------------------------------------------------*/
cpl_image*
sph_test_ngc_ir_simulator_create_detector_flat_image_wdep(
        sph_test_ngc_ir_simulator* self, cpl_image* wimage) {
    cpl_image* image = NULL;
    int xx = 0;
    int yy = 0;
    double lambda = 0.0;
    int bp = 0;

    if (self->flatim) {
        image = cpl_image_duplicate(self->flatim);
    } else {
        image = cpl_image_new(self->det_size_x, self->det_size_y,
                CPL_TYPE_DOUBLE);
        for (xx = 0; xx < self->det_size_x; ++xx) {
            for (yy = 0; yy < self->det_size_y; ++yy) {
                lambda = cpl_image_get(wimage, xx + 1, yy + 1, &bp);
                if (lambda > 0.0 || bp) {
                    cpl_image_set(
                            image,
                            xx + 1,
                            yy + 1,
                            sph_test_ngc_ir_simulator_get_response(self, xx, yy,
                                    lambda));
                } else {
                    cpl_image_set(image, xx + 1, yy + 1, 0.0);
                }
            }
        }
    }
    return image;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Crete the linear response (flat) as an image
 * @param self the ngc_simulator
 *
 * @return image or NULL
 *
 * Returns the detector flat as an image. Specifically, this
 * means that the linear coefficient of the response function
 * is returned as a function of pixel coordinate (i.e. as an image)
 *
 */
/*----------------------------------------------------------------------------*/
cpl_image*
sph_test_ngc_ir_simulator_create_detector_flat_image(
        sph_test_ngc_ir_simulator* self) {
    cpl_image* image = NULL;
    int xx = 0;
    int yy = 0;

    if (self->flatim) {
        image = cpl_image_duplicate(self->flatim);
    } else {
        image = cpl_image_new(self->det_size_x, self->det_size_y,
                CPL_TYPE_DOUBLE);
        for (xx = 0; xx < self->det_size_x; ++xx) {
            for (yy = 0; yy < self->det_size_y; ++yy) {
                cpl_image_set(image, xx + 1, yy + 1, self->det_linear_factor);
            }
        }
    }
    return image;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Set the flatfield to a gradient
 * @param self pointer to the ngc simulator
 * @param minval    the minimum value (in lower left corner)
 * @param maxval    the maximum value (in upper left corner)
 *
 * @return error code
 *
 */
/*----------------------------------------------------------------------------*/
void sph_test_ngc_ir_simulator_set_dff_gradient(sph_test_ngc_ir_simulator* self,
        double min, double max) {
    cpl_image* image = NULL;
    int xx = 0;
    int yy = 0;
    double step = 0.0;

    if (self->flatim) {
        cpl_image_delete(self->flatim);
        self->flatim = NULL;
    }
    image = cpl_image_new(self->det_size_x, self->det_size_y, CPL_TYPE_FLOAT);
    if (!image) {
        return;
    }

    step = (max - min) / (self->det_size_x + self->det_size_y);

    for (xx = 0; xx < self->det_size_x; ++xx) {
        for (yy = 0; yy < self->det_size_y; ++yy) {
            cpl_image_set(image, xx + 1, yy + 1, min + step * (xx + yy));
        }
    }
    self->flatim = image;
    return;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Set the standard dither pattern for the test ngc_simulator functions
 * @param self
 * @param number of dither steps -- after that cyclic repeat
 *
 * @return
 *
 * Set the standard dither pattern.
 *
 *
 */
/*----------------------------------------------------------------------------*/
void sph_test_ngc_ir_simulator_set_standard_ditherpattern(
        sph_test_ngc_ir_simulator* self, int ndithers) {
    int ii = 0;
    double dx = 0.0;
    double dy = 0.0;
    double stepx = 1.0;
    double stepy = 1.0;
    if (!self) {
        SPH_NO_SELF
        return;
    }
    if (ndithers < 1) {
        return;
    }
    if (self->ditherpattern_x) {
        cpl_vector_delete(self->ditherpattern_x);
        self->ditherpattern_x = NULL;
    }
    if (self->ditherpattern_y) {
        cpl_vector_delete(self->ditherpattern_y);
        self->ditherpattern_y = NULL;
    }

    self->ditherpattern_x = cpl_vector_new(ndithers);
    self->ditherpattern_y = cpl_vector_new(ndithers);
    for (ii = 0; ii < ndithers; ++ii) {
        dx = dx + stepx;
        dy = dy + stepy;
        cpl_vector_set(self->ditherpattern_x, ii, dx);
        cpl_vector_set(self->ditherpattern_x, ii, dy);
    }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Delete the ngc_simulator
 * @param self pointer to the ngc_simulator to delete
 *
 */
/*----------------------------------------------------------------------------*/
void sph_test_ngc_ir_simulator_delete(sph_test_ngc_ir_simulator* self) {
    if (self) {
        if (self->pRNG) {
            gsl_rng_free(self->pRNG);
        }
        if (self->ditherpattern_x) {
            cpl_vector_delete(self->ditherpattern_x);
        }
        if (self->ditherpattern_y) {
            cpl_vector_delete(self->ditherpattern_y);
        }
        if (self->flatim) {
            cpl_image_delete(self->flatim);
            self->flatim = NULL;
        }
        cpl_free(self);
    }
}

/*----------------------------------------------------------------------------*/
/*                  STATIC FUNCTIONS */
/*----------------------------------------------------------------------------*/

static
char * sph_test_ngc_ir_simulator_create_raw_cube_filename__(const char* namein,
                                                            int ii) {
    char base[256];
    char ext[256];

    sph_filemanager_split(namein, base, ext);
    return cpl_sprintf(ii > 999 ? "%s_%d_DIT.fits" : "%s_%04d_DIT.fits",
                       base, ii);
}

static cpl_frame*
sph_test_ngc_ir_simulator_create_raw_cube_frame__(const char* namein,
        const char* tag, int ii) {
    char * fname;
    cpl_frame* result = NULL;

    result = cpl_frame_new();
    fname = sph_test_ngc_ir_simulator_create_raw_cube_filename__(namein, ii);
    cpl_frame_set_tag(result, tag);
    cpl_frame_set_filename(result, fname);
    cpl_free(fname);
    cpl_frame_set_group(result, CPL_FRAME_GROUP_RAW);
    return result;
}

static
void sph_test_ngc_ir_simulator_save_im_in_fitscube__(cpl_image* im,
        cpl_frame* frame, cpl_propertylist* pl, int pp) {
    cpl_imagelist* imlist = NULL;
    imlist = cpl_imagelist_new();
    cpl_imagelist_set(imlist, im, 0);
    if (pp == 0) {
        cpl_imagelist_save(imlist, cpl_frame_get_filename(frame),
                CPL_TYPE_UNSPECIFIED, pl, CPL_IO_CREATE);
    } else {
        cpl_imagelist_save(imlist, cpl_frame_get_filename(frame),
                CPL_TYPE_UNSPECIFIED, pl, CPL_IO_APPEND);
    }
    cpl_imagelist_unset(imlist, 0);
    cpl_imagelist_delete(imlist);
    imlist = NULL;
}

static
void sph_test_ngc_ir_simulator_digest_image__(sph_test_ngc_ir_simulator* self,
        cpl_image* animage) {
    cpl_image* tmp = NULL;
    tmp = sph_test_ngc_ir_simulator_create_raw_image(self, animage,
            self->current_dx, self->current_dy);
    if (tmp) {
        cpl_image_multiply_scalar(animage, 0.0);
        cpl_image_add(animage, tmp);
        cpl_image_delete(tmp);
    }
}

static
int sph_test_ngc_ir_simulator_advance_dithering__(
        sph_test_ngc_ir_simulator* self) {
    cpl_ensure(self, CPL_ERROR_NULL_INPUT, -1);
    if (self->ditherpattern_x == NULL || self->ditherpattern_y == NULL) {
        return 0;
    }
    if (self->current_dither_index + 1
            >= cpl_vector_get_size(self->ditherpattern_x)) {
        self->current_dither_index = 0;
    } else {
        self->current_dither_index++;
    }
    self->current_dx = cpl_vector_get(self->ditherpattern_x,
            self->current_dither_index);
    self->current_dy = cpl_vector_get(self->ditherpattern_y,
            self->current_dither_index);
    cpl_ensure(cpl_error_get_code() == CPL_ERROR_NONE, cpl_error_get_code(),
            -1);
    return 1;
}

static
void sph_test_ngc_ir_simulator_image_add_bias_and_dark__(
        sph_test_ngc_ir_simulator* self, cpl_image* animage, double exptime) {
    cpl_image_add_scalar(animage, self->mean_bias + self->mean_dark * exptime);
}

static
void sph_test_ngc_ir_simulator_image_apply_flat__(
        sph_test_ngc_ir_simulator* self, cpl_image* animage) {
    if (!self->flatim) {
        self->flatim = sph_test_ngc_ir_simulator_create_detector_flat_image(
                self);
        SPH_RAISE_CPL;
    }
    cpl_image_multiply(animage, self->flatim);
    SPH_RAISE_CPL;
}
static
void sph_test_ngc_ir_simulator_image_add_noise__(
        sph_test_ngc_ir_simulator* self, cpl_image* animage) {
    if (self->poisson) {
        sph_test_image_tools_apply_poisson_noise(animage, self->pRNG);
    }
    if (self->add_ron_noise) {
        sph_test_image_tools_add_noise(animage, self->ron, self->pRNG);
        sph_test_image_tools_add_noise(animage, self->sigma_dark, self->pRNG);
        if (self->rom == SPH_TEST_NGC_ROM_SINGLE) {
            sph_test_image_tools_add_noise(animage, self->sigma_bias,
                    self->pRNG);
        } else {
            ; // If its not single readout, then bias noise wont matter.
        }
    }
}

