/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/
#include "sph_test_zpl_simulator.h"
#include <math.h>
#include "sph_error.h"
#include "sph_common_keywords.h"
#include "sph_zpl_keywords.h"
#include "sph_zpl_tags.h"
#include "sph_filemanager.h"
#include "sph_zpl_exposure.h"
#include "sph_zpl_exposure_imaging.h"
#include "sph_quad_image.h"
#include "sph_double_image.h"
#include "sph_triple_image.h"
#include "sph_master_frame.h"
#include "sph_fits.h"
#include "sph_cube.h"
#include "sph_fctable.h"
#include "sph_point.h"

#include "sph_test.h"
#include "sph_test_image_tools.h"

/*-----------------------------------------------------------------------------
 Static variables
 -----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
 Function declaration
 -----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
 Function code
 -----------------------------------------------------------------------------*/

sph_zpl_exposure*
sph_test_zpl_simulator_create_exposure(int nx, int ny) {
    sph_zpl_exposure* zplexp = NULL;

    zplexp = sph_test_zpl_simulator_create_exposure_4flat_double(nx, ny,
            ZERO_ODD_DOUBLE, ZERO_EVEN_DOUBLE, PI_ODD_DOUBLE, PI_EVEN_DOUBLE);

    if (!zplexp) {
        return NULL;
    }

    return zplexp;
}

cpl_propertylist*
sph_test_zpl_simulator_create_ovsc_propertylist(double* ovsc) {
    cpl_propertylist* result = NULL;

    result = cpl_propertylist_new();
    cpl_propertylist_append_double(result, ADU1_ZERO_OVSC_MEAN, ovsc[0]);
    cpl_propertylist_append_double(result, ADU1_ZERO_OVSC_RMS, ovsc[1]);
    cpl_propertylist_append_double(result, ADU2_ZERO_OVSC_MEAN, ovsc[2]);
    cpl_propertylist_append_double(result, ADU2_ZERO_OVSC_RMS, ovsc[3]);
    cpl_propertylist_append_double(result, ADU1_PI_OVSC_MEAN, ovsc[4]);
    cpl_propertylist_append_double(result, ADU1_PI_OVSC_RMS, ovsc[5]);
    cpl_propertylist_append_double(result, ADU2_PI_OVSC_MEAN, ovsc[6]);
    cpl_propertylist_append_double(result, ADU2_PI_OVSC_RMS, ovsc[7]);

    return result;

}

cpl_propertylist*
sph_test_zpl_simulator_create_ovsc_propertylist_imaging(double* ovsc) {
    cpl_propertylist* result = NULL;

    result = cpl_propertylist_new();
    cpl_propertylist_append_double(result, ADU1_OVSC_MEAN, ovsc[0]);
    cpl_propertylist_append_double(result, ADU1_OVSC_RMS, ovsc[1]);
    cpl_propertylist_append_double(result, ADU2_OVSC_MEAN, ovsc[2]);
    cpl_propertylist_append_double(result, ADU2_OVSC_RMS, ovsc[3]);

    return result;

}

sph_zpl_exposure_imaging*
sph_test_zpl_simulator_create_exposure_imaging(int nx, int ny) {
    sph_zpl_exposure_imaging* zplexp = NULL;

    zplexp = sph_test_zpl_simulator_create_exposure_imaging_2flat_double(nx, ny,
            ZERO_ODD_DOUBLE, ZERO_EVEN_DOUBLE);

    if (!zplexp) {
        return NULL;
    }

    return zplexp;
}

sph_zpl_exposure*
sph_test_zpl_simulator_create_exposure_4flat_double(int nx, int ny,
        double zero_odd, double zero_even, double pi_odd, double pi_even) {

    sph_zpl_exposure* zplexp = NULL;

    zplexp = sph_zpl_exposure_new_empty();
    zplexp->image_zero_odd = sph_test_image_tools_create_flat_image_double(nx,
            ny, zero_odd);
    zplexp->image_zero_even = sph_test_image_tools_create_flat_image_double(nx,
            ny, zero_even);
    zplexp->image_pi_odd = sph_test_image_tools_create_flat_image_double(nx, ny,
            pi_odd);
    zplexp->image_pi_even = sph_test_image_tools_create_flat_image_double(nx,
            ny, pi_even);

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        SPH_RAISE_CPL;
        return NULL;
    }
    return zplexp;
}

sph_zpl_exposure_imaging*
sph_test_zpl_simulator_create_exposure_imaging_2flat_double(int nx, int ny,
        double image_odd, double image_even) {

    sph_zpl_exposure_imaging* zplexp = NULL;

    zplexp = sph_zpl_exposure_imaging_new_empty();
    zplexp->image_odd = sph_test_image_tools_create_flat_image_double(nx, ny,
            image_odd);
    zplexp->image_even = sph_test_image_tools_create_flat_image_double(nx, ny,
            image_even);

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        SPH_RAISE_CPL;
        return NULL;
    }
    return zplexp;
}

sph_zpl_exposure*
sph_test_zpl_simulator_create_exposure_4flat_int(int nx, int ny, int zero_odd,
        int zero_even, int pi_odd, int pi_even) {

    sph_zpl_exposure* zplexp = NULL;

    zplexp = sph_zpl_exposure_new_empty();
    zplexp->image_zero_odd = sph_test_image_tools_create_flat_image_int(nx, ny,
            zero_odd);
    zplexp->image_zero_even = sph_test_image_tools_create_flat_image_int(nx, ny,
            zero_even);
    zplexp->image_pi_odd = sph_test_image_tools_create_flat_image_int(nx, ny,
            pi_odd);
    zplexp->image_pi_even = sph_test_image_tools_create_flat_image_int(nx, ny,
            pi_even);

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        SPH_RAISE_CPL;
        return NULL;
    }
    return zplexp;
}

cpl_frame*
sph_test_zpl_simulator_create_exposure_cube_frame(int nx, int ny, int nz,
        const char* filename) {
    cpl_frame* zexpcube = NULL;

    zexpcube = sph_test_zpl_simulator_create_exposure_cube_frame_4flat_double(
            nx, ny, nz, filename, ZERO_ODD_DOUBLE, ZERO_EVEN_DOUBLE,
            PI_ODD_DOUBLE, PI_EVEN_DOUBLE);
    if (!zexpcube) {
        return NULL;
    }

    return zexpcube;
}

cpl_frame*
sph_test_zpl_simulator_create_exposure_imaging_cube_frame(int nx, int ny,
        int nz, const char* filename) {
    cpl_frame* zexpcube = NULL;

    zexpcube =
            sph_test_zpl_simulator_create_exposure_imaging_cube_frame_2flat_double(
                    nx, ny, nz, filename, ZERO_ODD_DOUBLE, ZERO_EVEN_DOUBLE);
    if (!zexpcube) {
        return NULL;
    }

    return zexpcube;
}

cpl_frame*
sph_test_zpl_simulator_create_exposure_cube_frame_4flat_double(int nx, int ny,
        int nz, const char* filename, double zero_odd, double zero_even,
        double pi_odd, double pi_even) {
    sph_error_code rerr = CPL_ERROR_NONE;
    cpl_frame* zexpcube = NULL;
    cpl_frame* zexpframe = NULL;
    cpl_frameset* zexpframeset = NULL;
    sph_zpl_exposure* zexp = NULL;

    zexpcube = sph_test_zpl_simulator_frame_template(filename);

    zexpframeset = cpl_frameset_new();
    for (int ii = 0; ii < nz; ii++) {

        zexp = sph_test_zpl_simulator_create_exposure_4flat_double(nx, ny,
                zero_odd, zero_even, pi_odd, pi_even);
        zexpframe = sph_filemanager_create_temp_frame(filename,
                SPH_ZPL_TAG_PREPROC_CALIB, CPL_FRAME_GROUP_NONE);
        rerr = sph_zpl_exposure_save(zexp, zexpframe, NULL);

        if (rerr != CPL_ERROR_NONE) {
            SPH_ERR("save zpl exposure failed.");
            return NULL;
        }

        sph_zpl_exposure_delete(zexp);
        zexp = NULL;
        cpl_frameset_insert(zexpframeset, zexpframe);
    }

    if (!zexpframeset) {
        SPH_ERR("NULL pointer to zpl exposure frameset.");
        return NULL;
    }

    rerr = sph_test_zpl_simulator_create_cube_from_exposure_frames_double(
            zexpframeset, zexpcube);
    if (zexpframeset)
        cpl_frameset_delete(zexpframeset);
    zexpframeset = NULL;

    if (rerr != CPL_ERROR_NONE) {
        SPH_ERR("create cube from zpl exposure frames failed.");
        return NULL;
    }

    return zexpcube;
}

cpl_frame*
sph_test_zpl_simulator_create_exposure_imaging_cube_frame_2flat_double(int nx,
        int ny, int nz, const char* filename, double imaging_odd,
        double imaging_even) {
    sph_error_code rerr = CPL_ERROR_NONE;
    cpl_frame* zexpcube = NULL;
    cpl_frame* zexpframe = NULL;
    cpl_frameset* zexpframeset = NULL;
    sph_zpl_exposure_imaging* zexp = NULL;

    zexpcube = sph_test_zpl_simulator_frame_template(filename);

    zexpframeset = cpl_frameset_new();
    for (int ii = 0; ii < nz; ii++) {

        zexp = sph_test_zpl_simulator_create_exposure_imaging_2flat_double(nx,
                ny, imaging_odd, imaging_even);
        zexpframe = sph_filemanager_create_temp_frame(filename,
                SPH_ZPL_TAG_PREPROC_IMAGING_CALIB, CPL_FRAME_GROUP_NONE);
        rerr = sph_zpl_exposure_imaging_save(zexp, zexpframe, NULL);

        if (rerr != CPL_ERROR_NONE) {
            SPH_ERR("save zpl exposure imaging failed.");
            return NULL;
        }

        cpl_frameset_insert(zexpframeset, zexpframe);
        sph_zpl_exposure_imaging_delete(zexp);
    }

    if (!zexpframeset) {
        SPH_ERR("NULL pointer to zpl exposure imaging frameset.");
        return NULL;
    }

    rerr =
            sph_test_zpl_simulator_create_cube_from_exposure_imaging_frames_double(
                    zexpframeset, zexpcube);

    if (rerr != CPL_ERROR_NONE) {
        SPH_ERR("create cube from zpl exposure imaging frames failed.");
        return NULL;
    }

    cpl_frameset_delete(zexpframeset);
    return zexpcube;
}

sph_error_code sph_test_zpl_simulator_create_cube_from_exposure_frames_double(
        cpl_frameset* zexpframes, cpl_frame* zexpcube) {
    sph_error_code rerr = CPL_ERROR_NONE;
    cpl_frame* curframe = NULL;
    cpl_image* image = NULL;
    cpl_propertylist* pl = NULL;

    sph_error_reset();

    if (!zexpframes && !zexpcube) {
        SPH_NULL_ERROR;
        return sph_error_get_last_code();
    }

    if (cpl_frameset_get_size(zexpframes) < 1) {
        SPH_ERR("no input zpl exp frames");
        return sph_error_get_last_code();
    }

    unlink(cpl_frame_get_filename(zexpcube));
    curframe = cpl_frameset_get_first(zexpframes);
    while (curframe) {
        for (int extn = 0; extn < 4; extn++) {
            pl = cpl_propertylist_load_regexp(cpl_frame_get_filename(curframe),
                    extn, ".*ESO.*", extn);
            image = cpl_image_load(cpl_frame_get_filename(curframe),
                    CPL_TYPE_DOUBLE, 0, extn);
            rerr = sph_cube_append_image(cpl_frame_get_filename(zexpcube),
                    image, pl, extn);
            if (rerr) {
                SPH_ERR("cant append image to the given zpl exposure cube")
                sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR, "sph_fits_append_image failed: %d",
                        rerr);
                return rerr;
            }

            cpl_image_delete(image);
            cpl_propertylist_delete(pl);
            pl = NULL;

        }

        curframe = cpl_frameset_get_next(zexpframes);
    }

    sph_cube_finalise_file(cpl_frame_get_filename(zexpcube));
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        SPH_RAISE_CPL;
    }

    return sph_error_get_last_code();
}

sph_error_code sph_test_zpl_simulator_create_cube_from_exposure_imaging_frames_double(
        cpl_frameset* zexpframes, cpl_frame* zexpcube) {
    sph_error_code rerr = CPL_ERROR_NONE;
    cpl_frame* curframe = NULL;
    cpl_image* image = NULL;
    cpl_propertylist* pl = NULL;

    sph_error_reset();

    if (!zexpframes && !zexpcube) {
        SPH_NULL_ERROR;
        return sph_error_get_last_code();
    }

    if (cpl_frameset_get_size(zexpframes) < 1) {
        SPH_ERR("no input zpl exp imaging frames");
        return sph_error_get_last_code();
    }

    unlink(cpl_frame_get_filename(zexpcube));
    curframe = cpl_frameset_get_first(zexpframes);
    while (curframe) {
        for (int extn = 0; extn < 2; extn++) {
            //pl = cpl_propertylist_load_regexp( cpl_frame_get_filename( curframe ), extn, ".*ESO.*", 0);
            pl = cpl_propertylist_load(cpl_frame_get_filename(curframe), extn);
            image = cpl_image_load(cpl_frame_get_filename(curframe),
                    CPL_TYPE_DOUBLE, 0, extn);
            rerr = sph_cube_append_image(cpl_frame_get_filename(zexpcube),
                    image, pl, extn);
            if (rerr) {
                SPH_ERR(
                        "cant append image to the given zpl exposure imaging cube");
                sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR, "sph_fits_append_image failed: %d",
                        rerr);
                return rerr;
            }

            cpl_image_delete(image);
            cpl_propertylist_delete(pl);
            pl = NULL;

        }

        curframe = cpl_frameset_get_next(zexpframes);
    }

    sph_cube_finalise_file(cpl_frame_get_filename(zexpcube));
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        SPH_RAISE_CPL;
    }

    return sph_error_get_last_code();
}

sph_error_code sph_test_zpl_simulator_create_cube_from_exposure_frames_int(
        cpl_frameset* zexpframes, cpl_frame* zexpcube) {
    sph_error_code rerr = CPL_ERROR_NONE;
    cpl_frame* curframe = NULL;
    cpl_image* image = NULL;
    cpl_propertylist* pl = NULL;

    sph_error_reset();

    if (!zexpframes && !zexpcube) {
        SPH_NULL_ERROR;
        return sph_error_get_last_code();
    }

    if (cpl_frameset_get_size(zexpframes) < 1) {
        SPH_ERR("no input zpl exp frames");
        return sph_error_get_last_code();
    }

    curframe = cpl_frameset_get_first(zexpframes);
    unlink(cpl_frame_get_filename(zexpcube));
    while (curframe) {
        for (int extn = 0; extn < 4; extn++) {

            pl = cpl_propertylist_load_regexp(cpl_frame_get_filename(curframe),
                    extn, ".*ESO.*", 0);
            image = cpl_image_load(cpl_frame_get_filename(curframe),
                    CPL_TYPE_INT, 0, extn);
            rerr = sph_cube_append_image(cpl_frame_get_filename(zexpcube),
                    image, pl, extn);
            if (rerr) {
                SPH_ERR("cant append image to the given zpl exposure cube")
                sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR, "sph_fits_append_image failed: %d",
                        rerr);
                return rerr;
            }
            cpl_image_delete(image);
            image = NULL;
            cpl_propertylist_delete(pl);
            pl = NULL;
        }

        curframe = cpl_frameset_get_next(zexpframes);
    }

    sph_cube_finalise_file(cpl_frame_get_filename(zexpcube));
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        SPH_RAISE_CPL;
    }

    return sph_error_get_last_code();
}

cpl_frame* sph_test_zpl_simulator_frame_template(const char *filename) {
    cpl_frame* ftemplate = NULL;

    ftemplate = cpl_frame_new();
    cpl_frame_set_filename(ftemplate, filename);
    cpl_frame_set_tag(ftemplate, "NONE");
    cpl_frame_set_group(ftemplate, CPL_FRAME_GROUP_NONE);

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        SPH_RAISE_CPL;
        return NULL;
    }

    return ftemplate;
}

cpl_frame* sph_test_simulator_create_fctable(int nx, int ny, double* dx,
        double* dy, double* angle, int nframes) {
    sph_fctable* fctable = NULL;
    cpl_frame* frame = NULL;

    fctable = sph_fctable_new();
    for (int ii = 0; ii < nframes; ii++) {
        sph_fctable_add_row_single(fctable, 0.0, //TODO: add proper time here
                ny / 2.0 + dx[ii], ny / 2.0 + dy[ii], angle[ii]);
    }

    frame = sph_filemanager_create_temp_frame("test_fctable.tmp",
            SPH_ZPL_TAG_FIELD_CENTER_CALIB, CPL_FRAME_GROUP_CALIB);
    sph_fctable_save_ascii(fctable, cpl_frame_get_filename(frame));
    sph_fctable_delete(fctable); fctable = NULL;
    return frame;
}

/*
 cpl_frame* sph_test_simulator_create_fctable(cpl_frame* frame, int nx, int ny, double* dx,
 double* dy, double* angle, int nframes){
 sph_fctable*		 fctable	= NULL;
 cpl_frame*			 frame_fc		= NULL;
 char*                fname 		= NULL;
 char*                fname2 	= NULL;


 fctable = sph_fctable_new();
 for (int ii = 0; ii < nframes; ii++ ){
 sph_fctable_add_row_single(fctable, ny / 2.0 + dx[ii], ny / 2.0 + dy[ii], angle[ii]);
 }

 //frame = sph_filemanager_create_temp_frame("test_fctable.tmp", SPH_ZPL_TAG_STAR_CENTER_CALIB, CPL_FRAME_GROUP_CALIB);
 //create a filename of the given frame with  'txt' extension
 fname2 = sph_filemanager_filename_new_ext( cpl_frame_get_filename(frame), "txt");
 fname = sph_filemanager_new_filename_from_base( fname2, "fctable");
 sph_fctable_save_ascii(fctable, fname);

 frame_fc = cpl_frame_new();
 cpl_frame_set_filename( frame_fc, fname );
 cpl_frame_set_tag(frame_fc,SPH_ZPL_TAG_STAR_CENTER_CALIB);
 cpl_frame_set_group(frame_fc,CPL_FRAME_GROUP_CALIB);

 //sph_fctable_save_ascii(fctable, cpl_frame_get_filename(frame));
 cpl_free(fname2); fname2 = NULL;
 cpl_free(fname); fname = NULL;
 sph_fctable_delete(fctable); fctable = NULL;

 return frame;

 }

 */

void sph_test_simulator_create_zplpreproc_science_cube(cpl_frame** zexpcube,
        cpl_frame* fctable_frame, int nx, int ny) {
    sph_error_code rerr = CPL_ERROR_NONE;
    sph_fctable* fctable = NULL;
    sph_point* p = NULL;
    sph_zpl_exposure* zplexp = NULL;
    cpl_image* im = NULL;
    cpl_frameset* zexpframes = NULL;
    cpl_frame* zexpframe = NULL;
    const double flat_counts = 100.0;
    const double star_counts = 1000000.0;
    const double star_fwhm = 10;
    const double planet_counts = 10000.0;
    const double planet_fwhm = 1;
    double cx = 0;
    double cy = 0;
    double angle = 0;

    fctable = sph_fctable_load_ascii(cpl_frame_get_filename(fctable_frame));


    *zexpcube = sph_filemanager_create_temp_frame("test_science_cube.fits",
            SPH_ZPL_TAG_SCIENCE_P1_PREPROC_RAW, CPL_FRAME_GROUP_RAW);
    zexpframes = cpl_frameset_new();

    for (int ii = 0; ii < sph_fctable_get_size(fctable); ii++) {
        zplexp = sph_zpl_exposure_new_empty();

        cx = sph_fctable_get_centre_x_element(fctable, ii);
        cy = sph_fctable_get_centre_y_element(fctable, ii);
        angle = sph_fctable_get_angle(fctable, ii);
        SPH_ERROR_RAISE_INFO(
                SPH_ERROR_INFO_MESSAGE,
                "fctable->cx = %5.3f; fctable->cy = %5.3f; fctable->angle = %5.3f", cx, cy, angle);
        im = sph_test_image_tools_create_flat_image_double(nx, ny, flat_counts);
        //add star as a gauss with center coords (cx+0.5, cy+0.5); +0.5 is important because of the difference
        //in the coord system between sphere and fits: sphere starts from (0,0)pixel whereas fits from (1,1)
        sph_test_image_tools_add_gauss(im, cx + 0.5, cy + 0.5, star_fwhm,
                star_counts + ii * 100000.0);
        p = sph_point_new(cx + 20.0, cy); //rotation point
        sph_point_rotate_around(p, cx + .5, cy + 0.5,
                cos(angle * CPL_MATH_RAD_DEG), sin(angle * CPL_MATH_RAD_DEG),
                0.0, 0.0);
        SPH_ERROR_RAISE_INFO(SPH_ERROR_INFO_MESSAGE,
                "p->x = %5.3f; p->y = %5.3f", p->x, p->y);
        sph_test_image_tools_add_gauss(im, p->x, p->y, planet_fwhm,
                planet_counts); //planet

        //duplicate for all 4 zimpol images
        zplexp->image_zero_odd = cpl_image_duplicate(im);
        zplexp->image_zero_even = cpl_image_duplicate(im);
        cpl_image_add_scalar(zplexp->image_zero_odd, -100.0);
        zplexp->image_pi_odd = cpl_image_duplicate(im);
        zplexp->image_pi_even = cpl_image_duplicate(im);
        cpl_image_add_scalar(zplexp->image_pi_odd, +100.0);
        cpl_image_delete(im);
        im = NULL;
        sph_point_delete(p);
        p = NULL;

        zexpframe = sph_filemanager_create_temp_frame(
                "test_zplexp_science.fits", "NONE", CPL_FRAME_GROUP_NONE);
        rerr = sph_zpl_exposure_save(zplexp, zexpframe, NULL);

        if (rerr != CPL_ERROR_NONE) {
            SPH_ERR( "could not save zpl exposure");
            return;
        }
        cpl_frameset_insert(zexpframes, zexpframe);
        sph_zpl_exposure_delete(zplexp);

    }

    rerr = sph_test_zpl_simulator_create_cube_from_exposure_frames_double(
            zexpframes, *zexpcube);

    cpl_frameset_delete(zexpframes);
    sph_fctable_delete(fctable);


    /*
     for (ii = 0; ii < nframes; ++ii) {
     sph_fctable_add_row_single(fctable,ny / 2.0 + dithx[ii],ny / 2.0 + dithy[ii],angle);
     }

     for (ii = 0; ii < nframes; ++ii) {
     im = sph_test_image_tools_create_flat_image_double(nx,ny,100.0);
     cxl = sph_fctable_get_centre_x_left(fctable,ii);
     cyl = sph_fctable_get_centre_y_left(fctable,ii);
     sph_test_image_tools_add_gauss(im,cxl,cyl, 10.0, 1000000.0); //star
     p = sph_point(cxl + 10.0,cyl);
     sph_point_rotate_around(p,cxl,cyl,anlge);//rotation
     sph_test_image_tools_add_gauss(im, cxr + p->x, cyr + p->y, 10.0, 1000.0);//planet
     //sph_test_image_tools_add_noise(im,10.0,NULL);
     //sph_test_image_tools_apply_poisson
     cpl_frameset_insert(inframes,
     sph_test_frame_image(im,
     CPL_BPP_32_SIGNED,
     "SCIENCE_IDEAL",
     CPL_FRAME_GROUP_RAW));
     cpl_image_delete(im); im = NULL;
     sph_point_delete(p);
     }
     */

}

void sph_test_simulator_create_zplpreproc_science_cube_overscan(
        cpl_frame** zexpcube, cpl_frame* fctable_frame, int nx, int ny,
        double* ovsc) {
    sph_error_code rerr = CPL_ERROR_NONE;
    sph_fctable* fctable = NULL;
    sph_point* p = NULL;
    sph_zpl_exposure* zplexp = NULL;
    cpl_image* im = NULL;
    cpl_frameset* zexpframes = NULL;
    cpl_frame* zexpframe = NULL;
    const double flat_counts = 100.0;
    const double star_counts = 1000000.0;
    const double star_fwhm = 10;
    const double planet_counts = 10000.0;
    const double planet_fwhm = 1;
    double cx = 0;
    double cy = 0;
    double angle = 0;
    cpl_table* ovsc_table = NULL;
    cpl_frame* curframe = NULL;
    int iplane = 0;
    cpl_propertylist* ovsc_pl = NULL;
    double ovsc_duplicate[8];

    for (int kk = 0; kk < 8; kk++) {
        ovsc_duplicate[kk] = ovsc[kk];
    }

    fctable = sph_fctable_load_ascii(cpl_frame_get_filename(fctable_frame));

    *zexpcube = sph_filemanager_create_temp_frame(
            "test_science_cube_overscan.fits",
            SPH_ZPL_TAG_SCIENCE_P1_PREPROC_RAW, CPL_FRAME_GROUP_RAW);
    zexpframes = cpl_frameset_new();

    for (int ii = 0; ii < sph_fctable_get_size(fctable); ii++) {
        zplexp = sph_zpl_exposure_new_empty();

        cx = sph_fctable_get_centre_x_element(fctable, ii);
        cy = sph_fctable_get_centre_y_element(fctable, ii);
        angle = sph_fctable_get_angle(fctable, ii);
        SPH_ERROR_RAISE_INFO(
                SPH_ERROR_INFO_MESSAGE,
                "fctable->cx = %5.3f; fctable->cy = %5.3f; fctable->angle = %5.3f", cx, cy, angle);
        im = sph_test_image_tools_create_flat_image_double(nx, ny, flat_counts);
        //add star as a gauss with center coords (cx+0.5, cy+0.5); +0.5 is important because of the difference
        //in the coord system between sphere and fits: sphere starts from (0,0)pixel whereas fits from (1,1)
        sph_test_image_tools_add_gauss(im, cx + 0.5, cy + 0.5, star_fwhm,
                star_counts + ii * 100000.0);
        p = sph_point_new(cx + 20.0, cy); //rotation point
        sph_point_rotate_around(p, cx + .5, cy + 0.5,
                cos(angle * CPL_MATH_RAD_DEG), sin(angle * CPL_MATH_RAD_DEG),
                0.0, 0.0);
        SPH_ERROR_RAISE_INFO(SPH_ERROR_INFO_MESSAGE,
                "p->x = %5.3f; p->y = %5.3f", p->x, p->y);
        sph_test_image_tools_add_gauss(im, p->x, p->y, planet_fwhm,
                planet_counts); //planet

        //duplicate for all 4 zimpol images
        zplexp->image_zero_odd = cpl_image_duplicate(im);
        zplexp->image_zero_even = cpl_image_duplicate(im);
        cpl_image_add_scalar(zplexp->image_zero_odd, -100.0);
        zplexp->image_pi_odd = cpl_image_duplicate(im);
        zplexp->image_pi_even = cpl_image_duplicate(im);
        cpl_image_add_scalar(zplexp->image_pi_odd, +100.0);
        cpl_image_delete(im);
        im = NULL;
        sph_point_delete(p);
        p = NULL;

        zexpframe = sph_filemanager_create_temp_frame(
                "test_zplexp_science.fits", "NONE", CPL_FRAME_GROUP_NONE);
        rerr = sph_zpl_exposure_save(zplexp, zexpframe, NULL);
        sph_zpl_exposure_delete(zplexp);

        if (rerr != CPL_ERROR_NONE) {
            SPH_ERR( "could not save zpl exposure");
            return;
        }
        cpl_frameset_insert(zexpframes, zexpframe);
    }

    //rerr = sph_test_zpl_simulator_create_cube_from_exposure_frames_double( zexpframes, *zexpcube);

    //cubetest = sph_filemanager_create_temp_frame("test_science_cube_overscan_test.fits", SPH_ZPL_TAG_SCIENCE_P1_PREPROC_RAW, CPL_FRAME_GROUP_RAW);
    curframe = cpl_frameset_get_first(zexpframes);
    zplexp = sph_zpl_exposure_load(curframe, 0);

    ovsc_pl = sph_test_zpl_simulator_create_ovsc_propertylist(ovsc_duplicate);
    if (zplexp->ovsc)
        cpl_propertylist_delete(zplexp->ovsc);
    zplexp->ovsc = cpl_propertylist_duplicate(ovsc_pl);
    cpl_propertylist_delete(ovsc_pl);
    ovsc_pl = NULL;

    ovsc_table = sph_zpl_exposure_ovsc_table_create_empty();
    sph_zpl_exposure_save_open(zplexp, *zexpcube, NULL, ovsc_table);
    sph_zpl_exposure_delete(zplexp);

    curframe = cpl_frameset_get_next(zexpframes);
    while (curframe) {
        iplane++;
        SPH_ERROR_RAISE_INFO(SPH_ERROR_INFO, "curframe: iplane = %d", iplane);
        zplexp = sph_zpl_exposure_load(curframe, 0);

        ovsc_pl = sph_test_zpl_simulator_create_ovsc_propertylist(
                ovsc_duplicate);
        if (zplexp->ovsc)
            cpl_propertylist_delete(zplexp->ovsc);
        zplexp->ovsc = cpl_propertylist_duplicate(ovsc_pl);
        cpl_propertylist_delete(ovsc_pl);
        ovsc_pl = NULL;

        sph_zpl_exposure_save_append(zplexp, *zexpcube, ovsc_table);
        sph_zpl_exposure_delete(zplexp);
        curframe = cpl_frameset_get_next(zexpframes);
    }

    sph_zpl_exposure_finalize_file(*zexpcube, ovsc_table);
    cpl_frameset_delete(zexpframes);
    cpl_table_delete(ovsc_table);
    sph_fctable_delete(fctable);

    return;

    /*
     for (ii = 0; ii < nframes; ++ii) {
     sph_fctable_add_row_single(fctable,ny / 2.0 + dithx[ii],ny / 2.0 + dithy[ii],angle);
     }

     for (ii = 0; ii < nframes; ++ii) {
     im = sph_test_image_tools_create_flat_image_double(nx,ny,100.0);
     cxl = sph_fctable_get_centre_x_left(fctable,ii);
     cyl = sph_fctable_get_centre_y_left(fctable,ii);
     sph_test_image_tools_add_gauss(im,cxl,cyl, 10.0, 1000000.0); //star
     p = sph_point(cxl + 10.0,cyl);
     sph_point_rotate_around(p,cxl,cyl,anlge);//rotation
     sph_test_image_tools_add_gauss(im, cxr + p->x, cyr + p->y, 10.0, 1000.0);//planet
     //sph_test_image_tools_add_noise(im,10.0,NULL);
     //sph_test_image_tools_apply_poisson
     cpl_frameset_insert(inframes,
     sph_test_frame_image(im,
     CPL_BPP_32_SIGNED,
     "SCIENCE_IDEAL",
     CPL_FRAME_GROUP_RAW));
     cpl_image_delete(im); im = NULL;
     sph_point_delete(p);
     }
     */

}

void sph_test_simulator_create_zplpreproc_imaging_science_cube(
        cpl_frame** zexpcube, cpl_frame* fctable_frame, int nx, int ny) {
    sph_error_code rerr = CPL_ERROR_NONE;
    sph_fctable* fctable = NULL;
    sph_point* p = NULL;
    sph_zpl_exposure_imaging* zplexpimg = NULL;
    cpl_image* im = NULL;
    cpl_frameset* zexpframes = NULL;
    cpl_frame* zexpframe = NULL;
    const double flat_counts = 100.0;
    const double star_counts = 1000000.0;
    const double star_fwhm = 10;
    const double planet_counts = 10000.0;
    const double planet_fwhm = 1;
    double cx = 0;
    double cy = 0;
    double angle = 0;

    fctable = sph_fctable_load_ascii(cpl_frame_get_filename(fctable_frame));

    zplexpimg = sph_zpl_exposure_imaging_new_empty();

    *zexpcube = sph_filemanager_create_temp_frame(
            "test_science_imaging_cube.fits",
            SPH_ZPL_TAG_SCIENCE_IMAGING_PREPROC_RAW, CPL_FRAME_GROUP_RAW);
    zexpframes = cpl_frameset_new();

    for (int ii = 0; ii < sph_fctable_get_size(fctable); ii++) {
        cx = sph_fctable_get_centre_x_element(fctable, ii);
        cy = sph_fctable_get_centre_y_element(fctable, ii);
        angle = sph_fctable_get_angle(fctable, ii);
        SPH_ERROR_RAISE_INFO(
                SPH_ERROR_INFO_MESSAGE,
                "fctable->cx = %5.3f; fctable->cy = %5.3f; fctable->angle = %5.3f", cx, cy, angle);
        im = sph_test_image_tools_create_flat_image_double(nx, ny, flat_counts);
        //add star as a gauss with center coords (cx+0.5, cy+0.5); +0.5 is important because of the difference
        //in the coord system between sphere and fits: sphere starts from (0,0)pixel whereas fits from (1,1)
        sph_test_image_tools_add_gauss(im, cx + 0.5, cy + 0.5, star_fwhm,
                star_counts + ii * 100000.0);
        p = sph_point_new(cx + 20.0, cy); //rotation point
        sph_point_rotate_around(p, cx + .5, cy + 0.5,
                cos(angle * CPL_MATH_RAD_DEG), sin(angle * CPL_MATH_RAD_DEG),
                0.0, 0.0);
        SPH_ERROR_RAISE_INFO(SPH_ERROR_INFO_MESSAGE,
                "p->x = %5.3f; p->y = %5.3f", p->x, p->y);
        sph_test_image_tools_add_gauss(im, p->x, p->y, planet_fwhm,
                planet_counts); //planet

        //duplicate for 2 zimpol images (imaging mode)
        zplexpimg->image_odd = cpl_image_duplicate(im);
        zplexpimg->image_even = cpl_image_duplicate(im);
        cpl_image_add_scalar(zplexpimg->image_even, -100.0);
        cpl_image_delete(im);
        im = NULL;
        sph_point_delete(p);
        p = NULL;

        zexpframe = sph_filemanager_create_temp_frame(
                "test_zplexpimg_science.fits", "NONE", CPL_FRAME_GROUP_NONE);
        rerr = sph_zpl_exposure_imaging_save(zplexpimg, zexpframe, NULL);

        if (rerr != CPL_ERROR_NONE) {
            SPH_ERR( "could not save zpl exposure imaging");
            return;
        }
        cpl_frameset_insert(zexpframes, zexpframe);
    }

    rerr =
            sph_test_zpl_simulator_create_cube_from_exposure_imaging_frames_double(
                    zexpframes, *zexpcube);

}

cpl_frame* sph_test_simulator_create_zplpreproc_imaging_science_cube_with_overscan(
        cpl_frame* fctable_frame, int nx, int ny, double *ovsc) {
    sph_error_code rerr = CPL_ERROR_NONE;
    sph_fctable* fctable = NULL;
    sph_point* p = NULL;
    sph_zpl_exposure_imaging* zplexpimg = NULL;
    cpl_image* im = NULL;
//	cpl_frameset*				zexpframes		= NULL;
//	cpl_frame*					zexpframe		= NULL;
    const double flat_counts = 100.0;
    const double star_counts = 1000000.0;
    const double star_fwhm = 10;
    const double planet_counts = 10000.0;
    const double planet_fwhm = 1;
    double cx = 0;
    double cy = 0;
    double angle = 0;

    cpl_table* ovsc_table = NULL;
    double ovsc_duplicate[4];
    cpl_propertylist* ovsc_pl = NULL;
    cpl_frame* zexpcube = NULL;

    for (int kk = 0; kk < 4; kk++) {
        ovsc_duplicate[kk] = ovsc[kk];
    }

    fctable = sph_fctable_load_ascii(cpl_frame_get_filename(fctable_frame));

//	zplexpimg = sph_zpl_exposure_imaging_new_empty();

    zexpcube = sph_filemanager_create_temp_frame(
            "test_science_imaging_cube_with_overscan.fits",
            SPH_ZPL_TAG_SCIENCE_IMAGING_PREPROC_RAW, CPL_FRAME_GROUP_RAW);
    //zexpframes = cpl_frameset_new();

    for (int ii = 0; ii < sph_fctable_get_size(fctable); ii++) {
        //printf("ii = %d\n", ii);
        cx = sph_fctable_get_centre_x_element(fctable, ii);
        cy = sph_fctable_get_centre_y_element(fctable, ii);
        angle = sph_fctable_get_angle(fctable, ii);
        SPH_ERROR_RAISE_INFO(
                SPH_ERROR_INFO_MESSAGE,
                "fctable->cx = %5.3f; fctable->cy = %5.3f; fctable->angle = %5.3f", cx, cy, angle);
        im = sph_test_image_tools_create_flat_image_double(nx, ny, flat_counts);
        //add star as a gauss with center coords (cx+0.5, cy+0.5); +0.5 is important because of the difference
        //in the coord system between sphere and fits: sphere starts from (0,0)pixel whereas fits from (1,1)
        sph_test_image_tools_add_gauss(im, cx + 0.5, cy + 0.5, star_fwhm,
                star_counts + ii * 100000.0);
        p = sph_point_new(cx + 20.0, cy); //rotation point
        sph_point_rotate_around(p, cx + .5, cy + 0.5,
                cos(angle * CPL_MATH_RAD_DEG), sin(angle * CPL_MATH_RAD_DEG),
                0.0, 0.0);
        SPH_ERROR_RAISE_INFO(SPH_ERROR_INFO_MESSAGE,
                "p->x = %5.3f; p->y = %5.3f", p->x, p->y);
        sph_test_image_tools_add_gauss(im, p->x, p->y, planet_fwhm,
                planet_counts); //planet

        zplexpimg = sph_zpl_exposure_imaging_new_empty();
        //duplicate for 2 zimpol images (imaging mode)
        zplexpimg->image_odd = cpl_image_duplicate(im);
        zplexpimg->image_even = cpl_image_duplicate(im);
        cpl_image_add_scalar(zplexpimg->image_even, -100.0);
        cpl_image_delete(im);
        im = NULL;
        sph_point_delete(p);
        p = NULL;

        if (ii == 0) {
            //zexpframe = sph_filemanager_create_temp_frame("test_zplexpimg_science.fits", "NONE", CPL_FRAME_GROUP_NONE);
            ovsc_pl = sph_test_zpl_simulator_create_ovsc_propertylist_imaging(
                    ovsc_duplicate);
            zplexpimg->ovsc = cpl_propertylist_duplicate(ovsc_pl);
            cpl_propertylist_delete(ovsc_pl);
            ovsc_pl = NULL;
            ovsc_table = sph_zpl_exposure_imaging_ovsc_table_create_empty();
            rerr = sph_zpl_exposure_imaging_save_open(zplexpimg, zexpcube, NULL,
                    ovsc_table);
        } else {
            ovsc_pl = sph_test_zpl_simulator_create_ovsc_propertylist_imaging(
                    ovsc_duplicate);
            zplexpimg->ovsc = cpl_propertylist_duplicate(ovsc_pl);
            cpl_propertylist_delete(ovsc_pl);
            ovsc_pl = NULL;

            sph_zpl_exposure_imaging_save_append(zplexpimg, zexpcube,
                    ovsc_table);
            sph_zpl_exposure_imaging_delete(zplexpimg);
            zplexpimg = NULL;
        }

        if (rerr != CPL_ERROR_NONE) {
            SPH_ERR( "could not save zpl exposure imaging");
            return NULL;
        }
        //cpl_frameset_insert( zexpframes, zexpframe );
    }

    //rerr = sph_test_zpl_simulator_create_cube_from_exposure_imaging_frames_double( zexpframes, *zexpcube);

    sph_zpl_exposure_imaging_finalize_file(zexpcube, ovsc_table);
    cpl_table_delete(ovsc_table);

    return zexpcube;
}

void sph_test_simulator_create_zplpreproc_cube_set(cpl_frame** zexpcube,
        double zero_odd, double zero_even, double pi_odd, double pi_even,
        double dV, int nframes) {
    sph_zpl_exposure* zplexp = NULL;
    cpl_frame* zexpframe = NULL;
    cpl_frameset* zexpframes = NULL;
    //cpl_propertylist*    pl            = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;

    const int nx = 100;
    const int ny = 100;
    int nz;

    nz = nframes;
    //pl = cpl_propertylist_new();
    //cpl_propertylist_append_int( pl, SPH_ZPL_KEYWORD_CAMERA_NAME, SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID);
    *zexpcube = sph_filemanager_create_temp_frame(
            "test_zplexp_cube_minus.fits", "NONE", CPL_FRAME_GROUP_NONE);

    zexpframes = cpl_frameset_new();
    for (int i = 0; i < nz; i++) {
        zplexp = sph_test_zpl_simulator_create_exposure_4flat_double(nx, ny,
                zero_odd + i * dV, zero_even + i * dV, pi_odd + i * dV,
                pi_even + i * dV);

        zexpframe = sph_filemanager_create_temp_frame(
                "test_zplexp_single_frame_cam1_int_minus.fits", "NONE",
                CPL_FRAME_GROUP_NONE);
        rerr = sph_zpl_exposure_save(zplexp, zexpframe, NULL);

        if (rerr != CPL_ERROR_NONE) {
            SPH_ERR( "could not save zpl exposure");
            return;
        }
        sph_zpl_exposure_delete(zplexp);
        cpl_frameset_insert(zexpframes, zexpframe);
    }
    rerr = sph_test_zpl_simulator_create_cube_from_exposure_frames_int(
            zexpframes, *zexpcube);
    cpl_frameset_delete(zexpframes);

}

void sph_test_simulator_create_zplpreproc_cube_set_iff(cpl_frame** zexpcube,
        double zero_odd, double zero_even, double pi_odd, double pi_even,
        double dV, int nframes) {
    sph_zpl_exposure* zplexp = NULL;
    cpl_frame* zexpframe = NULL;
    cpl_frameset* zexpframes = NULL;
    cpl_propertylist* pl = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;

    const int nx = 100;
    const int ny = 100;
    int nz;

    nz = nframes;
    pl = cpl_propertylist_new();
    cpl_propertylist_append_int(pl, SPH_ZPL_KEYWORD_CAMERA_NAME,
            SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID);
    *zexpcube = sph_test_zpl_simulator_frame_template(
            "test_zplexp_cube_iff.fits");

    zexpframes = cpl_frameset_new();
    for (int i = 0; i < nz; i++) {
        zplexp = sph_test_zpl_simulator_create_exposure_4flat_double(nx, ny,
                zero_odd + i * dV, zero_even + 2 * i * dV, pi_odd + 3 * i * dV,
                pi_even + 4 * i * dV);

        zexpframe = sph_filemanager_create_temp_frame(
                "test_zplexp_single_frame_cam1_int_minus.fits", "NONE",
                CPL_FRAME_GROUP_NONE);
        rerr = sph_zpl_exposure_save(zplexp, zexpframe, NULL);

        if (rerr != CPL_ERROR_NONE) {
            SPH_ERR( "could not save zpl exposure");
            return;
        }
        cpl_frameset_insert(zexpframes, zexpframe);
    }
    rerr = sph_test_zpl_simulator_create_cube_from_exposure_frames_int(
            zexpframes, *zexpcube);

}

void sph_test_simulator_create_zplpreproc_cube_set_filename(
        cpl_frame** zexpcube, const char* filename, double zero_odd,
        double zero_even, double pi_odd, double pi_even, double dV, int nframes) {
    sph_zpl_exposure* zplexp = NULL;
    cpl_frame* zexpframe = NULL;
    cpl_frameset* zexpframes = NULL;
    cpl_propertylist* pl = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;

    const int nx = 100;
    const int ny = 100;
    int nz;

    nz = nframes;
    pl = cpl_propertylist_new();
    cpl_propertylist_append_int(pl, SPH_ZPL_KEYWORD_CAMERA_NAME,
            SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID);
    //cpl_propertylist_append_double( pl, SPH_ZPL_KEYWORD_HWP2_SETTING, hwp2 );
    *zexpcube = sph_test_zpl_simulator_frame_template(filename);
    //*zexpcube = sph_filemanager_create_temp_frame(filename, "NONE", CPL_FRAME_GROUP_NONE);

    zexpframes = cpl_frameset_new();
    for (int i = 0; i < nz; i++) {
        zplexp = sph_test_zpl_simulator_create_exposure_4flat_double(nx, ny,
                zero_odd + i * dV, zero_even + i * dV, pi_odd + i * dV,
                pi_even + i * dV);

        zexpframe = sph_filemanager_create_temp_frame(
                "test_zplexp_single_frame.fits", "NONE", CPL_FRAME_GROUP_NONE);
        rerr = sph_zpl_exposure_save(zplexp, zexpframe, pl);

        if (rerr != CPL_ERROR_NONE) {
            SPH_ERR( "could not save zpl exposure");
            return;
        }
        cpl_frameset_insert(zexpframes, zexpframe);
    }
    rerr = sph_test_zpl_simulator_create_cube_from_exposure_frames_int(
            zexpframes, *zexpcube);

}

void sph_test_simulator_create_zplpreproc_cube_set_hwpz(cpl_frame** zexpcube,
        const char* filename, double zero_odd, double zero_even, double pi_odd,
        double pi_even, double dV, double hwp2, int nframes) {
    sph_zpl_exposure* zplexp = NULL;
    cpl_frame* zexpframe = NULL;
    cpl_frameset* zexpframes = NULL;
    cpl_propertylist* pl = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;

    const int nx = 100;
    const int ny = 100;
    int nz;

    nz = nframes;
    pl = cpl_propertylist_new();
    cpl_propertylist_append_int(pl, SPH_ZPL_KEYWORD_CAMERA_NAME,
            SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID);
    cpl_propertylist_append_double(pl, SPH_ZPL_KEYWORD_HWPZ_SETTING, hwp2);
    *zexpcube = sph_filemanager_create_temp_frame( filename, "NONE", CPL_FRAME_GROUP_NONE);

    zexpframes = cpl_frameset_new();
    for (int i = 0; i < nz; i++) {
        zplexp = sph_test_zpl_simulator_create_exposure_4flat_double(nx, ny,
                zero_odd + i * dV, zero_even + i * dV, pi_odd + i * dV,
                pi_even + i * dV);

        zexpframe = sph_filemanager_create_temp_frame(
                "test_zplexp_single_frame.fits", "NONE", CPL_FRAME_GROUP_NONE);
        rerr = sph_zpl_exposure_save(zplexp, zexpframe, pl);

        if (rerr != CPL_ERROR_NONE) {
            SPH_ERR( "could not save zpl exposure");
            return;
        }
        cpl_frameset_insert(zexpframes, zexpframe);
    }
    rerr = sph_test_zpl_simulator_create_cube_from_exposure_frames_int(
            zexpframes, *zexpcube);

}

void sph_test_simulator_create_zplpreproc_cube_set_qplus(cpl_frame** zexpcube,
        const char* filename, double zero_odd, double zero_even, double pi_odd,
        double pi_even, double dV, int nframes) {
    sph_zpl_exposure* zplexp = NULL;
    cpl_frame* zexpframe = NULL;
    cpl_frameset* zexpframes = NULL;
    cpl_propertylist* pl = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;

    const int nx = 100;
    const int ny = 100;
    int nz;

    nz = nframes;
    pl = cpl_propertylist_new();
    cpl_propertylist_append_int(pl, SPH_ZPL_KEYWORD_CAMERA_NAME,
            SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID);
    cpl_propertylist_append_string(pl, SPH_ZPL_KEYWORD_POL_STOKES_NAME,
            SPH_ZPL_KEYWORD_VALUE_POL_STOKES_QPLUS);
    //*zexpcube = sph_test_zpl_simulator_frame_template(filename);
    *zexpcube = sph_filemanager_create_temp_frame( filename, "NONE", CPL_FRAME_GROUP_NONE);

    zexpframes = cpl_frameset_new();
    for (int i = 0; i < nz; i++) {
        zplexp = sph_test_zpl_simulator_create_exposure_4flat_double(nx, ny,
                zero_odd + i * dV, zero_even + i * dV, pi_odd + i * dV,
                pi_even + i * dV);

        zexpframe = sph_filemanager_create_temp_frame(
                "test_zplexp_single_frame.fits", "NONE", CPL_FRAME_GROUP_NONE);
        rerr = sph_zpl_exposure_save(zplexp, zexpframe, pl);

        if (rerr != CPL_ERROR_NONE) {
            SPH_ERR( "could not save zpl exposure");
            return;
        }
        cpl_frameset_insert(zexpframes, zexpframe);
        sph_zpl_exposure_delete(zplexp);
    }
    rerr = sph_test_zpl_simulator_create_cube_from_exposure_frames_int(
            zexpframes, *zexpcube);
    if (zexpframes)
        cpl_frameset_delete(zexpframes);
    if (pl)
        cpl_propertylist_delete(pl);
}

void sph_test_simulator_create_zplpreproc_cube_set_qminus(cpl_frame** zexpcube,
        const char* filename, double zero_odd, double zero_even, double pi_odd,
        double pi_even, double dV, int nframes) {
    sph_zpl_exposure* zplexp = NULL;
    cpl_frame* zexpframe = NULL;
    cpl_frameset* zexpframes = NULL;
    cpl_propertylist* pl = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;

    const int nx = 100;
    const int ny = 100;
    int nz;

    nz = nframes;
    pl = cpl_propertylist_new();
    cpl_propertylist_append_int(pl, SPH_ZPL_KEYWORD_CAMERA_NAME,
            SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID);
    cpl_propertylist_append_string(pl, SPH_ZPL_KEYWORD_POL_STOKES_NAME,
            SPH_ZPL_KEYWORD_VALUE_POL_STOKES_QMINUS);
    //*zexpcube = sph_test_zpl_simulator_frame_template(filename);
    *zexpcube = sph_filemanager_create_temp_frame( filename, "NONE", CPL_FRAME_GROUP_NONE);

    zexpframes = cpl_frameset_new();
    for (int i = 0; i < nz; i++) {
        zplexp = sph_test_zpl_simulator_create_exposure_4flat_double(nx, ny,
                zero_odd + i * dV, zero_even + i * dV, pi_odd + i * dV,
                pi_even + i * dV);

        zexpframe = sph_filemanager_create_temp_frame(
                "test_zplexp_single_frame.fits", "NONE", CPL_FRAME_GROUP_NONE);
        rerr = sph_zpl_exposure_save(zplexp, zexpframe, pl);

        if (rerr != CPL_ERROR_NONE) {
            SPH_ERR( "could not save zpl exposure");
            return;
        }
        cpl_frameset_insert(zexpframes, zexpframe);
        sph_zpl_exposure_delete(zplexp);
    }
    rerr = sph_test_zpl_simulator_create_cube_from_exposure_frames_int(
            zexpframes, *zexpcube);
    if (zexpframes)
        cpl_frameset_delete(zexpframes);
    if (pl)
        cpl_propertylist_delete(pl);

}

void sph_test_simulator_create_zplpreproc_cube_set_hwp2(cpl_frame** zexpcube,
        const char* filename, double zero_odd, double zero_even, double pi_odd,
        double pi_even, double dV, double hwp2, int nframes) {
    sph_zpl_exposure* zplexp = NULL;
    cpl_frame* zexpframe = NULL;
    cpl_frameset* zexpframes = NULL;
    cpl_propertylist* pl = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;

    const int nx = 100;
    const int ny = 100;
    int nz;

    nz = nframes;
    pl = cpl_propertylist_new();
    cpl_propertylist_append_int(pl, SPH_ZPL_KEYWORD_CAMERA_NAME,
            SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID);
    cpl_propertylist_append_double(pl, SPH_ZPL_KEYWORD_HWP2_SETTING, hwp2);
    //*zexpcube = sph_test_zpl_simulator_frame_template(filename);
    *zexpcube = sph_filemanager_create_temp_frame( filename, "NONE", CPL_FRAME_GROUP_NONE);

    zexpframes = cpl_frameset_new();
    for (int i = 0; i < nz; i++) {
        zplexp = sph_test_zpl_simulator_create_exposure_4flat_double(nx, ny,
                zero_odd + i * dV, zero_even + i * dV, pi_odd + i * dV,
                pi_even + i * dV);

        zexpframe = sph_filemanager_create_temp_frame(
                "test_zplexp_single_frame.fits", "NONE", CPL_FRAME_GROUP_NONE);
        rerr = sph_zpl_exposure_save(zplexp, zexpframe, pl);

        if (rerr != CPL_ERROR_NONE) {
            SPH_ERR( "could not save zpl exposure");
            return;
        }
        cpl_frameset_insert(zexpframes, zexpframe);
    }
    rerr = sph_test_zpl_simulator_create_cube_from_exposure_frames_int(
            zexpframes, *zexpcube);

}

void sph_test_simulator_create_zplpreproc_cube_minus(cpl_frame** zexpcube,
        int nframes) {
    sph_zpl_exposure* zplexp = NULL;
    cpl_frame* zexpframe = NULL;
    cpl_frameset* zexpframes = NULL;
    cpl_propertylist* pl = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;

    const int nx = 100;
    const int ny = 100;
    int zero_odd = 51;
    int zero_even = 151;
    int pi_odd = 151;
    int pi_even = 51;
    int dV = -1;
    int nz; //nframes

    nz = nframes;
    pl = cpl_propertylist_new();
    cpl_propertylist_append_int(pl, SPH_ZPL_KEYWORD_CAMERA_NAME,
            SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID);
    cpl_propertylist_append_string(pl, SPH_ZPL_KEYWORD_POL_STOKES_NAME,
            SPH_ZPL_KEYWORD_VALUE_POL_STOKES_QMINUS);
    *zexpcube = sph_filemanager_create_temp_frame("test_zplexp_cube_minus.fits", "NONE", CPL_FRAME_GROUP_NONE);

    zexpframes = cpl_frameset_new();
    for (int i = 0; i < nz; i++) {
        zplexp = sph_test_zpl_simulator_create_exposure_4flat_int(nx, ny,
                zero_odd + i * dV, zero_even + i * dV, pi_odd + i * dV,
                pi_even + i * dV);

        zexpframe = sph_filemanager_create_temp_frame(
                "test_zplexp_single_frame_cam1_int_minus.fits", "NONE",
                CPL_FRAME_GROUP_NONE);
        rerr = sph_zpl_exposure_save(zplexp, zexpframe, pl);

        if (rerr != CPL_ERROR_NONE) {
            SPH_ERR( "could not save zpl exposure");
            return;
        }
        sph_zpl_exposure_delete(zplexp);

        cpl_frameset_insert(zexpframes, zexpframe);
    }
    rerr = sph_test_zpl_simulator_create_cube_from_exposure_frames_int(
            zexpframes, *zexpcube);
    cpl_frameset_delete(zexpframes);
    cpl_propertylist_delete(pl);

}

void sph_test_simulator_create_zplpreproc_cube_plus(cpl_frame** zexpcube,
        int nframes) {
    sph_zpl_exposure* zplexp = NULL;
    cpl_frame* zexpframe = NULL;
    cpl_frameset* zexpframes = NULL;
    cpl_propertylist* pl = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;

    const int nx = 100;
    const int ny = 100;
    int zero_odd = 169;
    int zero_even = 29;
    int pi_odd = 29;
    int pi_even = 169;
    int dV = 1;
    int nz; //nframes

    nz = nframes;
    pl = cpl_propertylist_new();
    cpl_propertylist_append_int(pl, SPH_ZPL_KEYWORD_CAMERA_NAME,
            SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID);
    cpl_propertylist_append_string(pl, SPH_ZPL_KEYWORD_POL_STOKES_NAME,
            SPH_ZPL_KEYWORD_VALUE_POL_STOKES_QPLUS);
    //*zexpcube = sph_test_zpl_simulator_frame_template(
    //        "test_zplexp_cube_plus.fits");

    *zexpcube = sph_filemanager_create_temp_frame("test_zplexp_cube_plus.fits", "NONE", CPL_FRAME_GROUP_NONE);
    zexpframes = cpl_frameset_new();
    for (int i = 0; i < nz; i++) {
        zplexp = sph_test_zpl_simulator_create_exposure_4flat_int(nx, ny,
                zero_odd + i * dV, zero_even + i * dV, pi_odd + i * dV,
                pi_even + i * dV);

        zexpframe = sph_filemanager_create_temp_frame(
                "test_zplexp_single_frame_cam1_int_plus.fits", "NONE",
                CPL_FRAME_GROUP_NONE);
        rerr = sph_zpl_exposure_save(zplexp, zexpframe, pl);

        if (rerr != CPL_ERROR_NONE) {
            SPH_ERR( "could not save zpl exposure");
            return;
        }
        sph_zpl_exposure_delete(zplexp);

        cpl_frameset_insert(zexpframes, zexpframe);
    }
    rerr = sph_test_zpl_simulator_create_cube_from_exposure_frames_int(
            zexpframes, *zexpcube);
    cpl_frameset_delete(zexpframes);
    cpl_propertylist_delete(pl);
}

cpl_frame* sph_test_simulator_create_zpl_exposure_overscans_cube(int nx, int ny,
        int nz, const char* filename, double zero_odd, double zero_even,
        double pi_odd, double pi_even, double* ovsc, double dV, double dOVSC) {

    sph_zpl_exposure* zplexp = NULL;
    cpl_frame* zexpframe = NULL;
    cpl_propertylist* ovsc_pl = NULL;
    cpl_table* ovsc_table = NULL;
    double ovsc_duplicate[8];

    for (int kk = 0; kk < 8; kk++) {
        ovsc_duplicate[kk] = ovsc[kk];
    }

    zplexp = sph_test_zpl_simulator_create_exposure_4flat_double(nx, ny,
            zero_odd, zero_even, pi_odd, pi_even);

    zexpframe = sph_filemanager_create_temp_frame(filename, "PREPROC",
            CPL_FRAME_GROUP_NONE);
    ovsc_pl = sph_test_zpl_simulator_create_ovsc_propertylist(ovsc_duplicate);
    zplexp->ovsc = cpl_propertylist_duplicate(ovsc_pl);
    cpl_propertylist_delete(ovsc_pl);
    ovsc_pl = NULL;

    ovsc_table = sph_zpl_exposure_ovsc_table_create_empty();
    sph_zpl_exposure_save_open(zplexp, zexpframe, NULL, ovsc_table);
    sph_zpl_exposure_delete(zplexp);

    for (int ii = 1; ii < nz; ii++) {
        zplexp = sph_test_zpl_simulator_create_exposure_4flat_double(nx, ny,
                zero_odd + ii, zero_even + ii, pi_odd + ii, pi_even + ii);
        for (int kk = 0; kk < 8; kk = kk + 2) {
            ovsc_duplicate[kk] = ovsc[kk] + dOVSC * ii;
        }
        ovsc_pl = sph_test_zpl_simulator_create_ovsc_propertylist(
                ovsc_duplicate);
        zplexp->ovsc = cpl_propertylist_duplicate(ovsc_pl);
        cpl_propertylist_delete(ovsc_pl);

        sph_zpl_exposure_save_append(zplexp, zexpframe, ovsc_table);
        sph_zpl_exposure_delete(zplexp);
    }
    sph_zpl_exposure_finalize_file(zexpframe, ovsc_table);
    cpl_table_delete(ovsc_table);

    return zexpframe;

}

cpl_frame* sph_test_simulator_create_zpl_exposure_overscans_cube_set_hwpz(
        int nx, int ny, int nz, const char* filename, double zero_odd,
        double zero_even, double pi_odd, double pi_even, double* ovsc,
        double dV, double dOVSC, double hwp2) {

    sph_zpl_exposure* zplexp = NULL;
    cpl_frame* zexpframe = NULL;
    cpl_propertylist* ovsc_pl = NULL;
    cpl_propertylist* pl = NULL;
    cpl_table* ovsc_table = NULL;
    double ovsc_duplicate[8];

    for (int kk = 0; kk < 8; kk++) {
        ovsc_duplicate[kk] = ovsc[kk];
    }

    pl = cpl_propertylist_new();
    cpl_propertylist_append_int(pl, SPH_ZPL_KEYWORD_CAMERA_NAME,
            SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID);
    cpl_propertylist_append_double(pl, SPH_ZPL_KEYWORD_HWPZ_SETTING, hwp2);

    zplexp = sph_test_zpl_simulator_create_exposure_4flat_double(nx, ny,
            zero_odd, zero_even, pi_odd, pi_even);

    zexpframe = sph_filemanager_create_temp_frame(filename, "PREPROC",
            CPL_FRAME_GROUP_NONE);
    ovsc_pl = sph_test_zpl_simulator_create_ovsc_propertylist(ovsc_duplicate);
    zplexp->ovsc = cpl_propertylist_duplicate(ovsc_pl);
    cpl_propertylist_delete(ovsc_pl);
    ovsc_pl = NULL;

    ovsc_table = sph_zpl_exposure_ovsc_table_create_empty();
    sph_zpl_exposure_save_open(zplexp, zexpframe, pl, ovsc_table);
    sph_zpl_exposure_delete(zplexp);
    cpl_propertylist_delete(pl);

    for (int ii = 1; ii < nz; ii++) {
        zplexp = sph_test_zpl_simulator_create_exposure_4flat_double(nx, ny,
                zero_odd + ii, zero_even + ii, pi_odd + ii, pi_even + ii);
        for (int kk = 0; kk < 8; kk = kk + 2) {
            ovsc_duplicate[kk] = ovsc[kk] + dOVSC * ii;
        }
        ovsc_pl = sph_test_zpl_simulator_create_ovsc_propertylist(
                ovsc_duplicate);
        zplexp->ovsc = cpl_propertylist_duplicate(ovsc_pl);
        cpl_propertylist_delete(ovsc_pl);

        sph_zpl_exposure_save_append(zplexp, zexpframe, ovsc_table);
        sph_zpl_exposure_delete(zplexp);
    }
    sph_zpl_exposure_finalize_file(zexpframe, ovsc_table);
    cpl_table_delete(ovsc_table);

    return zexpframe;

}

cpl_frame* sph_test_simulator_create_zpl_exposure_imaging_overscans_cube(int nx,
        int ny, int nz, const char* filename, double iframe, double pframe,
        double* ovsc, double dV, double dOVSC) {

    sph_zpl_exposure_imaging* zplexp = NULL;
    cpl_frame* zexpframe = NULL;
    cpl_propertylist* ovsc_pl = NULL;
    cpl_table* ovsc_table = NULL;
    double ovsc_duplicate[4];

    for (int kk = 0; kk < 4; kk++) {
        ovsc_duplicate[kk] = ovsc[kk];
    }

    zplexp = sph_test_zpl_simulator_create_exposure_imaging_2flat_double(nx, ny,
            iframe, pframe);

    zexpframe = sph_filemanager_create_temp_frame(filename, "PREPROC",
            CPL_FRAME_GROUP_NONE);
    ovsc_pl = sph_test_zpl_simulator_create_ovsc_propertylist_imaging(
            ovsc_duplicate);
    if ( zplexp->ovsc ) cpl_propertylist_delete( zplexp->ovsc );
    zplexp->ovsc = cpl_propertylist_duplicate(ovsc_pl);
    cpl_propertylist_delete(ovsc_pl);
    ovsc_pl = NULL;

    ovsc_table = sph_zpl_exposure_imaging_ovsc_table_create_empty();
    sph_zpl_exposure_imaging_save_open(zplexp, zexpframe, NULL, ovsc_table);
    sph_zpl_exposure_imaging_delete(zplexp);

    for (int ii = 1; ii < nz; ii++) {
        zplexp = sph_test_zpl_simulator_create_exposure_imaging_2flat_double(nx,
                ny, iframe + ii, pframe + ii);
        for (int kk = 0; kk < 4; kk = kk + 2) {
            ovsc_duplicate[kk] = ovsc[kk] + dOVSC * ii;
        }
        ovsc_pl = sph_test_zpl_simulator_create_ovsc_propertylist_imaging(
                ovsc_duplicate);
        if ( zplexp->ovsc ) cpl_propertylist_delete( zplexp->ovsc );
        zplexp->ovsc = cpl_propertylist_duplicate(ovsc_pl);
        cpl_propertylist_delete(ovsc_pl);

        sph_zpl_exposure_imaging_save_append(zplexp, zexpframe, ovsc_table);
        sph_zpl_exposure_imaging_delete(zplexp);
    }
    sph_zpl_exposure_imaging_finalize_file(zexpframe, ovsc_table);
    cpl_table_delete(ovsc_table);

    return zexpframe;

}

void sph_test_simulator_create_zplpreproc_imaging_cube(cpl_frame** zexpcube,
        int nframes) {
    sph_zpl_exposure_imaging* zplexp = NULL;
    cpl_frame* zexpframe = NULL;
    cpl_frameset* zexpframes = NULL;
    cpl_propertylist* pl = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;

    const int nx = 100;
    const int ny = 100;
    double image_odd = 19.0;
    double image_even = 20.0;
    int dV = 1;
    int nz; //nframes

    nz = nframes;
    pl = cpl_propertylist_new();
    cpl_propertylist_append_int(pl, SPH_ZPL_KEYWORD_CAMERA_NAME,
            SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID);
    //*zexpcube = sph_test_zpl_simulator_frame_template( "test_zplexp_cube_imaging.fits");
    *zexpcube = sph_filemanager_create_temp_frame(
            "test_zplexp_imaging_cube.fits", "NONE", CPL_FRAME_GROUP_NONE);

    zexpframes = cpl_frameset_new();
    for (int i = 0; i < nz; i++) {
        zplexp = sph_test_zpl_simulator_create_exposure_imaging_2flat_double(nx,
                ny, image_odd + i * dV, image_even + i * dV);

        zexpframe = sph_filemanager_create_temp_frame(
                "test_zplexp_imaging_single_frame_cam1_double.fits", "NONE",
                CPL_FRAME_GROUP_NONE);
        rerr = sph_zpl_exposure_imaging_save(zplexp, zexpframe, NULL);

        if (rerr != CPL_ERROR_NONE) {
            SPH_ERR( "could not save zpl exposure imaging");
            return;
        }
        sph_zpl_exposure_imaging_delete(zplexp);

        cpl_frameset_insert(zexpframes, zexpframe);
    }
    rerr =
            sph_test_zpl_simulator_create_cube_from_exposure_imaging_frames_double(
                    zexpframes, *zexpcube);
    cpl_frameset_delete(zexpframes);
    cpl_propertylist_delete(pl);

}

void sph_test_simulator_create_zplpreproc_imaging_cube_ra_dec_time(
        cpl_frame** zexpcube, int nframes, float ra, float dec, double mjd_obs,
        const char* date_obs, double utc, double dit, int curid) {
    sph_zpl_exposure_imaging* zplexp = NULL;
    cpl_frame* zexpframe = NULL;
    cpl_frameset* zexpframes = NULL;
    cpl_propertylist* pl = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;

    const int nx = 100;
    const int ny = 100;
    double image_odd = 19.0;
    double image_even = 20.0;
    int dV = 1;
    int nz; //nframes

    nz = nframes;
    pl = cpl_propertylist_new();
    cpl_propertylist_append_int(pl, SPH_ZPL_KEYWORD_CAMERA_NAME,
            SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID);
    cpl_propertylist_append_float(pl, SPH_COMMON_KEYWORD_RA, ra);
    cpl_propertylist_append_float(pl, SPH_COMMON_KEYWORD_DEC, dec);
    cpl_propertylist_append_double(pl, SPH_COMMON_KEYWORD_MJD_OBS, mjd_obs);
    cpl_propertylist_append_string(pl, SPH_COMMON_KEYWORD_DATE_OBS, date_obs);
    cpl_propertylist_append_double(pl, SPH_COMMON_KEYWORD_UTC, utc);
    cpl_propertylist_append_double(pl, SPH_COMMON_KEYWORD_DIT, dit);
    cpl_propertylist_append_int(pl, SPH_COMMON_KEYWORD_READ_CURID, curid);
    cpl_propertylist_append_string( pl, SPH_COMMON_KEYWORD_DROT2_MODE,
                                        SPH_COMMON_KEYWORD_VALUE_DROT2_MODE_PUPIL_STAB );


    //*zexpcube = sph_test_zpl_simulator_frame_template( "test_zplexp_cube_imaging.fits");
    *zexpcube = sph_filemanager_create_temp_frame(
            "test_zplexp_imaging_cube.fits", "NONE", CPL_FRAME_GROUP_NONE);

    zexpframes = cpl_frameset_new();
    for (int i = 0; i < nz; i++) {
        zplexp = sph_test_zpl_simulator_create_exposure_imaging_2flat_double(nx,
                ny, image_odd + i * dV, image_even + i * dV);

        zexpframe = sph_filemanager_create_temp_frame(
                "test_zplexp_imaging_single_frame_cam1_double.fits", "NONE",
                CPL_FRAME_GROUP_NONE);
        rerr = sph_zpl_exposure_imaging_save(zplexp, zexpframe, pl);

        if (rerr != CPL_ERROR_NONE) {
            SPH_ERR( "could not save zpl exposure imaging");
            return;
        }
        sph_zpl_exposure_imaging_delete(zplexp);

        cpl_frameset_insert(zexpframes, zexpframe);
    }
    rerr =
            sph_test_zpl_simulator_create_cube_from_exposure_imaging_frames_double(
                    zexpframes, *zexpcube);
    cpl_frameset_delete(zexpframes);
    cpl_propertylist_delete(pl);

}

#ifdef SPH_USE_ZPL_SIMULATOR
void sph_test_simulator_create_zplpreproc_imaging_cube_ra_dec_time_interp(
        cpl_frame** zexpcube, int nframes, float ra, float dec, double dit,
        const char* frame_utc) {
    sph_zpl_exposure_imaging* zplexp = NULL;
    cpl_frame* zexpframe = NULL;
    cpl_frameset* zexpframes = NULL;
    cpl_propertylist* pl = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;

    const int nx = 100;
    const int ny = 100;
    double image_odd = 19.0;
    double image_even = 20.0;
    int dV = 1;
    int nz; //nframes

    nz = nframes;
    pl = cpl_propertylist_new();
    cpl_propertylist_append_int(pl, SPH_ZPL_KEYWORD_CAMERA_NAME,
            SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID);
    cpl_propertylist_append_float(pl, SPH_COMMON_KEYWORD_RA, ra);
    cpl_propertylist_append_float(pl, SPH_COMMON_KEYWORD_DEC, dec);
    cpl_propertylist_append_double(pl, SPH_COMMON_KEYWORD_DIT, dit);
    cpl_propertylist_append_string(pl, SPH_COMMON_KEYWORD_DET_FRAM_UTC,
            frame_utc);

    //*zexpcube = sph_test_zpl_simulator_frame_template( "test_zplexp_cube_imaging.fits");
    *zexpcube = sph_filemanager_create_temp_frame(
            "test_zplexp_imaging_cube_time_interp.fits", "NONE", CPL_FRAME_GROUP_NONE);

    zexpframes = cpl_frameset_new();
    for (int i = 0; i < nz; i++) {
        zplexp = sph_test_zpl_simulator_create_exposure_imaging_2flat_double(nx,
                ny, image_odd + i * dV, image_even + i * dV);

        zexpframe = sph_filemanager_create_temp_frame(
                "test_zplexp_imaging_single_frame_cam1_double_time_interp.fits", "NONE",
                CPL_FRAME_GROUP_NONE);
        rerr = sph_zpl_exposure_imaging_save(zplexp, zexpframe, pl);

        if (rerr != CPL_ERROR_NONE) {
            SPH_ERR( "could not save zpl exposure imaging");
            return;
        }
        sph_zpl_exposure_imaging_delete(zplexp);

        cpl_frameset_insert(zexpframes, zexpframe);
    }
    rerr =
            sph_test_zpl_simulator_create_cube_from_exposure_imaging_frames_double(
                    zexpframes, *zexpcube);
    cpl_frameset_delete(zexpframes);
    cpl_propertylist_delete(pl);

}
#endif

void sph_test_simulator_create_zplpreproc_imaging_cube_set(cpl_frame** zexpcube,
        double image_odd, double image_even, double dV, int nframes) {
    sph_zpl_exposure_imaging* zplexp = NULL;
    cpl_frame* zexpframe = NULL;
    cpl_frameset* zexpframes = NULL;
    cpl_propertylist* pl = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;

    const int nx = 100;
    const int ny = 100;
    // double                image_odd     = 19.0;
    // double                image_even    = 20.0;
    //int                    dV            = 1;
    int nz; //nframes

    nz = nframes;
    pl = cpl_propertylist_new();
    cpl_propertylist_append_int(pl, SPH_ZPL_KEYWORD_CAMERA_NAME,
            SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID);
    //*zexpcube = sph_test_zpl_simulator_frame_template( "test_zplexp_cube_imaging.fits");
    *zexpcube = sph_filemanager_create_temp_frame(
            "test_zplexp_imaging_cube.fits", "NONE", CPL_FRAME_GROUP_NONE);

    zexpframes = cpl_frameset_new();
    for (int i = 0; i < nz; i++) {
        zplexp = sph_test_zpl_simulator_create_exposure_imaging_2flat_double(nx,
                ny, image_odd + i * dV, image_even + i * dV);

        zexpframe = sph_filemanager_create_temp_frame(
                "test_zplexp_imaging_single_frame_cam1_double.fits", "NONE",
                CPL_FRAME_GROUP_NONE);
        rerr = sph_zpl_exposure_imaging_save(zplexp, zexpframe, NULL);

        if (rerr != CPL_ERROR_NONE) {
            SPH_ERR( "could not save zpl exposure imaging");
            return;
        }
        sph_zpl_exposure_imaging_delete(zplexp);

        cpl_frameset_insert(zexpframes, zexpframe);
    }
    rerr =
            sph_test_zpl_simulator_create_cube_from_exposure_imaging_frames_double(
                    zexpframes, *zexpcube);
    cpl_frameset_delete(zexpframes);
    cpl_propertylist_delete(pl);

    return;
}

void sph_test_simulator_create_zplpreproc_frames(cpl_frameset** zexpframes,
        int nframes) {
    sph_zpl_exposure* zplexp = NULL;
    cpl_frame* zexpframe = NULL;
    //cpl_frameset*        zexpframes    = NULL;
    cpl_propertylist* pl = NULL;

    sph_error_code rerr = CPL_ERROR_NONE;

    const int nx = 100;
    const int ny = 100;
    int zero_odd = 169;
    int zero_even = 29;
    int pi_odd = 29;
    int pi_even = 169;
    int dV = 1;

    int nz; //nframes

    nz = nframes;
    pl = cpl_propertylist_new();
    cpl_propertylist_append_int(pl, SPH_ZPL_KEYWORD_CAMERA_NAME,
            SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID);

    *zexpframes = cpl_frameset_new();
    for (int i = 0; i < nz; i++) {
        zplexp = sph_test_zpl_simulator_create_exposure_4flat_int(nx, ny,
                zero_odd + i * dV, zero_even + i * dV, pi_odd + i * dV,
                pi_even + i * dV);

        zexpframe = sph_filemanager_create_temp_frame(
                "test_zplexp_single_frame_cam1_int_plus.fits", "NONE",
                CPL_FRAME_GROUP_NONE);
        rerr = sph_zpl_exposure_save(zplexp, zexpframe, NULL);

        if (rerr != CPL_ERROR_NONE) {
            SPH_ERR( "could not save zpl exposure");
            return;
        }
        sph_zpl_exposure_delete(zplexp);
        zplexp = NULL;
        cpl_frameset_insert(*zexpframes, zexpframe);
    }
    cpl_propertylist_delete(pl);
    pl = NULL;
}

cpl_frameset* sph_test_simulator_create_quad_image_frames_from_zplexp_frames(void) {
    cpl_frameset* zexpframes = NULL;
    cpl_frameset* quadframes = NULL;
    const int nframes = 3;

    sph_test_simulator_create_zplpreproc_frames(&zexpframes, nframes);
    quadframes = sph_quad_image_create_quad_image_frameset_from_zplexp_frameset(
            zexpframes);

    if (!quadframes) {
        SPH_ERR( "could not create quad image frames from zplexp frames");
        return NULL;
    }

    cpl_frameset_delete(zexpframes);
    return quadframes;
}

#ifdef SPH_USE_ZPL_SIMULATOR
void sph_test_simulator_create_zplpreproc_imaging_frames(
        cpl_frameset** zexpframes, int nframes) {
    sph_zpl_exposure_imaging* zplexp = NULL;
    cpl_frame* zexpframe = NULL;
    //cpl_frameset*        zexpframes    = NULL;
    cpl_propertylist* pl = NULL;

    sph_error_code rerr = CPL_ERROR_NONE;

    const int nx = 100;
    const int ny = 100;
    int image_odd = 169;
    int image_even = 29;
    int dV = 1;

    int nz; //nframes

    nz = nframes;
    pl = cpl_propertylist_new();
    cpl_propertylist_append_int(pl, SPH_ZPL_KEYWORD_CAMERA_NAME,
            SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID);

    *zexpframes = cpl_frameset_new();
    for (int i = 0; i < nz; i++) {
        zplexp = sph_test_zpl_simulator_create_exposure_imaging_2flat_double(nx,
                ny, image_odd + i * dV, image_even + i * dV);

        zexpframe = sph_filemanager_create_temp_frame(
                "test_zplexp_imaging_single_frame_cam1_double.fits", "NONE",
                CPL_FRAME_GROUP_NONE);
        rerr = sph_zpl_exposure_imaging_save(zplexp, zexpframe, NULL);

        if (rerr != CPL_ERROR_NONE) {
            SPH_ERR( "could not save zpl exposure imaging");
            return;
        }
        cpl_frameset_insert(*zexpframes, zexpframe);
    }
    cpl_propertylist_delete(pl);
    pl = NULL;
}

cpl_frameset* sph_test_simulator_create_quad_image_frames_from_zplexp_cubes(void) {
    cpl_frame* zexpcube_plus = NULL;
    cpl_frame* zexpcube_minus = NULL;
    cpl_frameset* zexpcubes = NULL;
    cpl_frameset* quadframes = NULL;
    const int nframes = 3;

    zexpcubes = cpl_frameset_new();

    sph_test_simulator_create_zplpreproc_cube_plus(&zexpcube_plus, nframes);
    cpl_frameset_insert(zexpcubes, zexpcube_plus);

    sph_test_simulator_create_zplpreproc_cube_plus(&zexpcube_minus, nframes);
    cpl_frameset_insert(zexpcubes, zexpcube_minus);

    quadframes = sph_quad_image_create_quad_image_frameset_from_zplexp_cubes(
            zexpcubes);

    cpl_frameset_delete(zexpcubes);

    if (!quadframes) {
        SPH_ERR( "could not create quad image frames from zplexp cubes");
        return NULL;
    }

    return quadframes;
}
#endif

sph_double_image*
sph_test_zpl_simulator_create_double_image(int nx, int ny, double iframe_value,
        double pframe_value) {
    sph_double_image* di = NULL;
    cpl_image* i_image = NULL;
    cpl_image* p_image = NULL;
    sph_master_frame* iframe = NULL;
    sph_master_frame* pframe = NULL;

    //di = sph_double_image_new( nx, ny );
    //if ( di == NULL ){
    //    SPH_RAISE_CPL;
    //    return NULL;
    //}
    //di->iframe->image = sph_test_image_tools_create_flat_image_double( nx, ny, iframe_value);
    //di->pframe->image = sph_test_image_tools_create_flat_image_double( nx, ny, pframe_value);

    i_image = sph_test_image_tools_create_flat_image_double(nx, ny,
            iframe_value);
    p_image = sph_test_image_tools_create_flat_image_double(nx, ny,
            pframe_value);

    iframe = sph_master_frame_new_from_cpl_image(i_image);
    pframe = sph_master_frame_new_from_cpl_image(p_image);
    cpl_image_delete(i_image);
    cpl_image_delete(p_image);

    di = sph_double_image_new_from_master_frames(iframe, pframe);

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        SPH_RAISE_CPL;
        return NULL;
    }
    return di;
}

cpl_frame*
sph_test_zpl_simulator_create_double_image_frame(const char* filename, int nx,
        int ny, double val1, double val2) {
    cpl_frame* result = NULL;
    sph_double_image* dimage = NULL;
    cpl_propertylist* pl = NULL;
    double mjd_obs = 56799.00106531; // 2014-05-22T00:01:32.0429
    const char* frameut_string = "2014-05-22T00:04:57.0520";

    pl = cpl_propertylist_new();
    cpl_propertylist_update_double(pl, SPH_COMMON_KEYWORD_MJD_OBS, mjd_obs);
    cpl_propertylist_update_string(pl, SPH_COMMON_KEYWORD_DET_FRAM_UTC,
            frameut_string);
    dimage = sph_test_zpl_simulator_create_double_image(nx, ny, val1, val2);
    sph_double_image_save(dimage, filename, pl, CPL_IO_CREATE);

    sph_double_image_delete(dimage);
    cpl_propertylist_delete(pl);

    result = sph_test_zpl_simulator_frame_template(filename);

    return result;
}

sph_double_image*
sph_test_zpl_simulator_create_double_image_noise_uniform(int nx, int ny,
        double iframe_value, double pframe_value, double sigma_iframe,
        double sigma_pframe) {
    sph_double_image* di = NULL;
    sph_master_frame* iframe_image = NULL;
    sph_master_frame* pframe_image = NULL;
    cpl_image* i_image = NULL;
    cpl_image* p_image = NULL;

    //di = sph_double_image_new(nx, ny);

    i_image = sph_test_image_tools_create_flat_image_double(nx, ny,
            iframe_value);
    p_image = sph_test_image_tools_create_flat_image_double(nx, ny,
            pframe_value);

    iframe_image = sph_master_frame_new_from_cpl_image(i_image);
    pframe_image = sph_master_frame_new_from_cpl_image(p_image);
    cpl_image_delete(i_image);
    cpl_image_delete(p_image);

    di = sph_double_image_new_from_master_frames(iframe_image, pframe_image);

    sph_test_image_tools_add_noise(di->iframe->image, sigma_iframe, NULL);
    sph_test_image_tools_add_noise(di->pframe->image, sigma_pframe, NULL);

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        SPH_RAISE_CPL;
        return NULL;
    }
    return di;
}

sph_triple_image*
sph_test_zpl_simulator_create_triple_image(int nx, int ny, double val1,
        double val2, double val3) {
    sph_triple_image* result = NULL;
    sph_master_frame* mframe1 = NULL;
    sph_master_frame* mframe2 = NULL;
    sph_master_frame* mframe3 = NULL;

    mframe1 = sph_test_zpl_simulator_create_master_frame(nx, ny, val1);
    mframe2 = sph_test_zpl_simulator_create_master_frame(nx, ny, val2);
    mframe3 = sph_test_zpl_simulator_create_master_frame(nx, ny, val3);

    result = sph_triple_image_new_from_master_frames(mframe1, mframe2, mframe3);
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        SPH_RAISE_CPL;
        return NULL;
    }

    return result;
}

sph_quad_image*
sph_test_zpl_simulator_create_quad_image(int nx, int ny, double val1,
        double val2, double val3, double val4) {
    sph_quad_image* result = NULL;
    sph_master_frame* mframe1 = NULL;
    sph_master_frame* mframe2 = NULL;
    sph_master_frame* mframe3 = NULL;
    sph_master_frame* mframe4 = NULL;

    mframe1 = sph_test_zpl_simulator_create_master_frame(nx, ny, val1);
    if (!mframe1) {
        SPH_ERR("mframe1 is not created, null pointer")
    }
    mframe2 = sph_test_zpl_simulator_create_master_frame(nx, ny, val2);
    if (!mframe2) {
        SPH_ERR("mframe2 is not created, null pointer")
    }
    mframe3 = sph_test_zpl_simulator_create_master_frame(nx, ny, val3);
    if (!mframe3) {
        SPH_ERR("mframe3 is not created, null pointer")
    }
    mframe4 = sph_test_zpl_simulator_create_master_frame(nx, ny, val4);
    if (!mframe4) {
        SPH_ERR("mframe4 is not created, null pointer")
    }

    result = sph_quad_image_new_from_master_frames(mframe1, mframe2, mframe3,
            mframe4);

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        SPH_RAISE_CPL;
        return NULL;
    }

    return result;
}

cpl_frame*
sph_test_zpl_simulator_create_quad_image_frame(const char* filename, int nx,
        int ny, double val1, double val2, double val3, double val4) {
    cpl_frame* result = NULL;
    sph_quad_image* qimage = NULL;

    qimage = sph_test_zpl_simulator_create_quad_image(nx, ny, val1, val2, val3,
            val4);
    sph_quad_image_save(qimage, filename, NULL);
    result = sph_test_zpl_simulator_frame_template(filename);

    return result;
}

sph_master_frame*
sph_test_zpl_simulator_create_master_frame(int nx, int ny, double value) {
    sph_master_frame* mframe;
    mframe = sph_master_frame_new(nx, ny);
    cpl_image_delete(mframe->badpixelmap);
    mframe->badpixelmap = NULL;
    mframe->badpixelmap = sph_test_image_tools_create_flat_image_int(nx, ny, 0);
    cpl_image_delete(mframe->rmsmap);
    mframe->rmsmap = NULL;
    mframe->rmsmap = sph_test_image_tools_create_flat_image_double(nx, ny,
            value);
    cpl_image_delete(mframe->ncombmap);
    mframe->ncombmap = NULL;
    mframe->ncombmap = sph_test_image_tools_create_flat_image_double(nx, ny,
            value);
    cpl_image_delete(mframe->image);
    mframe->image = NULL;
    mframe->image = sph_test_image_tools_create_flat_image_double(nx, ny,
            value);
    return mframe;
}

cpl_frame*
sph_test_zpl_simulator_create_masterframe_frame(const char* filename, int nx,
        int ny, double val) {
    cpl_frame* result = NULL;
    sph_master_frame* mframe = NULL;

    mframe = sph_test_zpl_simulator_create_master_frame(nx, ny, val);
    sph_master_frame_save(mframe, filename, NULL);
    result = sph_test_zpl_simulator_frame_template(filename);

    return result;
}

