/* $Id: $
 *
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_ifs_master_dark.h"
#include "sph_common_keywords.h"
#include "sph_error.h"
#include "sph_test.h"
#include "sph_utils.h"

#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <float.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup  cutest_ifs_master_dark  Unit test of ifs_master_dark recipe and \
 *                                       associated functions.
 *
 * This tests verifies the very basic requirements of the sph_ifs_master_dark
 * recipe. All functions of the recipe object are tested in only one,
 * standard, test scenario. In this scenario, the parameters for the
 * recipe are the standard default parameters. The input raw frames
 * are created using the sph_test_create_ifs_raw_dark_frameset function.
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

static
int cutest_init_master_dark_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    return sph_test_nop_code();
}

static
int cutest_clean_master_dark_testsuite(void) {
    return sph_end_test();
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Common setup for the unit tests
 *
 * @param    frames        Address of pointer to the input frames to set up
 * @param   params        Address of pointer to the input parameterlist to set up
 *
 * This sets up the standard recipe parameter list and the input frames.
 * The frames an params pointers should not be initiallised yet.
 *
 * On return the pointers will point to the corresponding cpl structures that
 * are now filled with the setup data for the test.
 *
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ifs_master_dark_common_setup(cpl_frameset** frames,
        cpl_parameterlist** params, int nframes) {
    sph_error_code rerr = CPL_ERROR_NONE;
    cpl_error_reset();
    cpl_test_nonnull( frames );
    cpl_test_nonnull( params );

    *frames = cpl_frameset_new();
    rerr = sph_test_create_ifs_raw_dark_frameset(*frames, nframes);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    *params = sph_ifs_master_dark_create_paramlist();
    // PIPE-9384 - manually alter the master_dark output file name to avoid clashes between tests
    cpl_parameter* master_dark_param = cpl_parameterlist_find(*params, "ifs.master_dark.outfilename");
    cpl_parameter_set_string(master_dark_param, "master_dark_cutest_ifs_master_dark.fits");
//    printf("cutest_ifs_master_dark_common_setup parameters:\n");
//    cpl_parameterlist_dump(*params, stdout);

    rerr = cpl_error_get_code();
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_nonnull( *frames );
    cpl_test_nonnull( *params );
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Common teardown for the unit tests
 *
 * @param    frames        pointer to the input frames to clean up
 * @param   params        pointer to the input parameterlist to clean up
 *
 * This cleans up the frameset and parameterlist
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ifs_master_dark_common_teardown(cpl_frameset* frames,
        cpl_parameterlist* params) {

    cpl_test_nonnull( frames );
    cpl_test_nonnull( params );

    cpl_parameterlist_delete(params);
    sph_test_frameset_delete(frames);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_ifs_master_dark check frames function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ifs_master_dark_check_frames(void) {
    sph_ifs_master_dark* ifs_master_dark = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;

    /* Setup */
    cutest_ifs_master_dark_common_setup(&inframes, &inparams, 3);

    ifs_master_dark = sph_ifs_master_dark_new(inframes, inparams);
    cpl_test_nonnull( ifs_master_dark );
    // Delete some stuff to avoid duplicates and memleaks.
    cpl_frameset_delete(ifs_master_dark->current_raw_frameset);
    cpl_frameset_delete(ifs_master_dark->rawframes);

    /* Running code... */
    rerr = sph_ifs_master_dark_check_frames(ifs_master_dark);

    /* Verification */
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    cpl_test_error(CPL_ERROR_NONE);

    cpl_test_nonnull( ifs_master_dark->inframes );

    cpl_test_eq(cpl_frameset_get_size( ifs_master_dark->inframes), 3);

    /* Terdown */
    sph_ifs_master_dark_delete(ifs_master_dark);
    cutest_ifs_master_dark_common_teardown(inframes, inparams);
    return;

}
static
void cutest_ifs_master_dark_check_params(void) {
    sph_ifs_master_dark* ifs_master_dark = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;

    /* Setup */
    cutest_ifs_master_dark_common_setup(&inframes, &inparams, 3);
    cpl_test_error(CPL_ERROR_NONE);

    ifs_master_dark = sph_ifs_master_dark_new(inframes, inparams);
    cpl_test_error(CPL_ERROR_NONE);

    cpl_test_nonnull( ifs_master_dark );

    /* Running ... */

    rerr = sph_ifs_master_dark_check_params(ifs_master_dark);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    cpl_test_nonnull( ifs_master_dark->master_dark_outfilename );

    cpl_test_nonnull( ifs_master_dark->static_badpix_filename );

    cpl_test_nonnull( ifs_master_dark->framecomb_parameterlist );

    cpl_test_eq_string( ifs_master_dark->master_dark_outfilename,
            "master_dark_cutest_ifs_master_dark.fits");

    cpl_test_eq_string( ifs_master_dark->static_badpix_filename,
            "static_badpixels.fits");

    cpl_test_eq(
            cpl_parameterlist_get_size( ifs_master_dark->framecomb_parameterlist),
            2);
    /* Terdown */
    cutest_ifs_master_dark_common_teardown(inframes, inparams);
    sph_ifs_master_dark_delete(ifs_master_dark);
    return;

}

static
void cutest_ifs_master_dark_new(void) {
    sph_ifs_master_dark* ifs_master_dark = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;

    cutest_ifs_master_dark_common_setup(&inframes, &inparams, 3);

    ifs_master_dark = sph_ifs_master_dark_new(inframes, inparams);

    cpl_test_assert( ifs_master_dark);
    /* Terdown */
    cutest_ifs_master_dark_common_teardown(inframes, inparams);
    sph_ifs_master_dark_delete(ifs_master_dark);
    return;
}

static
void cutest_ifs_master_dark_delete(void) {
    sph_ifs_master_dark* ifs_master_dark = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;

    /* Setup */
    cutest_ifs_master_dark_common_setup(&inframes, &inparams, 3);

    ifs_master_dark = sph_ifs_master_dark_new(inframes, inparams);

    cpl_test_nonnull( ifs_master_dark );
    /* Run code... */
    rerr = sph_ifs_master_dark_delete(ifs_master_dark);
    /* Verify */
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    cpl_test_error(CPL_ERROR_NONE);
    /* Terdown */
    cutest_ifs_master_dark_common_teardown(inframes, inparams);

    return;
}

static
void cutest_ifs_master_dark_run(void) {
    sph_ifs_master_dark* ifs_master_dark = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    cpl_image* bpixs = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;

    /* Setup */
    cutest_ifs_master_dark_common_setup(&inframes, &inparams, 5);

    ifs_master_dark = sph_ifs_master_dark_new(inframes, inparams);

    cpl_test_nonnull( ifs_master_dark );

    /* Running */
    rerr = sph_ifs_master_dark_run(ifs_master_dark);

    /* Verification */
    cpl_test_eq_error( rerr, CPL_ERROR_NONE);

    bpixs = cpl_image_load(ifs_master_dark->master_dark_outfilename,
            CPL_TYPE_INT, 0, 1);
    cpl_test_nonnull( bpixs );
    cpl_test_lt(0, cpl_image_get_absflux( bpixs ));

    cpl_test_error(CPL_ERROR_NONE);
    cpl_image_delete(bpixs);
    bpixs = NULL;
    /* Terdown */
    cutest_ifs_master_dark_common_teardown(inframes, inparams);
    sph_ifs_master_dark_delete(ifs_master_dark);
    return;
}

static
void cutest_ifs_master_dark_returns_RON(void) {
    sph_ifs_master_dark* ifs_master_dark = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    cpl_image* bpixs = NULL;
    cpl_propertylist* pl = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;

    /* Setup */
    cutest_ifs_master_dark_common_setup(&inframes, &inparams, 5);
    cpl_test_error(CPL_ERROR_NONE);

    ifs_master_dark = sph_ifs_master_dark_new(inframes, inparams);
    cpl_test_error(CPL_ERROR_NONE);

    cpl_test_nonnull( ifs_master_dark );

    /* Running */
    rerr = sph_ifs_master_dark_run(ifs_master_dark);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    bpixs = cpl_image_load(ifs_master_dark->master_dark_outfilename,
            CPL_TYPE_INT, 0, 1);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( bpixs );
    cpl_test_lt(0, cpl_image_get_absflux( bpixs ));

    pl = cpl_propertylist_load(ifs_master_dark->master_dark_outfilename, 0);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test( cpl_propertylist_has(pl,"ESO QC MEAN RON"));
    cpl_test_error(CPL_ERROR_NONE);
    cpl_image_delete(bpixs);
    bpixs = NULL;
    /* Teardown */
    cutest_ifs_master_dark_common_teardown(inframes, inparams);
    sph_ifs_master_dark_delete(ifs_master_dark);
    cpl_propertylist_delete(pl);
    return;
}

static
void cutest_ifs_master_dark_run_test_illegal_reject(void) {
    sph_ifs_master_dark* ifs_master_dark = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    cpl_parameter* p = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;

    /* Setup */
    cutest_ifs_master_dark_common_setup(&inframes, &inparams, 5);

    ifs_master_dark = sph_ifs_master_dark_new(inframes, inparams);

    cpl_test_nonnull( ifs_master_dark );

    ifs_master_dark->clean_mean_reject_high = 2;
    ifs_master_dark->clean_mean_reject_low = 1;
    p = cpl_parameterlist_find(ifs_master_dark->framecomb_parameterlist,
            "clean_mean.reject_low");
    cpl_test_nonnull( p );
    cpl_parameter_set_int(p, ifs_master_dark->clean_mean_reject_low);
    p = cpl_parameterlist_find(ifs_master_dark->framecomb_parameterlist,
            "clean_mean.reject_high");
    cpl_test_nonnull( p );
    cpl_parameter_set_int(p, ifs_master_dark->clean_mean_reject_high);

    /* Running */
    rerr = sph_ifs_master_dark_run(ifs_master_dark);

    /* Verification */
    cpl_test_eq_error( rerr, CPL_ERROR_NONE);
    sph_ifs_master_dark_delete(ifs_master_dark);

    /* Terdown */
    cutest_ifs_master_dark_common_teardown(inframes, inparams);
    return;
}

static
void cutest_ifs_master_dark_run_more_than50(void) {
    sph_ifs_master_dark* ifs_master_dark = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;

    /* Setup */
    inframes = cpl_frameset_new();
    rerr = sph_test_create_ifs_raw_dark_frameset_big(inframes, 50);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    inparams = sph_ifs_master_dark_create_paramlist();

    ifs_master_dark = sph_ifs_master_dark_new(inframes, inparams);

    cpl_test_nonnull( ifs_master_dark );

    /* Running */
    rerr = sph_ifs_master_dark_run(ifs_master_dark);

    /* Verification */
    cpl_test_eq_error( rerr, CPL_ERROR_NONE);
    sph_ifs_master_dark_delete(ifs_master_dark);

    /* Terdown */
    cutest_ifs_master_dark_common_teardown(inframes, inparams);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit tests of techcal_master_dark recipe and associated functions
 */
/*----------------------------------------------------------------------------*/
int main(void) {

    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("master_dark_test", cutest_init_master_dark_testsuite,
            cutest_clean_master_dark_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "sph_ifs_master_dark_new",
                    cutest_ifs_master_dark_new)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_ifs_master_dark_check_frames",
                    cutest_ifs_master_dark_check_frames)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_ifs_master_dark_check_params",
                    cutest_ifs_master_dark_check_params)) {
        return sph_test_get_error();
    }
    if (cpl_msg_get_level() <= CPL_MSG_DEBUG) {
        if ( NULL == sph_test_do(pSuite, "sph_ifs_master_dark_run_more_than50",
                                 cutest_ifs_master_dark_run_more_than50 ) )
            {
                return sph_test_get_error();
            }
    }
    if (NULL
            == sph_test_do(pSuite, "sph_ifs_master_dark_returns_RON",
                    cutest_ifs_master_dark_returns_RON)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_ifs_master_dark_run",
                    cutest_ifs_master_dark_run)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "cutest_ifs_master_dark_run_test_illegal_reject",
                    cutest_ifs_master_dark_run_test_illegal_reject)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_ifs_master_dark_delete",
                    cutest_ifs_master_dark_delete)) {
        return sph_test_get_error();
    }

    return sph_test_end();
}

/**@}*/
