/* $Id: $
 *
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_ifs_master_detector_flat.h"
#include "sph_common_keywords.h"
#include <math.h>
#include "sph_error.h"
#include "sph_test.h"
#include "sph_utils.h"
#include "sph_filemanager.h"

#include <stdlib.h>
#include <string.h>
#include <time.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup  cutest_ifs_master_detector_flat  Unit test of the sph_ifs_master_\
 *                                         detector_flat recipe \
 *                                      and associated functions.
 *
 * This tests verifies the very basic requirements of the sph_ifs_master_detector
 * _flat recipe. All functions of the recipe object are tested in only one,
 * standard, test scenario. In this scenario, only the flat fielding for the
 * borad band lamp is tests and the parameters for the
 * recipe are the standard default parameters. The input raw frames
 * are created using the sph_test_create_ifs_raw_dff_frameset_broad_band
 * function.
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

static
int cutest_init_master_detector_flat_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    return sph_test_nop_code();
}

static
int cutest_clean_master_detector_flat_testsuite(void) {
    return sph_end_test();
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Common setup for the unit tests
 *
 * @param   frames      Address of pointer to the input frames to set up
 * @param   params      Address of pointer to the input parameterlist to set up
 *
 * This sets up the standard recipe parameter list and the input frames.
 * The frames an params pointers should not be initiallised yet.
 *
 * On return the pointers will point to the corresponding cpl structures that
 * are now filled with the setup data for the test.
 *
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ifs_master_detector_flat_common_setup(cpl_frameset** frames,
        cpl_parameterlist** params) {
    sph_error_code rerr = CPL_ERROR_NONE;

    cpl_test_nonnull( frames );
    cpl_test_nonnull( params );

    *frames = cpl_frameset_new();

    rerr = sph_test_create_ifs_raw_dff_frameset_broad_band(*frames, 3, 1);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    *params = sph_ifs_master_detector_flat_create_paramlist();

    cpl_test_nonnull( *frames );
    cpl_test_nonnull( *params );
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Common teardown for the unit tests
 *
 * @param   frames      pointer to the input frames to clean up
 * @param   params      pointer to the input parameterlist to clean up
 *
 * This cleans up the frameset and parameterlist
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ifs_master_detector_flat_common_teardown(cpl_frameset* frames,
        cpl_parameterlist* params) {

    cpl_test_nonnull( frames );
    cpl_test_nonnull( params );

    cpl_parameterlist_delete(params);
    sph_test_frameset_delete(frames);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the s[h_ifs_master_dark main function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ifs_master_detector_flat_check_frames(void) {
    sph_ifs_master_detector_flat* ifs_master_detector_flat = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;

    /* Setup */
    cutest_ifs_master_detector_flat_common_setup(&inframes, &inparams);

    ifs_master_detector_flat = sph_ifs_master_detector_flat_new(inframes,
            inparams);

    cpl_test_nonnull( ifs_master_detector_flat );
    cpl_frameset_delete(ifs_master_detector_flat->rawframes);
    cpl_frameset_delete(ifs_master_detector_flat->current_raw_frameset);
    /* Run code... */
    rerr = sph_ifs_master_detector_flat_check_frames(ifs_master_detector_flat);

    /* Verify */
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    cpl_test_error(CPL_ERROR_NONE);

    cpl_test_nonnull( ifs_master_detector_flat->inframes );

    cpl_test_eq( cpl_frameset_get_size( ifs_master_detector_flat->inframes),
            3);

    cpl_test_error(CPL_ERROR_NONE);

    sph_ifs_master_detector_flat_delete(ifs_master_detector_flat);
    cutest_ifs_master_detector_flat_common_teardown(inframes, inparams);

    return;
}

static
void cutest_ifs_master_detector_flat_check_params(void) {
    sph_ifs_master_detector_flat* ifs_master_dff = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;

    /* Setup */
    cutest_ifs_master_detector_flat_common_setup(&inframes, &inparams);

    ifs_master_dff = sph_ifs_master_detector_flat_new(inframes, inparams);

    cpl_test_nonnull( ifs_master_dff );

    /* Run code ... */
    rerr = sph_ifs_master_detector_flat_check_params(ifs_master_dff);

    /* Verify */
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_nonnull( ifs_master_dff->outfilename );
    cpl_test_nonnull( ifs_master_dff->badpix_filename );

    cpl_test_nonnull( ifs_master_dff->framecomb_parameterlist );

    cpl_test_eq_string( ifs_master_dff->badpix_filename,
            "dff_badpixels.fits");

    cpl_test_zero(ifs_master_dff->make_badpix);

    cpl_test_eq(
            cpl_parameterlist_get_size( ifs_master_dff->framecomb_parameterlist),
            2);
    cpl_test_error(CPL_ERROR_NONE);

    sph_ifs_master_detector_flat_delete(ifs_master_dff);
    cutest_ifs_master_detector_flat_common_teardown(inframes, inparams);
    return;

}

static
void cutest_ifs_master_detector_flat_new(void) {
    sph_ifs_master_detector_flat* ifs_master_dff = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;

    cpl_error_reset();
    /* Setup */
    cutest_ifs_master_detector_flat_common_setup(&inframes, &inparams);
    cpl_test_error(CPL_ERROR_NONE);

    /* Run code... */
    ifs_master_dff = sph_ifs_master_detector_flat_new(inframes, inparams);

    /* Verify */
    cpl_test_nonnull( ifs_master_dff );
    cpl_test_error(CPL_ERROR_NONE);
    sph_ifs_master_detector_flat_delete(ifs_master_dff);

    cutest_ifs_master_detector_flat_common_teardown(inframes, inparams);
    return;
}

static
void cutest_ifs_master_detector_flat_delete(void) {

    sph_ifs_master_detector_flat* ifs_master_dff = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;

    /* Setup */
    cutest_ifs_master_detector_flat_common_setup(&inframes, &inparams);

    ifs_master_dff = sph_ifs_master_detector_flat_new(inframes, inparams);
    cpl_test_error(CPL_ERROR_NONE);

    cpl_test_nonnull( ifs_master_dff );
    /* Run code... */
    rerr = sph_ifs_master_detector_flat_delete(ifs_master_dff);

    /* Verify */
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    cpl_test_error(CPL_ERROR_NONE);

    cutest_ifs_master_detector_flat_common_teardown(inframes, inparams);
    return;
}

static
void cutest_ifs_master_detector_flat_run(void) {
    sph_ifs_master_detector_flat* ifs_master_dff = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_propertylist* pl = NULL;
    cpl_frameset* inframes = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;

    /* Setup */
    cutest_ifs_master_detector_flat_common_setup(&inframes, &inparams);

    ifs_master_dff = sph_ifs_master_detector_flat_new(inframes, inparams);

    cpl_test_nonnull( ifs_master_dff );

    /* Run code... */
    rerr |= sph_ifs_master_detector_flat_run(ifs_master_dff);

    /* Verfiy */
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    cpl_test_error(CPL_ERROR_NONE);

    pl = cpl_propertylist_load("master_detector_flat_l1.fits", 0);
    cpl_test( cpl_propertylist_has(pl,"ESO QC LAMP FLUX AVG 1"));
    cpl_test( cpl_propertylist_has(pl,"ESO QC LAMP FLUX VARIANCE 1"));

    cpl_propertylist_delete(pl);
    pl = NULL;
    sph_ifs_master_detector_flat_delete(ifs_master_dff);
    cutest_ifs_master_detector_flat_common_teardown(inframes, inparams);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit tests of techcal_master_detector_flat
 recipe and associated functions
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("master_dff_test",
            cutest_init_master_detector_flat_testsuite,
            cutest_clean_master_detector_flat_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "sph_ifs_master_detector_flat_new",
                    cutest_ifs_master_detector_flat_new)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_ifs_master_detector_flat_check_frames",
                    cutest_ifs_master_detector_flat_check_frames)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_ifs_master_detector_flat_check_params",
                    cutest_ifs_master_detector_flat_check_params)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_ifs_master_detector_flat_delete",
                    cutest_ifs_master_detector_flat_delete)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_ifs_master_detector_flat_run",
                    cutest_ifs_master_detector_flat_run)) {
        return sph_test_get_error();
    }

    return sph_test_end();
}

/**@}*/
